package it.inaf.ia2.gms.service;

import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.persistence.LoggingDAO;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import it.inaf.ia2.gms.persistence.PermissionsDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import java.util.List;
import java.util.Optional;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class PermissionsService {

    private final PermissionsDAO permissionsDAO;
    private final LoggingDAO loggingDAO;

    @Autowired
    public PermissionsService(PermissionsDAO permissionsDAO, LoggingDAO loggingDAO) {
        this.permissionsDAO = permissionsDAO;
        this.loggingDAO = loggingDAO;
    }

    public List<PermissionEntity> getGroupPermissions(GroupEntity group) {
        return permissionsDAO.getGroupsPermissions(group.getId());
    }

    public List<PermissionEntity> findUserPermissions(GroupEntity group, String userId) {
        return permissionsDAO.findUserPermissions(userId, group.getPath());
    }

    public void removePermission(GroupEntity group, String userId) {
        permissionsDAO.deletePermission(group.getId(), userId);
        loggingDAO.logAction("Removed permission for " + userId + "in group " + group.getId());
    }

    public PermissionEntity addPermission(GroupEntity group, String userId, Permission permission) {

        Optional<PermissionEntity> existingPermissionEntity = permissionsDAO.findPermissionEntity(group.getId(), userId);

        PermissionEntity permissionEntity;
        if (existingPermissionEntity.isPresent()) {
            permissionEntity = existingPermissionEntity.get();
            Permission resultingPermission = Permission.addPermission(permissionEntity.getPermission(), permission);
            if (resultingPermission == permissionEntity.getPermission()) {
                return permissionEntity;
            } else {
                permissionEntity.setPermission(resultingPermission);
            }
        } else {
            permissionEntity = new PermissionEntity();
            permissionEntity.setGroupId(group.getId());
            permissionEntity.setUserId(userId);
            permissionEntity.setPermission(permission);
            permissionEntity.setGroupPath(group.getPath());
        }

        permissionEntity = permissionsDAO.createOrUpdatePermission(permissionEntity);
        loggingDAO.logAction("Added " + permission + " permission for " + userId + "in group " + group.getId());

        return permissionEntity;
    }

    public PermissionEntity createOrUpdatePermission(GroupEntity group, String userId, Permission permission) {

        PermissionEntity permissionEntity = new PermissionEntity();
        permissionEntity.setGroupId(group.getId());
        permissionEntity.setUserId(userId);
        permissionEntity.setPermission(permission);
        permissionEntity.setGroupPath(group.getPath());

        return permissionsDAO.createOrUpdatePermission(permissionEntity);
    }

    public PermissionEntity updatePermission(GroupEntity group, String userId, Permission permission) {

        PermissionEntity permissionEntity = permissionsDAO.findPermissionEntity(group.getId(), userId)
                .orElseThrow(() -> new IllegalArgumentException("Specified permission not found"));

        return permissionsDAO.updatePermission(permissionEntity, permission);
    }
}
