package it.inaf.ia2.gms.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import static it.inaf.ia2.gms.controller.ControllersMockData.*;
import it.inaf.ia2.gms.manager.GroupsManager;
import it.inaf.ia2.gms.manager.PermissionsManager;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.RapUserPermission;
import it.inaf.ia2.gms.model.request.AddPermissionRequest;
import it.inaf.ia2.gms.persistence.GroupsDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import it.inaf.ia2.gms.service.GroupNameService;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import static org.hamcrest.CoreMatchers.is;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.delete;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

@RunWith(MockitoJUnitRunner.class)
public class PermissionsControllerTest {

    @Mock
    private GroupsManager groupsManager;

    @Mock
    private PermissionsManager permissionsManager;

    @Mock
    private GroupsDAO groupsDAO;

    @InjectMocks
    private PermissionsController controller;

    private MockMvc mockMvc;

    private final ObjectMapper mapper = new ObjectMapper();

    private GroupEntity group;

    @Before
    public void init() {
        controller.groupNameService = new GroupNameService(groupsDAO);
        mockMvc = MockMvcBuilders.standaloneSetup(controller).build();

        group = new GroupEntity();
        group.setId("group_id");
        group.setName("GroupName");
        group.setPath("parent_id.group_id");

        when(groupsManager.getGroupById(eq("group_id"))).thenReturn(group);
    }

    @Test
    public void testGetPermission() throws Exception {

        when(permissionsManager.getDirectUserPermission(eq(group), eq("user_id"))).thenReturn(Permission.MANAGE_MEMBERS);

        mockMvc.perform(get("/ui/permission?groupId=group_id&userId=user_id")
                .contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.permission", is("MANAGE_MEMBERS")));
    }

    @Test
    public void testGetInexistentPermission() throws Exception {

        mockMvc.perform(get("/ui/permission?groupId=group_id&userId=user_id")
                .contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isNoContent());
    }

    @Test
    public void testAddPermissionPaginated() throws Exception {

        AddPermissionRequest request = new AddPermissionRequest();
        request.setGroupId("group_id");
        request.setUserId("user_id");
        request.setPermission(Permission.ADMIN);
        request.setPaginatorPage(1);
        request.setPaginatorPageSize(10);

        mockMvc.perform(post("/ui/permission")
                .content(mapper.writeValueAsString(request))
                .contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isCreated())
                .andExpect(jsonPath("$.currentPage", is(1)));

        verify(permissionsManager, times(1)).addPermission(eq(group), eq("user_id"), eq(Permission.ADMIN));
    }

    @Test
    public void testRemovePermissionPaginated() throws Exception {

        mockMvc.perform(delete("/ui/permission?groupId=group_id&userId=user_id&paginatorPage=1&paginatorPageSize=10"))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.currentPage", is(1)));

        verify(permissionsManager, times(1)).removePermission(eq(group), eq("user_id"));
    }

    @Test
    public void testGetGroupPermissions() throws Exception {

        when(groupsDAO.findGroupByParentAndName("", "LBT")).thenReturn(Optional.of(getLbtGroup()));
        when(groupsDAO.findGroupByParentAndName("lbt_id", "INAF")).thenReturn(Optional.of(getInafGroup()));

        List<RapUserPermission> permissions = new ArrayList<>();
        RapUserPermission up = new RapUserPermission();
        up.setUser(getRapUser());
        up.setPermission(Permission.ADMIN);
        permissions.add(up);
        when(permissionsManager.getAllPermissions(any())).thenReturn(permissions);

        mockMvc.perform(get("/permission?group=LBT.INAF").principal(getPrincipal())
                .accept(MediaType.TEXT_PLAIN))
                .andExpect(status().isOk())
                .andExpect(content().string("rap_user ADMIN\n"));
    }

    @Test
    public void testAddPermission() throws Exception {

        String userId = "target_user";
        Permission permission = Permission.ADMIN;

        GroupEntity lbt = getLbtGroup();
        GroupEntity inaf = getInafGroup();

        when(groupsDAO.findGroupByParentAndName("", "LBT")).thenReturn(Optional.of(lbt));
        when(groupsDAO.findGroupByParentAndName("lbt_id", "INAF")).thenReturn(Optional.of(inaf));

        PermissionEntity permissionEntity = new PermissionEntity();
        permissionEntity.setGroupId(inaf.getId());
        permissionEntity.setUserId(userId);
        permissionEntity.setPermission(permission);
        permissionEntity.setGroupPath(inaf.getPath());

        when(permissionsManager.addPermission(eq(inaf), eq(userId),
                eq(permission))).thenReturn(permissionEntity);

        mockMvc.perform(post("/permission")
                .param("group", "LBT.INAF")
                .param("user_id", userId)
                .param("permission", permission.toString())
                .accept(MediaType.TEXT_PLAIN)
                .contentType(MediaType.APPLICATION_FORM_URLENCODED))
                .andExpect(status().isOk());

        verify(permissionsManager, times(1))
                .addPermission(eq(inaf), eq(userId), eq(permission));
    }

    @Test
    public void testRemovePermission() throws Exception {

        GroupEntity lbt = getLbtGroup();
        GroupEntity inaf = getInafGroup();

        when(groupsDAO.findGroupByParentAndName("", "LBT")).thenReturn(Optional.of(lbt));
        when(groupsDAO.findGroupByParentAndName("lbt_id", "INAF")).thenReturn(Optional.of(inaf));

        mockMvc.perform(delete("/permission?group=LBT.INAF&user_id=userId")
                .accept(MediaType.TEXT_PLAIN))
                .andExpect(status().isNoContent());

        verify(permissionsManager, times(1)).removePermission(eq(inaf), eq("userId"));
    }
}
