package it.inaf.ia2.gms.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import static it.inaf.ia2.gms.controller.ControllersMockData.getInafGroup;
import static it.inaf.ia2.gms.controller.ControllersMockData.getLbtGroup;
import it.inaf.ia2.gms.manager.MembershipManager;
import it.inaf.ia2.gms.manager.PermissionsManager;
import it.inaf.ia2.gms.model.request.AddMemberRequest;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.persistence.GroupsDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.MembershipEntity;
import it.inaf.ia2.gms.service.GroupNameService;
import it.inaf.ia2.gms.service.GroupsService;
import java.util.Optional;
import static org.hamcrest.CoreMatchers.is;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.delete;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

@RunWith(MockitoJUnitRunner.class)
public class MembersControllerTest {

    @Mock
    private GroupsService groupsService;

    @Mock
    private PermissionsManager permissionsManager;

    @Mock
    private MembershipManager membershipManager;

    @Mock
    private GroupsDAO groupsDAO;

    @InjectMocks
    private MembersController controller;

    private MockMvc mockMvc;

    private final ObjectMapper mapper = new ObjectMapper();

    private GroupEntity lbt = getLbtGroup();
    private GroupEntity inaf = getInafGroup();

    @Before
    public void init() {
        controller.groupNameService = new GroupNameService(groupsDAO);
        mockMvc = MockMvcBuilders.standaloneSetup(controller).build();

        when(groupsService.getGroupById(eq("lbt_id"))).thenReturn(lbt);

        when(groupsDAO.findGroupByParentAndName("", "LBT")).thenReturn(Optional.of(lbt));
        when(groupsDAO.findGroupByParentAndName("lbt_id", "INAF")).thenReturn(Optional.of(inaf));
    }

    @Test
    public void testAddMemberPaginated() throws Exception {

        AddMemberRequest request = new AddMemberRequest();
        request.setGroupId("lbt_id");
        request.setUserId("user_id");
        request.setPaginatorPage(1);
        request.setPaginatorPageSize(10);
        request.setPermission(Permission.VIEW_MEMBERS);

        mockMvc.perform(post("/ui/member")
                .content(mapper.writeValueAsString(request))
                .contentType(MediaType.APPLICATION_JSON_VALUE))
                .andExpect(status().isCreated())
                .andExpect(jsonPath("$.currentPage", is(1)));

        verify(membershipManager, times(1)).addMember(any(), eq("user_id"));
    }

    @Test
    public void testRemoveMemberPaginated() throws Exception {

        mockMvc.perform(delete("/ui/member?groupId=group_id&userId=user_id&paginatorPage=1&paginatorPageSize=10"))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.currentPage", is(1)));

        verify(membershipManager, times(1)).removeMember(any(), eq("user_id"));
    }

    @Test
    public void testAddMember() throws Exception {

        when(groupsDAO.findGroupByParentAndName("", "LBT")).thenReturn(Optional.of(lbt));
        when(groupsDAO.findGroupByParentAndName("lbt_id", "INAF")).thenReturn(Optional.of(inaf));

        String userId = "target_user";

        MembershipEntity membership = new MembershipEntity();
        membership.setGroupId(lbt.getId());
        membership.setUserId(userId);

        mockMvc.perform(post("/membership")
                .param("group", "LBT.INAF")
                .param("user_id", userId)
                .accept(MediaType.TEXT_PLAIN)
                .contentType(MediaType.APPLICATION_FORM_URLENCODED))
                .andExpect(status().isOk());

        verify(membershipManager, times(1)).addMember(eq(inaf), eq(userId));
    }

    @Test
    public void testRemoveMember() throws Exception {

        mockMvc.perform(delete("/membership?group=LBT.INAF&user_id=userId"))
                .andExpect(status().isNoContent());

        verify(membershipManager, times(1)).removeMember(eq(inaf), eq("userId"));
    }
}
