const BASE_API_URL = process.env.VUE_APP_API_BASE_URL;

import axios from 'axios';

function apiRequest(options, showLoading = true, handleValidationErrors = false) {
  if (showLoading) {
    loading(true);
  }
  return new Promise((resolve, reject) => {
    axios(options)
      .then(response => {
        if (response.status === 204) {
          resolve({});
        } else {
          resolve(response.data);
        }
        loading(false);
      })
      .catch(error => {
        if (handleValidationErrors && error.response && error.response.status === 400) {
          reject(error.response.data);
        } else {
          dispatchApiErrorEvent(error);
        }
        loading(false);
      });
  });
}

function dispatchApiErrorEvent(error) {
  let event = new CustomEvent('apiError');
  let errorMessage;
  if (error.response && error.response.data && error.response.data.message) {
    errorMessage = error.response.data.message;
  } else if (error.message) {
    errorMessage = error.message;
  } else {
    errorMessage = 'Unknown error';
  }
  event.message = {
    title: error.error || 'Error',
    body: errorMessage
  };
  document.dispatchEvent(event);
}

/* For loading animation */
function loading(value) {
  let event = new CustomEvent('loading');
  event.value = value;
  document.dispatchEvent(event);
}

export default {
  fetchHomePageModel(input) {
    let url = BASE_API_URL +
      'ui/home?groupId=' + input.selectedGroupId +
      '&paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage;
    return apiRequest({
      method: 'GET',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      }
    });
  },
  fetchGroupsTab(input) {
    let url = BASE_API_URL +
      'ui/groups?groupId=' + input.selectedGroupId +
      '&paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage +
      '&onlyPanel=false';
    if (input.searchFilter !== null) {
      url += '&searchFilter=' + input.searchFilter;
    }
    return apiRequest({
      method: 'GET',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      }
    });
  },
  fetchGroupsPanel(input) {
    let url = BASE_API_URL +
      'ui/groups?groupId=' + input.selectedGroupId +
      '&paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage +
      '&onlyPanel=true';
    if (input.searchFilter !== null) {
      url += '&searchFilter=' + input.searchFilter;
    }
    return apiRequest({
      method: 'GET',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      }
    });
  },
  fetchMembersPanel(input) {
    let url = BASE_API_URL +
      'ui/members?groupId=' + input.selectedGroupId +
      '&paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage;
    return apiRequest({
      method: 'GET',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      }
    });
  },
  fetchPermissionsPanel(input) {
    let url = BASE_API_URL +
      'ui/permissions?groupId=' + input.selectedGroupId +
      '&paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage;
    return apiRequest({
      method: 'GET',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      }
    });
  },
  addGroup(newGroupName, leaf, input) {
    let url = BASE_API_URL + 'ui/group';
    return apiRequest({
      method: 'POST',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      },
      data: {
        newGroupName: newGroupName,
        parentGroupId: input.selectedGroupId,
        paginatorPageSize: input.paginatorPageSize,
        paginatorPage: input.paginatorPage,
        searchFilter: input.searchFilter,
        leaf: leaf
      }
    }, true, true);
  },
  updateGroup(groupId, newGroupName, leaf, input) {
    let url = BASE_API_URL + 'ui/group/' + groupId;
    return apiRequest({
      method: 'PUT',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      },
      data: {
        newGroupName: newGroupName,
        leaf: leaf,
        paginatorPageSize: input.paginatorPageSize,
        paginatorPage: input.paginatorPage,
        searchFilter: input.searchFilter
      }
    }, true, true);
  },
  removeGroup(groupId, input) {
    let url = BASE_API_URL + 'ui/group/' + groupId +
      '?paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage;
    if (input.searchFilter !== null) {
      url += '&searchFilter=' + input.searchFilter;
    }
    return apiRequest({
      method: 'DELETE',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      }
    });
  },
  searchUser(searchInput) {
    let url = BASE_API_URL + 'ui/users?search=' + searchInput;

    return apiRequest({
      method: 'GET',
      url: url,
      withCredentials: true,
      headers: {
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      }
    });
  },
  addPermission(userId, permission, input, override) {
    let url = BASE_API_URL + 'ui/permission';

    return apiRequest({
      method: 'POST',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      },
      data: {
        groupId: input.selectedGroupId,
        userId: userId,
        permission: permission,
        override: override,
        paginatorPageSize: input.paginatorPageSize,
        paginatorPage: input.paginatorPage
      }
    });
  },
  updatePermission(groupId, userId, permission) {
    let url = BASE_API_URL + 'ui/permission';

    return apiRequest({
      method: 'PUT',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      },
      data: {
        groupId: groupId,
        userId: userId,
        permission: permission
      }
    });
  },
  getPermission(groupId, userId) {
    let url = BASE_API_URL + 'ui/permission?groupId=' + groupId + '&userId=' + userId;

    return apiRequest({
      method: 'GET',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      }
    });
  },
  addMember(userId, permission, input) {
    let url = BASE_API_URL + 'ui/member';

    return apiRequest({
      method: 'POST',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      },
      data: {
        groupId: input.selectedGroupId,
        userId: userId,
        permission: permission,
        paginatorPageSize: input.paginatorPageSize,
        paginatorPage: input.paginatorPage
      }
    });
  },
  removeMember(userId, removeAlsoPermission, input) {
    let url = BASE_API_URL + 'ui/member' +
      '?groupId=' + input.selectedGroupId +
      '&userId=' + userId +
      '&removeAlsoPermission=' + removeAlsoPermission +
      '&paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage;
    return apiRequest({
      method: 'DELETE',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      }
    });
  },
  removePermission(userId, input) {
    let url = BASE_API_URL + 'ui/permission' +
      '?groupId=' + input.selectedGroupId +
      '&userId=' + userId +
      '&paginatorPageSize=' + input.paginatorPageSize +
      '&paginatorPage=' + input.paginatorPage;
    return apiRequest({
      method: 'DELETE',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      }
    });
  },
  search(input) {
    let url = BASE_API_URL + 'ui/search?query=' + input.genericSearch.filter +
      '&paginatorPage=' + input.genericSearch.paginatorPage +
      '&paginatorPageSize=' + input.genericSearch.paginatorPageSize +
      '&users=' + input.genericSearch.users + "&groups=" + input.genericSearch.groups;

    return apiRequest({
      method: 'GET',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      }
    });
  },
  openUserSearchResult(userId) {
    let url = BASE_API_URL + 'ui/search/user/' + userId;

    return apiRequest({
      method: 'GET',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      }
    });
  },
  keepAlive() {
    let url = BASE_API_URL + 'ui/keepAlive';

    return apiRequest({
      method: 'GET',
      url: url,
      withCredentials: true,
      headers: {
        'Cache-Control': 'no-cache'
      }
    }, false);
  },
  deleteInvitedRegistration(requestId, groupId) {
    let url = BASE_API_URL + 'ui/registration?' +
      'request_id=' + requestId + '&group_id=' + groupId;
    return apiRequest({
      method: 'DELETE',
      url: url,
      withCredentials: true,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
      }
    });
  }
};
