package it.inaf.ia2.gms.cli;

import it.inaf.ia2.gms.client.GmsClient;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import org.springframework.boot.CommandLineRunner;
import org.springframework.stereotype.Component;

@Component
public class CLI implements CommandLineRunner {

    private final GmsClient client;

    public CLI() throws IOException {

        File config = new File("gms.properties");
        if (!config.exists()) {
            System.err.println("Unable to find the file gms.properties");
            System.exit(1);
        }

        Properties properties = new Properties();
        try (InputStream in = new FileInputStream(config)) {
            properties.load(in);
        }

        String baseUrl = (String) properties.get("base_url");
        if (baseUrl == null) {
            System.err.println("Missing base_url in gms.properties");
            System.exit(1);
        }

        String clientId = (String) properties.get("client_id");
        if (clientId == null) {
            System.err.println("Missing client_id in gms.properties");
            System.exit(1);
        }

        String clientSecret = (String) properties.get("client_secret");
        if (clientSecret == null) {
            System.err.println("Missing client_secret in gms.properties");
            System.exit(1);
        }

        client = new GmsClient(baseUrl, clientId, clientSecret);
    }

    @Override
    public void run(String... args) throws Exception {
        if (args.length < 2) {
            displayUsage();
        }

        switch (args[0]) {
            case "create-group":
                client.addGroup(getNames(args, 1));
                System.out.println("Group created");
                break;
            case "delete-group":
                client.removeGroup(getNames(args, 1));
                System.out.println("Group deleted");
                break;
            case "add-member":
                if (args.length < 3) {
                    displayUsage();
                }
                client.addMember(getNames(args, 1, args.length - 2), args[args.length - 1]);
                System.out.println("Member added");
                break;
            case "remove-member":
                if (args.length < 3) {
                    displayUsage();
                }
                client.removeMember(getNames(args, 1, args.length - 2), args[args.length - 1]);
                System.out.println("Member removed");
                break;
            case "add-permission":
                if (args.length < 4) {
                    displayUsage();
                }
                client.addPermission(getNames(args, 1, args.length - 3), args[args.length - 2], args[args.length - 1]);
                System.out.println("Permission added");
                break;
            case "delete-permission":
                if (args.length < 4) {
                    displayUsage();
                }
                client.removePermission(getNames(args, 1, args.length - 2), args[args.length - 1]);
                System.out.println("Permission removed");
                break;
            default:
                displayUsage();
                break;
        }
    }

    private void displayUsage() {
        System.out.println("java -jar gms-client.jar\n"
                + "    create-group <name1 name2 name3>\n"
                + "    delete-group <name1 name2 name3>\n"
                + "    add-member <name1 name2 name3> <user_id>\n"
                + "    remove-member <name1 name2 name3> <user_id>\n"
                + "    add-permission <name1 name2 name3> <user_id> <permission>\n"
                + "    delete-permission <name1 name2 name3> <user_id>");
        System.exit(0);
    }

    private List<String> getNames(String[] args, int startIndex) {
        return getNames(args, startIndex, args.length - 1);
    }

    private List<String> getNames(String[] args, int startIndex, int endIndex) {
        List<String> names = new ArrayList<>();
        for (int i = startIndex; i <= endIndex; i++) {
            names.add(args[i]);
        }
        return names;
    }
}
