package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.authn.SessionData;
import it.inaf.ia2.gms.exception.UnauthorizedException;
import it.inaf.ia2.gms.model.CreateGroupRequest;
import it.inaf.ia2.gms.model.GroupNode;
import it.inaf.ia2.gms.model.GroupsModelRequest;
import it.inaf.ia2.gms.model.GroupsModelResponse;
import it.inaf.ia2.gms.model.PaginatedData;
import it.inaf.ia2.gms.model.PaginatedModelRequest;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.RenameGroupRequest;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.service.GroupsModelBuilder;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.service.GroupsTreeBuilder;
import it.inaf.ia2.gms.service.PermissionsService;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class GroupsController {

    @Autowired
    private SessionData session;

    @Autowired
    private PermissionsService permissionsService;

    @Autowired
    private GroupsService groupsService;

    @Autowired
    private GroupsModelBuilder groupsModelService;

    @Autowired
    private GroupsTreeBuilder groupsTreeBuilder;

    @GetMapping(value = "/groups", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public GroupsModelResponse getGroupsModelResponse(@Valid GroupsModelRequest groupsModelRequest) {
        return groupsModelService.getGroupsModel(groupsModelRequest, session.getUserId());
    }

    @PostMapping(value = "/group", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<GroupNode>> createGroup(@Valid @RequestBody CreateGroupRequest request) {

        GroupEntity parent = groupsService.getGroupById(request.getParentGroupId());

        if (permissionsService.getUserPermissionForGroup(parent, session.getUserId()) != Permission.ADMIN) {
            throw new UnauthorizedException("Missing admin permission");
        }

        GroupEntity newGroup = groupsService.addGroup(parent, request.getNewGroupName());

        PaginatedData<GroupNode> groupsPanel = getGroupsPanel(request.getParentGroupId(), request);

        return new ResponseEntity<>(groupsPanel, HttpStatus.CREATED);
    }

    @PutMapping(value = "/group/{groupId}", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<GroupNode>> renameGroup(@PathVariable("groupId") String groupId, @Valid @RequestBody RenameGroupRequest request) {

        GroupEntity group = groupsService.getGroupById(groupId);

        if (permissionsService.getUserPermissionForGroup(group, session.getUserId()) != Permission.ADMIN) {
            throw new UnauthorizedException("Missing admin permission");
        }

        GroupEntity renamedGroup = groupsService.renameGroup(group, request.getNewGroupName());

        GroupEntity parent = groupsService.getGroupByPath(renamedGroup.getParentPath());

        PaginatedData<GroupNode> groupsPanel = getGroupsPanel(parent.getId(), request);

        return ResponseEntity.ok(groupsPanel);
    }

    @DeleteMapping(value = "/group/{groupId}", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<?> deleteGroup(@PathVariable("groupId") String groupId, PaginatedModelRequest request) {

        GroupEntity group = groupsService.getGroupById(groupId);

        if (permissionsService.getUserPermissionForGroup(group, session.getUserId()) != Permission.ADMIN) {
            throw new UnauthorizedException("Missing admin permission");
        }

        GroupEntity parent = groupsService.deleteGroup(group);

        PaginatedData<GroupNode> groupsPanel = getGroupsPanel(parent.getId(), request);

        return ResponseEntity.ok(groupsPanel);
    }

    private PaginatedData<GroupNode> getGroupsPanel(String parentGroupId, PaginatedModelRequest paginatedRequest) {
        return groupsTreeBuilder.listSubGroups(parentGroupId, session.getUserId(), paginatedRequest);
    }
}
