#include <FileWrapper.h>

namespace DataImporter_ns
{

//==============================================================================
//      FileWrapper::FileWrapper()
//==============================================================================
FileWrapper::FileWrapper(Tango::DeviceImpl* deviceImpl_p,
    std::string storagePath, std::string filePath, int fileVersion,
    std::string fileName, boost::uint64_t totalFileSize) throw(std::logic_error)
    : Tango::LogAdapter(deviceImpl_p), m_storagePath(storagePath),
    m_filePath(filePath), m_fileVersion(fileVersion), m_fileName(fileName),
    m_expectedFileSize(totalFileSize)

{
    DEBUG_STREAM << "FileWrapper::FileWrapper()" << endl;

    m_outputFilePath /= storagePath;

    m_outputFilePath /= filePath;

    std::stringstream fileStream;
    fileStream << "/" << fileVersion;

    m_outputFilePath /= fileStream.str();

    try
    {
        if(!boost::filesystem::exists(m_outputFilePath))
            boost::filesystem::create_directories(m_outputFilePath);
    }
    catch(boost::filesystem::filesystem_error& ex)
    {
        throw std::logic_error(ex.what());
    }

    if(!boost::filesystem::is_directory(m_outputFilePath))
        throw std::logic_error("Destination path \'"
            + m_outputFilePath.string() + "\' is not a directory" );

    m_outputFilePath /= fileName;

    m_outputFileStream.open(m_outputFilePath.string(), std::ios::binary);
}

//==============================================================================
//      FileWrapper::~FileWrapper()
//==============================================================================
FileWrapper::~FileWrapper()
{
    DEBUG_STREAM << "FileWrapper::~FileWrapper()" << endl;

    m_outputFileStream.close();
}

//==============================================================================
//      FileWrapper::create()
//==============================================================================
FileWrapper::SP FileWrapper::create(Tango::DeviceImpl* deviceImpl_p,
    std::string storagePath, std::string filePath, int fileVersion,
    std::string fileName, boost::uint64_t fileSize)
    throw(std::logic_error)
{
    FileWrapper::SP d_sp(new FileWrapper(deviceImpl_p, storagePath, filePath,
        fileVersion, fileName, fileSize), FileWrapper::Deleter());

    return d_sp;
}

//==============================================================================
//      FileWrapper::getStoragePath()
//==============================================================================
std::string FileWrapper::getStoragePath()
{
    DEBUG_STREAM << "FileWrapper::getStoragePath()" << endl;

    return m_storagePath;
}

//==============================================================================
//      FileWrapper::getFilePath()
//==============================================================================
std::string FileWrapper::getFilePath()
{
    DEBUG_STREAM << "FileWrapper::getFilePath()" << endl;

    return m_filePath;
}

//==============================================================================
//      FileWrapper::getFileVersion()
//==============================================================================
int FileWrapper::getFileVersion()
{
    DEBUG_STREAM << "FileWrapper::getFileVersion()" << endl;

    return m_fileVersion;
}

//==============================================================================
//      FileWrapper::getFileName()
//==============================================================================
std::string FileWrapper::getFileName()
{
    DEBUG_STREAM << "FileWrapper::getFileName()" << endl;

    return m_fileName;
}

//==============================================================================
//      FileWrapper::isOpen()
//==============================================================================
bool FileWrapper::isOpen()
{
    return m_outputFileStream.is_open();
}

//==============================================================================
//      FileWrapper::isBad()
//==============================================================================
bool FileWrapper::isBad()
{
    return m_outputFileStream.bad();
}

//==============================================================================
//      FileWrapper::isCompleted()
//==============================================================================
bool FileWrapper::isCompleted()
{
    return (boost::uint64_t)m_outputFileStream.tellp() >= m_expectedFileSize;
}

//==============================================================================
//      FileWrapper::getLeftToWrite()
//==============================================================================
boost::uint64_t FileWrapper::getLeftToWrite()
{
    return m_expectedFileSize - (boost::uint64_t)m_outputFileStream.tellp();
}

//==============================================================================
//      FileWrapper::write()
//==============================================================================
void FileWrapper::write(std::vector<char>& writeBuff, boost::uint64_t& recvBytes)
{
    m_outputFileStream.write(&writeBuff[0], (std::streamsize)recvBytes);
}

//==============================================================================
//      FileWrapper::cleanUp()
//==============================================================================
void FileWrapper::cleanUp()
{
    DEBUG_STREAM << "FileWrapper::cleanUp()" << endl;

    if(m_outputFileStream.is_open())
        m_outputFileStream.close();

    boost::system::error_code errorCode;

    if(boost::filesystem::exists(m_outputFilePath))
        boost::filesystem::remove(m_outputFilePath, errorCode);
}

}   //namespace
