#ifndef CLIENT_H
#define	CLIENT_H

#include <Configuration.h>
#include <DBManager.h>
#include <ProtocolManager.h>
#include <Request.pb.h>
#include <Response.pb.h>
#include <FileWrapper.h>

#include <tango.h>

#include <boost/shared_ptr.hpp>
#include <boost/asio.hpp>
#include <boost/thread.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/cstdint.hpp>
#include <boost/array.hpp>

namespace DataImporter_ns
{

class DataImporter;

class Client : public Tango::LogAdapter
{
public:
//------------------------------------------------------------------------------
//  [Public] Shared pointer typedef
//------------------------------------------------------------------------------
    typedef boost::shared_ptr<Client> SP;

protected:
//------------------------------------------------------------------------------
//  [Protected] Constructor destructor
//------------------------------------------------------------------------------
    Client(DataImporter*, Configuration::SP);

    virtual ~Client();

public:
//------------------------------------------------------------------------------
//  [Public] Thread management methods
//------------------------------------------------------------------------------
    virtual void start();

    virtual void stop();

//------------------------------------------------------------------------------
//  [Public] Read state and status methods
//------------------------------------------------------------------------------
    virtual Tango::DevState readState();

    virtual std::string readStatus();

protected:
//------------------------------------------------------------------------------
//  [Protected] Write state and status methods
//------------------------------------------------------------------------------
    virtual void writeState(Tango::DevState);

    virtual void writeStatus(std::string);

//------------------------------------------------------------------------------
//  [Protected] IO service run thread method
//------------------------------------------------------------------------------
    virtual void run();

//------------------------------------------------------------------------------
//  [Protected] Lists update methods
//------------------------------------------------------------------------------
    virtual void startUpdateLists();

    virtual void handleUpdateLists(const boost::system::error_code&);

//------------------------------------------------------------------------------
//  [Protected] Endpoint resolution methods
//------------------------------------------------------------------------------
    virtual void startResolve();

    virtual void handleResolve(const boost::system::error_code&,
        boost::asio::ip::tcp::resolver::iterator);

//------------------------------------------------------------------------------
//  [Protected] Connection initialization methods
//------------------------------------------------------------------------------
    virtual void startConnect(boost::asio::ip::tcp::resolver::iterator) = 0;

    virtual void handleConnect(const boost::system::error_code&,
        boost::asio::ip::tcp::resolver::iterator) = 0;

//------------------------------------------------------------------------------
//  [Protected] Write request methods
//------------------------------------------------------------------------------
    virtual void startWriteRequest() = 0;

    virtual void handleWriteRequest(const boost::system::error_code&);

//------------------------------------------------------------------------------
//  [Protected] Read response header methods
//------------------------------------------------------------------------------
    virtual void startReadResponseHeader() = 0;

    virtual void handleReadResponseHeader(const boost::system::error_code&);

//------------------------------------------------------------------------------
//  [Protected] Read response body methods
//------------------------------------------------------------------------------
    virtual void startReadResponseBody(boost::uint32_t) = 0;

    virtual void handleReadResponseBody(const boost::system::error_code&);

//------------------------------------------------------------------------------
//  [Protected] Read data methods
//------------------------------------------------------------------------------
    virtual void startReadData(FileWrapper::SP) = 0;

    virtual void handleReadData(FileWrapper::SP, std::size_t,
        const boost::system::error_code&);

//------------------------------------------------------------------------------
//  [Protected] Protocol and transfer result handler methods
//------------------------------------------------------------------------------
    void onTransferCompleted(FileWrapper::SP);

    void onTransferFailed();

//------------------------------------------------------------------------------
//  [Protected] Connection reset and timeout handler methods
//------------------------------------------------------------------------------
    virtual void closeConnection() = 0;

    virtual void resetConnection();

//------------------------------------------------------------------------------
//  [Protected] Header encoding decoding methods
//------------------------------------------------------------------------------
    virtual void encodeHeader(std::vector<boost::uint8_t>&, boost::uint32_t)
        throw(std::runtime_error);

    virtual boost::uint32_t decodeHeader(std::vector<boost::uint8_t>&)
        throw(std::runtime_error);

//------------------------------------------------------------------------------
//  [Protected] Class variables
//------------------------------------------------------------------------------
    //Tango server class pointer
    DataImporter* m_dataImporter_p;

    //Configuration shared pointer
    Configuration::SP m_configuration_sp;

    //Database manager shared pointer
    DBManager::SP m_dBManager_sp;

    //Protocol manager shared pointer
    ProtocolManager::SP m_protocolManager_sp;

    //IO service instance
    boost::asio::io_service m_ioService;

    //Work IO service instance
    boost::scoped_ptr<boost::asio::io_service::work> m_work_sp;

    //DNS resolver instance
    boost::asio::ip::tcp::resolver m_resolver;

    //Thread for IO service run scoped pointer
    boost::scoped_ptr<boost::thread> m_thread_sp;

    //Connection write read timeout
    boost::asio::deadline_timer m_resetConnectionTimer;

    //File list update time
    boost::asio::deadline_timer m_listsUpdateTimer;

    //Tango state property mutex
    boost::mutex m_stateMutex;

    //Tango state property
    Tango::DevState m_state;

    //Tango status property mutex
    boost::mutex m_statusMutex;

    //Tango status property
    std::string m_status;

    //Address and port of remote endpoint
    std::string m_remoteEndpoint;

    //Header size on binary stream
    static const unsigned HEADER_SIZE = 4;

    //Buffer for binary data read from stream
    std::vector<boost::uint8_t> m_readBuff;	
	
    //Read buffer size
    static const boost::uint64_t BUFFER_SIZE = 40960;

    //Buffer for file data read from stream
    std::vector<char> m_fileBuff;
};

}   //End of namespace

#endif	/* CLIENT_H */

