#include <DBManager.h>

#include <boost/date_time.hpp>

#include <soci/mysql/soci-mysql.h>
#include <soci/use.h>

namespace DataExporter_ns
{

//==============================================================================
//      DBManager::DBManager()
//==============================================================================
DBManager::DBManager(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp) : Tango::LogAdapter(deviceImpl_p),
    m_configuration_sp(configuration_sp)
{
    DEBUG_STREAM << "DBManager::DBManager()" << endl;
}

//==============================================================================
//      DBManager::~DBManager()
//==============================================================================
DBManager::~DBManager()
{
    DEBUG_STREAM << "DBManager::~DBManager()" << endl;
}

//==============================================================================
//      DBManager::create()
//==============================================================================
DBManager::SP DBManager::create(Tango::DeviceImpl* deviceImpl_p,
    Configuration::SP configuration_sp)
{
    DBManager::SP d_sp(new DBManager(deviceImpl_p, configuration_sp),
        DBManager::Deleter());

    return d_sp;
}

//==============================================================================
//      DBManager::connect()
//==============================================================================
void DBManager::connect() throw(soci::soci_error)
{
    DEBUG_STREAM << "DBManager::connect()" << endl;

    boost::mutex::scoped_lock lock(m_connectionPoolMutex);

    unsigned int connectionNumber = m_configuration_sp->getDatabaseConnectionNumber();

    m_connectionPool_sp.reset(new soci::connection_pool(connectionNumber));

    std::stringstream connection;
    connection << " host=" << m_configuration_sp->getDatabaseHost();
    connection << " port=" << m_configuration_sp->getDatabasePort();
    connection << " user=" << m_configuration_sp->getDatabaseUsername();
    connection << " password=" << m_configuration_sp->getDatabasePassword();

    #ifdef VERBOSE_DEBUG
        INFO_STREAM << "DBManager::connect(): " << connection.str() << endl;
    #endif

    for(unsigned int i=0; i<connectionNumber; ++i)
    {
        m_connectionPool_sp->at(i).open(soci::mysql, connection.str());

        soci::mysql_session_backend* backend =
            static_cast<soci::mysql_session_backend*>(
            m_connectionPool_sp->at(i).get_backend());

        MYSQL* mysql = backend->conn_;

        bool reconnect = true;
        mysql_options(mysql, MYSQL_OPT_RECONNECT, &reconnect);
    }
}

//==============================================================================
//      DBManager::disconnect()
//==============================================================================
void DBManager::disconnect()
{
    DEBUG_STREAM << "DBManager::disconnect()" << endl;

    boost::mutex::scoped_lock lock(m_connectionPoolMutex);

    m_connectionPool_sp.reset();
}

//==============================================================================
//      DBManager::retrieveFileInfo()
//==============================================================================
DBManager::FileTuple DBManager::retrieveFileInfo(std::string schema,
    std::string table, int version, std::string name)
    throw(soci::soci_error)
{
    DEBUG_STREAM << "DBManager::retrieveFileInfo()" << endl;

    if(!m_connectionPool_sp)
        throw soci::soci_error("Connection pool not initialized");

    soci::session session(*m_connectionPool_sp);

    if(session.get_backend() == NULL)
        session.reconnect();

    soci::rowset<FileTuple> rows = (session.prepare << "select storage_path, "
        "file_path from " << schema << "." << table << " where "
        "file_version=:version and file_name like :name",
        soci::use(version, "version"), soci::use(name, "name"));

    std::vector<FileTuple> fileTupleList;

    std::copy(rows.begin(), rows.end(), std::back_inserter(fileTupleList));

    if(fileTupleList.empty())
    {
        std::stringstream errorStream;
        errorStream << "Table " << schema << "." << table
            << " does not contain file " << name << " version " << version;
        throw soci::soci_error(errorStream.str());
    }

    if(fileTupleList.size()>1)
    {
        std::stringstream errorStream;
        errorStream << "Table " << schema << "." << table
            << " has duplicate for file " << name << " version " << version;
        throw soci::soci_error(errorStream.str());
    }

    return fileTupleList.at(0);
}


}   //namespace
