#!/usr/bin/env python
#
"""
This set of libraries, analysisUtils.py, is a collection of often useful
python and/or CASA related functions.  It is an open package for anyone
on the science team to add to and generalize (and commit).  A few
practices will allow us to keep this useful.

1) Write the routines as generally as you can.

2) Before fundamentally changing the default behavior of a function
or class, consider your coworkers.  Do not modify the default behavior
without extreme need and warning.  If you need to modify it quickly,
consider a separate version until the versions can be blended (but please
do try to do the blending!).

3) There is a comment structure within the routines.  Please keep this
for additions because the documentation is automatically generated from
these comments.
 
Some of the examples in the inline help assume that you have imported 
the library to aU, as import analysisUtils as aU. You can of course do 
whatever you like, but the examples will thus have to be modified.

Thanks and good luck!  If you have any questions, bother Barkats or
Corder, then Robert.
 
S. Corder, 2010-11-07
"""

from __future__ import print_function  # prevents adding old-style print statements
if 1 :
    import os
    import shutil
    import distutils.spawn # used in class linfit to determine if dvipng is present
    import sys
    import re
    import telnetlib   
    import math
    import numpy as np
    import hashlib
    import binascii # used for debugging planet()
    from mpfit import mpfit
#    from pylab import *  # this is not good practice, and has been removed for python3/CASA6
    import pylab as pb
    import matplotlib.pyplot as plt
    import matplotlib.cm
    import matplotlib.colorbar
    from matplotlib.font_manager import FontProperties
    from numpy.fft import fft
    import fnmatch, pickle, traceback, copy as python_copy
    import scipy as sp
    from scipy.ndimage.filters import gaussian_filter
    import scipy.signal as spsig
    from scipy.interpolate import splev, splrep
    import inspect  # needed for python3/CASA6, but apparently not for python2

    import scipy.special # for Bessel functions
    import scipy.odr # for class linfit
#    import string # no longer used since string.join and string.upper are not available in CASA6
    import struct # needed for pngWidthHeight
    import glob
    import pprint
    import readscans as rs
    import time as timeUtilities
    import datetime
    import copy
    import tmUtils as tmu
    import compUtils  # used in class SQLD
    try:
        import pytz  # used in mjdToLocalTime
    except:
        print("pytz is not available in this python (used by au.mjdToLocalTime)")
    from scipy.special import erf, erfc  # used in class Atmcal
    from scipy import ndimage
    from scipy import polyfit
    from scipy import optimize # used by class linfit
    import random  # used by class linfit
    import matplotlib.ticker # used by plotWeather
    import matplotlib.colors
    from matplotlib import rc # used by class linfit
    from matplotlib.figure import SubplotParams
    from matplotlib.ticker import MultipleLocator # used by plotPointingResults
    from matplotlib.ticker import ScalarFormatter # used by plotWVRSolutions, among others
    import warnings
    import csv # used by getALMAFluxcsv and editIntentscsv
#    from StringIO import StringIO # needed for getALMAFluxcsv
    from io import BytesIO # python2/3 compatible, for getALMAFluxcsv
    from io import IOBase # python2/3 compatible, for checking if variable is a file
    import functools # python 2.6/3 compatible, for editIntents reduce
    import O2SounderPlayer
    import fileinput  # used by updateSBSummaryASDM
    mypath = os.path.dirname(__file__).replace('analysis_scripts',
                                               'scripts/R10.4_WORKING')
    sys.path.append(mypath)
    import collections # used by convertSMAAntennaPositionsToPads
    asdmLibraryAvailable = False
    try:
        if True:
            asdmPath = '/opt/software/acs/ACS-current/ACSSW/intlist/PIPELINE_INTROOT/lib/python/site-packages'
            if (os.path.exists(asdmPath)):
                if not os.path.exists(os.path.join(asdmPath,'mpl_toolkits')):
                    print("Appended path ", asdmPath)
                    sys.path.append(asdmPath)
                    asdmLibraryAvailable = True
            else:
#                asdmPath = '/alma/ACS-2016.5/ACSSW/lib/python/site-packages'
                asdmPath = '/alma/ACS-2018JUN/ACSSW/lib/python/site-packages'
                if (os.path.exists(asdmPath)):
                    if not os.path.exists(os.path.join(asdmPath,'mpl_toolkits')):
                        # some versions of ACS have an old mpl_toolkits that is missing mplot3d
                        print("Appended path ", asdmPath)
                        sys.path.append(asdmPath)
                        asdmLibraryAvailable = True
                    else:
                        import au_noASDMLibrary
#                        print("1) Imported au_noASDMLibrary")
                else:
                    import au_noASDMLibrary
#                    print("2) Imported au_noASDMLibrary")
    except:
        import au_noASDMLibrary
#        print("3) Imported au_noASDMLibrary")
    if asdmLibraryAvailable:
        try:
            from asdm import ASDM, ASDMParseOptions
        except:
#            print("Although ASDM.py is present, it failed to import")
            asdmLibraryAvailable = False
            import au_noASDMLibrary
#            print("4) Imported au_noASDMLibrary")
    try:
        import AntPosResult as APR
        APRAvailable = True
    except:
        APRAvailable = False
    try:
        # Try to import TelCal libraries
        import CompareAntPosResults as CAPR
        CAPRAvailable = True
    except:
        CAPRAvailable = False
    try:
        import plotbandpass3 as plotbp3
    except:
        plotbp3avail=False
    try:
        from importlib import reload
    except:
        pass  # reload is already available in python 2.x
    #
    # Beginning of CASA stuff.
    # 
    casaAvailable = False
    casaVersion = None
    # Check if this is CASA6  CASA 6
    try:
        import casalith
        casaVersion = casalith.version_string()
        plotxyAvailable = False
        pb.style.use('classic')
    except:
        # either we are importing into python, or CASA < 6
        if (os.getenv('CASAPATH') is not None):
            import casadef
            if casadef.casa_version >= '5.0.0':
                import casa as mycasa
                if 'cutool' in dir(mycasa):
                    cu = mycasa.cutool()
                    casaVersion = '.'.join([str(i) for i in cu.version()[:-1]]) + '-' + str(cu.version()[-1])
                else:
                    casaVersion = mycasa.casa['build']['version'].split()[0]
            else:
                casaVersion = casadef.casa_version
            print("casaVersion = ", casaVersion)
        else:
            casaVersion = None
            print("You appear to be importing analysisUtils into python (not CASA). version = ", '.'.join([str(i) for i in sys.version_info[:3]]))
    try:
        from six.moves import input as raw_input
    except:
        pass  # raw_input works in older versions where six.moves is unavailable
    if casaVersion is not None:
      try:
          from taskinit import *
          print("imported casatasks and tools using taskinit *")
          if casaVersion >= '5.9.9':
              from casatools import ctsys
              from casaplotms import plotms
              from casaviewer import imview
              from casatasks.private import simutil
              from casatasks.private import solar_system_setjy as sss
              useSolarSystemSetjy = True
              from casatools import calanalysis
              from casatools import synthesisutils 
              mysu = synthesisutils()
              getOptimumSize = mysu.getOptimumSize
              import checksource
      except:
          # in case not everyone has the taskinit.py provided by Darrell
          if casaVersion >= '5.9.9':
              # The following makes CASA 6 look like CASA 5 to a script like this.
              from casatasks import casalog
              from casatasks import gaincal
              from casatasks import tclean
              from casatasks import imstat
              from casatasks import imhead
              from casatasks import imregrid
              from casatasks import immath
              from casatasks import imsmooth
              from casatasks import makemask
              from casatasks import split
              from casatasks import importasdm
              from casatasks import clearstat
              from casatasks import gencal
              from casatasks import setjy
              from casatasks import listobs
              from casatasks import imsmooth
              from casatasks import flagmanager
              from casatasks import flagdata
              from casatasks import flagcmd
              from casatasks import makemask
              from casatasks import vishead
              from casatasks import visstat
              from casatasks import imsubimage
              from casatasks import imfit
              from casatasks import immath
              from casatasks import exportfits
              from casatasks import importfits
              from casatasks import imcollapse
              from casatasks import predictcomp
              from casatasks import simobserve
              from casatasks import simanalyze
              from casatasks import imhistory
              from casatasks import fixvis
              from casatasks import bandpass
              from almatasks import wvrgcal
              from casaviewer import imview
              # Tools
              from casatools import measures as metool
              from casatools import table as tbtool
              from casatools import atmosphere as attool
              from casatools import agentflagger as aftool
              from casatools import msmetadata as msmdtool
              from casatools import image as iatool
              from casatools import ms as mstool
              from casatools import regionmanager as rgtool
              from casatools import quanta as qatool
              from casatools import ctsys
              from casatools import synthesisutils 
              mysu = synthesisutils()
              getOptimumSize = mysu.getOptimumSize
              from casatasks.private import simutil
              from casatasks.private import solar_system_setjy as sss
              useSolarSystemSetjy = True
              print("imported casatasks and casatools individually")
              from casaplotms import plotms
              from casatools import calanalysis
              import checksource
      try:
        predictCompBodies = ['CALLISTO','CERES','EUROPA','GANYMEDE','IO',
                             'JUNO','JUPITER','MARS','NEPTUNE','PALLAS','TITAN',
                             'URANUS','VENUS','VESTA']
        if casaVersion < '5.9.9':
            import simutil # needed for obslist, getPadLOCsFromASDM; it has useful pad transforming functions
            # This arrangement allows you to call casa commands by name: - T. Hunter
            from listobs_cli import listobs_cli as listobs
            from gencal_cli import gencal_cli as gencal
            from imview_cli import imview_cli as imview
            from plotms_cli import plotms_cli as plotms
            from imstat_cli import imstat_cli as imstat
            from imsmooth_cli import imsmooth_cli as imsmooth
            from flagmanager_cli import flagmanager_cli as flagmanager
            from flagdata_cli import flagdata_cli as flagdata
            from flagcmd_cli import flagcmd_cli as flagcmd
            from makemask_cli import makemask_cli as makemask
            from vishead_cli import vishead_cli as vishead
            from importasdm_cli import importasdm_cli as importasdm
            calanalysis = casac.calanalysis
            try:
                from imsubimage_cli import imsubimage_cli as imsubimage
                useImsubimage = True
            except:
                useImsubimage = False
            from imfit_cli import imfit_cli as imfit
            from imhead_cli import imhead_cli as imhead
            from gaincal_cli import gaincal_cli as gaincal
            from clean_cli import clean_cli as clean
            try:
                from tclean_cli import tclean_cli as tclean
            except:
                pass
            from cleanhelper import cleanhelper # used for getOptimumSize
            getOptimumSize = cleanhelper.getOptimumSize
            try:
                from plotuv_cli import plotuv_cli as plotuv
            except:
                # casa <= 3.2 and >= 5.4.0
                pass
            from immath_cli import immath_cli as immath # used by complexToSquare()
            from imregrid_cli import imregrid_cli as imregrid # used by complexToSquare(),linmos
            from exportfits_cli import exportfits_cli as exportfits # used by makeSimulatedImage()
            from importfits_cli import importfits_cli as importfits # used by addGaussianToFITSImage()
            from visstat_cli import visstat_cli as visstat # used by scaleModel()
            from imcollapse_cli import imcollapse_cli as imcollapse
            try:
                from wvrgcal_cli import wvrgcal_cli as wvrgcal # used by wvrgcalStats()
                useWvrgcal = True
            except:
                useWvrgcal = False
            try:
                from predictcomp_cli import predictcomp_cli as predictcomp
                usePredictComp = True
            except:
                usePredictComp = False
                print("Could not import predictcomp")
            try:      # CASA 3.4
                from simobserve_cli import simobserve_cli as simobserve
                from simanalyze_cli import simanalyze_cli as simanalyze
            except:   # CASA 3.3
                try:
                    from sim_observe_cli import sim_observe_cli as simobserve
                    from sim_analyze_cli import sim_analyze_cli as simanalyze
                except:
                    pass  # 3.2
            if (casaVersion >= '4.6.0'):
                from imhistory_cli import imhistory_cli as imhistory
            try:
                import solar_system_setjy as sss
                useSolarSystemSetjy = True
            except:
                useSolarSystemSetjy = False
            from importasdm import importasdm
            from plotcal import plotcal
            try:  # this is no longer in CASA as of 5.0
                from sdplot import sdplot
                sdplotAvailable = True
            except:
                sdplotAvailable = False
                # The following could be workaround for SDcheckSpectra if necessary 
    #            from sdplotold import sdplotold as sdplot
            from split import split
            from setjy import setjy
            try:
                from plotxy import plotxy
                mymp = mp  # Check if the msplot tool is available, as it is needed by plotxy
                plotxyAvailable = True
            except:
                plotxyAvailable = False
            from fixvis import fixvis
            from bandpass import bandpass
            import viewertool
            from clearstat import clearstat
            if (casaVersion >= '3.3.0'):  # is not in 3.2, might be in 3.3
                from fixplanets import fixplanets
            try:
                import checksource
            except:
                pass
        # end of stuff to do in CASA < 6
        if (casaVersion >= '4.0.0'):
            # When this changes, be sure to update the inline help for au.planet
            defaultEphemeris = 'Butler-JPL-Horizons 2012'
        else:
            defaultEphemeris = 'Butler-JPL-Horizons 2010'
        if int(casaVersion.split('.')[0]) < 5:
            from scipy.stats import nanmean as scipy_nanmean
            from scipy.stats import nanmedian as scipy_nanmedian
            from scipy.stats import nanstd as scipy_nanstd
        else:
            try:
                from scipy import nanmean as scipy_nanmean
                from scipy import nanmedian as scipy_nanmedian
                from scipy import nanstd as scipy_nanstd
            except:
                # Some 5.0 stables were not yet updated to new scipy
                from scipy.stats import nanmean as scipy_nanmean
                from scipy.stats import nanmedian as scipy_nanmedian
                from scipy.stats import nanstd as scipy_nanstd
        if (casaVersion >= '5.9.9'):
            from casatasks.private.imagerhelpers.imager_base import PySynthesisImager
            from casatasks.private.imagerhelpers.input_parameters import ImagerParameters
        elif (casaVersion >= '5.0.0'):
            from imagerhelpers.imager_base import PySynthesisImager
            from imagerhelpers.input_parameters import ImagerParameters
        casaAvailable = True
        # end of casa stuff
      except:
        defaultEphemeris = 'Butler-JPL-Horizons 2012'
        if 'casadef' in locals():
            print("The import of casa items did not complete.  You may need to update the version of analysisUtils that you are using.\n See https://casaguides.nrao.edu/index.php/Analysis_Utilities")
            myversion = "$Id: analysisUtils.py,v 1.5049 2021/02/22 18:59:09 thunter Exp $"
            try:
                casaVersion = cu.version().tolist()
                casaVersion = '.'.join([str(i) for i in casaVersion[:-1]])
            except:
                casaVersion = casadef.casa_version
            print("au: %s, casa: %s" % (myversion.split()[2],casaVersion))
        else:
            print("You appear to be importing analysisUtils into python (not CASA). version = ", '.'.join([str(i) for i in sys.version_info[:3]]))
    else:
        print("CASAPATH is not defined, so I am skipping a lot of imports")
        defaultEphemeris = 'Butler-JPL-Horizons 2012'
    import fileIOPython as fiop
    from scipy.stats import scoreatpercentile, percentileofscore
    import types
    import operator
    from xml.dom import minidom
    import subprocess
    try:  # python 3
        from urllib.parse import urlparse, urlencode
        from urllib.request import urlopen, Request, HTTPPasswordMgrWithDefaultRealm, HTTPBasicAuthHandler, build_opener

        from urllib.error import HTTPError
        from subprocess import getstatusoutput, getoutput
        import XmlObjectifier_python3 as XmlObjectifier
    except ImportError:  # python2
        import XmlObjectifier
        from urlparse import urlparse
        from urllib import urlencode
        from urllib2 import urlopen, Request, HTTPError, HTTPPasswordMgrWithDefaultRealm, HTTPBasicAuthHandler, build_opener
        from commands import getstatusoutput, getoutput

    import itertools
    import calDatabaseQuery  # used by searchFlux
    import socket            # used by searchFlux to set tunnel default

    # pyfits is needed for getFitsBeam, numberOfChannelsInCube, findPixel, fixFitsHeader, setObject, getFitsDate, imageHistory, imageStdPerChannel
    if (casaVersion is None):
        # we are in python, so might want to look for astropy.io.fits first
        try:
            import pyfits
            pyfitsPresent = True
        except:
            pyfitsPresent = False
    elif (casaVersion >= '5.9.9'):
        try:
            with warnings.catch_warnings():
                # ignore pyfits deprecation message, maybe casa will include astropy.io.fits soon
                # Note: you cannot set category=DeprecationWarning in the call to filterwarnings() because the actual 
                #       warning is PyFITSDeprecationWarning, which of course is not defined until you import pyfits!
                warnings.filterwarnings("ignore") #, category=DeprecationWarning)  
                import pyfits 
                pyfitsPresent = True
        except:
            pyfitsPresent = False
    else:
        try:
            import pyfits
            pyfitsPresent = True
        except:
            pyfitsPresent = False
# endif 1

def version(short=False):
    """
    Returns the CVS revision number.
    """
    myversion = "$Id: analysisUtils.py,v 1.5049 2021/02/22 18:59:09 thunter Exp $"
    if (short):
        myversion = myversion.split()[2]
    return myversion

if casaAvailable:
    try:
        casalog.post('imported analysisUtils version %s from %s' % (version(),__file__))
    except:
        print("Failed to post to casa log")
        pass


"""
Constants that are sometimes useful.  Warning these are cgs, we might want to change them
to SI given the propensity in CASA to use SI.
"""
h=6.6260755e-27
k=1.380658e-16
c=2.99792458e10
c_mks=2.99792458e8
jy2SI=1.0e-26
jy2cgs=1.0e-23
pc2cgs=3.0857e18
au2cgs=1.4960e13
au2km=au2cgs*1e-5
solmass2g=1.989e33
earthmass2g=5.974e27
radiusEarthMeters=6371000
solLum2cgs = 3.826e33
mH = 1.673534e-24
G  = 6.67259e-8
Tcmb = 2.725
TROPICAL = 1
MID_LATITUDE_WINTER = 2
MID_LATITUDE_SUMMER = 3
ALMA_LONGITUDE=-67.754748 # ICT-4143,  -67.754694=JPL-Horizons,  -67.754929=CASA observatories
ALMA_LATITUDE=-23.029211  # ICT-4143,  -23.029167=JPL-Horizons,  -23.022886=CASA observatories
ARCSEC_PER_RAD=206264.80624709636
SECONDS_PER_YEAR = 31556925.

JPL_HORIZONS_ID = {'ALMA': '-7',
                   'VLA': '-5',
                   'GBT': '-9',
                   'MAUNAKEA': '-80',
                   'OVRO': '-81',
                   'geocentric': '500'
}
# Please keep these objects in this order (mean orbital radius from sun)
majorPlanets = ['SUN','MERCURY','VENUS','MOON','MARS','JUPITER','SATURN','URANUS','NEPTUNE','PLUTO']

maxIFbandDefinitions = {3: 8e9, 4:8e9, 5: 8e9, 6: 10e9, 7: 8e9, 8:8e9, 9:16e9, 10:16e9}

bandDefinitions = {
    1  : [35e9, 50e9  ],
    2  : [67e9  , 84e9  ], # upper end is actually 90, but we do not have logic to choose between 2 and 3
    3  : [84e9  , 116e9 ],
    4  : [125e9 , 163e9 ],
    5  : [163e9 , 211e9 ],
    6  : [211e9 , 275e9 ],
    7  : [275e9 , 373e9 ],
    8  : [385e9 , 500e9 ],
    9  : [602e9 , 720e9 ],
    10 : [787e9 , 950e9]
    }

almaReferencePosition = np.array([2225061.869, -5440061.953, -2481682.085]) # Robert Lucas
allWeatherStationPositions = {'Meteo129': np.array([ 2226292.373, -5440071.187, -2480490.57 ]),
                              'Meteo130': np.array([ 2223475.222, -5440620.327, -2481822.703]),
                              'Meteo131': np.array([ 2226146.018, -5439167.973, -2482751.669]),
                              'Meteo201': np.array([ 2218047.888, -5442740.475, -2480988.859]),
                              'Meteo309': np.array([ 2229937.944, -5435387.75 , -2486806.917]),
                              'Meteo410': np.array([ 2229279.046, -5440478.349, -2476637.931]),
                              'MeteoCentral': np.array([ 2225008.773, -5440202.705, -2481447.213]),
                              'MeteoTB1': np.array([ 2225262.839, -5440322.902, -2480961.371]),
                              'MeteoTB2': np.array([ 2225262.839, -5440322.902, -2480961.371]),
                              'MeteoOSF': np.array([ 2202176.215, -5445210.627, -2485352.924]),
                              'MeteoAPEX': np.array([ 2225039.5297, -5441197.6292, -2479303.3597]),
                              'MeteoASTE': np.array([ 2230817.87779425,  -5440189.64070941, -2475719.54322985])}

casaRevisionWithAlmaspws = '27481' # nominally '26688', but a pipeline version of 27480 does not have it
casaVersionForTClean = '5.6.0' # use tclean instead of clean in generateQA2Report
casaVersionWithMSMD = '4.1.0'
casaVersionWithMSMDFieldnames = '4.5'
casaVersionWithUndefinedFrame = '4.3.0'
weatherStationColors = ['b','r','g','c','m','brown','k','grey','orange','y','salmon']
DEFAULT_HOTLOAD_TEMP = 356
DEFAULT_AMBLOAD_TEMP = 287                

def getCasaVersion():
    """
    Uses casalith.version_string() in CASA 6; otherwise, 
    it uses cu.version().tolist() if available (i.e. CASA 5), 
    otherwise uses casadef.casa_version.  This is meant
    to replace all the pre-existing calls that simply used casadef.casa_version
    prior to the introduction of the cu tool and the move from svn to github.  
    New comparisons added to analysisUtils should be made using the function 
    au.casaVersionCompare.
    The actual code is now implemented up in the preamble, and this function
    simply returns the value of the global variable for backwards compatibility.
    -Todd Hunter
    """
#    try:
#        casaVersion = cu.version().tolist()
#        casaVersion = '.'.join([str(i) for i in casaVersion[:-1]])
#    except:
#        casaVersion = casadef.casa_version
    return casaVersion

def getCasaSubversionRevision():
    # could (equivalently) use the global variable (casaVersion) instead
    if getCasaVersion() >= '5.0':
        return '40000'
    else:
        return casadef.subversion_revision

def versionStringToArray(versionString):
    """
    Converts '5.3.0-22' to np.array([5,3,0,22], dtype=np.int32)
    -Todd Hunter
    """
    tokens = versionString.split('-')
    t = tokens[0].split('.')
    version = [np.int32(i) for i in t]
    if len(tokens) > 1:
        version += [np.int32(tokens[1])]
    return np.array(version)
    
def casaVersionCompare(comparitor, versionString):
    """
    This is the preferred way to perform CASA version checks in newly-written code going forward.
    It uses cu.compare_version in CASA >=5, otherwise uses string comparison as was done
    prior to the introduction of the cu tool.
    Example: casaVersionCompare('<', '5.3.0-22')
    versionString: can also be a list or array of integers in CASA >= 5, e.g. [5,3,0,22]
    -Todd Hunter
    """
    if getCasaVersion() < '5':
        if comparitor == '>=':
            comparison = casaVersion >= versionString
        elif comparitor == '>':
            comparison = casaVersion > versionString
        elif comparitor == '<':
            comparison = casaVersion < versionString
        elif comparitor == '<=':
            comparison = casaVersion <= versionString
        else:
            print("Unknown comparitor: ", comparitor)
            return False
    else:
        if type(versionString) == str:
            version = versionStringToArray(versionString)
        else:
            version = versionString
        if getCasaVersion() < '5.9.9':
            comparison = cu.compare_version(comparitor, version)
        else:
            comparison = ctsys.compare_version(comparitor, version)
    return comparison

def help(match='', showClassMethods=False, debug=False):
    """
    Print an alphabetized list of all the defined functions at the top level in
    analysisUtils.py.
    match: limit the list to those functions containing this string (case insensitive)
           If match is a list of strings or a list as a comma-delimited string, then
           the function must contain all strings to be considered a match.
    -- Todd Hunter
    """
    myfile = __file__
    if (myfile[-1] == 'c'):
        if (debug): print("au loaded from .pyc file, looking at .py file instead")
        myfile = myfile[:-1]
    aufile = open(myfile,'r')
    lines = aufile.readlines()
    if (debug): print("Read %d lines from %s" % (len(lines), __file__))
    aufile.close()
    commands = []
    if (type(match) == str and match.find(',')>0):
        match = match.split(',')
    for line in lines:
        if (line.find('def ') == 0):
            commandline = line.split('def ')[1]
            tokens = commandline.split('(')
            if (len(tokens) > 1):
                command = tokens[0]
            else:
                command = commandline.strip('\n\r').strip('\r\n')
            if (match == ''):
                commands.append(command)
            elif (type(match) == str):
                if (command.lower().find(match.lower()) >= 0):
                    commands.append(command)
            elif (type(match) == list or type(match) == np.ndarray):
                add = True
                for m in match:
                    if (command.lower().find(m.lower()) < 0):
                        add = False
                if (add): commands.append(command)
            else:
                print("The match argument must be a string or a list of strings.")
                return
    # Now identify matching classes
    classMethods = []
    for i,line in enumerate(lines):
        if (line.find('class ') == 0):
            myclass = line[6:-1]
            if (type(match) == str):
                match = match.split(',')
            for m in match:
                if line.lower().find(m.lower()) >= 0:
                    # now find all methods of this class
                    classMethods.append("Methods in class %s" % (myclass))
                    for j in range(i+1,len(lines)):
                        if lines[j].find('def') == 0:
                            # end of the current class
                            break
                        myline = lines[j]
                        loc = myline.find(' def ')
                        if loc >= 0 and myline.find('__init__')<0:
                            classMethods.append("  " + myline[loc+5:].split('(')[0])
                
    commands.sort()
    classMethods = np.unique(classMethods)
    for command in commands:
        print(command)
    if len(classMethods) > 1 and (len(commands) == 0 or showClassMethods):
        print(classMethods[0])
        classMethods = classMethods[1:]
        classMethods.sort()
        for command in classMethods:
            print(command)

def aggregate(object):
    """
    This function checks whether the object is a list, and if it is not,
    it wraps it in a list.
    """
    if type(object) == type([]):
        return object
    else:
        return [object]

def bandToFreqRange(band):
    """
    Returns a two-element integer list of the lowest and highest RF frequency
    for the specified ALMA band (in GHz).
    band: integer or string integer
    See also freqToBand(), and getBand().
    -Todd Hunter
    """
    band = int(band)
    if band not in bandDefinitions:
        print("Unrecognized band")
        return
    myrange = list(np.array(bandDefinitions[band],dtype=int)/1000000000)
    return myrange

def getBand(freq):
    """
    Converts a frequency into an ALMA band number, using the static
    dictionary called au.bandDefinitions
    freq: can be given either as a floating point value in Hz, or a string
          with units at the end (GHz, MHz, kHz, or Hz).
    See also freqToBand(), and bandToFreqRange().
    Todd Hunter
    """
    if (type(freq) == str):
        freq = parseFrequencyArgument(freq)
    for band in list(bandDefinitions.keys()):
        if ((freq <= bandDefinitions[band][1]) and (freq >= bandDefinitions[band][0])) :
            if band == 3 and freq < 90e9:
                print("Someday, Band 2 will also cover this frequency.")
            if band == 4 and freq > bandDefinitions[band+1][0]:
                print("Also observable in Band 5.")
            return band
    print("This frequency does not lie within any ALMA band.")
    return None

# A useful sequence of 19 unique matplotlib colors to cycle through
overlayColors = [
      [0.00,  0.00,  1.00],
      [0.00,  0.50,  0.00],
      [1.00,  0.00,  0.00],
      [0.00,  0.75,  0.75],
      [0.75,  0.00,  0.75],
      [0.25,  0.25,  0.25],
      [0.75,  0.25,  0.25],
#      [0.95,  0.95,  0.00],  yellow
      [0.25,  0.25,  0.75],
      [1.00,  0.75,  0.75], # [0.75, 0.75, 0.75] is invisible on gray border
      [0.00,  1.00,  0.00],
      [0.76,  0.57,  0.17],
      [0.54,  0.63,  0.22],
      [0.34,  0.57,  0.92],
      [1.00,  0.10,  0.60],
      [0.70,  1.00,  0.70], # [0.88,  0.75,  0.73], hard to see on gray
      [0.10,  0.49,  0.47],
      [0.66,  0.34,  0.65],
      [0.99,  0.41,  0.23]]
overlayMarkers = len(overlayColors) * 'o'
overlayColors += overlayColors + overlayColors + overlayColors
overlayMarkers += len(overlayMarkers) * '^' + len(overlayMarkers) * '*' + len(overlayMarkers) * 'v'

tableau20 = [(31, 119, 180), (174, 199, 232), (255, 127, 14), (255, 187, 120),
             (44, 160, 44), (152, 223, 138), (214, 39, 40), (255, 152, 150),
             (148, 103, 189), (197, 176, 213), (140, 86, 75), (196, 156, 148),
             (227, 119, 194), (247, 182, 210), (127, 127, 127), (199, 199, 199),
             (188, 189, 34), (219, 219, 141), (23, 190, 207), (158, 218, 229)]
tableau20 += tableau20 + tableau20
# Scale the RGB values to the [0, 1] range, which is the format matplotlib accepts.
for i in range(len(tableau20)):
    r, g, b = tableau20[i]
    tableau20[i] = (r / 255., g / 255., b / 255.) 

def makeTimeStamp():
    """
    Creates a current timestamp string like '2017-05-06T17:14:37'
    Used by manualAsdmExport.py.  See also convertTimeStamp(s)
    """
    return timeUtilities.strftime('%Y-%m-%dT%H:%M:%S')

def makeList(input) :
    """
    If a value is not a list, then make it a list.
    Used by locate, getAllanVariance, plotTrxFreq, plotTrxInfo, convertTimeStamps, etc.
    """
    if list not in [type(input)] : return [input]
    else : return input

def locate(msfile):
    """
    Script used to locate an ASDM or an MS file in the RADIO data directory tree.
    """
    host=os.getenv('HOSTNAME')
    if 'gns' in host:
        datadir='/groups/science/data/RADIO'
    elif 'red' in  host:
        datadir="/data/RADIO/"
    else:
        datadir="/data/RADIO/"
        
    if not msfile.find(':') == []:  msfile=msfile.replace('/','_').replace(':','_')
    print(msfile)

    # first search via locate utility
    a=os.popen('locate %s' %(msfile)).read()
    m=re.search('/data/RADIO/[A-Z]*/.*/[0-9].*/%s/' %msfile,a)
  
    if m is not None:
        location=m.group(0)
        print('using Unix locate')
        location=makeList(location)
    else:
        print('using Unix find')
        location=os.popen('find  %s -name %s' %(datadir,  msfile)).read().split('\n')
        
    if location != ['']:
        for i in range(len(location)):
            if ('ASDMBinary' not in location[i]):
                location=location[i]
                break
            else:
                print('could not find this file in local dir or in /data/RADIO')
                return
        dir=location.strip(msfile)
        return [location, dir]
    else:
        print('could not find this file in local dir or in /data/RADIO')
        return

def psd(d, fsample):
    """
    Function to take the psd of a timestream and return also the freq axis and the phase. 
    input should be 1D
    fsample in Hz
    """
    d = np.double(d)
    if np.floor(len(d) / 2) * 2 != len(d):
        d=d[1:]
        
    n = len(d) / 2
    transform = fft(d)
    transform=transform[0:n+1]
    freq = (fsample / 2.0) * (arange(n+1)) / n
    factor = repeat([2.0],n+1)
    factor[0]=1.
    factor[-1]=1.
    spec = pb.sqrt(factor / freq[1]) * transform/ (2*n);
    spec = abs(spec);
        
    return freq,spec


def avpsd(d, fsample, fres,  deg=1):
    """
    function [freq,spec,nrep]=avpsd(input, fsample, fres, med, deg)
    make an average psd by averaging the psd from small segments of resolution fres
    """
    n = len(d);
    nout = np.floor(fsample / fres)
    nrep = np.floor(n / nout)
    x=arange(nout)
    psdarr = zeros([nrep, nout/2+1])
    print("%i %i" %(nout, nrep))
    
    if (deg !=-1):
        for i in arange(nrep): 
            y=d[i*nout : (i+1)* nout]
            p =polyfit(x,y, deg);
            baseline=polyval(p,x);
            y=y-baseline;
            [freq,ps]=psd(y, fsample);        
            psdarr[i, :] = ps
    else:
        for i in arange(nrep):
            y=d[i*nout : (i+1) * nout]
            [freq,ps]=psd(y, fsample);
            psdarr[i, :] = ps;

    print(np.shape(psdarr))
    spec=np.sqrt(np.mean(psdarr*psdarr,0));
    
    return freq, spec

def smooth(x, window_len=10, window='hanning', verbose=False):
    """
    smooth the data using a window with requested size.
    
    This method is based on the convolution of a scaled window with the signal.
    The signal is prepared by introducing reflected copies of the signal 
    (with the window size) in both ends so that transient parts are minimized
    in the beginning and end part of the output signal.
    
    input:
        x: the input signal 
        window_len: the dimension of the smoothing window
        window: the type of window from 'flat', 'hanning', 'hamming', 'bartlett', 'blackman'
            flat window will produce a moving average smoothing.

    output:
        the smoothed signal
        
    example:

    t = linspace(-2,2,0.1)
    x = sin(t)+random.randn(len(t))*0.1
    y = smooth(x)
    
    see also: 
    
    numpy.hanning, numpy.hamming, numpy.bartlett, numpy.blackman, numpy.convolve
    scipy.signal.lfilter
 
    TODO: the window parameter could be the window itself if an array instead of a string   
    """

    if x.ndim != 1:
        raise ValueError("smooth only accepts 1 dimension arrays.")

    if x.size < window_len:
        raise ValueError("Input vector needs to be bigger than window size.")

    if window_len < 3:
        if verbose:
            print("Returning unsmoothed input data")
        return x

    if not window in ['flat', 'hanning', 'hamming', 'bartlett', 'blackman', 'gauss']:
        raise ValueError("Window is on of 'flat', 'hanning', 'hamming', 'bartlett', 'blackman', 'gauss'")

    s = np.r_[2*x[0]-x[window_len:1:-1], x, 2*x[-1]-x[-1:-window_len:-1]]
    if verbose:
        print("kernel length: %d, window_len: %d, data_len: %d" % (len(s), window_len, len(x)))
    
    if window == 'flat': #moving average
        w = np.ones(window_len,'d')
    elif window == 'gauss':
        w = gauss_kern(window_len)
    else:
        w = getattr(np, window)(window_len)
    y = np.convolve(w/w.sum(), s, mode='same')
    return y[window_len-1:-window_len+1]

def casaHanning(x, padOutput=False):
    """
    Simulates the CASA task hanningsmooth as described here:
    http://casa.nrao.edu/docs/taskref/hanningsmooth-task.html
    Inputs: 
    x: vector of values
    padOutput: if True, then include the 2 edge points with weights of 2/3 of edge channel 
               and 1/3 on adjacent channel
    Returns: array 
    -Todd Hunter
    """
    if (len(x) < 3):
        print("Spectrum has too few points")
        return
    output = np.zeros(len(x)-2)
    for i in range(len(output)):
        output[i] = x[i]*0.25 + x[i+1]*0.5 + x[i+2]*0.25
    if padOutput:
        output = [x[0]*2/3. + x[1]*1/3.] + list(output) + [x[-1]*2/3. + x[-2]*1/3.]
    return(np.array(output))

def compareHanning(npts=30000, bw=300, noise=0.5, ncomponents=50, maxtime=200,
                   xlimits = [45,80], ylimits = [-10, 3000], denominator=0.025,
                   drawstyle='default'):   
    """
    This was written to answer helpdesk ticket 5680.
    Compares Hanning smoothing in time domain (ALMA correlator) vs. CASA (hanningsmooth task)
    drawstyle: 'default' or 'steps' (stairsteps)
    -Todd Hunter
    """
    t = linspace(-maxtime,maxtime,num=npts)
    datastream = np.zeros(len(t))
    for freq in np.linspace(0.90, 1.1, num=ncomponents):
        amplitude = np.exp(-((freq-1.0)/denominator)**2)
        datastream += np.sin(t*freq)*amplitude + np.random.randn(len(t))*noise
    datastream /= np.max(datastream)
    almaHannWindow = scipy.signal.hann(len(datastream))
    almaDatastream = datastream * almaHannWindow
    almaDatastream /= np.max(almaDatastream)
    rawSpectrum = np.abs(np.fft.fft(datastream))
    casaSpectrum = np.append(np.abs(casaHanning(rawSpectrum)),[0])
    # Pad casa spectrum with two zeros
    casaSpectrum = np.append([0],casaSpectrum)
    almaSpectrum = np.abs(np.fft.fft(almaDatastream))
    pb.clf()
    desc = pb.subplot(211)
    pb.subplots_adjust(hspace=0.3)
    pb.plot(range(len(datastream)), datastream, 'k', range(len(almaHannWindow)), almaHannWindow, 'g',
            range(len(almaDatastream)), almaDatastream, 'r')
    pb.xlabel('Time sample')
    pb.title('Simulated timestream of a sum of sinewaves plus noise')
    pb.ylim([-1.1,1.1])
    size = 10
    pb.text(0.02, 0.18, 'Black: Raw data', transform=desc.transAxes, size=size)
    pb.text(0.02, 0.11, 'Green: Hann window in lag space', transform=desc.transAxes, size=size, color='g')
    pb.text(0.02, 0.04, 'Red: Data weighted by Hann window', transform=desc.transAxes, size=size, color='r')

    desc = pb.subplot(212)
    pb.plot(range(len(casaSpectrum[:npts/bw])), casaSpectrum[:npts/bw], 'b.-', 
            range(len(rawSpectrum[:npts/bw])), rawSpectrum[:npts/bw], 'k.-', 
            range(len(almaSpectrum[:npts/bw])), almaSpectrum[:npts/bw], 'r.-', markeredgewidth=0,
            drawstyle=drawstyle)
    pb.xlabel('Frequency bin')
    pb.xlim(xlimits)
    pb.ylim(ylimits)
    pb.title('Power Spectra')
    pb.text(0.02, 0.9, 'Black: Raw spectrum', transform=desc.transAxes, size=size)
    pb.text(0.02, 0.81, 'Blue: after CASA hanningsmooth', transform=desc.transAxes, size=size, color='b')
    pb.text(0.02, 0.72, 'Red: ALMA correlator output', transform=desc.transAxes, size=size, color='r')
    pb.text(0.07, 0.64, '(with Hann window weighting)', transform=desc.transAxes, size=size, color='r')
    rms = np.std(almaSpectrum-casaSpectrum)
    output = "rms of Red-Blue residual"
    pb.text(0.02, 0.5, output, transform=desc.transAxes, size=size)
    output = " = %f = %f%% of peak" % (rms, rms/np.max(almaSpectrum))
    pb.text(0.02, 0.42, output, transform=desc.transAxes, size=size)
    pb.draw()
    pb.savefig('compareHanning.png')

# some quick code I added.

def wtvar(X, W, method = "R"):
    """
    Only used by BaselineReducer.py, but commented-out there
    """
    sumW = sum(W)
    if method == "nist":
        xbarwt = sum([w * x for w,x in zip(W, X)])/sumW    # fixed.2009.03.07, divisor added.
        Np = sum([ 1 if (w != 0) else 0 for w in W])
        D = sumW * (Np-1.0)/Np
        return sum([w * (x - xbarwt)**2 for w,x in zip(W,X)])/D
    else: # default is R
        sumW2 = sum([w **2 for w in W])
        xbarwt = sum([(w * x)  for (w,x) in zip(W, X)])/sumW
        return sum([(w * (x - xbarwt)**2) for (w,x) in zip(W, X)])* sumW/(sumW**2 - sumW2)

def movingStD(a, w):
   """
   Not currently used anywhere.
   Moving standard deviation of array a calculated with window size w
   """
   res=[std(a[i:i+w]) for i in range(len(a)-w)]
   return res


def movStd(t,d, w,removeoutlier=True):
    """
    Not currently used anywhere.
    """
    trms=[]
    drms=[]

    #if removeoutlier:
    #    d = remove_outlier(d, sigma_th=10)
    
    s=len(d)
    
    for i in range(s//w):

        
        x=arange(w-1)
        y=d[w*i:w*(i+1)-1]
        polycoeffs=polyfit(x, y, 2)
        yfit=polyval(polycoeffs, x)
        #pl.clf()
        #pl.plot(x,y)
        #pl.plot(x,yfit)
        y=y-yfit
        #pl.plot(x,y)
        #raw_input()
           
        trms.append(t[w*(i+1)])
        drms.append(std(y))

    return trms, drms


def gauss_kern(size):
    """ 
    Returns a normalized 2D gauss kernel array for convolutions.  Used by smooth.
    """
    size = int(size)
    x= np.mgrid[-size:size+1]
    g = np.exp(-(x**2/float(size)))
    return g / g.sum()

def onedgaussianplus(x, H,A,dx,w, r):
    #fit a 1D gaussian plus a linear term
    return H+A*np.exp(-(x-dx)**2/(2*w**2))+r*(x-dx)

def onedgaussian(x,H,A,dx,w):
    """
    Returns a 1-dimensional gaussian of form
    H+A*np.exp(-(x-dx)**2/(2*w**2))
    """
    return H+A*np.exp(-(x-dx)**2/(2*w**2))

def onedgaussfit(xax,data,err=None,params=[0,1,0,1,0],fixed=[False,False,False,False,False],limitedmin=[False,False,False,False,False],
        limitedmax=[False,False,False,False,False],minpars=[0,0,0,0,0],maxpars=[0,0,0,0,0],quiet=True,shh=True):
    """
    Only called by Coherence.py
    Inputs:
       xax - x axis
       data - y axis
       err - error corresponding to data

       params - Fit parameters: Height of background, Amplitude, Shift, Width, Linear
       fixed - Is parameter fixed?
       limitedmin/minpars - set lower limits on each parameter
       limitedmax/maxpars - set upper limits on each parameter
       quiet - should MPFIT output each iteration?
       shh - output final parameters?

    Returns:
       Fit parameters
       Model
       Fit errors
       chi2
    """

    def mpfitfun(x,y,err):
        if err is None:
            def f(p,fjac=None): return [0,(y-onedgaussianplus(x,*p))]
        else:
            def f(p,fjac=None): return [0,(y-onedgaussianplus(x,*p))/err]
        return f

    if xax is None:
        xax = np.arange(len(data))

    parinfo = [ {'n':0,'value':params[0],'limits':[minpars[0],maxpars[0]],'limited':[limitedmin[0],limitedmax[0]],'fixed':fixed[0],'parname':"HEIGHT",'error':0} ,
                {'n':1,'value':params[1],'limits':[minpars[1],maxpars[1]],'limited':[limitedmin[1],limitedmax[1]],'fixed':fixed[1],'parname':"AMPLITUDE",'error':0},
                {'n':2,'value':params[2],'limits':[minpars[2],maxpars[2]],'limited':[limitedmin[2],limitedmax[2]],'fixed':fixed[2],'parname':"SHIFT",'error':0},
                {'n':3,'value':params[3],'limits':[minpars[3],maxpars[3]],'limited':[limitedmin[3],limitedmax[3]],'fixed':fixed[3],'parname':"WIDTH",'error':0},
                {'n':4,'value':params[4],'limits':[minpars[4],maxpars[4]],'limited':[limitedmin[4],limitedmax[4]],'fixed':fixed[4],'parname':"LINEAR",'error':0}]
    
    mp = mpfit(mpfitfun(xax,data,err),parinfo=parinfo,quiet=quiet)
    mp.status
    mpp = mp.params
    mpperr = mp.perror
    chi2 = mp.fnorm

    if not shh:
        for i,p in enumerate(mpp):
            parinfo[i]['value'] = p
            print(parinfo[i]['parname'],p," +/- ",mpperr[i])
        print("Chi2: ",mp.fnorm," Reduced Chi2: ",mp.fnorm/len(data)," DOF:",len(data)-len(mpp))

    return mpp,onedgaussianplus(xax,*mpp),mpperr,chi2


def almaToGeo(lon, lat, alma):
    """
    Convert the local (horizontal) coordinates into geocentric
    lon: longitude in radians, e.g. np.radians(au.ALMA_LONGITUDE)
    lat: latitude in radians, e.g. np.radians(au.ALMA_LATITUDE)
    alma: local coordinates: [E,N,U]
    """
    geo = [0, 0, 0]
    geo[0] = -math.sin(lon) * alma[0] \
             - math.cos(lon) * math.sin(lat) * alma[1] \
             + math.cos(lon) * math.cos(lat) * alma[2]
    geo[1] = math.cos(lon) * alma[0] \
             - math.sin(lon) * math.sin(lat) * alma[1] \
             + math.sin(lon) * math.cos(lat) * alma[2]
    geo[2] = math.cos(lat) * alma[1] + math.sin(lat) * alma[2]
    return geo

def geoToAlma(lon, lat, geo):
    """
    Convert the geocentric coordinates into the local (horizontal) ones.
    lon: longitude in radians (e.g. au.ALMA_LONGITUDE)
    lat: latitude in radians (e.g. au.ALMA_LATITUDE)
    geo: geocentric coordinates: [X,Y,Z]
    """
    alma = [0, 0, 0]
    alma[0] = -geo[0] * math.sin(lon) + geo[1] * math.cos(lon)
    alma[1] = -geo[0] * math.cos(lon) * math.sin(lat)  \
              - geo[1] * math.sin(lon) * math.sin(lat) \
              + geo[2] * math.cos(lat)
    alma[2] = geo[0] * math.cos(lon) * math.cos(lat) \
              + geo[1] * math.sin(lon) * math.cos(lat) \
              + geo[2] * math.sin(lat)
    return alma

def getDataColumnName(inputms):
    """
    Gets the name of the data column of a measurement set: either 'DATA'
    or 'FLOAT_DATA'
    """
    mytb = createCasaTool(tbtool)
    mytb.open(inputms)
    colnames = mytb.colnames()
    if 'FLOAT_DATA' in colnames:
        data_query= 'FLOAT_DATA'
    else:
        data_query = 'DATA'
    mytb.close()
    return(data_query)

def dataColumns(vis):
    """
    Returns the names of the data columns (data, model, corrected) in a measurement set.
    -Todd Hunter
    """
    mytb = createCasaTool(tbtool)
    mytb.open(vis)
    names = mytb.colnames()
    mytb.close()
    columns = []
    for i in ['DATA','FLOAT_DATA','MODEL_DATA','CORRECTED_DATA']:
        if i in names:
            columns.append(i)
    return columns
    
def getSpectralData(inputms, dd, scanum=[]):
    """
    Used by Coherence.py
    """
    mytb = tbtool()
    mytb.open(inputms)
    if size(scanum) == 0:
        specTb = mytb.query('ANTENNA1==0 && ANTENNA2==1 && DATA_DESC_ID==%d'%(dd))
    else:
        specTb = mytb.query('ANTENNA1==0 && ANTENNA2==1 && DATA_DESC_ID==%d && SCAN_NUMBER == %d'%(dd,scanum))
    if 'FLOAT_DATA' in specTb.colnames():
        data_query= 'FLOAT_DATA'
    else:
        data_query='DATA'
    specData = specTb.getcol(data_query)
    specTime = specTb.getcol('TIME')
    specTb.close()
    mytb.close()
    date  = int(specTime[0]/ 86400)
    specTime = specTime - date*86400
    return [specTime, specData]

def getSpectralAutoData(inputms,iant, dd, scanum=[]):
    """
    Used by Coherence.py, reduc_cutscans.py and reduc_oof.py
    """
    mytb = tbtool()
    mytb.open(inputms)
    if size(scanum) == 0:
        specTb = mytb.query('ANTENNA1==%d && ANTENNA2==%d && DATA_DESC_ID==%d'%(iant, iant,dd))
    else:
        specTb = mytb.query('ANTENNA1==%d && ANTENNA2==%d && DATA_DESC_ID==%d && SCAN_NUMBER == %d' % (iant, iant ,dd, scanum))
    if 'FLOAT_DATA' in specTb.colnames():
        data_query= 'FLOAT_DATA'
    else:
        data_query='DATA'
    specData = specTb.getcol(data_query)
    specTime = specTb.getcol('TIME')
    specTb.close()
    mytb.close()
    date  = int(specTime[0]/ 86400)
    specTime = specTime - date*86400
    return [specTime, specData]

#def getMeasFocus(msfile, aid, scanum):
#    """
#    Now obsolete
#    Look into EditfocusModel for more upto date versions
#    """
#    
#    try:
#        print "Focus values from actual measurements at start of Beam map."
#        if os.path.isdir(msfile+'/ASDM_FOCUS'):
#            tb.open(msfile+'/ASDM_FOCUS')
#            antid=tb.getcol('antennaId')
#            foc=tb.getcol('measuredFocusPosition')
#            asdmtime=tb.getcol('timeInterval')
#
#            q=find(antid== 'Antenna_%i' %aid)
#            asdmtime=asdmtime[0,q]
#            foc=foc[:,q]
#            
#            tb.open(msfile+'/ANTENNA')
#            antennas=tb.getcol('NAME')
#            tb.open(msfile)
#            scan=tb.getcol('SCAN_NUMBER')
#            t0=tb.getcol('TIME')
#            tb.close() 
#            f=find(scan==int(scanum))[0]
#            scanbeg=t0[f]
#           
#            # cross reference asdmtime and standard time for start of beam map.
#            asdmindex=find(asdmtime > scanbeg)[0]
#            focus=zeros([3])
#            um=1e6
#            antenna=antennas[aid]
#            focus[0]=foc[0,asdmindex]
#            focus[1]=foc[1,asdmindex]
#            focus[2]=foc[2,asdmindex]
#            print "%s (X  Y  Z)= (%.0f %.0f %.0f) microns " \
#                  % (antenna, focus[0]*um, focus[1]*um, focus[2]*um)
#            return focus
#        else:
#            print "The ms file is missing /ASDM_FOCUS table. Regenerate the MS file with that table. Using batchasdm2MS for exemple"
#
#    except:
#        return (NaN, NaN, NaN)


plotOption = {'0-1' : 'b.', '0-2' : 'r.'}
def getAllanVariance(vis,antenna1=None,antenna2=None,spwID=None,param='phase',scan=None,state=None,doPlot=True) :
    """
    Currently unused by anything else in analysis_scripts.
    """
    if param not in ['phase','real','imag'] : return 'you are a dumb fuck.'
    if spwID is None : spwID = getChanAverSpwIDBaseBand0(vis)
    if antenna2 is None : antenna2 = getAntennaNames(vis)
    else : antenna2 = makeList(antenna2)
    if antenna1 is None : antenna1 = getAntennaNames(vis)
    else : antenna1 = makeList(antenna1)
    data = Visibility(vis,antenna1=antenna1[0],antenna2=antenna2[0],spwID=spwID,scan=scan,state=state)
    aV = {}
    for i in antenna1 :
        for j in antenna2 :
            if i < j :
                data.setAntennaPair(i,j)
                aV[("%s-%s" % (i,j))] = allanVariance(data.phase,data.specTime,data.specFreq.mean())
                if doPlot : pb.plot(aV[("%s-%s" % (i,j))][:,0],np.log10(aV[("%s-%s" % (i,j))][:,1]),'.')
    return aV

def allanVariance(phase,time,ref_freq) :
    Nallan=[]
    ave=[]
    Ndata = len(time)
    Nmax = int(np.floor(Ndata/2))
    dt = (time[1:]-time[:-1]).mean()
    for i in range(1,Nmax):
        n=0
        y = []
        for j in range(Ndata):
            k=j+2*i
            if k > Ndata-1:
                break
            z=phase[:,:,k]-2*phase[:,:,i+j]+phase[:,:,j]
            y.append(z**2)
            n+=1
        ave.append(np.mean(y))
        Nallan.append(n)
    Ntau=len(ave)
    Allan=array(Ntau*[2*[0.0]],dtype=float)
    for i in range(Ntau):
        y=[]
        Allan[i][0]=dt*(i+1)
        Allan[i][1]=math.sqrt(ave[i]/2.0/(2*math.pi*ref_freq*Allan[i][0])**2)
    return Allan

def phaseClosure(vis, antenna1, antenna2, antenna3, spw, field, scan, vm=''):
     antennas=sorted([antenna1,antenna2,antenna3])
     antenna1=antennas[0]
     antenna2=antennas[1]
     antenna3=antennas[2]
     visVal=Visibility(vis,antenna1=antenna1,antenna2=antenna2,spwID=spw,field=field,scan=scan, vm=vm)
     phi01=visVal.phase
     visVal.setAntenna2(antenna3)
     phi02=visVal.phase
     visVal.setAntenna1(antenna2)
     phi12=visVal.phase
     close=phi01+phi02-phi12
     
     return close,visVal

def phaseClosureStats(msName='', scan='', intent='CALIBRATE_BANDPASS#ON_SOURCE', chanEdge=0.2):

    if casaVersion < casaVersionWithMSMD:
        sys.exit('ERROR: CASA versions earlier than %s are not supported.' % (casaVersionWithMSMD))

    if msName == '': sys.exit('ERROR: no ms name specified.')

    if scan == '':
        if casaVersion >= casaVersionWithMSMD:
            mymsmd = createCasaTool(msmdtool)
            mymsmd.open(msName)
            scan2 = mymsmd.scansforintent(intent).tolist()
            mymsmd.close()
        else:
            sys.exit('ERROR: no scan specified.')
    else:
        scan2 = scan.split(',')
        scan2 = [int(i) for i in scan2]
    mytb = tbtool()
    mytb.open(msName+'/ANTENNA')
    antIndex = range(mytb.nrows())
    mytb.close()
    mytb.open(msName+'/DATA_DESCRIPTION')
    spwIds1 = mytb.getcol('SPECTRAL_WINDOW_ID').tolist()
    mytb.close()

    phase2 = {}

    mytb.open(msName)

    for scan1 in scan2:

        phase2[scan1] = {}

        mymsmd.open(msName)
        spwIds = mymsmd.spwsforscan(scan1)
        spwIds = [i for i in spwIds if i in mymsmd.tdmspws().tolist()+mymsmd.fdmspws().tolist()]
        mymsmd.close()

        for i in spwIds:

            phase2[scan1][i] = {}

            tb1 = mytb.query('SCAN_NUMBER == '+str(scan1)+' AND DATA_DESC_ID == '+str(spwIds1.index(i)))

            time1 = tb1.getcol('TIME')
            ant1 = tb1.getcol('ANTENNA1')
            ant2 = tb1.getcol('ANTENNA2')
            data1 = tb1.getcol('DATA')
            data2 = data1.swapaxes(0,2)
            flagrow1 = tb1.getcol('FLAG_ROW')
            flag1 = tb1.getcol('FLAG')
            flag2 = flag1.swapaxes(0,2)

            phase3 = []

            for time2 in np.unique(time1):

                ij = itertools.combinations(antIndex, 3)

                for j in ij:

                    j = sorted(j)

                    ij2 = []

                    for k in range(3):

                        ij1 = np.argwhere((time1 == time2) & (((ant1 == j[k%3]) & (ant2 == j[(k+1)%3])) | ((ant1 == j[(k+1)%3]) & (ant2 == j[k%3])))).flatten()
                        if len(ij1) == 1: ij2.append(ij1[0]*math.copysign(1, ant2[ij1[0]] - ant1[ij1[0]]))

                    if len(ij2) == 3:

                        if flagrow1[abs(ij2[0])] == 0 and flagrow1[abs(ij2[1])] == 0 and flagrow1[abs(ij2[2])] == 0:

                            k1 = int(len(data2[0]) * chanEdge)
                            k2 = int(len(data2[0]) * (1-chanEdge))

                            for kl in range(len(data2[0][0])):

                                phase1 = []
                                for k in range(k1, k2):
                                    if flag2[abs(ij2[0])][k][kl] == 0 and flag2[abs(ij2[1])][k][kl] == 0 and flag2[abs(ij2[2])][k][kl] == 0:
                                        phase4 = np.rad2deg(math.copysign(1, ij2[0])*np.angle(data2[abs(ij2[0])][k][kl]) + math.copysign(1, ij2[1])*np.angle(data2[abs(ij2[1])][k][kl]) - math.copysign(1, ij2[2])*np.angle(data2[abs(ij2[2])][k][kl]))
                                        phase4 = phase4 - round(phase4/360.)*360.
                                        phase1.append(phase4)

                                phase3.append(np.mean(phase1))

            phase2[scan1][i]['min'] = min(phase3)
            phase2[scan1][i]['max'] = max(phase3)
            phase2[scan1][i]['mean'] = np.mean(phase3)
            phase2[scan1][i]['stddev'] = np.std(phase3)

    mytb.close()

#     pb.figure()
#     n, bins, patches = pb.hist( phase3, bins=range(int(np.ceil(max(phase3))+1)), histtype='bar')

    return phase2

class ValueMapping:
    """
    Input: The name of an MS dataset as a string.
    Purpose: This class provides details on the mapping of various parameters to each other.  For example, if you would like to
             know which scans observed a given field source or over which time interval a scan was executed, this is the place to look.
             Included in that are functions which map antenna name to antenna id and field name to field id.  This is useful in building
             other routines that allow you to not require the user to input one or the other type of information.  It also gets unique
             lists of items, like antennas, field sources, scans, intents, etc.

    Responsible: S. Corder and other contributors
    Example: vm = aU.ValueMapping('myInputMS.ms')
    Suggested Improvements:
          (done, 06-04-2011, scorder)1) Change some of the get methods to do methods because they aren't really returning anything
          2) Add spectral window mapping information, spectral windows, spectral windows to fields, spectral windows to scans,
             spectral windows to frequency (central and channelized): Basically make a dictionary of all the spectral line info stuff,
             per spectral window.  Rework combination of SensitivityCalculator and VM....
          3) Add integration time calculator per source
          4) Do sensitivity calculator (maybe needs to be separate function/class that inhereits this)
    """
    def __init__(self,inputMs):
        """
        Instantiation of this class calls this, i.e., vm = aU.ValueMapping('myInputMS.ms').  The dataset name is the only allowed
        input.  It generates the mappings are part of this instantiation.
        """
        self.inputMs = inputMs
        self.setInputMs(self.inputMs)

    def getInputMs(self):
        """
        Input: None
        Output: The active measurement set in the class as a string
        Responsible: S. Corder
        Purpose: Return the name of the active dataset
        """
        return self.inputMs

    def setInputMs(self,inputMs):
        """
        Input: New measurement set that you wish to become the active one, as a string and activate that change to all other parameters
        Output: None
        Responsible: S. Corder
        Purpose: This changes the active dataset and remakes the relevant mappings.  The order of the functions is very
                 important.
        """
        self.inputMs = inputMs
        self.doScanTimeMapping()
        self.doFieldsAndSources()   ;  self.doStatesAndIntents()
        self.doAntennasAndNames()
        self.doScanStateMapping()   ;  self.doFieldTimeMapping() ;
        self.doAntennaTimeMapping() ;  self.doAntennaStateMapping()
        self.doDataDescId()         ; self.doSpwAndDataDescId()
        self.doPolarizations()
        self.doSpwAndFrequency()    
        self.doSpwScanMapping()     ;  self.doSpwFieldMapping()
        self.doSpwIntentMapping()

    def doSpwAndFrequency(self,ignoreWVR=True) :
        """
        Input: None
        Output: None
        Responsible: S. Corder
        Purpose: Creates a dictionary (spwInfo) of spectral windows, with keys of the spectral window number.
                 For each spectral window, another dictionary is formed that has keys of bandwidth, sideband,
                 chanFreqs, chanWidth, numChannels, and meanFreq.
        """
        mytb = createCasaTool(tbtool)
        self.spwInfo = {}
        mytb.open("%s/SPECTRAL_WINDOW" % self.inputMs)
        specWinIds = range(mytb.nrows())
        junk = []
        for i in specWinIds :
            self.spwInfo[i] = {}
            self.spwInfo[i]["bandwidth"] = mytb.getcell("TOTAL_BANDWIDTH",i)
            self.spwInfo[i]["chanFreqs"] = mytb.getcell("CHAN_FREQ",i)
            self.spwInfo[i]["chanWidth"] = mytb.getcell("CHAN_WIDTH",i)[0]
            self.spwInfo[i]["edgeChannels"] = [min(self.spwInfo[i]["chanFreqs"]),max(self.spwInfo[i]["chanFreqs"])]
            netSideband  = mytb.getcell("NET_SIDEBAND",i)
            if netSideband == 2 : self.spwInfo[i]["sideband"] = 1
            else : self.spwInfo[i]["sideband"] = -1
            self.spwInfo[i]["meanFreq"]  = self.spwInfo[i]["chanFreqs"].mean()
            self.spwInfo[i]["numChannels"] = self.spwInfo[i]["chanFreqs"].shape[0]
            if ((ignoreWVR) and (self.spwInfo[i]['numChannels'] == 4)) :
                junk.append(i)
                self.spwInfo.pop(i)
        mytb.close()
        if ignoreWVR:
            if (len(junk) > 0):
                print("Ignoring spectral window %s because it is WVR related" % junk)

    def doSpwAndDataDescId(self) :
        mytb = createCasaTool(tbtool)
        mytb.open("%s/DATA_DESCRIPTION" % self.inputMs)
        self.spwForDataDescId = mytb.getcol('SPECTRAL_WINDOW_ID')
        mytb.close()

    def doSpwFieldMapping(self) :
        mytb = createCasaTool(tbtool)
        mytb.open("%s" % self.inputMs)
        self.fieldsForSpw = {}
        for i in self.spwForDataDescId :
            spw = self.spwForDataDescId[i]
            indices = np.where(self.dataDescId == i)
            self.fieldsForSpw[spw] = np.unique(self.fields[indices])
        mytb.close()
        return

    def doDataDescId(self) :
        mytb = createCasaTool(tbtool)
        mytb.open("%s" % self.inputMs)
        self.dataDescId = mytb.getcol('DATA_DESC_ID')
        mytb.close()

    def doPolarizations(self) :
        # Determine the number of polarizations for the first OBSERVE_TARGET intent.
        # Used by plotbandpass for BPOLY plots since the number of pols cannot be inferred
        # correctly from the caltable alone.  You cannot not simply use the first row, because
        # it may be a pointing scan which may have different number of polarizations than what
        # the TARGET and BANDPASS calibrator will have.
        # -- T. Hunter
        myscan = -1
        starttime = timeUtilities.time()
        for s in self.uniqueScans:
            intents = self.getIntentsForScan(s)
            for i in intents:
                if (i.find('OBSERVE_TARGET')>=0):
                    myscan = s
#                    print("First OBSERVE_TARGET scan = ", myscan)
                    break
            if (myscan >= 0):
                break
        if (myscan == -1):
            # if there is no OBSERVE_TARGET, then just use the first scan
            myscan = 0
        self.getDataColumnNames()
        mytb = createCasaTool(tbtool)
        mytb.open("%s" % self.inputMs)
        if (myscan == 0):
            # assume the first row in the table is for the first scan, to save time
            self.nPolarizations = np.shape(mytb.getcell(self.dataColumnName,0))[0]
        else:
            scans = mytb.getcol('SCAN_NUMBER')
            self.nPolarizations = 0
            for s in range(len(scans)):
                if (scans[s]==myscan):
                    self.nPolarizations = np.shape(mytb.getcell(self.dataColumnName,s))[0]
                    break
        mytb.close()
        donetime = timeUtilities.time()
#        print "doPolarizations took %.1f sec" % (donetime-starttime)

    def getDataColumnNames(self):
        mytb = createCasaTool(tbtool)
        mytb.open(self.inputMs)
        colnames = mytb.colnames()
        self.correctedDataColumnName = ''
        self.modelDataColumnName = ''
        if 'FLOAT_DATA' in colnames:
            self.dataColumnName = 'FLOAT_DATA'
            self.correctedDataColumnName = 'FLOAT_DATA'
        elif 'DATA' in colnames:
            self.dataColumnName = 'DATA'
        if 'CORRECTED_DATA' in colnames:
            self.correctedDataColumnName = 'CORRECTED_DATA'
        if 'MODEL_DATA' in colnames:
            self.modelDataColumnName = 'MODEL_DATA'
        mytb.close()
        return

    def doSpwScanMapping(self) :
        mytb = createCasaTool(tbtool)
        mytb.open("%s" % self.inputMs)
        self.scansForSpw = {}
        for i in self.spwForDataDescId :
            spw = self.spwForDataDescId[i]
            indices = np.where(self.dataDescId == i)
            self.scansForSpw[spw] = np.unique(self.scans[indices])
        mytb.close()
        return
    
    def doSpwIntentMapping(self) :
        mytb = createCasaTool(tbtool)
        mytb.open("%s" % self.inputMs)
        self.intentsForSpw = {}
        for i in self.spwForDataDescId :
            spw = self.spwForDataDescId[i]
            indices = np.where(self.dataDescId == i)
            statesForSpw = np.unique(self.states[indices])
            _intent = []
            for i in statesForSpw :
                __intent = []
# The 'if' statement is needed to support telescopes w/o intents. -T. Hunter
                if (len(self.intentsForStates) > 0):
                  for j in self.intentsForStates[i] :
#                    __map = j.split('#')[0]
                    __map = j
                    __intent.append(__map)
                  _intent += __intent
            self.intentsForSpw[spw] = np.unique(np.array(_intent))
        mytb.close()


    def getSpwsForIntent(self,intent) :
        spwsForIntent = []
        for i in list(self.intentsForSpw.keys()) :
            if (intent in self.intentsForSpw[i]) : spwsForIntent.append(i)
        return spwsForIntent

    def getIntentsForSpw(self,spw) :
        return self.intentsForSpw[spw]

    def getSpwsForField(self,field) :
        if not str(field).isdigit() : field = self.getFieldIdsForFieldName(field)
        spwsForField = []
        for i in list(self.fieldsForSpw.keys()) :
            if (field in self.fieldsForSpw[i]) : spwsForField.append(i)
        return spwsForField

    def getFieldsForSpw(self,spw,returnName = True) :
        if returnName :
            return self.getFieldNamesForFieldId(np.unique(np.array(self.fieldsForSpw[spw])))
        else :
            return np.unique(np.array(self.fieldsForSpw[spw]))

    def getSpwsForScan(self,scan):
        spwsForScan = []
        for i in list(self.scansForSpw.keys()) :
            if (scan in self.scansForSpw[i]) : spwsForScan.append(i)
        return spwsForScan

    def getScansForSpw(self,spw) :
        return self.scansForSpw[spw]

    def getAntennaNamesForAntennaId(self,id):
        """
        Input: Antenna id as an integer or string
        Output: Antenna name as a string
        Responsible: S. Corder
        Purpose: Allows translation between antenna id and antenna name.
        """

        return self.antennaNamesForAntennaIds[int(id)]

    def getAntennaIdsForAntennaName(self,antennaName):
        """
        Input: Antenna names as a string
        Output: Antenna index as an integer.
        Responsible: S. Corder
        Purpose: This allows translation between antenna name and antenna id
        """

        return np.where(self.antennaNamesForAntennaIds == antennaName)[0][0]

    def doStatesAndIntents(self):
        """
        Input: None
        Output: None
        Responsible: S. Corder
        Purpose: This function defines two attributes, uniqueStates, which is a python list of the different intents present in the dataset,
                 and intentsForStates is another python list which give the intents for state id as a nested list.  The first index of
                 the intentsForStates is the state id.  If you choose a state id, then the result is a list of intents for that state.
        """
        mytb = createCasaTool(tbtool)
        mytb.open("%s/STATE" % self.inputMs)
        intents = mytb.getcol("OBS_MODE")
        mytb.close()
        _intents = []
        for i in intents : _intents.append(i.split(','))
        self.intentsForStates = _intents
        self.uniqueIntents = []
        for i in self.intentsForStates : self.uniqueIntents.extend(i)
        self.uniqueIntents = np.unique(np.array(self.uniqueIntents))

    def doFieldsAndSources(self):
        """
        Input: None
        Output: None
        Responsible: S. Corder
        Purpose: This function defines two attributes, uniqueField and fieldNamesForFieldIds.  For the time being these are identical.
                 fieldNamesForFieldIds is simply a numpy array where the index is the field id and the value is the name of the field source.
        """
        mytb = createCasaTool(tbtool)
        mytb.open("%s/FIELD" % self.inputMs)
        self.fieldNamesForFieldIds = mytb.getcol('NAME')
#        print '%d field names = '%len(self.fieldNamesForFieldIds), self.fieldNamesForFieldIds
        self.uniqueFields = self.fieldNamesForFieldIds
        mytb.close()

    def doAntennasAndNames(self) :
        """
        Input: None
        Output: None
        Responsible: S. Corder
        Purpose: This function defines two attributes, uniqueAntennas (which is a little excessive) and antennaNamesForAntennaIds.
                 antennaNamesForAntennaIds is a numpy array and has indices that are the antenna ids and values that are the antenna names.
        """
        mytb = createCasaTool(tbtool)
        mytb.open("%s/ANTENNA" % self.inputMs)
        self.antennaNamesForAntennaIds = mytb.getcol('NAME')
        self.uniqueAntennas = np.unique(self.antennaNamesForAntennaIds)
        self.numAntennas = len(self.uniqueAntennas)
        mytb.close()

    def doScanStateMapping(self):
        """
        Input: None
        Output: None
        Responsible: S. Corder
        Purpose: This function defines an attribute, statesForScans, that is the mapping between states and scan numbers.  It is
                 a python dictionary that has keys of the scan number and values, in a list, of the states used in that scan.
        """
        mytb = createCasaTool(tbtool)
        mytb.open("%s" % self.inputMs)
        self.states = mytb.getcol("STATE_ID")
        mytb.close()
        self.statesForScans = {}
        for i in self.uniqueScans :
            indices = np.where(self.scans == i)
            self.statesForScans[i] = np.unique(self.states[indices])

    def doScanTimeMapping(self):
        """
        Input: None
        Output: None
        Responsible: S. Corder
        Purpose: This function defines four attributes, scans, time, uniqueScans and scansForTiems.  scans and time are simply
                 the scan number and time table from the main data table as python arrays.  uniqueScans is a numpy array of the independent scans
                 in the table.  scansForTimes is a python dictionary with keys as scans and values as times in which data was taken
                 for that scan.
        """
        mytb = createCasaTool(tbtool)
        mytb.open(self.inputMs)
        self.scans = mytb.getcol('SCAN_NUMBER')
        self.time = mytb.getcol('TIME')
        mytb.close()
        self.uniqueScans = np.unique(self.scans)
        self.scansForTimes = {}
        for i in self.uniqueScans :
            indices = np.where(self.scans == i)
            self.scansForTimes[i] = self.time[indices]

    def doFieldTimeMapping(self):
        """
        Input: None
        Output: None
        Responsible: S. Corder
        Purpose: This function defines two attributes, fields, a numpy array, and fieldsForTimes, a dictionary with keys of field name.
                 The fields is just the field id from the data table.  The values of fieldsForTimes are the times during which data was
                 collected for that field source.
        """
        mytb = createCasaTool(tbtool)        
        mytb.open(self.inputMs)
        self.fields = mytb.getcol('FIELD_ID')
        mytb.close()
        self.fieldsForTimes = {}
        for i in range(len(self.fieldNamesForFieldIds)) :
            indices = np.where(self.fields == i)
            self.fieldsForTimes[self.fieldNamesForFieldIds[i]] = self.time[indices]

    def doAntennaTimeMapping(self):
        """
        Input: None
        Output: None
        Responsible: S. Corder
        Purpose: This function defines three attributes. antenna1 and antenna2 are numpy arrays containing the antenna1 and antenna2 columns
                 from the data table.  antennasForTimes defines the times over which data was collected for that antenna.  It is a python
                 dictionary with keys of baseline (using antenna names) and values of numpy array of times.
        """
        mytb = createCasaTool(tbtool)
        mytb.open(self.inputMs)
        self.antennas1 = mytb.getcol('ANTENNA1')
        self.antennas2 = mytb.getcol('ANTENNA2')
        mytb.close()
        self.antennasForTimes = {}
        for i in range(len(self.uniqueAntennas)) :
            for j in range(len(self.uniqueAntennas)) :
                if i <= j :
                    antennaKey = "%s-%s" % (str(self.uniqueAntennas[j]),str(self.uniqueAntennas[i]))
                    indices = np.where((self.antennas1 == list(self.antennaNamesForAntennaIds).index(self.uniqueAntennas[i])) *
                                       (self.antennas2 == list(self.antennaNamesForAntennaIds).index(self.uniqueAntennas[j])))
                    self.antennasForTimes[antennaKey] = self.time[indices]

    def doAntennaStateMapping(self):
        """
        Input: None
        Output: None
        Responsible: S. Corder
        Purpose: This function defines one attribute, antennasForStates, a python dictionary.  The keys are baselines (using
                 antenna names) and values are staes used for that baseline.  Usually the autocorrelations are most useful.
        """

        self.antennasForStates = {}
        for i in range(len(self.uniqueAntennas)) :
            for j in range(len(self.uniqueAntennas)) :
                if i <= j :
                    antennaKey = "%s-%s" % (str(self.uniqueAntennas[j]),str(self.uniqueAntennas[i]))
                    indices = np.where((self.antennas1 == list(self.antennaNamesForAntennaIds).index(self.uniqueAntennas[i])) *
                                       (self.antennas2 == list(self.antennaNamesForAntennaIds).index(self.uniqueAntennas[j])))
                    self.antennasForStates[antennaKey] = self.states[indices]

    
    def getScansForTime(self,time,fudge=0.0):
        """
        Input: Time stamp in CASA native units as a string or float
        Output: Scan number associated with that time stamp.
        Responsible: S. Corder
        Purpose: This function returns the scan number for a specific timestamp.  It allows translation between time and scan.
        """
        for i in list(self.scansForTimes.keys()) :
            if ((float(time) >= self.scansForTimes[i][0]-fudge) and (float(time) <= self.scansForTimes[i][-1]+fudge)) :
                return i

    def getTimesForScans(self,scans):
        """
        Input: Scan number as an integer or string or list of integers
        Output: A list of time ranges over which data exists for that scan (or those scans), each as a numpy array.
        Responsible: S. Corder, copied from getTimesForScan and modified by T. Hunter
        Purpose: Return the times associated with a given timestamp.  This allows translation between scan and time.
        """
        times = []
        if (type(scans) == int or type(scans) == np.int32):
            scans = [scans]
        for scan in scans:
            times.append(self.scansForTimes[int(scan)])
        return (times)

    def getTimesForScan(self,scan):
        """
        Input: Scan number as an integer or string
        Output: Time range over which data exists for that scan as a numpy array.
        Responsible: S. Corder
        Purpose: Return the times associated with a given timestamp.  This allows translation between scan and time.
        """
        return self.scansForTimes[int(scan)]

    def getScansForState(self,state):
        """
        Input: State id as an integer or string
        Output: The scan numbers, as a list, that use that specific state.
        Responsible: S. Corder
        Purpose: Return the scans that used a specific state.  This allos translation between state and scan.
        """

        scansForState = []
        for i in self.uniqueScans :
            if int(state) in self.statesForScans[i] : scansForState.append(i)
        return scansForState

    def getStatesForScan(self,scan):
        """
        Input: Scan number as a string or integer
        Output: States used during that scan.
        Responsible: S. Corder
        Purpose: Returns the states used during a given scan.  This allows translation between scan and state
        """

        return self.statesForScans[int(scan)]

    def getIntentsForScan(self,scan) :
        """
        Input: Scan as an integer or string.
        Output: Intent as a an array of strings with the names of the intents as values.
        Responsible: S. Corder
        Purpose: This returns the intents used in a specific scan allowing translation between scan and intent.
        """

        intentsForScan = []
        for i in range(len(self.intentsForStates)) :
            subIntents = self.intentsForStates[i]
            if int(scan) in self.getScansForState(i) : intentsForScan.extend(subIntents)
        return np.unique(intentsForScan)
            
    def getScansForIntent(self,intent) :
        """
        Input: Intent (as a string)
        Output: A numpy array of scans using the input intent.
        Responsible: S. Corder
        Purpose: This returns the scans using a specific intent.  This allows flagging based on intent and translation
                 between intent and scan.
        """

        scansForIntent = []
        for i in range(len(self.states)) :
            if intent in self.intentsForStates[self.states[i]] :
                scansForIntent.extend(self.getScansForState(self.states[i]))
        return np.unique(scansForIntent)
        
    def getScansForFieldID(self,field):
        """
        Input: Field, as an id.
        Output: Scans using that field
        Responsible: T. Hunter
        Purpose: This takes a field ID and tells you what scans it was used in.  It was
                 created to avoid a strange behavior of getScansForField for integer inputs.
        """
        indices = np.where(self.fields == field)
        return np.unique(self.scans[indices])
        
    def getScansForField(self,field):
        """
        Input: Field, as a name or id.
        Output: Scans using that field
        Responsible: S. Corder
        Purpose: This takes a field source and tells you what scans it was used in.
        """

        if not str(field).isdigit() : field = self.getFieldIdsForFieldName(field)
        indices = np.where(self.fields == field)
        return np.unique(self.scans[indices])

    def getFieldsForScans(self,scans,returnName=True):
        slist = []
        for scan in scans:
            slist.append(self.getFieldsForScan(scan))
        return([item for sublist in slist for item in sublist])
    
    def getFieldsForScan(self,scan,returnName=True):
        """
        Input: Scan as an integer or string
        Output: Field ids observed during that scan.
        Responsible: S. Corder
        Purpose: This takes a scan number and returns a field observed during that scan.  This allows translation between
                 scan and field.
        """

        indices = np.where(self.scans == int(scan))
        if returnName : return self.getFieldNamesForFieldId(np.unique(self.fields[indices]))
        else : return np.unique(self.fields[indices])

    def getFieldsForIntent(self,intent,returnName=True):
        """
        Input: intent as a string
        Output: field id as integer or array of names
        Responsible: S. Corder
        Purpose: This retrieves all of the fields that have been assigned a given intent during an observation.  
        """        
        _fields = []
        scans = self.getScansForIntent(intent)
        for i in scans :
            _field = self.getFieldsForScan(i)
            if _field not in _fields : _fields.append(_field)
        if returnName :
            return _fields
        else :
            return self.getFieldIdsForFieldName(_fields)
        
    def getFieldIdsForFieldName(self,sourceName):
        """
        Input: source name as string
        Output: field id as integer (actually it is returning the source id-Todd)
        Responsible: S. Corder
        Purpose: This translates between source/field name and field id.
        """
# The following fails because the case varies when .title() is applied: QSO vs. Qso, and TW Hya vs Tw Hya
#        return np.where(upper == sourceName.title())[0][0]
        if (type(sourceName) == list):
            sourceName = sourceName[0]
#        print "looking for %s in " % (sourceName), self.fieldNamesForFieldIds
        return np.where(self.fieldNamesForFieldIds == sourceName)[0]
        
    def getFieldNamesForFieldId(self,sourceId):
        """
        Input: field id (as string or integer)
        Output: field name
        Responsible: S. Corder
        Purpose: This translates between field id and field/source name.
        """
        if (type(sourceId) == int or type(sourceId) == np.int32 or type(sourceId) == np.int64 or type(sourceId) == str):
            if (len(self.fieldNamesForFieldIds) > int(sourceId)):
                # prevent "index out of bounds" error if field is not present 
                return self.fieldNamesForFieldIds[int(sourceId)]
            else:
                return (None)
        else:
            # Todd added this check which was necessary for the Antennae Band 7 mosaic
            return [self.fieldNamesForFieldIds[s] for s in sourceId]

    def getFieldsForTime(self,time,returnName=True):
        """
        Input: Time in casa native units, returnName (boolean).  If returnName is true, the name is returned, else the id is returned.
               Default is returnName=True
        Output: Field name or id (depending on value of returnName).
        Responsible: S. Corder
        Purpose: Allows the field id/name to be returned for a specific observation time.
        """

        for i in list(self.fieldsForTimes.keys()) :
            if (time in self.fieldsForTimes[i]) :
                if returnName : return i
                else : return self.getFieldNamesForFieldId(i)

    def getTimesForField(self,field):
        """
        Input: Field name or id (as a string or integer)
        Output: Times as a numpy array over which that field was observed in casa native units.
        Responsible: S. Corder
        Purpose: This allows you to determine the data time stamps for observations of a specific field source.
        """

        if str(field).isdigit() : field = self.fieldNamesForFieldIds[int(field)]
        return self.fieldsForTimes[field]


class TsysExplorer:
    """
    Used only by plotWeeklyTrx().
    Put something in about updating the Tsys on TDM observations in real time.
    """
    def __init__(self,inputMs,antenna=None,spwID=None,autoSubtableQuery=True,queryString='',cross_auto_all='all'):
        if autoSubtableQuery==False and queryString=='' : return 'Must either automatically generate the (autoSubtableQuery=True) or provide a subtable query string (queryString)'
        self.inputMs = inputMs
        self.valueMapping = ValueMapping(inputMs)
        if antenna is None :
            antenna = self.valueMapping.getAntennaNamesForAntennaId(0)
        self.antenna = antenna
        self.checkAntenna()
        if spwID is None :
            spwID = getChanAverSpwIDBaseBand0(inputMs)
        self.spwID = spwID
        self.elevation = None
        self.time      = None
 #       self.tcal      = None
        self.trx       = None
        self.tsky      = None
        self.tsys      = None
        self.freq      = None
        self.elevTime  = None
        self.scan      = []
        self.field     = []
        self.autoSubtableQuery = autoSubtableQuery
        self.queryString = queryString
        if self.autoSubtableQuery == True : 
            self.makeAutoSubtable()
        else : self.makeSubtable()

    def getScanAndField(self):
        for i in self.time : 
            self.scan.append(self.valueMapping.getScansForTime(i))
            self.field.append(self.valueMapping.getFieldsForTime(i))

    def restrictTimeField(self):
        return

    def getElevation(self):
        mytb = tbtool()
        mytb.open("%s/POINTING" % self.inputMs)
        subtable = mytb.query("ANTENNA_ID == %s" % self.antenna)
        mytb.close()        
        self.elevation = subtable.getcol("DIRECTION")
        self.elevTime  = subtable.getcol("TIME")
        elev = []
        for i in self.time :
            diffTime = abs(self.elevTime - i)
            indy = np.where(diffTime == diffTime.min())
            elev.append(self.elevation[...,indy[0][0]]*180.0/math.pi)
        elev = np.array(elev)
#        print elev.shape,elev
        self.elevation = elev[:,1]

    def getFreq(self) :
        mytb = tbtool()
        mytb.open("%s/SPECTRAL_WINDOW" % self.inputMs)
        rows = mytb.selectrows(self.spwID)
        freqs = rows.getcol("CHAN_FREQ")
        mytb.close()
        self.freq = freqs
        
    def getTsysData(self) :
        self.time = self.subtable.getcol("TIME")-self.subtable.getcol("INTERVAL")/2.0
#        self.tcal = self.subtable.getcol("TCAL_SPECTRUM")
        self.trx  = self.subtable.getcol("TRX_SPECTRUM")
        self.tsky = self.subtable.getcol("TSKY_SPECTRUM")
        self.tsys = self.subtable.getcol("TSYS_SPECTRUM")

    def getAntenna(self) : return self.antenna

    def setAntenna(self,antenna) :
        self.antenna = antenna
        self.checkAntenna()
        if self.autoSubtableQuery == True : 
            self.makeAutoSubtable()
        else : self.makeSubtable()

    def getInputMs(self) : return self.inputMs

    def setInputMs(self,inputMs) :
        self.inputMs = inputMs
        if self.autoSubtableQuery == True : 
            self.makeAutoSubtable()
        else : self.makeSubtable()
        self.ValueMapping.setInputMs(inputMs)

    def checkAntenna(self) :
        if self.antenna != None : 
            self.antenna = str(self.antenna)
            if not self.antenna.isdigit() : self.antenna = getAntennaIndex(self.inputMs,self.antenna)

    def getSpwID(self) : return self.spwID

    def setSpwID(self,spwID) :
        self.spwID = spwID
        if self.autoSubtableQuery == True : 
            self.makeAutoSubtable()
        else : self.makeSubtable()

    def makeSubtableQuery(self) :
        self.parameterList = []
        queryString = ''
        if self.antenna != None : self.parameterList.append('ANTENNA_ID == %s' % self.antenna)
#        if self.field <> None    : self.parameterList.append('FIELD_ID == %s' % self.field)
        if self.spwID != None    : self.parameterList.append('SPECTRAL_WINDOW_ID == %s' % self.spwID)
#        if self.state <> None    : self.parameterList.append('STATE_ID == %s' % self.state)
#        if self.scan <> None     : self.parameterList.append('SCAN_NUMBER == %s' % self.scan)
        for i in self.parameterList : queryString = self.appendQuery(queryString,i)
        self.queryString = queryString

    def appendQuery(self,queryString,additive) :
        if queryString == '' :
            if additive == '' : return queryString
            else : return additive
        else :
            if additive == '' : return queryString
            else : 
                queryString = queryString + ' && ' + additive
                return queryString

    def makeAutoSubtable(self) :
        self.checkAntenna()
        self.makeSubtableQuery()
        mytb = tbtool()
        mytb.open("%s/SYSCAL" % self.inputMs)
        self.subtable = mytb.query(self.queryString)
        mytb.close()
        self.getTsysData()
        self.getFreq()
        #self.getElevation()
        self.getScanAndField()

    def makeSubtable(self) :
        mytb = tbtool()
        mytb.open("%s/SYSCAL" % self.inputMs)
        self.subtable = mytb.query(self.queryString)
        mytb.close()
        self.getTsysData()
        self.getFreq()
        #self.getElevation()
        self.getScanAndField()

    def setAutoSubtableQuery(self,autoSubtableQuery) :
        self.autoSubtableQuery = autoSubtableQuery

    def getAutoSubtableQuery(self) : return self.autoSubtableQuery
        
def plotWeeklyTrx(uid,showPlots=False) :
    msFile=uid+'.ms'
    vm = ValueMapping(msFile)
    tsys = TsysExplorer(msFile,antenna=0,spwID=1)
    tsysSpw = vm.getSpwsForIntent('CALIBRATE_ATMOSPHERE#OFF_SOURCE')[1:]
    figureIndex = 0
    band = getBand(tsys.freq.mean())
    myqa = createCasaTool(qatool)
    ObsDate=myqa.time({'value' : vm.time[0],'unit': 's'}, form=['ymd'])
    fnDate=(''.join([str(l) for l in (ObsDate.split(':')[0].split('/'))]))
    fnDate2=(''.join([str(l) for l in ((''.join([str(l2) for l2 in (ObsDate.split('/'))])).split(':'))]))
    fDir=('/data/RADIO/TRX/AOS/%.8s' %(fnDate))
    if not os.path.exists(fDir): os.makedirs(fDir)
    F = open(fDir+'/TrxRB%s_%s.txt' %(band,fnDate2),'w')
    print(('%s %s ALMARB_%s' % (msFile, ObsDate, band)), file=F)
    print(('Mean Freq. Mean Trx  Std. Dev.'), file=F)
    print(('GHz.       K         K'), file=F)
    badAnts = []
    for antenna in vm.uniqueAntennas :
        figureIndex +=1 
        if showPlots: pb.figure(figureIndex)
        meanVals = []
        meanFreq = []
        stdVals  = []
        for spw in tsysSpw :
            if vm.spwInfo[spw]['numChannels'] == 128 :
                tsys.setSpwID(int(spw))
                tsys.setAntenna(antenna)
                meanFreq.append(tsys.freq.mean())
                meanVals+=(list(tsys.trx[:,3:125].mean(1).transpose()[0]))
                stdVals+=(list(tsys.trx[:,3:125].std(1).transpose()[0]))
                if showPlots:
                    pb.plot(tsys.freq[3:125]/1e9,tsys.trx[0,3:125],'g.')
                    pb.plot(tsys.freq[3:125]/1e9,tsys.trx[1,3:125],'b.')
        print(antenna, file=F)
        print(('\n'.join(['%6.2f'%round(float(l1/1e9),2)+"     "+'%6.2f'%round(float(l2),2)+"     "+'%6.2f'%round(float(l3),2) for (l1,l2,l3) in zip(meanFreq,meanVals,stdVals)])), file=F)
#        specVal100 = {3: 60, 6: 136, 7: 219, 9: 261} #This is specification over 100% of the bands
        specVal80 = {3: 45, 6: 83, 7: 147, 9: 175} #This is specification over 80% of the bands

        if band == 3 : spec = specVal80[3]
        elif band == 6 : spec = specVal80[6]
        elif band == 7 : spec = specVal80[7]
        elif band == 9 : spec = specVal80[9]
        else: return 'Not a valid weekly Trx testing Frequency'
        for i in meanVals :
            if i >= spec :
                badAnts.append(antenna)
        if showPlots:
            pb.plot((100*np.arange(len(meanFreq))),(spec*np.ones(len(meanFreq))),'r-')
            pb.xlim(meanFreq[0]/1e9-10,meanFreq[len(meanFreq)-1]/1e9+10)
            pb.xlabel('Frequency (GHz)')
            pb.ylabel('Trx (K)')
            pb.suptitle('%s %s %s' % (antenna,myqa.time({'value' : vm.time[0],'unit': 's'},form=['ymd']),msFile), size='14')
            pb.title('Green--Pol0   Blue--Pol1   Red--Specification(80%)', size='12')
            pb.savefig(fDir+'/TrxRB%s%s_%s.png' % (band,antenna,fnDate2))
    print("Problematic Antennas: %s" % str(badAnts), file=F)
    F.close()    
    myqa.done()
    print("Antennas: %s seem to have problems." % str(badAnts))
    raw_input("Hit Return to quit: ")
#    pb.close('all')
#Keep on having problem with close all
    fig_numbers = [x.num
               for x in matplotlib._pylab_helpers.Gcf.get_all_fig_managers()]
    for n in fig_numbers: pb.close(n)

class CalTableExplorer:
    """Stuff: Only works for antenna based solutions"""
    """Stuff to add: start and end channels (started, but need to make it consaistent, good metrics for differences"""
    """plotting routines, interpolation? In freq and/or time?  Basics are done"""
    """I need to make the residual functions make sense, I think the names are screwy. """
    
    def __init__(self,inputTable,antenna=None,spwID=None,feed=None,scan=None,state=None,field=None,autoSubtableQuery=True,queryString='',startChan=None,endChan=None):
        self.inputTable   = inputTable
        self.inputMs      = self.getMS_NAME(self.inputTable)
        self.ValueMapping = ValueMapping(self.inputMs)
        self.antenna     = antenna
        self.checkAntenna()
        self.startChan   = startChan
        self.endChan     = endChan
        self.spwID        = spwID
        self.feed         = feed
        self.scan         = scan
        self.state        = state
        self.field        = field
        self.checkField()
        self.getCalDescSpwIDMapping()
        if spwID is None : self.spwID = self.calDescSpwIDMapping[0]
        self.autoSubtableQuery = autoSubtableQuery
        self.queryString  = queryString
        if self.autoSubtableQuery == True :
            self.makeAutoSubtable()
        else : self.makeSubtable()

    def getFreq(self) :
        mytb = tbtool()
        mytb.open("%s/SPECTRAL_WINDOW" % self.inputMs)
        rows = mytb.selectrows(self.spwID)
        freqs = rows.getcol("CHAN_FREQ")
        mytb.close()
        self.freq = freqs

    def getTimeAndInterval(self) :
        self.time     = self.subtable.getcol("TIME")
        self.mjd      = mjdSecondsVectorToMJD(self.time)
        self.ut       = mjdVectorToUTHours(self.mjd)
        self.interval = self.subtable.getcol("INTERVAL")

    def getFit(self) :
        self.gain     = self.subtable.getcol("GAIN")
        self.solOkay  = self.subtable.getcol("SOLUTION_OK")
        self.flags    = self.subtable.getcol("FLAG")
        self.fit      = self.subtable.getcol("FIT")
        self.snr      = self.subtable.getcol("SNR")
        self.real     = np.real(self.gain)
        self.imag     = np.imag(self.gain)
        self.phase = np.arctan2(self.imag,self.real)
        self.amp   = abs(self.gain)

    def plotFit(self,pol,xtype=''):
        date = (mjdSecondsToMJDandUT(self.time[0])[1]).split()[0]
        if (xtype=='ut'):
            pb.plot(self.ut,self.gain[pol,:,:][0],'b.')
            pb.xlabel('UT (hours)')
        elif (xtype=='mjd'):
            pb.plot(self.mjd,self.gain[pol,:,:][0],'b.')
            pb.xlabel('MJD days')
        else:
            pb.plot(self.time,self.gain[pol,:,:][0],'b.')
            pb.xlabel('MJD seconds')
        pb.ylabel('Gain')
        pb.title(date)
        
    def timeAverageSolutions(self) :
        self.tavgGain  = self.gain.mean(-1)
        self.tavgReal  = self.tavgGain.real
        self.tavgImag  = self.tavgGain.imag
        self.tavgPhase = np.arctan2(self.tavgImag,self.tavgReal)
        self.tavgAmp   = abs(self.tavgGain)
        
    def freqAverageSolutions(self) :
        self.favgGain  = self.gain[...,self.startChan:self.endChan,...].mean(1)
        self.favgReal  = self.favgGain.real
        self.favgImag  = self.favgGain.imag
        self.favgPhase = np.arctan2(self.favgImag,self.favgReal)
        self.favgAmp   = abs(self.favgGain)

    def generateSpectralResiduals(self) :
        self.fresidGain = self.gain
        for i in range(self.gain.shape[-1]) :
            self.fresidGain[:,:,i] = self.fresidGain[:,:,i]-self.tavgGain
        self.fresidGain = self.fresidGain
        self.fresidReal = self.fresidGain.real
        self.fresidImag = self.fresidGain.imag
        self.fresidPhase = np.arctan2(self.fresidImag,self.fresidReal)
        self.fresidAmp   = abs(self.fresidGain)

    def unwrapPhase(self,simple=True) :
        from math import pi
        phaseShape = self.phase.shape
        for i in range(phaseShape[2]-1) :
            diff = self.phase[:,:,i]-self.phase[:,:,i+1]
            _diffg = (diff > 1.*np.pi)*2*np.pi
            _diffl = (diff < -1.*np.pi)*2*np.pi
            self.phase[:,:,i+1] = self.phase[:,:,i+1]+_diffg-_diffl

    def generateTimeResiduals(self) :
        self.tresidGain = self.gain
        for i in range(self.gain.shape[1]) :
            self.tresidGain[:,i,:] = self.tresidGain[:,i,:]-self.favgGain
        self.tresidGain = self.tresidGain[...,self.startChan:self.endChan,...]
        self.tresidFreq = self.freq[self.startChan:self.endChan]
        self.tresidReal = self.tresidGain.real
        self.tresidImag = self.tresidGain.imag
        self.tresidPhase = np.arctan2(self.tresidImag,self.tresidReal)
        self.tresidAmp   = abs(self.tresidGain)

    def getMS_NAME(self,inputTable) :
        mytb = tbtool()
        mytb.open("%s/CAL_DESC" % inputTable)
        msFiles = mytb.getcol("MS_NAME")
        mytb.close()
        return np.unique(msFiles)[0]

    def getCalDescForSpwID(self,calDesc) :
        try:
            return np.where(self.calDescSpwIDMapping[0] == self.spwID)[0][0]
        except:
            print('The identified spwID does not have a solution in this table.')
            sys.exit()
        
    def getCalDescSpwIDMapping(self) :
        mytb = tbtool()
        mytb.open("%s/CAL_DESC" % self.inputTable)
        self.calDescSpwIDMapping = mytb.getcol("SPECTRAL_WINDOW_ID")
        mytb.close()

    def setField(self,field) :
        self.field = field
        self.checkField()
        if self.autoSubtableQuery : self.makeAutoSubtable()

    def getField(self) : return self.field

    def checkField(self) :
        if self.field != None : 
            self.field = str(self.field)
            if not self.field.isdigit() : self.field = self.ValueMapping.getFieldIdsForFieldName(self.field)[0]

    def checkAntenna(self) :
        if self.antenna != None : 
            self.antenna = str(self.antenna)
            if not self.antenna.isdigit() : self.antenna = getAntennaIndex(self.inputMs,self.antenna)

    def getAntenna(self) : return self.antenna

    def setAntenna(self,antenna) :
        self.antenna = antenna
        self.checkAntenna()
        if self.autoSubtableQuery == True : 
            self.makeAutoSubtable()
        else : self.makeSubtable()

    def getInputTable(self) : return self.inputTable

    def setInputTable(self,inputMs) :
        self.inputTable = inputTable
        if self.autoSubtableQuery == True : 
            self.makeAutoSubtable()
        else : self.makeSubtable()
        self.ValueMapping.setInputMs(inputMs)

    def getSpwID(self) : return self.spwID

    def setSpwID(self,spwID) :
        self.spwID = spwID
        if self.autoSubtableQuery == True : 
            self.makeAutoSubtable()
        else : self.makeSubtable()

    def setScan(self,scan) :
        self.scan = scan
        if self.autoSubtableQueyry == True :
            self.makeAutoSubtable()
        else : self.makeSubtable()

    def setFeed(self,feed) :
        self.feed = feed
        if self.autoSubtableQueyry == True :
            self.makeAutoSubtable()
        else : self.makeSubtable()

    def setState(self,state) : 
        self.state = state
        if self.autoSubtableQueyry == True :
            self.makeAutoSubtable()
        else : self.makeSubtable()

    def getScan(self) : return self.scan

    def getFeed(self) : return self.feed

    def getState(self) : return self.state

    def makeSubtableQuery(self) :
        self.parameterList = []
        queryString = ''
        if self.antenna != None : self.parameterList.append('ANTENNA1 == %s' % self.antenna)
        if self.field != None    : self.parameterList.append('FIELD_ID == %s' % self.field)
        if self.spwID != None    : self.parameterList.append('CAL_DESC_ID == %s' % self.getCalDescForSpwID(self.spwID))
        if self.state != None    : self.parameterList.append('STATE_ID == %s' % self.state)
        if self.scan != None     : self.parameterList.append('SCAN_NUMBER == %s' % self.scan)
        if self.feed != None     : self.parameterList.append('FEED_ID == %s' % self.feed)
        for i in self.parameterList : queryString = self.appendQuery(queryString,i)
        self.queryString = queryString

    def appendQuery(self,queryString,additive) :
        if queryString == '' :
            if additive == '' : return queryString
            else : return additive
        else :
            if additive == '' : return queryString
            else : 
                queryString = queryString + ' && ' + additive
                return queryString

    def makeAutoSubtable(self) :
        self.checkAntenna()
        self.makeSubtableQuery()
        mytb = tbtool()
        mytb.open("%s" % self.inputTable)
        self.subtable = mytb.query(self.queryString)
        mytb.close()
        self.getFreq()
        self.getTimeAndInterval()
        self.getFit()
#        self.freqAverageSolutions()
#        self.timeAverageSolutions()
#        self.generateSpectralResiduals()
#        self.generateTimeResiduals()

    def makeSubtable(self) :
        mytb = tbtool()
        mytb.open("%s" % self.inputTable)
        self.subtable = mytb.query(self.queryString)
        mytb.close()
        self.getFreq()
        self.getTimeAndInterval()
        self.getFit()
        self.freqAverageSolutions()
        self.timeAverageSolutions()
        self.generateFrequencyResiduals()
        self.generateTimeResiduals()

    def setAutoSubtableQuery(self,autoSubtableQuery) :
        self.autoSubtableQuery = autoSubtableQuery

    def getAutoSubtableQuery(self) : return self.autoSubtableQuery

class ScaleGainsClass(CalTableExplorer):
    
    def __init__(self,calTable) :
        self.calTable = calTable
        CalTableExplorer.__init__(self,self.calTable)
        self.vm      = ValueMapping(self.inputMs)

    def calculateGainsScaling(self,calfieldL,calfieldH,caltableL,caltableH):

        mytb = tbtool()
        mytb.open("%s" % caltableL,nomodify=False)
        tableRow = mytb.selectrows(0,'table_junk')
        polgainshape=len(tableRow.getcol('GAIN'))
        tb.close()	
        
        ####### Setting up dictionaries for low frequency table ######

        # Initialising cal to spw mapping dictionary
        self.calSpwMapL = {}

        #open descriptor file
        mytb.open("%s/CAL_DESC" % caltableL ,nomodify=False)

        for k in range(mytb.nrows()):
            tableRow = mytb.selectrows(k,'table_junk')
            self.calSpwMapL[k] = tableRow.getcol('SPECTRAL_WINDOW_ID')[...,0]
	       
        mytb.close()
	
        #Delete file containing each table row    
        os.system('rm -rf table_junk')

        ####### Setting up dictionaries for high frequency table ######

        # Initialising cal to spw mapping dictionary
        self.calSpwMapH = {}

        #open descriptor file
        mytb = tbtool()
        mytb.open("%s/CAL_DESC" % caltableH ,nomodify=False)

        for k in range(mytb.nrows()):
            tableRow = mytb.selectrows(k,'table_junk')
            self.calSpwMapH[k] = tableRow.getcol('SPECTRAL_WINDOW_ID')[...,0]
	       
        mytb.close()
	
        #Delete file containing each table row    
        os.system('rm -rf table_junk')

        phasediff = np.zeros([self.vm.uniqueAntennas.shape[0],len(self.calSpwMapL),polgainshape]) 

        for pol in range(polgainshape):
             for calid in self.calSpwMapL:
                for antname1 in self.vm.uniqueAntennas:
           
                    ant = self.vm.getAntennaIdsForAntennaName(antname1)
                    spw = self.calSpwMapL[calid]
                     
                    print("Antenna, spw, Corr", antname1, spw, pol)

                    ###### Caltable low: read in and unwrap phases ######

                    ct = CalTableExplorer("%s" % caltableL,spwID=spw,field=calfieldL)
                    
                    ct.setAntenna(antname1)
                    
                    phaseLpol=ct.phase[pol,0,:]
                    sizeL=ct.phase[pol,0,:].shape
           
                    ct.unwrapPhase()
                    
                    timeL=ct.time

                    pLinterp=interp1d(timeL,phaseLpol,kind=1,bounds_error=False,fill_value=np.nan)
                    
                    ###### Caltable high: read in and unwrap phases ######

                    if spw+4 in list(self.calSpwMapH.values()):

                        ct = CalTableExplorer("%s" % caltableH,spwID=spw+4,field=calfieldH)
                    
                        ct.setAntenna(antname1)

                        phaseHpol=ct.phase[pol,0,:]

                        ct.unwrapPhase()
                        
                        timeH =ct.time

                        # Find interpolated values of low frequency at high data times
                        phaseLtoH= pLinterp(timeH)

                    else:
                        phaseHpol=np.zeros(sizeL)
                        phaseLtoH=np.zeros(sizeL)

                    keep = ~np.isnan(phaseHpol) & ~np.isnan(phaseLtoH)
                    
                    phasediff[ant,spw,pol] = np.mean(phaseHpol[keep]-phaseLtoH[keep])
                    
        return phasediff


    def scaleGains(self,phasediff,newTable=None) :
        
        # Define table name
        if newTable is None :
            self.newTable = '%s.scaled' % self.calTable
        else :
            self.newTable = newTable
            
        # Create copy of table, deep=True copies all tables not just data
        mytb = createCasaTool(tbtool)
        mytb.open(self.calTable,nomodify=False)
        mytb.copy(self.newTable,deep=True,valuecopy=True)
        mytb.close()
        
        # Initialising cal to spw mapping dictionary, and row holders
        self.calSpwMap = {}

	#open descriptor file
        mytb.open("%s/CAL_DESC" % self.calTable,nomodify=False)

        #Initialise row holder
        descrowvals  = {}

        for k in range(mytb.nrows()):
            tableRow = mytb.selectrows(k,'table_junk')
	    #Convert row to a dictionary
            for i in tableRow.colnames():
                descrowvals[i] = tableRow.getcol(i)[...,0]
            self.calSpwMap[k] = descrowvals['SPECTRAL_WINDOW_ID'][0]
	       
	    # Write out to new table
	    #self.reconstructRow(descrowvalsp,"%s" % self.newTable)

        mytb.close()
	
	#Delete file containing each table row    
        os.system('rm -rf table_junk')

	#Open caltable, find number of rows in table
        mytb.open("%s" % self.calTable,nomodify=False)
        numsoln=mytb.nrows()
        mytb.close()

	#Initialise row holder
        rowvals  = {}

        # Loop over rows/solutions in caltable
        for k in range(numsoln) :

            mytb.open("%s" % self.calTable,nomodify=False)

            #Read each row and write it into table_junk
            tableRow = mytb.selectrows(k,'table_junk')
            
	    #Convert row to a dictionary
            for i in tableRow.colnames():
                if 'REF_' in i : continue
                else:
                    rowvals[i] = tableRow.getcol(i)[...,0]

            mytb.close()

            #Set atribute of this class = rowvals
            self.rowvals = rowvals
                            
	    #Copy parameters for a given solution
            self.rowvalsp = rowvals.copy()
                              
            #initialise scaled phase and amp arrays         
            sphase = np.zeros(self.rowvals['GAIN'].shape)
            samp = np.zeros(self.rowvals['GAIN'].shape)

            ant = self.rowvalsp['ANTENNA1']
            spw = self.calSpwMap[int(self.rowvalsp['CAL_DESC_ID'])]

            sphase[:,:]=np.angle(self.rowvals['GAIN']) + phasediff[ant,spw][:,np.newaxis]
            samp[:,:]=np.abs(self.rowvals['GAIN'])
   
            self.rowvalsp['GAIN'] = samp * np.exp(sp.sqrt(-1.)*sphase)
       
	    # Write out result to new table
            self.reconstructRow(self.rowvalsp,"%s" % self.newTable)
             
        mytb.open("%s" % self.newTable,nomodify=False)
        mytb.removerows(range(numsoln))
        mytb.close()

        #Delete table_junk file            
        os.system("rm -rf table_junk")
        
        #Remove table locks
        os.system("rm -rf %s/table.lock" % self.calTable)
        os.system("rm -rf %s/CAL_DESC/table.lock" % self.calTable)

    def reconstructRow(self,rowVals,tableName,tableRow=None) :
        mytb = createCasaTool(tbtool)
        mytb.open(tableName,nomodify=False)
        if tableRow is None :
            mytb.addrows()
            rownum = mytb.nrows()-1
        else :
            rownum = tableRow
        for i in list(rowVals.keys()) :
            if rowVals[i].shape == () : isRealArray = False
            else : isRealArray = True
            rownum,i,rowVals[i]
            try:
                dt = mytb.coldatatype(i)
                if dt == 'boolean' :
                    if not isRealArray :
                        if not rowVals[i] :
                            mytb.putcell(i,rownum,0)
                        else :
                            mytb.putcell(i,rownum,1)
                    else :
                        mytb.putcell(i,rownum,rowVals[i])
                elif dt == 'double' :
                    if not isRealArray :
                        mytb.putcell(i,rownum,float(rowVals[i]))
                    else :
                        mytb.putcell(i,rownum,rowVals[i])
                elif dt == 'float' :
                    if not isRealArray :
                        mytb.putcell(i,rownum,float(rowVals[i]))
                    else :
                        junk = np.array(rowVals[i],dtype='float32')
                        mytb.putcell(i,rownum,junk)                    
                elif dt == 'integer' :
                    if not isRealArray :
                        mytb.putcell(i,rownum,int(rowVals[i]))
                    else :
                        mytb.putcell(i,rownum,rowVals[i])
                elif dt == 'string' :
                    if not isRealArray :
                        mytb.putcell(i,rownum,str(rowVals[i]))
                    else :
                        mytb.putcell(i,rownum,rowVals[i])
                else :
                    mytb.putcell(i,rownum,rowVals[i])
            except:
                print('skipping')
        mytb.close()
#                tb.putcell(i,rownum,rowVals[i])


class InterpolateTsys(CalTableExplorer):
    """The Flag "iKnowWhatIAmDoing" should be used with care.  It implies that the user knows very well that the
       ordering of the basebands vs frequency was proper and consistent between the tdm and fdm mode and that the
       tdm mode was listed in the OT ranging from BB1, BB2, BB3 and BB4, not some other order.  Failure to use this
       options will in some cases result in slightly poorer performance and slightly better in other cases...thus
       you should really know what you are doing before you use it.
    """
       
    def __init__(self,calTable) :
        self.calTable = calTable
        self.inputMs = self.getMS_NAME(calTable)
        #CalTableExplorer.__init__(self,self.calTable)
        self.vm      = ValueMapping(self.inputMs)

    def getMS_NAME(self,inputTable) :
        tb.open("%s/CAL_DESC" % inputTable)
        msFiles = tb.getcol("MS_NAME")
        tb.close()
        return np.unique(msFiles)[0]

    def setCalTable(self,calTable) :
        self.calTable = calTable
        CalTableExplorer.__init__(self.calTable)
        self.vm.setInputMs(self.inputMs)

    def correctBadTimes(self, force=False):
        tb.open(self.calTable,nomodify=False)
        time=tb.getcol('TIME')
        interval = tb.getcol('INTERVAL')
        if max(time) < 7.0e9 and force == False:
            return "This process appears to have been done already."
        else :
            corr_time = time-interval/2.0
            tb.putcol("TIME",corr_time)
        tb.close()
        os.system("rm -rf %s/table.lock" % self.calTable)

    def assignFieldAndScanToSolution(self, iKnowWhatImDoing=False) :
        tb.open(self.calTable,nomodify=False)
        fieldId = tb.getcol("FIELD_ID")
        scans   = tb.getcol("SCAN_NUMBER")
        times   = tb.getcol("TIME")
        linesToRemove = []
        for i in range(tb.nrows()):
            _scan = self.vm.getScansForTime(times[i],5e-6)
            if _scan is not None:
                  fieldId[i] = self.vm.getFieldsForScan(_scan,False)
                  scans[i] = _scan
            else:
                  linesToRemove.append(i)
        if iKnowWhatImDoing:
              for i in sorted(linesToRemove, reverse=True):
                  fieldId = np.delete(fieldId, i)
                  scans = np.delete(scans, i)
                  tb.removerows(i)
        tb.putcol('FIELD_ID',fieldId)
        tb.putcol("SCAN_NUMBER",scans)
        tb.close()
        os.system("rm -rf %s/table.lock" % self.calTable)
        

    def getTdmFdmSpw(self,iKnowWhatIAmDoing=True):
        """
        Input: None
        Output: None
        Responsible: S. Corder
        Purpose:
        """
        tb.open("%s/SYSCAL" % self.inputMs)
        tsysTdmSpw = sorted(dict.fromkeys(tb.getcol('SPECTRAL_WINDOW_ID')).keys())
        tb.close()
        tb.open("%s/SPECTRAL_WINDOW" % self.inputMs)
        #self.tdm = tb.query("(NUM_CHAN == 128) or (NUM_CHAN == 256) or (NUM_CHAN == 64)")
        self.tdm = tb.selectrows([int(i) for i in tsysTdmSpw])
        self.tdmSpw = self.tdm.rownumbers()
        if self.tdm.nrows() == 1: self.tdmSpw = [self.tdmSpw]
        self.tdmBBC = self.tdm.getcol('BBC_NO')
        self.fdm = tb.query("(NUM_CHAN == 7680) or (NUM_CHAN == 3840) or (NUM_CHAN == 1920) or (NUM_CHAN == 4096)")
        self.fdmSpw = self.fdm.rownumbers()
        if self.fdm.nrows() == 1: self.fdmSpw = [self.fdmSpw]
        self.fdmBBC = self.fdm.getcol('BBC_NO')
        if iKnowWhatIAmDoing : self.fdmBBC.sort()
        fdmFreqs = self.fdm.getcol('CHAN_FREQ')
        tdmFreqs = self.tdm.getcol('CHAN_FREQ')
        self.fdmFreqs = fdmFreqs
        self.tdmFreqs = tdmFreqs
        tb.close()
        self.tdmFdmMap = {}
        self.fdmTdmMap = {}
        for i in range(len(fdmFreqs[0])) :
            for j in range(len(tdmFreqs[0])) :
                delT = abs(tdmFreqs[1,j]-tdmFreqs[0,j])   # Added by S. Corder 2012/5/18
                minF = np.min(fdmFreqs[:,i])
                maxF = np.max(fdmFreqs[:,i])
                minT = np.min(tdmFreqs[:,j])
                maxT = np.max(tdmFreqs[:,j])
#                if ((minF >= minT) and (maxF <= maxT)) :  # Change requested by S. Corder 2012/5/18
                if ((minF >= (minT-0.5*delT)) and (maxF <= (maxT+0.5*delT))) :
                    if self.fdmTdmMap.has_key(int(self.fdmSpw[i])) :
                        if self.fdmBBC[i] == self.tdmBBC[j] :
                           self.fdmTdmMap[self.fdmSpw[i]] = self.tdmSpw[j]
                    else :
                        self.fdmTdmMap[self.fdmSpw[i]] = self.tdmSpw[j]
        for k,v in list(self.fdmTdmMap.items()) :
            self.tdmFdmMap[v] = self.tdmFdmMap.get(v,[])
            self.tdmFdmMap[v].append(k)
        print('# Mapping of Tsys and science spws.')
        print('# Please check that you have only one science spw per Tsys spw.')
        print('# ' + str(self.tdmFdmMap))
            
    def interpolateTsys(self,newTable=None,interpType='linear') :
        self.badRows = []
        if newTable is None :
            self.newTable = '%s.fdm' % self.calTable
        else :
            self.newTable = newTable
        tb.open(self.calTable,nomodify=False)
        tb.copy(self.newTable,deep=True,valuecopy=True)
        tb.close()
        tb.open("%s/CAL_DESC" % self.calTable,nomodify=False)
        self.numCalSol = tb.nrows()
        self.calSpwMap = {}
        tb.close()
        noSpw = []
        for k in range(self.numCalSol) :
            tb.open("%s/CAL_DESC" % self.calTable,nomodify=False)            
            tableRow = tb.selectrows(k,'table_junk')
            rowvals  = self.extractRow(tableRow)
            x = self.fdmFreqs.shape[0]
            y = rowvals['CHAN_WIDTH'].shape[0]
            y1 = rowvals['CHAN_RANGE'].shape[0]
            self.calSpwMap[k] = rowvals['SPECTRAL_WINDOW_ID'][0]
            self.calSpwMap[tb.nrows()+k]  = self.fdmSpw[k]
            rowvals['MS_NAME'] = np.array(rowvals['MS_NAME'],'str')
            rowvals['JONES_TYPE'] = np.array(rowvals['JONES_TYPE'],'str')
            rowvals['NUM_CHAN'] = np.array([x],'int')
            rowvals['SPECTRAL_WINDOW_ID'] = np.array([self.fdmSpw[k]],'int')
            rowvals['CHAN_FREQ'] = np.zeros((1,x))
            rowvals['CHAN_WIDTH'] = np.zeros((1,x))
            rowvals['CHAN_RANGE'] = np.zeros((y1,1,x),'int')
            rowvals['POLARIZATION_TYPE'] = np.zeros((1,x),'str')
            if self.fdmSpw[k] in list(self.fdmTdmMap.keys()) : self.reconstructRow(rowvals,"%s/CAL_DESC" % self.newTable)
            tb.close()
        os.system('rm -rf table_junk')
        tb.open("%s/CAL_DESC" % self.calTable)
        calIds = tb.getcol("SPECTRAL_WINDOW_ID")
        tb.close()
        tb.open("%s" % self.calTable,nomodify=False)
        self.calrows = tb.nrows()
        self.spwCalMap = {}
        tb.close()
        noData = []
        for k,v in list(self.calSpwMap.items()) :
            self.spwCalMap[v] = self.spwCalMap.get(v,[])
            self.spwCalMap[v].append(k)
        counter = 0
        countMe = 0
        for k in range(self.calrows) :
#            if self.calSpwMap.has_key(int(rowvals['CAL_DESC_ID'])):
            tb.open("%s" % self.calTable,nomodify=False)
            tableRow = tb.selectrows(k,'table_junk')
            tb.close()
            rowvals = self.extractRow(tableRow)
            self.rowvals = rowvals
            if 1:
                tdmSpwID = self.calSpwMap[int(rowvals['CAL_DESC_ID'])]
                tdmRow = self.tdmSpw.index(tdmSpwID)
                tdmFreq = self.tdmFreqs[:,tdmRow]
                self.tdmFreq = tdmFreq
                if self.tdmFdmMap.has_key(tdmSpwID):
                    for i in self.tdmFdmMap[tdmSpwID] :
                        fdmSpwID = i
                        fdmRow = self.fdmSpw.index(i)
                        fdmFreq = self.fdmFreqs[:,fdmRow]
                        self.fdmFreq = fdmFreq
                        rowvalsp = rowvals.copy()
                        self.rowvalsp = rowvalsp
                        self.rowvals  = rowvals
                        val = int(self.spwCalMap[i][0])-self.numCalSol
                        rowvalsp['CAL_DESC_ID'] = np.array(int(self.spwCalMap[i][0])-self.numCalSol,'int')
                        if interpType == 'cubicspline' :
                            _real = (self.interpSpline(tdmFreq,fdmFreq,np.real(rowvals['GAIN'])))
                            _imag = (self.interpSpline(tdmFreq,fdmFreq,np.imag(rowvals['GAIN'])))
                            rowvalsp['GAIN'] = np.zeros(_real.shape,'complex64')
                            for i in range(rowvalsp['GAIN'].shape[0]) :
                                for j in range(rowvalsp['GAIN'].shape[1]) :
                                    rowvalsp['GAIN'][i,j] = np.complex(_real[i,j],_imag[i,j])
#                            rowvalsp['GAIN'] = (self.interpSpline(tdmFreq,fdmFreq,np.real(rowvals['GAIN'])))
                            rowvalsp['SOLUTION_OK'] = np.ones((rowvals['SOLUTION_OK'].shape[0],x),'bool') #self.interpSpline(tdmFreq,fdmFreq,rowvals['SOLUTION_OK'])
                            rowvalsp['FIT'] = np.ones((1,x),'float32') #self.interpSpline(tdmFreq,fdmFreq,rowvals['FIT'])
                            rowvalsp['FIT_WEIGHT'] = np.ones((1,x),'float32') #self.interpSpline(tdmFreq,fdmFreq,rowvals['FIT_WEIGHT'])
                            rowvalsp['FLAG'] = np.zeros((rowvals['FLAG'].shape[0],x),'bool') #self.interpSpline(tdmFreq,fdmFreq,rowvals['FLAG'])
                            rowvalsp['SNR'] = np.ones((rowvals['SNR'].shape[0],x),'float32') #self.interpSpline(tdmFreq,fdmFreq,rowvals['SNR'])
                        elif interpType == 'linear' :
                            _real = (self.interpLinear(tdmFreq,fdmFreq,np.real(rowvals['GAIN'])))
                            _imag = (self.interpLinear(tdmFreq,fdmFreq,np.imag(rowvals['GAIN'])))
                            rowvalsp['GAIN'] = np.zeros(_real.shape,'complex64')
                            for i in range(rowvalsp['GAIN'].shape[0]) :
                                for j in range(rowvalsp['GAIN'].shape[1]) :
                                    rowvalsp['GAIN'][i,j] = np.complex(_real[i,j],_imag[i,j])
#                            rowvalsp['GAIN'] = (self.interpLinear(tdmFreq,fdmFreq,np.real(rowvals['GAIN'])))
                            rowvalsp['SOLUTION_OK'] = np.ones((rowvals['SOLUTION_OK'].shape[0],x),'bool') #self.interpLinear(tdmFreq,fdmFreq,rowvals['SOLUTION_OK'])
                            rowvalsp['FIT'] = np.ones((1,x),'float32') #self.interpLinear(tdmFreq,fdmFreq,rowvals['FIT'])
                            rowvalsp['FIT_WEIGHT'] = np.ones((1,x),'float32') #self.interpLinear(tdmFreq,fdmFreq,rowvals['FIT_WEIGHT'])
                            rowvalsp['FLAG'] = np.zeros((rowvals['FLAG'].shape[0],x),'bool') #self.interpLinear(tdmFreq,fdmFreq,rowvals['FLAG'])
                            rowvalsp['SNR'] = np.ones((rowvals['SNR'].shape[0],x),'float32') #self.interpLinear(tdmFreq,fdmFreq,rowvals['SNR'])
                        else :
                            return "Invalid interpType, please pick linear or cubicspline."
                        self.reconstructRow(rowvalsp,"%s" % self.newTable,counter)
                        counter+=1
#                        sys.stdin.readline()
                    countMe+=1
                else :
                    #self.nullRow("%s" % self.newTable,k)
                    noData.append(counter)
        os.system("rm -rf table_junk")
        print(counter)
        print(noData)
        tb.close()
        if noData != [] : 
            if min(makeList(noData)) < self.calrows :
                while max(makeList(noData)) >= self.calrows :
                    noData.remove(max(makeList(noData)))
        os.system("rm -rf %s/table.lock" % self.newTable)    
        tb.open(self.newTable,nomodify=False)
        if noData != [] : tb.removerows(noData)
        tb.close()
        os.system("rm -rf %s/CAL_DESC/table.lock" % self.newTable)
        tb.open("%s/CAL_DESC" % self.newTable,nomodify=False)
        tb.removerows(range(self.numCalSol))
        tb.close()
        os.system("rm -rf %s/table.lock" % self.calTable)
        os.system("rm -rf %s/CAL_DESC/table.lock" % self.calTable)
        os.system("rm -rf table_junk")

       
    def interpLinear(self,tmpFreq,newFreq,tmpData) :
        tmpFreq,tmpData,checker = self.checkOrder(tmpFreq,tmpData)
        if newFreq[1]-newFreq[0] < 0 : newFreq = newFreq[::-1]
        newData = np.zeros((tmpData.shape[0],newFreq.shape[0]))
        for i in range(tmpData.shape[0]) :
            newData[i,:] = np.interp(newFreq,tmpFreq,tmpData[i,:])
        if checker :
            return newData.transpose()[::-1].transpose()
        else :
            return newData

    def interpSpline(self,tmpFreq,newFreq,tmpData) :
        tmpFreq,tmpData,checker = self.checkOrder(tmpFreq,tmpData)        
        newData = np.zeros((tmpData.shape[0],newFreq.shape[0]))
        for i in range(tmpData.shape[0]) :
            tck = splrep(tmpFreq,tmpData[i,:],s=0)
            newData[i,:] = splev(newFreq,tck,der=0)
        if checker :
            return newData.transpose()[::-1].transpose()
        else :
            return newData

    def checkOrder(self,inpFreq,inpData) :
        if ((inpFreq[1]-inpFreq[0]) > 0) :
            return inpFreq,inpData,0
        else :
            return inpFreq[::-1],inpData.transpose()[::-1].transpose(),1
        
    def extractRow(self,row) :
        rowvals = {}
        for i in row.colnames() :
            try:
                rowvals[i] = row.getcol(i)[...,0]
            except:
                print("Unable to extract data for %s" % i)
        return rowvals

    def reconstructRow1(self,tableName,tableRow=None) :
        tb.open(tableName,nomodify=False)
        if tableRow is None : tb.addrows()
        else :
            self.badRows.append(tableRow)
        tb.close()

    def nullRow(self,tableName,tableRow) :
        tb.open(tableName,nomodify=False)
        row = tb.selectrows(tableRow,'table_null')
        rowVals = self.extractRow(row)
        for i in list(rowVals.keys()) :
            if rowVals[i].shape == () : isRealArray = False
            else : isRealArray = True
            row,i,rowVals[i]
            try:
                dt = tb.coldatatype(i)
                if dt == 'boolean' :
                    if not isRealArray :
                        if not rowVals[i] :
                            tb.putcell(i,rownum,0)
                        else :
                            tb.putcell(i,rownum,1)
                    else :
                        tb.putcell(i,rownum,rowVals[i])
                elif dt == 'double' :
                    if not isRealArray :
                        tb.putcell(i,rownum,float(rowVals[i])*0)
                    else :
                        tb.putcell(i,rownum,rowVals[i]*0)
                elif dt == 'float' :
                    if not isRealArray :
                        tb.putcell(i,rownum,float(rowVals[i])*0)
                    else :
                        junk = np.array(rowVals[i]*0,dtype='float32')
                        tb.putcell(i,rownum,junk)                    
                elif dt == 'integer' :
                    if not isRealArray :
                        tb.putcell(i,rownum,int(rowVals[i]*0))
                    else :
                        tb.putcell(i,rownum,rowVals[i]*0)
                elif dt == 'string' :
                    if not isRealArray :
                        tb.putcell(i,rownum,str(rowVals[i]))
                    else :
                        tb.putcell(i,rownum,rowVals[i]*0)
                else :
                    tb.putcell(i,rownum,rowVals[i])
            except:
                print('skipping')
        tb.close()
        os.system("rm -rf table_null")

    def reconstructRow(self,rowVals,tableName,tableRow=None) :
        tb.open(tableName,nomodify=False)
        if tableRow is None:
            tb.addrows()
            rownum = tb.nrows()-1
        else :
            rownum = tableRow
        for i in list(rowVals.keys()) :
            if rowVals[i].shape == () : isRealArray = False
            else : isRealArray = True
            rownum,i,rowVals[i]
            try:
                dt = tb.coldatatype(i)
                if dt == 'boolean' :
                    if not isRealArray :
                        if not rowVals[i] :
                            tb.putcell(i,rownum,0)
                        else :
                            tb.putcell(i,rownum,1)
                    else :
                        tb.putcell(i,rownum,rowVals[i])
                elif dt == 'double' :
                    if not isRealArray :
                        tb.putcell(i,rownum,float(rowVals[i]))
                    else :
                        tb.putcell(i,rownum,rowVals[i])
                elif dt == 'float' :
                    if not isRealArray :
                        tb.putcell(i,rownum,float(rowVals[i]))
                    else :
                        junk = np.array(rowVals[i],dtype='float32')
                        tb.putcell(i,rownum,junk)                    
                elif dt == 'integer' :
                    if not isRealArray :
                        tb.putcell(i,rownum,int(rowVals[i]))
                    else :
                        tb.putcell(i,rownum,rowVals[i])
                elif dt == 'string' :
                    if not isRealArray :
                        tb.putcell(i,rownum,str(rowVals[i]))
                    else :
                        tb.putcell(i,rownum,rowVals[i])
                else :
                    tb.putcell(i,rownum,rowVals[i])
            except:
                print('skipping')
        tb.close()
#                tb.putcell(i,rownum,rowVals[i])

class Visibility:
    """
    Uses the tb tool to read visibility data from a measurement set.
    Instantiation requires the input MS file.  Also, if spwID is not
    set and there is more than one, beware as a failure will occur if
    the spw have different shapes.  If you create this instance, what
    you get is a structure with various attributes.  If you use the
    data.setX methods the table selection is redone,
    i.e. data.antenna1='DV01' (or 0, it interprets both, I am working
    on the same thing for field) will not make you a new table but
    data.setAntenna1('DV01') will make a new table with antenna1 as
    DV01 instead of whatever it was before.  You can also make the
    table not automatically create the subtable by setting
    autoSubtableQuery==False and then you can put in your own
    queryString.  cross_auto_all is set to 'all' by default but if you
    put in 'cross' or 'auto' it will select the relevant items.  There
    are a few functions that return the amplitude and phase (or
    recalculate them) and there is an unwrap and wrap phase option
    however, use this with caution as it depends on having alot of
    signal to noise in each measurement, i.e. it is not smart.  Let me
    know if you have questions, additions, or whatever...additions can
    just be made and a warning ;) Try to make changes backwards
    compatible...that'll make it ugly but it'll make it work!
    scan: can be a single number or a range: e.g. '1~3'
    """
    
    def __init__(self,inputMs,antenna1=0,antenna2=0,spwID=None,field=None,state=None,scan=None,autoSubtableQuery=True,queryString='',cross_auto_all='all',correctedData=False, vm=''):
        if autoSubtableQuery==False and queryString=='' : return 'Must either automatically generate the (autoSubtableQuery=True) or provide a subtable query string (queryString)'
        if spwID is None:
            spwID = getChanAverSpwIDBaseBand0(inputMs)
        self.inputMs = inputMs
        if (casaVersion >= casaVersionWithMSMD):
            self.mytb = createCasaTool(tbtool)
            mymsmd = createCasaTool(msmdtool)
            mymsmd.open(inputMs)
            self.fieldsforname = {}
            for f in range(mymsmd.nfields()):
                fname = mymsmd.namesforfields(f)[0]
                self.fieldsforname[fname] = f
            mymsmd.close()
        else:
            if vm == '':
                self.valueMap = ValueMapping(self.inputMs)
            else:
                self.valueMap = vm
        self.antenna1 = antenna1
        self.antenna2 = antenna2
        if self.antenna1 != None : self.antenna1    = str(antenna1)
        if self.antenna2 != None : self.antenna2    = str(antenna2)
        self.checkAntenna()
        if cross_auto_all.lower() in ['cross','auto','all'] :
            self.cross_auto_all = cross_auto_all
        else :
            return "Improper value for cross_auto_all, please select, cross, auto or all."
        self.spwID       = spwID
        self.field       = field
        self.checkField()
        self.correctedData = correctedData
        self.state       = state
        self.scan        = scan
        self.autoSubtableQuery = autoSubtableQuery
        self.queryString = queryString
        if self.autoSubtableQuery == True : 
            self.makeAutoSubtable()
        else : self.makeSubtable()
        self.getSpectralData()
        self.getAmpAndPhase()

    def checkAntenna(self) :
        if self.antenna1 != None : 
            self.antenna1 = str(self.antenna1)
            if not self.antenna1.isdigit() : self.antenna1 = getAntennaIndex(self.inputMs,self.antenna1)
        if self.antenna2 != None :
            self.antenna2 = str(self.antenna2)
            if not self.antenna2.isdigit() : self.antenna2 = getAntennaIndex(self.inputMs,self.antenna2)

    def makeAutoSubtable(self) :
        self.checkAntenna()
        self.makeSubtableQuery()
        mytb = tbtool()
        mytb.open(self.inputMs)
        self.subtable = mytb.query(self.queryString)
        mytb.close()
        self.getSpectralData()
        self.getAmpAndPhase()

    def makeSubtable(self) :
        mytb = tbtool()
        mytb.open(self.inputMs)
        self.subtable = mytb.query(self.queryString)
        mytb.close()

    def makeSubtableForWriting(self) :
        self.mytb = tbtool()
        self.mytb.open(self.inputMs, nomodify=False)
        self.subtable = self.mytb.query(self.queryString)
        self.rowsToWrite = self.subtable.rownumbers()  

    def setAutoSubtableQuery(self,autoSubtableQuery) :
        self.autoSubtableQuery = autoSubtableQuery

    def getAutoSubtableQuery(self) : return self.autoSubtableQuery

    def setAntennaPair(self,antenna1,antenna2) :
        self.antenna1 = str(antenna1)
        self.antenna2 = str(antenna2)
        self.checkAntenna()
        if self.autoSubtableQuery : self.makeAutoSubtable()

    def setAntenna1(self,antenna1) :
        self.antenna1 = str(antenna1)
        self.checkAntenna()
        if self.autoSubtableQuery : self.makeAutoSubtable()

    def setAntenna2(self,antenna2) :
        self.antenna2 = str(antenna2)
        self.checkAntenna()
        if self.autoSubtableQuery : self.makeAutoSubtable()

    def getAntenna1(self) : return self.antenna1

    def getAntenna2(self) : return self.antenna2

    def getAntennaPair(self) :
        return [self.antenna1,self.antenna2]

    def setSpwID(self,spwID) :
        self.spwID = spwID
        if self.autoSubtableQuery : self.makeAutoSubtable()

    def getSpwID(self) : return self.spwID

    def setField(self,field) :
        self.field = field
        self.checkField()
        if self.autoSubtableQuery : self.makeAutoSubtable()

    def getField(self) : return self.field

    def checkField(self) :
        if self.field != None : 
            self.field = str(self.field)
            if not self.field.isdigit(): 
                if (casaVersion >= casaVersionWithMSMD):
                    self.field = self.fieldsforname[self.field]
                else:
                    self.field = self.valueMap.getFieldIdsForFieldName(self.field)[0]

    def setState(self,state) :
        self.state = state
        if self.autoSubtableQuery : self.makeAutoSubtable()
    
    def getState(self) : return self.state
    
    def setScan(self,scan) :
        self.scan = scan
        if self.autoSubtableQuery : self.makeAutoSubtable()
    
    def getScan(self) : return self.scan
    
    def makeSubtableQuery(self) :
        self.parameterList = []
        queryString = ''
        if self.antenna1 != None : self.parameterList.append('ANTENNA1 == %s' % self.antenna1)
        if self.antenna2 != None : self.parameterList.append('ANTENNA2 == %s' % self.antenna2)
        if self.field != None    : self.parameterList.append('FIELD_ID == %s' % self.field)
        if self.spwID != None    : self.parameterList.append('DATA_DESC_ID == %s' % getDataDescriptionId(self.inputMs,self.spwID))
        if self.state != None    : self.parameterList.append('STATE_ID == %s' % self.state)
        if self.scan != None     :
            if (type(self.scan) == str):
                # Added by Todd on 2014-09-22
                if (self.scan.find('~') > 0):
                    self.parameterList.append('SCAN_NUMBER in %s' % str(range(int(self.scan.split('~')[0]),1+int(self.scan.split('~')[1]))))
                else:
                    self.parameterList.append('SCAN_NUMBER == %s' % self.scan)
            else:
                self.parameterList.append('SCAN_NUMBER == %s' % self.scan)
        if self.cross_auto_all == 'cross' : self.parameterList.append('ANTENNA1 <> ANTENNA2')
        elif self.cross_auto_all == 'auto' : self.parameterList.append('ANTENNA1 == ANTENNA2')
        for i in self.parameterList : queryString = self.appendQuery(queryString,i)
        self.queryString = queryString

    def getSpectralData(self) :
        if 'FLOAT_DATA' in self.subtable.colnames() :
            if self.correctedData :
                self.specData = self.subtable.getcol('FLOAT_DATA')
            else :
                self.specData = self.subtable.getcol('FLOAT_DATA')
        else :
            if self.correctedData:
                self.specData = self.subtable.getcol('CORRECTED_DATA')
            else :
                self.specData = self.subtable.getcol('DATA')
        self.specTime = self.subtable.getcol('TIME')
        self.specFreq = getFrequencies(self.inputMs,self.spwID)
        self.tavgSpecData = self.specData.mean(-1)
        self.favgSpecData = self.specData.mean(-2)
            
    def putSpectralData(self, data, i):
        if 'FLOAT_DATA' in self.subtable.colnames() :
            if self.correctedData :
                self.mytb.putcell('FLOAT_DATA', self.rowsToWrite[i], data)
            else :
                self.mytb.putcell('FLOAT_DATA', self.rowsToWrite[i], data)
        else :
            if self.correctedData:
                self.mytb.putcell('CORRECTED_DATA', self.rowsToWrite[i], data)
            else :
                self.mytb.putcell('DATA', self.rowsToWrite[i], data)

    def getAmpAndPhase(self) :
        rData = self.specData.real
        rTavgData = self.tavgSpecData.real
        rFavgData = self.favgSpecData.real
        iData = self.specData.imag
        iTavgData = self.tavgSpecData.imag
        iFavgData = self.favgSpecData.imag
        self.phase = np.arctan2(iData,rData)
        self.amp   = (rData**2.0+iData**2.0)**0.5
        self.tavgPhase = np.arctan2(iTavgData,rTavgData)
        self.tavgAmp   = (rTavgData**2.0+iTavgData**2.0)**0.5
        self.favgPhase = np.arctan2(iFavgData,rFavgData)
        self.favgAmp   = (rFavgData**2.0+iFavgData**2.0)**0.5

    def wrapPhase(self,simple=True) :
        from math import pi
        phaseShape = self.phase.shape
        for i in range(phaseShape[2]) :
            if self.phase[:,:,i] <  -pi : self.phase[:,:,i]=self.phase[:,:,i]+2*np.pi
            elif self.phase[:,:,i] > pi : self.phase[:,:,i]=self.phase[:,:,i]-2*np.pi
        
    def unwrapPhase(self,simple=True) :
        from math import pi
        phaseShape = self.phase.shape
        for i in range(phaseShape[2]-1) :
            diff = self.phase[:,:,i]-self.phase[:,:,i+1]
            _diffg = (diff > 1.*np.pi)*2*np.pi
            _diffl = (diff < -1.*np.pi)*2*np.pi
            self.phase[:,:,i+1] = self.phase[:,:,i+1]+_diffg-_diffl

    def appendQuery(self,queryString,additive) :
        if queryString == '' :
            if additive == '' : return queryString
            else : return additive
        else :
            if additive == '' : return queryString
            else : 
                queryString = queryString + ' && ' + additive
                return queryString
    # end of Visibility class

def plotCorrectedMinusModel(vis, spw, field='', antenna1=None, antenna2=None,
                            xaxis='uvdist', avgtime=False, avgchannel='',
                            plotfile=''):
    """
    Computes and plots the scalar amplitude difference between the corrected 
    data and the model.   Note: the avgtime feature might have missed timestamps
    prior to the changes made in CAS-12879 (fixed in CASA 6.2 onward).
    spw: integer or string (comma-delimited)
    field: integer or string ID or string name (optional)
    xaxis: 'uvdist' or 'chan'
    avgtime: if True, then avgtime across all selected rows 
    avgchannel: string or integer number of channels to average
    plotfile: name of plotfile to produce (default: <vis>.ampCorrectedMinusModel.png)
    -Todd Hunter
    """
    if (xaxis not in ['uvdist','chan']):
        print("xaxis must be either 'uvdist' or 'chan'")
        return
    if (xaxis == 'uvdist' and avgtime):
        print("avgtime not support for 'uvdist' (thanks to the ms tool)")
        return
    pb.clf()
    spws = parseSpw(vis,spw)
    for i,spw in enumerate(spws):
        difference, uvdist = correctedMinusModel(vis, spw, field, antenna1, antenna2, avgtime, avgchannel)
        # axes of difference: [pol][chan][time]
        nrows = len(uvdist)
        if avgtime:
            nchan = np.shape(difference)[1]
        if xaxis == 'chan':
            xpol = difference[0]
            ypol = difference[1]
            pb.plot(range(nchan), xpol, 'bo', range(nchan), ypol, 'go')
            pb.xlabel('Channel')
        elif xaxis == 'uvdist':
            for bin in range(np.shape(difference)[1]):
                xpol = difference[0][bin]
                ypol = difference[1][bin]
                pb.plot(uvdist, xpol, 'bo', uvdist, ypol, 'go')
            pb.xlabel('uvdist (m)')
    pb.ylabel('Amp corrected - Amp model')
    if plotfile != '':
        png = plotfile
    else:
        png = vis + '.ampCorrectedMinusModel.png'
    if field != '':
        field_id, fieldname = parseFieldArgument(vis, field)
        fieldstr = ' ' + ','.join(fieldname)
    pb.title(os.path.basename(vis) + ' spw %d'%spw + fieldstr)
    pb.savefig(png)
    pb.draw()

def correctedMinusModel(vis, spw, field='', antenna1=None, antenna2=None,
                        avgtime=False, avgchannel=''):
    """
    Uses the ms tool to retrieve and compute the difference between the 
    corrected column and model column of visibilities for the specified 
    spw and field.  Note: the avgtime feature might have missed timestamps
    prior to the changes made in CAS-12879 (fixed in CASA 6.2 onward).
    spw: integer 
    field: integer or string ID or string name (optional)
    antenna1,2: antenna ID or name
    avgtime: if True, then avgtime across all selected rows 
    avgchannel: string or integer number of channels to average

    Returns: 2 lists: difference of amplitudes, uvdist
    -Todd Hunter
    """
    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)
    datadescid = mymsmd.datadescids(spw=spw)[0]
    if (avgchannel != ''):
        spwchan = mymsmd.nchan(spw)
    mymsmd.close()
    myms = createCasaTool(mstool)
    myms.open(vis)
    myms.selectinit(datadescid=datadescid)
    if (field != ''):
        field_id, fieldname = parseFieldArgument(vis, field)
        print("Using field_id = ", field_id)
        myms.select({'field_id':field_id})
    if (type(spw) == str):
        spw = int(spw)
    if (antenna1 is not None):
        if type(antenna1) == str:
            antenna1 = parseAntenna(vis, antenna1)
        myms.select({'antenna1': antenna1})
    if (antenna2 is not None):
        if type(antenna2) == str:
            antenna2 = [int(i) for i in antenna2.split(',')]
        myms.select({'antenna2': antenna2})
    if (avgchannel != ''):
        width = int(avgchannel)
        mydata = myms.getdata(['flag'])
        if (len(np.shape(mydata['flag'])) == 0):
            print("No data found")
            return
        meanflag = np.mean(np.mean(mydata['flag'],axis=2),axis=0)
        goodchannels = np.where(meanflag < 1.0)[0]
        nchan = (goodchannels[-1]-goodchannels[0]+1)/width
        start = goodchannels[0]
        inc = width
        myms.selectchannel(nchan, start, width, inc)
    mydata = myms.getdata(['corrected_amplitude','model_amplitude','uvdist'], average=avgtime)
    myms.close()
    difference = mydata['corrected_amplitude'] - mydata['model_amplitude']
    return(difference, mydata['uvdist'])

def computeUVThreshold(vis, spw='', field='', nbins=15, mydict=None, 
                       avgtime=False, averageChannel=True, intent='TARGET',
                       pol=0, doplot=False, plotfile='', nsigma=[2,5.5], # 6.5,18], 
                       column='corrected_data', statistic='percentile', 
                       plotrange=[0,0,0,0], percentile=25, verbose=False,
                       twopanels='auto'):
    """
    Runs getCorrectedData, sorts by uvdist, divides into nbins and computes statistics
    spw: integer or string integer ID (default = first science spw)
    field: default = first science field
    nbins: number of uvdist bins -- their width varies inversely with density of points
    mydict: a prior result from getCorrectedData [uvdist,uvamp,Nantennas] (for testing only)
    statistic: 'mean', 'median' or 'percentile' which is 10-90%ile
    nsigma: single value, or list of 2 (lower, upper) or array of 2
    intent: passed to getCorrectedData to filter out fields present in the
            FIELD table but not present in the main table of the ms
    Returns: list of uvdist values, list of amplitude high thresholds,
        list of amplitude low thresholds
    """
    if not os.path.exists(vis):
        print("Measurement set not found.")
        return
    if plotfile != '':
        doplot = True
    if spw == '':
        spws = getScienceSpws(vis,returnString=False)
        if len(spws) == 0:
            print("No science spws, you must specify an spw.")
            return
        spw = spws[0]            
        print("Using first science spw: ", spw)
    spw = int(spw)
    if field == '':
        fields = getScienceTargets(vis, returnNames=True)
        if len(fields) == 0:
            print("No science fields, you must specify a field name or ID.")
            return
        field = fields[0]
        print("Using first science target: ", field)
    if type(nsigma) != list and type(nsigma) != np.ndarray:
        nsigma = np.array([nsigma, nsigma])
    else:
        nsigma = np.array(nsigma)
    if mydict is None:
        result = getCorrectedData(vis, spw, field, pol=pol, avgtime=avgtime, averageChannel=averageChannel, column=column, blendByName=True, intent=intent)
        if result is None:
            return
        uvdist, amp, Nantennas = result
    else:
        uvdist, amp, Nantennas = mydict
    idx = np.argsort(uvdist)
    uvdist = uvdist[idx]
    amp = amp[idx]
    ndata = len(amp)
    binwidth = float(ndata)/nbins
    medians = []
    mads = []
    uvdists = []
    percentileYmax = []
    percentileYmin = []
    percentileYrange = []
    for i in range(nbins):
        start = int(i*binwidth)
        stop = int(np.ceil((i+1)*binwidth))+1
        if stop > start:
            if statistic == 'median':
                medians.append(np.median(amp[start:stop]))
                mads.append(MAD(amp[start:stop]))
            elif statistic == 'mean':
                medians.append(np.mean(amp[start:stop]))
                mads.append(MAD(amp[start:stop]))
            elif statistic == 'percentile':
                percentileYmax.append(scoreatpercentile(amp[start:stop],100-percentile))
                percentileYmin.append(scoreatpercentile(amp[start:stop],percentile))
                percentileYrange.append(scoreatpercentile(amp[start:stop],100-percentile) - scoreatpercentile(amp[start:stop],percentile))
#                 The mean of the 10th and 90th percentile levels
#                medians.append(0.5*(scoreatpercentile(amp[start:stop],100-percentile) + scoreatpercentile(amp[start:stop],percentile)))
                high = scoreatpercentile(amp[start:stop],100-percentile)
                low = scoreatpercentile(amp[start:stop],percentile)
                idx = np.where((amp[start:stop] < high) * (amp[start:stop] > low))  
                medians.append(np.median(amp[start:stop][idx]))
                mads.append(MAD(amp[start:stop][idx]))
                if verbose:
                    print("uvdist=%f, low/high = %f / %f, mad = %f" % (np.mean(uvdist[start:stop]),low,high,mads[-1]))
            else:
                print("statistic must be either median or mean")
                return
            uvdists.append(np.mean(uvdist[start:stop]))
    if verbose:
        print("percentileYrange: ", percentileYrange)
        print("mads: ", mads)
    # account for edge of final bin
    if len(uvdist) == 0:
        print("No qualifying data")
        return
    uvdists.append(np.max(uvdist))
    medians.append(medians[-1])
    mads.append(mads[-1])
    uvdists = [np.min(uvdist)] + uvdists
    # prepend the same value to match the uvdist min
    medians = medians[:1] + medians
    mads = mads[:1] + mads
    mads = np.array(mads)
    medians = np.array(medians)
    if statistic in  ['mean','median']:
        threshold = medians + mads*nsigma[1]
        thresholdLower = medians - mads*nsigma[0]
    else:  # percentile
        # prepend the same value to match the uvdist min
        percentileYmax = percentileYmax[:1] + percentileYmax
        percentileYmin = percentileYmin[:1] + percentileYmin
        percentileYrange = percentileYrange[:1] + percentileYrange
        # append the final value to match the uvdist max
        percentileYmax.append(percentileYmax[-1])
        percentileYmin.append(percentileYmin[-1])
        percentileYrange.append(percentileYrange[-1])
        # convert to arrays
        percentileYmax = np.array(percentileYmax)
        percentileYmin = np.array(percentileYmin)
        percentileYrange = np.array(percentileYrange)
#        threshold = percentileYmax + percentileYrange*nsigma[1]
#        thresholdLower = percentileYmin - percentileYrange*nsigma[0]
        threshold = medians + percentileYrange*nsigma[1]
        thresholdLower = medians - percentileYrange*nsigma[0]
    # place a floor on amplitude values at zero
    idx = np.where(thresholdLower < 0)
    thresholdLower[idx] = 0
    if doplot:
        pb.clf()
        if twopanels == 'auto':
            twopanels = np.max(threshold)*5 < np.max(amp)
        if twopanels:
            pb.subplot(211)
        else:
            pb.subplot(111)
        pb.plot(uvdist, amp, 'k.', mfc='white')
        pb.errorbar(uvdists, medians, yerr=mads*nsigma[1], lw=2)
        pb.plot(uvdists, threshold, 'r-', drawstyle='steps-mid')
        pb.plot(uvdists, thresholdLower, 'r-', drawstyle='steps-mid')
        if plotrange[:2] != [0,0]:
            pb.xlim(plotrange[:2])
        if plotrange[2:] != [0,0]:
            pb.ylim(plotrange[2:])
        pb.xlabel('UV distance')
        pb.ylabel('Amplitude')
        if statistic == 'percentile':
            mystatistic = 'percentile%02d' % percentile
        else:
            mystatistic = statistic
        mytitle = os.path.basename(vis) + ', spw%s, field=%s, pol%s, nbins=%d, nsigma=%s, %s' % (str(spw),field,str(pol),nbins,nsigma,mystatistic)
        pb.title(mytitle, size=10)
        padPlotLimits(0.05, False,False,True,True)
        if twopanels:
            pb.subplot(212)
            pb.plot(uvdist, amp, 'k.', mfc='white')
            pb.errorbar(uvdists, medians, yerr=mads*nsigma[1], lw=2)
            pb.plot(uvdists, threshold, 'r-', drawstyle='steps-mid')
            pb.plot(uvdists, thresholdLower, 'r-', drawstyle='steps-mid')
            pb.xlabel('UV distance')
            pb.ylabel('Amplitude')
            pb.ylim([np.min(thresholdLower), np.max(threshold)])
            padPlotLimits(0.05, False,False,True,True)
        pb.draw()
        if plotfile != '':
            if plotfile == True:
                plotfile = vis + '.spw%d.%s.uvthreshold.png' % (int(spw),mystatistic)
            pb.savefig(plotfile)
            print("Wrote ", plotfile)
    outliers = []
    return uvdists, threshold, thresholdLower

def refantScoreHistogram(casalog, flagbins=50, totalbins=50, flagxlim=[0,0], 
                         refant='', vis=''):
    """
    Generates histogram of the flagging score and total score from hif_refant
    with antennas labeled.  If you simply want a histogram of the total score,
    then use au.plotRefantScore.
    refant: if specified, label it in red instead of black
    vis: if specified, use this measurement set, rather than the first one
    -Todd Hunter
    """
    flaggingScore, totalScore, vis = readRefantScoresFromCasalog(casalog, vis)
    if not flaggingScore:
        print("No scores found for vis=", vis)
        return
    pb.clf()
    desc = pb.subplot(211)
    pb.hist(flaggingScore.values(), bins=flagbins)
    yFormat = matplotlib.ticker.ScalarFormatter(useOffset=False)
    desc.xaxis.set_major_formatter(yFormat)
    pb.xlabel('Flagging score')
    if flagxlim != [0,0]:
        pb.xlim(flagxlim)
    trans = matplotlib.transforms.blended_transform_factory(desc.transData, desc.transAxes)
    antennas = flaggingScore.keys()
    print("Found %d antennas" % (len(antennas)))
    for a in antennas:
        if a != refant:
            pb.text(flaggingScore[a], 1, a, va='top', ha='center', transform=trans,
                    rotation=90)
    if refant != '':
        if refant not in flaggingScore:
            print("refant not found")
            return
        print("refant flagging score = ", flaggingScore[refant])
        pb.text(flaggingScore[refant], 1, refant, va='top', ha='center', 
                transform=trans, rotation=90, color='r')
    pb.title(os.path.basename(casalog) + ', ' + vis)

    desc2 = pb.subplot(212)
    pb.hist(totalScore.values(), bins=totalbins)
    trans2 = matplotlib.transforms.blended_transform_factory(desc2.transData, desc2.transAxes)
    desc2.xaxis.set_major_formatter(yFormat)
    for a in antennas:
        if a != refant:
            pb.text(totalScore[a], 1, a, va='top', ha='center', transform=trans2, 
                    rotation=90)
    if refant != '':
        pb.text(totalScore[refant], 1, refant, va='top', ha='center', 
                transform=trans2, rotation=90, color='r')
    pb.xlabel('Total score')
    png = '%s_refantScoreHistogram.png' % (casalog)
    pb.savefig(png)
    print("Wrote ", png)

def pipelineRefantScores(pipelinedir, vis='', verbose=True, fast=True):
    """
    Identifies CASA log file and calls readRefantScoresFromCasalog()
    fast: option passed to findPipelineWorkingDirectory (if necessary) and 
           findPipelineCasaLogfile
    -Todd Hunter
    """
    if pipelinedir[-9:].find('/working') >= 0:
        workingdir = pipelinedir
    else:
        workingdir = findPipelineWorkingDirectory(pipelinedir, fast=fast)
        if verbose:
            print("Found workingdir = ", workingdir)
    casalog = findPipelineCasaLogfile(workingdir, fast=fast)
    if verbose:
        print("Found casalog = ", casalog)
    return readRefantScoresFromCasalog(casalog, vis)

def readRefantScoresFromCasalog(casalog, vis=''):
    """
    Reads flagging score from hif_refant from a casa log file.
    vis: if specified, use this measurement set, rather than the first one
    Returns: 2 dictionaries: flaggingScore and totalScore, keyed by antenna name
          and the name of the measurement set
    -Todd Hunter
    """
    f = open(casalog,'r')
    lines = f.readlines()
    f.close()
    flaggingScore = {}; totalScore = {}
    searchForMS = False
    myvis = ''
    for line in lines:
        if line.find('Equivalent CASA call: hif_refant') > 0:
            searchForMS = True
        if searchForMS:
            if line.find(vis) > 0 and line.find('.ms') > 0 and myvis == '': # line.find('') returns 0
                myvis = line.split('.ms')[0].split("'")[-1].split('/')[-1] + '.ms'
                print("Found vis = ", myvis)
                vis = myvis
            elif vis == '' and line.find('.ms') > 0 and myvis == '':
                myvis = 'uid___' + line.split('.ms')[0].split('uid___')[-1] + '.ms'
        if line.find('findrefant::') > 0 and vis=='' or vis == myvis:
            if line.find(' flagging score ') > 0:
                token = line.split()
                antenna = token[5]
                flaggingScore[antenna] = float(token[8])
                totalScore[antenna] = float(token[11])
    return flaggingScore, totalScore, myvis

def flagScienceTargetOutliers(vis, spw, field=''):
    """
    Not yet fully implemented.
    """
    uvdists, threshold, thresholdLower = computeUVThreshold(vis, spw, field)
    # generate flagging commands
    f = open('%s_flagScienceTarget_spw%s.txt' % (vis,spw))
    f.write("flagdata('%s', spw='%s')" % (vis,spw))
    f.close()

def getCorrectedData(vis, spw, field='', pol=0, antenna1=None, antenna2=None, 
                     averageChannel=True, avgchannel='',column='corrected_data',
                     blendByName=False, verbose=False, intent='', 
                     takeNumpyAbsolute=False, returnWeight=False, use_tbcalc=False, 
                     channel='', useflags=True):
    """
    Uses the ms tool to retrieve the uvdist and specified data column of visibilities for the specified 
    spw and field.
    spw: integer or string (channel ranges not allowed)
    field: integer or string ID or string name (but must be specified if averageChannel is True)
    antenna1,2: antenna ID or name
    pol: 0 or 1, or [0,1] or -1 for both
    avgchannel: string or integer number of channels to average (different mechanism as averageChannel, 
                and not tested nearly as much)
    averageChannel: Boolean (if you set it False, it can overflow memory)  True uses tb.query(means)
    column: 'data' or 'corrected_data'
    blendByName: passed to parseFieldArgument
    takeNumpyAbsolute: default=False for vector average (which is what you want for flux density)
    use_tbcalc: True or False (uses ms.select) give virtually the same answer but False is 2x faster
    channel: either blank (all channels) or a single channel (integer or string integer)
    intent: if specified, then limit automatic field ID finding to those with
            this intent (useful to eliminate Tsys-only fields which are in the
            FIELD table but not the main table of a pipeline _target.ms)
            The value of intent will be surrounded by '*', so short names 'PHASE' will work
            Note: The data are not limited to scans with this intent, only fields with this intent!
    Returns: 2 lists + integer: uvdist, amplitudes (if one pol specified), Nantennas
         or 4 lists + integer: uvdist0,amp0,uvdist1,amp1,Nantennas
      if returnWeight=True, then it returns 3 or 6 lists:
          uvdist, amplitudes, Nantennas, weight, [uvdist1, amplitudes1, weight1]
    -Todd Hunter
    """
    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)
    Nantennas = mymsmd.nantennas()
    spw = int(spw)
    allspws = range(0,mymsmd.nspw())
    if spw not in allspws:
        print("Available spws: ", allspws)
        print("Science spws: ", np.intersect1d(mymsmd.spwsforintent('*TARGET*'),mymsmd.almaspws(tdm=True,fdm=True)))
        return
    datadescids = mymsmd.datadescids(spw=spw)
    if len(datadescids) > 1:
        print("Why is there more than one datadescid for spw %d" % (spw))
    datadescid = datadescids[0]
    spwchan = mymsmd.nchan(spw)
    if pol == -1:
        poln = [0,1]
    elif type(pol) == list:
        poln = pol
    else:
        poln = [pol]
    myms = createCasaTool(mstool)
    myms.open(vis)
    myms.selectinit(datadescid=datadescid)
    if averageChannel and field == '' and intent == '':
        print("You must specify a field ID or name or an intent when using averageChannel.")
        return
    if field == '' and intent != '':
        myfields = mymsmd.fieldsforintent('*'+intent+'*', asnames=False)
        field_id = myfields
        print("Picked fields: ", field_id)
    if (field != ''):
        field_id, fieldname = parseFieldArgument(vis, field, blendByName, mymsmd=mymsmd)
        if verbose:
            print("field_id = ", field_id)
        if field_id == []:
            return
        if intent != '':
            myfields = mymsmd.fieldsforintent('*'+intent+'*', asnames=False)
            field_id = np.intersect1d(myfields,field_id)
        if len(field_id) == 0 and field != '':
            print("Likely there is a mismatch between intent (%s) and field_id (%s)." % (intent,str(field_id)))
            if intent != '':
                print("fields for intent = ", myfields)
            myms.close()
            return
    if len(field_id) > 0:
        try:
            myms.select({'field_id':field_id})
            print("Selected field_id = ", field_id)
        except:
            myms.close()
            return

    if (type(spw) == str):
        spw = int(spw)
    if (antenna1 is not None):
        if type(antenna1) == str:
            antenna1 = parseAntenna(vis, antenna1, mymsmd=mymsmd)
        myms.select({'antenna1': antenna1})
    if (antenna2 is not None):
        if type(antenna2) == str:
            antenna2 = [int(i) for i in antenna2.split(',')]
        myms.select({'antenna2': antenna2})
    mymsmd.close()
    if (avgchannel != ''):
        width = int(avgchannel)
        mydata = myms.getdata(['flag'])
        if (len(np.shape(mydata['flag'])) == 0):
            print("No data found")
            return
        meanflag = np.mean(np.mean(mydata['flag'],axis=2),axis=0)
        if useflags:
            goodchannels = np.where(meanflag < 1.0)[0]
        else:
            goodchannels = np.where(meanflag < 10.0)[0]
        nchan = (goodchannels[-1]-goodchannels[0]+1)/width
        start = goodchannels[0]
        inc = width
        myms.selectchannel(nchan, start, width, inc)

    if averageChannel:
        uvdist = []
        scan = []
        amplitude = []
        weight = []
        if not use_tbcalc:
            # Need to get the weights before you select channel averaging, even though they are not channelized
            weights = myms.getdata(['weight'])
            # Extract channel-averaged data from MS.
            myms.selectchannel(1, 0, spwchan, 1)
            if verbose:
                print("column = ", column)
            alldata = myms.getdata([column, 'flag', 'uvdist', 'scan_number'])
            if verbose:
                print("alldata.keys() = ", str(list(alldata.keys())))
            mydata = np.squeeze(alldata[column], axis=1)
            flag_all = np.squeeze(alldata['flag'], axis=1)

            npol = len(mydata)
            print('*** npol = '+str(npol)+' ***')
            if npol < 2:
                if len(poln) == 1 and poln[0] == 1:
                    print("Skipping second polarization since this is single-pol data")
                    return
                if len(poln) == 2:
                    print("Data is single pol, so reverting to poln=[0]")
                    poln = [0]
            elif npol==4:
                if poln==[0,1]:
                    print("Data is full pol, so reverting to poln=[0,3]")
                    poln=[0,3]

            for icorr in poln:
                cmetric = mydata[icorr]
                flag = flag_all[icorr]
                print("pol%d: weights['weight'][%d][%d]=%s" % (icorr,icorr,0,weights['weight'][icorr][0]))
                print("pol%d: weights['weight'][%d][%d]=%s" % (icorr,icorr,341,weights['weight'][icorr][341]))
                print("pol%d: weights['weight'][%d][%d]=%s" % (icorr,icorr,396,weights['weight'][icorr][396]))
                myweight = weights['weight'][icorr]
                # Select for non-flagged data and non-NaN data.
                if useflags:
                    id_nonbad = np.where(np.logical_and(
                        np.logical_not(flag),
                        np.isfinite(cmetric)))
                else:
                    id_nonbad = np.where(np.isfinite(cmetric))
                print("%d: shape(id_nonbad[0]): " % (icorr), np.shape(id_nonbad[0]))
                print("%d: sum(id_nonbad[0]): " % (icorr), np.sum(id_nonbad[0]))
                cmetric_sel = cmetric[id_nonbad]
                if takeNumpyAbsolute:
                    amp = np.abs(cmetric_sel)
                else:
                    amp = cmetric_sel
                if verbose:
                    print("pol%d: mean(amp)=%f" % (icorr,np.mean(amp)))
                    print("pol%d: len(myweight)=%d" % (icorr,len(myweight[id_nonbad])))
                    print("pol%d: myweight[%d]=%f" % (icorr,id_nonbad[0][0],myweight[id_nonbad][0]))
                    print("pol%d: scan[%d] = %d" % (icorr, id_nonbad[0][0],alldata['scan_number'][id_nonbad][0]))
                    print("pol%d: mean(myweight)=%f" % (icorr,np.mean(myweight[id_nonbad])))
                    print("pol%d: sum(myweight)=%f" % (icorr,np.sum(myweight[id_nonbad])))
                    print("pol%d: mean(uvdist)=%f" % (icorr,np.mean(alldata['uvdist'][id_nonbad])))
                amplitude.append(amp)
                scan.append(alldata['scan_number'][id_nonbad])
                uvdist.append(alldata['uvdist'][id_nonbad])
                weight.append(myweight[id_nonbad])
        else:
            mydata = myms.getdata(['uvdist','flag'])
            myms.close()
            mytb = tbtool()
            myfield_id = '[' + ','.join([str(i) for i in field_id]) + ']'
            if column.find('corrected') >= 0:
                if useflags:
                    query = "[select means(CORRECTED_DATA[FLAG],1) from "+vis+" where DATA_DESC_ID=="+str(datadescid)+" && FIELD_ID in "+myfield_id+ "]"
                else:
                    query = "[select means(CORRECTED_DATA) from "+vis+" where DATA_DESC_ID=="+str(datadescid)+" && FIELD_ID in "+myfield_id+ "]"
                print("tb.calc('%s')" % (query))
                mycalc = mytb.calc(query)
                query = "[select WEIGHT from "+vis+" where DATA_DESC_ID=="+str(datadescid)+" && FIELD_ID in "+myfield_id+ "]"
                print("tb.calc('%s')" % (query))
                myweight = mytb.calc(query)
                mytb.close()
            elif column == 'data':
                if useflags:
                    query = "[select means(DATA[FLAG],1) from "+vis+" where DATA_DESC_ID=="+str(datadescid)+" && FIELD_ID in "+myfield_id+ "]"
                else:
                    query = "[select means(DATA) from "+vis+" where DATA_DESC_ID=="+str(datadescid)+" && FIELD_ID in "+myfield_id+ "]"
                print("tb.calc('%s')" % (query))
                mycalc = mytb.calc(query)
                query = "[select WEIGHT from "+vis+" where DATA_DESC_ID=="+str(datadescid)+" && FIELD_ID in "+myfield_id+ "]"
                print("tb.calc('%s')" % (query))
                myweight = mytb.calc(query)
                mytb.close()
            else:
                print("Unrecognized column: ", column)
                mytb.close()
                return
            if takeNumpyAbsolute:
                amp = np.abs(mycalc['0'])
            else:
                amp = mycalc['0']
            weights = myweight['0']
            ncorrs, rows = amp.shape

            npol = len(amp)
            if npol < 2:
                if len(poln) == 1 and poln[0] == 1:
                    print("Skipping second polarization since this is single-pol data")
                    return
                if len(poln) == 2:
                    print("Data is single pol, so reverting to poln=[0]")
                    poln = [0]
            elif npol==4:
                if poln==[0,1]:
                    print("Data is full pol, so reverting to poln=[0,3]")
                    poln=[0,3]

            for pol in poln:
                # initialize a zero array
                flag = np.zeros(rows, dtype='bool')
                idx = np.where(amp[pol] == 0j)[0]
                if (len(idx) > 0):  
                    # values that are identically zero were flagged in the data
                    if len(np.shape(mydata['flag'])) == 1:
                        nrow = np.shape(mydata['flag'])[0] // spwchan // npol
                        mydata['flag'] = np.reshape(mydata['flag'],(npol,spwchan,nrow))
                        print("reshaped to ", np.shape(mydata['flag']))
                        f = open('_'.join([os.path.basename(vis),str(spw),'reshaped']), 'w')
                        f.write('%s\n' % str(np.shape(mydata['flag'])))
                        f.close()
                    nchan = len(mydata['flag'][pol,:,0])
                    setFlags = 0
                    if useflags:
                        for i in idx:
                            if (np.sum(mydata['flag'][pol,:,i]) == nchan):
                                setFlags += 1
                                flag[i] = True
                    if verbose or True:
                        print("Set %d/%d flags for pol %d column %s" % (setFlags,len(amp[pol]),pol,column))
                unflagged = np.where(flag==False)
                # Check and print if any points have zero uvdistance
                idx = np.where(mydata['uvdist'][unflagged] <= 0)
                if verbose:
                    print("%d points with uvdist=0: " %(len(idx[0])), idx)
                uvdist.append(mydata['uvdist'][unflagged])
                amplitude.append(amp[pol][unflagged])
                weight.append(weights[pol][unflagged])
        if len(poln) == 1:
            if returnWeight:
                return(uvdist[0], amplitude[0], Nantennas, weight[0])
            else:
                return(uvdist[0], amplitude[0], Nantennas)
        else:
            if returnWeight:
                return(uvdist[0], amplitude[0], Nantennas, weight[0], uvdist[1], amplitude[1], weight[1])
            else:
                return(uvdist[0], amplitude[0], Nantennas, uvdist[1], amplitude[1])
    else: # single channel requested
        myms.selectchannel(1, int(channel), 1, 1)
        alldata = myms.getdata([column, 'flag', 'uvdist'])
        mydata = np.squeeze(alldata[column], axis=1)
        flag_all = np.squeeze(alldata['flag'], axis=1)
        weights = myms.getdata(['weight'])
        weight = weights['weight'][pol]
        myms.close()
#        print("shape(mydata)=%s, weight=%s" % (np.shape(mydata),np.shape(weight)))
        if takeNumpyAbsolute:
            myamp = np.abs(mydata)
        else:
            myamp = mydata
        if verbose:
            print("%s min/max = %f / %f" % (column, np.min(myamp), np.max(myamp)))
        if returnWeight:
            return(alldata['uvdist'], myamp, Nantennas, weight)
        else:
            return(alldata['uvdist'], myamp, Nantennas)

def alignFunctions(xaxis1, intensity1, xaxis2, intensity2, points=None, k=3):
    """
    Takes any two functions and returns spline-fit versions sampled onto a 
    common grid which is set to span the common range of their x-axes with 
    at least as many points as the input function.
    Optional inputs:
    points: the number of points desired in the output grid
    k: passed to interpolateSpectrum
    Returns:
    xaxis, function1, function2
    -Todd Hunter
    """
    newmin = np.max([np.min(xaxis1),np.min(xaxis2)])
    newmax = np.min([np.max(xaxis1),np.max(xaxis2)])
    if points is None:
        points = np.max([len(xaxis1), len(xaxis2)])
    xaxis = np.linspace(newmin,newmax,points)
    int1 = interpolateSpectrum(xaxis1, intensity1, xaxis, k)
    int2 = interpolateSpectrum(xaxis2, intensity2, xaxis, k)
    return xaxis, int1, int2
    
def interpolateSpectrum(inputFreq, inputSpec, outputFreq, k=3, verbose=True,
                        bounds_error=True, fill_value=0.0):
    """
    Interpolates a spectrum onto a finer grid of frequencies using a spline fit.
    If the inputFreq is narrower than the outputFreq, it will extrapolate the
    inputSpec with 'nearest' .
    k: if an integer, then use scipy.interpolate.splrep and splev
    k: if a string, then use scipy.interpolate.interp1d
       options: 'linear', 'nearest', 'zero', 'slinear', 'quadratic', 'cubic'
    inputFreq: input x-axis values
    inputSpec: input y-axis values
    outputFreq: desired x-axis values
    Returns:
    outputSpec: y-axis values corresponding to the desired x-axis values
    Todd Hunter
    """
    tmpFreq, tmpData, checker = checkOrder(inputFreq, inputSpec)
    if (k == 'linear' or k=='nearest' or k=='zero' or k=='slinear' or k=='quadratic' or k=='cubic'):
        myfunc = scipy.interpolate.interp1d(inputFreq,inputSpec, kind=k,
                                            bounds_error=bounds_error, fill_value=fill_value)
        fdmSpectrum = myfunc(outputFreq)
    elif (type(k) == str):
        print("Invalid value for k: must be an integer, or one of the following strings:")
        print("  'linear', 'nearest', 'zero', 'slinear', 'quadratic', 'cubic' ")
        return
    else:
        if (len(tmpFreq) <= k):
            k = len(tmpFreq)-1
            if verbose: print("Reducing spline order to %d due to too few data points (%d)" % (k,len(tmpFreq)))
        tck = splrep(tmpFreq, tmpData, s=0, k=k)
        fdmSpectrum = splev(outputFreq, tck, der=0)
    return(fdmSpectrum)

def multiSingleDishSpectrum(vis,antenna=0,spw=None,field=None,offstate=None,
                            scan=None,pol=0,intent='OBSERVE_TARGET#ON_SOURCE',asdm=None,
                            tsystable=None, plotfile='', ut='', showMeanIntegration=True,
                            showMaxIntegration=False,plotEveryIntegration=False, ylimits=[0,0],
                            scaleFactor=1.0, smoothing=1, cleanup=True, onstate=None,
                            tsysvis=None,tsysspw=None,verbose=False,xlimits=[0,0],
                            xlimitsFdm=[0,0],removeMedian=False):
    """
    A test function to try alternative SD calibration equations.
    -Todd Hunter
    """
    tcal = []
    sd = []
    tsys = []
    meanTcal = []
    atmcal = None
    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)
    antennaName = mymsmd.antennanames(antenna)[0]
    mymsmd.close()
    if (onstate is None):
        # find all the ON states
        a = singleDishSpectrum(vis,antenna,spw,field,offstate,
                               scan, pol, intent, asdm,
                               tsystable, plotfile, ut, showMeanIntegration,
                               showMaxIntegration, plotEveryIntegration, ylimits,
                               scaleFactor, smoothing, cleanup, onstate, atmcal=atmcal,
                               tsysvis=tsysvis,tsysspw=tsysspw,verbose=verbose,
                               xlimits=xlimits,xlimitsFdm=xlimitsFdm)
        onstate = a[5]
        attenuatorCorrectionFactor = a[7]
        scaleFactor = a[8]
        alpha = a[9]
        alphaFdm = a[10]
        print("all onstates = ", onstate)
    for state in onstate:
        result = singleDishSpectrum(vis,antenna,spw,field,offstate,
                                    scan, pol, intent, asdm,
                                    tsystable, plotfile, ut, showMeanIntegration,
                                    showMaxIntegration, plotEveryIntegration, ylimits,
                                    scaleFactor, smoothing, cleanup,onstate=state,atmcal=atmcal,
                                    attenuatorCorrectionFactor=attenuatorCorrectionFactor,
                                    tsysvis=tsysvis,tsysspw=tsysspw,verbose=verbose,
                                    xlimits=xlimits,xlimitsFdm=xlimitsFdm,alpha=alpha,alphaFdm=alphaFdm)
                                    
        freq, tcalMethod, sdimagingMethod, meanTsysMethod, meanTcalMethod, allOnStates, atmcal, attenuatorCorrectionFactor, scaleFactor, alpha, alphaFdm = result
        tcal.append(tcalMethod)
        sd.append(sdimagingMethod)
        tsys.append(meanTsysMethod)
        meanTcal.append(meanTcalMethod)
    tcalMean = np.mean(tcal,axis=0)
    sdMean = np.mean(sd,axis=0)
    tsysMean = np.mean(tsys,axis=0)
    meanTcalMean = np.mean(meanTcal,axis=0)
    pb.clf()
    pb.subplots_adjust(hspace=0.20)
    pb.subplots_adjust(wspace=0.20)
    adesc = pb.subplot(221)
    print(np.shape(freq), np.shape(tcalMean))
    mysize = 12
    if (tsysvis is None):
        pb.text(0,1.15,'%s %s'%(vis,antennaName), size=mysize, transform=adesc.transAxes)
    else:
        pb.text(0,1.15,'%s %s (Tsys=%s)'%(vis,antennaName,tsysvis), size=mysize-1, transform=adesc.transAxes)
    pb.plot(freq, tcalMean, 'k-')
    if (removeMedian):
        tcalMean -= computeYStatsForXLimits(freq, tcalMean, xlimits)['median']
        meanTcalMean -= computeYStatsForXLimits(freq, meanTcalMean, xlimits)['median']
        sdMean -= computeYStatsForXLimits(freq, sdMean, xlimits)['median']
        tsysMean -= computeYStatsForXLimits(freq, tsysMean, xlimits)['median']
    ylimits = np.array(computeYLimitsForXLimits(freq,sdMean,xlimits))*1.1
    pb.xlim(xlimits)
    pb.ylim(ylimits)
    pb.title('spectral Tcal method', size=mysize)
    ticksize = 10
    resizeFonts(adesc, ticksize)

    adesc = pb.subplot(222)
    pb.plot(freq, meanTcalMean, 'k-')
    pb.xlim(xlimits)
    pb.ylim(ylimits)
    pb.title('mean Tcal method', size=mysize)
    resizeFonts(adesc, ticksize)

    adesc = pb.subplot(223)
    pb.plot(freq, sdMean, 'k-')
    pb.xlim(xlimits)
    pb.ylim(ylimits)
    pb.title('spectral Tsys method (sdimaging)', size=mysize)
    pb.xlabel('Sky frequency (GHz)')
    resizeFonts(adesc, ticksize)

    adesc = pb.subplot(224)
    pb.plot(freq, tsysMean, 'k-')
    pb.xlim(xlimits)
    pb.ylim(ylimits)
    pb.xlabel('Sky frequency (GHz)')
    pb.title('mean Tsys method', size=mysize)
    resizeFonts(adesc, ticksize)

    pb.savefig('tcal_vs_sdimaging.png')

def computeYStatsForXLimits(x, y, xlimits):
    """
    Computes the Y-axis statistics (mean, std, median, min, max) over the 
    specified x-axis range, rather than over the whole x-axis range.  
    Returns a dictionary.
    -Todd Hunter
    """
    idx0 = np.where(x >= xlimits[0])[0]
    idx1 = np.where(x <= xlimits[1])[0]
    idx = np.intersect1d(idx1,idx0)
    if (len(idx) < 1):
        print("No points match the x-range.  Using entire range.")
        idx = range(len(y))
    return({'mean': scipy_nanmean(y[idx]), 'std': scipy_nanstd(y[idx]), 'median': scipy_nanmedian(y[idx]),
            'min': np.nanmin(y[idx]), 'max': np.nanmax(y[idx])})

def computeYLimitsForXLimits(x, y, xlimits, verbose=False):
    """
    Computes the Y-axis limits to autorange over the specified x-axis range, rather
    than over the whole x-axis range, which is what pylab plot does by default.
    -Todd Hunter
    """
    mydict = computeYStatsForXLimits(x, y, xlimits)
    if (verbose):
        print("full ylimits: %f,%f   within x-range: %f,%f" % (np.nanmin(y), np.nanmax(y), mydict['min'], mydict['max']))
    ylimits = [mydict['min'], mydict['max']]
    return(ylimits)
                            
def singleDishSpectrum(vis,antenna=0,spw=None,field=None,offstate=None,
                       scan=None,pol=0,intent='OBSERVE_TARGET#ON_SOURCE',asdm=None,
                       tsystable=None, plotfile='', ut='', showMeanIntegration=False,
                       showMaxIntegration=False,plotEveryIntegration=False, ylimits=[0,0],
                       scaleFactor=1.0, smoothing=1, cleanup=True,onstate=None,
                       atmcal=None, attenuatorCorrectionFactor=None,tsysvis=None,
                       tsysspw=None,verbose=False,xlimits=[0,0],xlimitsFdm=[0,0],
                       alpha=None,alphaFdm=None):
    """
    A test function to try alternative SD calibration equations.
    vis: the measurement set containing the ON, OFF data
    tsysvis: the measurement set containing the Tsys scan to use
    asdm: the dataset corresponding to tsysvis
    antenna = antenna number as integer
    spw: spw number as integer
    field: field number as integer
    onstate: integer STATE_ID for the on source integration
    offstate: integer STATE_ID for the off source integration
    pol: 0 or 1
    ut: a string representing UT time '12:00:00' (date will be prepended)
    showMaxIntegration: if True, show the single integration with largest mean
                        if False, show the single integration closest to the requested UT
    showMeanIntegration: if True, show the mean of all integrations in a row/state_id,
                         if False, show the single integration closest to the requested UT
    plotEveryIntegration: make a plot for every dump rather than the mean, and build PDF
    scaleFactor: the factor by which to multiply Robert's formula
    cleanup: if True, remove the png files if multiple ones were made
    attenuatorCorrectionFactor: the value to multiply the ON and OFF spectrum by
    xlimits: the x-axis limits for the Tcal/Tsys spectra (in GHz)
    xlimitsFdm: the x-axis limits for the FDM data spectra (in GHz)
    Returns: 10 items:
      chanfreqs, spectrum using Tcal method, spectrum using Tsys method, spectrum using mean Tsys
      method, list of all on-source state IDs, atmcal instance, attenuatorCorrectionFactor,
      scaleFactor, alpha, alphaFdm
    
    Todd Hunter
    """
    if (type(onstate) == list):
        print("onstate must be an integer, not a list")
        return
    if (asdmLibraryAvailable == False):
        print("The ASDM bindings library is not available on this machine.")
        return
    from almahelpers_localcopy import tsysspwmap
    if (tsysvis is None):
        tsysvis = vis
    if (asdm is None):
        asdm = tsysvis.split('.')[0]
    if (tsystable is None):
        tsystable = tsysvis+'.tsys'
    if (os.path.exists(tsystable) == False):
        print("Running gencal to create the tsys caltable.")
        gencal(tsysvis, caltype='tsys', caltable=tsystable)
    if (atmcal is None):
        atmcal = Atmcal(tsysvis)
    mymsmd = createCasaTool(msmdtool)
    mjdsec = 0
    if (ut != ''):
        mydate = getObservationStartDate(vis).split()[0]
        mydate = mydate + ' ' + ut
        mjdsec = dateStringToMJDSec(mydate)
        print("Finding closest data to %s = %f" % (mydate, mjdsec))
    mymsmd.open(vis)
    antennaName = mymsmd.antennanames(antenna)[0]
    if (spw is None):
        spws = spwsforintent_nonwvr_nonchanavg(mymsmd, intent)
        if (len(spws) < 1):
            print("No spws match this intent")
            return
        spw = spws[0]
        print("Picking spw = %d" % (spw))
    datadescid = mymsmd.datadescids(spw=spw)[0]
    if (field is None):
        fields = mymsmd.fieldsforintent(intent)
        if (len(fields) < 1):
            print("No fields match this intent")
            return
        field = fields[0]
        fieldName = mymsmd.namesforfields(field)[0]
        print("Picking field = %d = %s" % (field, fieldName))
    fieldName = mymsmd.namesforfields(field)[0]
    if (scan is None):
        scans = np.intersect1d(mymsmd.scansforintent(intent), mymsmd.scansforfield(field))
        if (len(scans) < 1):
            print("No scans match this intent+field")
            return
        scan = scans[0]
        print("Picking data scan = ", scan)
    chanfreqs = mymsmd.chanfreqs(spw) * 1e-9
    meantime = np.mean(mymsmd.timesforscan(scan))
    mymsmd.close()
    atmcalscan = atmcal.nearestCalScan(meantime)
    print("Picking ATMCal scan = ", atmcalscan)
    if (tsysspw is None):
        tcalSpw = tsysspwmap(vis, tsystable)[spw]
    else:
        tcalSpw = tsysspw
        if (tcalSpw not in list(atmcal.datadescids.keys())):
            print("spw %d is not an ATMCal spw in %s" % (tcalSpw, tsysvis))
            return
    skyLoad = atmcal.getSpectrum(atmcalscan, tcalSpw, pol, 'sky', antenna)
    ambLoad = atmcal.getSpectrum(atmcalscan, tcalSpw, pol, 'amb', antenna)
    hotLoad = atmcal.getSpectrum(atmcalscan, tcalSpw, pol, 'hot', antenna)
    if (verbose):
        print("Calling readTcal('%s', %d, %d, %f)" % (asdm, antenna, tcalSpw,meantime))
    tcal, tcalTime = readTcal(asdm, antenna, tcalSpw, meantime, verbose=verbose)
    if (verbose):
        print("Calling readTcal('%s', %d, %d, %f, spectrum='tsys')" % (asdm, antenna, tcalSpw,meantime))
    tsys, tsysTime = readTcal(asdm, antenna, tcalSpw, meantime, spectrum='tsys', verbose=verbose)
    tcal = tcal[pol]
    tsys = tsys[pol]
    mymsmd.open(tsysvis)
    tcalChanfreqs = mymsmd.chanfreqs(tcalSpw) * 1e-9
    mymsmd.close()
    if (verbose):
        print("%d chanfreqs: %f-%f,  %d tcalChanfreqs: %f-%f" % (len(chanfreqs),chanfreqs[0],chanfreqs[-1],
                                                                 len(tcalChanfreqs),tcalChanfreqs[0],tcalChanfreqs[-1]))
    tcalFdm = interpolateSpectrum(tcalChanfreqs, tcal, chanfreqs)
    tsysFdm = interpolateSpectrum(tcalChanfreqs, tsys, chanfreqs)
    meanTsysFdm = computeYStatsForXLimits(chanfreqs,tsysFdm,xlimits)['mean']
    skyLoadFdm = interpolateSpectrum(tcalChanfreqs, skyLoad, chanfreqs)
    ambLoadFdm = interpolateSpectrum(tcalChanfreqs, ambLoad, chanfreqs)
    hotLoadFdm = interpolateSpectrum(tcalChanfreqs, hotLoad, chanfreqs)
    
    mytb = createCasaTool(tbtool)
    mytb.open(vis)
    allstates = mytb.getcol('STATE_ID')
    myt = mytb.query('ANTENNA1==%d and ANTENNA2==%d and DATA_DESC_ID==%d and FIELD_ID==%d and SCAN_NUMBER==%d' % (antenna,antenna,datadescid,field,scan))
    rows = myt.rownumbers()
    times = myt.getcol('TIME')
    print("%d matching rows: %s" % (len(rows), str(rows)))
    if (len(rows) == 0):
        return
    states = myt.getcol('STATE_ID')
    myt.close()
    mytb.close()

    mytb.open(vis+'/STATE')
    obsMode = mytb.getcol('OBS_MODE')
    allOnStates = []
    if (onstate is None):
        if (mjdsec > 0):
            # Pick the closest row in time
            mindiff = 1e20
            for row in range(len(rows)):
                if (obsMode[allstates[rows[row]]].find('ON_SOURCE') >= 0):
                    deltaTime = abs(times[row]-mjdsec)
                    if (deltaTime < mindiff):
                        mindiff = deltaTime
#                        print "New mindiff on = ", mindiff
                        onstate = states[row]
                        onstateTime = times[row]
        else:            
            # Pick the first row that contains ON_SOURCE
            for row in range(len(rows)):
                if (obsMode[allstates[rows[row]]].find('ON_SOURCE') >= 0):
                    onstate = states[row]
                    onstateTime = times[row]
#                    print "onstate = ", onstate
                    break
    else:
        row = np.where(states == onstate)[0][0]
        onstateTime = times[row]
    # Just do this so I can return the value for other usage.
    for myrow in range(len(rows)):
        if (obsMode[allstates[rows[myrow]]].find('ON_SOURCE') >= 0):
            allOnStates.append(states[myrow])
    on_intents = obsMode[allstates[rows[row]]]
    print("on source intents = %s" % (str(on_intents)))
    if (offstate is None):
        if (mjdsec > 0):
            # Pick the closest row in time
            mindiff = 1e20
            for row in range(len(rows)):
                if (obsMode[allstates[rows[row]]].find('OFF_SOURCE') >= 0):
                    deltaTime = abs(times[row]-mjdsec)
                    if (deltaTime < mindiff):
                        mindiff = deltaTime
                        offstate = states[row]
                        offsourceRow = row
        else:            
            # Pick the row closest to the ON_SOURCE in time
            mindiff = 1e20
            for row in range(len(rows)):
                if (obsMode[allstates[rows[row]]].find('OFF_SOURCE') >= 0):
                    deltaTime = abs(times[row]-onstateTime)
                    if (deltaTime < mindiff):
                        mindiff = deltaTime
                        offstate = states[row]
                        offsourceRow = row
    else:
        row = np.where(states == offstate)[0][0]
        offsourceRow = row

    row = offsourceRow
    off_intents = obsMode[allstates[rows[row]]]
    print("offsource row = ", row)
    print("offsource rows[row] = ", rows[row])
    print("allstates[rows[row]] = ", allstates[rows[row]])    
    print("offstate = ", offstate)
    print("offsource intents = %s" % (str(off_intents)))
    v = Visibility(vis, antenna, antenna, spw, field, onstate, scan, cross_auto_all='auto')

    if (plotEveryIntegration):
        dumps = range(len(v.specTime))
        on_spectrum = []
        for dump in dumps:
            on_spectrum.append(np.abs(v.specData[pol,:,dump]))
        on_time = v.specTime[:]
        on_duration = v.specTime[1] - v.specTime[0]
    elif (showMeanIntegration == False):
        if (showMaxIntegration):
            peakspec = -1e10
            for s in range(len(v.specData[pol][0])):
                spec = v.specData[pol,:,s]
                meanspec = np.mean(spec)
                if (meanspec > peakspec):
                    peakspec = meanspec
                    pick = s
        elif (mjdsec > 0):
            # pick the one closest to the requested UT time
            nearestTime = 1e10
            for s in range(len(v.specData[pol][0])):
                deltaTime = abs(v.specTime[s]-mjdsec)
                if (deltaTime < nearestTime):
                    nearestTime = deltaTime
                    pick = s
        else:
            # Just pick the first one
            pick = 0
        on_spectrum = [np.abs(v.specData[pol,:,pick])]
        on_time = [v.specTime[pick]]
        if (pick == 0):
            on_duration = v.specTime[pick+1] - v.specTime[pick]
        else:
            on_duration = v.specTime[pick] - v.specTime[pick-1]
        dumps = [0]
    else:
        dumps = [0]
        on_spectrum = [np.mean(v.specData[pol],axis=1)]
        on_time = [np.mean(v.specTime)]
        on_duration = np.max(v.specTime) - np.min(v.specTime)
            
    v.setState(offstate)
    off_time = np.mean(v.specTime)
    off_spectrum = np.mean(v.specData[pol],axis=1)
    off_duration = np.max(v.specTime) - np.min(v.specTime)
    off_spectrum = abs(off_spectrum)
    
    channels = np.intersect1d(np.where(tcalChanfreqs >= np.min(chanfreqs))[0], np.where(tcalChanfreqs <= np.max(chanfreqs))[0])
    print("Channels to average = %d-%d" % (channels[0], channels[-1]))
    if (attenuatorCorrectionFactor == None):
        attenuatorCorrectionFactor = np.mean(skyLoad[channels]) / np.mean(off_spectrum)  
    off_spectrum *= attenuatorCorrectionFactor
    print("TDM/FDM plus attenuator correction factor = %f = %f dB" % (attenuatorCorrectionFactor, 10*np.log10(attenuatorCorrectionFactor)))
    if (alpha is None):
        jSky, jAmb, jHot, frequency, jatmDSB, jspDSB, jbg, tauA, alpha, gb, tebbsky = atmcal.computeJs(atmcalscan, antenna, pol, tcalSpw, computeJsky=True)
        alphaFdm =  interpolateSpectrum(tcalChanfreqs, np.array(alpha), chanfreqs)
    mysize = 10
    if (plotEveryIntegration):
        rows = 2
        cols = 1
        ticksize = 9
    else:
        rows = 3
        cols = 3
        ticksize = 7
    pnglist = []
    for dump in dumps:
      on_spectrum[dump] = abs(on_spectrum[dump]) * attenuatorCorrectionFactor
      pb.clf()
      pb.subplots_adjust(hspace=0.30)
      pb.subplots_adjust(wspace=0.40)
      scale = 1e8
      if (xlimits == [0,0]):
          xlimits = [np.min([np.min(tcalChanfreqs),np.min(chanfreqs)]), np.max([np.max(tcalChanfreqs),np.max(chanfreqs)])]
      if (xlimitsFdm == [0,0]):
          xlimitsFdm = [np.min(chanfreqs), np.max(chanfreqs)]
      if (plotEveryIntegration == False):
        adesc = pb.subplot(rows,cols,5)
        pb.plot(chanfreqs, on_spectrum[dump]/scale, 'k-', chanfreqs, off_spectrum/scale, 'r-')
        pb.xlim(xlimitsFdm)
        resizeFonts(adesc, ticksize)
        pb.ylabel('Raw data / %.0e'%(scale),size=mysize)
        print("Mean ON-OFF = %f,  Mean ON/OFF = %f" % (np.mean(on_spectrum[dump]-off_spectrum),
                                                       np.mean(on_spectrum[dump])/np.mean(off_spectrum)))
        pb.title('ON=black, OFF=red',size=mysize)
        
      if (plotEveryIntegration == False):
        adesc = pb.subplot(rows,cols,1)
        pb.plot(tcalChanfreqs, tcal, 'k-', chanfreqs, tcalFdm, 'r-')
        pb.xlim(xlimits)
        resizeFonts(adesc, ticksize)
#        pb.title('Tcal at %s (spw %d, %d)' % (mjdsecToUTHMS(tcalTime), tcalSpw, spw), size=mysize)
        pb.title('Tcal at %s (spw %d, %d)' % (mjdsecToUT(tcalTime), tcalSpw, spw), size=mysize)
        pb.ylabel('Temperature (K)',size=mysize)
        pb.text(-0.10*cols,1.17,vis + ', ant%02d=%s, spw=%02d, field=%d=%s, scan=%d, onstate=%d, offstate=%d'%(antenna, antennaName, spw, field, fieldName, scan, onstate, offstate), transform=adesc.transAxes, size=mysize)
    
      if (plotEveryIntegration == False):
        adesc = pb.subplot(rows,cols,2)
        pb.plot(tcalChanfreqs, tsys, 'k-', chanfreqs, tsysFdm, 'r-')
        pb.xlim(xlimits)
        pb.text(0.1,0.85,'mean=%.1f' % (meanTsysFdm),size=mysize+1-cols,transform=adesc.transAxes)
        resizeFonts(adesc, ticksize)
        pb.title('Tsys',size=mysize) # at %s (spw %d, %d)' % (mjdsecToUTHMS(tcalTime), tcalSpw, spw), size=mysize)
        pb.ylim(computeYLimitsForXLimits(tcalChanfreqs, tsys, xlimits))
        pb.ylabel('Temperature (K)',size=mysize)
        if ((rows-1)*cols < 3):
            pb.xlabel('Sky frequency (GHz)',size=mysize)
    
      if (plotEveryIntegration == False):
        adesc = pb.subplot(rows,cols,4)
        pb.plot(tcalChanfreqs, ambLoad/scale,'k-', tcalChanfreqs, hotLoad/scale,'k-',
                chanfreqs, ambLoadFdm/scale,'r-', chanfreqs, hotLoadFdm/scale,'r-',
                tcalChanfreqs, skyLoad/scale, 'k-', chanfreqs, skyLoadFdm/scale,'r-')
        pb.ylim([np.min(ambLoad/scale), np.max(hotLoad/scale)])
        resizeFonts(adesc, ticksize)
        pb.xlim(xlimits)
        pb.title('Ambient, Hot load, Sky', size=mysize)
        pb.ylabel('Raw data / %.0e'%scale,size=mysize)
        pb.text(0.05,0.28, 'C=mean(skyLoad/OFF)=%.2f' % (attenuatorCorrectionFactor),
                size=7, transform=adesc.transAxes)
    
      if (plotEveryIntegration == False and False):
        adesc = pb.subplot(rows,cols,5)
        yaxis = skyLoad/scale
        pb.plot(tcalChanfreqs, yaxis, 'k-', chanfreqs, skyLoadFdm/scale,'r-')
        resizeFonts(adesc, ticksize)
        pb.xlim(xlimits)
        pb.ylim(computeYLimitsForXLimits(tcalChanfreqs, yaxis, xlimits))
        pb.title('Sky load', size=mysize)
        pb.text(0.05,0.88, 'C=mean(skyLoad/OFF)=%.2f' % (attenuatorCorrectionFactor),
                size=7, transform=adesc.transAxes)
        pb.ylabel('Raw data / %.0e'%scale,size=mysize)
    
      if (plotEveryIntegration == False):
        adesc = pb.subplot(rows,cols,3)
        pb.plot(tcalChanfreqs, alpha,'k-', chanfreqs, alphaFdm, 'r-')
        resizeFonts(adesc, ticksize)
        pb.xlim(xlimits)
        pb.title('Alpha (red=interpolated to FDM)', size=mysize)
        pb.ylabel('Unitless',size=mysize)
    
      if (plotEveryIntegration == False):
          panel = 8
      else:
          panel = 1
      adesc = pb.subplot(rows,cols,panel)
      if (plotEveryIntegration):
          pb.text(-0.10*cols,1.17,vis + ', ant%02d=%s, spw=%02d, field=%d=%s, scan=%d, onstate=%d, offstate=%d'%(antenna, antennaName, spw, field, fieldName, scan, onstate, offstate), transform=adesc.transAxes, size=mysize)
          pb.text(-0.08*cols,-0.08*rows,'ON: %s (%.2f sec duration)  OFF: %s (%.2f sec duration)' % (mjdsecToUTHMS(on_time[dump],prec=8), on_duration,
                                                                                                mjdsecToUTHMS(off_time,prec=8), off_duration),
                  transform=adesc.transAxes, size=mysize)
          pb.text(0.9,-0.08*rows, '%2d/%d'%(dump+1,len(dumps)), transform=adesc.transAxes, size=mysize)
      print("mean(alpha) = %f, mean(tcalFdm)=%f" % (np.mean(alphaFdm), np.mean(tcalFdm)))
      print("mean(ambLoad) = %e, mean(hotLoad) = %e, mean(OFF) = %e" % (np.mean(ambLoadFdm), np.mean(hotLoadFdm), np.mean(off_spectrum)))
      onoff_sdimaging = tsysFdm * (on_spectrum[dump] - off_spectrum) / off_spectrum
      onoff_meanTsys = meanTsysFdm * (on_spectrum[dump] - off_spectrum) / off_spectrum
      onoff_cso = atmcal.loadTemperatures[antenna][atmcalscan]['amb'] * (on_spectrum[dump] - off_spectrum) / (ambLoadFdm - off_spectrum)
      if (True):
          # original posting to CSV-2986
          onoff = scaleFactor * tcalFdm * (on_spectrum[dump] - off_spectrum) / ((1-alphaFdm)*hotLoadFdm + alphaFdm*ambLoadFdm - off_spectrum)
          onoff_meanTcal = scaleFactor * np.mean(tcalFdm) * (on_spectrum[dump] - off_spectrum) / ((1-alphaFdm)*hotLoadFdm + alphaFdm*ambLoadFdm - off_spectrum)
          if (scaleFactor != 1.0):
              pb.title('%g*C*Tcal*(ON-OFF)/((1-alpha)*HOT+alpha*AMB-C*OFF)' % (scaleFactor), size=8)
          else:
              scaleFactor = computeYStatsForXLimits(chanfreqs,onoff_sdimaging,xlimits)['mean']/computeYStatsForXLimits(chanfreqs,onoff,xlimits)['mean']
              scaleFactor = np.round(scaleFactor*10)/10.0
              pb.title('C*Tcal*(ON-OFF)/((1-alpha)*HOT+alpha*AMB-C*OFF)', size=8)
      else:
          onoff = tcalFdm * (on_spectrum[dump] - off_spectrum) / ((1-alphaFdm)*ambLoadFdm + alphaFdm*hotLoadFdm - off_spectrum) 
          pb.title('C*Tcal*(ON-OFF)/((1-alpha)*AMB+alpha*HOT-C*OFF)', size=8)
      if (smoothing > 1):
          onoff = smooth(onoff, window_len=smoothing, window='flat')
          onoff_cso = smooth(onoff_cso, window_len=smoothing, window='flat')
          onoff_meanTcal = smooth(onoff_meanTcal, window_len=smoothing, window='flat')
          onoff_sdimaging = smooth(onoff_sdimaging, window_len=smoothing, window='flat')
          onoff_meanTsys = smooth(onoff_meanTsys, window_len=smoothing, window='flat')

      pb.plot(chanfreqs, onoff,'k-')
      pb.text(0.1,0.9,'mean=%.2f, std=%.3f'%(computeYStatsForXLimits(chanfreqs,onoff,xlimits)['mean'],
                                             computeYStatsForXLimits(chanfreqs,onoff,xlimits)['std']),
                                             size=mysize+1-cols, transform=adesc.transAxes)
      resizeFonts(adesc, ticksize)
      pb.xlim(xlimitsFdm)
      if (ylimits == [0,0] or ylimits == []):
          ylimits = [np.min([np.min(onoff_sdimaging), np.min(onoff)]), np.max([np.max(onoff_sdimaging),np.max(onoff)])]
          ylimits = [ylimits[0]-0.1*ylimits[1], ylimits[1]*1.1]
          if (ylimits[0] > 0): ylimits[0] = 0
      pb.ylim(ylimits)
      pb.ylabel('Temperature (K)',size=mysize)

      if (plotEveryIntegration == False):
        adesc = pb.subplot(rows,cols,6)
        # CSO method
        pb.plot(chanfreqs, onoff_cso,'k-')
        pb.title('CSO method = Tamb*(ON-OFF)/(AMB-SKY)', size=mysize-1)
        pb.ylabel('Temperature (K)',size=mysize)
        resizeFonts(adesc, ticksize)
        pb.xlim(xlimitsFdm)
        pb.ylim(ylimits)
        pb.text(0.1,0.9,'mean=%.2f, std=%.3f'%(computeYStatsForXLimits(chanfreqs,onoff_cso,xlimits)['mean'],
                                               computeYStatsForXLimits(chanfreqs,onoff_cso,xlimits)['std']),
                size=mysize+1-cols, transform=adesc.transAxes)
        pb.text(0.1,0.1,'Tamb=%.2fK'%(atmcal.loadTemperatures[antenna][atmcalscan]['amb']),
                size=mysize+1-cols, transform=adesc.transAxes)
    
      if (plotEveryIntegration == False):
        adesc = pb.subplot(rows,cols,7)
        if (False):
            onoff = (on_spectrum[dump] - off_spectrum) / off_spectrum
            pb.plot(chanfreqs, onoff,'k-')
            pb.title('(ON-OFF)/OFF', size=mysize)
            pb.ylabel('Raw data',size=mysize)
            pb.text(0.1,0.9,'mean=%.2f, std=%.3f'%(computeYStatsForXLimits(chanfreqs,onoff,xlimits)['mean'],
                                                   computeYStatsForXLimits(chanfreqs,onoff,xlimits)['std']),
                    size=mysize+1-cols, transform=adesc.transAxes)
        else:
            pb.plot(chanfreqs, onoff_meanTsys, 'k-')
            pb.title('mean(Tsys)*(ON-OFF)/OFF', size=mysize-2)
            pb.ylim(ylimits)
            pb.ylabel('Temperature (K)',size=mysize)
            pb.text(0.1,0.9,'mean=%.2f, std=%.3f'%(computeYStatsForXLimits(chanfreqs,onoff_meanTsys,xlimits)['mean'],
                                                   computeYStatsForXLimits(chanfreqs,onoff_meanTsys,xlimits)['std']),
                    size=mysize+1-cols, transform=adesc.transAxes)
        resizeFonts(adesc, ticksize)
        pb.xlim(xlimitsFdm)
        pb.text(-0.08*cols,-0.08*rows,'ON: %s (%.2f sec duration)  OFF: %s (%.2f sec duration)' % (mjdsecToUTHMS(on_time[dump],prec=8), on_duration,
                                                                                     mjdsecToUTHMS(off_time,prec=8), off_duration),
                transform=adesc.transAxes, size=mysize)
    
      adesc = pb.subplot(rows,cols,panel+1)
      pb.plot(chanfreqs, onoff_sdimaging,'k-')
      pb.text(0.1,0.9,'mean=%.2f, std=%.3f'%(computeYStatsForXLimits(chanfreqs,onoff_sdimaging,xlimits)['mean'],
                                             computeYStatsForXLimits(chanfreqs,onoff_sdimaging,xlimits)['std']),
              size=mysize+1-cols, transform=adesc.transAxes)
      resizeFonts(adesc, ticksize)
      pb.xlim(xlimitsFdm)
      pb.ylim(ylimits)
      pb.title('Tsys*(ON-OFF)/OFF', size=mysize)
      pb.ylabel('Temperature (K)',size=mysize)
      pb.xlabel('Sky frequency (GHz)',size=mysize)
      print("mean of sdimaging / Robert = ", computeYStatsForXLimits(chanfreqs,onoff_sdimaging,xlimits)['mean']/computeYStatsForXLimits(chanfreqs,onoff,xlimits)['mean'])
      pb.draw()
      if (plotfile != ''):
          if (plotfile == True):
              if (len(dumps) > 1):
                  png = vis+'.%s.%s.spw%02d.pol%d.dump%02d.png' % (fieldName,antennaName,spw,pol,dump)
                  pb.savefig(png)
                  pnglist.append(png)
              else:
                  pb.savefig(vis+'.%s.%s.spw%02d.pol%d.png' % (fieldName,antennaName,spw,pol))
          else:
              pb.savefig(plotfile)
    if (len(pnglist) > 0):
        if (smoothing > 1):
            pdf = vis+'.%s.%s.spw%02d.pol%d.pdf' % (fieldName,antennaName,spw,pol)
        else:
            pdf = vis+'.%s.%s.spw%02d.pol%d.smooth%g.pdf' % (fieldName,antennaName,spw,pol,smoothing)
        buildPdfFromPngs(pnglist, pdf)
        if (cleanup):
            for png in pnglist:
                os.system('rm -f %s' % png)
    else:
        if (len(allOnStates) > 0):
            print("on states = ", np.unique(allOnStates))
        return(chanfreqs, onoff, onoff_sdimaging, onoff_meanTsys, onoff_meanTcal,
               np.unique(allOnStates), atmcal, attenuatorCorrectionFactor,scaleFactor,
               alpha, alphaFdm)

def getAntennaIndex(msFile,antennaName) :
    """
    Returns the index number of the specified antenna in the specified ms.
    The antennaName must be a string, e.g. DV01.  Obsoleted by msmd.antennaids('DV01')[0].
    """
    if str(antennaName).isdigit() : 
        return antennaName
    else :
        ids = getAntennaNames(msFile)
        if (antennaName not in ids):
            print("Antenna %s is not in the dataset.  Available antennas = %s" % (antennaName, str(ids)))
            return(-1)
        return np.where(ids == antennaName)[0][0]

def getAntennaName(msFile,antennaID) :
    """
    Returns the antenna name of the specified antenna in the specified ms.
    The antennaID must be an integer.Obsoleted by msmd.antennanames(1)[0].
    Todd Hunter
    """
    names = getAntennaNames(msFile)
    if (antennaID >= 0 and antennaID < len(names)):
        return (names[antennaID])
    else:
        print("This antenna ID (%d) is not in the ms." % (antennaID))
        return ('')

def commonAntennas(vislist, returnPads=False, returnDict=False):
    """
    Returns a list of the common antenna names between two (or more) datasets, or
    between all the obsids of a single multi-obsid dataset.
    vis: either a comma-delimited string, or a python list of strings
    returnPads: if True, then return the list of pads that are common between these datasets
    returnDict: if True, then return dictionary keyed by vis name, with values being a comma-delimited
             string of antenna names that occupy the subset of pads that is common to all datasets
             (does not yet work for the case of a single multi-obsid dataset)
    -Todd Hunter
    """
    if (type(vislist) == str):
        if vislist.find('*') >= 0:
            vislist = sorted(glob.glob(vislist))
        else:
            vislist = vislist.split(',')
    vis = vislist[0].strip()
    if (len(vislist) == 1):
        mytb = createCasaTool(tbtool)
        mytb.open(vis)
        obsids = np.unique(mytb.getcol('OBSERVATION_ID'))
        mytb.close()
        if returnPads or returnDict:
            print("Finding pads common to %d obsids" % (len(obsids)))
            common = getAntennaPadsForObsID(vis, obsids[0])
            for obsid in obsids[1:]:
                common = np.intersect1d(common, getAntennaPadsForObsID(vis, obsid))
        else:
            print("Finding antennas common to %d obsids" % (len(obsids)))
            common = getAntennaNamesForObsID(vis, obsids[0])
            for obsid in obsids[1:]:
                common = np.intersect1d(common, getAntennaNamesForObsID(vis, obsid))
    else:
        if returnPads or returnDict:
            common = getAntennaPads(vis)
            if returnDict:
                antennaPads = {}
                antennaNames = {}
                antennaPads[vis] = list(common)
                antennaNames[vis] = np.array(getAntennaNames(vis))
        else:
            common = getAntennaNames(vislist[0].strip())
        for vis in vislist[1:]:
            vis = vis.strip()
            if returnPads or returnDict:
                ants = getAntennaPads(vis)
                if returnDict:
                    antennaPads[vis] = list(ants)
                    antennaNames[vis] = np.array(getAntennaNames(vis))
            else:
                ants = getAntennaNames(vis)
            common = np.intersect1d(common,ants)
    if returnDict and len(vislist) > 1:
        mydict = {}
        for vis in vislist:
            mylist = [antennaPads[vis].index(pad) for pad in common]
            mydict[vis] = ','.join(sorted(antennaNames[vis][mylist]))
        return(mydict)
    else:
        return(list(common))

def getAntennaNamesForObsID(vis, obsid=0):
    """
    Gets the list of antenna names associated to a specific obsid.
    -Todd 
    """
    if (not os.path.exists(vis)):
        print("Could not find measurement set")
        return
    mytb = createCasaTool(tbtool)
    mytb.open(vis)
    obsids = np.unique(mytb.getcol('OBSERVATION_ID'))
    if (obsid not in obsids):
        print("obsid=%d is not in the dataset, which contains obsids: %s" % (obsid, str(obsids)))
        mytb.close()
        return
    t = mytb.query('OBSERVATION_ID == %d'%obsid)
    ant1 = np.unique(t.getcol('ANTENNA1'))
    ant2 = np.unique(t.getcol('ANTENNA2'))
    t.close()
    ids = np.union1d(ant1,ant2)
    mytb.close()
    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)
    names = list(mymsmd.antennanames(ids))
    mymsmd.close()
    return(names)

def getAntennaNames(msFile) :
    """
    Returns the list of antenna names in the specified ms ANTENNA table using tbtool.
    Obsoleted by msmd.antennanames(range(msmd.nantennas())), but kept because it is faster.
    """
    if (msFile.find('*') >= 0):
        mylist = glob.glob(msFile)
        if (len(mylist) < 1):
            print("getAntennaNames: Could not find measurement set.")
            return
        msFile = mylist[0]
    mytb = createCasaTool(tbtool)
    mytb.open(msFile+'/ANTENNA')
    names = mytb.getcol('NAME')
    mytb.close()
    return names

def convertTimeStamps(timesIn) :
    """
    Converts a list of makeTimeStamp strings to a list of Julian day numbers
    """
    timesIn = makeList(timesIn)
    timesOut = []
    for i in timesIn :
        timesOut.append(convertTimeStamp(i))
    return timesOut

def convertTimeStamp(timeIn) :
    """
    Converts a makeTimeStamp string to Julian day number
    """
    conv   = timeIn.split('T')
    date   = conv[0].split('-')
    time   = conv[1].split(':')
    year   = float(date[0])
    month  = float(date[1])
    day    = float(date[2])
    hour   = float(time[0])
    minute = float(time[1])
    second = float(time[2])
    ut=hour+minute/60+second/3600
    if (100*year+month-190002.5)>0:
        sig=1
    else:
        sig=-1
    return 367*year - int(7*(year+int((month+9)/12))/4) + int(275*month/9) + day + 1721013.5 + ut/24 - 0.5*sig +0.5

def parseTrx(antennaSel,polSel=0,bandNumberSel=3,filename='/data/checkTrx.txt') :
    data = fiop.readcolPy(filename,'s,f,s,f,i,s,f,f,f,f,f')
    recTime = np.array(data[0])
    elev    = np.array(data[1])
    jdTime  = recTime
    for i in range(len(recTime)) : jdTime[i] = convertTimeStamp(recTime[i])-2455198
    antenna = np.array(data[2])
    freq    = np.array(data[3])*1e9
    pol     = np.array(data[4])
    chan    = np.array(data[5])
    trx     = np.array(data[6])
    errtrx  = np.array(data[7])
    gain    = np.array(data[8])
    errgain = np.array(data[9])
    tsys    = np.array(data[10])
    indexAnt  = np.where(antenna == antennaSel)[0]
    indexPol  = np.where(pol == polSel)[0]
    indexFreqLow = np.where(freq <= bandDefinitions[bandNumberSel][1])[0]
    indexFreqHigh = np.where(freq >= bandDefinitions[bandNumberSel][0])[0]
    indexVal = list((set(indexAnt) & set(indexPol) & set(indexFreqLow) & set(indexFreqHigh)))
    return jdTime[indexVal],elev[indexVal],recTime[indexVal],chan[indexVal],trx[indexVal],errtrx[indexVal],gain[indexVal],errgain[indexVal],freq[indexVal],tsys[indexVal]

def parseTrxInfo(antennaList=['DV01','DV02','DV03','DV04','DV05','PM02','PM03'],polList=[0,1],bandList=[3,6],filename=None) :
    if filename == None : filename='/data/checkTrx.txt'
    info = {}
    for i in antennaList :
        info[i] = {}
        for j in bandList :
            info[i][j] = {}
            for k in polList :
                info[i][j][k] = {'jdTime' : [], 'recTime' : [], 'chan' : [], 'trx' : [], 'errtrx' : [], 'gain' : [], 'errgain' : [], 'freq' : [],'elev' : [],'tsys':[]}
                info[i][j][k]['jdTime'],info[i][j][k]['elev'],info[i][j][k]['recTime'],info[i][j][k]['chan'],info[i][j][k]['trx'],info[i][j][k]['errtrx'],info[i][j][k]['gain'],info[i][j][k]['errgain'],info[i][j][k]['freq'],info[i][j][k]['tsys'] = parseTrx(i,k,j,filename)
    return info


def plotTrxInfo(antennaList=['DV01','DV02','DV03','DV04','DV05','PM02','PM03'],polList=[0,1],bandList=[3,6],filename=None) :
    if filename == None : filename='/data/checkTrx.txt'
    antennaList = makeList(antennaList)
    polList = makeList(polList)
    bandList = makeList(bandList)
    colorList = ['b','g','r','c','m','y','k']
    pointList = ['x','.','o','^','<','>','s','+',',','D','1','2','3','4','h','H','o','|','_']
    info = parseTrxInfo(antennaList,polList,bandList,filename)
    pb.clf()
    limits = {3 : 100, 6: 100, 7:300, 9:500}
    spec = {3 : 30, 6: 70, 7:137, 9: 500}
    pb.subplot(len(bandList),1,1)
#    pb.hold(True)
    rcParams['font.size'] = 9.0
    for i in range(len(antennaList)) :
        for j in range(len(bandList)) :
            pb.subplot(len(bandList),1,j+1)
            for k in range(len(polList)) :
                legendInfo = 'Antenna %s' % (antennaList[i])
                time = info[antennaList[i]][bandList[j]][polList[k]]['jdTime']
                trx  = info[antennaList[i]][bandList[j]][polList[k]]['trx']
                err  = info[antennaList[i]][bandList[j]][polList[k]]['errtrx']
                psym = colorList[i]+pointList[k]
#                errorbar(time,trx,yerr=err,fmt=None)
                try:
                   if k==0 : pb.plot(time,trx,psym,label=legendInfo)
                   else : pb.plot(time,trx,psym)
                except:
                    print(len(info[antennaList[i]][bandList[j]][polList[k]]['trx']))
                    print('invalid data for antenna %s, polarization %i, band %i' % (antennaList[i],polList[k],bandList[j]))
                print(len(time),spec[bandList[j]])
                try: pb.plot(time,[spec[bandList[j]]]*len(time),'k-')
                except: continue
            pb.legend(loc=0)
            pb.title('Trx vs time')
            pb.xlabel('Julian Date')
            pb.ylabel('Receiver Temperature (K)')
            pb.ylim(0,limits[bandList[j]])
#    pb.legend(loc=1)
    pb.show()

def makeSpecTrx(freq,band) :
    newSpec = []
    for i in freq :
        low = (bandDefinitions[band][1]-bandDefinitions[band][0])*0.1/1e9+bandDefinitions[band][0]/1e9+6
        high = -(bandDefinitions[band][1]-bandDefinitions[band][0])*0.1/1e9+bandDefinitions[band][1]/1e9-6
        print(low,high)
        if band == 3 :
            if (i < low) or (i > high) : alpha = 10
            else : alpha = 6
        if band == 6 :
            if (i < low) or (i > high) : alpha = 10
            else : alpha = 6
            alpha = 6
        if band == 7 :
            if (i < low) or (i > high) : alpha = 12
            else : alpha = 8
        if band == 9 :
            if (i < low) or (i > high) : alpha = 15
            else : alpha = 10
        newSpec.append(0.048*alpha*i+4)
    return newSpec
                     
def plotTrxFreq(antennaList=['DV01','DV02','DV03','DV04','DV05','PM02','PM03'],polList=[0,1],bandList=[3,6],filename=None) :
    if filename == None : filename='/data/checkTrx.txt'
    
    antennaList = makeList(antennaList)
    polList = makeList(polList)
    bandList = makeList(bandList)
    colorList = ['b','g','r','c','m','y','k']
    pointList = ['x','o','.','^','<','>','s','+',',','D','1','2','3','4','h','H','o','|','_']
    info = parseTrxInfo(antennaList,polList,bandList,filename)
    pb.clf()
    pb.subplot(len(bandList),1,1)
    limits = {3 : 100, 6: 150, 7:300, 9:500}
    rcParams['font.size'] = 9.0
#    pb.hold(True)
    for i in range(len(antennaList)) :
        for j in range(len(bandList)) :
            pb.subplot(len(bandList),1,j+1)
            legendInfo = 'Antenna %s, Band %s' % (antennaList[i],bandList[j])
            for k in range(len(polList)) :
                freq = info[antennaList[i]][bandList[j]][polList[k]]['freq']/1e9
                trx  = info[antennaList[i]][bandList[j]][polList[k]]['trx']
                err  = info[antennaList[i]][bandList[j]][polList[k]]['errtrx']
                psym = colorList[j]+pointList[i] ; print(psym,len(trx),len(freq))
                pb.plot(freq,trx,psym)
                newSpec = makeSpecTrx(freq,bandList[j])
                newSpec = np.array(newSpec)
                freq = np.array(freq)
                print(newSpec.shape,freq.shape)
                freq.sort()
                newSpec.sort()
                pb.plot(freq,newSpec,'k-')
                pb.ylim(0,limits[bandList[j]])
        pb.legend(loc=0)
        pb.title('Trx vs Frequency')
        pb.xlabel('Frequency (GHz)')
        pb.ylabel('Receiver Temperature (K)')
#    pb.ylim(20,200)
    pb.subplot(len(bandList),1,1)
    pb.title('Trx vs Frequency')
    pb.show()
    return newSpec

class MakeTable: 
    def __init__(self,inputMs,queryString='') :
        self.inputMs = inputMs
        self.queryString = queryString
        mytb = tbtool()
        self.makeSubtable(mytb)
        self.data = {}
        for i in self.subtable.colnames() : self.data[i] = self.subtable.getcol(i)
        mytb.close()

    def makeSubtable(self, mytb) :
        mytb.open(self.inputMs)
        self.subtable = mytb.query(self.queryString)

class Weather(MakeTable):
    def __init__(self,inputMs,pressureCut=700,location='AOS'):
        queryString = ("PRESSURE < %s" % pressureCut)
        MakeTable.__init__(self,"%s/WEATHER" % inputMs,queryString)
        self.location      = location
        self.getAtmProfile()
        self.data['ATM_TEMP'] = self.getAtmProfile()
        
    def getAtmProfile(self) :
        if self.location == 'AOS'   : alt = casac.Quantity(5000.0,'m')
        elif self.location == 'OSF' : alt = casac.Quantity(3000.0,'m')
        tatm = []
        tProfile = []
        for i in range(len(self.data["REL_HUMIDITY"])) : 
            tmp = casac.Quantity(self.data['TEMPERATURE'][i],'K')
            pre = casac.Quantity(self.data['PRESSURE'][i],'mbar')
            maxA = casac.Quantity(48.0,'km')
            hum = self.data["REL_HUMIDITY"][i]
            myatm   = at.initAtmProfile(alt,tmp,pre,maxA,hum)
            tempPro = at.getProfile()['temperature'] 
            tProfile.append(np.array(tempPro.value))
            tatm.append(sum(tempPro.value)/len(tempPro.value))
        return np.array(tatm)
        
class InterpolateTableTime: #I think this also would work for Frequency without trying at all, just have to fix verbage?  I also think the call in Tsys is wrong..and probably giving me all sorts of errors.
    def __init__(self,table,timeSeries=None,nonRealInterp='nearest',realInterp='linear',ifRepeat='average',tableQuery=None) :
        print("""Warning: Interpolation of many variables may be inaccurate, be careful of using anything interpolated.""")
        if nonRealInterp != 'nearest' and nonRealInterp != 'change' :
            print("""You must enter nearetst or change for the nonRealInterp value.  Nearest is nearest neighbor, change
                     is the same value is repeated until the boolean changes values.""")
            return
        else :
            self.nonrealInterpType = nonRealInterp
        if realInterp != 'linear' and realInterp != 'cubicspline' :
            print("""You must enter linear or cubicspline for the nonRealInterp value.  Nearest is nearest neighbor, change
                     is the same value is repeated until the boolean changes values.""")
            return
        else :
            self.realInterpType = realInterp
        self.table = table
        print(self.table)
        self.colNames = list(table.data.keys())
        if "TIME" not in self.colNames :
            print("""CRITICAL: Time must be a component of the table to use this function.""")
            return
        self.time = self.table.data["TIME"]
        self.oldData = self.table.data.copy()
        self.oldTime = self.oldData.pop("TIME")
        self.colNames.remove('TIME')
        self.timeSeries = timeSeries
        
    def interpolateData(self,timeSeries,quiet=False) :
        self.newTime = timeSeries
        self.newData = {}
        self.newTime.sort()
        if self.newTime.shape != np.unique(self.newTime).shape :
            if not quiet : print("Removing repeated times.")
            self.newTime = np.unique(self.newTime)

        for i in self.colNames :
            if not quiet : print("Doing parameter %s" % i)
            if 'float' in str(self.oldData[i].dtype) or 'complex' in str(self.oldData[i].dtype) :
                if self.oldTime.shape != np.unique(self.oldTime).shape :
                    tmpTime,tmpData = self.handleTimeRepeats(i)
                else :
                    tmpTime = self.oldTime ;  tmpData = self.oldData[i]
                tuppleMax  = self.oldData[i].shape[:-1]
                indexList  = np.ndindex(tuppleMax)
                _newTmp = []
                for j in indexList :
                    _tmpData = np.transpose(tmpData)[j][:]
                    if self.realInterpType == 'cubicspline' : _new = self.interpSpline(tmpTime,_tmpData)
                    elif self.realInterpType == 'linear' : _new = self.interpLinear(tmpTime,_tmpData)
                    _newTmp.append(np.transpose(_new))
                self.newData[i] = np.array(_newTmp)
                for j in range(len(self.newTime)) :
                    if self.oldTime[0]-self.newTime[j] > 0 :
                        self.newData[i][...,j] = self.oldData[i][...,0]
            else :
                if self.nonrealInterpType == 'change' : self.interpChange(i)
                elif self.nonrealInterpType == 'nearest' : self.interpNearest(i)
            self.newData[i] = np.squeeze(self.newData[i])

    def handleTimeRepeats(self,colname) : 
        tmpTime = []
        tmpData = []
        for j in np.unique(self.oldTime) :
            locos = np.where(self.oldTime == j)
            tmpTime.append(np.mean(self.oldTime[locos]))
            tmpData.append(np.mean(self.oldData[colname][...,locos],-1))
        tmpTime = np.array(tmpTime)
        tmpData = (np.squeeze(np.array(tmpData)))
        return tmpTime,tmpData

    def interpLinear(self,tmpTime,tmpData) :
        return np.interp(self.newTime,tmpTime,tmpData)

    def interpSpline(self,tmpTime,tmpData) :
        tck = splrep(tmpTime,tmpData,s=0)
        return splev(self.newTime,tck,der=0)

    def interpChange(self,colname) :
        self.newData[colname] = []
        for j in self.newTime :
            timeDiff = self.oldTime-j
            goodIndex = max(np.where(timeDiff < 0)[0])
            self.newData[colname].append(self.oldData[colname][goodIndex])
            self.newData[colname] = np.array(self.newData[colname])

    def interpNearest(self,colname) :
        self.newData[colname] = []
        for j in self.newTime :
            timeDiff = (self.oldTime-j)
            indexCount = abs(timeDiff).min()
            locos      = np.where(abs(timeDiff) == indexCount)
            self.newData[colname].append(np.mean(self.oldData[colname][locos]))
        self.newData[colname] = np.array(self.newData[colname])        

def getSourceFieldMapping(inputMs) :
    """
    Returns two items: a dictionary keyed by source name with values of field ID,
      and an array of source names from the NAME column of the FIELD table.
    """
    if not os.path.exists(inputMs):
        print("No such ms found.")
        return
    mytb = createCasaTool(tbtool)
    mytb.open("%s/FIELD" % (inputMs) )
    sourceNames = mytb.getcol('NAME')
    sourceIds = {}
    for i in range(len(sourceNames)) :
        sourceIds[sourceNames[i]] = i
    mytb.close()
    return sourceIds,sourceNames

def getSourceScans(inputMs,sourceName) :
    if str(sourceName).isdigit() : fieldId = sourceName
    else:
        sourceIds,sourceNames = getSourceFieldMapping(inputMs)
        try:
            fieldId = sourceIds[sourceName]
        except:
            return 'What you talking about Willis?'
    mytb = createCasaTool(tbtool)
    mytb.open(inputMs)
    scans = mytb.getcol('SCAN_NUMBER')
    fields = mytb.getcol('FIELD_ID')
    mytb.close()
    list1 = pb.where(fields == fieldId)
    fieldscans = scans[list1]
    return np.unique(fieldscans)

def getBasebandNumber(inputMs,spwId) :
    """
    Returns the number of the baseband for the specified spw in the specified ms.
    Obsoleted by msmd.baseband.
    """
    if (os.path.exists(inputMs) == False):
        print("au.getBasebandNumber(): measurement set not found")
        return -1
    mytb = createCasaTool(tbtool)
    mytb.open("%s/SPECTRAL_WINDOW" % inputMs)
    if ('BBC_NO' in mytb.colnames()):
        bbNums = mytb.getcol("BBC_NO")
    else:
        return -1
    mytb.close()
    return bbNums[spwId]

def getBasebandNumbers(inputMs) :
    """
    Returns the baseband numbers associated with each spw in the specified ms.
    Does not use msmd.
    Todd Hunter
    """
    if (os.path.exists(inputMs) == False):
        print("au.getBasebandNumbers(): measurement set not found")
        return -1
    mytb = createCasaTool(tbtool)
    mytb.open("%s/SPECTRAL_WINDOW" % inputMs)
    if ("BBC_NO" in mytb.colnames()):
        bbNums = mytb.getcol("BBC_NO")
    else:
        mytb.close()
        return(-1)
    mytb.close()
    return bbNums

def getTelescopeNameFromCaltable(caltable):
    return(plotbp3.getTelescopeNameFromCaltable(caltable))

def getFieldNamesFromCaltable(caltable):
    """
    Returns the field names in the specified caltable using casac.calanalysis.
    See also getFieldsFromCaltable
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("caltable not found")
        return -1
    myca = calanalysis()
    myca.open(caltable)
    fieldnames = myca.field()
    myca.close()
    return fieldnames

def getFieldIDsFromCaltable(caltable):
    """
    Returns the field IDs in the specified caltable using casac.calanalysis.
    (See getFieldsFromCaltable for a different method that uses the tb tool.)
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("caltable not found")
        return -1

    mytb = tbtool()
    mytb.open(caltable)
    tableType = mytb.getkeyword('VisCal')
    mytb.close()
    if tableType.find('A Mueller') >= 0:
        # CAS-12595
        return getFieldsFromCaltable(caltable)
    myca = calanalysis()
    myca.open(caltable)
    fieldids = [int(i) for i in myca.field(name=False)]
    myca.close()
    return fieldids

def unifyFieldTimes(vislist, field):
    """
    Copies the time for a field in the FIELD table from one ms to a list of
    ms.
    vislist: a list of measurement sets, or comma-delimited string
    field: integer ID or name (name is safer if IDs vary per ms)
    -Todd Hunter
    """
    if type(vislist) == str:
        vislist = vislist.split(',')
    ids, names = parseFieldArgument(vislist[0],field)
    field = names[0]
    mjdsec = getFieldTimes(vislist[0], ids[0])
    vislist = vislist[1:]
    myfield = field
    for vis in vislist:
        ids, names = parseFieldArgument(vis,field)
        result = setFieldTime(vis, field, mjdsec)
        if (result == -1): return
        # I don't see why the following is needed, but it gets an error without it
        # Objects/listobject.c:169: bad argument to internal function
        if (vis == vislist[-1]): break

def setFieldTime(vis, field, mjdsec):
    """
    Sets a cell in the TIME column of the field table.
    field: integer ID or name
    Todd Hunter
    """
    if (not os.path.exists(vis)):
        print("vis not found: ", vis)
        return -1
    ids, names = parseFieldArgument(vis,field)
    mytb = createCasaTool(tbtool)
    mytb.open(vis+'/FIELD', nomodify=False)
    oldmjdsec = mytb.getcol('TIME')
    print("Updating field %d = %s in %s from %f to %f" % (ids[0],names[0],vis,oldmjdsec[ids[0]],mjdsec))
    oldmjdsec[ids[0]] = mjdsec
    mytb.putcol('TIME',oldmjdsec)
    mytb.close()

def getFieldTime(vis, field):
    """
    Returns the time (in mjd seconds) in the FIELD table for the specified
    field.
    field: integer ID or string ID or name
    -Todd Hunter
    """
    ids, names = parseFieldArgument(vis, field)
    mytimes = getFieldTimes(vis)
    return mytimes[ids[0]]

def getFieldTimes(vis, field=''):
    """
    Returns the TIME column of the field table from one measurement set,
    or a list of measurement sets.
    vis: string, list, or comma-delimited string
    field: integer ID or name (name is safer if IDs vary by ms)
    Todd Hunter
    """
    if (type(vis) == str):
        vislist = vis.split(',')
    else:
        vislist = vis
    mjdsecs = []
    for vis in vislist:
        if (not os.path.exists(vis)):
            print("vis not found")
            return -1
        if (field != -1 and field != ''):
            ids, names = parseFieldArgument(vis,field)
            myfield = ids[0]
        else:
            myfield = field
        if field != '' and field != -1:
            print("%s: field = %d" % (vis, myfield))
        mytb = createCasaTool(tbtool)
        mytb.open(vis+'/FIELD')
        mjdsec = mytb.getcol('TIME')
        mytb.close()
        if field == -1 or field == '':
            mjdsecs.append(mjdsec)
        else:
            mjdsecs.append(mjdsec[myfield])
    if len(mjdsecs) == 1:
        return mjdsecs[0]
    else:
        return mjdsecs
        

def getBasebandNumbersFromCaltable(caltable):
    """
    Returns the baseband numbers associated with each spw in the specified 
    caltable.
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("au.getBasebandNumbersFromCaltable(): caltable not found")
        return -1
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    spectralWindowTable = mytb.getkeyword('SPECTRAL_WINDOW').split()[1]
    mytb.close()
    mytb.open(spectralWindowTable)
    if ("BBC_NO" in mytb.colnames()):
        bbNums = mytb.getcol("BBC_NO")
    else:
        # until CAS-6853 is solved, need to get it from the name
#        print "BBC_NO not in colnames (CAS-6853).  Using NAME column."
        names = mytb.getcol('NAME')
        bbNums = []
        trivial = True
        for name in names:
            if (name.find('#BB_') > 0):
                bbNums.append(int(name.split('#BB_')[1].split('#')[0]))
                trivial = False
            else:
                bbNums.append(-1)
        if (trivial): bbNums = -1
    mytb.close()
    return bbNums

def getMeasurementSetFromCaltable(caltable) :
    """
    Returns the name of the parent measurement set from a caltable using
    the casac tool (except for A Mueller tables for which it uses the tb
    tool instead -- CAS-12595).
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("au.getBasebandNumbersFromCaltable(): caltable not found")
        return -1
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    if mytb.getkeyword('VisCal').find('A Mueller') >= 0:
        # CAS-12595 goes CPU bound on such tables (or is at least very slow)
        msname = mytb.getkeyword('MSName')
        mytb.close()
        return msname
    mytb.close()
    myca = calanalysis()
    myca.open(caltable)
    spectralWindowTable = myca.msname()
    myca.close()
#    mytb = createCasaTool(tbtool)
#    mytb.open(caltable)
#    spectralWindowTable = mytb.getkeyword('MSName')
#    mytb.close()
    return(spectralWindowTable)

def getAntennaNamesFromCaltable(caltable) :
    """
    Returns the antenna names from the specified caltable.
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("au.getAntennaNamesFromCaltable(): caltable not found")
        return -1
#    myca = calanalysis()
#    myca.open(caltable)
#    names = myca.antenna()  # a list
#    myca.close()
    mytb = createCasaTool(tbtool)
    mytb.open(caltable+'/ANTENNA')
    names = mytb.getcol('NAME')  # an array
    mytb.close()
    return names

def getMJDSecFromCaltable(caltable):
    """
    Returns the mean value of MJD seconds for the specified caltable.
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("Caltable not found")
        return -1
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    mjdsec = np.mean(mytb.getcol('TIME'))
    mytb.close()
    return mjdsec

def getMeanGainFromCaltable(caltable, spw, field):
    """
    Compute mean complex gain over all antennas for a given spw and field for
    a non-spectral gain table.
    caltable: file name
    spw: int or string
    field: field ID (int or string) not field name
    -Todd Hunter
    """
    spw = str(spw)
    field = str(field)
    if not field[0].isdigit():
        names = getFieldNamesFromCaltable(caltable)
        if field not in names:
            print("Field not in caltable. Available: ", names)
            return
        i = names.index(field)
        field = str(i)
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    print("Calling tb.query('SPECTRAL_WINDOW_ID == %s && FIELD_ID == %s')" % (spw,field))
    myt = mytb.query('SPECTRAL_WINDOW_ID == %s && FIELD_ID == %s' % (spw,field))
    gains = myt.getcol('CPARAM')
    myt.close()
    mytb.close()
    gains = gains.flatten()
    print("Found %d values" % (len(gains)))
    return np.mean(gains)

def getFlagsFromCaltable(caltable, returnDict=False, includeZeros=False, 
                         antenna='', spw='', returnFullyFlaggedDict=False,
                         includePartialFlaggedDict=False, 
                         showChannelsFlagged=False, bandpassPreAverageFactor=1):
    """
    Reads the number of flags in an antenna-based caltable, or reports a 
    dictionary of the number of flags keyed by scan, spw ID, and antenna name.
    antenna: limit to these specified antenna name(s) (comma-delimited string)
    Counts the number of fully-flagged spectra.
    spw: limit to these specified spws(s) (comma-delimited string)
    returnFullyFlaggedDict: if True, then return dictionary keyed by scan, spw ID
           with values being a list of antenna names
    includePartialFlaggedDict: if True, then also return a dictionary of spectra
            with partial channel flagging
    showChannelsFlagged: if False, the partial channel flagging dictionary
           contains the number of flags per spectrum; if True, it contains
           a list of flagged channels
    See also: au.snrFromCaltable  
          and au.getTsysFromCaltable(getflags=True) to get the FLAG vector
    -Todd Hunter
    """
    spws = getSpwsFromCaltable(caltable)
    antennas = getAntennaIDsFromCaltable(caltable)
    names = getAntennaNamesFromCaltable(caltable)
    scans = getScansFromCaltable(caltable)
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    antennaNames = antenna.split(',')
    d = {}
    nflags = 0
    spwIDs = []
    if spw != '':
        if type(spw) == str:
            vis = getMeasurementSetFromCaltable(caltable)
            if os.path.exists(vis):
                spwIDs = parseSpw(vis, spw)
            else:
                spwIDs = [int(i) for i in str(spw).split(',')]
        elif type(spw) == list or type(spw) == np.ndarray:
            spwIDs = [int(i) for i in spw]
        else:
            spwIDs = [spw]
    fullyFlaggedSpectra = 0
    totalSpectra = 0
    anyFlaggedSpectra = 0 # spectra with any channels flagged
    anyFlagged = {} # spectra with partial channels flagged
    fullyFlagged = {} # spectra with all channels flagged
    for scan in scans:
        d[scan] = {}
        for myspw in spws:
            if (spw == '' or myspw in spwIDs):
                d[scan][myspw] = {}
                for ant in antennas:
                    if (antenna == '' or names[ant] in antennaNames):
                        myt = mytb.query('ANTENNA1 == %d and SPECTRAL_WINDOW_ID == %d and SCAN_NUMBER == %d' % (ant, myspw, scan))
                        flags = myt.getcol('FLAG')
                        nflag = sum(flags.flatten())
                        nflags += nflag
                        totalSpectra += 1
                        if nflag > 0:
                            anyFlaggedSpectra += 1
                        if nflag == len(flags.flatten()) and nflag > 4:
                            fullyFlaggedSpectra += 1
                            if scan not in fullyFlagged:
                                fullyFlagged[scan] = {}
                            if myspw not in fullyFlagged[scan]:
                                fullyFlagged[scan][myspw] = []
                            fullyFlagged[scan][myspw].append(names[ant])
                        elif nflag > 0:
                            if scan not in anyFlagged:
                                anyFlagged[scan] = {}
                            if myspw not in anyFlagged[scan]:
                                anyFlagged[scan][myspw] = {}
                            if showChannelsFlagged:
                                value = np.where(flags == 1)
                            else:
                                value = nflag
                            anyFlagged[scan][myspw][names[ant]] = value[1]*bandpassPreAverageFactor
                        myt.close()
                        if (nflag > 0 or includeZeros):
                            d[scan][myspw][names[ant]] = nflag
    mytb.close()
    print("Saw %d fully-flagged spectra out of %d spectra with any flags out of %d total spectra." % (fullyFlaggedSpectra, anyFlaggedSpectra, totalSpectra))
    if fullyFlaggedSpectra > 0:
        print("Here is the dictionary of fully flagged spectra, keyed by scan number, then spw ID")
        print(fullyFlagged)
    if anyFlaggedSpectra > 0:
        print("Here is the dictionary of spectra with partial flags, keyed by scan number, then spw ID")
        print(anyFlagged)
    if returnDict:
        return d
    elif returnFullyFlaggedDict:
        if includePartialFlaggedDict:
            return fullyFlagged, anyFlagged
        else:
            return fullyFlagged
    else:
        return nflags
    
def getAntennaIDsFromCaltable(caltable) :
    """
    Returns the antenna IDs for which solutions exist in the specified caltable
    using the tb tool.
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("au.AntennaIDsFromCaltable(): caltable not found")
        return -1
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    antennas = np.unique(mytb.getcol('ANTENNA1'))
    mytb.close()
    return antennas

def compareColumnsBetweenVis(vis1, vis2, spw, columns='TIME,TIME_CENTROID,UVW,DATA,CORRECTED_DATA,MODEL_DATA,SIGMA,WEIGHT,FLAG'):
    """
    Calls compareColumnBetweenVis for a sequence of columns.
    -Todd Hunter
    """
    columns = columns.split(',')
    for column in columns:
        compareColumnBetweenVis(vis1, vis2, spw, column)

def rowsForSpw(vis, spw):
    """
    Returns a list of row numbers corresponding to specified spw.
    spw: single spw (integer or string)
    -Todd Hunter
    """
    mytb = createCasaTool(tbtool)
    mytb.open(vis)
    spw = int(spw)
    dd = getDataDescriptionId(vis, spw)
    myt = mytb.query('DATA_DESC_ID == %d' % (dd))
    rows = myt.rownumbers()
    myt.close()
    mytb.close()
    return rows

def compareFlagVersions(flag1, flag2='', column='FLAG', vis='', spw=None, 
                        rows=None, verbose=False, sortByTime=True):
    """
    Compares FLAG column between two tables in the flagversions directory of a 
    measurement set.
    flag1: name of flag file, e.g. 'mydata.ms.flagversions/flags.Original'
           or the parent directory:  'mydata.ms.flagversions' for which each
               "flags.x" file will be processed
    flag2: if you only wnat the hash of flag1, then do not specify flag2
    column: for multi-spw data, it is only safe to compare FLAG_ROW
            unless you also specify rows or vis+spw, otherwise you will 
            get a table conformance error
    rows: restrict to this list of rows
    vis, spw: alternative to specifying rows (it will figure it out)
        if vis is blank, it will extract name from flag1 up to '.flagversions'
    Returns: True if flag1==flag2, else False; or hash value if flag2==''
    -Todd Hunter
    """
    flag1 = flag1.rstrip('/')
    flag2 = flag2.rstrip('/')
    if flag1[-13:] == '.flagversions':
        if sortByTime:
            key = os.path.getmtime
        else:
            key = None
        flags1 = sorted(glob.glob(flag1+'/flags.*'), key=key)
        flags2 = sorted(glob.glob(flag2+'/flags.*'), key=key)
    else:
        flags1 = [flag1]
        flags2 = [flag2]
    mytb1 = createCasaTool(tbtool)
    if flag2 != '':
        mytb2 = createCasaTool(tbtool)
    if rows is None:
        if vis != '' and spw is not None:
            rows = rowsForSpw(vis, spw)
            print("Found %d rows for spw %d" % (len(rows),spw))
        elif vis != '': 
            print("You have specified vis, but not spw.")
            return
        elif spw is not None:
            loc = flag1.find('.flagversions')
            vis = flag1[:loc]
            print("You have specified spw, but not vis. Extracted vis from flag1: ", vis)
            rows = rowsForSpw(vis, spw)
            print("Found %d rows for spw %d" % (len(rows),spw))
    if column == 'FLAG' and rows is None:
        print("With column FLAG, you must specify either rows or vis,spw")
        return
    for i,flag1 in enumerate(flags1):
        mytb1.open(flag1)
        if column not in mytb1.colnames():
            print("%s column not present" % (column))
            mytb1.close()
            return
        if rows is None:
            data1 = mytb1.getcol(column)
            print("Using all %d rows" % (len(data1)))
        else:
            myt1 = mytb1.selectrows(rows)
            data1 = myt1.getcol(column)
            if verbose:
                print("retrieved shape %s" % str(np.shape(data1)))
            myt1.close()
        mytb1.close()
        print("len(data1) = ", len(data1))
        data1 = data1.flatten()
        print("len(data1) after flattening = ", len(data1))
        hash1 = hashlib.sha1(data1).hexdigest()
        if flag2 != '':
            flag2 = flags2[i]
            mytb2.open(flag2)
            if rows is None:
                data2 = mytb2.getcol(column)
            else:
                myt2 = mytb2.selectrows(rows)
                data2 = myt2.getcol(column)
                if verbose:
                    print("retrieved shape %s" % str(np.shape(data2)))
                myt2.close()
            mytb2.close()
            data2 = data2.flatten()
            result = np.array_equal(data1, data2)
            hash2 = hashlib.sha1(data2).hexdigest()
            if result and hash1 == hash2:
                print("%s columns are identical, hash = %s" % (column, hash1))
            else:
                print("%s columns are different" % (column))
                print("hash is different: %s, %s" % (hash1, hash2))
                diff = np.where(data1 != data2)[0]
                print("Shape of data1.flatten(): ", np.shape(data1))
                print("Shape of difference: ", np.shape(diff))
            if len(flags1) == 1:
                return result
        else:
            print("hash = %s for %s" % (hash1,os.path.basename(flag1)))
    if flags2 == '':
        return hash1

def compareColumnBetweenVis(vis1, vis2, spw, column='DATA', verbose=False):
    """
    Uses np.array_qual and hashlib.sha to compare data arrays.
    If you just want the hash value of vis1, then set vis2=''
    -Todd Hunter
    """
    dd1 = getDataDescriptionId(vis1, spw)
    if verbose:
        print("Using DATA_DESC_ID 1 = %d" % (dd1))
    mytb1 = createCasaTool(tbtool)
    mytb2 = createCasaTool(tbtool)
    mytb1.open(vis1)
    if column not in mytb1.colnames():
        print("%s column not present" % (column))
        mytb1.close()
        return
    myt1 = mytb1.query('DATA_DESC_ID == %s' % (str(dd1)))
    data1 = myt1.getcol(column)
    myt1.close()
    mytb1.close()
    data1 = data1.flatten()
    hash1 = hashlib.sha1(data1).hexdigest()
    if vis2 != '':
        dd2 = getDataDescriptionId(vis2, spw)
        if verbose:
            print("Using DATA_DESC_ID 2 = %d" % (dd2))
        mytb2.open(vis2)
        myt2 = mytb2.query('DATA_DESC_ID == %s' % (str(dd2)))
        data2 = myt2.getcol(column)
        myt2.close()
        mytb2.close()
        data2 = data2.flatten()
        result = np.array_equal(data1, data2)
        hash2 = hashlib.sha1(data2).hexdigest()
    else:
        print("%s column hash = %s" % (column, hash1))
        return hash1
    if result and hash1 == hash2:
        print("%s columns are identical, hash = %s" % (column, hash1))
    else:
        print("%s columns are different" % (column))
        print("hash is different: %s, %s" % (hash1, hash2))
        diff = np.where(data1 != data2)[0]
        print("Shape of data1.flatten(): ", np.shape(data1))
        print("Shape of difference: ", np.shape(diff))
    return result

def delayCalculations(efficiency=0.99, bandwidth=8e9, channelWidth=1e3, 
                      maxBaseline=35, numerator=0.245317809, spwBandwidth=None):
    """
    Compute useful quantities for the ALMA next generation correlator specs.
    frequencyOffset: in Hz
    bandwidth: of the baseband, in Hz
    channelWidth: in the correlator, in Hz
    maxBaseline: in km
    spwBandwidth: in Hz, None implies equal to bandwidth
    numerator: force a specific floating point value, such as 0.245317809, 
               which is exact (but only for the case of efficiency=0.99)
               or None --> use sqrt(6*(1-efficiency))
      which is based on Maclaurin approximation first term for sin(theta)/theta 
                                                 = 1 - (theta^2)/6
    """
    if spwBandwidth is None:
        spwBandwidth = bandwidth
    if (efficiency == 0.99):
        print("efficiency is 0.99")
    if numerator != 0.245317809:
        print("numerator is not the value for 0.99")
    if (numerator == 0.245317809 and efficiency != 0.99) or (numerator != 0.245317809 and efficiency == 0.99):
        print("You must set numerator to a value appropriate for your efficiency, or set numerator=None to use the Maclaurin approximation.")
        return
    channelWidth = float(channelWidth)
    loss = 1-efficiency
    sampleRate = bandwidth*2
    if numerator is None:
        factor = np.sqrt(loss*6)
    else:
        factor = numerator
    print("numerator for deltaT equation = %g" % (factor))
    frequencyOffset = 0.5*channelWidth
    deltaT = factor / (2*np.pi*spwBandwidth)
    print("deltaT = %g sec/sec" % (deltaT))
    siderealRate = np.pi*2 / 86400.
    print("siderealRate = %g rad/sec" % (siderealRate))
    maxDelayRate = siderealRate * maxBaseline*1000 / c_mks 
    print("maximum sidereal delayRate = %g" % (maxDelayRate))
    delayUpdateRate = deltaT / maxDelayRate
    print("required delayUpdateRate = %g ms" % (1000*delayUpdateRate))
    print("    (Minimum channel width possible (for all delay correction in correlator) = %g Hz)" % (1/delayUpdateRate))
    power = int(np.ceil(math.log(sampleRate/channelWidth, 2)))
    samples = 2**power
    print("Sample rate = %g GSa/sec" % (sampleRate))
    print("samples = %d (2^%d)" % (samples, power))
    print("segment length = %f ms" % (1000*samples/sampleRate))
    realChannelWidth = sampleRate/(1000*samples)
    print("Real channel width = %f kHz" % (realChannelWidth))
    return deltaT
    
def caltableChecksums(vis, column='auto', suffix='*.tbl'):
    """
    Finds all caltables of a pipeline run (*.tbl) and runs compareCparamFromCaltables
    on them to print the checksum.
    column: 'auto' means 'CPARAM' if present, otherwise 'FPARAM'
         or you could give the actual name, like 'FLAG' 
    -Todd Hunter
    """
    caltables = sorted(glob.glob(vis+suffix), key=os.path.getmtime)
    mytb = createCasaTool(tbtool)
    for caltable in caltables:
        mytb.open(caltable)
        if column == 'auto':
            if 'CPARAM' in mytb.colnames():
                column = 'CPARAM'
            else:
                column = 'FPARAM'
        mytb.close()
        compareCparamFromCaltables(caltable, column=column)

def compareCparamFromCaltableList(filename, dir1='', dir2='jao_caltables', 
                                  phase=False, realAndImaginary=False, 
                                  fractional=False, ignoreAmpAbove=None):
    """
    Reads a list of pipeline caltable names from filename (one per row)
    and runs au.compareCparamFromCaltables in those files in directory dir1
    vs. the same names in a second working directory (dir2).
    realAndImaginary: 'auto' or True or False
       'auto': will use np.real and np.imag for tables with 'gpcal' in the name
              and amplitude (i.e. np.absolute) for all other tables
        True or False: will pass directly to compareCparamFromCaltabes()
    phase: passed directly to compareCparamFromCaltabes()
    fractional: this option is not yet commissioned!
    -Todd Hunter
    """
    f = open(filename,'r')
    lines = f.readlines()
    f.close()
    results = []
    realAndImaginaryInput = realAndImaginary
        
    for line in lines:
        caltable = line.split()[0]
        if realAndImaginaryInput == 'auto':
            if caltable.find('gacal') > 0 or caltable.find('fcal') > 0:
                realAndImaginary = False
            else:
                realAndImaginary = True
        results.append(compareCparamFromCaltables(os.path.join(dir1,caltable), os.path.join(dir2,caltable), phase=phase, realAndImaginary=realAndImaginary, fractional=fractional, ignoreAmpAbove=ignoreAmpAbove))
        results[-1][0]['caltable'] = caltable  # store this in the first pol
    lenSuffix = []
    for result in results:
        suffix = '.'.join(result[0]['caltable'].split('.ms.')[1:])
        lenSuffix.append(len(suffix))
        print("%s" % (suffix))
        for pol in range(len(result)):
            print("stats of pol%d diff: max= %+.9f  min= %+.9f  mean= %+.9f  median= %+.9f" % (pol, result[pol]['max'], result[pol]['min'], result[pol]['mean'], result[pol]['median']))
        print("------------------")
    print("||  caltable  ||  pol.product  ||  max diff  ||  min diff  ||  mean diff  ||  median diff  ||  value 1  ||  value 2  ||  nrows with abs>1e-7  ||  row with max(abs(diff))  ||")
    stringLength = int(np.max(lenSuffix)) # "%*s" wants an int not np.int64
    for result in results:
        for pol in range(len(result)):
            if 'maxI' not in result[0].keys():
                print("| %*s | %d amp | %+.9f  | %+.9f  | %+.9f  | %+.9f  |  %+.9f  |  %+.9f  |  %d  |  %d (%s spw%d) |" % (stringLength, '.'.join(result[0]['caltable'].split('.ms.')[1:]), pol, result[pol]['max'], result[pol]['min'], result[pol]['mean'], result[pol]['median'], result[pol]['v1'], result[pol]['v2'], result[pol]['nrows'], result[pol]['row'], result[pol]['antenna'], result[pol]['spw']))
            else:
                print("| %*s | %d real | %+.9f  | %+.9f  | %+.9f  | %+.9f  |  %+.9f  |  %+.9f  |  %d  |  %d (%s spw%d)  |" % (stringLength, '.'.join(result[0]['caltable'].split('.ms.')[1:]), pol, result[pol]['max'], result[pol]['min'], result[pol]['mean'], result[pol]['median'], result[pol]['v1'], result[pol]['v2'], result[pol]['nrows'], result[pol]['row'], result[pol]['antenna'], result[pol]['spw']))
                print("| %*s | %d imag | %+.9f  | %+.9f  | %+.9f  | %+.9f  |  %+.9f  |  %+.9f  |  %d  |  %d (%s spw%d)  |" % (stringLength, '.'.join(result[0]['caltable'].split('.ms.')[1:]), pol, result[pol]['maxI'], result[pol]['minI'], result[pol]['meanI'], result[pol]['medianI'], result[pol]['v1'], result[pol]['v2'], result[pol]['nrows'], result[pol]['row'], result[pol]['antenna'], result[pol]['spw']))
#    return results

def compareCparamFromCaltables(caltable='', caltable2='',myfilter='*gpcal.*wvr.tbl',
                               vis='', column='CPARAM', ignoreFlags=True, 
                               phase=False, spw='', zeroIndices=1, scan='', 
                               perscan=False, antenna='', perantenna=False, 
                               returnmessages=False, realAndImaginary=False, 
                               fractional=False, computeDiff=False, 
                               ignoreAmpAbove=None):
    """
    Written to diagnose small difference in gfluxscale's gacal caltable in CASA6.
    caltable: either a caltable or a working directory containing a caltable
         that matches myfilter, or a pipeline run directory
    caltable2: either a caltable or a working directory containing a caltable
         that matches myfilter, or a pipeilne run directory
         if you only want the hash value for caltable, leave caltable2=''
    If caltable and caltable2 are blank, then search for tables that match
       the myfilter string and the vis string
    vis: limit the caltables to include this string (preceeding myfilter)
    myfilter: limit the caltables to include this string (following vis)
    column: could also be FPARAM, e.g. for Tsys
          but note that if spws vary in channel number, you need to limit with spw parameter
          or else you will get a conformance error
    ignoreFlags: if True, then ignore rows in which either pol is flagged
    scan: limit comparison to this scan (int or string)
    phase: if True, use np.arctan2(imag,real) instead of np.absolute() for
           computing max, min, mean and median
    realAndImaginary: if True, compute difference of real components and of the
       imaginary components
    zeroIndices: set to 0 to examine full spectral solutions (like Tsys)
             set to 1 to examine only the first channel of a spectral solution
    fractional: this option is not yet commissioned!
    Returns True if tables are identical
    -Todd Hunter
    """
    if caltable == '' and caltable2  == '':
        if vis != '':
            caltables = sorted(glob.glob('*%s*%s' % (vis,myfilter)))
        else:
            caltables = sorted(glob.glob(myfilter))
        caltable = caltables[0]
        caltable2 = caltables[1]
        print("Using caltable:  ", caltable)
        print("Using caltable2: ", caltable2)
    if caltable.replace('/','')[-38:-35] in ['E2E', '201']:
        caltable = findPipelineWorkingDirectory(caltable)
        print("caltable findPipelineWorkingDirectory returns: ", caltable)
    differences = []  # 2-element list of dictionaries (one for each pol)
    if caltable[-9:].find('/working') >= 0:
        caltables = sorted(glob.glob('%s/*%s*%s*' % (caltable,vis, myfilter)))
        if len(caltables) < 1:
            print("No matching cal tables found in this directory: ", caltable)
            return
        if len(caltables) > 1:
            print("More than one caltable1 matches your selection. Use the vis or myfilter parameters: ")
            for caltable in caltables:
                print(os.path.basename(caltable))
            return
        caltable = caltables[0]
        print("Found caltable=%s" % (caltable))
    if not os.path.exists(caltable):
        print("no such table: ", caltable)
        return
    if caltable2 != '':
        if caltable2.replace('/','')[-38:-35] in ['E2E', '201']:
            caltable2 = findPipelineWorkingDirectory(caltable2)
            print("caltable2 findPipelineWorkingDirectory returns: ", caltable2)
        if caltable2[-9:].find('/working') >= 0:
            caltables = sorted(glob.glob('%s/*%s*%s*' % (caltable2,vis, myfilter)))
            if len(caltables) < 1:
                print("No matching cal tables found in this directory: ", caltable2)
                return
            if len(caltables) > 1:
                print("More than one caltable2 matches your selection. Use the vis or myfilter parameters: ")
                for caltable in caltables:
                    print(os.path.basename(caltable))
                return
            caltable2 = caltables[0]
            print("Found caltable2=%s" % (caltable2))

        if not os.path.exists(caltable2):
            print("no such table: ", caltable2)
            return
    result = getTimesFromCaltable(caltable,returnAntennaSpw=True,uniqueValues=False,column=column, returnFlags=ignoreFlags, spw=spw)
    if result is None:  
        print("Failed on first caltable")
        return
    hash1 = hashlib.sha1(result[0].flatten()).hexdigest()
    if caltable2 == '':
        print ("%s = %s %s" % (hash1, column, os.path.basename(caltable)))
        return hash1
    result2 = getTimesFromCaltable(caltable2,returnAntennaSpw=True, uniqueValues=False, column=column, returnFlags=ignoreFlags, spw=spw)
    if result2 is None:
        print("Failed on second caltable")
        return
    if ignoreFlags:
        myt1, ant1, spw1, scan1, field1, flag1 = result
        myt2, ant2, spw2, scan2, field2, flag2 = result2
    else:
        myt1, ant1, spw1, scan1, field1 = result
        myt2, ant2, spw2, scan2, field2 = result2
    calscans = np.unique(scan1)
    if perantenna:
        antennas = np.unique(ant1)
        print("antennas = ", antennas)
    else:
        if type(antenna) == str and antenna != '':
            antennaNames = getAntennaNamesFromCaltable(caltable)
            antenna = list(antennaNames).index(antenna)
        antennas = [antenna]
    if perscan:
        allscans = calscans
    elif scan == '':
        allscans = [-1]
    else:
        allscans = [int(scan)]
    messages = []
    for antenna in antennas:
        for myscan in allscans:
            if perscan or perantenna:
                if perscan:
                    print("Processing scan %d of %s" % (myscan,allscans))
                if perantenna:
                    print("Processing antenna %s" % (antenna))
                # reset values to whole caltable
                if ignoreFlags:
                    myt1, ant1, spw1, scan1, field1, flag1 = result
                    myt2, ant2, spw2, scan2, field2, flag2 = result2
                else:
                    myt1, ant1, spw1, scan1, field1 = result
                    myt2, ant2, spw2, scan2, field2 = result2
            if myscan > -1:
                if myscan not in calscans:
                    print("Scan not found. Available: ", calscans)
                    return
                print("Processing scan: ", myscan)
                idx = np.where(scan1 == myscan)[0]
                myt1 = myt1[:,:,idx]
                ant1 = ant1[idx]
                spw1 = spw1[idx]
                scan1 = scan1[idx]
                field1 = field1[idx]
                flag1 = flag1[:,:,idx]
                idx = np.where(scan2 == myscan)[0]
                myt2 = myt2[:,:,idx]
                ant2 = ant2[idx]
                spw2 = spw2[idx]
                scan2 = scan2[idx]
                field2 = field2[idx]
                flag2 = flag2[:,:,idx]
            else:
                print("Processing all scans: ", allscans)
            if antenna != '':
                idx = np.where(antenna == ant1)[0]
                myt1 = myt1[:,:,idx]
                ant1 = ant1[idx]
                spw1 = spw1[idx]
                scan1 = scan1[idx]
                field1 = field1[idx]
                flag1 = flag1[:,:,idx]
                idx = np.where(antenna == ant2)[0]
                myt2 = myt2[:,:,idx]
                ant2 = ant2[idx]
                spw2 = spw2[idx]
                scan2 = scan2[idx]
                field2 = field2[idx]
                flag2 = flag2[:,:,idx]
                
            originalShape = np.shape(myt1)
            originalLen = len(myt1)
            for i in range(originalLen):
                if zeroIndices == 0:
                    t1 = myt1[i] 
                    t2 = myt2[i] 
                elif zeroIndices == 1:
                    t1 = myt1[i][0]
                    t2 = myt2[i][0]
                print("************ shape(myt1) = %s" % str(originalShape))
                print("Comparing %s: t1[%d] shape=%s out of shape=%s, len=%s" % (column, i, np.shape(t1), originalShape, originalLen))
                if zeroIndices == 1:
                    print("caltable:  %d values, t1[0]=%s" % (len(t1), str(t1[0])))
                    print("caltable2: %d values, t2[0]=%s" % (len(t2), str(t2[0])))
                else:
                    # show first channel and last channel of first row
                    print("Showing first and last channel of first row:")
                    print("caltable:  %d values, t1[0][0]..t1[-1][0]=%s..%s" % (len(t1), str(t1[0][0]), str(t1[-1][0])))
                    print("caltable2: %d values, t2[0][0]..t2[-1][0]=%s..%s" % (len(t2), str(t2[0][0]), str(t2[-1][0])))
                idx = np.where(t1 != t2)[0]
                print("%d rows show differences: " % len(idx))
                ants = np.unique(ant1[idx])
                spws = np.unique(spw1[idx])
                scans = np.unique(scan1[idx])
                fields = np.unique(field1[idx]) # these are field IDs
                antennaNames = getAntennaNamesFromCaltable(caltable)
                antennaNames2 = getAntennaNamesFromCaltable(caltable2)
                allFieldNames = np.array(getFieldNamesFromCaltable(caltable))
                fieldIDs = getFieldIDsFromCaltable(caltable)
                fieldNames = []
                for field in fields:
                    fieldNames.append(allFieldNames[fieldIDs.index(field)])
                # Here we assume that antenna IDs start at zero
                if len(idx) > 0:
                    print("%2d affected spws (of %2d): %s" % (len(spws), len(np.unique(spw1)), spws))
                    print("     %2d antennas (of %2d): %s = %s" % (len(ants),len(np.unique(ant1)), ants, antennaNames[ants]))
                    print("        %2d scans (of %2d): %s" % (len(scans),len(np.unique(scan1)), scans))
                    print("       %2d fields (of %2d): %s = %s" % (len(fields), len(np.unique(field1)), fields, fieldNames))
                if ignoreFlags:
                    # ignore any rows that have all channels flagged
                    nchan = np.shape(flag1[i])[0]
                    idx = np.where((np.sum(flag1[i],axis=0) < nchan) * (np.sum(flag2[i],axis=0) < nchan))
                    flag1_idx = np.where(np.sum(flag1[i],axis=0) == nchan)
                    flag2_idx = np.where(np.sum(flag2[i],axis=0) == nchan)
        #   older simpler approach which ignores rows with any channel flagged
        #            idx = np.where((flag1[0][0] == False) * (flag2[0][0] == False))
        #            flag1_idx = np.where(flag1[0][0] == True)
        #            flag2_idx = np.where(flag2[0][0] == True)
        #            flagged_idx = np.where((flag1[0][0] == True) | (flag2[0][0] == True))
                    flagged_idx = np.where((np.sum(flag1[i],axis=0) == nchan) | (np.sum(flag2[i],axis=0) == nchan))
                    flaggedAntennas = np.unique(ant1[flag1_idx])
                    flaggedAntennas2 = np.unique(ant2[flag2_idx])
                    print("Ignoring %d flagged rows (covering antennas in first table: %s = %s)" % (len(flag1[0][0])-len(idx[0]), flaggedAntennas, antennaNames[flaggedAntennas]))
                    print("                                          (in second table: %2s = %4s)" % (flaggedAntennas2, antennaNames2[flaggedAntennas2]))
                    originalRowidx = idx[0][:]
                    t1 = np.transpose(np.transpose(t1)[idx])
                    t2 = np.transpose(np.transpose(t2)[idx])
                    idx = np.where(t1 != t2)[0]  # idx is the indices of the rows that differ
                else:
                    idx = range(len(t1))
                if len(t1) == len(t2) and len(t1) > 0:
                    if phase:
                        print("statistics of direct difference degrees(arctan2(t1) - arctan2(t2)): ")
                        d = np.degrees(np.arctan2(np.imag(t1), np.real(t1))) - np.degrees(np.arctan2(np.imag(t2), np.real(t2)))
                        v1 = np.degrees(np.arctan2(np.imag(t1), np.real(t1)))
                        v2 = np.degrees(np.arctan2(np.imag(t2), np.real(t2)))
                    elif realAndImaginary:
                        print("statistics of direct difference (np.real(t1)-np.real(t2)): ")
                        if fractional:
                            d = (np.real(t1) - np.real(t2)) / (0.5*(np.real(t1)+np.real(t2)))
                            d2 = (np.imag(t1) - np.imag(t2)) / (0.5*(np.imag(t1)+np.imag(t2)))
                            v1 = np.real(t1)
                            v2 = np.real(t2)
                        else:
                            d = np.real(t1) - np.real(t2)
                            d2 = np.imag(t1) - np.imag(t2)
                            v1 = np.real(t1)
                            v2 = np.real(t2)
                    else:
                        print("statistics of direct difference (np.absolute(t1)-np.absolute(t2)): ")
                        v1 = np.absolute(t1)
                        v2 = np.absolute(t2)
                        d = v1 - v2
                    if len(d.flatten()) > 0:
                        if ignoreAmpAbove is None or phase or realAndImaginary:
                            maxvalue = np.max(d)
                            row = np.argmax(np.abs(d)) # this will not be the row of the caltable since we have filtered by t1!=t2 above
                            caltableRow = originalRowidx[row] # now it is the row in the caltable
                            minvalue = np.min(d)
                            meanvalue = np.mean(d)
                            medianvalue = np.median(d)
                        else:
                            goodidx = np.where((np.abs(v1) < ignoreAmpAbove) * (np.abs(v2) < ignoreAmpAbove))[0]
                            row = goodidx[np.argmax(np.abs(d[goodidx]))] # this will not be the row of the caltable since we have filtered by t1!=t2 above
                            caltableRow = originalRowidx[row] # now it is the row in the caltable
                            maxvalue = np.max(d[goodidx])
                            minvalue = np.min(d[goodidx])
                            meanvalue = np.mean(d[goodidx])
                            medianvalue = np.median(d[goodidx])
                        nrowidx = np.where(np.abs(d.flatten()) > 1e-7)[0]
                        nrows = len(nrowidx)
                        differences.append({'max': maxvalue, 'min': minvalue, 'mean': meanvalue, 'median': medianvalue, 
                                            'row': caltableRow, 'v1': v1[row], 'v2': v2[row], 'nrows': nrows, 
                                            'antenna': antennaNames[ant1[caltableRow]], 'spw': spw1[caltableRow]})
                        if realAndImaginary:
                            differences[-1]['maxI'] = np.max(d2)
                            differences[-1]['minI'] = np.min(d2)
                            differences[-1]['meanI'] = np.mean(d2)
                            differences[-1]['medianI'] = np.median(d2)
                    if len(idx) > 0 and not phase and not realAndImaginary:
                        # compute means over the rows that differ
                        mymean = np.mean(np.absolute(t1[idx]))
                        mymean2 = np.mean(np.absolute(t2[idx]))
                        message = ''
                        if perscan:
                            message += "scan %3d: " % (myscan)
                        if perantenna:
                            message += "antenna %2d=%s: " % (antenna,antennaNames[antenna])
                        message += " diff in percentage: max=%+.9f  min=%+.9f  mean=%+.9f (%d values)" % (np.max(d)*100/mymean, np.min(d)*100/mymean, (mymean-mymean2)*100/mymean, len(d))
                        messages.append(message)
                        print(message)
                    # Now repeat over the rows that differ
                    if len(idx) > 0:
                        if phase:
                            d = np.degrees(np.arctan2(np.imag(t1[idx]),np.real(t1[idx])) - np.arctan2(np.imag(t2[idx]), np.real(t2[idx])))
                        elif realAndImaginary:
                            if fractional:
                                d = (np.real(t1[idx]) - np.real(t2[idx])) / (0.5*(np.real(t1[idx])+np.real(t2[idx])))
                                d2 = (np.imag(t1[idx]) - np.imag(t2[idx])) / (0.5*(np.imag(t1[idx])+np.imag(t2[idx])))
                            else:
                                d = np.real(t1[idx]) - np.real(t2[idx])
                                d2 = np.imag(t1[idx]) - np.imag(t2[idx])
                        else:
                            d = np.absolute(t1[idx]) - np.absolute(t2[idx])
                        print("over differing rows: max=%+.9f  min=%+.9f  mean=%+.9f  median=%+.9f (%d values)" % (np.max(d), np.min(d), np.mean(d), np.median(d), len(idx)))
                        if realAndImaginary:
                            print("   imaginary: max=%+.9f  min=%+.9f  mean=%+.9f  median=%+.9f (%d values)" % (np.max(d2), np.min(d2), np.mean(d2), np.median(d2), len(idx)))
                        if True: # not phase:
                            mymean = np.mean(np.absolute(t1[idx]))
                            mymean2 = np.mean(np.absolute(t2[idx]))
                            message = ''
                            if perscan:
                                message += "scan %3d: " % (myscan)
                            if perantenna:
                                message += "antenna %2d=%s: " % (antenna,antennaNames[antenna])
                            # "d" now has only the rows that differ
                            message += " diff in percentage: max=%+.9f  min=%+.9f  mean=%+.9f (%d values)" % (np.max(d)*100/mymean, np.min(d)*100/mymean, (mymean-mymean2)*100/mymean, len(idx))
#                          Don't append these, to avoid confusion in the summary at the end of the output
#                            messages.append(message)
                            print(message)
                    # Now compute the np.diff over all rows (adjacent value differences)
                    if computeDiff:
                        if len(t1.flatten()) > 0:
                            if phase:
                                print("statistics of np.diff(arctan2(t1)) minus np.diff(arctan2(t2)): ")
                                d = np.degrees(np.diff(np.arctan2(np.imag(t1),np.real(t1))) - np.diff(np.arctan2(np.imag(t2),np.real(t2))))
                            elif realAndImaginary:
                                print("statistics of np.diff(real(t1)) minus np.diff(real(t2)): ")
                                if not fractional:
                                    d = np.diff(np.real(t1)) - np.diff(np.real(t2))
                                    d2 = np.diff(np.imag(t1)) - np.diff(np.imag(t2))
                                else:
                                    d = (np.diff(np.real(t1)) - np.diff(np.real(t2))) / (0.5*(np.diff(np.real(t1))+np.diff(np.real(t2))))
                                    d2 = (np.diff(np.imag(t1)) - np.diff(np.imag(t2))) / (0.5*(np.diff(np.imag(t1))+np.diff(np.imag(t2))))
                            else:
                                print("statistics of np.diff(abs(t1)) minus np.diff(abs(t2)): ")
                                d = np.diff(np.absolute(t1)) - np.diff(np.absolute(t2))
                            print("max=%g  min=%g  mean=%g  median=%g" % (np.max(d), np.min(d), np.mean(d), np.median(d)))
                            if realAndImaginary:
                                print("imaginary: max=%g  min=%g  mean=%g  median=%g" % (np.max(d2), np.min(d2), np.mean(d2), np.median(d2)))

                    if (ant1==ant2).all():
                        print("Antenna1 columns match")
                    else:
                        print("WARNING: Antenna1 columns do not match")
                    if (spw1==spw2).all():
                        print("spw columns match")
                    else:
                        print("WARNING: spw columns do not match")
                    if (scan1==scan2).all():
                        print("scan columns match")
                    else:
                        print("WARNING: spw columns do not match")
                    if (field1==field2).all():
                        print("field_id columns match")
                    else:
                        print("WARNING: spw columns do not match")
    if perscan or perantenna:
        for message in messages:
            print(message)
        print("Used caltable:  ", caltable)
        print("Used caltable2: ", caltable2)
        if returnmessages:
            return messages
        if len(messages) > 0:
            return False
        else:
            return True
    return differences
    
def compareTimesFromCaltables(caltable='', caltable2='', myfilter='*gpcal.*wvr.tbl',
                              vis=''):
    """
    Written to diagnose PIPE-406.  Also works for ms subtables that have a TIME column.
    caltable: either a caltable or a working directory containing a caltable
         that matches myfilter, or a pipeline run directory
    caltable2: either a caltable or a working directory containing a caltable
         that matches myfilter, or a pipeilne run directory
    If caltable and caltable2 are blank, then search for tables that match
       the myfilter string and the vis string.
    vis: limit the caltables to include this string (preceeding myfilter)
    myfilter: limit the caltables to include this string (following vis), only
             relevant if caltable or caltable2 is a directory rather than a table
    Returns True if they match.
    -Todd Hunter
    """
    if caltable == '' and caltable2  == '':
        if vis != '':
            caltables = sorted(glob.glob('*%s%s' % (vis,myfilter)))
        else:
            caltables = sorted(glob.glob(myfilter))
        caltable = caltables[0]
        caltable2 = caltables[1]
        print("Using caltable:  ", caltable)
        print("Using caltable2: ", caltable2)
    if caltable.replace('/','')[-38:-35] in ['E2E', '201']:
        caltable = findPipelineWorkingDirectory(caltable)
        print("caltable findPipelineWorkingDirectory returns: ", caltable)
    if caltable2.replace('/','')[-38:-35] in ['E2E', '201']:
        caltable2 = findPipelineWorkingDirectory(caltable2)
        print("caltable2 findPipelineWorkingDirectory returns: ", caltable2)
    if caltable[-9:].find('/working') >= 0:
        caltables = sorted(glob.glob('%s/%s*%s*' % (caltable,vis, myfilter)))
        if len(caltables) < 1:
            print("No matching cal tables found in this directory: ", caltable)
            return
        if len(caltables) > 1:
            print("More than one caltable1 matches your selection. Use the vis or myfilter parameters.")
            for caltable in caltables:
                print(os.path.basename(caltable))
            return
        caltable = caltables[0]
        print("Found caltable=%s" % (caltable))
    if caltable2[-9:].find('/working') >= 0:
        caltables = sorted(glob.glob('%s/%s*%s*' % (caltable2,vis, myfilter)))
        if len(caltables) < 1:
            print("No matching cal tables found in this directory: ", caltable2)
            return
        if len(caltables) > 1:
            print("More than one caltable2 matches your selection. Use the vis or myfilter parameters.")
            for caltable in caltables:
                print(os.path.basename(caltable))
            return
        caltable2 = caltables[0]
        print("Found caltable2=%s" % (caltable2))

    if not os.path.exists(caltable):
        print("no such table: ", caltable)
        return
    if not os.path.exists(caltable2):
        print("no such table: ", caltable2)
        return
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    names = mytb.colnames()
    mytb.close()
    if 'ANTENNA1' in names:
        returnAntennaSpw = True
    else:
        print("Did not find ANTENNA1 in names: ", names)
        returnAntennaSpw = False
    result1 = getTimesFromCaltable(caltable,returnAntennaSpw=returnAntennaSpw)
    result2 = getTimesFromCaltable(caltable2,returnAntennaSpw=returnAntennaSpw)
    if returnAntennaSpw:
        t1, ant1, spw1, scan1, field1 = result1
        t2, ant2, spw2, scan2, field2 = result2
    else:
        t1 = result1
        t2 = result2
    print("caltable:  %d unique times (mean=%.6f MJDseconds = %s)" % (len(t1), np.mean(t1), mjdsecToDatestring(np.mean(t1))))
    print("caltable2: %d unique times (mean=%.6f MJDseconds = %s)" % (len(t2), np.mean(t2), mjdsecToDatestring(np.mean(t2))))
    if len(t1) == len(t2):
        idx = np.where(t1 != t2)[0]
        print("%d rows out of %d show differences: " % (len(idx), len(t1)))
        if returnAntennaSpw:
            print("These rows include the following unique antenna IDs: %s" % str(sorted(np.unique(ant1[idx]))))
            antennaNames = getAntennaNamesFromCaltable(caltable)
            print("These rows include the following unique antenna names: %s" % str(antennaNames[sorted(np.unique(ant1[idx]))]))
        print("statistics of direct difference (t1-t2): ")
        d = t1-t2
        print("max=%g  min=%g  median=%g" % (np.max(d), np.mean(d), np.median(d)))
        print("statistics of np.diff(t1) minus np.diff(t2): ")
        d = np.diff(t1) - np.diff(t2)
        print("max=%g  min=%g  median=%g" % (np.max(d), np.mean(d), np.median(d)))
        if returnAntennaSpw:
            if (ant1==ant2).all():
                print("Antenna1 columns match")
            else:
                print("WARNING: Antenna1 columns do not match")
            if (spw1==spw2).all():
                print("spw columns match")
            else:
                print("WARNING: spw columns do not match")
            if (scan1==scan2).all():
                print("scan columns match")
            else:
                print("WARNING: spw columns do not match")
            if (field1==field2).all():
                print("field_id columns match")
            else:
                print("WARNING: spw columns do not match")
        if np.max(d) != 0 or np.mean(d) != 0 or np.median(d) != 0:
            return False
        else:
            return True
    else:
        print("No further analysis attempted due to differing number of times.")
        return False
    
def getTimesFromCaltable(caltable, datestring=False, prec=8, keyBy='',
                         antenna='', spw='', returnAntennaSpw=False, 
                         column='TIME', uniqueValues=True, returnFlags=False):
    """
    Returns the unique list of times in MJD seconds for which solutions exist 
    in the specified caltable. 
    datestring: if True, then return the date strings 
    prec: precision to use for datestring (6: nearest second, 7: tenth sec, etc)
    antenna: restrict search to times for this antenna1 ID
    spw: restrict search to times for this spw ID
    keyBy: 'antenna', 'antennaName' or 'spw';  if set, then return a dictionary keyed by this
           parameter; be sure to set  uniqueValues=False for a complete dictionary
    returnAntennaSpw: if True, and if keyBy is not set, then also return
          matching lists for the antenna1, spw, scan, and field_id columns
    column: specify a different column other than TIME
    returnFlags: warning: if there are multiple spws with different channel
        numbers, this option will fail with array conformance error
    Todd Hunter
    """
    if not os.path.exists(caltable):
        print("au.getTimesFromCaltable(): caltable not found")
        return None
    if not os.path.isdir(caltable):
        print("%s is not a caltable" % (caltable))
        return None
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    names = mytb.colnames()
    if 'ANTENNA1' in names:
        antennaIDs = mytb.getcol('ANTENNA1')
        spws = mytb.getcol('SPECTRAL_WINDOW_ID')
    elif column not in mytb.colnames() and returnAntennaSpw:
        print("Column name not found. Available: ", mytb.colnames())
        mytb.close()
        return
    if spw == '':
        myt = mytb
    else:
        myt = mytb.query('SPECTRAL_WINDOW_ID == %s' % (str(spw)))
    times = myt.getcol(column)
    if 'SCAN_NUMBER' in names:
        scans = myt.getcol('SCAN_NUMBER')
        fields = myt.getcol('FIELD_ID')
    if returnFlags:
        flags = myt.getcol('FLAG')
    if (antenna != ''):
        idx = np.where(antennaIDs == int(antenna))
        times = times[idx]
        spws = spws[idx]
        antennaIDs = antennaIDs[idx]
#    if (spw != ''):
#        idx = np.where(spws == int(spw))
#        times = times[idx]
#        spws = spws[idx]
#        antennaIDs = antennaIDs[idx]
    if uniqueValues:
        times, idx = np.unique(times, return_index=True)
    else:
        idx = np.array(range(len(times)))
    myt.close()
    if keyBy == 'antennaName':
        mytb.open(caltable+'/ANTENNA')
        antennaNames = mytb.getcol('NAME')
    if spw != '':
        mytb.close()
    if datestring:
        times = mjdsecToUT(times, prec=prec)
    if keyBy in ['antenna','spw','antennaName']:
        mydict = {}
        for i in range(len(times)):
            if (keyBy == 'antenna'):
                key = antennaIDs[idx[i]]
            elif (keyBy == 'antennaName'):
                key = antennaNames[antennaIDs[idx[i]]]
            elif (keyBy == 'spw'):
                key = spws[idx[i]]
            if key not in mydict:
                mydict[key] = []
            mydict[key].append(times[i])
        return mydict
    else:
        if returnAntennaSpw:
            if returnFlags:
                return times, antennaIDs, spws, scans, fields, flags
            else:
                return times, antennaIDs, spws, scans, fields
        else:
            return times

def compareASDMSizeToCalibratedMS(path):
    """
    Computes the ratio of the total size of *.ms files in a pipeline run
    to the total size of the rawdata ASDMs.
    path: single string, list, or string with wildcard
          Assumes the subtree is S*/G*/M*/rawdata working etc.
    Returns: the ratio, or list of ratios
    -Todd Hunter
    """
    if type(path) == list:
        paths = path
    elif path.find('*') > 0:
        paths = glob.glob(path)
    else:
        paths = [path]
    ratios = []
    for path in paths:
        asdms = path+'/S*/G*/M*/rawdata/uid*'
        asdmSize = directorySize(asdms)
        if asdmSize <= 0:
            print("No ASDMs found. Skipping ", path)
            continue
        working = path+'/S*/G*/M*/working/*.ms'
        msSize = directorySize(working)
        ratio = msSize / float(asdmSize)
        ratios.append(ratio)
    if len(ratios) == 1:
        return ratios[0]
    else:
        print("Median +- scaled MAD: %.2f +- %.2f" % (np.median(ratios),MAD(ratios)))
        return ratios

def directorySize(path):
    """
    Returns the size in bytes of a directory tree, using os.path.getsize for
    each file.  Accepts wildcards, and will compute the sum of those paths.
    -Todd Hunter
    """
    total_size = 0
    if path.find('*') < 0:
        paths = [path]
    else:
        paths = glob.glob(path)
        print("Using %d paths" % (len(paths)))
    for path in paths:
        for dirpath, dirnames, filenames in os.walk(path):
            for f in filenames:
                fp = os.path.join(dirpath, f)
                total_size += os.path.getsize(fp)
    return total_size

def tsysFromMOUSListobs(mous):
    """
    Parses a pipeline MOUS directory for all the listobs.txt files that are not associated with
    _target.ms and runs tsysFromListobs on all of them.
    -Todd Hunter
    """
    files = []
    for dirpath, dnames, fnames in os.walk(mous):
        for f in fnames:
            if f == 'listobs.txt':
                fullname = os.path.join(dirpath, f)
                if fullname.find('_target.ms') < 0:
                    files.append(fullname)
    for f in files:
        tsysFromListobs(f)
        print("")
    
def tsysFromListobs(listfile):
    """
    Parses a listobs text file and determines the number of Tsys scans on the phase calibrator(s)
    and science target(s).
    -Todd Hunter
    """
    if not os.path.exists(listfile):
        print("Could not find file: ", listfile)
        return
    phaseCals = []
    scienceTargets = []
    phaseCalTsysScans = []
    scienceTargetTsysScans = []
    result = grep(listfile,'MeasurementSet Name: ')[0]
    vis = result.split(':')[2].strip().split()[0]
    print(os.path.basename(vis))
    result = grep(listfile,'PHASE')[0]
    if result == '':
        print("No phase calibrators.")
    else:
        lines = result.split('\n')
        for line in lines:
            loc = line.find('[')
            if loc > 0:
                firstHalf = line[:loc]
                lineNumber, startTime, dash, endTime, scan, fieldID, fieldName, nrows = firstHalf.split()
                spwlist, intents = line[loc:].split('] [')
                intents = intents.strip(']').split(',')
                phaseCals.append(fieldName)
        phaseCals = np.unique(phaseCals)
    result = grep(listfile,'TARGET')[0]
    if result == '':
        print("No science targets.")
    else:
        lines = result.split('\n')
        for line in lines:
            loc = line.find('[')
            if loc > 0:
                firstHalf = line[:loc]
                lineNumber, startTime, dash, endTime, scan, fieldID, fieldName, nrows = firstHalf.split()
                spwlist, intents = line[loc:].split('] [')
                intents = intents.strip(']').split(',')
                scienceTargets.append(fieldName)
        scienceTargets = np.unique(scienceTargets)
    result = grep(listfile,'ATMOSPHERE')[0]
    if result == '':
        print("No Tsys scans")
    else:
        lines = result.split('\n')
        for line in lines:
            loc = line.find('[')
            if loc > 0:
                firstHalf = line[:loc]
                lineNumber, startTime, dash, endTime, scan, fieldID, fieldName, nrows = firstHalf.split()
                if fieldName in phaseCals:
                    phaseCalTsysScans.append(scan)
                if fieldName in scienceTargets:
                    scienceTargetTsysScans.append(scan)
    print("%d phaseCalibrator Tsys scans: '%s'  %s" % (len(phaseCalTsysScans), ','.join(phaseCalTsysScans), ','.join(phaseCals)))
    print("%d scienceTarget   Tsys scans: '%s'  %s" % (len(scienceTargetTsysScans), ','.join(scienceTargetTsysScans), ','.join(scienceTargets)))

def getTsysBands(vis, mymsmd=None):
    """
    Returns list of band numbers that have Tsys measurements
    -Todd Hunter
    """
    needToClose = True
    if mymsmd is None:
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        needToClose = True
    tsysSpws = getTsysSpws(vis, mymsmd=mymsmd)
    spwnames = mymsmd.namesforspws(tsysSpws)
    if needToClose:
        mymsmd.close()
    bands = []
    for n,name in enumerate(spwnames):
        band = int(name.split('RB_')[1].split('#')[0])
        bands.append(band)
    bands = sorted(np.unique(bands))
    return bands

def getTsysSpwsForBand(vis, band, mymsmd=None):
    """
    Returns list of Tsys spws for a specific band number
    band: integer: 1..10
    -Todd Hunter
    """
    needToClose = True
    if mymsmd is None:
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        needToClose = True
    tsysSpws = getTsysSpws(vis, mymsmd=mymsmd)
    spwnames = mymsmd.namesforspws(tsysSpws)
    if needToClose:
        mymsmd.close()
    spws = []
    for n,name in enumerate(spwnames):
        if name.find('RB_%02d'%band) > 0:
            spws.append(tsysSpws[n])
    return spws

def compareScanTimesWithSysCal(vis, scans=[], verbose=False, maxRows=-1,
          colorsForBands = ['','k','k','k','k','k','k','grey','k','k','k'],
                               antennaToPlot=0, plotfile='', fontsize=8):
    """
    Finds which scan each row of the SYSCAL table is aligned with (if any).
    verbose: if False, then only list rows that align with any scan
    scans: limit search to this list of scans
    plotfile: Boolean or string (default name: vis+'.compareScanTimes.png')
    maxRows: -1 means all rows
    """
    syscal = os.path.join(vis,'SYSCAL')
    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)
    if len(scans) == 0:
        scans = mymsmd.scannumbers()
    times = {}
    pb.clf()
    t0 = np.min(mymsmd.timesforscan(scans[0]))
    lw = 4
    for i,scan in enumerate(scans):
        times[scan] = mymsmd.timesforscan(scan)
        scanspws = ','.join([str(j) for j in getTsysSpwsForScan(vis,scan,mymsmd=mymsmd)])
        y = 0.01+i*0.02
        pb.plot([np.min(times[scan])-t0,np.max(times[scan])-t0], 2*[y],'-',lw=lw)
        pb.text(np.mean(times[scan])-t0, y, 'scan %d (%s)' % (scan,scanspws), ha='center', size=fontsize)
    y0 = 0.01+(len(scans)+1)*0.02
    pb.xlabel('Time (seconds) relative to start of scan %d' % (scans[0]))
    pb.draw()
    pb.title(os.path.basename(vis) + ' (antenna=%d)' % antennaToPlot)
    mytb = createCasaTool(tbtool)
    mytb.open(syscal)
    tsys_start_times = mytb.getcol('TIME')
    tsys_intervals = mytb.getcol('INTERVAL')
    antennas = mytb.getcol('ANTENNA_ID')
    spws = mytb.getcol('SPECTRAL_WINDOW_ID')
    bands = []
    for spw in spws:
        bands.append(bandforspw(spw, vis, mymsmd))
    mytb.close()
    tsys_start_times = tsys_start_times - 0.5*tsys_intervals
    tsys_end_times = tsys_start_times + tsys_intervals
    mymsmd.close()
    rowsPlotted = 0
    for i in range(len(tsys_start_times)):
        found = False
        if i < maxRows or maxRows < 0:
            y = y0 + rowsPlotted*0.02
            if antennaToPlot == antennas[i]:
                if tsys_end_times[i] - tsys_start_times[i] < 1e5:
                    pb.plot([tsys_start_times[i]-t0, tsys_end_times[i]-t0], 2*[y], '-', lw=lw, color=colorsForBands[bands[i]])
                    pb.text(tsys_end_times[i]-t0+25, y, 'row%d'%i, size=fontsize, ha='left', va='center')
                else:
                    # interval often goes off to infinity on final scan
                    pb.plot([tsys_start_times[i]-t0], [y], 'o', color=colorsForBands[bands[i]])
                    pb.text(tsys_start_times[i]-t0+25, y, 'row%d'%i, size=fontsize, ha='left', va='center')
                x0 = 0.11*np.diff(pb.xlim())
                pb.text(tsys_start_times[i]-t0-x0, y, 'B%d,spw%d' % (bands[i],spws[i]), ha='left', size=fontsize, va='center')
                
                rowsPlotted += 1
        for scan in scans:
            if np.min(times[scan]) <= tsys_end_times[i] and np.max(times[scan]) >= tsys_start_times[i] and antennas[i] == antennaToPlot:
                found = True
                print("Row %d (antenna %d, spw %d) aligns with scan %d (interval=%.1f)" % (i,antennas[i],spws[i],scan,tsys_intervals[i]))
        if not found and verbose:
            print("Row %d (antenna %d, spw %d) aligns with no scan" % (i,antennas[i],spws[i]))
    y0,y1 = pb.ylim()
    pb.ylim([y0, y1+0.05*(y1-y0)])
    x0,x1 = pb.xlim()
    pb.xlim([x0-0.08*(x1-x0), x1+0.1*(x1-x0)])
    if plotfile != '':
        if plotfile == True:
            plotfile = os.path.basename(vis) + '.compareScanTimes.png'
        pb.savefig(plotfile)

def getScansFromCaltable(caltable, field=''):
    """
    Returns the unique list of scans for which solutions exist in 
    the specified caltable. 
    field: if specified, restrict to this field ID or name, or list thereof
         including comma-delimited string
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("au.getScansFromCaltable(): caltable not found")
        return -1
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    scans = []
    if field == '':
        scans = list(np.unique(mytb.getcol('SCAN_NUMBER')))
    else:
        vis = getMeasurementSetFromCaltable(caltable)
        if not os.path.exists(vis):
            vis = os.path.join(os.path.dirname(caltable), vis)
        if os.path.exists(vis):
            fields = parseFieldArgument(vis,field)[0]
            if len(fields) > 0:
                for field in fields:
                    myt = mytb.query('FIELD_ID==%d'%(field))
                    scans += list(np.unique(myt.getcol('SCAN_NUMBER')))
                myt.close()
                scans = np.unique(scans) # probably not necessary
        else:
            print("Could not find measurement set.")
    mytb.close()
    return np.array(scans)

def getFieldsFromCaltable(caltable):
    """
    Returns the unique list of field IDs for which solutions exist in 
    the specified caltable.  See also getFieldIDsFromCaltable (which 
    uses the casac tool instead of the tb tool.)
    See also au.getFieldNamesFromCaltable
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("au.getFieldsFromCaltable(): caltable not found")
        return -1
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    fields = np.unique(mytb.getcol('FIELD_ID'))
    mytb.close()
    return fields

def snrFromCaltables(caltable, intent='PHASE', stat='median', bins=None, 
                     plotfile='', snr=[], xlim2=[0,58], ylim2=[0,0.71]):
    """
    Generates a histogram of the per-spw, per-field, per-interval SNRs (using 
    the requested statistic over antennas and polarizations) from the specified
    list of caltables.
    caltable: python list or wildcard string containing a 'hifa' table type
     e.g. '/lustre/naasc/sciops/comm/sbooth/pipeline/root/5.1.0_validation/*/S*/G*/M*/working/*hifa_timegaincal*solintint.gpcal.tbl'
      or  '/lustre/naasc/sciops/comm/sbooth/pipeline/root/5.1.0_validation/*/S*/G*/M*/working/*hifa_gfluxscale.s15*solintint.gpcal.tbl'
    intent: limit the solutions to this intent (must be 'PHASE' or 'CHECK')
    stat: 'max', 'median', 'min', 'scaledMad', 'fractionBelow3sigma', 'fractionBelow2sigma'
    bins: default is range(0,max,1)
    plotfile: default is snrFromCaltables.png
    snr: a pre-computed list from a prior run of this function, to allow rapid
         re-plotting after changing the code
    Returns: list of SNRs
    """
    if type(caltable) == str:
        if (caltable.find('*')>=0):
            caltables = glob.glob(caltable)
        else:
            caltables = caltable.split(',')
        tableType = caltable.split('hifa_')[1].split('.tbl')[0]
    else:
        caltables = caltable
        tableType = caltable[0].split('hifa_')[1].split('.tbl')[0]
    projects = 0
    lowsnr = {}
    if len(snr) == 0:
        print("Working on %d caltables" % (len(caltables)))
        intents = {}
        for i,caltable in enumerate(caltables):
            msname = getMeasurementSetFromCaltable(caltable)
            vis = os.path.join(os.path.dirname(caltable),msname)
            if not os.path.exists(vis):
                print("Could not find measurement set")
                continue
            if intent == 'PHASE':
                # Here we assume that there will be a phase calibrator.
                # Otherwise, we would have to do similar logic as for CHECK.
                field = getPhaseCalibrators(vis)[0]
            elif intent.find('CHECK') >= 0:
                if msname not in intents:
                    mymsmd = createCasaTool(msmdtool)
                    mymsmd.open(vis)
                    intents[msname] = mymsmd.intents()
                    mymsmd.close()
                myintents = intents[msname]
                if 'OBSERVE_CHECK_SOURCE#ON_SOURCE' in myintents:
                    field = getPhaseCalibrators(vis, intent='OBSERVE_CHECK_SOURCE#ON_SOURCE')
                else:
                    field = []
                if len(field) == 0: 
                    print("%3d: No CHECKSOURCE in %s (%s)" % (i, os.path.basename(caltable), os.path.dirname(caltable)))
                    continue
                field = field[0]
            else:
                print("Unrecognized intent.  Must be either PHASE or CHECK.")
                return
            mydict = snrFromCaltable(caltable, field=field)
            for spw in mydict:
                for field in list(mydict[spw].keys()):
                    mysnr = mydict[spw][field][stat]
                    print("SNR=%.1f in spw %d field %d in %s" % (mysnr, spw, field, msname))
                    if mysnr < 3:
                        if caltable not in lowsnr:
                            lowsnr[caltable] = {}
                        lowsnr[caltable][spw] = mysnr
                    snr.append(mysnr)
            projects += 1
    else:
        print("No recomputing results, since snr was specified: ", snr)
    print("min %s snr, max %s snr = " % (stat,stat), min(snr), max(snr))
    print("lowsnr results: ", lowsnr)
    if bins is None:
        inc = 1
        bins = np.arange(0, np.ceil(max(snr))+inc*1.5, inc)
    pb.clf()
    pb.subplot(211)
    pb.hist(snr, bins)
    pb.xlim([0,max(snr)+max([1,0.01*max(snr)])])
    pb.xlabel('%s SNR in %s table' % (stat,tableType))
    pb.ylabel('Number of occurrences')
    pb.title(intent+' Calibrators from %d C5.1P2 pipeline benchmark datasets (%d spws)' % (projects, len(snr))) # len(caltables),len(snr)))

    adesc = pb.subplot(212)
    pb.hist(snr, bins, cumulative=True, normed=True)
    pb.xlim(xlim2)
    pb.ylim(ylim2)
    pb.xlabel('%s SNR in %s table' % (stat,tableType))
    pb.ylabel('Cumulative fraction with SNR < x')
    pb.grid(True)
    majorLocator = MultipleLocator(5)
    adesc.xaxis.set_major_locator(majorLocator)
    adesc.yaxis.grid(True,which='both')
    pb.show()
    if plotfile == '':
        plotfile = 'snrFromCaltables.png'
    pb.savefig(plotfile)
    return snr

def lowSnrFractionFromCaltable(caltable, spw='', field='', minsnr=2):
    """
    Uses snrFromCaltable to build a subset dictionary keyed by spw
    and field ID, with values being >0 and either fractionBelow2sigma or
    fractionBelow3sigma, depending on value of minsnr.
    spw: limit spw to this spw or list of spws (integer, string, list or 
         comma-delimited string)
    field: limit field selection to this field ID, name or list thereof (or
         comma-delimited string)
    minsnr: 2 or 3
    -Todd Hunter
    """
    mydict = snrFromCaltable(caltable, spw, field)
    spws = mydict
    newdict = {}
    for spw in spws:
        newdict[spw] = {}
        fields = mydict[spw]
        for field in fields:
            newdict[spw][field] = {}
            if minsnr == 2:
                newdict[spw][field] = mydict[spw][field]['fractionBelow2sigma']
            elif minsnr == 3:
                newdict[spw][field] = mydict[spw][field]['fractionBelow3sigma']
    return newdict

def snrFromCaltable(caltable, spw='', field='', doplot=False, plotfile='',
                    perscan=False, titleFontsize=10, plotvstime=False, 
                    plotvschan=False, intent='', maxbinwidth=10, 
                    projectCode='', verbose=False, ignoreZerosInPlot=True, 
                    startXAxisAtZero=False, snrRange=[2,3]):
    """
    Extracts the SNR from a caltable (either timegaincal or bandpass), on a per-spw, 
    per-field basis and computes the median, min, max, and scaled MAD of each,
    and the fraction values between two values of SNR (set via snrRange).
    The polarizations (if there are more than one) are all considered together.
    caltable: path to caltable to use (or './' to find first one in current dir)
    spw: limit spw to this spw or list of spws (integer, string, list or 
         comma-delimited string)
    field: limit field selection to this field ID, name or list thereof (or
         comma-delimited string)
    intent: limit field selection to this intent (if field=='')
    doplot: if True, plot a histogram of the values per spw per field
            multi-panel plot has: spws = columns, fields = rows
    plotfile: string, or True=automatic name: caltable+'_histogram.png'
    perscan: if True, then break down statistics per scan
    maxbinwidth: used to compute second argument to pb.hist(); if bins<10 bins, then maxbinwidth/2,4 etc. is used
    plotvstime: if True, then plot snr vs. time instead of a histogram (only if perscan=False)
    plotvschan: if True, then plot snr vs. channel instead of a histogram (only if perscan=False)
    projectCode: add this to title
    Returns: a dictionary of SNR values keyed by spw, with subdictionaries keyed
       by field ID (integer), and optionally, subdictionaries keyed by scan number
       The '10%ile' and '25%ile' values are computed over non-zero values, unless all are zero.
    See also: au.getFlagsFromCaltable
    -Todd Hunter
    """
    if caltable == './':
        caltable = findPipelineAmpCalTable('./',returnFirstTable=True)
        print("Found ", caltable)
    if (not os.path.exists(caltable)):
        print("au.getSpwsFromCaltable(): caltable not found")
        return -1
    caltable = caltable.rstrip('/')
    spws = getSpwsFromCaltable(caltable)
    fields = getFieldsFromCaltable(caltable)
    snrs = {}
    if spw != '':
        spws = parseSpwArgument(spw, spws)
    vis = getMeasurementSetFromCaltable(caltable)
    if not os.path.exists(vis):
        vis = os.path.join(os.path.dirname(caltable), vis)
    if os.path.exists(vis):
        mymsmd = msmdtool()
        mymsmd.open(vis)
        if field != '':
            fields,fieldNames = parseFieldArgument(vis, field, mymsmd=mymsmd)
        elif intent != '':
            fields = getPhaseCalibrators(vis,intent=intent,byname=False,mymsmd=mymsmd)
            fieldNames = getPhaseCalibrators(vis,intent=intent,byname=True,mymsmd=mymsmd)
            print("Using fields: ", fieldNames)
        else:
#            print("Calling au.parseFieldArgument('%s','%s',False)" % (vis,fields))
            # we need False here to protect against E2E5.1.00015 where phase and check source are the same as the 2 targets!
            fields,fieldNames = parseFieldArgument(vis, fields, False)
    else:
        fieldNames = ''
    if plotvschan:
        uniqueAntennaNames = getAntennaNamesFromCaltable(caltable) 
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    if plotfile != '':
        doplot = True
    if doplot:
        pb.clf()
        singlePanelDesc = pb.subplot(1,1,1)
        i = 0
    nspws = len(spws)
    nfields = len(fields)
    medianValues = {}
    nAntennas = len(np.unique(getAntennaNamesFromCaltable(caltable)))
    for myspw in spws:
        snrs[myspw] = {}
        bandwidth = getBandwidthFromCaltable(caltable,myspw)
        meanfreq = getSpwMeanFreqFromCaltable(caltable,myspw)
        for field in fields:
            print("Working field: ", field)
            if doplot:
                i += 1
                if nfields > 1:
                    nrows = nfields
                    ncols = nspws
                else:
                    if plotvschan:
                        # 8 spws will be 4 rows, 2 columns
                        ncols = int(np.sqrt(nspws))
                        nrows = nspws/ncols
                    else:
                        # 8 spws will be 2 rows, 4 columns
                        nrows = int(np.sqrt(nspws))
                        ncols = nspws/nrows
                if (nrows * ncols * i) == 1:
                    desc = singlePanelDesc  # avoid deprecation warning
                else:
                    desc = pb.subplot(nrows, ncols, i)
                if len(fields) > 3:
                    pb.subplots_adjust(hspace=0.45)
                else:
                    pb.subplots_adjust(hspace=0.3)
            if perscan:
                scans = getScansFromCaltable(caltable, field)
                snrs[myspw][field] = {}
                allsnr = np.array([])
                for scan in scans:
                    myt = mytb.query('SPECTRAL_WINDOW_ID == %d && FIELD_ID == %d && SCAN_NUMBER == %d' % (myspw, field, scan))
                    snr = myt.getcol('SNR')
                    fraction = np.where((snr < 3)*(snr>0))[0]
                    fraction = float(np.shape(fraction)[0]) / np.prod(np.shape(snr))
                    fraction2 = np.where((snr < 2)*(snr>0))[0]
                    fraction2 = float(np.shape(fraction2)[0]) / np.prod(np.shape(snr))
                    snrNonZero = snr.flatten()[np.where(snr.flatten()>0)[0]]
                    if len(snrNonZero) < 1:
                        print("No SNRs > 0")
                        snrNonZero = snr
                    idx = np.where((snrNonZero < snrRange[1]) * (snrNonZero > snrRange[0]))[0]
                    fractionInRange = float(len(idx)) / len(snrNonZero)
                    snrs[myspw][field][scan] = {'median': np.median(snr), 'min': np.min(snr), 'max': np.max(snr), 'scaledMAD': MAD(snr), 'fractionBelow3sigma': fraction, 'fractionBelow2sigma': fraction2, '25%ile': scoreatpercentile(snrNonZero,25), '10%ile': scoreatpercentile(snrNonZero,10), 'fractionInRange': inRange}
                    allsnr = np.append(allsnr,snr)
                snr = allsnr
                fraction = np.where((snr < 3)*(snr>0))[0]
                fraction = float(np.shape(fraction)[0]) / np.prod(np.shape(snr))
                fraction2 = np.where((snr < 2)*(snr>0))[0]
                fraction2 = float(np.shape(fraction2)[0]) / np.prod(np.shape(snr))
                snrNonZero = snr.flatten()[np.where(snr.flatten()>0)[0]]
                if len(snrNonZero) < 1:
                    print("No SNRs > 0")
                    snrNonZero = snr
                snrs[myspw][field]['allscans'] = {'median': np.median(snr), 'min': np.min(snr), 'max': np.max(snr), 'scaledMAD': MAD(snr), 'fractionBelow3sigma': fraction, 'fractionBelow2sigma': fraction2, '25%ile': scoreatpercentile(snrNonZero,25), '10%ile': scoreatpercentile(snrNonZero,10)}
            else:
                query = 'SPECTRAL_WINDOW_ID == %d && FIELD_ID == %d' % (myspw, field)
                myt = mytb.query(query)
                if verbose:
                    print("Query: ", query)
                snr = myt.getcol('SNR')
                antenna1 = myt.getcol('ANTENNA1')
                antenna2 = myt.getcol('ANTENNA2')
                mytime = myt.getcol('TIME')
                uniqueTimes = np.unique(mytime)
                if len(uniqueTimes) > 1:
                    medianSolutionInterval = np.median(np.diff(np.unique(mytime)))
                    minSolutionInterval = np.min(np.diff(np.unique(mytime)))
                    maxSolutionInterval = np.max(np.diff(np.unique(mytime)))
                else:
                    medianSolutionInterval = 0
                    minSolutionInterval = 0
                    maxSolutionInterval = 0
                if os.path.exists(vis):
                    scans = getScansFromCaltable(caltable, field)
                    print("Scans on field %s: %s" % (str(field), scans))
                    myscans = mymsmd.scansforspw(myspw)
                    integrationTime = None
                    for scan in scans:
                        if scan in myscans:
                            integrationTime = mymsmd.exposuretime(scan, myspw)
                            break
                    if integrationTime is None:
                        print("No data for spw %d in scans %s" % (myspw, scans))
                        continue
                else:
                    integrationTime = None
                if len(np.shape(snr)) == 0 or np.prod(np.shape(snr))==0:
                    print("Skipping spw=%s, field=%s, because shape(snr) = " % (str(myspw),str(field)), np.shape(snr))
                    continue
                if np.shape(snr)[0] == 1:
                    print("Found a single pol solution in spw %d of %s" % (myspw, caltable))
#                else:
#                    print "Found a %d-pol solution in spw %d of %s" % (np.shape(snr)[0], myspw, caltable)
                fraction = np.where((snr < 3)*(snr>0))[0]
                fraction = float(np.shape(fraction)[0]) / np.prod(np.shape(snr))
                fraction2 = np.where((snr < 2)*(snr>0))[0]
                fraction2 = float(np.shape(fraction2)[0]) / np.prod(np.shape(snr))
                snrNonZero = snr.flatten()[np.where(snr.flatten()>0)[0]]
                if len(snrNonZero) < 1:
                    print("No SNRs > 0 in spw %d" % myspw)
                    snrNonZero = snr
                else:
                    print("%d SNRs > 0 in spw %d" % (len(snrNonZero), myspw))
                idx = np.where((snrNonZero < snrRange[1]) * (snrNonZero > snrRange[0]))[0]
                fractionInRange = float(len(idx)) / len(snrNonZero)
#                print("types: myspw=%s, field=%s" % (type(myspw),type(field)))
                snrs[myspw][field] = {'median': np.median(snr), 'min': np.min(snr), 'max': np.max(snr), 'scaledMAD': MAD(snr), 'fractionBelow3sigma': fraction, 'fractionBelow2sigma': fraction2, 'medianInterval': medianSolutionInterval, 'minInterval': minSolutionInterval, 'maxInterval': maxSolutionInterval, '25%ile': scoreatpercentile(snrNonZero,25), '10%ile': scoreatpercentile(snrNonZero,10), 'medianIgnoringZeros': np.median(snrNonZero), 'fractionInRange': fractionInRange}
                if integrationTime is not None:
                    snrs[myspw][field]['integrationTime'] = integrationTime
                myt.close()
            if doplot:
                if plotvstime:
                    timeStamps = pb.date2num(mjdSecondsListToDateTime(mytime))
                    pb.plot_date(timeStamps,snr[0][0],'ko')
                    setXaxisTimeTicks(desc, np.min(mytime), np.max(mytime))
                    if len(snr) > 1:
                        pb.plot(timeStamps,snr[1][0],'ro')
                    pb.xlabel('Time')
                    pb.ylabel('SNR in spw %d' % (myspw))
                elif plotvschan:
                    # shape is pol, chan, antenna
                    npols = np.shape(snr)[0]
                    nchans = np.shape(snr)[1]
                    nants = np.shape(snr)[2]
                    # colorize by antenna ID = antenna1 array
                    medianValues[myspw] = {}
                    for ant in range(len(uniqueAntennaNames)):
                        medianValues[myspw][ant] = []
                    for pol in range(npols):
                        for chan in range(nchans):
                            for ant in range(nants):
                                pb.plot(chan, snr[pol][chan][ant], '.', 
                                        color=overlayColors[antenna1[ant]], 
                                        mec=overlayColors[antenna1[ant]])
                                medianValues[myspw][antenna1[ant]] += [snr[pol][chan][ant]]
                    for ant in range(len(uniqueAntennaNames)):
                        medianValues[myspw][ant] = np.median(medianValues[myspw][ant])
                    pb.xlabel('Solution Channel')
                    padPlotLimits(0.05, True, False, True, False)
                    pb.ylabel('SNR (spw %d)' % (myspw))
                else:
                    snr = snr.flatten()
                    if ignoreZerosInPlot:
                        snr = snr[np.where(snr > 0)]
                    bins = len(np.arange(np.min(snr), np.max(snr), maxbinwidth))
                    while bins < 10:
                        maxbinwidth /= 2.0
                        bins = len(np.arange(np.min(snr), np.max(snr), maxbinwidth))
                    pb.hist(snr, bins)
                    if startXAxisAtZero:
                        pb.xlim([0,pb.xlim()[1]])
                    pb.xlabel('SNR in spw %d (freq=%.1f GHz, bandwidth=%.1f MHz)' % (myspw,meanfreq,bandwidth*1e-6))
                    if ((i-1) % ncols) > 0:
                        desc.set_yticklabels([])
                    elif nfields == 1:
                        pb.ylabel('Occurrences')
                    else:
                        pb.ylabel('Field %s' % (str(field)))
    mytb.close()
    if doplot:
        pb.text(0.5,0.99, os.path.basename(caltable), 
                transform=pb.gcf().transFigure, va='top', ha='center', size=titleFontsize)
        if len(fields) == 1:
            mytext = '%s (%d antennas) field: %s = %s' % (projectCode, nAntennas, str(fields),str(fieldNames))
        else:
            mytext = '%s (%d antennas)' % (projectCode, nAntennas) # fields are given in the y-axis label
        if plotvschan:
            mytext += '  (colored by antenna)'
        pb.text(0.5,0.94, mytext, transform=pb.gcf().transFigure, va='top', 
                ha='center', size=12)
        if plotvschan:
            if len(spws) == 1:
                DrawAntennaLegend(desc, uniqueAntennaNames, medianValues[spws[0]])
            else:
                DrawAntennaLegend(singlePanelDesc, uniqueAntennaNames)
        pb.draw()
        if plotfile != '' and plotfile != False:
            caltable = os.path.basename(caltable)
            if plotfile == True:
                if plotvstime:
                    plotfile = caltable+'_SNRvstime.png'
                elif plotvschan:
                    if spw == '':
                        plotfile = caltable+'_SNRspectrum.png'
                    else:
                        plotfile = caltable+'_SNRspectrum_spw%s.png' % ('_'.join([str(i) for i in spws]))
                else:
                    plotfile = caltable+'_SNRhistogram.png'
            pb.savefig(plotfile)
            print("Wrote ", plotfile)
    if os.path.exists(vis):
        mymsmd.close()
    return snrs

def getSpwsFromCaltable(caltable, getNumberOfChannels=False):
    """
    Returns the unique list of spw IDs for which solutions exist in 
    the specified caltable, using the tb tool. 
    getNumberOfChannels: if True, then return a dictionary, with values=nchan
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("au.getSpwsFromCaltable(): caltable not found")
        return -1
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    spws = np.unique(mytb.getcol('SPECTRAL_WINDOW_ID'))
    mytb.close()
    if getNumberOfChannels:
        mytb.open(caltable+'/SPECTRAL_WINDOW')
        spwdict = {}
        for spw in spws:
            spwdict[spw] = len(mytb.getcell('CHAN_FREQ',spw))
        mytb.close()
        return spwdict
    else:
        return spws

def getSpwMeanFreqFromCaltable(caltable, spw):
    """
    Returns the mean frequency (in GHz) of the specified spw in a caltable.
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("caltable not found")
        return -1
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    spectralWindowTable = mytb.getkeyword('SPECTRAL_WINDOW').split()[1]
    mytb.close()
    mytb.open(spectralWindowTable)
    spws = range(len(mytb.getcol('MEAS_FREQ_REF')))
    chanFreqGHz = []
    for i in spws:
        # The array shapes can vary.
        chanFreqGHz.append(np.mean(1e-9 * mytb.getcell('CHAN_FREQ',i)))
    mytb.close()
    return chanFreqGHz[spw]

def getChanWidthFromCaltable(caltable, spw=None):
    """
    Returns the width of a channel (in Hz) of the specified spw in a caltable.
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("caltable not found")
        return -1
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    spectralWindowTable = mytb.getkeyword('SPECTRAL_WINDOW').split()[1]
    mytb.close()
    mytb.open(spectralWindowTable)
    if spw is None:
        chanwidth = mytb.getcol('CHAN_WIDTH')[0]
    else:
        chanwidth = mytb.getcell('CHAN_WIDTH',spw)[0]
    return chanwidth
    
def getBandwidthFromCaltable(caltable, spw=None):
    """
    Returns the bandwidth (in Hz) of the specified spw in a caltable.
    spw: if None, then it returns the value for the first spw.
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("caltable not found")
        return -1
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    spectralWindowTable = mytb.getkeyword('SPECTRAL_WINDOW').split()[1]
    mytb.close()
    mytb.open(spectralWindowTable)
    if spw is None:
        bandwidth = mytb.getcol('TOTAL_BANDWIDTH')[0]
    else:
        nrows = mytb.nrows()
        if spw >= nrows:
            print("There are only %d rows in this table" % (nrows))
            mytb.close()
            return
        bandwidth = mytb.getcell('TOTAL_BANDWIDTH',spw)
    mytb.close()
    return bandwidth
    
def getNChanFromCaltable(caltable, spw=None):
    """
    Returns the number of channels of the specified spw in a caltable.
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("caltable not found")
        return -1
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    spectralWindowTable = mytb.getkeyword('SPECTRAL_WINDOW').split()[1]
    mytb.close()
    mytb.open(spectralWindowTable)
    if spw is None:
        nchan = mytb.getcol('NUM_CHAN')
    else:
        nchan = mytb.getcell('NUM_CHAN',spw)
    return nchan
    
def getChanFreqFromCaltable(caltable, spw, channel=None):
    """
    Returns the frequency (in GHz) of the specified spw channel in a caltable.
    channel: if not specified, then return array of all channel frequencies
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("caltable not found")
        return -1
    mytb = createCasaTool(tbtool)
    mytb.open(caltable)
    spectralWindowTable = mytb.getkeyword('SPECTRAL_WINDOW').split()[1]
    mytb.close()
    mytb.open(spectralWindowTable)
    spws = range(len(mytb.getcol('MEAS_FREQ_REF')))
    chanFreqGHz = {}
    for i in spws:
        # The array shapes can vary, so read one at a time.
        spectrum = mytb.getcell('CHAN_FREQ',i)
        chanFreqGHz[i] = 1e-9 * spectrum
    mytb.close()
    if channel is None:
        return chanFreqGHz[spw]
    if channel > len(chanFreqGHz[spw]):
        print("spw %d has only %d channels"% (spw, len(chanFreqGHz[spw])))
    else:
        return chanFreqGHz[spw][channel]

def getConfig(vis):
    """
    Reads the ASDM_EXECBLOCK table for the configName. (available Cycle 5 onward)
    -Todd Hunter
    """
    if not os.path.exists(vis):
        print("Could not find measurement set")
        return
    mytable = os.path.join(vis,'ASDM_EXECBLOCK')
    if not os.path.exists(mytable):
        print("Could not find ASDM_EXECBLOCK table")
        return
    mytb = tbtool()
    mytb.open(mytable)
    configNames = mytb.getcol('configName')
    mytb.close()
    return configNames[0]

def getConfigFromASDM(asdm):
    """
    Reads the ASDM ExecBlock.xml file for the configName.
    -Todd Hunter
    """
    if not os.path.exists(asdm):
        print("Could not find ASDM.")
        return
    result = grep(asdm+'/ExecBlock.xml', 'configName')[0]
    if len(result) == 0:
        print("No configName in the ExecBlock.xml file.")
        return
    configName = result.split('configName>')[1].replace('</','')
    return configName

def configs(telescope='alma', cycle='', verbose=True):
    """
    List the configurations in the data/alma/simmos directory.  To view them:
    os.system('ls '+os.getenv('CASAPATH').split()[0]+'/data/alma/simmos')
    cycle: integer or string integer;  use 'cycle' to ignore the 'outN.cfg' files
    -Todd Hunter
    """
    if verbose:
        print("Searching directory = ", os.getenv('CASAPATH').split()[0]+'/data/alma/simmos')
    if telescope != 'WSRT':
        telescope = telescope.lower()
    if (cycle != ''):
        if cycle != 'cycle':
            cycle = 'cycle' + str(cycle)
        else:
            cycle = 'cycle'
    myglob = os.getenv('CASAPATH').split()[0]+'/data/alma/simmos/%s*%s*'%(telescope,cycle)
    biglist = sorted([os.path.basename(i) for i in glob.glob(myglob)])
    uniqueList = []
    # remove the duplicates for cycle 1
    for b in biglist:
        if b.find('_') < 0:
            uniqueList.append(b)
    return uniqueList

def getBaselineLengthForStations(station1, station2, config='alma.all.cfg'):
    """
    Computes the baseline length between two ALMA stations.
    station1: name of first pad
    station2: name of second pad
    config: config file to search.  For ACA, use alma.aca.cfg, for VLA-A, use vla.a.cfg
    -Todd Hunter
    """
    if (not os.path.exists(config)):
        config = os.getenv('CASAPATH').split()[0]+'/data/alma/simmos/'+config
        if (not os.path.exists(config)):
            print("Could not find config file")
            return
    mylist = np.array(getBaselineLengths(config=config, verbose=False))
    mykey = 'pad'+station1+'-pad'+station2
    mylengths = mylist[:,1]
    mypads = mylist[:,0]
    if (mykey in mypads):
        print("found %s" % (mykey))
        return float(mylengths[np.where(mypads==mykey)[0][0]])
    else:
        myOtherkey = 'pad'+station2+'-pad'+station1
        if (myOtherkey in mypads):
            print("found %s" % (myOtherkey))
            return float(mylengths[np.where(mypads==myOtherkey)[0][0]])
        else:
            print("Baseline not found: %s or %s" % (mykey, myOtherkey))
            return 0

def getAntennaStationsFromCaltable(caltable) :
    """
    Returns the antenna names from the specified caltable.
    Todd Hunter
    """
    if (os.path.exists(caltable) == False):
        print("au.getAntennaStationsFromCaltable(): caltable not found")
        return -1
    mytb = createCasaTool(tbtool)
    mytb.open(caltable+'/ANTENNA')
    names = mytb.getcol('STATION')
    mytb.close()
    return names

def imagesChannelFrequency(imglist, channel=0):
    """
    Calls imageChannelFrequency for a list of images and
    returns a dictionary.
    imglist: a list or a comma-delimited string or a string that contains a wildcard
    """
    if type(imglist) == str:
        if imglist.find('*') >= 0:
            imglist = glob.glob(imglist)
        else:
            imglist = imglist.split(',')
    mydict = {}
    for img in imglist:
        freq = imageChannelFrequency(img, channel)
        mydict[img] = freq
        print("%.14f %s" % (freq, img))
    return mydict

def imageChannelFrequency(img, channel=0, myia=None):
    """
    Returns the frequency (in GHz) of the specified channel of any CASA or FITS
    image cube with a spectral axis.
    channel: can be a scalar or a list/vector, if None then return list for all channels, 
             if -1 then return the final channel
    Returns: value in GHz; scalar (if one channel) or array (if more than 1 channel)
    -Todd Hunter
    """
    if not os.path.exists(img):
        print("Could not find image")
        return
    pixel = [0,0,0,channel]
    if myia is None or myia == '':
        myia = createCasaTool(iatool)
        try:
            myia.open(img)
        except:
            print("Could not open image: %s, check permissions" % (img))
            return
        needToClose = True
    else:
        needToClose = False
    maxchan = numberOfChannelsInCube(img, myia=myia) - 1
    if type(channel) not in [list, range, np.ndarray]:
        if channel is not None:
            if (channel < -1 or channel > maxchan):
                print("Invalid channel %d, max=%d" % (channel,maxchan))
                if needToClose:
                    myia.close()
                return
    elif type(channel) == range:
        channel = list(channel)
    spectralAxis = findSpectralAxis(myia)
    freq = []
    nchan = myia.shape()[spectralAxis]
    if type(channel) != list and type(channel) != np.ndarray:
        if channel is None:
            channels = range(nchan)
        elif channel == -1:
            channels = [nchan-1]
        else:
            channels = [channel]
    else:
        channels = channel
    for channel in channels:
        pixel[spectralAxis] = channel
        mydict = myia.toworld(pixel)
        if (len(mydict['numeric']) <= spectralAxis):
            print("Not a cube?")
            return
        freq.append(mydict['numeric'][spectralAxis])
    if needToClose:
        myia.close()
    freq = np.array(freq)
#    print("type(freq) = ", type(freq[0]))
    if len(freq) == 1:
        # the first channel is all we have
        freq = freq[0]
    return freq * 1e-9

def imageFrequencyChannel(frequencies, img=None, n=None, firstfreq=None, lastfreq=None, 
                          returnInt=True, findcont=True, myia=None):
    """
    Returns the channel closest in frequency to the specified frequency in an
    image cube with a spectral axis.
    frequencies: can be a python list or comma-delimited string of floating point values
         in Hz, GHz, or string with units (threshold is 10000)
    img: read n, firstfreq, lastfreq from this image
    n: number of channels
    firstfreq: frequency of first channel (in Hz or GHz)
    lastfreq: frequency of last channel (in Hz or GHz)
    Returns: channel number, either floating point or nearest integer
    -Todd Hunter
    """
    if img is not None:
        if not os.path.exists(img):
            print("Could not find image")
            return
        n, firstfreq, lastfreq = numberOfChannelsInCube(img, returnFreqs=True, myia=myia)
    elif (n is None or firstfreq is None or lastfreq is None):
        print("If you do not specify img, you need to specify, n, firstfreq and lastfreq.")
        return
    else:
        firstfreq = parseFrequencyArgumentToHz(firstfreq)
        lastfreq = parseFrequencyArgumentToHz(lastfreq)
    if type(frequencies) != list and type(frequencies) != np.ndarray:
        if type(frequencies) == str:
            frequencies = frequencies.split(',')
        else:
            frequencies = [frequencies]
    channels = []
    for i,frequency in enumerate(frequencies):
        frequency = parseFrequencyArgumentToHz(frequency)
        cdelt = (lastfreq-firstfreq)/(n-1)
        channel = (frequency-firstfreq)/cdelt
        if returnInt:
            if findcont:
                if i==0:
                    channel = int(np.ceil(channel))
                else:
                    channel = int(np.floor(channel))
            else:
                channel = int(np.round(channel))
        channels.append(channel)
    if len(channels) == 1:
        channels = channels[0]
    return channels

def imageChannel(img, velocity):
    """
    Returns the channel corresponding to the specified velocity (in km/s) 
    in an image cube with a spectral axis.
    velocity: can be a scalar or a list/vector (in km/s)
    Returns: scalar (if one channel) or array (if more than 1 velocity)
    Not yet implemented.
    -Todd Hunter
    """
    if not os.path.exists(img):
        print("Could not find cube.")
        return
    myia = createCasaTool(iatool)
    myia.open(img)
    pixel = np.zeros(len(myia.shape()))  # initialize an array to [0,0,0,0]
    spectralAxis = findSpectralAxis(myia)
    nchannels = myia.shape()[spectralAxis]
    mychannel = []
    if type(velocity) != list and type(velocity) != np.ndarray:
        velocities = [velocity]
    else:
        velocities = velocity
    for velocity in velocities:
        # stopped here
        chanvelocity = []
        for channel in range(nchannels):
            pixel[spectralAxis] = channel
            mydict = myia.toworld(pixel, 'm', dovelocity=True)
            vel = mydict['measure']['spectral']['radiovelocity']['m0']['value']
            unit = mydict['measure']['spectral']['radiovelocity']['m0']['unit']
            if unit=='m/s':
                vel *= 0.001
            elif unit != 'km/s':
                print("Unrecognized velocity unit: ", unit)
                return
            chanvelocity.append(vel)
        idx = np.argsort(np.abs(np.array(chanvelocity)-velocity))
        deltaV = np.abs(chanvelocity[idx[0]] - chanvelocity[idx[1]])
        mychannel.append((np.abs(chanvelocity[idx[0]]-velocity)*idx[1] + np.abs(chanvelocity[idx[1]]-velocity)*idx[0]) / deltaV)
    myia.close()
    mychannel = np.array(mychannel)
    if len(mychannel) == 1:
        mychannel = mychannel[0]
    return mychannel
    
def imageChannelVelocityWidth(img='', showEquation=False, myia=''):
    """
    Returns velocity of channel 1 minus velocity of channel 0.
    -Todd Hunter
    """
    v0 = imageChannelVelocity(img, 0, showEquation, myia)
    v1 = imageChannelVelocity(img, 1, showEquation, myia)
    return (v1-v0)

def imageBandwidth(img='', showEquation=False, myia=None, nchan=None):
    """
    Returns absolute value in GHz.
    """
    if nchan is None:
        nchan = numberOfChannelsInCube(img, myia=myia)
    return nchan * abs(imageChannelFrequencyWidth(img,myia))

def imageChannelFrequencyWidth(img='', myia=None):
    """
    Returns frequency of channel 1 minus channel 0 (in GHz).
    myia: currently unused
    -Todd Hunter
    """
    if not os.path.exists(img):
        print("Could not find image")
        return
    v0 = imageChannelFrequency(img, channel=0, myia=myia)
    if v0 is None:
        return
    v1 = imageChannelFrequency(img, channel=1, myia=myia)
    if v1 is None:
        return
    return (v1-v0)

def imageChannelVelocity(img='', channel=0, showEquation=False, myia=None):
    """
    Returns the velocity (in km/s) of the specified channel of any CASA or FITS
    image cube with a spectral axis.
    channel: can be a scalar or a list/vector
    showEquation: if True, then derive an equivalent equation that could be used
    Returns: scalar (if one channel) or array (if more than 1 channel)
    -Todd Hunter
    """
    if img == '' and myia == '':
        print("You must specify either img or myia")
        return
    pixel = [0,0,0,0]
    if not os.path.exists(img):
        print("Could not find image")
        return
    if myia == '' or myia is None:
        myia = createCasaTool(iatool)
        myia.open(img)
        needToClose = True
    else:
        needToClose = False
    spectralAxis = findSpectralAxis(myia)
    velocity = []
    if type(channel) != list and type(channel) != np.ndarray:
        channels = [channel]
    else:
        channels = channel
    for channel in channels:
        pixel[spectralAxis] = channel
        mydict = myia.toworld(pixel, 'm', dovelocity=True)
        vel = mydict['measure']['spectral']['radiovelocity']['m0']['value']
        unit = mydict['measure']['spectral']['radiovelocity']['m0']['unit']
        if unit=='m/s':
            vel *= 0.001
        elif unit != 'km/s':
            print("Unrecognized velocity unit: ", unit)
            if needToClose: myia.close()
            return
        velocity.append(vel)
        if channel == channels[0] and showEquation:
            pixel[spectralAxis] = channel+1
            mydict = myia.toworld(pixel, 'm', dovelocity=True)
            vel2 = mydict['measure']['spectral']['radiovelocity']['m0']['value'] 
            if unit=='m/s':
                vel2 *= 0.001
            increment = vel2 - vel
            print("Equivalent equation: velocity = %f + channel*%f" % (vel,increment))
    if needToClose: myia.close()
    velocity = np.array(velocity)
    if len(velocity) == 1:
        velocity = velocity[0]
    return velocity

def imageRestFrequency(img):
    """
    Returns the rest frequency (in GHz) stored in a CASA or FITS image header
    -Todd Hunter
    """
    restfreq = parseFrequencyArgumentToGHz(imhead(img, mode='get', hdkey='restfreq'))
    return restfreq

def listImageRestFreq(img, includeChan0freq=True):
    """
    Prints the rest frequency from the header of one or more images.
    img: a list of images as a list or comma-delimited string; or a string with wildcards
    (See tt.fixRestFreq to change the value, or imageRestFrequency to return value for 1 image)
    -Todd Hunter
    """
    if (type(img) == str):
        if (img.find('*') >= 0):
            img = sorted(glob.glob(img))
        else:
            img = img.split(',')
    print("RestFreq Chan0  image")
    for i in img:
        restfreq = parseFrequencyArgumentToGHz(imhead(i, mode='get', hdkey='restfreq'))
        if (includeChan0freq):
            freq = imageChannelFrequency(i)
            print("%.6f %.6f GHz  %s" % (restfreq, freq, i))
        else:
            print("%.6f GHz  %s" % (restfreq, i))

def medianTsysForRestFrequencySpw(vis, restfreq, field='3'):
    """
    Finds the median Tsys of the Tsys spw that maps to the science spw
    that contains the requested rest frequency (as defined in the OT).
    restfreq: requested value in GHz, Hz, or a string with units
    """
    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)
    spw = spwForRestFrequency(vis, restfreq, mymsmd=mymsmd)
    tsysspw = tsysspwmapWithNoTable(vis, field, spw, mymsmd=mymsmd)
    print("Using science spw = %d (tsys=%d)" % (spw, tsysspw))
    tsys = medianTsysForFieldFromSyscalTable(vis, field, tsysspw, mymsmd=mymsmd)
    mymsmd.close()
    return tsys

def spwForRestFrequency(vis, restfreq, warningThresholdMHz=10, mymsmd=''):
    """
    Finds the spw that contains the rest frequency definition (from the OT)
    that is closest to the requested value.
    restfreq: requested value in GHz, Hz or a string with units
    """
    hz = parseFrequencyArgumentToHz(restfreq)
    mydict = restFrequencies(vis, mymsmd=mymsmd)
    values = []
    spws = []
    for i in mydict:
        spws += [i]*len(mydict[i])
        values += list(mydict[i])
    values = np.array(values)
    freqdiff = np.abs(hz - values)
    idx = np.argmin(freqdiff)
    spw = spws[idx]
    if freqdiff[idx] > warningThresholdMHz*1e6:
        print("Closest rest frequency differs by more than %.1fMHz from the request." % (warningThresholdMHz))
    return spw
    
def restFrequencies(vis, spw='', source='', intent='OBSERVE_TARGET', 
                    verbose=False, showIF=False, showSpwFreq=False, 
                    showBB=False, mymsmd=''):
    """
    Returns a dictionary of rest frequencies (in Hz) for specified spw(s) (and source).
    spw: list or comma-delimited string;  if blank, then use all science spws
    source: can be integer ID or string name or list or comma-delimited string
    intent: if source is blank, then use first one with matching intent and spw
    showIF: if True, then show the IF frequency as well in the dictionary
    showSpwFreq: if True, then show the spw center frequency as well in the dictionary
    showBB: if True, then show the center frequency of the baseband (via the SQLD spws)
    """
    if (not os.path.exists(vis)):
        print("Could not find ms")
        return
    freq = {}
    needToClose = False
    if mymsmd == '':
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        needToClose = True
    if (spw == ''):
        spw = getScienceSpws(vis, returnString=False, mymsmd=mymsmd)
    else:
        spw = parseSpw(vis, spw, mymsmd=mymsmd)
    if showIF:
        LOs = interpretLOs(vis, mymsmd=mymsmd)  # dictionary keyed by spw
    if showBB:
        basebandFreqs = getScienceBasebandFrequencies(vis, mymsmd=mymsmd)
        if basebandFreqs is None:
            print("The showBB option is not available for this dataset.")
            showBB = False
    for myspw in spw:
        freq[myspw] = restFrequency(vis, myspw, source, intent, verbose, 
                                    mymsmd=mymsmd)
        if showIF or showBB or showSpwFreq:
            freq[myspw] = {'frequency': freq[myspw]}
        if showIF:
            ifFreq = freq[myspw]['frequency'] - LOs[myspw]  # negative values are LSB
            freq[myspw]['IF'] = ifFreq
        if showSpwFreq:
            spwFreq = getMeanFreqOfSpwlist(vis, myspw, mymsmd=mymsmd)
            freq[myspw]['spwCenterFreq'] = spwFreq
        if showBB:
            baseband = mymsmd.baseband(myspw)
            freq[myspw]['basebandFreq'] = basebandFreqs[baseband]
            freq[myspw]['baseband'] = baseband
    if needToClose:
        mymsmd.close()
    return freq

def restFrequenciesASDM(asdm):
    """
    Reads the rest frequencies for the science target spws in an ASDM and returns a dictionary,
    keyed by the spw number, as they will appear in a measurement set once it is imported.
    -Todd Hunter
    """
    if (not os.path.exists(asdm)):
        print("Could not find ASDM")
        return
    if (not os.path.exists(asdm+'/Source.xml')):
        print("Could not find Source.xml")
        return
    xmlscans = minidom.parse(asdm+'/Source.xml')
    rowlist = xmlscans.getElementsByTagName("row")
    restFreqs = {}
    for rownode in rowlist:
        row = rownode.getElementsByTagName("velRefCode")
        if (len(row) > 0):
            row = rownode.getElementsByTagName("spectralWindowId")
            spw = int(str(row[0].childNodes[0].nodeValue).split('_')[1])
            row = rownode.getElementsByTagName("restFrequency")
            tokens = (row[0].childNodes[0].nodeValue).split()
            restFreqs[spw] = []
            for i in range(int(tokens[1])):
                restFreqs[spw].append(float(tokens[2+i]))
    scienceSpwsASDM = list(restFreqs.keys())
    spwmap = asdmspwmap(asdm)
    scienceSpws = []
    mydict = getSpwsFromASDM(asdm)
    restFreqMS = {}
    for spw in scienceSpwsASDM:
        myspw = spwmap.index(spw)
        if mydict[myspw]['numChan'] > 1:
            restFreqMS[myspw] = restFreqs[spw]
    return restFreqMS

def restFrequency(vis, spw, source='', intent='OBSERVE_TARGET', verbose=True, 
                  mymsmd=None):
    """
    Returns the list of rest frequencies (in Hz) for specified spw (and source).
    source: can be integer ID or string name
    intent: if source is blank, then use first one with matching intent and spw
    """
    if (not os.path.exists(vis)):
        print("restFrequency: Could not find measurement set")
        return
    needToClose = False
    if mymsmd is None:
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        needToClose = True
    spw = int(spw)
    if (spw >= mymsmd.nspw()):
        print("spw %d not in the dataset" % (spw))
        if needToClose:
            mymsmd.close()
        return
    meanfreq = mymsmd.meanfreq(spw)
    bandwidth = mymsmd.bandwidths(spw)
    chanwidth = mymsmd.chanwidths(spw)[0]
    mytb = createCasaTool(tbtool)
    mytb.open(vis+'/SOURCE')
    spws = mytb.getcol('SPECTRAL_WINDOW_ID')
    sourceIDs = mytb.getcol('SOURCE_ID')
    names = mytb.getcol('NAME')
    spw = int(spw)
    if (type(source) == str):
        if (source.isdigit()):
            source = int(source)
        elif (source == ''):
            # pick source
            fields = mymsmd.fieldsforintent(intent+'*')
            fields2 = mymsmd.fieldsforspw(spw)
            fields = np.intersect1d(fields,fields2)
            source = mymsmd.namesforfields(fields[0])[0]
            if verbose:
                print("For spw %d, picked source: " % (spw), source)
    if (type(source) == str):
        sourcerows = np.where(names==source)[0]
        if (len(sourcerows) == 0):
            # look for characters ()/ and replace with underscore
            names = np.array(sanitizeNames(names))
            if verbose:
                print("Checking %s against sanitized names = " % source, names)
            sourcerows = np.where(names==source)[0]
    else:
        sourcerows = np.where(sourceIDs==source)[0]
    if needToClose:
        mymsmd.close()
    spwrows = np.where(spws==spw)[0]
    row = np.intersect1d(spwrows, sourcerows)[0]
    if mytb.iscelldefined('REST_FREQUENCY',row):
        freq = mytb.getcell('REST_FREQUENCY',row)
    else:
        freq = 0
        print("Rest frequency cell not defined for this source/spw combination (row=%s)." % row)
        return([])
    mytb.close()
    velocity = c_mks*0.001*(freq-meanfreq)/freq
    deltav = c_mks*0.001*bandwidth/freq
    if verbose:
        print("Velocity of central channel = %f km/s (width = %f km/s = %f Hz)" % (velocity, chanwidth*c_mks*0.001/freq, chanwidth))
        print("Width of spw = %f km/s,  87.5%% = %f km/s, half that = %f km/s" % (deltav,0.85*deltav,0.5*0.85*deltav))
    return(freq)

def approximateBarycentricCoordinatesOfEarth(date=''):
    """
    Based on astronomical almanac for 1998.
    """
    jd = mjdToJD(dateStringToMJD(date,verbose=False))
    X = -0.00450 + 0.999760*np.cos(2*np.pi * (jd - 2451078.5) / 365.24219 )
    Y = -0.01389 + 0.916742*np.cos(2*np.pi * (jd - 2451169.5) / 365.24219 )
    return(X,Y)

def meanObliquity(date=''):
    """
    Returns the mean obliquity in radians for the specified date/time.
    Explanatory supplment to astronomical almanac, equation 3.222-1
    date: blank means use the current time; otherwise:
    Input date format: 2011/10/15 05:00:00  or   2011/10/15-05:00:00
                    or 2011-10-15 05:00:00  or   2011-10-15-05:00:00
                    or 2011-10-15T05:00:00  or   2011-Oct-15 etc.
    """
    if (date == ''):
        mjd = getMJD()
    else:
        mjd = dateStringToMJD(date,verbose=False)
    t = (mjdToJD(mjd) - 2451545.0)/36525.
    epsilon0 = 23+26/60.+21.448/3600.+(-46.8160*t - 0.00059*t**2 + 0.001813*t**3)/3600.
    return(np.radians(epsilon0))

def radecParallaxToRadec(parallax, radec='', date='', vis='', field=-1,
                         verbose=False):
    """
    parallax: a value in arcsec
    radec: a sexagesimal string
    date: date format: 2011/10/15 05:00:00  or   2011/10/15-05:00:00
                    or 2011-10-15 05:00:00  or   2011-10-15-05:00:00
                    or 2011-10-15T05:00:00  or   2011-Oct-15 etc.
    vis: measurement set from which to read date (if it is not specified)
    field: field in vis from which to read radec
    """
    if (radec == ''):
        if (vis== '' or field<0):
            print("Must specify vis&field or radec")
            return
        if (not os.path.exists(vis)):
            print("Could not find measurement set.")
            return
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        radec = direction2radec(mymsmd.phasecenter(int(field)))
        mymsmd.close()
        if (date == ''):
            date = getObservationStartDate(vis)
    rarad, decrad = radec2rad(radec)
    c = np.cos(rarad)/np.cos(decrad)/15.0 # see B23 of 1998 almanac for /15
    d = np.sin(rarad)/np.cos(decrad)/15.0 # see B23 of 1998 almanac for /15
    epsilon0 = meanObliquity(date)
    cprime = np.tan(epsilon0)*np.cos(decrad) - np.sin(rarad)*np.sin(decrad)
    dprime = np.cos(rarad)*np.sin(decrad)
    X,Y = approximateBarycentricCoordinatesOfEarth(date)
    rao = 15*np.cos(decrad)* parallax * (d*X - c*Y)
    deco = parallax * (dprime*X - cprime*Y)
    if verbose:
        print("X=%f, Y=%f, c=%f, d=%f, c'=%f, d'=%f, rao=%f, dec=%f" % (X, Y, c, d, cprime, dprime, rao, deco))
    newradec = radecOffsetToRadec([rarad,decrad], rao, deco, verbose=False)
    result = angularSeparationOfStrings(radec,newradec,returnComponents=True,verbose=False)
    separation,raSep,decSep,raSepCosDec,positionAngle = result
    print('Shift = %f arcsec = %.2f%% (%f" in RA, %f" in Dec)' % (separation*3600, separation*360000/parallax, raSepCosDec*3600, decSep*3600))
    return newradec

def applyProperMotion(vis, source, field=-1, otCoordinates='', parallax=0, 
                      verbose=True):
    """
    Reads the proper motion for a source, then subtracts it from the phase center
    of the specified field in order to produce the epoch 2000 position, which is returned
    as a sexagesimal string.  The measurement set is unchanged.  You can use au.editFieldDirection
    if you want to set this new value.
    source: ID (integer or string integer or name)
    field: ID (integer or string integer or name), if not specified, then use first field for the source
    otCoordinates: if specified, then also compute angular separation in arcsec from these coordinates
    -Todd Hunter
    """
    if (not os.path.exists(vis)):
        print("properMotion: Could not find measurement set")
        return
    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)
    if type(source) == str:
        if source not in mymsmd.sourcenames():
            if source.isdigit():
                source = int(source)
            else:
                print("Source name not found among: %s" % (str(mymsmd.sourcenames())))
                mymsmd.close()
                return
        # leave it as a name string, since properMotion can accept it this way
    else:
        source = int(source)
    if (field < 0 and type(source) is not str):
        uniqueIDs = np.unique(mymsmd.sourceidsfromsourcetable())
        if source in uniqueIDs:
            field = mymsmd.fieldsforsource(source)[0]
        else:
            print("There are only %d sources in this ms." % (len(uniqueIDs)))
            return
    else:
        if field < 0:
            field = source
        fieldids, fieldnames = parseFieldArgument(vis, field)
        field = fieldids[0]
    pmra, pmdec = properMotion(vis, source)
    mydir = mymsmd.phasecenter(field)
    mytime = np.mean(mymsmd.timesforfield(field))
    mymsmd.close()
    ra = mydir['m0']['value']
    dec = mydir['m1']['value']
    phaseCenter = rad2radec(ra,dec, prec=7, verbose=False)
    if verbose: print("Phase center = ", phaseCenter)
    years = (mytime - dateStringToMJDSec('2000/01/01 12:00',verbose=False))/SECONDS_PER_YEAR
    print("Applying %f years of motion" % (years))
    rao = -pmra*years
    deco = -pmdec*years
    radec = radecOffsetToRadec([ra,dec], rao, deco, verbose=False)
    if verbose: print("Inferred epoch 2000 center = ", radec)
    if (otCoordinates != ''):
        separation = angularSeparationOfStrings(radec,otCoordinates, verbose=False)
        if (parallax > 0):
            percentage = 3600*100.*separation/parallax
            print("Difference from OT coordinates = %f arcsec = %.2f%% of the parallax." % (separation*3600,percentage))
        else:
            print("Difference from OT coordinates = %f arcsec" % (separation*3600))
    else:
        separation, longsep, latsep, longsepcosdec, pa = angularSeparationOfStrings(radec, phaseCenter, returnComponents=True, returnArcsec=True)
        print("Angular motion = %g arcsec:  RA = %g arcsec, Dec = %g arcsec" % (separation, longsepcosdec, latsep))
    return radec

def properMotionConversion(value,value2=None):
    """
    value: string with units: either rad/s or arcsec/yr
    value2: second value (optional)
    Returns: the value in the other units.  If value2 is specified, then also return their vector sum.
    """
    if value.find('rad/s') >= 0:
        value = float(value.replace('rad/s','')) * ARCSEC_PER_RAD * SECONDS_PER_YEAR
    elif value.find('arcsec/yr') >= 0:
        value = float(value.replace('arcsec/yr','')) / ARCSEC_PER_RAD / SECONDS_PER_YEAR
    else:
        print("Unrecognized unit, must be rad/s or arcsec/yr")
    if value2 is None:
        return value
    if value2.find('rad/s') >= 0:
        value2 = float(value2.replace('rad/s','')) * ARCSEC_PER_RAD * SECONDS_PER_YEAR
    elif value2.find('arcsec/yr') >= 0:
        value2 = float(value2.replace('arcsec/yr','')) / ARCSEC_PER_RAD / SECONDS_PER_YEAR
    else:
        print("Unrecognized unit for value2, must be rad/s or arcsec/yr")
    return value,value2,np.linalg.norm([value,value2])
    
def properMotionASDM(asdm, source=None, arcsecPerYear=True):
    """
    Reads the proper motion from the ASDM Source.xml file 
    and returns a dictionary, keyed by the source name
    -Todd Hunter
    """
    if (not os.path.exists(asdm)):
        print("Could not find ASDM")
        return
    if (not os.path.exists(asdm+'/Source.xml')):
        print("Could not find Source.xml")
        return
    xmlscans = minidom.parse(asdm+'/Source.xml')
    rowlist = xmlscans.getElementsByTagName("row")
    properMotion= {}
    for rownode in rowlist:
        row = rownode.getElementsByTagName("sourceName")
        tokens = (row[0].childNodes[0].nodeValue).split()
        sourceName = str(tokens[0])
        row = rownode.getElementsByTagName("properMotion")
        if (len(row) > 0 and sourceName not in properMotion):
            tokens = (row[0].childNodes[0].nodeValue).split()
            properMotion[sourceName] = [float(tokens[2]), float(tokens[3])]
            if arcsecPerYear:
                properMotion[sourceName] = list(np.degrees(np.array(properMotion[sourceName]))*3600*SECONDS_PER_YEAR)
    if source is not None and source in properMotion:
        properMotion = properMotion[source]
    return properMotion

def properMotion(vis, source=None, spw='', arcsecPerYear=True):
    """
    Reads the SOURCE table of a measurement set and returns the proper 
    motion for specified source in native units (rad/sec) or in arcsec/year.
    source: can be integer ID or string name, if None, then show all of them
    -Todd Hunter
    """
    if (not os.path.exists(vis)):
        print("properMotion: Could not find measurement set")
        return
    mytb = createCasaTool(tbtool)
    mytb.open(vis+'/SOURCE')
    spws = mytb.getcol('SPECTRAL_WINDOW_ID')
    sources = mytb.getcol('SOURCE_ID')
    names = mytb.getcol('NAME')
    if (type(source) == str):
        if (source.isdigit()):
            source = int(source)
    if (type(source) == str):
        source = sources[np.where(names==source)[0]][0]
    
    if source is not None:
        sourcerows = np.where(sources==source)[0]
        if (len(sourcerows) == 0):
            print("Did not find %s in %s" % (str(source), str(sources)))
            return
        if (spw == ''):
            row = sourcerows[0]
        else:
            spwrows = np.where(spws==int(spw))[0]
            row = np.intersect1d(spwrows, sourcerows)[0]
        try:
            propermotion = mytb.getcell('PROPER_MOTION',row)
        except:
            propermotion = 0
            print("Source does not match with spw.")
        mytb.close()
        if arcsecPerYear:
            propermotion = np.degrees(propermotion)*3600*SECONDS_PER_YEAR
        return(propermotion)
    else:
        pm = {}
        for i,source in enumerate(sources):
            sourcerows = np.where(sources==source)[0]
            if (spw == ''):
                row = sourcerows[0]
            else:
                spwrows = np.where(spws==int(spw))[0]
                row = np.intersect1d(spwrows, sourcerows)[0]
            try:
                pm[names[i]] = mytb.getcell('PROPER_MOTION',row)
            except:
                pm[names[i]] = 0
                print("Source does not match with spw.")
            if arcsecPerYear:
                pm[names[i]] = np.degrees(pm[names[i]])*3600*SECONDS_PER_YEAR
        return pm

def radialVelocity(vis, source, spw=''):
    """
    Returns the systemic velocity (or list) for specified spw and source (in m/s).
    source: can be integer ID or string name
    spw: integer or string; if not specified, then use first TDM or
         FDM spw with OBSERVE_TARGET intent (that also has a defined velocity,
         as the opposite sideband in Bands9 and 10 do not)
    -Todd Hunter
    """
    if (not os.path.exists(vis)):
        print("radialVelocity: Could not find measurement set")
        return
    mytb = createCasaTool(tbtool)
    mytb.open(vis+'/SOURCE')
    spws = mytb.getcol('SPECTRAL_WINDOW_ID')
    sources = mytb.getcol('SOURCE_ID')
    names = mytb.getcol('NAME')
    if (type(source) == str):
        if (source.isdigit()):
            source = int(source)
    if (type(source) == str):
        sourcerows = np.where(names==source)[0]
    else:
        sourcerows = np.where(sources==source)[0]
    if (spw == ''):
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        almaspws = mymsmd.almaspws(tdm=True,fdm=True)
        targetspws = mymsmd.spwsforintent('OBSERVE_TARGET*')
        mymsmd.close()
        potentialSpws = np.intersect1d(np.intersect1d(targetspws, spws),almaspws)
        # not all spws will have a velocity defined if 90deg Walsh-switching is engaged
        for spw in potentialSpws:
            spwrows = np.where(spws==spw)[0]
            row = np.intersect1d(spwrows, sourcerows)[0]
            if mytb.iscelldefined('SYSVEL',row):
                print("Choosing spw = ", spw)
                break
    else:
        spw = int(spw)
    spwrows = np.where(spws==spw)[0]
    row = np.intersect1d(spwrows, sourcerows)[0]
    frame = 'unknown frame'
    veltype = 'velocity'
    units = 'm/s'
    try:
        velocity = mytb.getcell('SYSVEL',row)
        if len(velocity) == 1:
            velocity = velocity[0]
        mydict = mytb.getcolkeywords('SYSVEL')
        if 'MEASINFO' in mydict:
            frame = mydict['MEASINFO']['Ref']
            veltype = mydict['MEASINFO']['type']
        if 'QuantumUnits' in mydict:
            units = mydict['QuantumUnits'][0]
    except:
        velocity = 0
        print("Source does not match with spw.")
    mytb.close()
    velocity_kms = velocity*1e-3
    print("%s = %s %s (%s)" % (veltype, str(velocity), units, frame))
    return(velocity)

def representativeSpw(vis, checkTarget=True, verbose=True):
    """
    Reads the representative frequency from the measurement set, then computes which science
    spw(s) contains it.
    checkTarget: if True, then check whether the rep target is actually obsreved in the 
        rep spw (SCIREQ-1735, PIPE-377)
    -Todd Hunter
    """
    freq = representativeFrequency(vis, verbose, reportSpw=False)
    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)
    spws = getScienceSpwsForFrequency(vis, freq, mymsmd=mymsmd)
    if (len(spws) == 1):
        value = spws[0]
    elif (len(spws) == 0):
        print("No spws cover the representative frequency (%g GHz)" % (freq))
        spws = getScienceSpws(vis, mymsmd=mymsmd, returnString=False)
        print("Spw central frequencies in GHz: ", np.array([mymsmd.meanfreq(spw) for spw in spws]) * 1e-9)
        value = None
    else:
        print("Multiple spws (%s) cover the representative frequency (%g GHz)" % (str(spws),freq))
        print("Returning the one nearest to the center.")
        spw = getScienceSpwsForFrequency(vis, freq, nearestOnly=True, mymsmd=mymsmd)
        value = spw
    if checkTarget:
        mydict = representativeSource(vis, verbose, mymsmd)
        fieldID = mydict.keys()[0]
        fieldName = mydict.values()[0]
        spws = mymsmd.spwsforfield(fieldID)
        if value not in spws:
            print("WARNING: representativeSource (%s) was not observed in representativeSpw (%d)" % (fieldName,value))
        else:
            print("The representativeSource (%s) was indeed observed in representativeSpw (%d)" % (fieldName,value))
    mymsmd.close()
    return value

def observingMode(vis):
    """
    Reads the observing mode from the ASDM_SUMMARY table (if imported).
    -Todd Hunter
    """
    if (not os.path.exists(vis+'/ASDM_SBSUMMARY')):
        print("ASDM_SBSUMMARY table does not exist for this measurement set.")
        return
    mytb = createCasaTool(tbtool)
    mytb.open(vis+'/ASDM_SBSUMMARY')
    mode = mytb.getcol('observingMode')
    return(mode)

def representativeFrequency(vis, verbose=True, reportSpw=True):
    """
    Get the representative frequency from the ASDM_SBSUMMARY table of a
    measurement set, if it has been imported with asis.
    e.g. [representativeFrequency = 230.0348592858192 GHz, ...] 
    verbose: if True, then also print the min/max acceptable angular resolutions
    reportSpw: if True, then also report the spw that contains this frequency
    Returns the value in GHz.
    """
    if (not os.path.exists(vis)):
        print("Could not find measurement set.")
        return
    mytb = createCasaTool(tbtool)
    if (not os.path.exists(vis+'/ASDM_SBSUMMARY')):
        print("Could not find ASDM_SBSUMMARY table.  Did you not import it with asis='SBSummary'?")
        return
    mytb.open(vis+'/ASDM_SBSUMMARY')
    scienceGoal = mytb.getcol('scienceGoal')
    mytb.close()
    freq = 0
    minAcceptableResolution = 0
    maxAcceptableResolution = 0
    bw = None
    for args in scienceGoal:
        for arg in args:
            loc = arg.find('representativeFrequency')
            if (loc >= 0):
                tokens = arg[loc:].split()
                freq = parseFrequencyArgumentToGHz(tokens[2]+tokens[3])
            loc = arg.find('representativeBandwidth')
            if (loc >= 0):
                tokens = arg[loc:].split()
                bw = parseFrequencyArgumentToGHz(tokens[2]+tokens[3])
            loc = arg.find('minAcceptableAngResolution')
            if (loc >= 0):
                tokens = arg[loc:].split()
                minAcceptableResolution = float(tokens[2])
                minUnits = tokens[3]
            loc = arg.find('maxAcceptableAngResolution')
            if (loc >= 0):
                tokens = arg[loc:].split()
                maxAcceptableResolution = float(tokens[2])
                maxUnits = tokens[3]
    if verbose:
        print("minAcceptableResolution = %f %s" % (minAcceptableResolution, minUnits))
        print("maxAcceptableResolution = %f %s" % (maxAcceptableResolution, maxUnits))
        if bw is not None:
            print("representativeBandwidth = %f GHz" % (bw))
        if reportSpw:
            print("Looking for spw that contains the representative frequency (%.3f GHz)..." % (freq))
            spw = representativeSpw(vis, verbose=False)
            if spw is not None:
                print("representative spw = ", spw)
    return(freq)

def representativeSource(vis, verbose=True, mymsmd=None):
    """
    Get the representative source from the ASDM_SBSUMMARY table of a
    measurement set, if it has been imported with asis, via the scienceGoal column.
    Also checks if the centerDirection, assumed to be in RA(deg), Dec(deg), matches this source.
    If the direction is not (0,0), i.e. Cycle 5 onward data, then return a 
    dictionary of {fieldID: fieldName}
    -Todd Hunter
    """
    if (not os.path.exists(vis)):
        print("Could not find measurement set.")
        return
    mytb = createCasaTool(tbtool)
    if (not os.path.exists(vis+'/ASDM_SBSUMMARY')):
        print("Could not find ASDM_SBSUMMARY table.  Did you not import it with asis='SBSummary'?")
        return
    mytb.open(vis+'/ASDM_SBSUMMARY')
    goals = mytb.getcol('scienceGoal')
    fieldName = ''
    for goal in goals:
        if goal[0].find('Source') > 0:
            fieldName = goal[0].split('=')[1].strip()
    direction = mytb.getcol('centerDirection')   # value is in degrees
    directionCode = mytb.getcol('centerDirectionCode')
    directionEquinox = mytb.getcol('centerDirectionEquinox')
    mytb.close()
    if (direction[0][0] == 0. and direction[1][0] == 0.):
        print("The values are not filled.  It should be present starting in Cycle 5 data (CPM6).")
        return
    fieldID,separation = findNearestFieldInVis(vis, deg2radec(direction[0][0], direction[1][0], verbose=False), 
                                               returnSeparation=True)
    if verbose:
        print("Separation = %f deg" % separation)
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    else:
        needToClose = False
    fieldNameCheck = mymsmd.namesforfields(fieldID)[0]
    if needToClose:
        mymsmd.close()
    if fieldName == '':
        print("There was no representativeSource specified in the ASDM_SBSUMMARY table.")
    elif fieldNameCheck != fieldName:
        print("There is a mismatch between the scienceGoal (%s) and the centerDirection (%s)" % (fieldName, fieldNameCheck))
    elif verbose:
        print("The scienceGoal name (%s) is indeed the closest field to the centerDirection (%s)" % (fieldName, fieldNameCheck))
    return({fieldID: fieldName})

def updateRepresentativeSource(vis, newSource):
    """
    Simpler alternative to updateSBSummary when you only want to change the source.
    """
    mytb = tbtool()
    mytb.open(vis+'/ASDM_SBSUMMARY', nomodify=False)
    scienceGoal = mytb.getcol('scienceGoal')
    newValues = []
    for i,args in enumerate(scienceGoal):
        for arg in args:
            loc = arg.find('representativeSource')
            if loc < 0:
                newValues += [[arg]]
            else:
                newValues += [['representativeSource = %s' % newSource]]
    newValues = np.array(newValues)
    mytb.putcol('scienceGoal', newValues)
    mytb.close()

def updateSBSummary(vis, representativeFrequency=None, 
                    minAcceptableAngResolution=None, 
                    maxAcceptableAngResolution=None,
                    dynamicRange=None, representativeBandwidth=None,
                    representativeSource=None, representativeSPW=None,
                    maxAllowedBeamAxialRatio=None):
    """
    Updates the ASDM_SBSUMMARY table of a measurement set with one or more new
    values.  If a value is not present in the existing table and also not 
    specified, then it will remain not present in the updated table.
    representativeFrequency: float value in typical units (GHz), 
            or string with units (space before units is optional)
    minAcceptableAngResolution: float value in typical units (arcsec), 
            or string with units (space before units is required)
    maxAcceptableAngResolution: float value in typical units (arcsec), 
            or string with units (space before units is required)
    dynamicRange: float value
    representativeBandwidth: float value in typical units (GHz), 
            or string with units (space before units is optional)
    representativeSource: string
    representativeSPW: string
    maxAllowedBeamAxialRatio: float
    -Todd Hunter
    """
    if not os.path.exists(vis):
        print("Could not find ms.")
        return
    t = vis+'/ASDM_SBSUMMARY'
    if not os.path.exists(t):
        print("Could not find ASDM_SBSUMMARY table for this ms.  Was it imported?")
        return
    if (representativeFrequency is not None or 
        minAcceptableAngResolution is not None or 
        maxAcceptableAngResolution is not None or maxAllowedBeamAxialRatio is not None or
        dynamicRange is not None or representativeBandwidth is not None or
        representativeSource is not None or representativeSPW is not None):
        update = True
    else:
        update = False
    mytb = createCasaTool(tbtool)
    mytb.open(vis+'/ASDM_SBSUMMARY', nomodify=False)
    scienceGoal = mytb.getcol('scienceGoal')
    numScienceGoal = mytb.getcol('numScienceGoal')[0]
    representativeSpwPresent = False
    axialRatioPresent = False
    # Read the existing values for those that were not specified
    for i,args in enumerate(scienceGoal):
        # args will look like: 
        # array(['representativeFrequency = 219.55647641503566 GHz'])
        for arg in args:
            # arg will look like: 
            #  'representativeFrequency = 219.55647641503566 GHz'
            loc = arg.find('representativeFrequency')
            if (loc >= 0 and representativeFrequency is None):
                tokens = arg[loc:].split()
                representativeFrequency = float(tokens[2])
                freqUnits = tokens[3]
            loc = arg.find('minAcceptableAngResolution')
            if (loc >= 0 and minAcceptableAngResolution is None):
                tokens = arg[loc:].split()
                minAcceptableAngResolution = float(tokens[2])
                minUnits = tokens[3]
            loc = arg.find('maxAcceptableAngResolution')
            if (loc >= 0 and maxAcceptableAngResolution is None):
                tokens = arg[loc:].split()
                maxAcceptableAngResolution = float(tokens[2])
                maxUnits = tokens[3]
            loc = arg.find('dynamicRange')
            if (loc >= 0 and dynamicRange is None):
                tokens = arg[loc:].split()
                dynamicRange = float(tokens[2])
            loc = arg.find('representativeBandwidth')
            if (loc >= 0 and representativeBandwidth is None):
                tokens = arg[loc:].split()
                representativeBandwidth = float(tokens[2])
                bwUnits = tokens[3]
            loc = arg.find('representativeSource')
            if (loc >= 0 and representativeSource is None):
                tokens = arg[loc:].split()
                representativeSource = str(tokens[2])
            loc = arg.find('representativeWindow')
            if (loc >= 0 and representativeSPW is None):
                tokens = arg[loc:].split()
                representativeSPW = str(tokens[2])
                representativeSpwPresent = True
            loc = arg.find('maxAllowedBeamAxialRatio')
            if (loc >= 0 and maxAllowedBeamAxialRatio is None):
                tokens = arg[loc:].split()
                maxAllowedBeamAxialRatio = float(tokens[2])
                axialRatioPresent = True
    if update:
        # convert any command-line arguments from string to value and units
        if type(representativeFrequency) is str:
            representativeFrequency = parseFrequencyArgumentToGHz(representativeFrequency)
            freqUnits = 'GHz'
        if type(representativeBandwidth) is str:
            representativeBandwidth = parseFrequencyArgumentToGHz(representativeBandwidth) * 1000
            bwUnits = 'MHz'
        if type(dynamicRange) is str:
            dynamicRange = float(dynamicRange)
        if type(minAcceptableAngResolution) is str:
            result = minAcceptableAngResolution.split()
            if len(result) == 1:
                value = result
                minUnits = 'arcsec'
            else:
                value, minUnits = result
            minAcceptableAngResolution = float(value)
        if type(maxAcceptableAngResolution) is str:
            result = maxAcceptableAngResolution.split()
            if len(result) == 1:
                value = result
                maxUnits = 'arcsec'
            else:
                value, maxUnits = result
            maxAcceptableAngResolution = float(value)
        newvalues = []
        if representativeFrequency is not None:
            newvalues += [['representativeFrequency = %f %s'%(representativeFrequency,freqUnits)]]
        if minAcceptableAngResolution is not None:
            newvalues += [['minAcceptableAngResolution = %f %s'%(minAcceptableAngResolution, minUnits)]]
        if maxAcceptableAngResolution is not None:
            newvalues += [['maxAcceptableAngResolution = %f %s'%(maxAcceptableAngResolution, maxUnits)]]
        if dynamicRange is not None:
            newvalues += [['dynamicRange = %f'%(dynamicRange)]]
        if representativeBandwidth is not None:
            newvalues += [['representativeBandwidth = %f %s'%(representativeBandwidth, bwUnits)]]
        if representativeFrequency is not None:
            newvalues += [['representativeSource = %s'%representativeSource]]
        if representativeSPW is not None:
            newvalues += [['representativeWindow = %s'%str(representativeSPW)]]
        if maxAllowedBeamAxialRatio is not None:
            newvalues += [['maxAllowedBeamAxialRatio = %f'%maxAllowedBeamAxialRatio]]
        newvalues = np.array(newvalues, dtype=str)
        if len(newvalues) != numScienceGoal:
            print("Updating numScienceGoal to %d" % (len(newvalues)))
            mytb.putcol('numScienceGoal',[len(newvalues)])
            casalog.post('Wrote new value of numScienceGoal to %s/ASDM_SBSUMMARY: %d'%(vis,len(newvalues)))
        print("Putting new values:\n", newvalues)
        mytb.putcol('scienceGoal',newvalues)
        casalog.post('Wrote new values to %s/ASDM_SBSUMMARY: %s'%(vis,str(newvalues)))
    else:
        print("Current values: shape=%s\n" % (str(np.shape(scienceGoal))), scienceGoal)
        if not representativeSpwPresent:
            print("Looking for spw that contains the representative frequency...")
            spw = representativeSpw(vis,verbose=False)
            if spw is not None:
                print("spw = ", spw)
    mytb.close()

def updateSBSummaryASDM(asdm, representativeFrequency=None, 
                        minAcceptableAngResolution=None, 
                        maxAcceptableAngResolution=None,
                        dynamicRange=None, representativeBandwidth=None,
                        representativeSource=None, representativeSPW=None,
                        maxAllowedBeamAxialRatio=None):
    """
    Updates the SBSummary.xml file of an ASDM with one or more new
    values.  If a value is not present in the existing table and also not 
    specified, then it will remain not present in the updated table.
    representativeFrequency: float value in typical units (GHz), 
            or string with units (space before units is optional)
    minAcceptableAngResolution: float value in typical units (arcsec), 
            or string with units (space before units is required)
    maxAcceptableAngResolution: float value in typical units (arcsec), 
            or string with units (space before units is required)
    dynamicRange: float value
    representativeBandwidth: float value in typical units (GHz), 
            or string with units (space before units is optional)
    representativeSource: string
    representativeSPW: string
    maxAllowedBeamAxialRatio: float
    -Todd Hunter
    """
    if not os.path.exists(asdm):
        print("Could not find asdm.")
        return
    t = asdm+'/SBSummary.xml'
    if not os.path.exists(t):
        print("Could not find SBSummary.xml for this ASDM.")
        return
    if (representativeFrequency is not None or 
        minAcceptableAngResolution is not None or 
        maxAcceptableAngResolution is not None or maxAllowedBeamAxialRatio is not None or
        dynamicRange is not None or representativeBandwidth is not None or
        representativeSource is not None or representativeSPW is not None):
        update = True
    else:
        update = False
    result = grep(t, 'numScienceGoal')
    if len(result[0]) == 0:
        print("Could not find numScienceGoal")
        return
    numScienceGoal = int(result[0].split('>')[1].split('<')[0])
    result = grep(t, 'scienceGoal')
    if len(result[0]) == 0:
        print("Could not find scienceGoal")
        return
    scienceGoalString = result[0].split('<scienceGoal>')[1].split('</scienceGoal')[0]
    scienceGoals = scienceGoalString.split('"')
    scienceGoal = []
    for s in scienceGoals:
        if (s.find('=') > 0):
            scienceGoal.append(s)
    
    # Read the existing values for those that were not specified
    for i,arg in enumerate(scienceGoal):
        # arg will look like: 
        #  'representativeFrequency = 219.55647641503566 GHz'
        loc = arg.find('representativeFrequency')
        if (loc >= 0 and representativeFrequency is None):
            tokens = arg[loc:].split()
            representativeFrequency = float(tokens[2])
            freqUnits = tokens[3]
        loc = arg.find('minAcceptableAngResolution')
        if (loc >= 0 and minAcceptableAngResolution is None):
            tokens = arg[loc:].split()
            minAcceptableAngResolution = float(tokens[2])
            minUnits = tokens[3]
        loc = arg.find('maxAcceptableAngResolution')
        if (loc >= 0 and maxAcceptableAngResolution is None):
            tokens = arg[loc:].split()
            maxAcceptableAngResolution = float(tokens[2])
            maxUnits = tokens[3]
        loc = arg.find('dynamicRange')
        if (loc >= 0 and dynamicRange is None):
            tokens = arg[loc:].split()
            dynamicRange = float(tokens[2])
        loc = arg.find('representativeBandwidth')
        if (loc >= 0 and representativeBandwidth is None):
            tokens = arg[loc:].split()
            representativeBandwidth = float(tokens[2])
            bwUnits = tokens[3]
        loc = arg.find('representativeSource')
        if (loc >= 0 and representativeSource is None):
            tokens = arg[loc:].split()
            representativeSource = str(tokens[2])
        loc = arg.find('representativeWindow')
        if (loc >= 0 and representativeSPW is None):
            tokens = arg[loc:].split()
            representativeSPW = str(tokens[2])
        loc = arg.find('maxAllowedBeamAxialRatio')
        if (loc >= 0 and maxAllowedBeamAxialRatio is None):
            tokens = arg[loc:].split()
            maxAllowedBeamAxialRatio = float(tokens[2])
    if update:
        # convert any command-line arguments from string to value and units
        shutil.copyfile(t, t+'.backup')
        if type(representativeFrequency) is str:
            representativeFrequency = parseFrequencyArgumentToGHz(representativeFrequency)
            freqUnits = 'GHz'
        if type(representativeBandwidth) is str:
            representativeBandwidth = parseFrequencyArgumentToGHz(representativeBandwidth) * 1000
            bwUnits = 'MHz'
        if type(dynamicRange) is str:
            dynamicRange = float(dynamicRange)
        if type(minAcceptableAngResolution) is str:
            value, minUnits = minAcceptableAngResolution.split()
            minAcceptableAngResolution = float(value)
        if type(maxAcceptableAngResolution) is str:
            value, maxUnits = maxAcceptableAngResolution.split()
            maxAcceptableAngResolution = float(value)
        newvalues = []
        if representativeFrequency is not None:
            newvalues += ['representativeFrequency = %f %s'%(representativeFrequency,freqUnits)]
        if minAcceptableAngResolution is not None:
            newvalues += ['minAcceptableAngResolution = %f %s'%(minAcceptableAngResolution, minUnits)]
        if maxAcceptableAngResolution is not None:
            newvalues += ['maxAcceptableAngResolution = %f %s'%(maxAcceptableAngResolution, maxUnits)]
        if dynamicRange is not None:
            newvalues += ['dynamicRange = %f'%(dynamicRange)]
        if representativeBandwidth is not None:
            newvalues += ['representativeBandwidth = %f %s'%(representativeBandwidth, bwUnits)]
        if representativeFrequency is not None:
            newvalues += ['representativeSource = %s'%representativeSource]
        if representativeSPW is not None:
            newvalues += ['representativeWindow = %s'%str(representativeSPW)]
        if maxAllowedBeamAxialRatio is not None:
            newvalues += ['maxAllowedBeamAxialRatio = %f'%(maxAllowedBeamAxialRatio)]
        print("newvalues = ", newvalues)
        newvalues = np.array(newvalues, dtype=str)
        if len(newvalues) != numScienceGoal:
            print("Updating numScienceGoal to %d" % (len(newvalues)))
        print("Putting new values:\n", newvalues)
        if casaAvailable:
            casalog.post('Wrote new values to %s/SBSummary.xml: %s'%(asdm,str(newvalues)))
        for line in fileinput.input(t, inplace=True):
            if line.find('numScienceGoal') > 0:
                line = "    <numScienceGoal>%d</numScienceGoal>\n" % (len(newvalues))
            elif line.find('scienceGoal') > 0:
                line = "    <scienceGoal>1 %d " % (len(newvalues))
                for sg in newvalues:
                    line += '"%s" ' % (sg)
                line += "</scienceGoal>\n"
            print(line[:-1])
    else:
        print("Current values: \n ", scienceGoal)

def cmd_exists(cmd):
    return subprocess.call("type " + cmd, shell=True, 
        stdout=subprocess.PIPE, stderr=subprocess.PIPE) == 0

def projectCodeFromDataset(dataset, wget='wget', verbose=False, overwrite=False):
    """
    Consults an online table on F. Stoehr's machine in order to locate the 
    project code, SB name, and MOUS for a dataset.  (See SCIREQ-412).
    dataset: ASDM or ms (underscores)
    wget: the full path to the wget executable
    verbose: if True, then print complete information 
    overwrite: if True, then get new table even if one exists in the PWD
    Returns: a tubple of: project code, SB name, MOUS, region, and status (all as strings)
    -Todd Hunter
    """
    dataset = uidToUnderscores(dataset)
    dataset = dataset.rstrip('/')
    dataset = os.path.basename(dataset).split('.')[0] # strip off .ms or .vis, etc.
    if (not cmd_exists(wget)): 
        wget = '/opt/local/bin/wget'
        if (not cmd_exists(wget)): 
            wget = '/usr/local/bin/wget'
            if (not cmd_exists(wget)): 
                return('Could not find wget executable on this machine.')
    try:
        output = getFelixTable(wget, overwrite, verbose)
        for line in output.split('\n'):
            loc = line.find(dataset)
            if (loc >= 0):
                if (verbose):
                    print("Project_code  Hidden_OUS  Science_Goal_OUS  Group_OUS  Member_OUS  EB  Obs_Date  SB_Status  SB_UID  SB_NAME  Region  Status  Type")
                    print(line)
                tokens = line.split()
                if (len(tokens) > 1):
                    return(tokens[0], tokens[-4], tokens[4], tokens[-3], tokens[-2])
                else:
                    return('Error in parsing the result.')
        return('This EB does not appear to be associated with a science project.')
    except:
        return('Could not reach the server.  Try again at a different time.')

def getFelixTable(wget, overwrite=False, verbose=True):
    """
    Opens a local copy (if present) or downloads the table of project/SB/EB
    information from Felix's machine.
    overwrite: if True, then get new table even if one exists in the PWD
    -Todd Hunter
    """
    url = 'http://www.eso.org/~fstoehr/project_ous_eb_hierarchy.txt'
    filename = 'fstoehr.txt'
    if not os.access('./', os.W_OK):
        filename = os.path.join('/tmp',filename)
    if (not os.path.exists(filename) or overwrite):
        output = subprocess.check_output('%s -q -O - %s' % (wget,url), 
                                         shell=True)
        f = open(filename,'w')
        f.write(output)
        f.close()
    else:
        if verbose:
            print("Using existing search results file: %s.  Set overwrite=True to refresh it." % (filename))
        f = open(filename,'r')
        output = f.read()
        f.close()
    return output

def datasetsForProjectCode(project,wget='wget',verbose=False,overwrite=False,
                           sevenMeter=True, twelveMeter=True, totalPower=True,
                           status='', sb='', exactStatus=True, dataset='',
                           showRejects=False):
    """
    Consults an online table on F. Stoehr's machine in order to locate the 
    executions for a project.
    project: project code
    wget: the full path to the wget executable
    verbose: if True, then print complete information 
    overwrite: if True, then download new copy of the table (fstoehr.txt)
    status: '' or 'Pass' or 'Fail' or 'SemiPass' or 'None' (case-insensitive)
    exactStatus: if True, then prepend a 'tab' character to status 
                 (e.g. to differentiate between Pass and SemiPass)
    sb: '' or name of sb to match
    dataset: if specified, then only print the details for this EB
    Returns: project code, SB name, MOUS, and region (all as strings)
    -Todd Hunter
    """
    if (not cmd_exists(wget)): 
        wget = '/opt/local/bin/wget'
        if (not cmd_exists(wget)): 
            wget = '/usr/local/bin/wget'
            if (not cmd_exists(wget)): 
                return('Could not find wget executable on this machine.')
    if (status != '' and exactStatus):
        status = '\t'+status
    try:
        output = getFelixTable(wget, overwrite)
        executions = []
        if (verbose):
            print("Project_code  Hidden_OUS  Science_Goal_OUS  Group_OUS  Member_OUS  Execution Block  Obs_Date  SB_Status  SB_UID  SB_NAME  ARC  Status  ArrayType")
            print("Project_code      Hidden_OUS            Science_Goal_OUS     Group_OUS            Member_OUS                          ExecutionBlock            Observation_Date                SB_Status               SB_UID                     SB_NAME                         ARC     Status          ArrayType")
        for line in output.split('\n'):
            loc = line.find(project)
            if (loc >= 0):
                lline = line.lower()
                if (((lline.find('7m"')>0 or lline.find('_ac"')>0) and sevenMeter) or
                    ((lline.find('_12m"')>0 or lline.find('_12m_')>0 or lline.find('12m ')>0 or lline.find('_te"')>0 or 
                      lline.find('_tc"')>0) and twelveMeter) or
                    (lline.find('_tp"')>0 and totalPower)):
                    if (status == '' or line.lower().find(status.lower())>=0):
                        if (sb == '' or line.find(sb)>=0):
                            tokens = line.split()
                            if (len(tokens) > 1): 
                                executions.append(tokens[5])
                                if (verbose):
                                    if dataset == '' or line.find(dataset) > 0:
                                        print(line)
                else:
                    if showRejects:
                        print("rejected line: ", line)
        if (len(executions) < 1):
            return('This project code does not appear to be associated with a science project.')
        else:
            return executions
    except:
        return('Could not reach the server.  Try again at a different time.')

def projectCodeFromSB(sb, wget='wget', allEBs=False, verbose=False, 
                      overwrite=False):
    """
    Consults an online table on F. Stoehr's machine in order to locate the 
    project code, MOUS, region and first EB for an SB.
    sb: SB uid (underscores)
    wget: the full path to the wget executable
    allEBs: if True, then return a list of all EBs
    verbose: if True, then print complete information 
    overwrite: if True, then download new copy of table even if it exists in PWD
    Returns: project code, region, MOUS and EB (all as strings)
    -Todd Hunter
    """
    dataset = uidToUnderscores(sb)
    dataset = dataset.split('.')[0] # strip off .ms or .vis, etc.
    if (not cmd_exists(wget)): 
        wget = '/opt/local/bin/wget'
        if (not cmd_exists(wget)): 
            wget = '/usr/local/bin/wget'
            if (not cmd_exists(wget)): 
                return('Could not find wget executable on this machine.')
    try:
        output = getFelixTable(wget, overwrite)
        results = []
        for line in output.split('\n'):
            loc = line.find(dataset)
            if (loc >= 0):
                if (verbose):
                    print("Project_code  Hidden_OUS  Science_Goal_OUS  Group_OUS  Member_OUS  EB  Obs_Date  SB_Status  SB_UID  SB_NAME  Region  Status  Type")
                    print(line)
                tokens = line.split()
                if (len(tokens) > 1):
                    result = tokens[0], tokens[-4], tokens[4], tokens[-3], tokens[-2]
                    if (allEBs):
                        results.append(tokens[5])
                    else:
                        return(result)
                else:
                    return('Error in parsing the result.')
        if (allEBs):
            return results
        return('This SB does not appear to be associated with a science project.')
    except:
        return('Could not reach the server.  Try again at a different time.')

def sbnameForMOUS(mous, wget='wget', verbose=False, overwrite=False):
    """
    Consults an online table on F. Stoehr's machine in order to locate the 
    SB name for an MOUS.
    verbose: if True, then print complete information 
    overwrite: if True, then download new copy of table even if it exists in PWD
    -Todd Hunter
    """
    dataset = uidToUnderscores(mous.lstrip('MOUS_'))
    dataset = dataset.split('.')[0] # strip off .ms or .vis, etc.
    if (not cmd_exists(wget)): 
        wget = '/opt/local/bin/wget'
        if (not cmd_exists(wget)): 
            wget = '/usr/local/bin/wget'
            if (not cmd_exists(wget)): 
                return('Could not find wget executable on this machine.')
    try:
        output = getFelixTable(wget, overwrite)
        results = []
        for line in output.split('\n'):
            loc = line.find(dataset)
            if (loc >= 0):
                if (verbose):
                    print("Project_code  Hidden_OUS  Science_Goal_OUS  Group_OUS  Member_OUS  EB  Obs_Date  SB_Status  SB_UID  SB_NAME  Region  Status  Type")
                    print(line)
                tokens = line.split()
                if (len(tokens) > 1):
                    result = tokens[10]
                else:
                    return('Error in parsing the result.')
                return result
        return('This SB does not appear to be associated with a science project.')
    except:
        return('Could not reach the server.  Try again at a different time.')
    
def projectCodeForMOUS(mous, wget='wget',verbose=False, overwrite=False):
    """
    Consults an online table on F. Stoehr's machine in order to locate the 
    project code for an MOUS.
    mous: mous uid (underscores or colon/slash format)
    wget: the full path to the wget executable
    verbose: if True, then print complete information 
    overwrite: if True, then download new copy of table even if it exists in PWD
    Returns: project code string
    -Todd Hunter
    """
    dataset = uidToUnderscores(mous.lstrip('MOUS_'))
    dataset = dataset.split('.')[0] # strip off .ms or .vis, etc.
    if (not cmd_exists(wget)): 
        wget = '/opt/local/bin/wget'
        if (not cmd_exists(wget)): 
            wget = '/usr/local/bin/wget'
            if (not cmd_exists(wget)): 
                return('Could not find wget executable on this machine.')
    try:
        output = getFelixTable(wget, overwrite)
        results = []
        for line in output.split('\n'):
            loc = line.find(dataset)
            if (loc >= 0):
                if (verbose):
                    print("Project_code  Hidden_OUS  Science_Goal_OUS  Group_OUS  Member_OUS  EB  Obs_Date  SB_Status  SB_UID  SB_NAME  Region  Status  Type")
                    print(line)
                tokens = line.split()
                if (len(tokens) > 1):
                    result = tokens[0]
                else:
                    return('Error in parsing the result.')
                return result
        return('This SB does not appear to be associated with a science project.')
    except:
        return('Could not reach the server.  Try again at a different time.')

def maxBaselineForRequestedResolutionFromASDM(asdm, resolution=None):
    """
    Uses the representative frequency and requested resolution from an ASDM
    and returns the maximum baseline length for the nominal ALMA configuration 
    that would achieve it.  To do this, it first computes the L80 baseline
    using the formula in the technical handbook, then scales this upward by
    the sine of the median elevation of science targets observed, then finds 
    the Cycle 4 configuration with the nearest (but larger) L80 value, finds
    its maximum baseline, then scales L80 by max_config)/L80_config
    resolution: if specified (in floating point arcsec), use this value instead
                of what is in the ASDM
    -Todd Hunter
    """
    if (not os.path.exists(asdm)):
        print("Could not find ASDM")
        return
    if resolution is None:
        resolution = np.min(requestedResolutionFromASDM(asdm))
        if resolution is None:
            return 0
        if (resolution <= 0):
            print("Requested resolution not present in the ASDM.")
            return 0
    L80 = 3600*np.degrees(1)*0.574*c_mks/(representativeFrequencyFromASDM(asdm)*1e9*resolution)
    result = getElevationStatsForIntentFromASDM(asdm)
    if (result == None):
        return
    minElev, medianElev, maxElev = result
    print("elevation: min = %.1f, median = %.1f, max = %.1f deg" % (minElev, medianElev, maxElev))
    print("Required L80 = %.0f m" % (L80))
    L80 /= np.sin(np.radians(minElev))
    print("Required L80 (accounting for minimum elevation) = %.0f m" % (L80))
    for config in range(1,10):
        L80config = getBaselineStats(config='alma.cycle4.%d.cfg'%config, percentile=80, verbose=False)[0]
        if (L80config > L80):
            print("Nominal required configuration: Cycle4-%d" % config)
            break
    maxBaselineConfig = getBaselineStats(config='alma.cycle4.%d.cfg'%config, verbose=False)[2]
    maxBaseline = L80 * (maxBaselineConfig/L80config)
    return(maxBaseline)

def requestedResolutionFromASDM(asdm):
    """
    Get the min/max acceptable resolution from the SBSummary.xml file of an ASDM
    Returns a tuple of values in arcsec
    -Todd Hunter
    """
    if (not os.path.exists(asdm)):
        print("Could not find ASDM")
        return
    if (not os.path.exists(asdm+'/SBSummary.xml')):
        print("Could not find SBSummary.xml, are you sure that this is an ASDM?")
        return
    f = open(asdm+'/SBSummary.xml')
    minAcceptableReoslution = 0
    maxAcceptableReoslution = 0
    for line in f.readlines():
        loc = line.find('minAcceptableAngResolution')
        if (loc >= 0):
            tokens = line[loc:].split()
            minAcceptableResolution = float(tokens[2])
            minUnits = tokens[3]
        loc = line.find('maxAcceptableAngResolution')
        if (loc >= 0):
            tokens = line[loc:].split()
            maxAcceptableResolution = float(tokens[2])
            maxUnits = tokens[3]
    f.close()
    return(minAcceptableResolution, maxAcceptableResolution)
    return(freq)
    
def requestedResolution(vis):
    """
    Get the min/max acceptable resolution from the ASDM_SBSUMMARY table of
    a measurement set, if it has been imported with the asis parameter set.   
    Returns a tuple of values in arcsec
    -Todd Hunter
    """
    if (not os.path.exists(vis)):
        print("Could not find measurement set.")
        return
    mytb = createCasaTool(tbtool)
    if (not os.path.exists(vis+'/ASDM_SBSUMMARY')):
        print("Could not find ASDM_SBSUMMARY table.  Did you not import it with asis='SBSummary'?")
        print("If not, you can run this on the asdm: au.requestedResolutionFromASDM(asdm)")
        return
    mytb.open(vis+'/ASDM_SBSUMMARY')
    scienceGoal = mytb.getcol('scienceGoal')
    mytb.close()
    minAcceptableResolution = 0
    maxAcceptableResolution = 0
    for args in scienceGoal:
        for arg in args:
            loc = arg.find('minAcceptableAngResolution')
            if (loc >= 0):
                tokens = arg[loc:].split()
                minAcceptableResolution = float(tokens[2])
                minUnits = tokens[3]
            loc = arg.find('maxAcceptableAngResolution')
            if (loc >= 0):
                tokens = arg[loc:].split()
                maxAcceptableResolution = float(tokens[2])
                maxUnits = tokens[3]
    return(minAcceptableResolution, maxAcceptableResolution)
    
def representativeFrequencyFromASDM(asdm, verbose=True):
    """
    Get the representative frequency from the SBSummary.xml file of an ASDM
    e.g. <scienceGoal>1 4 "representativeFrequency = 230.0348592858192 GHz" 
    Returns the value in GHz.
    verbose: if True, then also print the min/max acceptable angular resolutions
    -Todd Hunter
    """
    if (not os.path.exists(asdm)):
        print("Could not find ASDM")
        return
    if (not os.path.exists(asdm+'/SBSummary.xml')):
        print("Could not find SBSummary.xml, are you sure that this is an ASDM?")
        return
    f = open(asdm+'/SBSummary.xml')
    minAcceptableResolution = 0
    maxAcceptableResolution = 0
    representativeBandwidth = None
    representativeSource = None
    for line in f.readlines():
        loc = line.find('representativeFrequency')
        if (loc > 0):
            tokens = line[loc:].split()
            freq = parseFrequencyArgumentToGHz(tokens[2]+tokens[3])
        loc = line.find('minAcceptableAngResolution')
        if (loc >= 0):
            tokens = line[loc:].split()
            minAcceptableResolution = float(tokens[2])
            minUnits = tokens[3].strip('"')
        loc = line.find('maxAcceptableAngResolution')
        if (loc >= 0):
            tokens = line[loc:].split()
            maxAcceptableResolution = float(tokens[2])
            maxUnits = tokens[3].strip('"')
        loc = line.find('representativeBandwidth')
        if (loc >= 0):
            tokens = line[loc:].split()
            representativeBandwidth = float(tokens[2].strip('"'))
            bwUnits = tokens[3].strip('"')
            if bwUnits.find('Hz') < 0:
                bwUnits = ''
        loc = line.find('representativeSource')
        if (loc >= 0):
            tokens = line[loc:].split()
            representativeSource = str(tokens[2]).split('"')[0]
    f.close()
    if verbose:
        print("minAcceptableResolution = %f %s" % (minAcceptableResolution, minUnits))
        print("maxAcceptableResolution = %f %s" % (maxAcceptableResolution, maxUnits))
        print("representative frequency = %f GHz" % (freq))
        if representativeBandwidth is not None:
            print("representative bandwidth = %f %s" % (representativeBandwidth, bwUnits))
        if representativeSource is not None:
            print("representative source = %s" % (representativeSource))
    return(freq)

def sanitizeNames(names, newchar='_'):
    """
    Replaces various special characters with specified character.
    -Todd Hunter
    """
    if (type(names) == str):
        names = names.split(',')
    newnames = []
    for name in names:
        newname = name
        for ch in [')','(','/']:
            newname = newname.replace(ch,newchar)
        newnames.append(newname)
    return newnames

def transitions(vis, spws='', source='', intent='OBSERVE_TARGET', mymsmd=None):
    """
    Returns a dictionary of the line transition names for each science spw of the
    specified measurement set.
    spw: list or comma-delimited string;  if blank, then use all science spws
    -Todd Hunter
    """
    if (spws == ''):
        spws = getScienceSpws(vis, returnString=False, mymsmd=mymsmd)
        print("Got science spws: ", spws)
    elif (type(spws) == str):
        spws = parseSpw(vis, spws, mymsmd)
    elif (type(spws) != list and type(spws) != np.ndarray):
        spws = [int(spws)]
    lines = {}
    if source == '':
        targets = getScienceTargets(vis)
        source = targets[0]
        print("Picked target='%s' for all spws" % (source))
    for spw in spws:
        lines[spw] = transition(vis,spw,source,intent,mymsmd=mymsmd)
    return lines

def transitionsASDM(asdm):
    """
    Returns a dictionary, keyed by spw ID, of the spectral line transition 
    names listed in an ASDM.
    -Todd Hunter
    """
    scienceSpws, transitions = getScienceSpwsFromASDM(asdm, True)
    transition = {}
    for i,spw in enumerate(scienceSpws):
        transition[spw] = transitions[i]
    return transition

def isSingleDish(vis):
    """
    Checks whether the string StandardSingleDish apperas in the observingScript column
    of the ASDM_EXECBLOCK table of a measurement set.
    -Todd Hunter
    """
    if not os.path.exists(vis):
        print("Could not find vis.")
        return
    execblock = os.path.join(vis,'ASDM_EXECBLOCK')
    if not os.path.exists(execblock):
        print("Could not find ASDM_EXECBLOCK for this vis.")
        return
    mytb = createCasaTool(tbtool)
    mytb.open(execblock)
    mycell = mytb.getcell('observingScript',0)
    result = mycell.find('StandardSingleDish') >= 0
    mytb.close()
    return result

def isSingleDishASDM(asdm):
    """
    Searches for StandardSingleDish in the ExecBlock.xml
    -Todd Hunter
    """
    if not os.path.exists(asdm):
        print("Could not find ASDM.")
        return
    result = grep(asdm+'/ExecBlock.xml','StandardSingleDish')
    return len(result[0]) > 0

def isSingleContinuumASDM(asdm):
    """
    Checks whether the phrase Single_Continuum appears *anywhere* in the Source.xml file of an ASDM.
    -Todd Hunter
    """
    if not os.path.exists(asdm):
        print("Could not find ASDM.")
        return
    result = grep(asdm+'/Source.xml','Single_Continuum')
    return len(result[0]) > 0
    
def isSingleContinuum(vis, spw='', source='', intent='OBSERVE_TARGET', verbose=False, mymsmd=None):
    """
    Checks whether the first science spw (or specific spw) was defined as single continuum in the OT
    by looking at the transition name.
    -Todd Hunter
    """
    if not os.path.exists(vis):
        print("Could not find vis.")
        return
    needToClose = False
    if spw=='':
        if mymsmd is None:
            needToClose = True
            mymsmd = createCasaTool(msmdtool)
            mymsmd.open(vis)
        spw = getScienceSpws(vis, returnString=False, mymsmd=mymsmd)[0]
    info = transition(vis, spw, source, intent, verbose, mymsmd)
    if needToClose:
        mymsmd.close()
    if len(info) > 0:
        if info[0].find('Single_Continuum') >= 0:
            return True
    return False
    
def isSpectralScanASDM(asdm):
    """
    Checks whether the phrase Spectral_Scan appears in the Source.xml file of an ASDM.
    -Todd Hunter
    """
    if not os.path.exists(asdm):
        print("Could not find ASDM.")
        return
    result = grep(asdm+'/Source.xml','Spectral_Scan')
    return len(result[0]) > 0

def isMosaic(vis='', mymsmd=None, returnScienceNames=False):
    """
    Uses msmd to check whether any science target is observed as a mosaic.  It first finds all field
    names with OBSERVE_TARGET intent, then for each name, retrieves the field IDs for that name.
    If more than 1 field is found, it counts the number of those fields that have OBSERVE_TARGET intent
    and if it is greater than 1, it returns True; otherwise it returns False.
    returnScienceNames: if True, then also return the list of mosaic field names
    -Todd Hunter
    """
    needToClose = False
    if mymsmd is None:
        if not os.path.exists(vis):
            print("Could not find measurement set.")
            return
        needToClose = True
        mymsmd = msmdtool()
        mymsmd.open(vis)
    field_names = np.unique(mymsmd.fieldsforintent('OBSERVE_TARGET#ON_SOURCE', asnames=True))
    mosaic = False
    scienceNames = []
    for name in field_names:
        fields = mymsmd.fieldsforname(name)
        science = 0
        if len(fields) > 1:
            for field in fields:
                if 'OBSERVE_TARGET#ON_SOURCE' in mymsmd.intentsforfield(field):
                    science += 1
            if science > 1:
                mosaic = True
                scienceNames.append(name)
                break
    if needToClose: mymsmd.close()
    if returnScienceNames:
        return mosaic, scienceNames
    return mosaic
    
def isSpectralScan(vis, spw='', source='', intent='OBSERVE_TARGET', verbose=False, mymsmd=None):
    """
    Checks whether a project was defined as Spectral Scan in the OT
    by looking at the transition name in the first science spw.
    -Todd Hunter
    """
    if not os.path.exists(vis):
        print("Could not find vis.")
        return
    needToClose = False
    if (spw == ''):
        if mymsmd is None:
            needToClose = True
            mymsmd = createCasaTool(msmdtool)
            mymsmd.open(vis)
        spws = getScienceSpws(vis, returnString=False, mymsmd=mymsmd)
        if (len(spws) < 1):
            print("No science spws in this measurement set.")
            return
        spw = spws[0]
    info = transition(vis, spw, source, intent, verbose, mymsmd)
    if needToClose:
        mymsmd.close()
    if len(info) > 0:
        if info[0].find('Spec_') >= 0 and info[0].find('_Scan') >= 0:
            return True
    return False
    
def transition(vis, spw='', source='', intent='OBSERVE_TARGET', verbose=True, mymsmd=None):
    """
    Returns the list of transitions for specified spw (and source).
    source: can be integer ID or string name
    intent: if source is blank then use first one with matching intent and spw
    """
    if (not os.path.exists(vis)):
        print("Could not find measurement set")
        return
    needToClose = False
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    if (spw >= mymsmd.nspw()):
        print("spw not in the dataset")
        if needToClose:
            mymsmd.close()
        return
    mytb = createCasaTool(tbtool)
    mytb.open(vis+'/SOURCE')
    spws = mytb.getcol('SPECTRAL_WINDOW_ID')
    sourceIDs = mytb.getcol('SOURCE_ID')
    names = mytb.getcol('NAME')
    spw = int(spw)
    if (type(source) == str):
        if (source.isdigit()):
            source = int(source)
        elif (source == ''):
            # pick source
            fields1 = mymsmd.fieldsforintent(intent+'*')
            fields2 = mymsmd.fieldsforspw(spw)
            fields = np.intersect1d(fields1,fields2)
            source = mymsmd.namesforfields(fields[0])[0]
            if verbose:
                print("For spw %d, picked source: " % (spw), source)
    if (type(source) == str or type(source) == np.string_):
        sourcerows = np.where(names==source)[0]
        if (len(sourcerows) == 0):
            # look for characters ()/ and replace with underscore
            names = np.array(sanitizeNames(names))
            sourcerows = np.where(source==names)[0]
    else:
        sourcerows = np.where(sourceIDs==source)[0]
        
    spwrows = np.where(spws==spw)[0]
    row = np.intersect1d(spwrows, sourcerows)
    if (len(row) > 0):
        if (mytb.iscelldefined('TRANSITION',row[0])):
            transitions = mytb.getcell('TRANSITION',row[0])
        else:
            transitions = []
    else:
        transitions = []
    if (len(transitions) == 0):
        print("No value found for this source/spw (row=%s)." % row)
    mytb.close()
    if needToClose:
        mymsmd.close()
    return(transitions)

def getFrequencies(inputMs,spwId) :
    """
    Returns the list of channel frequencies in the specified spw in the
    specified ms. Obsoleted by msmd.chanfreqs.
    """
    mytb = tbtool()
    mytb.open("%s/SPECTRAL_WINDOW" % inputMs)
    chanFreq = mytb.getcol("CHAN_FREQ",startrow=spwId,nrow=1)
    mytb.close()
    return chanFreq
    
def getChanAverSpwIDBaseBand0(inputMs) :
    """
    Called by getFlux, getAllanVariance and classes: Tsys, TsysExplorer & Visibility
    """
    mytb = tbtool()
    mytb.open("%s/SPECTRAL_WINDOW" % inputMs)
    bbc_no = mytb.getcol('BBC_NO')
    ind1 = np.where(bbc_no == 1)[0]
    num_chan = mytb.getcol('NUM_CHAN')
    mytb.close()
    ind2 = np.where(num_chan == 1)[0]
    return np.intersect1d(ind1,ind2)[0]

def getDataDescriptionId(inputMs, spwId) :
    """
    Called by class Visibility.  Uses tb tool.
    """
    mytb = tbtool()
    mytb.open("%s/DATA_DESCRIPTION" % inputMs)
    spectralWindows = mytb.getcol("SPECTRAL_WINDOW_ID")
    mytb.close()
    ids = np.where(spectralWindows == spwId)[0]
    return int(ids)

# No longer called by anything.  Should be deleted.
#def getSpectralWindowId(inputMs,dataDesId) :
#    tb.open("%s/DATA_DESCRIPTION" % inputMs)
#    spectralWindows = tb.getcol("SPECTRAL_WINDOW_ID")
#    tb.close()
#    return spectralWindows[dataDesId]
    
def getFlux(inputMs,spwID=None,jyPerK=33,badAntennas=[],useCorrected=False) :
    if spwID == None :
        spwID = getChanAverSpwIDBaseBand0(inputMs)
    sourceIds,sourceNames = getSourceFieldMapping(inputMs)
    antennas = getAntennaNames(inputMs)
    tsys = Tsys(inputMs,spwID=spwID)
    sourceFlux = {}
    averageFlux = {}
    for i in range(len(badAntennas)) :
        badAntennas[i] = getAntennaIndex(inputMs,badAntennas[i])
    for i in range(len(sourceIds)) :
        fieldId = sourceIds[sourceNames[i]]
        sourceName = sourceNames[i]
        sourceFlux[sourceName] = {}
        averageFlux[sourceName] = {}
        sourceScans = getSourceScans(inputMs,fieldId)
        for k in range(len(sourceScans)//2) :
            tsysScan = sourceScans[2*k]
            sourceScan = sourceScans[2*k+1]
            tsys_ = {}
            sourceFlux[sourceName][sourceScan] = {}
            for j in range(len(antennas)) :
#                print antennas[j]
                tsys_[antennas[j]] = tsys.sysInfo[antennas[j]][tsysScan]['Tsys']['value']
            averageFlux[sourceName][sourceScan] = {'Flux' : np.zeros(tsys_[list(tsys.sysInfo.keys())[0]].shape)}
            for j in range(len(antennas)) :
                for m in range(len(antennas)) :
                    if j < m :
                        if j in badAntennas or m in badAntennas :
                            continue
                        else :
                            sourceFlux_ = Visibility(inputMs,antenna1=j,antenna2=m,spwID=spwID,field=fieldId,scan=sourceScan,correctedData=useCorrected)
                            flux_  = ((tsys_[antennas[j]]*tsys_[antennas[m]])**0.5)*sourceFlux_.amp.mean(-1)*jyPerK
                            dflux_ = ((tsys_[antennas[j]]*tsys_[antennas[m]])**0.5)*sourceFlux_.amp.std(-1)*jyPerK/sourceFlux_.amp.shape[-1]**0.5
                            baseline = ('%i-%i' % (j,m))
                            sourceFlux[sourceName][sourceScan][baseline] = {'Flux' : flux_, 'Error' : dflux_}
            for j in list(sourceFlux[sourceName][sourceScan].keys()) :
                averageFlux[sourceName][sourceScan]['Flux'] = averageFlux[sourceName][sourceScan]['Flux']+sourceFlux[sourceName][sourceScan][j]['Flux']/len(list(sourceFlux[sourceName][sourceScan].keys()))
    return sourceFlux,tsys,averageFlux


class Tsys(Weather):
    def __init__(self,inputMs,spwID=None,tau=0.05,etaF=0.99,doRefSub=False):
        if spwID == None :
            spwID = getChanAverSpwIDBaseBand0(inputMs)
        Weather.__init__(self,inputMs)
        self.inputMs = inputMs
        self.atm = AtmStates(inputMs)
        self.loads = list(self.atm.antennaInfo[self.atm.antennaNames[0]].keys())
        self.spwID = spwID
        self.tau = tau
        self.etaF = etaF
        interTab = InterpolateTableTime(self,self.spwID)
        mytb = tbtool()
        mytb.open("%s/SPECTRAL_WINDOW" % inputMs)
        self.freq = mytb.getcol("REF_FREQUENCY")[self.spwID]
        mytb.close()
        self.specFreq = getFrequencies(inputMs,spwID)
        self.atmRes = {}
        for i in self.atm.antennaNames :
#            print i
            _visVal = Visibility(inputMs,spwID=spwID,antenna1=i,antenna2=i)
            scanNums = np.unique(_visVal.subtable.getcol('SCAN_NUMBER'))
            self.atmRes[i] = {}
            noScan = []
            for m in scanNums :
                self.atmRes[i][m] = {}
                for j in self.loads :
                    stateVal = self.atm.antennaInfo[i][j]['state']
                    self.atmRes[i][m][j] = {}
                    for k in stateVal :
                        try:
                            visVal = Visibility(inputMs,spwID=spwID,antenna1=i,antenna2=i,scan=m,state=k)
                            self.atmRes[i][m][j]['power'] = np.mean(visVal.amp,len(visVal.amp.shape)-1)
                            self.atmRes[i][m][j]['error'] = np.std(visVal.amp,len(visVal.amp.shape)-1)
                            self.atmRes[i][m][j]['time']  = np.mean(visVal.subtable.getcol("TIME"))
                            if j in ["HOT","AMB"] :
                                states = self.atm.antennaInfo[i][j]['state']
                                loadTemps = self.atm.antennaInfo[i][j]['loadTemp']
                                checker = states.index(k)
                                self.atmRes[i][m][j]['loadTemp'] = loadTemps[checker]
                        except:
                            continue
                if list(self.atmRes[i][m]["AMB"].keys()) == [] : noScan.append(m)
            noScan = np.unique(np.array(noScan))
            for m in noScan :
                self.atmRes[i].pop(m)
            counter = 0
            for m in list(self.atmRes[i].keys()) :
                try:
                    if list(self.atmRes[i][m]['REF'].keys()) == [] : self.atmRes[i][m]['REF'] = self.atmRes[i][scanNums[counter-1]]['REF']
                    if list(self.atmRes[i][m]['HOT'].keys()) == [] : self.atmRes[i][m]['HOT'] = self.atmRes[i][scanNums[counter-1]]['HOT']
                    counter+=1
                except:
                    continue
        self.sysInfo = {}
        for i in list(self.atmRes.keys()) :
            self.sysInfo[i] = {}
            for m in list(self.atmRes[i].keys()) :
                print(m,i)
                self.sysInfo[i][m] = {}
                pHot = self.atmRes[i][m]['HOT']['power']
                eHot = self.atmRes[i][m]['HOT']['error']
                timetHot = self.atmRes[i][m]['HOT']['time']
                tHot = jVal(self.freq,self.atmRes[i][m]['HOT']['loadTemp'])
                pAmb = self.atmRes[i][m]['AMB']['power']
                eAmb = self.atmRes[i][m]['AMB']['error']
                timeAmb = self.atmRes[i][m]['AMB']['time']
                tAmb = jVal(self.freq,self.atmRes[i][m]['AMB']['loadTemp'])
#                print self.atmRes[i][m]['AMB']
                try:
                    pRef = self.atmRes[i][m]['REF']['power']
                    eRef = self.atmRes[i][m]['REF']['error']
                    timeRef = self.atmRes[i][m]['REF']['time']
                except:
                    pRef = np.zeros(self.atmRes[i][m]['AMB']['power'].shape)
                    eRef = pRef
                    timeRf = self.atmRes[i][m]['AMB']['time']
                pSky = self.atmRes[i][m]['SKY']['power']
                eSky = self.atmRes[i][m]['SKY']['error']
                timeSky = self.atmRes[i][m]['SKY']['time']
                tCmb = jVal(self.freq,Tcmb)
                Gain,dGain,Trx,dTrx,Tsky,dTsky,y,dy = calcTrxGain(pHot,pAmb,pSky,tHot,tAmb,pRef,eHot,eAmb,eRef,doRefSub=doRefSub)
                meanTime = (timeAmb+timeSky)/2.0
                interTab.interpolateData(np.array(meanTime),quiet=True)
                tOut = interTab.newData['TEMPERATURE'].mean()
                tAtm = interTab.newData['ATM_TEMP'].mean()
                alph = solveAlpha(tHot,tAmb,tAtm,tOut,etaF)
                tCal,tSys = solveTsys(tAtm,pHot,pAmb,pSky,tCmb,alph,pRef,doRefSub=doRefSub)
                self.sysInfo[i][m]['gain'] = {'value' : Gain, 'error' : dGain}
                self.sysInfo[i][m]['Trx']  = {'value' : Trx, 'error' : dTrx}
                self.sysInfo[i][m]['Tsky'] = {'value' : Tsky, 'error' : dTsky}
                self.sysInfo[i][m]['y']    = {'value' : y, 'error' : dy}
                self.sysInfo[i][m]['Tcal'] = {'value' : tCal, 'error' : 0}
                self.sysInfo[i][m]['Tsys'] = {'value' : tSys, 'error' : 0}
                self.sysInfo[i][m]['Time'] = {'value' : meanTime, 'error' : 0}
#                self.sysInfo[i][m]['Freq'] = {'value' : , 'error' : 0}

def solveTsys(tAtm,pHot,pAmb,pSky,tCmb,alpha,pRef,doRefSub=False) :
    if not doRefSub : pRef = pRef-pRef
    tCal  = tAtm-tCmb
    pLoad = alpha*pHot+(1-alpha)*pAmb
    tSys  = tCal*(pSky-pRef)/(pLoad-pSky)
    return tCal,tSys

def calcTrxGain(pHot,pAmb,pSky,tHot,tAmb,pRef=0,eHot=0,eAmb=0,eSky=0,eRef=0,etHot=0,etAmb=0,Gain=None,dGain=None,Trx=None,dTrx=None,doRefSub=False) :
    if not doRefSub : pRef = pRef-pRef
    if Gain == None  : Gain  = (pHot-pAmb)/(tHot-tAmb)
    if dGain == None : dGain = (((eHot**2.0+eAmb**2.0)/(tHot-tAmb)**2.0)+((pHot-pAmb)**2.0/(tHot-tAmb)**4.0)*(etHot**2.0+etAmb**2.0))**0.5
    if Trx == None   : Trx   = ((pHot-pRef)/Gain)-tHot
    if dTrx == None  : dTrx  = ((eHot/Gain)**2.0+(eRef/Gain)**2.0+((pHot-pRef)*dGain/Gain**2.0)**2.0 + etHot**2.0)**0.5
    Tsky  = tAmb-(pAmb-pSky)/Gain
    dTsky = ((eAmb/Gain)**2.0+(eSky/Gain)**2.0+((pAmb-pSky)*dGain/Gain**2.0)**2.0+etAmb**2.0)**0.5
    y     = (pHot-pRef)/(pAmb-pRef)
    dy    = ((eHot/(pAmb-pRef))**2.0+((pHot-pRef)*eAmb/(pAmb-pRef)**2.0)**2.0+(eRef/(pAmb-pRef)+(pAmb*eRef)/(pAmb-pRef)**2.0)**2.0)**0.5
    return Gain,dGain,Trx,dTrx,Tsky,dTsky,y,dy

def solveAlpha(tHot,tAmb,tAtm,tOut,etaF) :
    # Equation 16 of Lucas & Corder "Dual Load Amplitude Calibration in ALMA"
    return (etaF*tAtm-tAmb+(1-etaF)*tOut)/(tHot-tAmb)

def jVal(freq,temp) :
    import math as m
    x = h*freq/k
    return x*(m.exp(x/temp)-1)**(-1)

def djVal(freq,temp,detemp) :
    x = h*freq*1e9/k
    return abs(x*(m.exp(x/temp)-1)**(-2)*(x*dtemp/temp**2.0)*m.exp(x/temp))

def overlayTsys(vis1,vis2,antenna=0,spw=1,scan=1,pol=0,plotrange=[0,0,0,0],
                overlayTelcal=True, titleFontsize=10):
    """
    Calls the Atmcal class to overlay a Tsys result from one scan of two 
    different DelayCal measurement sets.  It was written to compare TDM 
    and FDM Tsys spectra.
    -Todd Hunter
    """
    a1 = Atmcal(vis1)
    tsys1, freqHz1, trec, tsky, tcal = a1.computeTsys(antenna=antenna, spw=spw, scan=scan, pol=pol)
    a2 = Atmcal(vis2)
    tsys2, freqHz2, trec, tsky, tcal = a2.computeTsys(antenna=antenna, spw=spw, scan=scan, pol=pol)
    pb.clf()
    adesc = pb.subplot(111)
    freqHz1 *= 1e-9
    freqHz2 *= 1e-9
    if (overlayTelcal):
        tsys1_telcal = a1.getTelcalTsys(antenna,spw,scan,pol) 
        tsys2_telcal = a2.getTelcalTsys(antenna,spw,scan,pol) 
        pb.plot(freqHz1, tsys1_telcal, 'k-', freqHz2, tsys2_telcal, 'k-', lw=3)
#        pb.hold(True) # not available in CASA6, but also not needed 
    pb.plot(freqHz1, tsys1, 'r-', freqHz2, tsys2, 'r-')
    if (len(tsys1) <= 256):
        tdmMedian = np.median(tsys1)
        fdmMedian = np.median(tsys2)
    else:
        tdmMedian = np.median(tsys2)
        fdmMedian = np.median(tsys1)

    pb.xlabel('Frequency (GHz)')
    pb.ylabel('Tsys (K)')
    if (plotrange != [0,0,0,0]):
        if (plotrange[0] != 0 or plotrange[1] != 0):
            pb.xlim(plotrange[:2])
        if (plotrange[2] != 0 or plotrange[3] != 0):
            pb.ylim(plotrange[2:])
    adesc.xaxis.grid(True,which='major')
    adesc.yaxis.grid(True,which='major')
    if (type(antenna) != str):
        antennaName = a1.antennaNames[antenna]
    else:
        antennaName = antenna
    pb.title('%s / %s  %s  scan=%d  spw=%d  pol=%d' % (vis1,vis2,antennaName, scan, spw, pol), fontsize=titleFontsize)
    pb.text(0.1,0.95,'Black = TelCal,  Red = au.Atmcal().computeTsys', transform=adesc.transAxes)
    xoffset = pb.xlim()[0] + (pb.xlim()[1]-pb.xlim()[0])*0.03
    pb.text(xoffset, tdmMedian, "TDM")
    pb.text(xoffset, fdmMedian, "FDM")
    pb.draw()
    png = '%s_%s.%s.scan%d.spw%d.pol%d.tsys.png' % (vis1, vis2, antennaName,scan,spw,pol)
    pb.savefig(png)

def overlayTrx(vis1,vis2,antenna=0,spw=1,scan=1,pol=0,plotrange=[0,0,0,0],
                overlayTelcal=True):
    """
    Calls the Atmcal class to overlay a Trx result from one scan of two 
    different DelayCal measurement sets.  It was written to compare TDM 
    and FDM Trx spectra.
    -Todd Hunter
    """
    a1 = Atmcal(vis1)
    trx1, gain, tsky, freqHz1 = a1.computeTrec2(antenna=antenna, spw=spw, scan=scan, pol=pol)
    a2 = Atmcal(vis2)
    trx2, gain, tsky, freqHz2 = a2.computeTrec2(antenna=antenna, spw=spw, scan=scan, pol=pol)
    pb.clf()
    adesc = pb.subplot(111)
    freqHz1 *= 1e-9
    freqHz2 *= 1e-9
    if (overlayTelcal):
        trx1_telcal = a1.getTelcalTrx(antenna,spw,scan,pol) 
        trx2_telcal = a2.getTelcalTrx(antenna,spw,scan,pol) 
        pb.plot(freqHz1, trx1_telcal, 'k-', freqHz2, trx2_telcal, 'k-', lw=3)
#        pb.hold(True) # not available in CASA6, but also not needed 
    pb.plot(freqHz1, trx1, 'r-', freqHz2, trx2, 'r-')
    if (len(trx1) <= 256):
        tdmMedian = np.median(trx1)
        fdmMedian = np.median(trx2)
    else:
        tdmMedian = np.median(trx2)
        fdmMedian = np.median(trx1)
    
    pb.xlabel('Frequency (GHz)')
    pb.ylabel('Trx (K)')
    if (plotrange != [0,0,0,0]):
        if (plotrange[0] != 0 or plotrange[1] != 0):
            pb.xlim(plotrange[:2])
        if (plotrange[2] != 0 or plotrange[3] != 0):
            pb.ylim(plotrange[2:])
    adesc.xaxis.grid(True,which='major')
    adesc.yaxis.grid(True,which='major')
    if (type(antenna) != str):
        antennaName = a1.antennaNames[antenna]
    else:
        antennaName = antenna
    pb.title('%s / %s  %s  scan=%d  spw=%d  pol=%d' % (vis1,vis2,antennaName, scan, spw, pol), fontsize=10)
    pb.text(0.1,0.95,'Black = TelCal,  Red = au.Atmcal().computeTrx', transform=adesc.transAxes)
    xoffset = pb.xlim()[0] + (pb.xlim()[1]-pb.xlim()[0])*0.03
    pb.text(xoffset, tdmMedian, "TDM")
    pb.text(xoffset, fdmMedian, "FDM")
    pb.draw()
    png = '%s_%s.%s.scan%d.spw%d.pol%d.trx.png' % (vis1, vis2, antennaName,scan,spw,pol)
    pb.savefig(png)

def repairSidebandRatio(asdm, scan, showplot=False):
    """
    Prepares an ASDM for running offline casapy-telcal's tc_sidebandratio() command
    to regenerate the CalAtmosphere.xml table.  Useful for computing sideband ratios
    if the values in the ASDM are simply the default values, yet sideband_ratio data
    exists in the ASDM. Renames the CalAtmosphere.xml and .bin tables (to *.old), sets the
    number CalAtmosphere rows to zero in the ASDM.xml table, then
    runs tc_sidebandratio on one scan at a time (if the command is available).
    scan: list of scans, e.g. '1,2,3' or [1,2,3]
    -Todd Hunter
    """
    if (type(scan) == str):
        scan = scan.split(',')
    elif (type(scan) == int):
        scan = [scan]
    if (os.path.exists(asdm) == False):
        print("Could not find ASDM")
        return
    f = open(asdm+'/ASDM.xml')
    fc = f.read()
    f.close()
    asdmBlockList = re.findall('<Table>.*?</Table>', fc, re.DOTALL|re.MULTILINE)
    if len(asdmBlockList) == 0:
        print('Found 0 blocks.')
        return
    for i in range(len(asdmBlockList)):
        if re.search('<Name> *CalAtmosphere *</Name>', asdmBlockList[i]) is not None or re.search('<Name> *CalAtmosphere *</Name>', asdmBlockList[i]) is not None:
            asdmBlockList1 = re.sub('<NumberRows> *[0-9]+ *</NumberRows>', '<NumberRows> 0 </NumberRows>', asdmBlockList[i])
            fc = re.sub(asdmBlockList[i], asdmBlockList1, fc)

    f = open(asdm+'/ASDM.xml', 'w')
    f.write(fc)
    f.close()
    if (os.path.exists(asdm+'/CalAtmosphere.xml')):
        os.system('mv '+asdm+'/CalAtmosphere.xml '+asdm+'/CalAtmosphere.xml.old')
    if (os.path.exists(asdm+'/CalAtmosphere.bin')):
        os.system('mv '+asdm+'/CalAtmosphere.bin '+asdm+'/CalAtmosphere.bin.old')
    try:
        from tc_sidebandratio_cli import tc_sidebandratio_cli as tc_sidebandratio
        for s in scan:
            print("Running tc_sidebandratio('%s', dataorigin='avercross', calresult='%s', showplot=%s, scans='%s')" % (asdm,asdm,showplot,str(s)))
            tc_sidebandratio(asdm, dataorigin='avercross', calresult=asdm, showplot=showplot, scans=str(s))
    except:
        print("Now you can start casapy-telcal and run the following on each sideband_ratio scan:")
        print("tc_sidebandratio('%s', dataorigin='avercross', calresult='%s', showplot=False, scans='')" % (asdm,asdm))
        print("(Note: At present, the scans parameter of tc_sidebandratio only accepts one scan number.)")

def getAtmcalStateIDsFromASDM(asdm):
    """
    Reads the State.xml file of an ASDM and returns a dictionary of the form:
    {'AMBIENT_LOAD': 'State_1', 'HOT_LOAD': 'State_2', 'NONE': 'State_0'}
    -Todd Hunter
    """
    if (os.path.exists(asdm) == False):
        print("Could not find ASDM.")
        return
    f = open(asdm + '/State.xml', 'r')
    lines = f.readlines()
    f.close()
    mydict = {}
    for line in lines:
        if (line.find('<stateId>') >= 0):
            stateId = line.split('<stateId>')[1].split('</stateId>')[0]
        if (line.find('<calDeviceName>') >= 0):
            mydict[line.split('<calDeviceName>')[1].split('</calDeviceName>')[0]] = stateId
    return(mydict)
            
def repairAtmcalStateIDs(asdm, dryrun=False):
    """
    Replaces <stateId> entries in the Main.xml file of an ASDM for all the
    AtmCal scans, so that offline casapy-telcal can then be run in order
    to regenerate the SysCal.xml file.  It forces all appearances of
    "State_0" to be "State_X" such that X = the stateId from the State.xml table
    corresponding to load for the subscan number (1=NONE, 2=AMBIENT, 3=HOT for
    a 3-subscan AtmCal, and 1=NONE, 2=NONE, 3=AMBIENT, 4=HOT for a 4-subscan Atmcal.
    asdm: the name of the ASDM
    dryrun: if True, then simply print the scans that would be changed, do not change them
    -Todd Hunter
    """
    if (os.path.exists(asdm) == False):
        print("Could not find ASDM.")
        return
    calscans, nsubscans = getScanNumbersFromASDM(asdm,'CALIBRATE_ATMOSPHERE')
    AtmcalStateIds = getAtmcalStateIDsFromASDM(asdm)
    print("AtmcalStateIds = ", AtmcalStateIds)
    f = open(asdm + '/Main.xml', 'r')
    if (not dryrun):
        o = open(asdm + '/Main.xml.new', 'w')
    lines = f.readlines()
    f.close()
    scan = 0
    subscan = 0
    changed = 0
    scansToFix = []
    print("Read %d lines from Main.xml" % (len(lines)))
    for line in lines:
        originalLine = line[:]
        if (line.find('<scanNumber>') >= 0):
            scan = int(line.split('>')[1].split('<')[0])
        elif (line.find('<subscanNumber>') >= 0):
            subscan = int(line.split('>')[1].split('<')[0])
        elif (line.find('<stateId>') >= 0):
            if (scan in calscans):
                if (nsubscans[calscans.index(scan)] == 3):
                    if (subscan == 1):
                        for state in range(3):
                            line = line.replace('State_%d'%state,AtmcalStateIds['NONE'])
                    elif (subscan == 2):  
                        for state in range(3):
                            line = line.replace('State_%d'%state,AtmcalStateIds['AMBIENT_LOAD'])
                    elif (subscan == 3):
                        for state in range(3):
                            line = line.replace('State_%d'%state,AtmcalStateIds['HOT_LOAD'])
                else:
                    if (subscan == 1 or subscan == 2):  
                        for state in range(4):
                            line = line.replace('State_%d'%state,AtmcalStateIds['NONE'])
                    elif (subscan == 3):  
                        for state in range(4):
                            line = line.replace('State_%d'%state,AtmcalStateIds['AMBIENT_LOAD'])
                    elif (subscan == 4):
                        for state in range(4):
                            line = line.replace('State_%d'%state,AtmcalStateIds['HOT_LOAD'])
        if (line != originalLine):
            changed += 1
            scansToFix.append(scan)
        if not dryrun:
            o.write(line)
        
    if dryrun:
        if (len(scansToFix) > 0):
            print("Scans that need fixing: ", np.unique(scansToFix))
        else:
            print("No scans need fixing.")
    else:
        print("Changed %d lines" % (changed))
        o.close()
        os.rename(asdm + '/Main.xml', asdm + '/Main.xml.old') 
        os.rename(asdm + '/Main.xml.new', asdm + '/Main.xml') 
        
def repairSysCal(asdm, sidebandgainoption='observed', showplot=False, 
                 sidebandgain=-1, water='', tsysmode='ALPHA', verbose=True,
                 scan=''):
    """
    Prepares an ASDM for running offline casapy-telcal's tc_atmosphere() command
    to regenerate the SysCal.xml table.  It first moves the SysCal.xml table to
    Syscal.xml.old, then sets the number of SysCal rows to zero in the ASDM.xml
    table.  It then tries to run that task if you are running casapy-telcal.
    This task is useful for computing Tsys solutions that are missing in the
    ASDM.  Based on Neil Phillips' posting to PRTSIR-305.  Before running
    tc_atmosphere, it preserves the other .xml files and copies them back to
    avoid issues with flagcmd.
    scan: use only the specified scans, otherwise, use all ATMOSPHERE scans
    sidebandgainoption: 'observed' or 'fixed' or 'userdefined'
    sidebandgain: use this for 'userdefined'
    water: PWV (in meters) pass this to tc_atmosphere
    tsysmode: either 'WVR' or 'ALPHA' (default)
    -Todd Hunter
    """
    if (os.path.exists(asdm) == False):
        print("Could not find ASDM")
        return
    f = open(asdm+'/ASDM.xml')
    fc = f.read()
    f.close()
    asdmBlockList = re.findall('<Table>.*?</Table>', fc, re.DOTALL|re.MULTILINE)
    if (scan != ''):
        scan = scan.replace(',',' ')
    else:
        # find the scans automatically
        d = readscans(asdm)
        scan = ''
        for s in list(d[0].keys()):
            if (d[0][s]['intent'].find('ATMOSPHERE') > 0):
                print("adding scan %d to list" % (s))
                if (len(scan) > 0): scan += ' '
                scan += str(s)
        if (len(scan) == 0):
            print("No ATMOSPHERE scans found.")
            return
    if len(asdmBlockList) == 0:
        print('Found 0 blocks.')
        return
    for i in range(len(asdmBlockList)):
        if re.search('<Name> *SysCal *</Name>', asdmBlockList[i]) is not None or re.search('<Name> *CalAtmosphere *</Name>', asdmBlockList[i]) is not None:
            asdmBlockList1 = re.sub('<NumberRows> *[0-9]+ *</NumberRows>', '<NumberRows> 0 </NumberRows>', asdmBlockList[i])
            fc = re.sub(asdmBlockList[i], asdmBlockList1, fc)

    f = open(asdm+'/ASDM.xml', 'w')
    f.write(fc)
    f.close()
    newdir = asdm + '.originalxml'
    print("Creating directory ", newdir)
    if (os.path.exists(newdir) == False):
        os.mkdir(newdir)
    print("Copying all .xml files to %s..." % (newdir))
    os.system('cp %s/*.xml %s/' % (asdm, newdir))
    modifiedFiles = ['SysCal.xml','ASDM.xml']
    print("...except ", modifiedFiles)
    for mF in modifiedFiles:
        if (os.path.exists(newdir+'/'+mF)):
            os.remove(newdir+'/'+mF)              
    if (os.path.exists(asdm+'/SysCal.xml')):
        os.system('mv '+asdm+'/SysCal.xml '+asdm+'/SysCal.xml.old')
#    print "CASA_TELCAL = ", os.getenv('CASA_TELCAL')
#    tasksum = {}  # needed to prevent exception with execfile
#    execfile(os.getenv('CASA_TELCAL')) # still fails upon tc_atmosphere
    try:
        from tc_atmosphere_cli import tc_atmosphere_cli as tc_atmosphere
        print("Running tc_atmosphere('%s', dataorigin='specauto', calresult='%s', showplot=%s, sidebandgainoption='%s', sidebandgain=%f, water='%s', tsysmode='%s',verbose=%s,scan='%s')" % (asdm,asdm, showplot, sidebandgainoption, sidebandgain, water, tsysmode, verbose, scan))
        tc_atmosphere(asdm, dataorigin='specauto', calresult=asdm, 
                      showplot=showplot, sidebandgainoption=sidebandgainoption,
                      sidebandgain=sidebandgain, water=water,tsysmode=tsysmode,
                      verbose=verbose, scans=scan)
        print("Copying the original .xml files back to the ASDM.")
        os.system('cp %s/*.xml %s/' % (newdir,asdm))
        print("Done")
    except:
        print("Now you can start casapy-telcal and run:")
        print("execfile(os.getenv('CASA_TELCAL'))")
        print("tc_atmosphere('%s', dataorigin='specauto', calresult='%s', showplot=False, sidebandgainoption='%s', sidebandgain=%f, water='%s', tsysmode='%s', verbose=%s, scans='%s')" % (asdm,asdm,sidebandgainoption,sidebandgain,water,tsysmode,verbose,scan))

def uidToSlashes(uid):
    """
    Converts uid___A002_Xabcd_X001 to uid://A002_Xabcd_X001.
    For the inverse function, see uidToUnderscores.
    -Todd Hunter
    """
    return(uid.replace('_',':',1).replace('_','/'))

def uidToUnderscores(asdm):
    """
    Converts uids from native name to underscore-delimited name (if necessary), preserving any initial path 
    if it has one.  Will not corrupt a pathname that was previously converted.
    -Todd Hunter
    """
    if (asdm.find('uid://') >= 0):
        asdm = asdm.replace('uid://','uid___')
        asdm = asdm.split('uid___')[0] + 'uid___' + asdm.split('uid___')[1].replace('/','_')
    return asdm

def importandlist(asdmlist, suffix='.listobs', outpath='./', 
                  asis='Antenna Station Receiver Source CalAtmosphere CalWVR CorrelatorMode SBSummary ExecBlock',
                  bdfflags=True, applyflags=False, tbuff=0.0, overwrite=False,
                  rungencal=False, process_caldevice=False, runplotbandpass=False):
    """
    Run importasdm (if necessary) followed by listobs on a list of ASDMs
    asdmlist: either a list ['uid1','uid2'] or a wildcard string 'uid*')
    overwrite: passed to importasdm and listobs
    rungencal: if True, then create Tsys table
    runplotbandpass: if True, then create Tsys table and plot it with overlay='time'
    bdfflags, asis, applyflags, tbuff, process_caldevice: passed to importasdm
    Todd Hunter
    """
    if (type(asdmlist) == str):
        mylist = glob.glob(asdmlist)
        if (len(mylist) == 0):
            uids = asdmlist.split(',')
            asdmlist = []
            for uid in uids:
                asdmlist += glob.glob(uidToUnderscores(uid))
        else:
            asdmlist = mylist
    vislist = []
    if not process_caldevice:
        print("Note: process_caldevice=False.  Set to True if you are experimenting with FDM Tsys.")
    for asdm in asdmlist:
        if (asdm[-1] == '/'): asdm = asdm[:-1]
        if (asdm[-3:] == '.ms' or asdm[-8:] == '.listobs' or
            asdm[-4:] == '.log' or asdm[-5:] == '.last' or
            asdm[-13:] == '.flagversions' or asdm[-8:] == '_cmd.txt'): continue
        asdm = uidToUnderscores(asdm)
        outvis = outpath + os.path.basename(asdm) + '.ms'
        if (not os.path.exists(outvis) or overwrite):
            for xmlfile in asis.split():
                if len(glob.glob(asdm+'/'+xmlfile.strip()+'.xml')) < 1:
                    print("Could not find an xml file for asis parameter: %s. Dropping from asis." % (xmlfile))
                    asis = asis.replace(xmlfile,'')
            if (casaVersion >= '4.3.0'):
                print("Running importasdm('%s', vis='%s', asis='%s', bdfflags=%s, applyflags=%s, tbuff=%f, process_caldevice=%s)" % (asdm, outvis, asis, bdfflags, applyflags, tbuff, process_caldevice))
                importasdm(asdm, vis=outvis, asis=asis, bdfflags=bdfflags, process_caldevice=process_caldevice,
                           applyflags=applyflags, tbuff=tbuff, overwrite=overwrite)
            else:
                print("Running importasdm('%s', vis='%s', asis='%s', applyflags=%s, tbuff=%f)" % (asdm,outvis, asis,applyflags,tbuff))
                importasdm(asdm, vis=outvis, asis=asis, applyflags=applyflags, tbuff=tbuff, 
                           overwrite=overwrite, process_caldevice=process_caldevice)
        else:
            print("Not running importasdm because ms exists and overwrite=False")
        if (rungencal or runplotbandpass):
            gencal(outvis, caltype='tsys', caltable=outvis+'.tsys')
        if (runplotbandpass):
            plotbandpassOverlayTime(outvis+'.tsys')
        vislist.append(outvis)
    listobslist(vislist, overwrite=overwrite, outpath='')
    
def asdmExportImport(asdmlist, args='', suffix='.listobs', outpath='./', 
                     asis='Antenna Station Receiver Source CalAtmosphere CalWVR CorrelatorMode SBSummary',
                     bdfflags=True, applyflags=False, tbuff=0.0, overwrite=False,
                     rungencal=False, process_caldevice=False, runplotbandpass=False):
    """
    Runs asdmExportLight on the NRAO-CV Lustre system on one ASDM or a list
    of ASDMs, then runs au.importandlist.
    args: passed to asdmExportLight
    rest of arguments: passed to au.importandlist
    -Todd Hunter
    """
    if (outpath[-1] != '/'): outpath += '/'
    asdmlist = asdmExport(asdmlist, args, outpath)
    print("Running importandlist(%s,outpath='%s')" % (asdmlist, outpath))
    importandlist(asdmlist, suffix, outpath, asis, bdfflags, applyflags, tbuff,
                  overwrite, rungencal, process_caldevice, runplotbandpass)

def fillAsdmList(asdmlist):
    """
    Takes a list like: 'uid___A002_Xabf8b9_X504, X38f'
         and returns ['uid___A002_Xabf8b9_X504',
                      'uid___A002_Xabf8b9_X38f']
    -Todd Hunter
    """
    asdmlist = asdmlist.split(',')
    asdms = []
    for i,asdm in enumerate(asdmlist):
        asdm = uidToUnderscores(asdm)
        if (len(asdm.split('_')) < 6):
            if (i==0):
                print("First ASDM in the list must be a complete name (with 5 underscores)")
                return
            loc = asdms[0].replace('_','-',4).find('_')
            asdms.append(asdms[0][:loc+1] + asdm.strip())
        else:
            asdms.append(asdm.strip())
    return(asdms)

def asdmUpdate(asdm, options='-y'):
    """
    Checks whether an ASDM has been updated in the Archive.
    -Todd Hunter
    """
    cmd = 'bash -c "source /lustre/naasc/sciops/pipeline/pipeline_env.asdmExportLight.sh ; asdmUpdate %s %s"' % (options,asdm)
    os.system(cmd)

def asdmsFromPPR(ppr):
    """
    Returns the list of ASDMs in a PPR.
    -Todd Hunter
    """
    if not os.path.exists(ppr):
        print("Could not find PPR: ", ppr)
        return
    lines = grep(ppr, 'AsdmDiskName')[0].split('\n')
    asdmlist = []
    for line in lines:
        if (len(line) > 0):
            asdmlist.append(line.split('>')[1].split('<')[0])
    return asdmlist

def asdmExportFromPPR(ppr, args='', outpath='./', dryrun=False):
    """
    Runs asdmExportLight on all ASDMs found in a PPR.
    -Todd Hunter
    """
    if not os.path.exists(ppr):
        print("Could not find PPR: ", ppr)
        return
    asdmlist = asdmsFromPPR(ppr)
    print("Working on list: ", asdmlist)
    if not dryrun:
        asdmExport(asdmlist, args, outpath)

def unpackWeblogs(mydir, dryrun=False):
    """
    Given a list of directories, unpacks the first *.tar.gz (or *.tgz) weblog in all of them.
    mydir: list of strings, or single comma-delimited string with optional wildcard characters
    Example: au.unpackWeblogs('*qa')
    -Todd Hunter
    """
    if (mydir.find('*') >= 0):
        mydir = glob.glob(mydir)
    elif type(mydir) == str:
        mydir = mydir.split(',')
    for d in mydir:
        tarball = glob.glob(d+'/*z')
        if (len(tarball) > 0):
            cmd = 'tar xzf %s -C %s' % (tarball[0],d)
            print("Running: ", cmd)
            if not dryrun:
                os.system(cmd)

def rebuildWeblog(stages, context='last'):
    """
    Execute this from the working directory of a pipeline run.
    Commands posted to CAS-8906 by Stewart Williams.
    stages: either a single stage number (17) or a range '17~20'
         will invoke the new style triggering.  The following strings invoke the old:
        'lowgainflag' -->  hif_lowgainflag (experimental)
        'imaging' --> uvcontfit, uvcontsub, makeimlist, makeimages
        'timegaincal' --> hifa_timegaincal
        'imageprecheck' --> hifa_imageprecheck and hif_checkproductsize (never worked)
    context: 'last', None, or the name of a pickle file
    -Todd Hunter
    """
    import pipeline
    import pipeline.infrastructure.renderer.htmlrenderer
    if stages in ['imaging','timegaincal','imageprecheck','lowgainflag']:
        import pipeline.infrastructure.renderer.weblog as weblog
        if stages == 'imaging':
            weblog.add_renderer(pipeline.hif.tasks.findcont.findcont.FindCont, pipeline.hif.tasks.findcont.renderer.T2_4MDetailsFindContRenderer(always_rerender=True), group_by=weblog.UNGROUPED)
            weblog.add_renderer(pipeline.hif.tasks.uvcontsub.uvcontfit.UVcontFit, pipeline.hif.tasks.uvcontsub.renderer.T2_4MDetailsUVcontFitRenderer(always_rerender=True), group_by=weblog.UNGROUPED)
            weblog.add_renderer(pipeline.hif.tasks.uvcontsub.uvcontsub.UVcontSub, pipeline.hif.tasks.uvcontsub.renderer.T2_4MDetailsUVcontSubRenderer(always_rerender=True), group_by=weblog.UNGROUPED)
            weblog.add_renderer(pipeline.hif.tasks.makeimlist.makeimlist.MakeImList, pipeline.infrastructure.renderer.basetemplates.T2_4MDetailsDefaultRenderer(uri='makeimlist.mako', description='Compile a list of cleaned images to be calculated', always_rerender=True), group_by=weblog.UNGROUPED)
            weblog.add_renderer(pipeline.hif.tasks.makeimages.makeimages.MakeImages, pipeline.hif.tasks.tclean.renderer.T2_4MDetailsTcleanRenderer(description='Calculate clean products', always_rerender=True), group_by=weblog.UNGROUPED)
        elif stages == 'timegaincal':
            weblog.add_renderer(pipeline.hifa.tasks.gaincal.timegaincal.TimeGaincal, pipeline.hif.tasks.gaincal.renderer.T2_4MDetailsGaincalRenderer(description='Gain calibration', always_rerender=True), group_by='session')
        elif stages == 'lowgainflag':  # experimental
            weblog.add_renderer(pipeline.hif.tasks.lowgainflag.lowgainflag.Lowgainflag, pipeline.hif.tasks.lowgainflag.renderer.T2_4MDetailsLowgainFlagRenderer(description='Flag antennas with low gain', always_rerender=True), group_by='session')
        elif stages == 'imageprecheck':  # hifa_imageprecheck, hif_checkproductsize
            weblog.add_renderer(pipeline.hifa.tasks.imageprecheck.ImagePreCheck, pipeline.hifa.tasks.imageprecheck.renderer.T2_4MDetailsCheckProductSizeRenderer(description='Image precheck', always_rerender=True), group_by=weblog.UNGROUPED)
            weblog.add_renderer(pipeline.hif.tasks.checkproductsize.checkproductsize.CheckProductSize, pipeline.hif.tasks.checkproductsize.renderer.T2_4MDetailsCheckProductSizeRenderer(description='Check product size', always_rerender=True), group_by=weblog.UNGROUPED)
        else:
            print("stages must be either 'timegaincal', 'imaging', or 'imageprecheck'")
            return
        context = pipeline.Pipeline(context=context).context
        pipeline.infrastructure.renderer.htmlrenderer.WebLogGenerator.render(context)
    else:
        if type(stages) == str:
            if stages.find('~') > 0:
                firstStage,finalStage = stages.split('~')
            else:
                firstStage = str(stages)
                finalStage = str(stages)
        elif type(stages) == list:
            firstStage,finalStage = stages
        else:
            firstStage = stages
            finalStage = stages
        for i,stage in enumerate(range(int(firstStage),int(finalStage)+1)):
            print("Rendering stage ", stage)
            os.environ['WEBLOG_RERENDER_STAGES'] = str(stage)
            if i == 0:
                context = pipeline.Pipeline(context=context).context
            pipeline.infrastructure.renderer.htmlrenderer.WebLogGenerator.render(context)

def mousExport(mouslist='',args='', outpath='./', env='/lustre/naasc/sciops/comm/rindebet/pipeline/scripts/pipeline_env_CASA5.6.1-8.sh', filename='', status='Pass', firstEBonly=True):
    """
    Finds the first science EB from an MOUS and exports it using asdmExport.
    status: QA0 status
    firstEBonly: if False, then export all EBs
    """
    if mouslist == '' and filename == '':
        print("You must specify either mouslist or filename")
        return
    if mouslist == '':
        if not os.path.exists(filename):
            print("Could not find file: ", filename)
            return
        f = open(filename,'r')
        lines = f.readlines()
        f.close()
        mouslist = []
        for line in lines:
            mouslist.append(uidToUnderscores(line.strip('\n')))
    if (type(mouslist) == str):
        mouslist = fillAsdmList(mouslist)
        if (mouslist == None): return
    asdmlist = []
    for mous in mouslist:
        datasets = datasetsForMOUS(mous, status=status)
        if len(datasets) < 1 or datasets[0] == 'T':
            print("No EBs found for %s" % (mous))
            continue
        if firstEBonly:
            asdmlist += [datasets[0]]
        else:
            asdmlist += datasets
    print("Calling asdmExport(%s, %s, %s, %s, %s)" % (asdmlist, args, outpath, env, filename))
    asdmExport(asdmlist, args, outpath, env, filename)

def asdmExport(asdmlist='', args='', outpath='./', env='/lustre/naasc/sciops/comm/rindebet/pipeline/scripts/pipeline_env_CASA5.6.1-8_newpmr.sh', filename=''):
    """
    Runs asdmExportLight on the NRAO-CV Lustre system on one ASDM or
    a list of ASDMs.  Replaces '.ms' with ''.  Replaces '://' with '___' etc.
    asdmlist: comma-delimited list: 'asdm1,asdm2',  or list of strings: ['asdm1','asdm2']
    args: for example "-m" to export only the metadata (XML files)
    filename: read ASDM names from a text file, one per line
    -Todd Hunter
    """
    if asdmlist == '' and filename == '':
        print("You must specify either asdmlist or filename")
        return
    if asdmlist == '':
        if not os.path.exists(filename):
            print("Could not find file: ", filename)
            return
        f = open(filename,'r')
        lines = f.readlines()
        f.close()
        asdmlist = []
        for line in lines:
            asdmlist.append(uidToUnderscores(line.strip('\n')))
    if (type(asdmlist) == str):
        asdmlist = fillAsdmList(asdmlist)
        if (asdmlist == None): return
    newlist = []
    if (outpath[-1] != '/'):
        outpath += '/'
    if args == 'm':
        print("Are you sure you didn't mean -m?")
        return
    print("Processing %d ASDMs" % (len(asdmlist)))
    for asdm in asdmlist:
        asdm = asdm.replace('.ms','')
        print("*** Exporting '%s'." % (asdm))
        if (outpath != './' and outpath != ''):
            cmd = 'bash -c "source %s ; asdmExportLight %s --outputdirectory=%s %s"' % (env,args,outpath,asdm)
        else:
            cmd = 'bash -c "source %s ; asdmExportLight %s %s"' % (env,args,asdm)
#            cmd = 'bash -c "source /lustre/naasc/sciops/pipeline/pipeline_env.asdmExportLight.sh ; asdmExportLight %s %s"' % (args,asdm)
        print("Running ", cmd)
        os.system(cmd)
        newlist.append(outpath+asdm)
    return(newlist)

def examineContext(context, msNumber=0, use_qa_tos=False):
    """
    Examines a pipeline context file for the PI science goals and prints them
       f = open(context, 'r')' 
       c = pickle.load(f)
       for attr in c.observing_run.measurement_sets[msNumber].science_goals: print attr
    use_qa_tos: if True then print quantities as human readable strings
    -Todd Hunter
    """
#  old version that worked for python 2.7 generated contexts
#    f = open(context,'r')
#    c = pickle.load(f)
    c = pickle.load(open(context,'rb'))
    print(c)
    sgs = c.observing_run.measurement_sets[msNumber].science_goals
    for i,attr in enumerate(sgs):
        if use_qa_tos:
            myqa = createCasaTool(qatool)
            print('%s: %s' % (attr, myqa.tos(myqa.quantity(list(sgs.values())[i]))))
            myqa.done()
        else:
            print(attr, list(sgs.values())[i])
    return c
#    f.close()

def pipelineAntpos(workingdir, returnAntennaNames=False):
    """
    Reports the number of corrections in the antennapos.csv file of a pipeline run.
    workingdir: a csv file, or the path to the directory containing it
    returnAntennaNames: if 'list', then return a unique list of antenna names;
                        if 'string', return a comma-delimited string
    -Todd Hunter
    """
    if os.path.isdir(workingdir):
        csvfile = os.path.join(workingdir,'antennapos.csv')
    else:
        csvfile = workingdir
    if not os.path.exists(csvfile):
        print("No antennapos.csv file found.")
        return 0
    result = grep(csvfile,'uid')[0].split('\n')
    if returnAntennaNames != False:
        antennas = []
        for corr in result:
            if len(corr) > 0:
                antennas.append(corr.split(',')[1])
        if returnAntennaNames == 'list':
            return sorted(np.unique(antennas))
        else:
            return ','.join(sorted(np.unique(antennas)))
    result = np.where(np.array(result) != '')[0]
    return len(result)

def pipelineRobust(workingdir):
    """
    Reports the value of robust used by a pipeline run by searching for the log
    message emitted at the end of hifa_imageprecheck in the CASA log.
    e.g.   grep "hifa_imageprecheck.*robust=" casa.log
    -Todd Hunter
    """
    log = findPipelineCasaLogfile(workingdir, fast=True, 
                                  searchString=' ...hifa_imageprecheck')
    print("Found log: ", log)
    result = grep(log, 'hifa_imageprecheck::::.*robust=')[0]
    robust = None
    if result.find('robust=') > 0:
        robust = float(result.split('robust=')[1].rstrip('\n'))
    else:
        print("Did not find value in matched string: ", result)
    return robust
    
def pipelineSevere(casalog, keystring='SEVERE', secondList=0):
    """
    Extracts any line containing SEVERE printed in a casa log.
    secondList: if not zero, then return a second list of matches
       that are this many lines relative to the first match (positive = after)
    -Todd Hunter
    """
    result = grep(casalog, keystring)
    lines = result[0].split('\n')
    if secondList == 0:
        return lines
    else:
        secondLines = []
        allLines = getLineFromFile(casalog)
        for line in lines:
            if line.find(':') > 0:
                lineNumber = int(line.split(':')[0])
                secondLine = allLines[lineNumber+secondList-1] # getLineFromFile(casalog,lineNumber+secondList-1)
                secondLines.append(secondLine.rstrip('\n'))
        return lines, secondLines

def plotmsMemory(casalog, keystring=' GB of memory', returnMaxLine=False):
    """
    Extracts the plotms cache requirements printed in a casa log and reports
    the min, median and max values in GB.
    -Todd Hunter
    """
    result = grep(casalog, 'plotms cache will require')
    lines = result[0].split('\n')
    memory = []
    mylines = []
    for line in lines:
        if line.find(keystring) > 0:
            memory.append(float(line.split(keystring)[0].split()[-1]))
            mylines.append(line)
    if len(memory) > 0:
        if returnMaxLine:
            idx = np.argmax(memory)
            return np.min(memory), np.median(memory), np.max(memory), mylines[idx]
        else:
            return np.min(memory), np.median(memory), np.max(memory)
    else:
        return

def pipelineMakeRequest(ous, env='pipeline_env_C5P2.sh', imaging=True, downloadAsdms=False):
    """
    Runs pipelineMakeRequest on the NRAO-CV Lustre system on an ALMA OUS and recipe.
    ous: name of uid (either underscore delimited or colon slash delimited)
    env: name of script in Remy's pipeline/scripts area
    imaging: if True, then use procedure_hifa.xml, otherwise use procedure_hifacal.xml
    downloadAsdms: if True, then retrieve the ASDMs
    Example run: pipelineMakeRequest <mous> intents_hifa.xml procedure_hifa.xml False
    -Todd Hunter
    """
    if imaging:
        proc = 'procedure_hifa_calimage.xml'
    else:
        proc = 'procedure_hifa_cal.xml'
    ous = uidToSlashes(ous)
    cmd = 'bash -c "source /lustre/naasc/sciops/comm/rindebet/pipeline/scripts/%s ; pipelineMakeRequest %s intents_hifa.xml %s %s"' % (env, ous,proc,str(downloadAsdms))
    print("Running ", cmd)
    os.system(cmd)

def listobslist(vislist, suffix='.listobs', outpath='', overwrite=False, verbose=True, field=''):
    """
    Run listobs on a list of measurement sets
    vislist: either a list ['a.ms','b.ms'], a comma-delimited string, or a wildcard string e.g. '*.ms'
    outpath: if blank, write to same directory as measurement set, otherwise 
             write to specified directory, using basename of measurement set
    -Todd Hunter
    """
    if (type(vislist) == str):
        if vislist.find('*') >= 0:
            vislist = glob.glob(vislist)
        else:
            vislist = vislist.split(',')
        if len(vislist) == 0:
            print("No qualifying measurement sets found.")
    for vis in vislist:
        vis = vis.rstrip('/')
        if len(outpath) > 0:
            listfile = os.path.join(outpath,os.path.basename(vis)+suffix)
        else:
            listfile = vis+suffix
        if (not os.path.exists(listfile) or overwrite):
            if (os.path.exists(listfile)):
                os.remove(listfile)
            print("Running listobs('%s', listfile='%s', field='%s')" % (vis, listfile, field))
            listobs(vis, listfile=listfile, field=field)

class CrossScan:
    def __init__(self, vis, intent='CALIBRATE_POINTING#ON_SOURCE'):
        self.vis = vis
        self.myms = createCasaTool(mstool)
        self.myms.open(self.vis)
        self.mymsmd = createCasaTool(msmdtool)
        self.mymsmd.open(self.vis)
        self.scans = self.mymsmd.scansforintent(intent)
        self.spws = self.mymsmd.spwsforintent(intent)
        try:
            self.sqldspws = np.intersect1d(self.mymsmd.almaspws(sqld=True), self.spws)
        except:
            self.sqldspws = np.intersect1d(self.mymsmd.chanavgspws(), self.spws)
        self.mymsmd.close()
        print("Scans for %s = %s" % (intent, self.scans))
        print("Spws for %s = %s" % (intent, self.spws))
        print("SQLDs for %s = %s" % (intent, self.sqldspws))
        self.datadescids = {}
        for spw in self.spws:
            self.datadescids[spw]=spw  # assume this is true for now, use msmd in casa 4.3

    def onMinusOffOverOff(self, antenna, spw, scan, subscan, pol=0, verbose=False):
        includeDate = False
        result = computeDurationOfScan(scan, vis=self.vis, returnSubscanTimes=True, 
                                       verbose=verbose,includeDate=includeDate)
        self.timestamps = {}
        self.timestamps[scan] = result[2]
        print("Found %d subscans" % (len(self.timestamps[scan])))
        self.myms.selectinit(datadescid=self.datadescids[spw])
        self.myms.select({'time':self.timestamps[scan][subscan],
                          'antenna1':antenna, 'antenna2':antenna})
        data0 = self.myms.getdata(['amplitude'])['amplitude']  # keyed by pol: 0, 1
        pb.clf()
        print("shape(self.timestamps[scan][subscan]=%s, shape(data0[pol])=%s" % (np.shape(self.timestamps[scan][subscan]), np.shape(data0[pol])))
        pb.plot(self.timestamps[scan][subscan], data0[pol])
        pb.draw()

def plotSQLDs(vis, plotfile='', xaxis='time', yaxis='amp', coloraxis='scan', field='',
              customflaggedsymbol=True, flaggedsymbolshape='square', baseband=1,
              ignoreIntent='CALIBRATE_POINTING#ON_SOURCE',avgtime='1.008'):
    """
    Plots the SQLD spws for the specified baseband on the specified field.
    customflaggedsymbol: only needed to be set if data were flagged (which pipeline does)
    baseband: single integer: 1..4
    ignoreIntent: single value (pointing data take forever to plot, and rarely worth examining)
    avgtime: value in seconds (floating point or string)
    field: argument passed to plotms
    -Todd Hunter
    """
    if not os.path.exists(vis):
        print("Could not find measurement set")
        return
    mymsmd = msmdtool()
    mymsmd.open(vis)
    scan = list(mymsmd.scannumbers())
    if ignoreIntent != '':
        scans = mymsmd.scansforintent(ignoreIntent)
        for s in scans:
            scan.remove(s)
    scan = ','.join([str(i) for i in scan])
    sqldSpws = mymsmd.almaspws(sqld=True)
    print("sqld spws: ", sqldSpws)
    spws = mymsmd.spwsforbaseband(baseband)
    if len(spws) < 1:
        print("No spws in that baseband")
        return
    print("spws for baseband: ", spws)
    spws = [str(i) for i in np.intersect1d(spws, sqldSpws)]
    spwsPlotted = []
    for spw in spws:
        intents = list(mymsmd.intentsforspw(int(spw)))
        if ignoreIntent in intents:
            intents.remove(ignoreIntent)
        if 'CALIBRATE_WVR#ON_SOURCE' in intents:
            intents.remove('CALIBRATE_WVR#ON_SOURCE')
        if len(intents) > 0:
            spwsPlotted.append(spw)
    if len(spwsPlotted) == 1:
        meanFreq = ', %.1fGHz' % (mymsmd.meanfreq(int(spw))*1e-9)
    else:
        meanFreq = ''
    mymsmd.close()
    spw = ','.join(spwsPlotted)
    print("scan=", scan)
    print("spw=", spw)
    if plotfile == True:
        plotfile = vis + "_spw%s_SQLD.png" % (spw)
        print("Set plotfile: %s" % (plotfile))
    plotms(vis, spw=spw, xaxis=xaxis, yaxis=yaxis, coloraxis=coloraxis, 
           plotfile=plotfile, overwrite=True, field=field, 
           customflaggedsymbol=customflaggedsymbol, avgtime=str(avgtime),
           flaggedsymbolshape=flaggedsymbolshape, scan=scan, 
           title=os.path.basename(vis)+' spw'+spw+meanFreq)

def sqldspws(vis):
    """
    Return the SQLD spws for version of casa prior to 26688 which do 
    not have msmd.almaspws().  - Todd Hunter
    """
    mytb = createCasaTool(tbtool)
    mytb.open(vis+'/SPECTRAL_WINDOW')
    names = mytb.getcol('NAME')
    mytb.close()
    sqld = []
    for i,name in enumerate(names):
        if (name.find('#SQLD')>0):
            sqld.append(i)
    return(sqld)

class SQLD():
    def __init__(self, vis, copy_ms=False, gettmcdb=False, outpath='./', verbose=False, dbm=True):
        """
        vis: the measurement set to operate with
        copy_ms: if set to True, it will first copy vis.ms into vis.sqld in the pwd
                 and then operate on that dataset instead of vis.ms
        gettmcdb: if True, then also get the BB detector values from the TMC DB
        outpath: the path to write the TMC DB data (default = './')
        dbm: if True, convert TMC DB values to dBm;  if False, convert to mW
        """
        if not os.path.exists(vis):
            print("Could not find measurement set.")
            return
        if not os.path.exists(vis+'/table.dat'):
            print("This does not appear to be a measurement set.")
            return
        if copy_ms:
            basename = vis.rstrip('.ms')
            newname = os.path.basename(basename)
            if (os.path.exists(newname+'.sqld')):
                print("Removing existing .sqld directory")
                shutil.rmtree(newname+'.sqld')
            cmd = 'rsync -au %s/ %s.sqld' % (vis, newname)
            print("Running: %s" % (cmd))
            os.system(cmd)
            vis = newname + '.sqld'
            print("Setting vis=%s" % (vis))
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        self.vis = vis

        if (getCasaSubversionRevision() < casaRevisionWithAlmaspws):
            self.sqldspws = sqldspws(self.vis)
        else:
            self.sqldspws = mymsmd.almaspws(sqld=True)
        if (getCasaSubversionRevision() > '27480'):
            self.antennaNames = mymsmd.antennanames() 
        else:
            self.antennaNames = getAntennaNames(self.vis)
        self.intents = mymsmd.intents()
        if ('CALIBRATE_ATMOSPHERE#ON_SOURCE' not in self.intents):
            self.atmcalscans = []
        else:
            self.atmcalscans = mymsmd.scansforintent('CALIBRATE_ATMOSPHERE#ON_SOURCE')
        if ('OBSERVE_TARGET#ON_SOURCE' not in self.intents):
            self.observeTargetScans = []
        else:
            self.observeTargetScans = mymsmd.scansforintent('OBSERVE_TARGET#ON_SOURCE')
        if (len(self.sqldspws) == 0):
            print("There are no SQLD spws in this dataset.")
            if (not gettmcdb):
                gettmcdb = True
                print("Setting gettmcdb to True")
        self.datadescIDs = {}
        self.basebands = {}
        self.pols = [0,1]  # call findNumberOfPolarizations
        self.date = getObservationStartDate(self.vis).split()[0]  # YYYY-MM-DD
        self.yyyymmdd = self.date
        if (self.yyyymmdd < '2013-05-29' and gettmcdb):
            gettmcdb = False
            print("These data are too old to have BB detector values stored in the TMCDB.")
        self.yyyymmdd2 = getObservationStopDate(self.vis).split()[0]
        if (self.yyyymmdd != self.yyyymmdd2):
            print("WARNING: This dataset spans two UT days, which is not yet fully supported.")
        self.IFProc = {}
        self.startTime = np.min(mymsmd.timesforscan(mymsmd.scannumbers()[0]))
        self.endTime =   np.max(mymsmd.timesforscan(mymsmd.scannumbers()[-1]))
        self.scanTimes = {}
        self.intents = {}
        for scan in mymsmd.scannumbers():
            self.scanTimes[scan] = [np.min(mymsmd.timesforscan(scan)),
                                    np.mean(mymsmd.timesforscan(scan)),
                                    np.max(mymsmd.timesforscan(scan))]
            self.intents[scan] = mymsmd.intentsforscan(scan)
        for spw in self.sqldspws:
            try:
                self.datadescIDs[spw] = mymsmd.datadescids(spw)
            except:
                self.datadescIDs[spw] = spw
            self.basebands[spw] = mymsmd.baseband(spw)
        self.sqldData = {}
        self.sqldDataDBm = {}
        self.sqldDataUTC = {}
        self.dbm = dbm
        if (gettmcdb):
            print("Retrieving the periodic measurements from TMCDB.")
            self.getTMCDBData(outpath=outpath, verbose=verbose)
        mymsmd.close()

    def plotTMCDB(self, antenna='', pol='', basebands='', wholeDay=False, 
                  labelscans=True, labelintents=True, plotfile=True, yrange=[0,5],
                  verbose=False, markdbm=[+2.4,+3.8], timeBuffer=40, units='dBm'):
        """
        Plots the power vs. time of the total power detectors as read from the
        ALMA TMC database.
        antenna: integer, string, or list of antenna IDs or names
        pol: 0, 1 or [0,1] (default)
        basebands: string or list of basebands numbers to plot (1..4)
        wholeDay: if True, plot the whole day of data, not just during the dataset
        labelscans: if True, demarcate and label the scan numbers
        labelintents: if True, label ATM, POI, SID, OBS intent scans
        plotfile: path and name of png to produce, or True for automatic
        yrange: the y-axis plotrange: [y0,y1]
        markdbm: draw a horizontal line at the specified level(s), set to None for none
        timeBuffer: value in seconds to widen the beginning and end of the dataset
        units: the y-axis units (either 'dBm' or 'milliwatt')
        """
        if (pol==''):
            polList = [0,1]
        antennaList = parseAntenna(self.vis, antenna)
        pb.clf()
#        pb.hold(True)  # not available in CASA6, but also not needed 
        if (basebands == ''):
            basebands = list(self.sqldDataDBm[self.antennaNames[antennaList[0]]][polList[0]].keys())
        else:
            basebands = parseBasebandArgument(basebands)
        linestyles = ['.-','.--']
        for baseband in basebands:
            if (len(basebands) == 1):
                desc = pb.subplot(1,1,1)
                nplots = 1
                nrows = 1
                column = 1
                row = 1
            else:
                desc = pb.subplot(2,2,baseband)
                nplots = 4
                nrows = 2
                if (baseband % 2 == 1):
                    column = 1
                else:
                    column = 2
                if (baseband < 3):
                    row = 1
                else:
                    row = 2
            pb.title('Baseband %d' % (baseband))
            for antennaID in antennaList:
                antenna = self.antennaNames[antennaID]
                if (verbose): print("Working on antenna %s baseband %d" % (antenna, baseband))
                for pol in polList:
                    if (wholeDay):
                        timeStamps = pb.date2num(mjdSecondsListToDateTime(list(self.sqldDataUTC[antenna][pol])))
                        day = mjdsecToUT(self.sqldDataUTC[antenna][pol][0]).split()[0]
                        yaxis = self.sqldDataDBm[antenna][pol][baseband]
                        if (units.lower() == 'milliwatt' or units.lower() == 'mw'):
                            yaxis = 10**(0.1*yaxis)
                        pb.plot_date(timeStamps, yaxis, linestyles[pol], color=overlayColors[antennaID])
                        setXaxisTimeTicks(desc, np.min(self.sqldDataUTC[antenna][pol]),
                                          np.max(self.sqldDataUTC[antenna][pol]))
                    else:
                        idx1 = np.where(self.sqldDataUTC[antenna][pol] >= self.startTime-timeBuffer)[0]
                        idx2 = np.where(self.sqldDataUTC[antenna][pol] <= self.endTime+timeBuffer)[0]
                        idx = np.intersect1d(idx1,idx2)
                        timeStamps = pb.date2num(mjdSecondsListToDateTime(list(np.array(self.sqldDataUTC[antenna][pol])[idx])))
                        day = mjdsecToUT(self.sqldDataUTC[antenna][pol][idx[0]]).split()[0]
                        yaxis = self.sqldDataDBm[antenna][pol][baseband][idx]
                        if (units.lower() == 'milliwatt' or units.lower() == 'mw'):
                            yaxis = 10**(0.1*yaxis)
                        pb.plot_date(timeStamps, yaxis, 
                                     linestyles[pol], color=overlayColors[antennaID])
                        setXaxisTimeTicks(desc, self.startTime, self.endTime, verbose=False)
                if (nplots==1 or baseband==2):
                    pb.text(1.02,1.04-antennaID*0.025*nrows, antenna, size=9,
                            transform=desc.transAxes,color=overlayColors[antennaID])
                    pb.text(0.5, 1.08, os.path.basename(self.vis))
            if (column == 1):
                if (self.dbm and units.lower()=='dbm'):
                    pb.ylabel('Power (dBm)')
                else:
                    pb.ylabel('Power (mW)')
                if (row == 2):
                    pb.xlabel(os.path.basename(self.vis))
            if (yrange != [0,0]):
                pb.ylim(yrange)
            if (labelscans):
                y0 = pb.ylim()[1]-(pb.ylim()[1]-pb.ylim()[0])*0.04*nrows
                for scan in list(self.scanTimes.keys()):
                    myTimes = pb.date2num(mjdSecondsListToDateTime([self.scanTimes[scan][0],self.scanTimes[scan][0]]))
                    pb.plot(myTimes, pb.ylim(), ':', color='k')
                    if (scan == list(self.scanTimes.keys())[-1]):
                        myTimes = pb.date2num(mjdSecondsListToDateTime([self.scanTimes[scan][2],self.scanTimes[scan][2]]))
                        pb.plot(myTimes, pb.ylim(), ':', color='k')

                    pb.text(mjdSecondsListToDateTime([self.scanTimes[scan][1]])[0], y0, str(scan), size=8)
                pb.xlim(pb.date2num(mjdSecondsListToDateTime([self.startTime-timeBuffer, self.endTime+timeBuffer])))
            if (yrange != [0,0]):
                pb.ylim(yrange)
            if (labelintents):
                for scan in list(self.scanTimes.keys()):
                    for intent in ['ATM','SIDEBAND','POINTING','OBSERVE','AMPLI','FLUX','BANDPASS']:
                        y0 = pb.ylim()[1]-(pb.ylim()[1]-pb.ylim()[0])*0.11*nrows
                        if (','.join(self.intents[scan]).find(intent) > 0):
                            pb.text(mjdSecondsListToDateTime([self.scanTimes[scan][1]])[0], y0, intent[:5], size=7, rotation='vertical', ha='left', va='bottom')
                            break  # be sure only one intent is written per scan
            if (self.dbm):
                if (markdbm is not None and markdbm != []):
                    if (type(markdbm) != list):
                        markdbm = [markdbm]
                    for mrk in markdbm:
                        pb.plot(pb.xlim(), [mrk,mrk], 'k:')
            else:
                if (markdbm is not None and markdbm != []):
                    if (type(markdbm) != list):
                        markdbm = [markdbm]
                    for mrk in markdbm:
                        pb.plot(pb.xlim(), [10**(mrk*0.1),10**(mrk*0.1)], 'k:')
            if (nrows == 2):
                pb.setp(desc.get_xticklabels(), fontsize=8)
                pb.setp(desc.get_yticklabels(), fontsize=8)
            if (yrange != [0,0]):
                pb.ylim(yrange)
        pb.xlabel('Time (UT on %s)'%(day))        
        yFormat = matplotlib.ticker.ScalarFormatter(useOffset=False)
        desc.yaxis.set_major_formatter(yFormat)
        pb.draw()
        if (plotfile != ''):
            if (plotfile == True):
                if (self.dbm):
                    plotfile = self.vis.replace('.sqld','') + '.sqld.dBm.png'
                else:
                    plotfile = self.vis.replace('.sqld','') + '.sqld.mW.png'
            dirname = os.path.dirname(plotfile)
            if (dirname == ''):
                dirname = './'
            if (os.access(dirname,os.W_OK) == False):
                plotfile = '/tmp/' + os.path.basename(plotfile)
            pb.savefig(plotfile)
            print("Plot left in: ", plotfile)

    def convertTMCDBtoPower(self, calibration='auto', verbose=False):
        """
        convert self.sqldData from voltage to dBm
        into self.sqldDataDBm.  The structure of sqldDataDBm is:
        [antenna][pol][baseband: 'A','B','C','D'][list of voltages vs. time]
        """
        query = False
        basebands = [0, 'A','B','C','D']
        if (self.IFProc == {}):  # contains the calibration coefficients
            query = True
        else:
            for antenna in self.antennaNames:
                if (antenna not in list(self.IFProc.keys())):
                    query = True
        if (query):
            for i,antenna in enumerate(self.antennaNames):
                if (antenna not in list(self.IFProc.keys())):
                    if (antenna not in list(self.sqldDataDBm.keys())):
                        self.sqldDataDBm[antenna] = {}
                        self.sqldDataUTC[antenna] = {}
                    self.IFProc[antenna] = self.readSQLDCalibration(antenna, calibration)
                    if (verbose):
                        print("Antenna %2d = %s: " % (i,antenna), self.IFProc[antenna])
                    for pol in self.pols:
                        self.sqldDataUTC[antenna][pol] = self.sqldData[antenna][pol][0]
                        self.sqldDataDBm[antenna][pol] = {}
                        for channel, baseband in enumerate(np.unique(list(self.basebands.values()))):  # need unique in case multiple groups of 4 SQLD spws
                            basebandLetter = basebands[baseband]
                            inputPowerAtZeroVoltage = self.IFProc[antenna][pol][basebandLetter]['icept']
                            # Scale by the gain slope, then add the offset
#                            print "shape(sqldData) = ", np.shape(self.sqldData[antenna][pol][1])
#                            print "len(np.shape(sqldData)) = ", len(np.shape(self.sqldData[antenna][pol][1]))
                            if (len(np.shape(self.sqldData[antenna][pol][1])) < 2):
                                print("There is a problem with the TMCDB data for %s pol %d (probably missing)" % (antenna,pol))
                                break
                            else:
                                voltages = np.transpose(np.array(self.sqldData[antenna][pol][1]))
                                self.sqldDataDBm[antenna][pol][baseband] = voltages[channel]*self.IFProc[antenna][pol][basebandLetter]['slope']+self.IFProc[antenna][pol][basebandLetter]['icept']
                                if (self.dbm):
                                    self.sqldDataDBm[antenna][pol][baseband] = 10*np.log10(self.sqldDataDBm[antenna][pol][baseband])
        return
        
    def getTMCDBData(self, outpath='./', verbose=False):
        monitorPoint = 'DATA_MONITOR_2'  # BB SQLDs  (_1=SB SQLDs)
        self.sqldData = {}
        for antenna in self.antennaNames:
            self.sqldData[antenna] = {}
            for pol in self.pols:  
                # need to get file for second date if necessary
                localfile = tmu.retrieve_daily_tmc_data_file_name_only(antenna,'IFProc'+str(pol), monitorPoint, self.yyyymmdd, outpath=outpath)
                if (os.path.exists(localfile)):
                    tmcfile_ifproc = localfile
                else:
                    try:
                        mydict=tmu.get_tmc_data(antenna,'IFProc'+str(pol),monitorPoint, self.yyyymmdd, self.yyyymmdd, outpath=outpath)
                        tmcfile_ifproc = mydict['files'][0]
                        self.tmcfile_ifproc = tmcfile_ifproc
                    except:
                        print("Failed to retrieve data for antenna %s" % (antenna))
                        continue
                self.sqldData[antenna][pol] = self.readIFProcBBSQLDs(tmcfile_ifproc, pol)
        self.convertTMCDBtoPower(verbose=verbose)
        return

    def readIFProcBBSQLDs(self, tmcfile, pol=0):
        """
        Returns 2 lists: dateTimeStamp in MJD second, and SQLD readings
        """
        loc = tmcfile.find('IFProc')+6
        if (loc > 0):
            tmcfile = tmcfile[:loc] + str(pol) + tmcfile[loc+1:]
            print("Using existing file = ", tmcfile)
        if (os.path.exists(tmcfile) == False):
            print("Could not open IF Proc TMC database file")
            return
        tmclines = open(tmcfile,'r').readlines()
        dateTimeStamp = []
        voltages = []
        for line in tmclines:
            tokens = line.split()
            if (len(tokens) == 0):
                print("len(tokens) = %d" % (len(tokens)))
            dateTimeStamp.append(dateStringToMJDSec(tokens[0],verbose=False))
            voltages.append([float(x) for x in tokens[1:]])
        return(dateTimeStamp, voltages)

    def convertSQLDtodBm(self, antenna='', calibration='auto', spw='',
                         zeroLevel=-20, verbose=False):
        """
        Converts the amplitude data in the SQLD spws of an ALMA measurement set
        from counts to dBm.
        Inputs:
        * vis: the measurement set
        Optional inputs:
        * calibration: a text file that contains the slope and intercept values
                 for each of the IF channels (A,B,C,D for each sideband)
        * antenna: the list of antenna names or IDs to convert
        * spw: the list of spws to convert (default is all SQLD spws)
        * zeroLevel: what value (in dBm) to assign to data points with 0 counts
        Theory:
        a) convert from counts to mV using (counts/65536)*2.5
        b) apply the slope and intercept to convert to mW
        d) convert from mW to dBm by 10*log10(mW)
        -Todd Hunter
        """
        if (len(self.sqldspws) == 0):
            print("There are no SQLD spws in this dataset!")
            return
        antennaList = parseAntenna(self.vis, antenna)
        query = False
        if (self.IFProc == {}):
            query = True
        else:
            for antenna in antennaList:
                if (antenna not in list(self.IFProc.keys())):
                    query = True
        if (query):
            for antenna in antennaList:
                if (antenna not in list(self.IFProc.keys())):
                    self.IFProc[antenna] = self.readSQLDCalibration(self.antennaNames[antenna], calibration)
                    print("Antenna %2d = %s: " % (antenna,self.antennaNames[antenna]), self.IFProc[antenna])
        basebands = [0, 'A','B','C','D']
        channel = 0
        if (spw == ''):
            spws = self.sqldspws
        else:
            spws = [int(x) for x in spw.split(',')]
        dataColumnName = getDataColumnName(self.vis)
        myt = createCasaTool(tbtool)
        myt.open(self.vis, nomodify=False)
        history = []
        for spw in spws:
            for antenna in antennaList:
                ddid = self.datadescIDs[spw]
                print("Working on spw %d (DD=%d), antenna %d (%s)" % (spw, ddid, antenna, self.antennaNames[antenna]))
                mytb = myt.query('DATA_DESC_ID==%d and ANTENNA1==%d and ANTENNA2==%d' % (ddid,antenna,antenna))
                complexData = mytb.getcol(dataColumnName)
                baseband = basebands[self.basebands[spw]]
                for pol in range(len(complexData)):
                    if (verbose):
                        print("min/max of spw=%d, pol=%d is %f/%f" % (spw,pol,np.min(np.real(complexData[pol][channel])), np.max(np.real(complexData[pol][channel]))))
                        print("complexData[pol][channel] = ", str(complexData[pol][channel]))
                    inputPowerAtZeroVoltage = self.IFProc[antenna][pol][baseband]['icept']
                    # Scale by the gain slope, then add the offset
                    arg = (complexData[pol][channel]*2.5/65536.)*self.IFProc[antenna][pol][baseband]['slope']+self.IFProc[antenna][pol][baseband]['icept']
                    if verbose:
                        print("    inputPowerAtZeroVoltage level = %f mW" % (inputPowerAtZeroVoltage))
                        if (len(np.where(np.real(arg)<=inputPowerAtZeroVoltage)[0]) > 0):
                            print("Replacing %d/%d zero values in pol 0 with -20dBm" % (len(np.where(np.real(arg)<=inputPowerAtZeroVoltage)[0]),len(arg)))
                    if (self.dbm):
                        if (dataColumnName == 'DATA'):
                            arg[np.where(np.real(arg) <= inputPowerAtZeroVoltage)[0]] = np.complex(10**(zeroLevel*0.1),0)  # i.e. convert -20 to 1e-2
                        else: # FLOAT_DATA
                            arg[np.where(np.real(arg) <= inputPowerAtZeroVoltage)[0]] = 10**(zeroLevel*0.1)
                        complexData[pol][channel] = 10*np.log10(arg)
                    else:
                        complexData[pol][channel] = arg
                        
                    if (verbose):
                        print("    after scaling, min/max is %f/%f" % (np.nanmin(np.real(complexData[pol][channel])), 
                                                                       np.nanmax(np.real(complexData[pol][channel]))))
                mytb.putcol(dataColumnName,complexData)
                mytb.close()
                history.append('Scaled antenna %2d SQLD spw %2d data into dBm' % (antenna,spw))
        myt.close()
        myms = createCasaTool(mstool)
        myms.open(self.vis, nomodify=False)
        for h in history:
            myms.writehistory(h)
        myms.listhistory()
        myms.close()

    def plotms(self, antenna='*&&&', scan='', spw='', coloraxis='spw',
               correlation='XX', iteraxis='antenna', avgtime='1s',
               plotrange=[0,0,0,0], intent='OBSERVE_TARGET',
               plotfile='', buildpdf=False, overwrite=True):
        """
        Runs plotms on a measurement set with useful parameters
        scan: if '', then plot all scans with specified intent(s)
        spw: if '', then find all SQLD spws
        intent: a list of intents, as a comma-delimited string
        buildpdf: if True, build a PDF of all antennas, one per page
                  if False, use iteraxis='antenna'
        """
        if (scan == '' and intent != ''):
            targetscans = []
            if (intent.find('OBSERVE_TARGET')>=0):
                scan = ','.join([str(i) for i in self.observeTargetScans])
                targetscans = scan
            if (intent.find('CALIBRATE_ATMOSPHERE')>=0):
                scan = ','.join(targetscans+[str(i) for i in self.atmcalscans])
                    
        if (spw == ''):
            spw = ','.join([str(i) for i in self.sqldspws])
        if (buildpdf == False):
            showgui = True
            if (antenna.find('&&&') < 0):
                antenna += '&&&'
            print("Running plotms('%s', antenna='%s', coloraxis='%s', yaxis='real', xaxis='time', correlation='%s', ylabel='Power', spw='%s', scan='%s', iteraxis='%s', avgtime='%s', plotrange=%s, plotfile='%s', overwrite=%s, showgui=%s)" % (self.vis,antenna,coloraxis,correlation,spw,scan,iteraxis,avgtime,str(plotrange),plotfile,str(overwrite),showgui))
            plotms(self.vis, antenna=antenna, coloraxis=coloraxis, yaxis='real',
                   xaxis='time',correlation=correlation,ylabel='Power',
                   spw=spw, scan=scan, iteraxis=iteraxis, avgtime=avgtime, showgui=showgui,
                   plotrange=plotrange, plotfile=plotfile, overwrite=overwrite)
        else:
            plotfiles = []
            showgui = False
            for antname in self.antennaNames:
                plotfile = self.vis + '.' + antname + '.png'
                ant = antname + '&&&'
                mytitle = os.path.basename(self.vis) + ' ' + antname + ' ' + correlation + ' Power vs. time'
                print("Running plotms('%s', antenna='%s', coloraxis='%s', yaxis='real', xaxis='time', correlation='%s', ylabel='Power (dBm)', spw='%s', scan='%s', avgtime='%s', plotrange=%s, plotfile='%s', showgui=%s, overwrite=%s, title='%s')" % (self.vis,ant,coloraxis,correlation,spw,scan,avgtime,str(plotrange),plotfile,showgui,str(overwrite),mytitle))
                plotms(self.vis, antenna=ant, coloraxis=coloraxis, yaxis='real',
                       xaxis='time',correlation=correlation,ylabel='Power (dBm)',
                       spw=spw, scan=scan, avgtime=avgtime, title=mytitle,
                       plotrange=plotrange, plotfile=plotfile, showgui=showgui,
                       overwrite=overwrite)
                plotfiles.append(plotfile)
            pdfname = self.vis + '.pdf'
            buildPdfFromPngs(plotfiles, pdfname=pdfname)
            
    def readSQLDCalibration(self, antenna, calibration='auto',overwrite=False,verbose=False):
        """
        Reads the gains (slope and intercept) of the SQLDs from a
        container log text file.
        calibration: if 'auto', then retrieve the files from the computing web server
                     if '', then use the default values for DA64 on April 1, 2014
                     if a filename, then search for values in it
        overwrite: if False (default), then check if the files have already
                   been retrieved in the current working directory
                   if True, then re-retrieve them
        -Todd Hunter
        """
        if (calibration == 'auto'):
            cal = open(compUtils.retrieve_abm_container_data_files(antenna, self.date, overwrite, verbose=False), 'r')
            lines = cal.readlines()
            cal.close()
        else:
            if not os.path.exists(calibration):
                print("Could not find calibration table. Using defaults.")
                calibration = ''
            if (calibration==''):
                lines = ['2014-03-31T22:40:08.142   IFProc0:\n',
                     '<DET ch="A" slope="1.645000" icept="-0.140000"/>\n',
                     '<DET ch="B" slope="1.585000" icept="-0.122000"/>\n',
                     '<DET ch="C" slope="1.515000" icept="-0.099200"/>\n',
                     '<DET ch="D" slope="1.490000" icept="-0.122000"/> \n',
                     '2014-03-31T22:40:33.310   IFProc1:\n',
                     '<DET ch="USB" slope="0.004900" icept="-0.000440"/>\n',
                     '<DET ch="LSB" slope="0.005900" icept="-0.000585"/>\n',
                     '<DET ch="A" slope="1.595000" icept="-0.140000"/>\n',
                     '<DET ch="B" slope="1.555000" icept="-0.113000"/>\n',
                     '<DET ch="C" slope="1.460000" icept="-0.115000"/>\n',
                     '<DET ch="D" slope="1.560000" icept="-0.118000"/>\n']
            else:
                cal = open(calibration,"r")
                lines = cal.readlines()
                cal.close()
        processor = -1
        IFProc = {}
        for line in lines:
            if (line.find('IFProc0') >= 0):
                processor = 0
            if (line.find('IFProc1') >= 0):
                processor = 1
            if (processor >= 0):
                if (line.find('DET ch=') >= 0 and line.find('SB') < 0):
                    a,b,c,d = line.split()
                    if (processor not in list(IFProc.keys())):
                        IFProc[processor] = {}
                    if (b[4] not in list(IFProc[processor].keys())):
                        IFProc[processor][b[4]] = {}
                    IFProc[processor][b[4]]['slope'] = float(c.split('"')[1])
                    IFProc[processor][b[4]]['icept'] = float(d.split('"')[1])
        return(IFProc)
    
def readAttenuatorSettings(dataset, antenna=0, pol=0, outpath='./'):
    """
    Queries the TMCDB for attenuator settings during an ALMA dataset.
    If text file already exists in the output, then it will read it.
    antenna: the antenna ID (integer)
    pol: 0 or 1
    """
    if (not os.path.exists(dataset)):
        print("Could not find dataset")
        return
    if (os.path.exists(dataset+'/table.dat')):
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(dataset)
        antennaNames = np.array(mymsmd.antennanames())
        mymsmd.close()
        yyyymmdd = getObservationStartDate(dataset).split()[0]
    else:
        antennaNames = readAntennasFromASDM(dataset, verbose=False)
        yyyymmdd = getObservationStartDateFromASDM(dataset)[0].split()[0]
    sidebands = [1,2]
    localfile = tmu.retrieve_daily_tmc_data_file_name_only(antennaNames[antenna],'IFProc'+str(pol),
                                                           'GAINS', yyyymmdd, outpath=outpath)
    if (os.path.exists(localfile)):
        tmcfile_ifproc = localfile
    else:
        try:
            mydict = tmu.get_tmc_data(antennaNames[antenna],'IFProc'+str(pol),'GAINS', yyyymmdd, yyyymmdd, outpath=outpath)
            tmcfile_ifproc = mydict['files'][0]
            tmcfile_ifproc = tmcfile_ifproc
        except:
            return False
    tmcfile_ifswitch = {}
    for sideband in sidebands:
        localfile = tmu.retrieve_daily_tmc_data_file_name_only(antennaNames[antenna],'FrontEnd_IFSwitch',
                                                               'CHANNEL%d%d_ATTENUATION'%(pol,sideband), yyyymmdd, outpath=outpath)
        if (os.path.exists(localfile)):
            tmcfile_ifswitch[sideband] = localfile
        else:
            mydict = tmu.get_tmc_data(antennaNames[antenna],'FrontEnd_IFSwitch',
                               'CHANNEL%d%d_ATTENUATION'%(pol,sideband), yyyymmdd, yyyymmdd, outpath=outpath)
            tmcfile_ifswitch[sideband] = mydict['files'][0]

    ifproc_time, ifproc_dB = readIFProcAttenuatorSettings(tmcfile_ifproc, pol=pol)
    ifswitchLSB_time, ifswitchLSB_dB = readIFSwitchAttenuatorSettings(tmcfile_ifswitch, pol=pol, sideband=1)
    ifswitchUSB_time, ifswitchUSB_dB = readIFSwitchAttenuatorSettings(tmcfile_ifswitch, pol=pol, sideband=2)
    return(ifproc_time, ifproc_dB, ifswitchLSB_time, ifswitchLSB_dB, ifswitchUSB_time, ifswitchUSB_dB)

def listOfPrimaryIntents():
    return(['POINTING', 'BANDPASS', 'FLUX', 'AMPLITUDE', 'ATMOSPHERE', 'PHASE',
            'TARGET', 'CHECK', 'DELAY', 'POLARIZATION'])

def getPrimaryIntent(intentList):
    """
    Given a list of intents, or a comma- or space-delimited string,
    return only the most important, fundamental intent, in the following
    order of precedence: POINTING, BANDPASS, FLUX/AMPLITUDE, ATMOSPHERE,
       PHASE, TARGET, CHECK, DELAY, POLARIZATION, first intent
    """
    if type(intentList) == str:
        if intentList.find(',') > 0:
            intentList = intentList.split(',')
        else:
            intentList = intentList.split()
    intentlist = ' '.join(intentList)
    for primaryIntent in listOfPrimaryIntents():
        if intentlist.find(primaryIntent) > 0:
            return primaryIntent
    return intentList[0]

def getShortIntent(intent):
    """
    Converts an intent to short name used in pipeline image filenames:
        {'PHASE': 'ph', 'BANDPASS': 'bp', 'CHECK': 'chk', 'TARGET': 'sci', 'FLUX': 'flux', 'POLARIZATION': 'pol'}
    -Todd Hunter
    """
    mydict = {'PHASE': 'ph', 'BANDPASS': 'bp', 'CHECK': 'chk', 'TARGET': 'sci', 'FLUX': 'flux', 'POLARIZATION': 'pol'}
    if intent not in mydict:
        print("Unrecognized intent: %s. Available intents: " % (intent), mydict.keys())
        return
    return(mydict[intent])

def primaryIntentLetter(intent):
    """
    Convert a primary intent into a unique letter:
    POINTING: 'G'
    BANDPASS: 'B'
    PHASE: 'P'
    ATMOSPHERE: 'A'
    AMPLITUDE: 'M'
    FLUX: 'F'
    TARGET: 'T'
    CHECK: 'C'
    DELAY: 'D'
    POLARIZATION: 'Z'
    other: '?'
    """
    if intent.find('POINTING') >= 0: return 'G'
    if intent.find('BANDPASS') >= 0: return 'B'
    if intent.find('PHASE') >= 0: return 'P'
    if intent.find('ATMOSPHERE') >= 0: return 'A'
    if intent.find('AMPLITUDE') >= 0: return 'M'
    if intent.find('FLUX') >= 0: return 'F'
    if intent.find('TARGET') >= 0: return 'T'
    if intent.find('CHECK') >= 0: return 'C'
    if intent.find('DELAY') >= 0: return 'D'
    if intent.find('POLARIZATION') >= 0: return 'Z'
    print("Returning ? for %s" % (intent))
    return('?')

def plotAttenuatorSettings(dataset, antenna='', pol='0,1', outpath='./', 
                           buildPDF=True, path='', verbose=False, figsize=(15,6),
                           intentLetterSize=9, marker='.'):
    """
    Reads the attenuators settings from the TMCDB for a specified ALMA dataset
    (measurement set or ASDM) and plots them vs. time.
    dataset: name of measurement set or ASDM
      ASDM will also label the plot with the letter of the primary scan intent
    antenna: name or ID (or a list of names or IDs), set to '' for all
    pol: 0 or 1, or '0' or '1', or set to '' for both in sequence
    outpath: the path to look for (or save) the attenuator files
    path: to look for dataset in
    marker: set to '.' (small) or 'o' (large) to have visible points shown on the line 
    Note: plots and PDF will be written to the current working directory
    """
    dataset = os.path.join(path,dataset)
    if (not os.path.exists(dataset)):
        print("Could not find dataset: ", dataset)
        return
    dataset = dataset.rstrip('/')
    if pol == 0 or pol == '0' or pol == [0]:
        pols = [0]
    elif pol == 1 or pol == '1' or pol == [1]:
        pols = [1]
    elif pol == '' or pol == [0,1] or pol == '0,1':
        pols = [0,1]
    else:
        print("pol must be either: 0, 1, '0,1' or '' (for both)")
        return
    if (os.path.exists(dataset+'/table.dat')): # dataset is a measurement set
        antennaIDs = parseAntenna(dataset, antenna)
        yyyymmdd = getObservationStartDate(dataset).split()[0]
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(dataset)
        scans = mymsmd.scannumbers()
        scantimes = []
        primaryIntentLetters = []
        for scan in scans:
            intents = mymsmd.intentsforscan(scan)
            primaryIntent = getPrimaryIntent(intents)
            primaryIntentLetters.append(primaryIntentLetter(primaryIntent))
            if verbose:
                print("%d: %s = %s" % (scan, primaryIntent,primaryIntentLetters[-1]))
            scantimes.append(np.min(mymsmd.timesforscan(scan)))
        antennaNames = np.array(mymsmd.antennanames())
        mymsmd.close()
        startMJDSec = getObservationStart(dataset)
        stopMJDSec = getObservationStop(dataset)
    else:  # dataset is an ASDM
        antennaNames = readAntennasFromASDM(dataset, verbose=False)
        antennaIDs = parseAntennaASDM(dataset, antenna)
        yyyymmdd = getObservationStartDateFromASDM(dataset)[0].split()[0]
        scaninfo = readscans(dataset)
        scans = list(scaninfo[0].keys())
        scantimes = []
        primaryIntentLetters = []
        for scan in scans:
            intents = scaninfo[0][scan]['intent']
            primaryIntent = getPrimaryIntent(intents)
            primaryIntentLetters.append(primaryIntentLetter(primaryIntent))
            scantimes.append(scaninfo[0][scan]['startmjd']*86400)
        startMJDSec = getObservationStartDateFromASDM(dataset)[1]
        stopMJDSec = getObservationEndDateFromASDM(dataset)[1]
    pngs = []
    for antenna in antennaIDs:
      for pol in pols:
        antennaName = antennaNames[antenna]
        ifproc_time, ifproc_dB, ifswitchLSB_time, ifswitchLSB_dB, ifswitchUSB_time, ifswitchUSB_dB = readAttenuatorSettings(dataset, antenna, pol, outpath)
        pb.close('all')
        # Top panel is IF Processor
        pb.figure(1,figsize=figsize)
        desc = pb.subplot(211)
#        pb.hold(True)  # not available in CASA6, but also not needed 
        col = ['k', 'b', 'r', 'g']
        idx1 = np.where(ifproc_time >= startMJDSec)[0]
        idx2 = np.where(ifproc_time <= stopMJDSec)[0]
        idx = np.intersect1d(idx1,idx2)
        timeStamps = pb.date2num(mjdSecondsListToDateTime(ifproc_time[idx]))
        for bb in range(4):
            pb.plot_date(timeStamps, ifproc_dB[bb][idx], '%s%s-' % (col[bb],marker), mec=col[bb], mfc=col[bb]) 
        pb.ylabel('IF Processor (dB)')
        pb.title(os.path.basename(dataset) + ' ' + antennaName + ' pol ' + str(pol)+' (dotted: scan start times)')
        desc.xaxis.set_major_formatter(matplotlib.dates.DateFormatter('%H:%M'))
        desc.xaxis.set_major_locator(matplotlib.dates.MinuteLocator(byminute=range(0,60,20)))
        desc.xaxis.set_minor_locator(matplotlib.dates.MinuteLocator(byminute=range(0,60,5)))
        desc.fmt_xdata = matplotlib.dates.DateFormatter('%H:%M')
        pb.xlabel('UT on '+yyyymmdd+' (BB0:black, 1:blue, 2:red, 3:green)')
        timeStamps = pb.date2num(mjdSecondsListToDateTime(scantimes))
        ylims = pb.ylim()
        pb.ylim([ylims[0]-0.5, ylims[1]+0.5])
        ylims = pb.ylim()
        yrange = ylims[1]-ylims[0]
        for i, timestamp in enumerate(timeStamps):
            if (i==0):
                t = 'scan %d' % scans[i]
                ha = 'center'
            else:
                t = str(scans[i])
                ha = 'left'
            pb.text(timestamp, ylims[1]-0.01-0.05*yrange*(1+i%2), t, size=8, ha=ha)
            pb.text(timestamp, ylims[0]+0.01+0.05*yrange*(1+i%2), primaryIntentLetters[i], size=intentLetterSize, ha=ha)
            pb.plot_date([timestamp,timestamp], pb.ylim(), 'k:')
        for i,primaryIntent in enumerate(listOfPrimaryIntents()):
            pb.text(1.01, 1-i*0.1, primaryIntentLetter(primaryIntent)+': '+primaryIntent,size=intentLetterSize,ha='left',transform=desc.transAxes)

        # Bottom panel is IF Switch
        desc = pb.subplot(212)
        idx1 = np.where(ifswitchLSB_time >= startMJDSec)[0]
        idx2 = np.where(ifswitchLSB_time <= stopMJDSec)[0]
        idx = np.intersect1d(idx1,idx2)
        timeStampsLSB = pb.date2num(mjdSecondsListToDateTime(ifswitchLSB_time[idx]))
        pb.plot_date(timeStampsLSB, ifswitchLSB_dB[idx], '%s%s-' % (marker,col[0]), mec=col[0], mfc=col[1]) 

        idx1 = np.where(ifswitchUSB_time >= startMJDSec)[0]
        idx2 = np.where(ifswitchUSB_time <= stopMJDSec)[0]
        idx = np.intersect1d(idx1,idx2)
        timeStampsUSB = pb.date2num(mjdSecondsListToDateTime(ifswitchUSB_time[idx]))
        pb.plot_date(timeStampsUSB, ifswitchUSB_dB[idx], '%s%s-' % (marker,col[1]), mec=col[1], mfc=col[1]) 
        pb.ylabel('IF switch (dB)')
        desc.xaxis.set_major_formatter(matplotlib.dates.DateFormatter('%H:%M'))
        desc.xaxis.set_major_locator(matplotlib.dates.MinuteLocator(byminute=range(0,60,20)))
        desc.xaxis.set_minor_locator(matplotlib.dates.MinuteLocator(byminute=range(0,60,5)))
        desc.fmt_xdata = matplotlib.dates.DateFormatter('%H:%M')
        pb.xlabel('UT on '+yyyymmdd+' (LSB=black, USB=blue)')
        ylims = pb.ylim()
        pb.ylim([ylims[0]-0.5, ylims[1]+0.5])
        ylims = pb.ylim()
        yrange = ylims[1]-ylims[0]
        for i, timestamp in enumerate(timeStamps):
            if (i==0):
                t = 'scan %d' % scans[i]
                ha = 'center'
            else:
                t = str(scans[i])
                ha = 'left'
            pb.text(timestamp, ylims[1]-0.01-0.05*yrange*(1+i%2), t, size=8, ha=ha)
            pb.text(timestamp, ylims[0]+0.01+0.05*yrange*(1+i%2), primaryIntentLetters[i], size=intentLetterSize, ha=ha)
            pb.plot_date([timestamp,timestamp], pb.ylim(), 'k:')
        pb.draw()
        png = os.path.basename(dataset)+'.%s.pol%d.attenuators.png' % (antennaName,pol)
        pngs.append(png)
        pb.savefig(png) # , bbox_inches='tight') # this cuts off the intent labeling
        print("Wrote plot = ", png)
    if (buildPDF and (len(pngs) > 1)):
        pdf = os.path.basename(dataset)+'.attenuators.pdf'
        buildPdfFromPngs(pngs, pdf)

def readIFProcAttenuatorSettings(tmcfile, pol=0):
    """
    Parses an IF processor attenuator text file read from the TMCDB and returns
    an array of timestamps (MJDsec) and an array of attenuator values 
    (in dB, one per sideband) for the specified polarization and sideband.
    -Todd Hunter
    """
    loc = tmcfile.find('IFProc')+6
    if (loc > 0):
        tmcfile = tmcfile[:loc] + str(pol) + tmcfile[loc+1:]
        print("Using file = ", tmcfile)
    if (os.path.exists(tmcfile) == False):
        print("Could not open IF Proc TMC database file")
        return
    tmclines = open(tmcfile,'r').readlines()
    dateTimeStamp = []
    dB = []
    for line in tmclines:
        tokens = line.split()
        if (len(tokens) == 0):
            print("len(tokens) = %d" % (len(tokens)))
        dateTimeStamp.append(dateStringToMJDSec(tokens[0],verbose=False))
        dB.append([float(x) for x in tokens[1:]])
    return(np.array(dateTimeStamp), np.array(np.transpose(dB)))

def readIFSwitchAttenuatorSettings(tmcfile_ifswitch, pol=0, sideband=1):
    """
    Parses an IF switch attenuator text file read from the TMCDB and returns
    an array of timestamps (MJDsec) and an array of attenuator 
    values (in dB) for the specified polarization and sideband.
    -Todd Hunter
    """
    tmcfile = tmcfile_ifswitch[sideband]
    loc = tmcfile.find('CHANNEL')+7
    if (loc > 0):
        tmcfile = tmcfile[:loc] + str(pol) + str(sideband) + tmcfile[loc+2:]
        print("Using file = ", tmcfile)
    if (os.path.exists(tmcfile) == False):
        print("Could not open IF Switch TMC database file")
        return
    tmclines = open(tmcfile,'r').readlines()
    dateTimeStamp = []
    dB = []
    for line in tmclines:
        tokens = line.split()
        dateTimeStamp.append(dateStringToMJDSec(tokens[0],verbose=False))
        dB.append(float(tokens[1]))
    return(np.array(dateTimeStamp), np.array(dB))

def createTsysTable(vis, caltable='', partype='Float', caltype='B TSYS', 
                    singlechan=False, overwrite=True):
    """
    Creates a new blank Tsys table for a measurement set.
    caltable: if '', then append '.tsys' to ms, if that exists, append '.tsys_new'
    -Todd Hunter
    """
    mycb = createCasaTool(cbtool)
    mycb.open(vis, addcorr=False, addmodel=False) # default of addcorr=True which adds model and corrected columns
    if caltable == '':
        caltable = vis + '.tsys'
        if os.path.exists(caltable):
            caltable = vis + '.tsys_new_' + casaVersion
    if os.path.exists(caltable) and overwrite:
        print("Removing existing table: ", caltable)
        shutil.rmtree(caltable)
    print("Creating ", caltable)
    mycb.createcaltable(caltable, partype=partype, caltype=caltype, singlechan=singlechan)
    mycb.close()
    return caltable

def compareTsysSpectralWindowTableChanFreqsWithMS(caltable, spw, vis=None):
    """
    Checks the agreement between the spectral window frequencies in a caltable
    with its parent measurement set.
    """
    if vis is None:
        vis = getMeasurementSetFromCaltable(caltable)
    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)
    visfreqs = mymsmd.chanfreqs(spw)
    mymsmd.close()
    calfreqs = getChanFreqFromCaltable(caltable, spw) * 1e9
    difference = np.mean(visfreqs) - np.mean(calfreqs)
    chanwidth = visfreqs[1] - visfreqs[0]
    if abs(difference) < 0.01*chanwidth:
        print("difference = %.0f Hz (less than 1% of channel width, which is %.4f Hz)" % (difference,chanwidth))
    else:
        print("difference = %.0f Hz = %.3f channel width, which is %.4f Hz" % (difference, abs(difference/chanwidth),chanwidth))

class Atmcal:
    """
    This class examines the CAL_ATMOSPHERE scans in a measurement set and
    determines the time ranges associated with the sky, ambient, and hot load
    in each scan based on the data itself, and checks it for sanity against
    the subscan order specified in the STATE table.  It also has methods to
    independently compute and/or plot the Trx and Tsys, with an option to
    overlay the TelCal result.  It can also compute and apply the FDM 
    quantization correction.  For the Trec2 and Tsys solving code, I have 
    endeavored to use the same function names and variables as the original 
    Telcal C-code.
    The source code for Telcal can be found on machines at the OSF at $ACSROOT, e.g.
    red-osf.osf.alma.cl:/alma/ACS-current/ACSSW/Sources/Engines/src/AtmosphereScan.cpp
    -Todd Hunter
    """
    def __init__(self,vis,verbose=False,showSummary=False,readAttenuatorSettings=False,
                 decimalDigits=2, includeDate=False, restoreBackup=False, outpath='./',
                 includeSQLD=False, maxscans=0, warnIfNoLoadTemperatures=True, mymsmd='',
                 loadLOs=False, distinguishLoads=True):
        """
        loadLOs: only needed for generateHighResTsysTable
        """
        self.vis = vis[:]
        self.visBasename = os.path.basename(vis)
        # Remove the following line once everything works well in fixSyscalTable.
        if (os.path.exists(self.vis+'/SYSCAL.backup') and restoreBackup):
            print("Copying the SYSCAL backup  into place")
            os.system('rm -rf %s' % (self.vis+'/SYSCAL'))
            os.system('rsync -vau %s %s' % (self.vis+'/SYSCAL.backup/', self.vis+'/SYSCAL'))
        self.correctedColumnPresent = 'CORRECTED_DATA' in dataColumns(vis)
        startTime = timeUtilities.time()
        if (outpath == ''): outpath = './'
        if (outpath[-1] != '/'): outpath += '/'
        self.sidebands = [1,2]
        self.polarizations = [0,1]  # assume we always want to operate on X and Y
        self.verbose = verbose
        if (os.path.exists(vis) == False):
            print("Cannot find this measurement set.")
            return
        if (os.path.exists(vis+'/table.dat') == False):
            print("No table.dat.  This does not appear to be an ms.")
            return
        if (verbose): print("Dataset begins at %s." % (mjdsecToUT(getObservationStart(self.vis))))
        print("Analyzing the Atmospheric Cal scans and various metadata may take a minute...")
        if (verbose): print("Reading the STATE table")
        self.readStateTable()
        if (verbose): print("Finding the cal scans")
        self.needToCloseMymsmd = False
        if mymsmd == '':
            #print "Opening msmd..."
            self.mymsmd = createCasaTool(msmdtool)
            self.mymsmd.open(vis)
            self.needToCloseMymsmd = True
        else:
            self.mymsmd = mymsmd
        self.findCalScans(includeDate,includeSQLD,maxscans,verbose=verbose,
                          warnIfNoLoadTemperatures=warnIfNoLoadTemperatures)
        if len(self.scans) < 1:
            return
        self.calscandict = buildCalDataIdDictionary(self.vis, mymsmd=self.mymsmd)
        if distinguishLoads:
            if (verbose): print("Distinguishing the load data")
            self.distinguishLoadData(verbose)
        if (verbose): print("Finding scans in the SYSCAL table")
        self.findSyscalScans(maxscans, verbose=verbose)
        if (showSummary): self.printCalScans(decimalDigits, includeDate)
        self.scienceSpws = getScienceSpws(self.vis, mymsmd=self.mymsmd)
        if len(self.scienceSpws) == 0:
            self.scienceSpws = getScienceSpws(self.vis, mymsmd=self.mymsmd, intent='CALIBRATE_FLUX#ON_SOURCE')
            if len(self.scienceSpws) == 0:
                self.scienceSpws = getScienceSpws(self.vis, mymsmd=self.mymsmd, intent='CALIBRATE_AMPLI#ON_SOURCE')
        self.hanningSmoothed = {}
        self.onlineChannelAveraging = {}
        scienceSpwsList = [int(i) for i in self.scienceSpws.split(',')]
        # include science and Tsys windows since they will differ for BLC and old ACA datasets
        for spw in np.unique(scienceSpwsList + self.spws):  
            spw = int(spw)
            self.onlineChannelAveraging[spw] = onlineChannelAveraging(self.vis, spw, self.mymsmd)
            if self.onlineChannelAveraging[spw] == 1:
                self.hanningSmoothed[spw] = True
            else:
                self.hanningSmoothed[spw] = False
        self.initializeAttenuatorDictionaries()
        self.radec = {}
        if (readAttenuatorSettings):
            if (verbose): print("Reading the attenuator settings")
            if (self.readAttenuatorSettings(outpath=outpath) == False):
                print("Stopping")
                return
        else:
            if (verbose): print("Not reading the attenuator settings (to do this: set readAttenuatorSettings=True)")
        if loadLOs:
            self.lo1s = interpretLOs(self.vis, parentms=None, mymsmd=self.mymsmd)
        else:
            self.lo1s = None
        stopTime = timeUtilities.time()
        self.xxyy = {} 
        for spw in self.spws:
            # fills in self.xxyy dictionary: will be [9,12] for dual-linear
                                          # or [9,10,11,12] for full-pol linear
            self.findPolarizationProducts(spw)  
                                               
        print("Atmcal class initialization completed in %.0f seconds" % (stopTime-startTime))

    # The following six functions are from R. Amestica (Nov 2013)
    def afun(self, sigma1, sigma2):
        """
        Compute 'a' parameter as defined in memo 583, section 7.3. Input values must be 
        3 bits sigma levels.
        """
        s1 = 0.
        s2 = 0.
        for i in range(1, 4):
            s1 += np.exp(- (i ** 2) / (2 * double(sigma1) ** 2))
            s2 += np.exp(- (i ** 2) / (2 * double(sigma2) ** 2))
        return (np.pi * sigma1 * sigma2) / (2 * (1. + 2 * s1) * (1. + 2 * s2))
    
    def bfun(self, sigma1, sigma2):
        """
        Compute 'b' parameter as defined in memo 583, section 7.3. Input values must be 3 bits sigma levels.
        """ 
        return self.afun(sigma1, sigma2) * self.R0(np.sqrt(sigma1 * sigma2), 8) - sigma1 * sigma2
    
    def STAR2(self, x):
        """
        Just a helper artifact.
        """
        return x * x
    
    def sign(self, a, b):
        """
        Just a helper artifact.
        """
        if b < 0.0:
            return -np.fabs(a)
        return np.fabs(a)
    
    def R0(self, sigma, n):
        """
        Equation 4 in memo 583.
        """
    
        ret = 0.
        for k in range(1, n // 2 - 1 + 1):
            ret += k * erf(k / (np.sqrt(2) * sigma))
        return (n - 1) ** 2 - 8. * ret 
    
    def sigma(self, zerolag, n):
        """
        Inverse of equation 4 in memo 583.
        """
        x = 0.0
        SQRT_05_PI = pb.sqrt(2.0 / pi)
        M_SQRT2 = pb.sqrt(2.0)
        if n % 2 == 0:
            x = 1.
        itmax = 30
        tol = 1.0e-8
        for i in range(itmax):
            f = zerolag
            fp = 0.0
            if n % 2 == 1:
                for k in range((n - 1) // 2):
                    kd = np.double(k + 1)
                    f = f - (2.0 * kd - 1.0) * erfc((2.0 * kd - 1.0) * x / M_SQRT2)
                    fp = fp + SQRT_05_PI * self.STAR2(2.0 * kd - 1.0) * np.exp(-0.5 * self.STAR2((2.0 * kd - 1.0) * x));
            else:
                f = f - double(1)
                for k in range((n - 1) / 2):
                    kd = np.double(k + 1)
                    f = f - 8 * kd * erfc(kd * x / M_SQRT2);
                    fp = fp + 8.0 * self.STAR2(kd) * SQRT_05_PI * np.exp(-0.5 * self.STAR2(kd * x));
            deltax = -f / fp
            deltax = self.sign(1.0, deltax) * fmin(0.5, np.fabs(deltax));
            x += deltax;
            if n % 2 == 1:
                x = fmax(0, x);
            if np.fabs(deltax / x) < tol:
                break
        return 1. / x;
    # The preceeding six functions are from R. Amestica (Nov 2013)

    def correctVisibilities(self, fdmscan, tdmscan=None, tdm=None, dataFraction=[0.0,1.0], 
                            ignoreFlags=False, useGetSpectrum=True):
        """
        Corrects cross-correlation FDM data for quantization on the basis of TDM 
        auto-correlation on the two antennas.
        fdmscan: the scan number to correct in the current Atmcal instance
        tdmscan: the scan number to use as the total power (default=same as fdmscan, but
                     in different measurement set)
        tdm: the name of the measurement set to get the TDM data from, or the Atmcal
               instance created from this different measurement set
        dataFraction: the portion of the subscan to use
        ignoreFlags: set to True to ignore the flag column
        useGetSpectrum: set to False to use Visibility class to get TDM data
        Note: in the future, I should pass self.mymsmd to Visibility() to save time
        """
        warnings.simplefilter("ignore", np.ComplexWarning)
        if (tdm == None):
            tdm = self  # the TDM data are in the same MS as the FDM data
        else:
            # the TDM data are in a different MS from the FDM data
            if (type(tdm) == str):
                # the name of the TDM measurement set was passed as 'tdm'
                tdm = Atmcal(tdm)
        tdmv = Visibility(tdm.vis)
        fdmv = Visibility(self.vis)
        fdmv.autoSubtableQuery = False
        if (tdmscan == None): tdmscan = fdmscan
        print("tdm.spwsforscan_nonchanavg = ", tdm.spwsforscan_nonchanavg)
        if (tdmscan not in tdm.spwsforscan_nonchanavg):
            print("tdmscan=%d is not an Atmcal scan" % (tdmscan))
            return
        tdmspws = tdm.spwsforscan_nonchanavg[tdmscan]
        if (fdmscan not in self.spwsforallscans_nonchanavg):
            print("fdmscan=%d is not an Atmcal scan" % (fdmscan))
        fdmspws = self.spwsforallscans_nonchanavg[fdmscan]
        nBaselines = len(self.antennas)*(len(self.antennas)-1)/2
        for spw in range(len(tdmspws)):
            baseline = 0
            for ant1 in self.antennas[:-1]:
                for ant2 in range(ant1+1, len(self.antennas)):
                    baseline += 1
                    print("Applying TDM spw %d to FDM spw %d for baseline %02d-%02d (%d/%d)" % (tdmspws[spw], fdmspws[spw], ant1, ant2, baseline, nBaselines))
                    tdm1spectrum = []
                    for pol in range(2):
                        if (useGetSpectrum):
                            tdm1spectrum.append(tdm.getSpectrum(tdmscan,tdmspws[spw],pol,'sky',ant1,dataFraction,ignoreFlags=ignoreFlags))
                        else:
                            tdmv.setAntennaPair(ant1,ant1)
                            tdmv.setSpwID(tdmspws[spw])
                            tdmv.setScan(tdmscan)
                            tdm1spectrum.append(np.mean(np.abs(tdmv.specData)[pol],axis=1))
                    tdm2spectrum = []
                    for pol in range(2):
                        if (useGetSpectrum):
                            tdm2spectrum.append(tdm.getSpectrum(tdmscan,tdmspws[spw],pol,'sky',ant2,dataFraction,ignoreFlags=ignoreFlags))
                        else:
                            tdmv.setAntennaPair(ant2,ant2)
                            tdm2spectrum.append(np.mean(np.abs(tdmv.specData)[pol],axis=1))
                    fdmv.setAntennaPair(ant1,ant2)
                    fdmv.setSpwID(fdmspws[spw])
                    fdmv.setScan(fdmscan)
                    fdmv.makeSubtableQuery()
                    fdmv.makeSubtableForWriting()
                    fdmv.getSpectralData()
                    fdmSpectra = fdmv.specData

                    # need to get individual integrations, not the average, so cannot use getSpectrum
                    nSpectra = len(fdmSpectra[0][0])
#                    print "Got %d spectra, shape = %s" % (nSpectra, np.shape(fdmv.specData))
                    npol = len(fdmv.specData)
                    for i in np.arange(nSpectra):
                        fdmSpectra = []
                        for pol in range(npol):
                            fdmSpectra.append([])
                            fdmSpectrum = fdmv.specData[pol][:][i]
                            fdmSpectra[pol] = self.applyCorrectionToFDMSpectrum(fdmSpectrum, tdm1spectrum[pol],
                                                                                tdm2spectrum[pol])
                        fdmv.putSpectralData(fdmSpectra, i)
        fdmv.subtable.close()
        fdmv.mytb.close() # write out the changes

    def applyCorrectionToFDMSpectrum(self, spectrum, TDM_X, TDM_Y, 
                                     useZeroLagsBinaryAttachment=False):
        """
        spectrum: the FDM spectrum to be corrected
        TDM_X: TDM spectrum for antenna X (can also be a single-valued SQLD "spectrum")
        TDM_Y: TDM spectrum for antenna Y (can also be a single-valued SQLD "spectrum")
        """
#        print "applyCorrectionToFDMSpectrum(): np.shape(spectrum) = ", np.shape(spectrum)
        if (useZeroLagsBinaryAttachment):
            print("Not yet implemented")
            sigma4_X = self.sigma(R4_X0, 4)
            sigma4_Y = self.sigma(R4_Y0, 4)
        else:
            sigma4_X = pb.sqrt(np.mean(TDM_X))
            sigma4_Y = pb.sqrt(np.mean(TDM_Y))
        sigma8_X = 2*sigma4_X
        sigma8_Y = 2*sigma4_Y
        a = self.afun(sigma8_X, sigma8_Y)
        b = self.bfun(sigma8_X, sigma8_Y)
        print("corrected=spectrum*a - b  where  a=%f, b=%f" % (a,b))
        if (type(spectrum[0]) == np.complex128):
            correctedSpectrum = []
            for chan in spectrum:
                correctedSpectrum.append(np.complex(np.real(spectrum[chan])*a - b, np.imag(spectrum[chan])))
        else:
            correctedSpectrum = spectrum*a - b
        return(correctedSpectrum)
        
    def initializeAttenuatorDictionaries(self):
        self.IFProc = {}
        self.IFSwitch = {}
        self.IFProcMin = {}
        self.IFProcMax = {}
        self.IFSwitchMin = {}
        self.IFSwitchMax = {}
        for a in self.antennas:
            self.IFProc[a] = {}
            self.IFSwitch[a] = {}
            self.IFProcMin[a] = {}
            self.IFProcMax[a] = {}
            self.IFSwitchMin[a] = {}
            self.IFSwitchMax[a] = {}
            for pol in self.polarizations:
                self.IFProc[a][pol] = None
                self.IFSwitch[a][pol] = {}
                self.IFSwitchMin[a][pol] = {}
                self.IFSwitchMax[a][pol] = {}
                for sb in self.sidebands:
                    self.IFSwitch[a][pol][sb] = None

    def fixSyscalTable(self, scan='', dataFraction=[0.0,1.0], ignoreFlags=False, spws='', clear=False):
        """
        Copies the existing SYSCAL table and appends entries calculated offline in casa
        for the specified scan.
        scan: which scan number(s) to process (default='' which means all)
              (integer, integer list, or comma-delimited string)
        dataFraction: which part of each load scan to use in the calculations
        ignoreFlags: if True, then ignore flags on the autocorrelation data
        spws: which spw(s) to process (default='' which means all)
        clear: if True, then remove all rows before appending new rows
               Select this option if you are trying to regenerate the table.
        """
        if (scan == None or scan == '' or scan == []):
            scan = self.scans
        else:
            if (self.unrecognizedScan(scan)): return
        startTime = timeUtilities.time()
        originalName = self.vis+'/SYSCAL.old'
        os.system('cp -r %s %s' % (self.vis+'/SYSCAL', originalName))
        calscandict = buildCalDataIdDictionary(self.vis)
        mytb = createCasaTool(tbtool)
        mytb.open(self.vis+'/SYSCAL', nomodify=False)
        nrows = len(mytb.getcol('ANTENNA_ID'))
        originalRows = nrows
        if (spws is not None and spws != '' and spws != []):
            if (type(spws) != list and type(spws) != np.ndarray):
                if (type(spws) == str):
                    spws = [int(i) for i in spws.split(',')]
                else:
                    spws = list([spws])
            restrictSpws = True
        else:
            restrictSpws = False
                    
        if (type(scan) != list and type(scan) != np.ndarray):
            if (type(scan) == str):
                scan = [int(i) for i in scan.split(',')]
            else:
                scan = list([scan])
        # Copy the list of scans, so that we can re-use the existing variable name as a scalar
        myscans = scan[:]
        if (clear):
            print("Clearing out all rows")
            mytb.removerows(range(nrows))
            nrows = 0
        for iscan,scan in enumerate(myscans):
          for antenna in self.antennas:
            if (not restrictSpws):
                spws = self.spwsforscan_nonchanavg[scan]
            for spw in spws:
                if (restrictSpws):
                    if (spw not in self.spwsforscan_nonchanavg[scan]): continue
                print("-------- Working on scan=%d (%d of %d)  antenna=%s (%d/%d), spw=%d (%d/%d) ----------" % (scan, iscan+1, len(myscans), self.antennaNames[antenna],antenna+1,len(self.antennas),spw,list(spws).index(spw)+1,len(spws)))
                mytb.addrows(1)
                mytb.putcell('ANTENNA_ID',nrows,antenna)
                mytb.putcell('SPECTRAL_WINDOW_ID',nrows,spw)
                mytb.putcell('FEED_ID',nrows,0)
                mytb.putcell('TIME',nrows,self.meantime[scan]+0.5*self.interval[scan])
                mytb.putcell('INTERVAL',nrows,self.interval[scan])
                result_pol0 = self.computeTsys(scan,antenna,0,spw,altscan='auto',
                                               dataFraction=dataFraction,
                                               ignoreFlags=ignoreFlags,calscandict=calscandict)
                result_pol1 = self.computeTsys(scan,antenna,1,spw,altscan='auto',
                                               dataFraction=dataFraction,
                                               ignoreFlags=ignoreFlags,calscandict=calscandict)
                if (result_pol0 is not None and result_pol1 is not None):
                    tsys0, freqHz, trx0, tsky0, tcal0 = result_pol0
                    tsys1, freqHz, trx1, tsky1, tcal1 = result_pol1
                    mytb.putcell('TCAL_SPECTRUM',nrows,np.array([tcal0,tcal1]))
                    mytb.putcell('TRX_SPECTRUM',nrows,np.array([trx0,trx1]))
                    mytb.putcell('TSKY_SPECTRUM',nrows,np.array([tsky0,tsky1]))
                    mytb.putcell('TSYS_SPECTRUM',nrows,np.array([tsys0,tsys1]))
#                    flags = np.array([np.array(tsys0)*0, np.array(tsys1)*0])
#                    idx = np.where(np.array([tsys0,tsys1]) <= 0)
#                    flags[idx] = 1
#                    print "Flagged %d/%d channels for negative Tsys" % (len(idx), len(tsys0))
#                    mytb.putcell('FLAG', nrows, flags)
                else:
                    print("Stopping due to missing data.  Try removing the online flags.")
                    mytb.close()
                    return
                mytb.putcell('TANT_SPECTRUM',nrows,np.array([],np.float))
                mytb.putcell('TANT_TSYS_SPECTRUM',nrows,np.array([],np.float))
                # The following values are what all normal SYSCAL tables contain.
                mytb.putcell('TCAL_FLAG',nrows,1)
                mytb.putcell('TRX_FLAG',nrows,1)
                mytb.putcell('TSKY_FLAG',nrows,1)
                mytb.putcell('TSYS_FLAG',nrows,1)
                mytb.putcell('TANT_FLAG',nrows,0)
                mytb.putcell('TANT_TSYS_FLAG',nrows,0)
                nrows += 1
        mytb.addreadmeline('Table has been modified by au.Atmcal().fixSyscalTable')
        mytb.close()
        print("Added %d rows.  The original table was moved to %s" % (nrows-originalRows,os.path.basename(originalName)))
        stopTime = timeUtilities.time()
        print("Task completed in %.0f seconds" % (stopTime-startTime))
        
    def nearestCalScan(self, mjdsec):
        mindiff=1e38
        myscan = None
        for scan in self.scans:
            mydiff = abs(self.meantime[scan]-mjdsec)
            if (mydiff < mindiff):
                mindiff = mydiff
                myscan = scan
        return myscan
    
    def findSyscalScans(self, maxscans=0, verbose=False):
        mytb = createCasaTool(tbtool)
        mytb.open(self.vis+'/SYSCAL')
        times = mytb.getcol('TIME')
        interval = mytb.getcol('INTERVAL')
        times -= 0.5*interval
#        print "times-0.5*interval: min=%f=%s max=%f=%s" % (np.min(times), mjdsecToUTHMS(np.min(times)), np.max(times), mjdsecToUTHMS(np.max(times)))
#        print "interval: min=%f max=%f" % (np.min(interval), np.max(interval))
        mytb.close()
        times = np.unique(times)
        scans = []
        timestamps = []
        for t in times:
            scan = self.nearestCalScan(t)
            if scan not in scans:
                scans.append(scan)
                timestamps.append(mjdsecToUTHMS(t))
        print("scans = %s" % (str(scans)))
        print("times = %s" % (str(timestamps)))
        self.syscalScans = scans
        if (maxscans > 0):
            self.syscalScans = scans[:maxscans]
        missingScans = np.setdiff1d(self.scans, scans)
        if (len(missingScans) > 0):
            print("There are %d atmcal scans missing from the SYSCAL table: %s" % (len(missingScans),str(missingScans)))
            spwString = ''
            for scan in missingScans:
                spwString += str(np.setdiff1d(np.setdiff1d(self.mymsmd.spwsforscan(scan),self.mymsmd.wvrspws()),
                                              self.mymsmd.chanavgspws()))
            print("They correspond to spws: %s" % (spwString))
        
    def getScans(self):
        return(self.scans)
            
    def distinguishLoadData(self, verbose=False):
        myms = createCasaTool(mstool)
        myms.open(self.vis)
        # Just use the first scan and first spw and first antenna
        antenna = 0
        datadescid = self.datadescids[self.spws[0]]
        if (verbose):
            print("Using datadescid=%d for spw=%d" % (datadescid, self.spws[0]))
            print("ms.selectinit(datadescid=%d)" % (datadescid))
        myms.selectinit(datadescid=datadescid)
        scan = sorted(self.timestamps.keys())[0]
        mytimestamps = self.timestamps[scan][self.loadsubscans[0]]
        if verbose:
            print("self.loadsubscans[0] = ", self.loadsubscans[0])
            print("Timerange = %s" % (mjdsecToTimerange(self.timestamps[scan][self.loadsubscans[0]])))
            print("Timerange = ", self.timestamps[scan][self.loadsubscans[0]])
        myms.select({'time':self.timestamps[scan][self.loadsubscans[0]], 'antenna1':antenna, 'antenna2':antenna})
        data0 = myms.getdata(['amplitude'])['amplitude']
        if verbose: print("data0[0][0] = ", data0[0][0])

        myms.selectinit(reset=True)  # required for CASA 5.3, (see CAS-11088)
        myms.selectinit(datadescid=datadescid)
        mytimestamps = self.timestamps[scan][self.loadsubscans[1]]
        if verbose:
            print("self.loadsubscans[1] = ", self.loadsubscans[1])
            print("Timerange = %s" % (mjdsecToTimerange(mytimestamps)))
            print("Timerange = ", mytimestamps)
            print("Calling myms.select({'time': %s, 'antenna1': %d, 'antenna2':%d})" % (mytimestamps,antenna,antenna))
        myms.select({'time': mytimestamps, 'antenna1':antenna, 'antenna2':antenna})
        data1 = myms.getdata(['amplitude'])['amplitude']
        if verbose: print("data1[0][0] = ", data1[0][0])
        Yfactor = []
        
        for pol in range(len(data0)):
            if (np.median(data0[pol]) < np.median(data1[pol])):
                Yfactor.append(np.median(data1[pol])/np.median(data0[pol]))
                self.target[self.loadsubscans[1]] = 'hot'
                self.target[self.loadsubscans[0]] = 'amb'
            else:
                Yfactor.append(np.median(data0[pol])/np.median(data1[pol]))
                self.target[self.loadsubscans[0]] = 'hot'
                self.target[self.loadsubscans[1]] = 'amb'
        self.targetInverse[self.target[self.loadsubscans[0]]] = self.loadsubscans[0]
        self.targetInverse[self.target[self.loadsubscans[1]]] = self.loadsubscans[1]
        if (verbose):
            print("Information from scan %d:" % (scan))
            print("Y-factors for pol %s: %s" % (range(len(data0)), str(Yfactor)))
        self.yfactor = {}
        self.yfactor[scan] = Yfactor  # only holds the first scan, for now
        if (len(Yfactor) > 1):
            if ((Yfactor[0] < 1 and Yfactor[1] > 1) or (Yfactor[0] > 1 and Yfactor[1] < 1)):
                print("The two polarizations do not agree on the order of the subscans (ambient vs. hot load)")
            else:
                loadlist = ''
                durationlist = ''
                totalDuration = 0
                for t in range(len(self.sub_scan_unique)):
                    if (t>0):
                        loadlist += ','
                        durationlist += ','
                    loadlist += self.target[self.sub_scan_unique[t]]
#                    print "self.timestamps = %s" % (self.timestamps.keys())
#                    print "self.timestamps[scan=%d] = %s" % (scan, self.timestamps[scan].keys())
#                    print "self.sub_scan_unique = %s" % (self.sub_scan_unique)
#                    print "self.sub_scan_unique[t=%d] = %s" % (t,self.sub_scan_unique[t])
                    myduration = self.timestamps[scan][self.sub_scan_unique[t]][1] - \
                                 self.timestamps[scan][self.sub_scan_unique[t]][0] + \
                                 self.integrationTime[scan]  # because need to add 1/2 integration time on each end
                    durationlist += '%.2f' % (myduration)
                    totalDuration += myduration
                if (verbose):
                    print("Standard order of the load subscans is confirmed (%s)" % (loadlist))
                    print("Subscan durations: %ss; total=%.2fs; integration=%.3fs" % (durationlist,totalDuration,self.integrationTime[scan]))
        myms.close()
        
    def readStateTable(self):
        mytb = createCasaTool(tbtool)
        mytb.open('%s/STATE' % self.vis)
        sub_scan = mytb.getcol('SUB_SCAN')
        obs_mode = mytb.getcol('OBS_MODE')
        sig = mytb.getcol('SIG')
        mytb.close()
        calatm_rows = []
        self.stateID_off_source = []
        for row in range(len(obs_mode)):
            om = obs_mode[row]
            if (om.find('CALIBRATE_ATMOSPHERE') >= 0):
                calatm_rows.append(row)
                if (om.find('OFF_SOURCE') >= 0 and sig[row]==0):
                    self.stateID_off_source.append(row)
        self.sub_scan = sub_scan[calatm_rows]
        self.sub_scan_unique = np.unique(self.sub_scan)
        self.obs_mode = obs_mode[calatm_rows]
        self.loadsubscans = []
        self.refsubscan = None
        self.ambsubscan = None
        self.hotsubscan = None
        for i in range(len(self.obs_mode)):
            if (self.obs_mode[i].find('OFF_SOURCE')>=0):
                self.skysubscan = self.sub_scan[i]
            elif (self.obs_mode[i].find('REFERENCE')>=0):
                self.refsubscan = self.sub_scan[i]
            elif (self.obs_mode[i].find('HOT')>=0):
                self.loadsubscans.append(self.sub_scan[i])
                self.hotsubscan = self.sub_scan[i]
            elif (self.obs_mode[i].find('AMBIENT')>=0):
                self.loadsubscans.append(self.sub_scan[i])
                self.ambsubscan = self.sub_scan[i]
            else:
                self.loadsubscans.append(self.sub_scan[i])
        self.loadsubscans = np.unique(self.loadsubscans)
        self.nsubscans = len(self.sub_scan_unique)
        self.target = dict.fromkeys(self.sub_scan_unique)
        self.targetInverse = {}
        for i in self.sub_scan_unique:
            if (i == self.hotsubscan):
                self.target[i] = 'hot'
            elif (i == self.ambsubscan):
                self.target[i] = 'amb'
            elif (i in self.loadsubscans):
                self.target[i] = 'load'
            elif (i==self.refsubscan):
                self.target[i] = 'ref'
                self.targetInverse['ref'] = i
            else:
                self.target[i] = 'sky'
                self.targetInverse['sky'] = i

    def setLoadTemperatures(self, ambient=None, hot=None):
        """
        Set a new value of ambient and hot load temperatures for all scans on all antennas.
        """
        if (ambient == None and hot == None):
            print("No changes requested.")
            return
        for antenna in list(self.loadTemperatures.keys()):
            for scan in list(self.loadTemperatures[antenna].keys()):
                if (ambient is not None):
                    self.loadTemperatures[antenna][scan]['amb'] = ambient
                if (hot is not None):
                    self.loadTemperatures[antenna][scan]['hot'] = hot
    
    def deleteScan(self, scan):
        for i in [self.timestamps, self.timestampsString, self.timerange, self.timerangeString, self.meantime,
                  self.meantimestring, self.spwsforscan, self.spwsforscan_nonchanavg]:
            if scan in i: del i[scan]
        self.scans = list(self.scans)
        self.scans.remove(scan)
        self.scans = np.array(self.scans)
        
    def findCalScans(self, includeDate, includeSQLD=False, maxscans=0, verbose=False, 
                     warnIfNoLoadTemperatures=True):
        self.loadTemperatures = getLoadTemperatures(self.vis, mymsmd=self.mymsmd, warnIfNoLoadTemperatures=warnIfNoLoadTemperatures)
        self.antennas = range(self.mymsmd.nantennas())
        self.antennaNames = np.array(self.mymsmd.antennanames(self.antennas))
        self.basebands = getBasebands(self.mymsmd)
        self.observationStart = getObservationStart(self.vis)
        myscans = self.mymsmd.scannumbers()
        if len(myscans) < 1:
            print("No scans found.")
        self.observationStop = np.max(self.mymsmd.timesforscan(int(myscans[-1])))
        self.telescopeName = self.mymsmd.observatorynames()[0]
        self.yyyymmdd = mjdSecondsToMJDandUT(self.mymsmd.timesforscans(int(myscans[0]))[0])[1].split()[0]
        self.intentsforscan = {}
        for scan in self.mymsmd.scannumbers():
            self.intentsforscan[scan] = self.mymsmd.intentsforscan(scan)
        self.intents = self.mymsmd.intents()
        calIntents = ['CALIBRATE_ATMOSPHERE#ON_SOURCE', 'CALIBRATE_ATMOSPHERE#OFF_SOURCE',
                      'CALIBRATE_ATMOSPHERE#HOT']
        self.scans = []
        for calIntent in calIntents:
            if (calIntent in self.intents):
                self.scans = self.mymsmd.scansforintent(calIntent)
                break
        if (len(self.scans) == 0):
            print("No atm cal intents found.")
            return
        if (len(self.scans) == self.scans[-1]):
            self.attenuatorTest = True
        else:
            self.attenuatorTest = False
        self.spws = spwsforintent_nonwvr_nonchanavg(self.mymsmd, calIntent, includeSQLD)
        if verbose: print("Found spws: ", self.spws)
        self.nchan = self.mymsmd.nchan(self.spws[0]) # assume all are the same
        self.datadescids = {}
        self.meanfreq = {}
        self.sidebandsforspw = {}
        self.chanfreqs = {}
        for spw in self.spws:
            self.datadescids[spw]=spw  # assume this is true for now, but may not be someday
            self.meanfreq[spw] = self.mymsmd.meanfreq(spw)
            self.chanfreqs[spw] = self.mymsmd.chanfreqs(spw)
            self.sidebandsforspw[spw] = sidebandToNetSideband(self.mymsmd.sideband(spw))
        self.timestamps = {}
        self.timestampsString = {}
        self.timerangeString = {}
        self.timerange = {}
        self.meantime = {}
        self.interval = {}
        self.meantimestring = {}
        self.integrationTime = {}
        self.spwsforscan = {}
        self.spwsforscan_nonchanavg = {}
        self.spwsforallscans_nonchanavg = spwsForScan(self.mymsmd)
        if (maxscans > 0):
            self.scans = self.scans[:maxscans]
        for scan in self.scans:
            if False:
                t = self.mymsmd.timesforscan(scan)
                # Round to nearest tenth of a second, to avoid long processing times for 0.016-sec SQLD data 
                t = np.unique(np.round(t,1)) # July 31, 2015
            else:
                t = None # June 6, 2017
            result = computeDurationOfScan(scan, t, vis=self.vis, returnSubscanTimes=True, 
                                           verbose=self.verbose,includeDate=includeDate, mymsmd=self.mymsmd)
            if (result[1] != len(self.sub_scan_unique)):
                print("Found %d subscans in scan %d instead of %d!" % (result[1], scan, len(self.sub_scan_unique)))
                line = '     '
                for mysubscan in range(result[1]):
                    line += result[3][mysubscan+1] + ', '
                print(line)
            else:
                if verbose: print("findCalScans(): Working on scan %d" % (scan))
            self.timestamps[scan] = result[2]
            self.timestampsString[scan] = result[3]
            self.timerange[scan] = [result[2][1][0], result[2][len(result[2])][1]]
            self.timerangeString[scan] = [mjdsecToUTHMS(result[2][1][0]), mjdsecToUTHMS(result[2][len(result[2])][1])]
            self.meantime[scan] = np.mean(self.timerange[scan])
            self.meantimestring[scan] = mjdsecToUTHMS(self.meantime[scan])
            self.spwsforscan[scan] = np.setdiff1d(self.mymsmd.spwsforscan(scan),self.mymsmd.wvrspws())  # get rid of WVR spws
            self.spwsforscan_nonchanavg[scan] = np.setdiff1d(self.spwsforscan[scan],self.mymsmd.chanavgspws())
            if verbose: 
                print("chanavgspws = ", self.mymsmd.chanavgspws())
                print("wvrspws = ", self.mymsmd.wvrspws())
                print("self.spwsforscan[%d] = " % (scan), self.spwsforscan[scan])
                print("self.spwsforscan_nonchanavg[%d] = " % (scan), self.spwsforscan_nonchanavg[scan])
            if (len(self.spwsforscan_nonchanavg[scan]) == 0):
                print("Removing scan %d because it appears to be missing data (PRTSPR-21604)." % scan)
                self.deleteScan(scan)
                continue
            if verbose: 
                print("Running exposuretime with spwid=", self.spwsforscan_nonchanavg[scan][0])
                startTime = timeUtilities.time()
            exposureTimeDict = self.mymsmd.exposuretime(scan=scan, spwid=self.spwsforscan_nonchanavg[scan][0])
            if verbose: 
                stopTime = timeUtilities.time()
                print("done after %f sec" % (stopTime-startTime))
            self.integrationTime[scan] = exposureTimeDict['value']
#            This was the original, slower method
#            self.integrationTime[scan] = getIntegrationTime(self.vis, scan=scan, intent='CALIBRATE_ATMOSPHERE#ON_SOURCE', verbose=False)
        for scan in self.scans:
            if (scan < self.scans[-1]):
                nextscan = self.scans[list(self.scans).index(scan)+1]
                self.interval[scan] = self.meantime[nextscan]-self.meantime[scan]
            else:
                self.interval[scan] = 3600

    def getSubscanTimes(self, scan, stringFormat=False, decimalDigits=2, includeDate=False):
        # currently not used
        timestamps = {}
        timestamp = self.timestamps[scan]
        for subscan in list(timestamp.keys()):
            if (stringFormat):
                timestamps[self.target[subscan]] = mjdsecToTimerange(timestamp[subscan][0],
                                                                     timestamp[subscan][1],
                                                                     decimalDigits,includeDate)
            else:
                timestamps[self.target[subscan]] = timestamp[subscan]
        return(timestamps)
    
    def printCalScans(self, decimalDigits=1, includeDate=False):
        for scan in sorted(self.timestamps.keys()):
            timestamp = self.timestamps[scan]
            for subscan in list(timestamp.keys()):
                print('%2d: %d: %4s %s [%.2f,%.2f]' % (scan, subscan, self.target[subscan],
                                              mjdsecToTimerange(timestamp[subscan][0],
                                                                timestamp[subscan][1],
                                                                decimalDigits,
                                                                includeDate),
                                              timestamp[subscan][0], timestamp[subscan][1]
                                              ))
            print("")

    def findPolarizationProducts(self, spw):
        """
        Translates the CORR_TYPE value from the POLARIZATION table that
        corresponds to the POLARIZATION_ID in the DATA_DESCRIPTION table for a specified spw.
        Returns: a list of 1, 2 or 4 values,  selected from 9,10,11,12 (for linear 
                 polarization products)
        """
        mytb = createCasaTool(tbtool)
        mytb.open(self.vis +'/DATA_DESCRIPTION')
        polId = mytb.getcol('POLARIZATION_ID') # will have length = number of spws
        # each integer value of polId corresponds to a set of correlations
        # so for an ACA dataset with dualpol pointing and 4-pol observations, 
        #      0 == XX,YY; 1 = XX,XY,YX,YY
        # for a 12m dataset with dualpol only, then 0 = WVR, 1=XX,YY
        mytb.close()
        mytb.open(self.vis +'/POLARIZATION')
        corrType = mytb.getcell('CORR_TYPE',polId[spw])
        mytb.close()
        xxpol = -1
        yypol = -1
        for i,corr in enumerate(corrType):
            if (corr == 12):
                yypol = i
            if (corr == 9):
                xxpol = i
        self.xxyy[spw] = [xxpol, yypol]
        return(corrType)

    def getSpectrum(self, scan, spw, pol, target, antenna, 
                    dataFraction=[0.0,1.0],
                    median=False, ignoreFlags=False, antenna2=None):
        """
        Uses the ms tool to retrieve the average spectrum over all 
        integrations in the specified scan, for the subscan corresponding 
        to the target, and the specified spw, polarization and antenna.
        scan: integer
        spw: integer
        pol: integer
        target: 'hot', 'sky', amb'
        antenna: the antenna ID
        """
        scan = int(scan)
        spw = int(spw)
        pol = int(pol)
        mypol = pol
        if (spw not in self.xxyy):
            print("spw not found, did you run Atmcal(includeSQLD=True)?")
            return
        pol = self.xxyy[spw][pol]
#        print "Chose position %d for pol %d" % (pol, mypol)
        antenna = int(antenna)
        antennaName = self.antennaNames[antenna]
        dataColumnsDefined = dataColumns(self.vis)
        myms = createCasaTool(mstool)
        myms.open(self.vis)
        subscan = self.targetInverse[target]
        myms.selectinit(datadescid=self.datadescids[spw])
        # normally, one could just bass 'time':self.timestamps[scan][subscan]
        # but here I give the option to use only part of the subscan
        endtime = self.timestamps[scan][subscan][1]
        starttime = self.timestamps[scan][subscan][0]
        timerange = [starttime+(endtime-starttime)*dataFraction[0],
                     starttime+(endtime-starttime)*dataFraction[1]]
        # because need to add 1/2 integration time on each end
        duration = timerange[1]-timerange[0]+self.integrationTime[scan]  
        if (antenna2 == None):
            antenna2 = antenna
        myms.select({'time':timerange,'antenna1':antenna, 'antenna2':antenna2})
        if 'float_data'.upper() not in dataColumnsDefined:
            datadict = myms.getdata(['amplitude'])
            if ('amplitude' not in list(datadict.keys())):
                print("No amplitude in ms.getdata result: %s" % (str(list(datadict.keys()))))
                return
            data = datadict['amplitude']
            if len(data) == 0:
                print("No amplitude found.")
                return
        else:
            datadict = myms.getdata(['float_data'])
            data = datadict['float_data']
            if len(data) == 0:
                print("No float_data found.")
                return
        datamean = np.zeros(len(data[pol]))
        flag = myms.getdata(['flag'])['flag']
        myms.close()
        nflags = np.sum(flag[pol].flatten())
        line = ''
        if nflags > 0:
            line = "scan%d spw%d pol%d %s: %s flags: %d/%d" % (scan,spw,pol,antennaName, target, nflags,
                                                               len(flag[pol].flatten()))
        try:
            if (ignoreFlags):
                datamean = np.mean(data[pol], 1)
            else:
                weights = 1.0-flag[pol].astype(float)
                if np.sum(weights) == 0.:
                    line +=  " ignoring flags since all data are flagged"
                    if self.correctedColumnPresent:
                        line += " (probably since this is a calibrated dataset)"
                    datamean = np.mean(data[pol], 1)
                    #weights = flag[pol].astype(float)
                    # the following produces a MaskedIterator which cannot be later used with things like np.min
                    #datamean = np.ma.average(data[pol], 1, weights=weights)
                else:
                    # The following fails if the weights sum to zero
                    datamean = np.average(data[pol], 1, weights=weights)
        except:
            datamean = -1
            print("This subscan data (scan=%d, spw=%d, pol=%d, antenna=%d) is either missing or totally flagged (%s)." % (scan,spw,pol,antenna,self.vis))
        if len(line) > 0:
            print(line)
        return(datamean)
        
    def computeMeanSpectrum(self, spw, pol, target, antenna, dataFraction=[0.0,1.0], median=False):
        """
        antenna: the antenna ID
        """
        means = []
        myms = createCasaTool(mstool)
        myms.open(self.vis)
        subscan = self.targetInverse[target]
        for i in range(len(self.scans)):
            # produce a mean spectrum for each scan
            scan = self.scans[i]
            myms.selectinit(datadescid=self.datadescids[spw])
            endtime = self.timestamps[scan][subscan][1]
            starttime = self.timestamps[scan][subscan][0]
            timerange = [starttime+(endtime-starttime)*dataFraction[0], starttime+(endtime-starttime)*dataFraction[1]]
            duration = timerange[1]-timerange[0]
            myms.select({'time':timerange, 'antenna1':antenna, 'antenna2':antenna})
            data = myms.getdata(['amplitude'])['amplitude']
            if (median):
                datamean = np.median(data[pol], 1)
                means.append(datamean/np.median(datamean))
            else:
                datamean = np.mean(data[pol], 1)
                means.append(datamean/np.mean(datamean))
        myms.close()
        # compute the mean spectrum over all scans
        if (median):
            return(np.median(means,0),duration)
        else:
            return(np.mean(means,0),duration)

    def readIFProcAttenuatorSettings(self, tmcfile, pol=0):
        """
        Returns 2 lists: dateTimeStamp in MJD second, and attenuation in dB
        """
        loc = tmcfile.find('IFProc')+6
        if (loc > 0):
            tmcfile = tmcfile[:loc] + str(pol) + tmcfile[loc+1:]
            print("Using file = ", tmcfile)
        if (os.path.exists(tmcfile) == False):
            print("Could not open IF Proc TMC database file")
            return
        tmclines = open(tmcfile,'r').readlines()
        dateTimeStamp = []
        dB = []
        for line in tmclines:
            tokens = line.split()
            if (len(tokens) == 0):
                print("len(tokens) = %d" % (len(tokens)))
            dateTimeStamp.append(dateStringToMJDSec(tokens[0],verbose=False))
            dB.append([float(x) for x in tokens[1:]])
        return(dateTimeStamp, dB)

    def readIFSwitchAttenuatorSettings(self, tmcfile_ifswitch, pol=0, sideband=1):
        """
        Returns the timestamp and (single) value for the specified polarization and sideband.
        """
        tmcfile = tmcfile_ifswitch[sideband]
        loc = tmcfile.find('CHANNEL')+7
        if (loc > 0):
            tmcfile = tmcfile[:loc] + str(pol) + str(sideband) + tmcfile[loc+2:]
            print("Using file = ", tmcfile)
        if (os.path.exists(tmcfile) == False):
            print("Could not open IF Switch TMC database file")
            return
        tmclines = open(tmcfile,'r').readlines()
        dateTimeStamp = []
        dB = []
        for line in tmclines:
            tokens = line.split()
            dateTimeStamp.append(dateStringToMJDSec(tokens[0],verbose=False))
            dB.append(float(tokens[1]))
        return(dateTimeStamp, dB)

    def getAttenuatorSettings(self, antenna=0, pol=0, tmcfile_ifswitch=None, tmcfile_ifproc=None):
        if (self.IFProc[antenna][pol] == None or self.IFSwitch[antenna][pol][1] == None):
            self.readAttenuatorSettings(antenna,pol,tmcfile_ifswitch,tmcfile_ifproc)
        return(self.IFProc[antenna][pol], self.IFSwitch[antenna][pol])

    def plotAttenuatorSettings(self, antenna=0, pol=0, dateTimeStamp=None, dB=None):
        """
        Plot the IF Proc attenuator settings.
        """
        if (self.IFProc == {}):
            print("You must first readAttenuatorSettings()")
            return
        if (dateTimeStamp == None or dB == None):
            dateTimeStamp, dB = self.readIFProcAttenuatorSettings(self.tmcfile_ifproc, pol)
            idx0 = np.where(dateTimeStamp < self.observationStop)[0]
            idx1 = np.where(dateTimeStamp > self.observationStart)[0]
            indices = np.intersect1d(idx0,idx1)
            dateTimeStamp = np.array(dateTimeStamp)[indices]
            print("np.shape(dB) = ", np.shape(dB))
            print("np.shape(dB[0]) = ", np.shape(dB[0]))
            print("indices = ", indices)
            dB[0] = np.array(dB[0])[indices]
            print("np.shape(dB) = ", np.shape(dB))
        timeStamps = pb.date2num(mjdSecondsListToDateTime(list(dateTimeStamp)))
        for bb in range(1):
            print("bb=%d: len(timeStamps)=%d, len(dB[bb])=%d" % (bb, len(timeStamps), len(dB[bb])))
            pb.plot_date(timeStamps, dB[bb], color=overlayColors[bb])
#            pb.hold(True) # not needed
        pb.ylabel('dB')
        pb.xlabel('Universal time')
        pb.title('%s: Antenna %d, pol %d' % (os.path.basename(self.vis), antenna, pol))
        png = os.path.basename(self.vis) + '.ifproc.ant%d.pol%d.png' % (antenna,pol)
        pb.savefig(png)
        pb.draw()
        return(dateTimeStamp, dB)
                               
    def readAttenuatorSettings(self, antenna=0, pol=0, tmcfile_ifswitch=None, tmcfile_ifproc=None, outpath='./'):
        # Example: tmu.get_tmc_data('DA62','FrontEnd_IFSwitch','CHANNEL01_ATTENUATION','2013-07-23','2013-07-23')
        if (tmcfile_ifproc is None):
            localfile = tmu.retrieve_daily_tmc_data_file_name_only(self.antennaNames[antenna],'IFProc'+str(pol),
                                                                   'GAINS', self.yyyymmdd, outpath=outpath)
            if (os.path.exists(localfile)):
                tmcfile_ifproc = localfile
            else:
                try:
                    mydict=tmu.get_tmc_data(self.antennaNames[antenna],'IFProc'+str(pol),'GAINS', self.yyyymmdd, self.yyyymmdd, outpath=outpath)
                    tmcfile_ifproc = mydict['files'][0]
                    self.tmcfile_ifproc = tmcfile_ifproc
                except:
                    return False
        if (tmcfile_ifswitch is None):
            tmcfile_ifswitch = {}
            for sideband in self.sidebands:
                localfile = tmu.retrieve_daily_tmc_data_file_name_only(self.antennaNames[antenna],'FrontEnd_IFSwitch',
                                                                       'CHANNEL%d%d_ATTENUATION'%(pol,sideband), self.yyyymmdd, outpath=outpath)
                if (os.path.exists(localfile)):
                    tmcfile_ifswitch[sideband] = localfile
                else:
                    mydict = tmu.get_tmc_data(self.antennaNames[antenna],'FrontEnd_IFSwitch',
                                       'CHANNEL%d%d_ATTENUATION'%(pol,sideband), self.yyyymmdd, self.yyyymmdd, outpath=outpath)
                    tmcfile_ifswitch[sideband] = mydict['files'][0]
        else:
            print("tmcfile_ifswitch is not None = ", tmcfile_ifswitch)
            
        self.associateAttenuatorSettingsToScans(tmcfile_ifproc, tmcfile_ifswitch, antenna, pol)
        print("Range of attenuator settings for pol %d: IFproc=%4.1f-%4.1f, IFswitch:sb1=%4.1f-%4.1f, sb2=%4.1f-%4.1f" % (pol, self.IFProcMin[antenna][pol],
                                                                              self.IFProcMax[antenna][pol],
                                                                              self.IFSwitchMin[antenna][pol][1],
                                                                              self.IFSwitchMax[antenna][pol][1],
                                                                              self.IFSwitchMin[antenna][pol][2],
                                                                              self.IFSwitchMax[antenna][pol][2]
                                                                                             ))
        return(True)

    def associateAttenuatorSettingsToScans(self, tmcfile_ifproc,
                                           tmcfile_ifswitch=None,
                                           antenna=0, pol=0, debug=False,
                                           subscan=2, verbose=False):
        """
        Builds two dictionaries, keyed by scan number, and baseband number:
        (1) IFProcessor attenuator settings: a vector of 4 values (one for each baseband).
        (2) IFSwitch attenuator settings: a single value
        pol: 0 or 1
        """
        if (pol != 0 and pol != 1):
            print("Invalid pol, must be 0 or 1")
            return
        dateTimeStamp, dB = self.readIFProcAttenuatorSettings(tmcfile_ifproc, pol)
        self.IFProc[antenna][pol] = {}     # start with empty list of scans
        self.IFSwitch[antenna][pol][1] = {}
        self.IFSwitch[antenna][pol][2] = {}
        alldB = []
#        print "IF Proc TMC timestamps range from %f to %f" % (dateTimeStamp[0], dateTimeStamp[-1])
        for scan in self.scans:
            startTime = self.timestamps[scan][1][0]
            endTime = self.timestamps[scan][self.nsubscans][-1]
            # Look for the attenuator measurements after the start time of the first subscan.
            indices1 = np.where(dateTimeStamp > startTime+1)[0]
            # Look for the attenuator measurements before the end of the last subscan.
            indices2 = np.where(dateTimeStamp < endTime-2.5)[0]
            if (debug): print("startTime-endTime = %s-%s" % (mjdsecToUTHMS(startTime+1), mjdsecToUTHMS(endTime-2.5)))
            indices = np.intersect1d(indices1,indices2)
            if (len(indices) < 1):
                print("The IFProc TMC file (%s) does not contain data within scan %d (%s-%s)." % (tmcfile_ifproc,scan,mjdsecToUTHMS(startTime),mjdsecToUTHMS(endTime)))
                self.IFProc[antenna][pol][scan] = -1
            if (verbose):
                outline = "Found %d %s IF proc pol%d attenuator measurements during scan %2d.  " % (len(indices),self.antennaNames[antenna],pol,scan)
            else:
                outline = ''
            if (len(indices) > 1):
                for bb in range(4):
                    if (dB[indices[0]][bb] != dB[indices[-1]][bb]):
                        outline += "The IFProc pol%d:bb%d attenuator value changed from %4.1f to %4.1f during scan %d!" % (pol,bb+1,dB[indices[0]][bb], dB[indices[-1]][bb], scan)
                if (verbose==False and len(outline)>0):
                    print(outline)
                    for j in range(len(indices)):
                        print("start = %.1f    end = %.1f" % (startTime, endTime))
                        print("%s dB at %.1f (%.1f--%.1f sec)\n" % (str(dB[indices[j]]),
                                                                    dateTimeStamp[indices[j]],
                                                                    dateTimeStamp[indices[j]]-startTime,
                                                                    endTime-dateTimeStamp[indices[j]]
                                                                    ))
                else:
                    outline += "No inconsistencies seen."
            if (verbose):
                print(outline)
            if (len(indices) > 0):
                firstIndex = indices[0]
                self.IFProc[antenna][pol][scan] = dB[firstIndex]
                alldB += dB[firstIndex]
        if (len(alldB) < 1):
            self.IFProcMin[antenna][pol] = -1
            self.IFProcMax[antenna][pol] = -1
        else:
            self.IFProcMin[antenna][pol] = np.min(alldB)
            self.IFProcMax[antenna][pol] = np.max(alldB)
        # Build baseband keys
        for bb in range(4):
            self.IFProc[antenna][pol]['bb%d'%(bb+1)] = []
            for scan in self.scans:
                if (scan in self.IFProc[antenna][pol]):
                    if (self.IFProc[antenna][pol][scan] != -1):
                        self.IFProc[antenna][pol]['bb%d'%(bb+1)].append(self.IFProc[antenna][pol][scan][bb])
                    else:
                        print("Scan %d is not in the IFProc dictionary" % (scan))
                else:
                    print("Scan %d is not in the IFProc dictionary" % (scan))
        if (tmcfile_ifswitch is not None):
          for sideband in self.sidebands:
            dateTimeStamp, dB = self.readIFSwitchAttenuatorSettings(tmcfile_ifswitch, pol, sideband)
            alldB = []
            for scan in self.scans:
                startTime = self.timestamps[scan][1][0]
                endTime = self.timestamps[scan][self.nsubscans][-1]
                # Look for the attenuator measurements after the start time of the first subscan
                if (debug): print("startTime of scan %d = %f" % (scan,startTime))
                indices1 = np.where(dateTimeStamp > startTime+1)[0]
                # Look for the attenuator measurements before then end of the last subscan
                indices2 = np.where(dateTimeStamp < endTime-2.5)[0]
                indices = np.intersect1d(indices1,indices2)
                if (len(indices) < 1):
                    print("The IFSwitch TMC file does not contain data during scan %d (%s-%s)." % (scan,mjdsecToUTHMS(startTime+1), mjdsecToUTHMS(endTime-2.5)))
                    self.IFSwitchMin[antenna][pol][sideband] = -1
                    self.IFSwitchMax[antenna][pol][sideband] = -1
                if (verbose):
                    outline = "Found %d %s IF switch pol%d:sb%d attenuator measurements during scan %2d.  " % (len(indices),self.antennaNames[antenna],pol,sideband,scan)
                else:
                    outline = ''
                if (len(indices) > 1):
                    if (dB[indices[0]] != dB[indices[-1]]):
                        outline += "The attenuation changed from %.1f to %.1f during scan %d!" % (dB[indices[0]], dB[indices[-1]],scan)
                        if (verbose==False and len(outline)>0):
                            print(outline)
                            for j in range(len(indices)):
                                print("start = %.1f    end = %.1f" % (startTime, endTime))
                                print("%s dB at %.1f (%.1f--%.1f sec)\n" % (str(dB[indices[j]]),
                                                                    dateTimeStamp[indices[j]],
                                                                    dateTimeStamp[indices[j]]-startTime,
                                                                    endTime-dateTimeStamp[indices[j]]
                                                                    ))
                    else:
                        outline += "No inconsistencies seen."
                if (verbose): print(outline)
                if (len(indices) > 0):
                    firstIndex = indices[0]
                    self.IFSwitch[antenna][pol][sideband][scan] = dB[firstIndex]
                    alldB.append(dB[firstIndex])
            if (len(indices) > 0):
                self.IFSwitchMin[antenna][pol][sideband] = np.min(alldB)
                self.IFSwitchMax[antenna][pol][sideband] = np.max(alldB)
            # Build baseband keys
            self.IFSwitch[antenna][pol][sideband]['bb0'] = []
            for scan in self.scans:
                if (scan in self.IFSwitch[antenna][pol][sideband]):
                    self.IFSwitch[antenna][pol][sideband]['bb0'].append(self.IFSwitch[antenna][pol][sideband][scan])
                else:
                    print("Scan %d not in IFSwitch dictionary" % (scan))

    def computeBitRanges(self, delta=1.0):
        """
        returns a matrix:  len=64 levels, nBits entries for each
        and its transpose: len=nBits, 64 entries for each
        and a dictionary keyed by each possible attenuation value with the value being
           the number of bits that must be changed when the increment is delta from there.
        """
        a = np.arange(64.0)
        atten = a*0.5
        levels = [16,8,4,2,1,0.5]  # dB
        attenlevels = []  # will hold 0 or 1 for each of the bit levels
        for a in range(len(atten)):
            i = atten[a]
            levs = []
            att = i
            for l in levels: 
                if (i >= l):
                    levs.append(1)
                    i = i-l
                else:
                    levs.append(0)
            attenlevels.append(levs)  # 0->0dB, 1->0.5dB
        deltaBits = {}
        if (delta > 0):
            start = 0
            end = 32-delta
            inc = 0.5
        else:
            start = 32
            end = 0-delta
            inc = -0.5
        for a in np.arange(start,end,inc):
            # deltaBits = how many bits changed when going from the key dB
            #             to key+delta dB
            startBits = attenlevels[int(a*2)]
            endBits = attenlevels[int((a+delta)*2)]
            deltaBits[a] = abs(np.array(startBits)-np.array(endBits)).sum()
#            print "%4.1f dB: startBits=%s, endBits=%s, deltaBits=%d" % (a, startBits, endBits, deltaBits[a])
            
        bitstatus = np.transpose(attenlevels) # len=nBits, each entry=64 levels
        return(attenlevels, bitstatus, deltaBits)

    def plotCalScansAll(self, target=['sky','amb','hot'],
                        normalize='meanspectrum', plotfile=True,
                        startdBm=4.0, incrementdBm=-1.0, plotSlopes=True,
                        maxSlope=20, edge=5, xaxis='dB', 
                        pdfname=None, tmcfile_ifswitch=None,
                        tmcfile_ifproc=None, xlim=[8,26], y_autoscale=True,
                        dataFraction=[0.0, 1.0], logPower=False, median=False,
                        printSlopes=False, debug=False,ylimitsForRatio=[0.9,1.1]):
        """
        Calls plotCalScans repeatedly for each antenna, pol, spw.
        dataFraction: use this range of the each integration
        median: normalize by the median rather than the mean spectrum
        """
        pngs = []
        for antenna in self.antennas:
            for pol in self.polarizations:
                for spw in self.spws:
                    png = self.plotCalScans(spw=spw, pol=pol, antenna=antenna,
                                            target=target,
                                            normalize=normalize,
                                            plotfile=plotfile,
                                            startdBm=startdBm,
                                            incrementdBm=incrementdBm,
                                            plotSlopes=plotSlopes,
                                            maxSlope=maxSlope, edge=edge,
                                            xaxis=xaxis,
                                            tmcfile_ifswitch=tmcfile_ifswitch,
                                            tmcfile_ifproc=tmcfile_ifproc,
                                            xlim=xlim, y_autoscale=y_autoscale,
                                            dataFraction=dataFraction, logPower=logPower,
                                            median=median, printSlopes=printSlopes,debug=debug,
                                            ylimitsForRatio=ylimitsForRatio)
                    pngs.append(png)
        if (normalize==''):
            normalize = 'raw'
        if (pdfname == None):
            if (dataFraction != [0.0,1.0]):
                pdfname = self.vis + '.%s.%.1f-%.1f.plotCalScans.pdf' % (normalize,dataFraction[0],dataFraction[1])
            else:
                pdfname = self.vis + '.%s.plotCalScans.pdf' % (normalize)
        mystatus = buildPdfFromPngs(pngs, pdfname=pdfname)
        print(mystatus)
        
    def plotCalScans(self, spw=0, pol=0, antenna=0, target='hot',
                     normalize=False, plotfile=None, startdBm=4.0,
                     incrementdBm=-1.0, plotSlopes=False,
                     maxSlope=20, edge=5, xaxis='dB',
                     tmcfile_ifswitch=None, tmcfile_ifproc=None, xlim=None,
                     y_autoscale=True, dataFraction=[0.0,1.0], logPower=False,
                     median=False, printSlopes=False, debug=False,
                     ylimitsForRatio=[0.9,1.1], fontsize=12, overlay=False):
        """
        Plot the atm cal subscans for a specified spw/pol/antenna combination,
        and the specified subscans.
        
        plotSlopes: if True, make 2 rows of plots, with the slope vs. scan
                    on the bottom row
        edge: number of edge channels to ignore when fitting a linear slope
        maxSlope: use this to set the +-y axis range when plotSlopes=True
        xlim: use this to set the x axis range when plotSlopes=True
        xaxis: 'dB' to put the IFProc attenuator setting on the x-axis
               (otherwise, use scan#)
        y_autoscale: True will set y-axis range to automatic
                     False will set y-axis range to [min(all_targets),max(all_targets)]
        dataFraction: use this range of the each integration
        logPower: if True, show the power spectra in log units instead of linear
        overlay: if True, show all on the same panel
        target: 'hot', 'sky', amb' or lists:  ['hot','sky','amb']
        fontsize: for tick labels, axis labels and target labels
        """
        loadNames = {'amb': 'ambient load', 'hot': 'hot load', 'sky': 'sky'}
        spw = int(spw)
        antenna = int(antenna)
        if (self.unrecognizedSpw(spw)): return
        if (self.unrecognizedAntenna(antenna)): return
        pb.clf()
#        pb.hold(True) # not needed
        if (type(target) != list):
            target = [target]
        if (plotSlopes):
            nx = 2
            if (normalize==False):
                normalize = 'meanspectrum'
                print("Since plotSlopes=True, set normalize to '%s'" % (normalize))
        else:
            nx = 1
            if (normalize==False or normalize==''):
                normalize = 'raw'
        if (not overlay):
            adesc = pb.subplot(nx,len(target),1)
        else:
            adesc = pb.subplot(1,1,1)
#            pb.hold(True) # not needed
        if (y_autoscale):
            wspace=0.15
        else:
            wspace=0.1
        pb.subplots_adjust(wspace=wspace, right=0.85, hspace=0.2)
        data0max = 0
        data0min = 1e38
        for t in range(len(target)):
            mytarget = target[t]
            if (not overlay):
                adesc = pb.subplot(nx,len(target),t+1)
            adesc.xaxis.grid(True,which='major')
            adesc.yaxis.grid(True,which='major')
            if (mytarget not in self.targetInverse):
                print("The %s subscan is not in this dataset (valid targets = %s)." % (mytarget,str(self.targetInverse)))
                return
            subscan = self.targetInverse[mytarget]
            if (normalize == 'meanspectrum'):
                self.meanspectrum,duration = self.computeMeanSpectrum(spw,pol,mytarget,
                                                             antenna,dataFraction,median)
            myms = createCasaTool(mstool)
            myms.open(self.vis)
            if (overlay):
                pb.text(0.1,0.9-t*0.1,loadNames[mytarget],size=fontsize,
                        transform=adesc.transAxes,color=overlayColors[t])
            else:
                pb.text(0.1,0.9,loadNames[mytarget],size=fontsize,transform=adesc.transAxes)
            slopes = []
            for i in range(len(self.scans)):
                scan = self.scans[i]
                myms.selectinit(datadescid=self.datadescids[spw])
                myms.select({'time':self.timestamps[scan][subscan],
                             'antenna1':antenna, 'antenna2':antenna})
                data0 = myms.getdata(['amplitude'])['amplitude']
                data0mean = np.median(data0[pol], 1)
                if (normalize == 'meanvalue'):
                    if (median):
                        data0mean /= np.median(data0mean)
                    else:
                        data0mean /= np.mean(data0mean)
                elif (normalize == 'meanspectrum'):
                    if (median):
                        data0mean /= np.median(data0mean)
                    else:
                        data0mean /= np.mean(data0mean)
                    data0mean /= self.meanspectrum
                if ((i < len(self.scans)/2) or (normalize=='raw')):
                    ls = '-'
                else:
                    ls = '--'
                if (logPower):
                    yaxisData = 10*np.log10(data0mean)
                else:
                    yaxisData = data0mean
                if (overlay):
                    colorIndex = t # i*len(target)+t
                else:
                    colorIndex = i
                pb.plot(range(len(data0mean)), yaxisData, ls=ls,
                        color=overlayColors[colorIndex],
                        markerfacecolor=overlayColors[colorIndex], lw=2.0)
                resizeFonts(adesc,fontsize)
                pb.xlim([-2, len(data0mean)+1])
                data0max = np.max([data0max,np.max(data0mean)])
                data0min = np.min([data0min,np.min(data0mean)])
                if (overlay):
                    if (data0min > 0): data0min = 0
                slope,intercept = linfit().linfit(range(len(data0mean[edge:-edge])),
                                                  data0mean[edge:-edge],
                                                  data0mean[edge:-edge]*0.0001)
                slopes.append(slope)
#                print "%4s %2d = %+f" % (mytarget, scan, slope*1000)
            # end for 'i'
            baseband = self.basebands[spw]
            antspwpol= self.antennaNames[antenna]+'.spw%02d.bb%d.pol'%(spw,baseband)+str(pol)
            titleString = mytarget+'.'+antspwpol
            pb.title(titleString,size=13-len(target))
            if (normalize == 'meanvalue' or normalize=='meanspectrum'):
                pb.ylim(ylimitsForRatio)
                if (t != 0 and not overlay):
                    adesc.set_yticklabels([])
            if (t==0):
                obsdateString = mjdsecToUT(getObservationStart(self.vis))
                pb.text(0.1,1+0.05*nx,self.vis + '  ' + obsdateString + ' %.3fGHz'%(self.meanfreq[spw]*1e-9),transform=adesc.transAxes)
                if (normalize == 'meanvalue'):
                    pb.ylabel('Normalized amplitude', size=fontsize)
                elif (normalize == 'meanspectrum'):
                    if (plotSlopes):
                        pb.ylabel('Normalized_Amp / Mean_of_all_scans', size=fontsize)
                    else:
                        pb.ylabel('Normalized_Amplitude / Mean_of_all_scans', size=fontsize)
                else:  # raw
                    if (logPower):
                        pb.ylabel('Relative Amplitude (dB)', size=fontsize)
                    else:
                        pb.ylabel('Amplitude', size=fontsize)

            pb.xlabel('Channel', size=fontsize)
            if (t==len(target)-1):
                font0 = FontProperties()
                font = font0.copy()
                font.set_weight('heavy')
                if (self.attenuatorTest):
                    heading = 'Scan=ReqPower'
                else:
                    heading = 'Scan'
                pb.text(1+0.02*len(target), 1.06, heading,
                        transform=adesc.transAxes, size=9)
                pb.text(1+0.02*len(target), 1.13, 'nsubscans=%d'%self.nsubscans,
                        transform=adesc.transAxes, size=9)
                for j in range(len(self.scans)):
                    if (startdBm == None or self.attenuatorTest==False):
                        mytext = str(self.scans[j])
                    else:
                        mytext = str(self.scans[j]) + "=%+.1fdBm"%(startdBm+j*incrementdBm)
                    if ((j < len(self.scans)/2) or (normalize=='raw')):
                        mytext = mytext
                    else:
                        mytext = ":" + mytext
                    pb.text(1+0.01*len(target), 1-j*0.04*nx, mytext,
                            color=overlayColors[j], size=9,
                            transform=adesc.transAxes, fontproperties=font)
            if (plotSlopes):
                adesc = pb.subplot(nx,len(target),t+1+len(target))
                slopes = np.array(slopes)*100*self.nchan
                bitranges,bitstatus,deltaBits = self.computeBitRanges()
                baseband = self.basebands[spw]
                for i in range(len(self.scans)):
                    if (xaxis=='dB'):
                        self.getAttenuatorSettings(tmcfile_ifswitch=tmcfile_ifswitch,
                                                   tmcfile_ifproc=tmcfile_ifproc, pol=pol, antenna=antenna)
                        if (debug):
                            print("self.IFProc[antenna][pol] = ", self.IFProc[antenna][pol])
                            print("self.IFProc[antenna][pol][self.scans[i]] = ", self.IFProc[antenna][pol][self.scans[i]])
                            print("baseband = ", baseband)
                        dB = self.IFProc[antenna][pol][self.scans[i]][baseband-1]
                        if (debug):
                            print("spw%d: bb%d: dB = %f, slope=%f" % (spw, self.basebands[spw], dB, slopes[i]))
                        pb.plot(dB, slopes[i], 'o',markerfacecolor=overlayColors[i],
                                markeredgecolor=overlayColors[i])
                        if (i>0):
                            pb.text(dB, slopes[i]+1.2, str(deltaBits[previousAttenuation]),
                                    color=overlayColors[i], size=9)
                        previousAttenuation = dB
                    else:
                        pb.plot(self.scans[i],slopes[i],'o',markerfacecolor=overlayColors[i],
                                markeredgecolor=overlayColors[i])
                    if (printSlopes and t==len(target)-1):
                        print("slopes(hot) = ", slopes[i])
                resizeFonts(adesc,9)
                if (xaxis=='dB'):
                    adesc.xaxis.set_minor_locator(MultipleLocator(1.0))
                    adesc.xaxis.grid(True,which='minor')
                    pb.xlabel('IFProc%d:bb%d atten (dB)' % (pol,baseband))
                    # draw a square waveform corresponding to the status of each bit
                    yrange = 2*maxSlope
                    if (xlim == None):
                        startdB = self.IFProcMin[antenna][pol]-1
                        stopdB =  self.IFProcMax[antenna][pol]+1
                    else:
                        startdB, stopdB = xlim
                    pb.xlim([startdB,stopdB])
                    for i in range(len(bitstatus)):
                        xlevel = np.arange(self.IFProcMin[antenna][pol], self.IFProcMax[antenna][pol]+0.6, 0.5)
                        # 0dB --> 0,  0.5dB --> 1, 1dB--> 2
                        ylevel = bitstatus[i][(xlevel*2).astype(int)]*0.015*yrange + (maxSlope-yrange*(0.11+0.05*(i-1)))
                        xlevel -= 0.25
                        pb.plot(xlevel, ylevel, 'k-', drawstyle='steps-post')
                        dB = [16,8,4,2,1,0.5][i]
                        pb.text(stopdB+0.25, np.min(ylevel), str(dB), size=8)
                        if (t == len(target)-1):
                            if (dB < 1):
                                pb.text(stopdB+0.5+0.6*(3), np.min(ylevel)-0.3, 'off', size=5)
                                pb.text(stopdB+0.5+0.6*(3), np.min(ylevel)+0.8, 'on', size=5)
                            else:
                                pb.text(stopdB+0.5+0.6*(1+dB/10), np.min(ylevel)-0.3, 'off', size=5)
                                pb.text(stopdB+0.5+0.6*(1+dB/10), np.min(ylevel)+0.8, 'on', size=5)
                            if (i==0):
                                pb.text(stopdB+0.5, 5, 'duration=%.1f sec' % (duration), size=8)
                                pb.text(stopdB+0.5, 2.5, '(%.1f-%.1f)' % (dataFraction[0],dataFraction[1]), size=8)
                                pb.text(stopdB+0.5, -0.5, 'colored digits', size=8)
                                pb.text(stopdB+0.5, -3, 'denote the number', size=8)
                                pb.text(stopdB+0.5, -6, 'of bits changed', size=8)
                                pb.text(stopdB+0.5, -9, 'when switching', size=8)
                                pb.text(stopdB+0.5, -12, 'to this level', size=8)
                                pb.text(stopdB+0.5, -16, 'IFswSB1:%.1f-%.1fdB'%(self.IFSwitchMin[antenna][pol][1],
                                                                                    self.IFSwitchMax[antenna][pol][1]),size=8)
                                pb.text(stopdB+0.5, -19, 'IFswSB2:%.1f-%.1fdB'%(self.IFSwitchMin[antenna][pol][2],
                                                                                    self.IFSwitchMax[antenna][pol][2]),size=8)
                    pb.xlim([startdB,stopdB])
                else:
                    pb.xlabel('scan number')
                if (t == 0):
                    pb.ylabel('fitted slope (% per BW)')
                else:
                    adesc.set_yticklabels([])
                pb.ylim([-maxSlope, maxSlope])
            # endif plotSlopes
            resizeFonts(adesc,fontsize)
            myms.close()
        #end 'for' t in range(len(target)):
        # set the ylimits for the amp. vs channel plots
        for t in range(len(target)):
            if (not overlay):
                adesc = pb.subplot(nx,len(target),t+1)
            if (y_autoscale == False):
                if (logPower==False):
                    pb.ylim([data0min,data0max])
                else:
                    pb.ylim([np.min(10*np.log10(data0min)),np.max(10*np.log10(data0max))])
                if (t > 0 and not overlay):
                    adesc.set_yticklabels([])
        pb.draw()
        if (plotfile is not None):
            if (plotfile == True):
                png = '.'.join(target) + '.' + antspwpol
                if (normalize != ''):
                    png += '.%s' % (normalize)
                png += '.png'
            else:
                png = plotfile
            pb.savefig(png)
            print("plot saved to %s" % (png))
            return(png)

    def getAntenna(self, antenna):
        """
        Converts an antenna ID or name into a tuple of ID, name.
        """
        if (str(antenna) in self.antennaNames):
            antennaName = antenna
            antennaId = self.antennaNames.index(antennaName)
        elif (antenna in self.antennas or antenna in [str(a) for a in self.antennas]):
            antennaId = int(antenna)
            antennaName = self.antennaNames[antennaId]
        else:
            print("Antenna %s is not in the dataset" % (str(antenna)))
            return None
        return(antennaId, antennaName)

    def getTelcalTrx(self, antenna, spw, scan, pol):
        return(self.getTelcalTspectrum('TRX_SPECTRUM', antenna, spw, scan, pol))
        
    def getTelcalTsys(self, antenna, spw, scan, pol):
        return(self.getTelcalTspectrum('TSYS_SPECTRUM', antenna, spw, scan, pol))
        
    def getTelcalTsky(self, antenna, spw, scan, pol):
        return(self.getTelcalTspectrum('TSKY_SPECTRUM', antenna, spw, scan, pol))
        
    def getTelcalTspectrum(self, tspec, antenna, spw, scan, pol):
        if (self.unrecognizedAntenna(antenna)): return
        if (self.unrecognizedScan(scan)): return
        if (self.unrecognizedSpw(spw)): return
        antennaId, antennaName = self.getAntenna(antenna)
        mytb = createCasaTool(tbtool)
        mytb.open(self.vis+'/SYSCAL')
        myt = mytb.query('ANTENNA_ID == %s and SPECTRAL_WINDOW_ID == %d' % (antennaId,spw))
        times = myt.getcol('TIME')
        trx = myt.getcol(tspec)
        myt.close()
        mytb.close()
        mindiff = 1e38
        pickrow = -1
        if (scan in self.syscalScans):
            for row in range(len(times)):
                diff = abs(times[row] - self.meantime[scan])
                if (diff < mindiff):
                    mindiff = diff
                    pickrow = row
        if (pickrow < 0):
            print("There are no TelCal-generated Trx/Tsys spectra found for this spw/antenna/pol/scan.")
            return None
        else:
            return(trx[pol,:,pickrow])

    def unrecognizedAntenna(self, antenna):
        if (antenna not in self.antennas and antenna not in self.antennaNames and
            antenna not in [str(a) for a in self.antennas]):
            print("antenna %s is not available.  valid antennas = %s, %s" % (str(antenna), str(self.antennas), str(self.antennaNames)))
            return(True)
        return(False)

    def unrecognizedSpw(self, spw):
        if (spw not in self.spws):
            print("spw %d is not available.  valid spws = %s" % (spw, str(self.spws)))
            return(True)
        return(False)

    def nonExistentTable(self, mytable):
        exists = os.path.exists(mytable)
        if (not exists):
            print("Could not find table = %s" % (mytable))
            print("Did you importasdm with asis='CalAtmosphere' ?")
        return not exists

    def unrecognizedScan(self, scan):
        if (type(scan) != int and type(scan) != np.int32):
            if (scan.isdigit()):
                scan = int(scan)
            else:
                print("The scan number must be an integer or integer string.")
                return(False)
        if (int(scan) not in self.scans):
            print("Scan %d is not a cal scan.  Available scans = %s" % (int(scan), str(self.scans)))
            return(True)
        return(False)

    def plotTsys(self, scan, antenna, pol, spw, tdmspw=None, asdm=None, etaF=0.98, lo1=None,
                 dataFraction=[0.0, 1.0], parentms=None, verbose=False,
                 siteAltitude_m=5000, computeJsky=True, altscan=None, overlayTelcal=True,
                 plotrange=[0,0,0,0], fdmCorrection=False, tdmscan=None, tdmdataset=None, plotfile='',
                 showAttenuators=False, takeLoadsFromTdmDataset=False):
        """
        Computes Tsys and then plots the newly-calculated Tsys.
        See also plotTsysTrec2 to plot both Tsys and Trec2.
        """
        spw = int(spw)
#        if ((tdmspw is not None or tdmdataset is not None or tdmspw is not None) and takeLoadsFromTdmDataset==False):
#            if (fdmCorrection == False): print "Setting fdmCorrection to True"
#            fdmCorrection = True
        if (self.unrecognizedAntenna(antenna)): return
        if (self.unrecognizedScan(scan)): return
        if (spw == 'auto'):
            spw = self.spwsforscan[scan][0]
            print("Choosing spw = %d" % (spw))
        if (self.unrecognizedSpw(spw)): return
        antennaId, antennaName = self.getAntenna(antenna)
        startTime = timeUtilities.time()
        result = self.computeTsys(scan, antenna, pol, spw, tdmspw, asdm, etaF, lo1,
                                  dataFraction, parentms, verbose,
                                  siteAltitude_m, computeJsky, altscan, fdmCorrection=fdmCorrection,
                                  tdmscan=tdmscan, tdmdataset=tdmdataset,
                                  takeLoadsFromTdmDataset=takeLoadsFromTdmDataset)
        stopTime = timeUtilities.time()
        if (stopTime-startTime > 5):
            print("Computation required %.1f seconds" % (stopTime-startTime))
        if (result == None):
            return
        if computeJsky:
            tsys, freqHz, trec, tsky, tcal, atmosphere = result
        else:
            trec, gain, tsky, freqHz = result
        pb.clf()
        pol = int(pol)
        adesc = pb.subplot(111)
        freqHz = self.chanfreqs[spw]
        casaTsysColor = 'r'
        pb.plot(freqHz*1e-9, tsys, '-', color=casaTsysColor)
        if (verbose):
            print("len(freqHz) = %d,  shape(tsys) = %s" % (len(freqHz), str(np.shape(tsys))))
        if (overlayTelcal):
            if (fdmCorrection):
                lw = 1
            else:
                lw = 3
            tsys_telcal = self.getTelcalTsys(antenna,spw,scan,pol)  # TelCal's result
            if (tsys_telcal == None):
                print("No Tsys result is available from TelCal ")
                overlayTelcal = False
            else:
                pb.plot(freqHz*1e-9, tsys_telcal, 'g-', lw=lw)
                y0,y1 = pb.ylim()
                pb.ylim([y0,y1+(y1-y0)*0.2])
                if (fdmCorrection):
                    mylabel = ' FDM'
                else:
                    mylabel = ''
                pb.text(0.05,0.95,'TelCal'+mylabel, color='g', transform=adesc.transAxes) # ,weight='extra bold')
                pb.text(0.42,0.95,'casa'+mylabel, color=casaTsysColor, transform=adesc.transAxes)
                pb.text(0.05,0.80, 'HotLoad = %.2fK' % (self.loadTemperatures[antennaId][scan]['hot']),
                        transform=adesc.transAxes,color='k')
                pb.text(0.05,0.75, 'AmbLoad = %.2fK' % (self.loadTemperatures[antennaId][scan]['amb']),
                        transform=adesc.transAxes,color='k')
                if (showAttenuators):
                    if (self.IFProc[antennaId][pol] == None or self.IFSwitch[antennaId][pol][1] == None):
                        self.readAttenuatorSettings(antennaId,pol)
                    if (scan in self.IFProc[antennaId][pol]):
                      if (self.IFProc[antennaId][pol][scan] != -1):
                        pb.text(0.05,0.85,'IFSw %.1fdB, IFPr %.1fdB'%(self.IFSwitch[antennaId][pol][self.sidebandsforspw[spw]][scan],
                                                       self.IFProc[antennaId][pol][scan][self.basebands[spw]-1]),
                            color='g',transform=adesc.transAxes)
#                pb.hold(True) # not needed
            if (fdmCorrection or takeLoadsFromTdmDataset):
                if (tdmspw == None): tdmspw = spw
                if (tdmscan == None): tdmscan = scan
                if (tdmdataset == None):
                    tdmdataset = self
                tsys = tdmdataset.getTelcalTsys(antenna,tdmspw,tdmscan,pol)  # TelCal's result for the TDM spectrum
                freqHz = tdmdataset.chanfreqs[tdmspw]
                pb.plot(freqHz*1e-9, tsys, 'r-')
                y0,y1 = pb.ylim()
                pb.ylim([y0,y1+(y1-y0)*0.2])
                pb.text(0.65,0.95, 'TelCal TDM (%s)' % (mjdsecToUTHMS(np.mean(tdmdataset.timerange[tdmscan]))),
                        color='r', transform=adesc.transAxes)
                pb.text(0.60,0.90, tdmdataset.visBasename, color='r', transform=adesc.transAxes)
                tdmAntennaId, antennaName = tdmdataset.getAntenna(antenna)
                pb.text(0.60,0.80, 'HotLoad = %.2fK' % (tdmdataset.loadTemperatures[tdmAntennaId][tdmscan]['hot']),transform=adesc.transAxes,color='r')
                pb.text(0.60,0.75, 'AmbLoad = %.2fK' % (tdmdataset.loadTemperatures[tdmAntennaId][tdmscan]['amb']),transform=adesc.transAxes,color='r')
                if (showAttenuators):
                    if (tdmdataset.IFProc[antennaId][pol] == None or tdmdataset.IFSwitch[antennaId][pol][1] == None):
                        tdmdataset.readAttenuatorSettings(antennaId,pol)
#                    print "tdmdataset.IFSwitch[antennaId=%d][pol=%d] = %s" % (antennaId,pol,str(tdmdataset.IFSwitch[antennaId][pol]))
                    if (tdmscan in tdmdataset.IFProc[antennaId][pol]):
                      if (tdmdataset.IFProc[antennaId][pol][tdmscan] != -1):
                        pb.text(0.65,0.85,'IFSw %.1fdB, IFPr %.1fdB'%(tdmdataset.IFSwitch[antennaId][pol][tdmdataset.sidebandsforspw[spw]][tdmscan],
                                                       tdmdataset.IFProc[antennaId][pol][tdmscan][tdmdataset.basebands[spw]-1]),
                            color='r',transform=adesc.transAxes)
        pb.xlabel('Frequency (GHz)')
        pb.ylabel('Tsys (K)')
        if (plotrange != [0,0,0,0]):
            if (plotrange[0] != 0 or plotrange[1] != 0):
                pb.xlim(plotrange[:2])
            if (plotrange[2] != 0 or plotrange[3] != 0):
                pb.ylim(plotrange[2:])
        adesc.xaxis.grid(True,which='major')
        adesc.yaxis.grid(True,which='major')
        ut = mjdsecToUTHMS(self.meantime[scan])
        pb.title('%s  %s  scan=%d  spw=%d  pol=%d  mean_time=%s' % (os.path.basename(self.vis), antennaName, scan, spw, pol, ut), fontsize=11)
        pb.draw()
        if (plotfile == '' or plotfile==True):
            png = os.path.basename(self.vis) + '.%s.scan%02d.spw%02d.pol%d.tsys.png' % (antennaName,scan,spw,pol)
        else:
            png = plotfile
        pb.savefig(png)
        print("Result left in ", png)
        return png

    def plotTsysTrec2(self, scan, antenna, pol, spw, tdmspw=None, asdm=None, etaF=0.98,
                      lo1=None, dataFraction=[0.0, 1.0], parentms=None, verbose=False,
                      siteAltitude_m=5000, computeJsky=True, altscan=None,
                      overlayTelcal=True,
                      plotrange=[0,0,0,0], fdmCorrection=False, tdmscan=None,
                      tdmdataset=None,
                      plotfile='', showAttenuators=False, trxDifferences=None,
                      tsysDifferences=None, takeLoadsFromTdmDataset=False):
        """
        Plots Tsys in one panel and Trec2 in another panel on the same page, for a specified
        combination of scan, antenna, pol, spw.
        """
        spw = int(spw)
#        if ((tdmspw is not None or tdmdataset is not None or tdmspw is not None) and takeLoadsFromTdmDataset==False):
#            if (fdmCorrection == False): print "Setting fdmCorrection to True"
#            fdmCorrection = True
        if (self.unrecognizedAntenna(antenna)): return
        if (self.unrecognizedScan(scan)): return
        if (spw == 'auto'):
            spw = self.spwsforscan[scan][0]
            print("Choosing spw = %d" % (spw))
        if (self.unrecognizedSpw(spw)): return
        antennaId, antennaName = self.getAntenna(antenna)
        antenna = antennaId
        startTime = timeUtilities.time()
        result = self.computeTsys(scan, antenna, pol, spw, tdmspw, asdm, etaF, lo1,
                                  dataFraction, parentms, verbose,
                                  siteAltitude_m, computeJsky, altscan, fdmCorrection=fdmCorrection,
                                  tdmscan=tdmscan, tdmdataset=tdmdataset,
                                  takeLoadsFromTdmDataset=takeLoadsFromTdmDataset)
        stopTime = timeUtilities.time()
        if (stopTime-startTime > 5):
            print("Computation required %.1f seconds" % (stopTime-startTime))
        if (result == None):
            return
        tsys, freqHz, trec, tsky, tcal = result
        pb.clf()
        pol = int(pol)
        freqHz = self.chanfreqs[spw]
        adesc = pb.subplot(211)
        pb.plot(freqHz*1e-9, trec, 'k-', lw=2)
#        print "Median Trec (black) = ", np.median(trec)
        if (verbose):
            print("len(freqHz) = %d,  shape(trec) = %s" % (len(freqHz), str(np.shape(trec))))
        if (overlayTelcal):
            trx_telcal = self.getTelcalTrx(antenna,spw,scan,pol)  # TelCal's result
            if (trx_telcal == None):
                print("No Trx result is available from TelCal ")
                overlayTelcal = False
            else:
                pb.plot(freqHz*1e-9, trx_telcal, 'g-', lw=1)
                y0,y1 = pb.ylim()
                pb.ylim([y0,y1+(y1-y0)*0.2])
                if (fdmCorrection):
                    mylabel = ' FDM'
                else:
                    mylabel = ''
                pb.text(0.05,0.92,'TelCal'+mylabel,color='g',transform=adesc.transAxes) # ,weight='extra bold')
                pb.text(0.42,0.92,'casa'+mylabel,color='k',transform=adesc.transAxes)
#                pb.hold(True) # not needed
                if (showAttenuators):
                    if (self.IFProc[antenna][pol] == None or self.IFSwitch[antenna][pol][1] == None):
                        self.readAttenuatorSettings(antenna,pol)
                    if (scan in self.IFProc[antenna][pol]):
                      if (self.IFProc[antenna][pol][scan] != -1):
                        pb.text(0.05,0.84,'IFSw %.1fdB, IFPr %.1fdB'%(self.IFSwitch[antenna][pol][self.sidebandsforspw[spw]][scan],
                                                       self.IFProc[antenna][pol][scan][self.basebands[spw]-1]),
                            color='g',transform=adesc.transAxes)
            if (fdmCorrection or takeLoadsFromTdmDataset):
                if (tdmspw == None): tdmspw = spw
                if (tdmscan == None): tdmscan = scan
                if (tdmdataset == None):
                    tdmdataset = self
                trx = tdmdataset.getTelcalTrx(antenna,tdmspw,tdmscan,pol)  # TelCal's result for the TDM spectrum
                freqHzTDM = tdmdataset.chanfreqs[tdmspw]
                pb.plot(freqHzTDM*1e-9, trx, 'r-')
                pb.text(0.65,0.92, 'TelCal TDM (%s)' % (mjdsecToUTHMS(np.mean(tdmdataset.timerange[tdmscan]))),
                        color='r', transform=adesc.transAxes)
                pb.text(0.65,0.84, tdmdataset.vis,color='r',transform=adesc.transAxes)
                trxDiff = np.median(trx)-np.median(trec)
                pb.text(0.42,0.84, 'diff %.1fK'%(trxDiff),color='k',transform=adesc.transAxes)
                if (showAttenuators):
                    if (tdmdataset.IFProc[antenna][pol] == None or tdmdataset.IFSwitch[antenna][pol][1] == None):
                        tdmdataset.readAttenuatorSettings(antenna,pol)
                    print("tdmdataset.IFSwitch[antenna][pol] = ", tdmdataset.IFSwitch[antenna][pol])
                    print("tdmdataset.IFProc[antenna][pol] = ", tdmdataset.IFProc[antenna][pol])
                    if (tdmscan in tdmdataset.IFProc[antenna][pol]):
                      if (tdmdataset.IFProc[antenna][pol][tdmscan] != -1):
                        pb.text(0.65,0.84,'IFSw %.1fdB, IFPr %.1fdB'%(tdmdataset.IFSwitch[antenna][pol][tdmdataset.sidebandsforspw[spw]][tdmscan],
                                                       tdmdataset.IFProc[antenna][pol][tdmscan][tdmdataset.basebands[spw]-1]),
                            color='r',transform=adesc.transAxes)
                        if (scan in self.IFProc[antenna][pol] and
                            scan in self.IFSwitch[antenna][pol][self.sidebandsforspw[spw]]):
                          if ((tdmdataset.IFSwitch[antenna][pol][tdmdataset.sidebandsforspw[spw]][tdmscan] ==
                             self.IFSwitch[antenna][pol][self.sidebandsforspw[spw]][scan]) and
                            (tdmdataset.IFProc[antenna][pol][tdmscan][tdmdataset.basebands[spw]-1] ==
                             self.IFProc[antenna][pol][scan][self.basebands[spw]-1])):
                            if (trxDifferences is not None):
                                if (antenna not in list(trxDifferences['antenna'].keys())):
                                    trxDifferences['antenna'][antenna] = []
                                if (spw not in list(trxDifferences['spw'].keys())):
                                    trxDifferences['spw'][spw] = []
                                trxDifferences['antenna'][antenna].append(trxDiff)
                                trxDifferences['spw'][spw].append(trxDiff)
        pb.xlabel('Frequency (GHz)')
        pb.ylabel('Trec2 (K)')
        if (plotrange != [0,0,0,0]):
            if (plotrange[0] != 0 or plotrange[1] != 0):
                pb.xlim(plotrange[:2])
            if (plotrange[2] != 0 or plotrange[3] != 0):
                pb.ylim(plotrange[2:])
        else:
            if (fdmCorrection or takeLoadsFromTdmDataset):
                pb.xlim([np.min(freqHzTDM)*1e-9, np.max(freqHzTDM)*1e-9])
                # trec, trx_telcal  and if fdmCorrection, you have trx 
                pb.ylim([np.min([np.min(trec), np.min(trx), np.min(trx_telcal)]), np.max([np.max(trec),np.max(trx),np.max(trx_telcal)])])
            else:
                pb.xlim([np.min(freqHz)*1e-9, np.max(freqHz)*1e-9])
                pb.ylim([np.min([np.min(trec), np.min(trx_telcal)]), np.max([np.max(trec), np.max(trx_telcal)])])
            y0,y1 = pb.ylim()
            if (y0 < 0 and y0 > -100000 and np.median(trec)>0 and np.median(trx_telcal)>0): y0 = 0
            pb.ylim([y0,y1+(y1-y0)*0.2])

        adesc.xaxis.grid(True,which='major')
        adesc.yaxis.grid(True,which='major')
        ut = mjdsecToUTHMS(self.meantime[scan])
        pb.title('%s  %s  scan=%d  spw=%d  pol=%d  mean_time=%s' % (os.path.basename(self.vis), antennaName, scan, spw, pol, ut), fontsize=11)

        adesc = pb.subplot(212)
        pb.plot(freqHz*1e-9, tsys, 'k-')
        if (verbose):
            print("len(freqHz) = %d,  shape(tsys) = %s" % (len(freqHz), str(np.shape(tsys))))
        if (overlayTelcal):
            tsys_telcal = self.getTelcalTsys(antenna,spw,scan,pol)  # TelCal's result
            if (tsys_telcal == None):
                print("No Tsys result is available from TelCal ")
                overlayTelcal = False
            else:
                pb.plot(freqHz*1e-9, tsys_telcal, 'g-', lw=1)
                if (fdmCorrection):
                    mylabel = ' FDM'
                else:
                    mylabel = ''
                pb.text(0.05,0.92,'TelCal'+mylabel,color='g',transform=adesc.transAxes) # ,weight='extra bold')
                pb.text(0.42,0.92,'casa'+mylabel,color='k',transform=adesc.transAxes)
#                pb.hold(True) # not needed
            if (fdmCorrection or takeLoadsFromTdmDataset):
                if (tdmspw == None): tdmspw = spw
                if (tdmscan == None): tdmscan = scan
                if (tdmdataset == None):
                    tdmdataset = self
                tsysTDM = tdmdataset.getTelcalTsys(antenna,tdmspw,tdmscan,pol)  # TelCal's result for the TDM spectrum
                freqHzTDM = tdmdataset.chanfreqs[tdmspw]
                pb.plot(freqHzTDM*1e-9, tsysTDM, 'r-')
                pb.text(0.65,0.92, 'TelCal TDM (%s)' % (mjdsecToUTHMS(np.mean(tdmdataset.timerange[tdmscan]))),
                        color='r', transform=adesc.transAxes)
                tsysDiff = np.median(tsysTDM)-np.median(tsys)
                pb.text(0.42,0.84, 'diff %.1fK'%(tsysDiff),color='k',transform=adesc.transAxes)
                if (tsysDifferences is not None):
                  if (scan in self.IFProc[antenna][pol] and
                      scan in self.IFSwitch[antenna][pol][self.sidebandsforspw[spw]]):
                    if ((tdmdataset.IFSwitch[antenna][pol][tdmdataset.sidebandsforspw[spw]][tdmscan] ==
                         self.IFSwitch[antenna][pol][self.sidebandsforspw[spw]][scan]) and
                        (tdmdataset.IFProc[antenna][pol][tdmscan][tdmdataset.basebands[spw]-1] ==
                         self.IFProc[antenna][pol][scan][self.basebands[spw]-1])):
                        if (antenna not in list(tsysDifferences['antenna'].keys())):
                            tsysDifferences['antenna'][antenna] = []
                        if (spw not in list(tsysDifferences['spw'].keys())):
                            tsysDifferences['spw'][spw] = []
                        tsysDifferences['antenna'][antenna].append(tsysDiff)
                        tsysDifferences['spw'][spw].append(tsysDiff)

        pb.xlabel('Frequency (GHz)')
        pb.ylabel('Tsys (K)')
        if (plotrange != [0,0,0,0]):
            if (plotrange[0] != 0 or plotrange[1] != 0):
                pb.xlim(plotrange[:2])
            if (plotrange[2] != 0 or plotrange[3] != 0):
                pb.ylim(plotrange[2:])
        else:
            if (fdmCorrection or takeLoadsFromTdmDataset):
                pb.xlim([np.min(freqHzTDM)*1e-9, np.max(freqHzTDM)*1e-9])
                pb.ylim([np.min([np.min(tsys), np.min(tsys_telcal), np.min(tsysTDM)]), np.max([np.max(tsys),np.max(tsys_telcal),np.max(tsysTDM)])])
            else:
                pb.xlim([np.min(freqHz)*1e-9, np.max(freqHz)*1e-9])
                pb.ylim([np.min([np.min(tsys), np.min(tsys_telcal)]), np.max([np.max(tsys),np.max(tsys_telcal)])])
            y0,y1 = pb.ylim()
            if (y0 < 0 and y0 > -100000): y0 = 0
            pb.ylim([y0,y1+(y1-y0)*0.2])
        adesc.xaxis.grid(True,which='major')
        adesc.yaxis.grid(True,which='major')
        ut = mjdsecToUTHMS(self.meantime[scan])
        pb.draw()
        if (plotfile == '' or plotfile==True):
            png = os.path.basename(self.vis) + '.%s.scan%02d.spw%02d.pol%d.tsys_trx.png' % (antennaName,scan,spw,pol)
        else:
            png = plotfile
        pb.savefig(png)
        print("Result left in ", png)
        return png

    def computeTsys(self, scan, antenna, pol, spw, tdmspw=None, asdm=None, etaF=0.98, lo1=None,
                    dataFraction=[0.0, 1.0], parentms=None, verbose=False,
                    siteAltitude_m=5000, computeJsky=True, altscan=None, ignoreFlags=False,
                    calscandict=None, fdmCorrection=False, tdmscan=None, tdmdataset=None,
                    takeLoadsFromTdmDataset=False, showplot=False, atmosphere=None):
        """
        Computes FDM Tsys from an FDM CALIBRATE_ATMOSPHERE scan by applying
        quantization correction using either a prior calscan in a TDM spw or
        the concurrent scan in an SQLD spw.
        tdmspw:  the spw from which to get the total power in order to apply fdmCorrection
        tdmscan: the scan from which to get the data from the tdmspw in order to apply fdmCorrection
        tdmdataset: the Atmcal instance for the dataset containing the TDM scans
        takeLoadsFromTdmDataset: set to True to use the hot/amb from one dataset, and sky from another
            So, to compute new TDM Tsys for dataset1 using the TDM hot/amb from dataset2,
            set tdmdataset=Atmcal(dataset2), self=Atmcal(dataset1), with fdmCorrection=False.
        fdmCorrection: if True, apply FDM quantization correction
        atmosphere: dictionary keyed by 'signal', 'image', with values returned by
                    au.CalcAtmosphere()
        Returns 5 masked arrays and a dict:  tsys, freqHz, trec, tsky, tcal, atmosphere
        """
        spw = int(spw)
        if (spw == 'auto'):
            spw = self.spwsforscan[scan][0]
            print("Choosing spw = %d" % (spw))
        if (self.unrecognizedAntenna(antenna)): return
        if (self.unrecognizedScan(scan)): return
        if (self.unrecognizedSpw(spw)): return
        if (spw not in self.spwsforscan[scan]):
            print("spw %d is not in scan %d.  Available spws are: %s" % (spw, scan, str(self.spwsforscan[scan])))
            return
        result = self.computeTrec2(scan, antenna, pol, spw, tdmspw,
                                   asdm, etaF, lo1,dataFraction,
                                   parentms, verbose, siteAltitude_m,
                                   computeJsky, altscan,ignoreFlags=ignoreFlags,
                                   calscandict=calscandict, fdmCorrection=fdmCorrection,
                                   tdmscan=tdmscan, tdmdataset=tdmdataset,
                                   takeLoadsFromTdmDataset=takeLoadsFromTdmDataset, 
                                   showplot=showplot, atmosphere=atmosphere)
        if (result == None):
            print("No result from computeTrec2()")
            return(None)
        if (computeJsky):
            trec, gain, tsky, freqHz, tcal, tsys, atmosphere = result
            return(tsys, freqHz, trec, tsky, tcal, atmosphere)
        else:
            trec, gain, tsky, freqHz = result
            return(result)

    def generateHighResTsysTable(self, oldcaltable='', newcaltable='', 
                                 maxRows=-1, maxScans=-1, verbose=False, 
                                 showplots=False, keepLowRes=True, 
                                 simulateTDMTsysScan=-1, showplotLoads=False, 
                                 writeNewLowResSpectrum=False, antenna='',
                                 spw='', filterOrder=8, showpoints=True,
                                 separateFigures=False, scan='', maxAltitude=60,
                                 s=0, mode='difference', highresEB='', pwv=None):
        """
        Takes an existing TDM Tsys table for the active measurement set that 
        contains FDM science data scans, and builds a high resolution version 
        of the Tsys spectra using the atmospheric model and an upsampled
        version of the Trx spectra, producing one resampled Tsys spw 
        per science spw.
        oldcaltable: name of existing TDM Tsys table to read
        newcaltable: name for new FDM Tsys table to write
        maxRows: if > 0, then limit the number of rows processed
        maxScans: if > 0, then limit the number of scans processed
        keepLowRes: if True, then also copy the low resolution Tsys solutions 
            to the new table
        simulateTDMTsysScan: if > 0, then smooth the high-res Tsys scan data 
            to this many channels before using it
        writeNewLowResSpectrum: if True, and simulateTDMTsysScan<=0, then instead 
           of copying the existing low-res spectrum from the old table to the new, 
           write the newly-calculated low-res spectrum;   if simulateTDMTsysScan>0,
           then write another new cal table containing the simulated TDM Tsys result
        antenna: if specified, then restrict new caltable to this list of 
            antennas (python list of IDs, or comma-delimited string of names)
        scan: if specified, then restrict to these scans (python list of integers,
             single integer, or comma-delimited string)
        showplots: passed to simulateHighResTsys
        s: smoothing parameter passed to scipy.interpolate.UnivariateSpline when
           up-interpolating to high-res Tsys again
        mode: 'model', 'data', 'ratio', or 'difference' (passed to simulateHighResTsys)
        maxAltitude: of the atmosphere, in km
        highresEB: if the high-resolution data is in a different EB from the TDM Tsys
          scan, then set this parameter to the former
        pwv: if specified, then use this value to override what was in the ASDM
        Note: the ATM model spectrum is Hanning smoothed if self.hanningSmoothed[spw] == True,
            which is set automatically upon initialization of the Atmcal class.  To override
            this feature, one can set that dictionary entry to True or False before calling
            this function.
        """
        # loop over scan, spw, antenna, pol
        if oldcaltable == '':
            oldcaltable = self.vis + '.tsys'
            if not os.path.exists(oldcaltable):
                print("You must specify the existing Tsys table name.")
                return
            print("Using ", oldcaltable)
        if highresEB != '':
            print("Creating new caltable for: ", highresEB)
            newcaltable = createTsysTable(highresEB, newcaltable)
        else:
            newcaltable = createTsysTable(self.vis, newcaltable)
        siteAltitude_m = getObservatoryAltitude(getObservatoryName(self.vis))
        mytbold = createCasaTool(tbtool)
        mytbnew = createCasaTool(tbtool)
        mytblowres = createCasaTool(tbtool)
        if writeNewLowResSpectrum and simulateTDMTsysScan > 0:
            lowrescaltable = createTsysTable(self.vis, newcaltable+'_lowres')
            mytblowres.open(newcaltable+'_lowres', nomodify=False)
        print("Opening existing table: ", oldcaltable)
        mytbold.open(oldcaltable)
        print("Opening new table:      ", newcaltable)
        mytbnew.open(newcaltable, nomodify=False)
        nrows = mytbold.nrows()
        if (maxRows > 0):
            myrows = np.min([maxRows,nrows])
        else:
            myrows = nrows
        scienceSpws = {}
        atmLowRes = {}
        atmHighRes = {}
        myscans = list(self.scans)
        startTime = timeUtilities.time()
        scanNumberPerRow = mytbold.getcol('SCAN_NUMBER')
        timePerRow = mytbold.getcol('TIME')
        fieldPerRow = mytbold.getcol('FIELD_ID')
        spwPerRow = mytbold.getcol('SPECTRAL_WINDOW_ID')
        antennaPerRow = mytbold.getcol('ANTENNA1')
        intervalPerRow = mytbold.getcol('INTERVAL')
        obsidPerRow = mytbold.getcol('OBSERVATION_ID')
        print("spws in the caltable: ", np.unique(spwPerRow))
        newrows = 0
        if antenna != '':
            antennaList = parseAntenna(self.vis, antenna, self.mymsmd)
            print("Processing antenna IDs: ", antennaList)
        if spw != '':
            spwList = parseSpw(self.vis, spw, self.mymsmd)
            print("Processing spw IDs: ", spwList)
        if scan != '':
            if type(scan) == str:
                scansRequested = [int(i) for i in scan.split(',')]
            elif type(scan) == list:
                scansRequested = scan
            else:
                scansRequested = [scan]
        else:
            scansRequested = myscans
        print("Processing scans: ", scansRequested)
        for i in range(myrows):
            if i % 10 == 0:
                line =  "***** Working row %d/%d *****" % (i+1,myrows)
            if antenna != '':
                if antennaPerRow[i] not in antennaList:
                    continue
            if spw != '':
                if spwPerRow[i] not in spwList:
                    continue
            if scansRequested != '':
                if scanNumberPerRow[i] not in scansRequested:
                    continue
            if i > 10:
                eta = (timeUtilities.time()-startTime)*(myrows-i)/(3600.*i)
                if maxScans > 0:
                    eta *= maxScans/float(len(myscans))
                line += "***** ETA: %.2f hours *****" % (eta)
            if i % 10 == 0:
                print(line)
            scan = scanNumberPerRow[i]
            if (myscans.index(scan) >= maxScans and maxScans > 0):
                print("Stopping due to maxScans.")
                break
            mytime = timePerRow[i]
            field = fieldPerRow[i]
            antennaID = antennaPerRow[i]
            if field not in self.radec:
                self.radec[field] = getRADecForField(self.vis, field, usemstool=True, mymsmd=self.mymsmd)
            spwid = spwPerRow[i]
            if spwid not in scienceSpws:
                scienceSpws[spwid] = inverseTsysspwmapWithNoTable(self.vis, field, spwid, mymsmd=self.mymsmd,
                                                                  alternateIntents=['OBSERVE_TARGET#ON_SOURCE',
                                                                                    'CALIBRATE_AMPLI#ON_SOURCE'])
                print("Tsys spw %d: science spws = " % (spwid), scienceSpws[spwid])
            defaultError = 0.1
            defaultSNR = 1.0
            if keepLowRes:
                myspws = sorted(np.unique([spwid] + scienceSpws[spwid]))
            else:
                myspws = sorted(np.unique(scienceSpws[spwid]))
            for myspw in myspws:
                tsys = mytbold.getcell('FPARAM', i)
                if scan not in list(atmLowRes.keys()):
                    atmLowRes[scan] = {}
                if spwid not in atmLowRes[scan]:
                    atmLowRes[scan][spwid] = None
                if scan not in list(atmHighRes.keys()):
                    atmHighRes[scan] = {}
                if myspw not in list(atmHighRes[scan].keys()):
                    atmHighRes[scan][myspw] = None
                mytbnew.addrows(1)
                mytbnew.putcell('SPECTRAL_WINDOW_ID',newrows,myspw)
                mytbnew.putcell('TIME',newrows,mytime)
                mytbnew.putcell('FIELD_ID',newrows,field)
                mytbnew.putcell('ANTENNA1',newrows,antennaID)
                mytbnew.putcell('ANTENNA2',newrows,antennaID)
                mytbnew.putcell('INTERVAL',newrows,intervalPerRow[i])
                mytbnew.putcell('SCAN_NUMBER',newrows,scan)
                mytbnew.putcell('OBSERVATION_ID', newrows, obsidPerRow[i])
                if writeNewLowResSpectrum and simulateTDMTsysScan > 0:
                    mytblowres.addrows(1)
                    mytblowres.putcell('SPECTRAL_WINDOW_ID',newrows,myspw)
                    mytblowres.putcell('TIME',newrows,mytime)
                    mytblowres.putcell('FIELD_ID',newrows,field)
                    mytblowres.putcell('ANTENNA1',newrows,antennaID)
                    mytblowres.putcell('ANTENNA2',newrows,antennaID)
                    mytblowres.putcell('INTERVAL',newrows,intervalPerRow[i])
                    mytblowres.putcell('SCAN_NUMBER',newrows,scan)
                    mytblowres.putcell('OBSERVATION_ID', newrows, obsidPerRow[i])
                npol = len(np.shape(tsys))
                trec = {}
                gain = {}
                tsky = {}
                tcal = {}
                tsysHighRes = {}
                tsysLowRes = {}
                for j in range(npol):
                    trec[j] = []
                    gain[j] = []
                    tsky[j] = []
                    tcal[j] = []
                    tsysHighRes[j] = []
                    tsysLowRes[j] = []
                if myspw != spwid or highresEB != '':
                    print("Working on FDM spw %d for ant%2d=%s in %s" % (myspw, antennaID, self.antennaNames[antennaID], highresEB))
                    newFDMFrequencyAxis = getChanFreqFromCaltable(newcaltable, myspw) * 1e9
                    for j in range(npol): 
                        result = self.simulateHighResTsys(scan, antennaID, j, spwid, myspw, atmosphereLowRes=atmLowRes[scan][spwid], atmosphereHighRes=atmHighRes[scan][myspw], verbose=verbose, showplot=showplots, filterOrder=filterOrder, showpoints=showpoints, separateFigures=separateFigures, maxAltitude=maxAltitude, s=s, mode=mode, highresEB=highresEB, newFDMFrequencyAxis=newFDMFrequencyAxis, pwv=pwv, siteAltitude_m=siteAltitude_m)
                        if result is None:
                            return
                        trec[j], gain[j], tsky[j], newfreqHz, tcal[j], tsysHighRes[j], atmLowRes[scan][spwid], atmHighRes[scan][myspw], tsysLowRes[j] = result
                    channel0 = getChanFreqFromCaltable(newcaltable, myspw, 0)
                    finalchannel = getChanFreqFromCaltable(newcaltable, myspw, -1)
                    if verbose:
                        print("start: newfreqGHz[0]=%f  spw%dFreq[0]=%f, diff=%f, meanfreq=%f" % (newfreqHz[0]*1e-9, myspw, channel0, (newfreqHz[0]*1e-9)-channel0, getSpwMeanFreqFromCaltable(newcaltable,myspw)))
                        print("  end: newfreqGHz[-1]=%f  spw%dFreq[-1]=%f, meanfreq=%f" % (newfreqHz[-1]*1e-9, myspw, finalchannel, getSpwMeanFreqFromCaltable(newcaltable,myspw)))
                        idx = np.argmax(tsky[0])
                        print("generateHighResTsysTable(): Peak tsky[0]=%f at freq=%f at idx=%d" % (tsky[0][idx], newfreqHz[idx], idx))
                        idx = np.argmax(tsysHighRes[0])
                        print("generateHighResTsysTable(): Peak tsys[0]=%f at freq=%f at idx=%d" % (tsysHighRes[0][idx], newfreqHz[idx], idx))
                        caltableFreq = getChanFreqFromCaltable(newcaltable, myspw, idx)
                        print("                            channel %d in SPECTRAL_WINDOW table of caltable = %fGHz, diff=%fGHz" % (idx, caltableFreq, newfreqHz[idx]*1e-9-caltableFreq))
                        if highresEB != '':
                            mymsmd = createCasaTool(msmdtool)
                            mymsmd.open(highresEB)
                            msFreq = mymsmd.chanfreqs(myspw)[idx] * 1e-9
                            mymsmd.close()
                            print("                            channel %d in SPECTRAL_WINDOW table of ms       = %fGHz, diff=%fGHz" % (idx, msFreq, newfreqHz[idx]*1e-9 - msFreq))
                    if npol == 0:
                        tsys = np.array([tsysHighRes[0]])
                        tsysLowRes = np.array([tsysLowRes[0]])
                    else:
                        tsys = np.array([tsysHighRes[0],tsysHighRes[1]])
                        tsysLowRes = np.array([tsysLowRes[0],tsysLowRes[1]])
                    if writeNewLowResSpectrum:
                        # print "Writing the low-res spectrum"
                        mytbnew.putcell('FLAG', i, tsysLowRes<0)
                        mytbnew.putcell('FPARAM', i, tsysLowRes)
                        mytbnew.putcell('SNR', i, defaultSNR*tsysLowRes/tsysLowRes)
                        mytbnew.putcell('PARAMERR', i, defaultError*tsysLowRes/tsysLowRes)
                    else:
                        # print "Writing the new high-res spectrum"
                        mytbnew.putcell('FLAG', newrows, tsys<0)
                        mytbnew.putcell('FPARAM', newrows, tsys)
                        mytbnew.putcell('SNR', newrows, defaultSNR*tsys/tsys)
                        mytbnew.putcell('PARAMERR', newrows, defaultError*tsys/tsys)
                elif simulateTDMTsysScan > 0:
                    for j in range(npol):
                        print("Calling simulateHighResTsys(simulateTDMTsysScan=%d, pol=%d, antenna=%d)" % (simulateTDMTsysScan, j, antennaID))
                        result = self.simulateHighResTsys(scan, antennaID, j, spwid, myspw, atmosphereLowRes=atmLowRes[scan][spwid], atmosphereHighRes=atmHighRes[scan][myspw], verbose=verbose, showplot=showplots, simulateTDMTsysScan=simulateTDMTsysScan, showplotLoads=showplotLoads, filterOrder=filterOrder, showpoints=showpoints, separateFigures=separateFigures, maxAltitude=maxAltitude, s=s, mode=mode, newFDMFrequencyAxis=newFDMFrequencyAxis, pwv=pwv, siteAltitude_m=siteAltitude_m)
                        if result is None:
                            return
                        trec[j], gain[j], tsky[j], newfreqHz, tcal[j], tsysHighRes[j], atmLowRes[scan][spwid], atmHighRes[scan][myspw], tsysLowRes[j] = result
                    if npol == 0:
                        tsys = np.array([tsysHighRes[0]])
                        tsysLowRes = np.array([tsysLowRes[0]])
                    else:
                        tsys = np.array([tsysHighRes[0],tsysHighRes[1]])
                        tsysLowRes = np.array([tsysLowRes[0],tsysLowRes[1]])
                    mytbnew.putcell('FLAG', newrows, tsys<0)
                    mytbnew.putcell('FPARAM', newrows, tsys)
                    mytbnew.putcell('SNR', newrows, defaultSNR*tsys/tsys)
                    mytbnew.putcell('PARAMERR', newrows, defaultError*tsys/tsys)
                    if writeNewLowResSpectrum:
                        print("Writing the low-res spectrum, row %d" % (i))
                        mytblowres.putcell('FLAG', newrows, tsysLowRes<0)
                        mytblowres.putcell('FPARAM', newrows, tsysLowRes)
                        mytblowres.putcell('SNR', newrows, defaultSNR*tsysLowRes/tsysLowRes)
                        mytblowres.putcell('PARAMERR', newrows, defaultError*tsysLowRes/tsysLowRes)
                else:
                    print("Copying the existing Tsys spw (%d) to the output table." % (spwid))
                    mytbnew.putcell('FLAG', newrows, mytbold.getcell('FLAG',i))
                    mytbnew.putcell('FPARAM', newrows, tsys)
                    mytbnew.putcell('SNR', newrows, mytbold.getcell('SNR',i))
                    mytbnew.putcell('PARAMERR', newrows, mytbold.getcell('PARAMERR',i))
                newrows += 1
        mytbold.close()
        mytbnew.close()
        if writeNewLowResSpectrum and simulateTDMTsysScan > 0:
            mytblowres.close()
            for spw in spwList:
                nchan = getNChanFromCaltable(newcaltable+'_lowres', spw)
                spwtable = newcaltable+'_lowres/SPECTRAL_WINDOW'
                print("Calling smoothSpectralWindowTable('%s', %d, %d)" % (spwtable, spw, nchan/simulateTDMTsysScan))
                smoothSpectralWindowTable(spwtable, spw, nchan/simulateTDMTsysScan)
        print("total time = %.1f seconds" % (timeUtilities.time() - startTime))
        return

    def simulateHighResTsys(self, scan, antenna, pol, spw, fdmspw, asdm=None, etaF=0.98, 
                            lo1=None, dataFraction=[0.0, 1.0], parentms=None, verbose=False,
                            siteAltitude_m=5000, altscan=None, ignoreFlags=False,
                            tdmscan=None, tdmdataset=None, showplot=True,
                            showplotLoads=False, atmosphereLowRes=None, atmosphereHighRes=None,
                            showpoints=False, simulateTDMTsysScan=-1, 
                            plotedge=10, listpeak=False, filterOrder=8, 
                            separateFigures=False, maxAltitude=60, s=0, mode='difference',
                            highresEB='', newFDMFrequencyAxis=None, pwv=None):
        """
        For a specified scan, antenna, pol, spw in the current caltable, 
        takes the low-resolution Trx spectrum and predicts a high resolution
        Trx and Tsys using the atmospheric model for this spw.  Trims resulting
        spectra to match the extent of the FDM spectrum.
        scan: integer or string
        antenna: integer ID, string ID or string name
        pol: 0 or 1, integer or string
        spw: Tsys spw (integer or string integer)
        fdmspw: science spw (integer or string integer), only the channel 
                width is used
        simulateTDMTsysScan: if > 0, then smooth the high-res Tsys scan data 
                to this many channels
        plotedge: skip this many edge channels when plotting Tsys
        separateFigures: if True, then open a new gui for each figure
        s: smoothing parameter passed to scipy.interpolate.UnivariateSpline when
          up-interpolating to high-res Tsys again
        mode: 'model', 'data', 'ratio', or 'difference'
        maxAltitude: of the atmosphere, in km
        highresEB: if the high-resolution data is in a different EB from the TDM Tsys
          scan, then set this parameter to the former
        newFDMFrequencyAxis: use this spectral grid to compute Tsys (e.g. from the proto-caltable)
        pwv: if specified, then use this value to override what was in the ASDM
        filterOrder: only used in simulating a TDM Tsys scan from a higher resolution one (e.g. ACA)
        Returns: trec, gain, tsky, freqHz, tcal, tsys, atmosphereLowRes, atmosphereHighRes, tsysLowRes
                  1D,   1D,   1D,    1D,    2D,  2D,   dictionary, dictionary, 2D
        Note: the ATM model spectrum is Hanning smoothed if self.hanningSmoothed[spw] == True,
            which is set automatically upon initialization of the Atmcal class.  To override
            this feature, one can set that dictionary entry to True or False before calling
            this function.
        """
        if fdmspw != spw and verbose:
            print("FDM spw = %d, Tsys spw = %d" % (fdmspw, spw))
        if separateFigures:
            pb.close('all')
        figctr = 0
        spw = int(spw)  # allow string integers
        pol = int(pol)  # allow string integers
        if (self.nonExistentTable(self.vis+'/ASDM_CALATMOSPHERE')): return
        if (self.unrecognizedAntenna(antenna)): return
        if (self.unrecognizedScan(scan)): return
        if (self.unrecognizedSpw(spw)): return
        if showpoints:
#            print "Setting marker to ."
            marker = '.'
        else:
#            print "Setting marker to (None)"
            marker = ''
        antennaId, antennaName = self.getAntenna(antenna)
        p0 = self.getSpectrum(scan,spw,pol,'amb',antennaId,dataFraction,ignoreFlags=ignoreFlags)
        if p0 is None:
            return
        p1 = self.getSpectrum(scan,spw,pol,'hot',antennaId,dataFraction,ignoreFlags=ignoreFlags)
        sky = self.getSpectrum(scan,spw,pol,'sky',antennaId,dataFraction,ignoreFlags=ignoreFlags)
        p0observed = p0
        p1observed = p1
        skyObserved = sky
        newFrequencyAxis = None
        if simulateTDMTsysScan > 0 and len(p0) not in [64,128,256]:
            mychanfreqs = self.mymsmd.chanfreqs(spw)
            window_len = len(p0) / simulateTDMTsysScan
            if verbose: print("Will smooth and decimate by %d with filter order %d" % (window_len, filterOrder))
            newFrequencyAxis, p0 = filterAndDecimate(mychanfreqs, p0, window_len, filterOrder)
            newFrequencyAxis, p1 = filterAndDecimate(mychanfreqs, p1, window_len, filterOrder)
            newFrequencyAxis, sky = filterAndDecimate(mychanfreqs, sky, window_len, filterOrder)
        else:
            if verbose: print("Not filtering since number of channels %d is in %s" % (len(p0),[64,128,256]))
            newFrequencyAxis = self.mymsmd.chanfreqs(spw)
        computeJsky = True
        startTime = timeUtilities.time()
        if (len(newFrequencyAxis) % 2 == 0 and casaVersion < '5.0'):
            # adjust by half a channel so that output spectrum from atm 
            # will match what is in the spectral window table 
            print("******* applying half channel frequency offset because this is CASA < 5.0")
            freqOffset = 0.5*self.mymsmd.chanwidths(spw)[0]
        else:
            freqOffset = 0
        result = self.computeJs(scan, antenna, pol, spw, asdm, etaF, lo1,
                                dataFraction, parentms, verbose, siteAltitude_m,
                                computeJsky, altscan=altscan, atmosphere=atmosphereLowRes,
                                calscandict=self.calscandict, newFrequencyAxis=newFrequencyAxis-freqOffset,
                                maxAltitude=maxAltitude, pwv=pwv)
        if verbose: print("time spent in computeJs lowres = %.1f seconds" % (timeUtilities.time() - startTime))
        if (result is None):
            print("computeJs returned None. Aborting")
            return None
        if computeJsky:
            JskyLowRes, t0, t1, freqHz, jatmDSBLowRes, jspDSBLowRes, jbgLowRes, tauALowRes, alphaLowRes, gb, atmosphereLowRes, tebbskyLowRes = result
            tebbskyLowRes = tebbskyLowRes[0]*gb[0] + tebbskyLowRes[1]*gb[1]
        else:
            JskyLowRes, t0, t1, freqHz = result
        if (abs(freqHz[0] - newFrequencyAxis[0]) > 1):
            print("low-res input freq[0]=%f output[0]=%f diff=%f" % (freqHz[0], newFrequencyAxis[0], freqHz[0]-newFrequencyAxis[0]))
        if verbose:
            print("len(newFrequencyAxis)=%d, len(p1)=%d, p0=%d, sky=%d, alphaLowRes=%d" % (len(newFrequencyAxis), len(p1),len(p0),len(sky),len(alphaLowRes)))
        tcalLowRes, tsysLowRes = self.solveTsys(p1, p0, sky, jatmDSBLowRes, jbgLowRes, gb, alphaLowRes, tauALowRes, verbose)
        if verbose:
            print("lowres sky: min=%f, max=%f, p0: min=%f, max=%f;  p1: min=%f, max=%f" % (np.nanmin(sky), np.nanmax(sky), np.nanmin(p0), np.nanmax(p0), np.nanmin(p1), np.nanmax(p1)))
        a = t1*p0 - t0*p1
        c = p1-p0
        b = t1-t0
        p0LowRes = p0
        p1LowRes = p1
        trecLowRes = a/c
        gainLowRes = c/b
        tskyLowRes = sky/gainLowRes - trecLowRes
        freqGHz = freqHz*1e-9
        suffix = 'spw%d_pol%d_%s' % (spw,pol,antennaName)
        plotTitle = self.vis + ', spw%d:%d, pol%d, scan%d, ant%d=%s' % (spw,fdmspw,pol,scan,antennaId,antennaName)
        if showplotLoads:
            figctr += 1
            if separateFigures: pb.figure(figctr)
            pb.clf()
            if verbose: print("shapes: ", np.shape(freqGHz), np.shape(sky), np.shape(p0), np.shape(p1))
            desc = pb.subplot(111)
            pb.plot(freqGHz, sky, 'b-')
#            pb.hold(True) # not needed
            pb.plot(freqGHz, p0, 'k-')
            pb.plot(freqGHz, p1, 'r-', mec='r')
            if simulateTDMTsysScan > 0:
                pb.plot(mychanfreqs*1e-9, p0observed, 'k--')
                pb.plot(mychanfreqs*1e-9, p1observed, 'r--', mec='r')
                pb.plot(mychanfreqs*1e-9, skyObserved, 'b--', mec='r')
                asciifile = 'sky_hot_ambient.txt'
                f = open(asciifile,'w')
                for i in range(len(mychanfreqs)):
                    f.write('%f %f\n' % (mychanfreqs[i]*1e-9, p0observed[i]))
                f.close()
                print("wrote ", asciifile)
            pb.xlabel('Frequency (GHz)')
            pb.ylabel('Counts')
            pb.title(plotTitle,size=12)
            pb.text(0.5,0.95,'Lowres: blue = sky,  red = hot_load,  black = ambient_load',
                    transform=desc.transAxes,ha='center')
            pb.text(0.75,0.01,'casa %s' % casaVersion, transform=desc.transAxes)
            addDateToPlot()
            pb.draw()
            png = 'sky_hot_ambient_lowres_%s.png' % (suffix)
            pb.savefig(png)
            print("Wrote ", png)
#            showplotLoads = False
#            showplot = False

        xaxis = freqHz
        if simulateTDMTsysScan > 0:
            tdmfreqs = newFrequencyAxis
            tdmwidth = self.mymsmd.chanwidths(spw)[0]*self.mymsmd.nchan(spw)/len(newFrequencyAxis)
#            print "Scaling native chanwidth of %f by *%d/%d to get simulated TDM width = %f" % (self.mymsmd.chanwidths(spw)[0],self.mymsmd.nchan(spw),len(newFrequencyAxis),tdmwidth)
        else:
            tdmfreqs = self.mymsmd.chanfreqs(spw)    
            tdmwidth = self.mymsmd.chanwidths(spw)[0]  # this will be negative in LSB


        if highresEB == '':
            mymsmd = self.mymsmd
        else:
            mymsmd = createCasaTool(msmdtool)
            mymsmd.open(highresEB)
        fdmwidth = mymsmd.chanwidths(fdmspw)[0]     # these will be negative in LSB
        if verbose:
            print("fdmwidth=%f, tdmwidth=%f" % (fdmwidth,tdmwidth))

        if newFDMFrequencyAxis is None:
            # You should really be passing this in from the proto-caltable. But this was the old way
            # of doing it, which got within 11.66 MHz.
            # compute edge to edge TDM freq, then offset to middle of first high-res fdm channel
            freqstart = tdmfreqs[0] - 0.5*tdmwidth + 0.5*fdmwidth
            freqstop = tdmfreqs[-1] + 0.5*tdmwidth - 0.5*fdmwidth 
            newFDMFrequencyAxis = np.linspace(freqstart, freqstop, len(tdmfreqs)*tdmwidth/fdmwidth) 
        if verbose:
            print("Defined newFDMFrequencyAxis from %f to %f by %f (mean=%f)" % (newFDMFrequencyAxis[0], newFDMFrequencyAxis[-1], newFDMFrequencyAxis[1]-newFDMFrequencyAxis[0],np.mean(newFDMFrequencyAxis)))
        fdmXaxis = newFDMFrequencyAxis # in Hz
        # Store the original spectra
        oldp0 = p0
        oldp1 = p1
        oldsky = sky

        # Resample the three original spectra to higher resolution
        # print "Resampling ambient,hot,sky to %d channels." % (len(fdmXaxis))
        if False:
            # spline method -- produces artifacts
            p0 = p0model(fdmXaxis)
            p1 = p1model(fdmXaxis)
            sky = skymodel(fdmXaxis)
            JskyLowResResampled = Jskymodel(fdmXaxis)
            tebbskyLowResResampled = tebbskymodel(fdmXaxis)
        else:
            reverse = False
            if xaxis[1] < xaxis[0]:
                # np.interp requires x-axis in increasing order
                reverse = True
                fdmXaxis = fdmXaxis[::-1]
                xaxis = xaxis[::-1]
                p0 = p0[::-1]
                p1 = p1[::-1]
                sky = sky[::-1]
                JskyLowRes = JskyLowRes[::-1]
                tebbskyLowRes = tebbskyLowRes[::-1]
            if verbose:
                if xaxis[0] < xaxis[1]:
                    print("spw %d: xaxis order is increasing" % (spw))
                else:
                    print("NOT EXPECTED !!!!  spw %d: xaxis order is decreasing" % (spw))
                if fdmXaxis[0] < fdmXaxis[1]:
                    print("spw %d: fdmXaxis order is increasing" % (spw))
                else:
                    print("NOT EXPECTED !!!!  spw %d: fdmXaxis order is decreasing" % (spw))
            p0 = np.interp(fdmXaxis, xaxis, p0)
            p1 = np.interp(fdmXaxis, xaxis, p1)
            sky = np.interp(fdmXaxis, xaxis, sky)
            JskyLowResResampled = np.interp(fdmXaxis, xaxis, JskyLowRes)
            tebbskyLowResResampled = np.interp(fdmXaxis, xaxis, tebbskyLowRes)
            if reverse:
                # Put them back to the original format
                xaxis = xaxis[::-1]
                fdmXaxis = fdmXaxis[::-1]
                p0 = p0[::-1]
                p1 = p1[::-1]
                sky = sky[::-1]
                JskyLowRes = JskyLowRes[::-1]
                tebbskyLowRes = tebbskyLowRes[::-1]
                # the following 2 lines were added on the plane
                JskyLowResResampled = JskyLowResResampled[::-1]
                tebbskyLowResResampled = tebbskyLowResResampled[::-1]
            if verbose:
                if xaxis[0] < xaxis[1]:
                    if fdmXaxis[0] < fdmXaxis[1]:
                        print("spw %d: xaxis and fdmXaxis orders match (increasing)" % (spw))
                    else:
                        print("NOT EXPECTED !!!!  spw %d: xaxis and fdmXaxis order is opposite" % (spw))
                if xaxis[0] > xaxis[1]:
                    if fdmXaxis[0] > fdmXaxis[1]:
                        print("spw %d: xaxis and fdmXaxis orders match (decreasing)" % (spw))
                    else:
                        print("NOT EXPECTED !!!!  spw %d: xaxis and fdmXaxis order is opposite" % (spw))
        if showplot:
            figctr += 1
            if separateFigures: pb.figure(figctr)
            pb.clf()
            desc = pb.subplot(111)
            pb.plot(fdmXaxis*1e-9,p0,'k-', fdmXaxis*1e-9,p1,'r-',fdmXaxis*1e-9,sky,'b-',
                    xaxis*1e-9,oldp0,'k--', xaxis*1e-9,oldp1,'r--',xaxis*1e-9,oldsky,'b--')
            pb.text(0.05,0.8,'solid: resampled to FDM',color='k',transform=desc.transAxes)
            pb.text(0.05,0.85,'dashed: smoothed to TDM',color='k',transform=desc.transAxes)
            if simulateTDMTsysScan > 0:
#                pb.hold(True) # not needed
                pb.text(0.05,0.9,'dotted: observed',color='k',transform=desc.transAxes)
                pb.plot(mychanfreqs*1e-9,p0observed,'k:', mychanfreqs*1e-9,p1observed,'r:', 
                        mychanfreqs*1e-9,skyObserved,'b:')
            pb.xlabel('Frequency (GHz)')
            pb.title(plotTitle,size=12)
            pb.ylabel('Counts (before scaling the sky scan)')
            addDateToPlot()
            pb.draw()
            png = 'sky_hot_ambient_lowres_highres_%s.png' % (suffix)
            pb.savefig(png)

        # compute a higher resolution Jsky, Trec and Tsys
        startTime = timeUtilities.time()
        if (len(newFDMFrequencyAxis) % 2 == 0 and casaVersion < '5.0'):
            # adjust by half a channel so that output spectrum from atm 
            # will match what we want
            print("******* applying half channel frequency offset because this is CASA < 5.0")
            freqOffset = 0.5*fdmwidth  # note that fdmwidth will be negative in LSB spws
        else:
            freqOffset = 0
        result = self.computeJs(scan, antenna, pol, spw, asdm, etaF, lo1,
                                dataFraction, parentms, verbose, siteAltitude_m,
                                computeJsky, altscan=altscan, calscandict=self.calscandict, 
                                atmosphere=atmosphereHighRes, newFrequencyAxis=newFDMFrequencyAxis-freqOffset,
                                maxAltitude=maxAltitude, pwv=pwv)
        if verbose: print("time spent in computeJs highres = %.1f seconds" % (timeUtilities.time() - startTime))
        if computeJsky:
            JskyHighRes, t0, t1, ignorefreqHz, jatmDSB, jspDSB, jbg, tauA, alpha, gb, atmosphereHighRes, tebbsky = result
            tebbskyDSB = tebbsky[0]*gb[0] + tebbsky[1]*gb[1]
        else:
            JskyHighRes, t0, t1, ignorefreqHz = result

        if showplot:
            pb.clf()
            desc = pb.subplot(111)
            pb.plot(fdmXaxis*1e-9, tebbskyDSB,'k-', fdmXaxis*1e-9,tebbskyLowResResampled,'r-', 
                    xaxis*1e-9, tebbskyLowRes,'b-')
            pb.ylabel('TebbskyDSB')
            pb.xlabel('Frequency (GHz)')
            pb.title(plotTitle,size=12)
            pb.text(0.05,0.85,'high-res',color='k',transform=desc.transAxes)
            pb.text(0.05,0.80,'low-res resampled',color='r',transform=desc.transAxes)
            pb.text(0.05,0.75,'low-res',color='b',transform=desc.transAxes)
            addDateToPlot()
            pb.draw()
            png = 'tebbsky_lowres_highres_%s.png' % (suffix)
            pb.savefig(png) 

        if (abs(ignorefreqHz[0] - newFDMFrequencyAxis[0]) > 1):
            print("input freq[0]=%f output[0]=%f" % (newFDMFrequencyAxis[0], ignorefreqHz[0]))
        if (abs(ignorefreqHz[-1] - newFDMFrequencyAxis[-1]) > 1):
            print("input freq[-1]=%f output[-1]=%f" % (newFDMFrequencyAxis[-1], ignorefreqHz[-1]))
        newfreqGHz = newFDMFrequencyAxis*1e-9
        # ignorefreqGHz is equal to newfreGHz
        if showplot:
            figctr += 1
            if separateFigures: pb.figure(figctr)
            pb.clf()
            desc = pb.subplot(211)
            pb.plot(newfreqGHz, JskyHighRes, 'r-', freqGHz, JskyLowRes, 'k%s-'%marker, 
                    newfreqGHz, JskyLowResResampled, 'b%s-'%marker)
            pb.text(0.02,0.9,'low resolution (%d pts)'%(len(JskyLowRes)),color='k',transform=desc.transAxes)
            pb.text(0.02,0.85,'low resolution resampled',color='b',transform=desc.transAxes)
            pb.text(0.02,0.8,'high resolution (%d pts)'%(len(JskyHighRes)),color='r',transform=desc.transAxes)
            yFormatter = ScalarFormatter(useOffset=False)
            desc.xaxis.set_major_formatter(yFormatter)
            pb.xlim([np.min(newfreqGHz),np.max(newfreqGHz)])
            pb.xlabel('Frequency (GHz)')
            pb.ylabel('Jsky')
            pb.title(plotTitle,size=12)

            desc = pb.subplot(212)
            pb.plot(newfreqGHz, JskyHighRes-JskyLowResResampled, 'k-')
            desc.xaxis.set_major_formatter(yFormatter)
            pb.ylabel('Residual (JskyHighRes-LowResResampled)')
            pb.xlabel('Frequency (GHz)')
            pb.xlim([np.min(newfreqGHz),np.max(newfreqGHz)])
            addDateToPlot()
            pb.draw()
            png = 'jsky_lowres_highres_%s.png' % (suffix)
            pb.savefig(png)
            print("Wrote ", png)

            figctr += 1
            if separateFigures: pb.figure(figctr)
            pb.clf()
            desc = pb.subplot(111)
            pb.plot(freqGHz, alphaLowRes, 'k-',newfreqGHz, alpha, 'r-')
            pb.text(0.1,0.9,'low resolution',color='k',transform=desc.transAxes)
            pb.text(0.1,0.8,'high resolution',color='r',transform=desc.transAxes)
            pb.xlabel('Frequency (GHz)')
            pb.ylabel('alpha')
            pb.title(plotTitle,size=12)
            addDateToPlot()
            pb.draw()
            png = 'alpha_%s.png' % (suffix)
            pb.savefig(png)
            print("Wrote ", png)

            figctr += 1
            if separateFigures: pb.figure(figctr)
            pb.clf()
            desc = pb.subplot(111)
            pb.plot(freqGHz, jatmDSBLowRes[0]*gb[0]+jatmDSBLowRes[1]*gb[1], 'k-',
                    newfreqGHz, jatmDSB[0]*gb[0]+jatmDSB[1]*gb[1], 'r-')
            pb.text(0.1,0.9,'low resolution',color='k',transform=desc.transAxes)
            pb.text(0.1,0.8,'high resolution',color='r',transform=desc.transAxes)
            pb.xlabel('Frequency (GHz)')
            pb.ylabel('JatmDSB (sum of both sidebands)')
            pb.title(plotTitle,size=12)
            addDateToPlot()
            pb.draw()
            png = 'JatmDSB_%s.png' % (suffix)
            pb.savefig(png)
            print("Wrote ", png)

            if False:
                figctr += 1
                if separateFigures: pb.figure(figctr)
                pb.clf()
                desc = pb.subplot(111)
                pb.plot(freqGHz, jspDSBLowRes, 'k-',newfreqGHz, jspDSB, 'r-')
                pb.text(0.1,0.9,'low resolution',color='k',transform=desc.transAxes)
                pb.text(0.1,0.8,'high resolution',color='r',transform=desc.transAxes)
                pb.xlabel('Frequency (GHz)')
                pb.ylabel('JspDSB')
                pb.title(plotTitle,size=12)
                addDateToPlot()
                pb.draw()
                png = 'JspDSB_%s.png' % (suffix)
                pb.savefig(png)
                print("Wrote ", png)

                figctr += 1
                if separateFigures: pb.figure(figctr)
                pb.clf()
                desc = pb.subplot(111)
                pb.plot(freqGHz, jbgLowRes[0], 'k-',newfreqGHz, jbg[0], 'r-')
                pb.text(0.1,0.9,'low resolution',color='k',transform=desc.transAxes)
                pb.text(0.1,0.8,'high resolution',color='r',transform=desc.transAxes)
                pb.xlabel('Frequency (GHz)')
                pb.ylabel('Jbg')
                pb.title(plotTitle,size=12)
                addDateToPlot()
                pb.draw()
                png = 'Jbg_%s.png' % (suffix)
                pb.savefig(png)
                print("Wrote ", png)

            figctr += 1
            if separateFigures: pb.figure(figctr)
            pb.clf()
            desc = pb.subplot(111)
            pb.plot(freqGHz, tauALowRes[0], 'k-', newfreqGHz, tauA[0], 'r-')
            pb.text(0.1,0.9,'low resolution',color='k',transform=desc.transAxes)
            pb.text(0.1,0.8,'high resolution',color='r',transform=desc.transAxes)
            pb.xlabel('Frequency (GHz)')
            pb.ylabel('tauA')
            pb.title(plotTitle,size=12)
            addDateToPlot()
            pb.draw()
            png = 'tauA_%s.png' % (suffix)
            pb.savefig(png)
            print("Wrote ", png)

        if False:
            skyRatio = JskyHighRes/JskyLowResResampled
            figctr += 1
            if separateFigures: pb.figure(figctr)
            pb.clf()
            desc = pb.subplot(111)
            print("Ignoring %d edge pixels on each side" % (plotedge))
            pb.plot(newfreqGHz[plotedge:-plotedge], skyRatio[plotedge:-plotedge], 'k%s-'%marker)
            yFormatter = ScalarFormatter(useOffset=False)
            desc.yaxis.set_major_formatter(yFormatter)
            pb.title(plotTitle,size=12)
            pb.ylabel('Jsky ratio (highres / lowres)')
            pb.xlabel('Frequency (GHz)')
            addDateToPlot()
            png = 'Jsky_ratio_lowres_highres_%s.png' % (suffix)
            pb.draw()
            pb.savefig(png)
            print("Wrote ", png)

        # compute higher resolution Trec and Tsys
        a = t1*p0 - t0*p1
        c = p1-p0
        b = t1-t0
        trec = a/c
        gain = c/b
        print("spw %d: Median gain = " % (spw), np.median(gain))
        if mode == 'data':
            # Try to transfer the high resolution features from atmospheric model (as encoded 
            # into JskyHighRes) into the sky scan
            skyIncrement = (JskyHighRes - JskyLowResResampled)*np.median(sky)/np.median(JskyLowResResampled)
            percentage = np.max(skyIncrement)*100/np.median(sky)
            print("Sky: median=%f,  max increment added: %f (%.3f%%)" % (np.median(sky), np.max(skyIncrement), percentage))
            sky += skyIncrement
            tsky = sky/gain - trec
        elif mode == 'ratio':
            tsky = sky/gain - trec
            # Compute high-resolution sky scan
            sky = (tsky*tebbskyDSB/tebbskyLowResResampled + trec)*gain  
            tsky = sky/gain - trec
        elif mode == 'difference':
            tsky = sky/gain - trec
            # Compute high-resolution sky scan: add full-res model and subtract low-res model
            sky = (tsky + tebbskyDSB - tebbskyLowResResampled + trec)*gain  
            tsky = sky/gain - trec
        elif mode == 'model':
            # Just use the Tsky from the model, and infer a theoretical sky spectrum
            tsky = tebbskyDSB
            print("**** Setting Tsky spectrum to signal sideband model, median = %f" % (np.median(tsky)))
            sky = (tsky + trec)*gain
        if verbose: print("highres: sky: min=%f, max=%f, p0: min=%f, max=%f;  p1: min=%f, max=%f" % (np.nanmin(sky), np.nanmax(sky), np.nanmin(p0), np.nanmax(p0), np.nanmin(p1), np.nanmax(p1)))

        tcal, tsys = self.solveTsys(p1, p0, sky, jatmDSB, jbg, gb, alpha, tauA, verbose)
        if showplot:
            figctr += 1
            if separateFigures: pb.figure(figctr)
            pb.clf()
            desc = pb.subplot(111)
            if verbose: print("Ignoring %d edge pixels on each side" % (plotedge))
            pb.plot(freqGHz[plotedge:-plotedge], tsysLowRes[plotedge:-plotedge], 'k%s-'%marker)
            n = len(tsysLowRes)/10
            idx = np.argmax(tsysLowRes[n:-n])
            if listpeak:
                print("peak of low-res spectrum:  %fK at %f GHz" % (tsysLowRes[idx+n],freqGHz[idx+n]))
#            pb.hold(True) # not needed
            pb.plot(newfreqGHz[plotedge:-plotedge], tsys[plotedge:-plotedge], 'r%s-'%marker, mec='r')
            n = len(tsys)/10
            idx = np.argmax(tsys[n:-n])
            if listpeak:
                print("peak of high-res spectrum: %fK at %f GHz" % (tsys[idx+n],newfreqGHz[idx+n]))
            pb.text(0.1,0.9,'low resolution',color='k',transform=desc.transAxes)
            pb.text(0.1,0.8,'high resolution',color='r',transform=desc.transAxes)
            pb.title(plotTitle,size=12)
            pb.ylabel('Tsys (K)')
            pb.xlabel('Frequency (GHz)')
            addDateToPlot()
            pb.draw()
            png = 'Tsys_lowres_highres_%s.png' % (suffix)
            pb.savefig(png)
            print("Wrote ", png)
        if showplotLoads:
            figctr += 1
            if separateFigures: pb.figure(figctr)
            pb.clf()
            desc = pb.subplot(111)
            if verbose: print("shapes: ", np.shape(freqGHz), np.shape(sky), np.shape(p0), np.shape(p1))
            pb.plot(newfreqGHz, sky, 'b-', newfreqGHz, p0, 'k-', newfreqGHz, p1, 'r-')
            pb.xlabel('Frequency (GHz)')
            pb.ylabel('Counts (after scaling the sky scan)')
            pb.title(plotTitle,size=12)
            pb.text(0.5,0.95,'Highres: blue = sky,  red = hot_load,  black = ambient_load',transform=desc.transAxes,ha='center')
            addDateToPlot()
            pb.draw()
            png = 'sky_hot_ambient_highres_%s.png' % (suffix)
            pb.savefig(png)
            print("Wrote ", png)
            pb.clf()
            pb.plot(xaxis*1e-9, oldsky, 'k-', newfreqGHz, sky, 'r-')
            pb.text(0.5,0.95,'Low-res sky: black,  high-res sky: red',transform=desc.transAxes,ha='center')
            pb.xlabel('Frequency (GHz)')
            pb.ylabel('Sky counts')
            pb.title(plotTitle,size=12)
            addDateToPlot()
            pb.draw()
            png = 'oldsky_newsky_%s.png' % (suffix)
            print("Wrote ", png)
            pb.savefig(png)

        fdmnchan = mymsmd.nchan(fdmspw)
        if highresEB != '':
            mymsmd.close()
        edge = (len(trec)-fdmnchan)/2
        # Note that a 468 MHz spw will require 16384 channels to span, so literally
        # thousands of channels need to be trimmed from each side for narrow FDM spws.
        if edge > 0:
            trec = trec[edge:-edge]
            if verbose:
                print("Trimming %d pixels from each edge to be width = %d" % (edge,len(trec)))
            tsky = tsky[edge:-edge]
            tsys = tsys[edge:-edge]
            tcal = tcal[edge:-edge]
            newFDMFrequencyAxis = newFDMFrequencyAxis[edge:-edge]
        if verbose:
            idx = np.argmax(tsky)
            print("simulateHighResTsys(): Peak Tsky=%f at freq=%f" % (tsky[idx], newFDMFrequencyAxis[idx]))
        return(trec, gain, tsky, newFDMFrequencyAxis, tcal, tsys, atmosphereLowRes, atmosphereHighRes, tsysLowRes)

    def joinNearbyWindows(self, selection, signal=None, sigma=0, verbose=False):
        """
        Takes [1,2,3,4,5,8,9,10,11] and converts to [1,2,3,4,5,6,7,8,9,10,11], i.e. removes the gap
        if mean of the signal[6:7] minues mean of the signal[4,5,8,9]
        is more than sigma*np.std([4,5,8,9])
        selection: a list or array of consecutive integer values that must be within 0..len(signal)-1
        signal: a list or array of all y-axis values (with no gaps)
        sigma: if > 0, then only join if gapSignal is larger than sigma*std, otherwise join if 
            gap is narrower than both neighbors
        """
        if signal is not None:
            signal = np.array(signal) # e.g. [0,1,1,1,1,1,5,5,1,2,3,4]   of length 12
        idxs = splitListIntoContiguousLists(selection)
        droppedWindows = 0
        newselection = []
        skipNext = False
        for i in range(len(idxs)-1):
            if skipNext:
                skipNext = False
                continue
            idx = idxs[i]        # e.g. [1,2,3,4,5]
            nextidx = idxs[i+1]  # e.g. [8,9,10,11]
            gapWidth = nextidx[0]-idx[-1]-1  # eg. 2
            if sigma > 0:
                x0 = np.max([0,len(idx)-gapWidth])  # e.g.  3  idx[3] = 4
                x1 = np.min([len(nextidx)-1, gapWidth])  # e.g. 2 nextidx[2] = 10
                leftside = signal[idx[x0]:idx[x0]+gapWidth] #  [1,1]
                rightside = signal[nextidx[:x1]]            #  [1,2]
                sides = list(leftside) + list(rightside)    #  [1,1,1,2]
                gap = np.arange(idx[-1]+1, nextidx[0])      #  [6,7]
                if verbose: print("gap=%s" % (str(gap)))
                gapSignal = np.mean(signal[gap])        #  5.0
                gapNetSignal = gapSignal - np.mean(sides)  # 3.75
                drop = gapNetSignal > sigma*np.std(sides)    #  3.75 > sigma*0.433  True
                if verbose: 
                    print("gapSignal=%f  gapNetSignal=%f  gapNetSignal/std = %f" % (gapSignal, gapNetSignal, gapNetSignal/np.std(sides)))
            else:
                if len(idx) > gapWidth and len(nextidx) > gapWidth:
                    drop = True
                else:
                    drop = False
            if drop:
                newselection += range(idx[0],nextidx[-1]+1)
                droppedWindows += 1
                skipNext = True
            else:
                newselection += idx
                if i == len(idxs)-2:
                    newselection += nextidx
        if droppedWindows > 0: 
            print("Dropped %d/%d windows because they were narrow." % (droppedWindows,len(idxs)))
#        print "n=%d, unique=%d" % (len(newselection), len(np.unique(newselection)))
        newselection = np.unique(newselection)
        return newselection

    def computeTrec2(self, scan, antenna, pol, spw, tdmspw=None, asdm=None, etaF=0.98, lo1=None,
                     dataFraction=[0.0, 1.0], parentms=None, verbose=False,
                     siteAltitude_m=5059, computeJsky=False, altscan=None,ignoreFlags=False,
                     calscandict=None, fdmCorrection=False, tdmscan=None, tdmdataset=None,
                     takeLoadsFromTdmDataset=False, showplot=False, atmosphere=None):
        """
        Compute Trec and saturation parameter from a 2-load measurement.
        This is the normal ALMA calibration procedure. The formula is:
        p = G * (tRec+t)
        If computeJsky is True, then it also computes Tsys.
        """
        spw = int(spw)
        if (self.nonExistentTable(self.vis+'/ASDM_CALATMOSPHERE')): return
        if (self.unrecognizedAntenna(antenna)): return
        if (self.unrecognizedScan(scan)): return
        if (self.unrecognizedSpw(spw)): return
        antennaId, antennaName = self.getAntenna(antenna)
        p0 = self.getSpectrum(scan,spw,pol,'amb',antennaId,dataFraction,ignoreFlags=ignoreFlags)
        p1 = self.getSpectrum(scan,spw,pol,'hot',antennaId,dataFraction,ignoreFlags=ignoreFlags)
        sky = self.getSpectrum(scan,spw,pol,'sky',antennaId,dataFraction,ignoreFlags=ignoreFlags)
        print("sky: min=%f, max=%f, p0: min=%f, max=%f;  p1: min=%f, max=%f" % (np.nanmin(sky), np.nanmax(sky), np.nanmin(p0), np.nanmax(p0), np.nanmin(p1), np.nanmax(p1)))
        if (takeLoadsFromTdmDataset):
            tdmAntennaId, antennaName = tdmdataset.getAntenna(antennaName)
            print("Translated %s from id=%d in one dataset to %d in the other (%s)" % (antennaName,antennaId,tdmAntennaId,antennaName))
            amb = tdmdataset.getSpectrum(tdmscan,tdmspw,pol,'amb',tdmAntennaId,dataFraction,ignoreFlags=ignoreFlags)
            hot = tdmdataset.getSpectrum(tdmscan,tdmspw,pol,'hot',tdmAntennaId,dataFraction,ignoreFlags=ignoreFlags)
            scalingFactor = np.median(amb)/np.median(p0)
            p0 = amb
            p1 = hot
            print("Applying scaling factor %f = %.2f dB to the sky subscan." % (scalingFactor, 10*np.log10(scalingFactor)))
            sky *= scalingFactor
        if (fdmCorrection):
            if (tdmscan == None): tdmscan = scan
            if (tdmspw == None): tdmspw = spw
            if (tdmdataset == None):
                tdmdataset = self
            tdmSpectrumAmb = tdmdataset.getSpectrum(tdmscan,tdmspw,pol,'amb',antennaId,dataFraction,ignoreFlags=ignoreFlags)
            tdmSpectrumHot = tdmdataset.getSpectrum(tdmscan,tdmspw,pol,'hot',antennaId,dataFraction,ignoreFlags=ignoreFlags)
            tdmSpectrumSky = tdmdataset.getSpectrum(tdmscan,tdmspw,pol,'sky',antennaId,dataFraction,ignoreFlags=ignoreFlags)
            print("Applying FDM correction using %d-channel total power" % (len(tdmSpectrumAmb)))
            p0 = self.applyCorrectionToFDMSpectrum(p0, tdmSpectrumAmb, tdmSpectrumAmb)
            p1 = self.applyCorrectionToFDMSpectrum(p1, tdmSpectrumHot, tdmSpectrumHot)
            sky = self.applyCorrectionToFDMSpectrum(sky, tdmSpectrumSky, tdmSpectrumSky)
        if (type(p0) != np.ndarray and type(p0) != np.ma.core.MaskedArray):
            print("type(p0) = %s, len(p0) = %d" % (str(type(p0)), len(p0)))
            return None
        result = self.computeJs(scan, antenna, pol, spw, asdm, etaF, lo1,
                                dataFraction, parentms, verbose, siteAltitude_m,
                                computeJsky, altscan=altscan, 
                                calscandict=calscandict, atmosphere=atmosphere)
        if (computeJsky):
#            skyResult, t0, t1, freqHz, jatmDSB, jspDSB, jbg, tauA, alpha, gb, atmosphere = result
#            tcal, tsys = self.solveTsys(p1, p0, sky, jatmDSB, jbg, gb, alpha, tauA, verbose)
            skyResult, t0, t1, freqHz, jatm, jspDSB, jbg, tauA, alpha, gb, atmosphere, tebbsky = result
            tcal, tsys = self.solveTsys(p1, p0, sky, jatm, jbg, gb, alpha, tauA, verbose)
        else:
            skyResult, t0, t1, freqHz = result
        a = t1*p0-t0*p1
        c = p1-p0
        b = t1-t0
        trec = a/c
        gain = c/b
        print("mean gain (deltaLoadPower/deltaT) = %f counts/Kelvin" % (np.mean(gain)))
        tsky = sky/gain - trec
        if (showplot):
            freqGHz = freqHz*1e-9
            pb.clf()
            pb.plot(freqGHz, sky, 'b-', freqGHz, p0, 'k-', freqGHz, p1, 'r-')
            pb.xlabel('Frequency (GHz)')
            pb.ylabel('Counts')
            pb.title('blue = sky,  red = hot_load,  black = ambient_load')
            pb.draw()
        if (computeJsky):
            return(trec, gain, tsky, freqHz, tcal, tsys, atmosphere)
        else:
            return(trec, gain, tsky, freqHz)

    def plotTrec2All(self, spw, tdmspw=None, asdm=None, etaF=0.98, lo1=None,
                     dataFraction=[0.0, 1.0], parentms=None, siteAltitude_m=5059):
        spw = int(spw)
        pngs = []
        if (self.unrecognizedSpw(spw)): return
        for scan in self.scans:
            print("Working on scan ", scan)
            for antenna in self.antennas:
                print("Working on antenna %d/%d" % (antenna+1,len(self.antennas)))
                for pol in range(2):
                    png = self.plotTrec2(scan,antenna,pol,spw, tdmspw, asdm, etaF, lo1,
                                         dataFraction, parentms, siteAltitude_m=siteAltitude_m)
                    if (png is not None):
                        pngs.append(png)
        buildPdfFromPngs(pngs, pdfname='%s.trec2.pdf'%(self.vis))

    def plotTsysTrec2AllMedian(self, spw=None, scans=None, tdmspw=None, asdm=None, 
                               antennas=None, etaF=0.98, lo1=None,
                               dataFraction=[0.0, 1.0], parentms=None, siteAltitude_m=5059,
                               fdmCorrection=False, tdmscan=None, tdmdataset=None, showAttenuators=False,
                               buildPDF=True, verbose=False):
        """
        Plots channel-medianed Trx and Tsys vs. antenna ID on a 2-panel plot, with one plot 
        per spw/scan combination (the two pols are shown in green and blue).
        """
        if (spw is not None):
            spw = int(spw)
            if (self.unrecognizedSpw(spw)): return
            spws = [spw]
        else:
            spws = self.spws
        if (scans == None):
            scans = self.scans
        elif type(scans) == str:
            scans = [int(i) for i in scans.split(',')]
        if (antennas is None):
            antennas = self.antennas
        elif (type(antennas) == str):
            antennas = [int(i) for i in antennas.split(',')]
        pngs = []
        trxDifferences = {'antenna': {}, 'spw':{}}
        tsysDifferences = {'antenna': {}, 'spw':{}}
        computeJsky = True # must be True when asking for Tsys
        for spw in spws:
          for scan in scans:
            print("Working on scan ", scan)
            tsys = []
            trec = []
            atmosphere = None
            for antenna in antennas:
                print("Working on spw %d antenna %s (%d/%d)" % (spw, self.antennaNames[antenna], antenna+1,len(antennas)))
                tsyspol = []
                trecpol = []
                for pol in range(2):
                    result = self.computeTsys(scan,antenna,pol,spw, tdmspw, asdm, etaF, lo1,
                                              dataFraction, parentms, verbose, siteAltitude_m,
                                              computeJsky, fdmCorrection=fdmCorrection, tdmscan=tdmscan,
                                              tdmdataset=tdmdataset, atmosphere=atmosphere)
                    atmosphere = result[-1]
                    tsyspol.append(np.median(result[0]))
                    trecpol.append(np.median(result[2]))
                tsys.append(tsyspol)
                trec.append(trecpol)
            pb.clf()
            adesc = pb.subplot(211)
            pb.plot(range(len(antennas)), trec, 'o', color=overlayColors[0])
            pb.ylabel('Trec')
            pb.title(os.path.basename(self.vis) + ' spw %d scan %d' % (spw,scan))
            xlim = pb.xlim()
            pb.xlim([xlim[0]-0.5, xlim[1]+0.5])
            xlim = pb.xlim()
            adesc.xaxis.set_major_locator(MultipleLocator(1))
            adesc.set_xticklabels(self.antennaNames[antennas], rotation='vertical', size=8, ha='left')
            adesc = pb.subplot(212)
            pb.plot(range(len(antennas)), tsys, 'o', color=overlayColors[1])
            pb.xlim(xlim)
            pb.xlabel('Antenna')
            pb.ylabel('Tsys')
            adesc.xaxis.set_major_locator(MultipleLocator(1))
            adesc.set_xticklabels(self.antennaNames[antennas], rotation='vertical', size=8, ha='left')
            png = self.vis+'_spw%d_scan%d_tsys.png'%(spw,scan)
            pb.savefig(png)
            pngs.append(png)
        pdf = self.vis+'_tsys.pdf'
        buildPdfFromPngs(pngs,pdf)

    def plotTsysTrec2All(self, spw=None, tdmspw=None, asdm=None, antennas=None, etaF=0.98, lo1=None,
                         dataFraction=[0.0, 1.0], parentms=None, siteAltitude_m=5059,
                         fdmCorrection=False, tdmscan=None, tdmdataset=None, showAttenuators=False):
        """
        Plots Tsys in one panel and Trec2 in another panel on the same page, for all requested
        combinations of antennas and spws.
        spw: a single spw ID (integer or string)
        antennas: an integer list of antenna IDs, or a comma-delimited string
        """
        if (spw is not None):
            spw = int(spw)
            if (self.unrecognizedSpw(spw)): return
            spws = [spw]
        else:
            spws = self.spws
        if (scans == None):
            scans = self.scans
        elif type(scans) == str:
            scans = [int(i) for i in scans.split(',')]
        if (antennas is None):
            antennas = self.antennas
        elif (type(antennas) == str):
            antennas = [int(i) for i in antennas.split(',')]
        pngs = []
        trxDifferences = {'antenna': {}, 'spw':{}}
        tsysDifferences = {'antenna': {}, 'spw':{}}
        for spw in spws:
          for scan in self.scans:
            print("Working on scan ", scan)
            for antenna in antennas:
                print("Working on spw %d antenna %s (%d/%d)" % (spw, self.antennaNames[antenna], antenna+1,len(self.antennas)))
                for pol in range(2):
                    png = self.plotTsysTrec2(scan,antenna,pol,spw, tdmspw, asdm, etaF, lo1,
                                             dataFraction, parentms, siteAltitude_m=siteAltitude_m,
                                             fdmCorrection=fdmCorrection, tdmscan=tdmscan,
                                             tdmdataset=tdmdataset, showAttenuators=showAttenuators,
                                             trxDifferences=trxDifferences, tsysDifferences=tsysDifferences)
                    if (png is not None):
                        pngs.append(png)
        buildPdfFromPngs(pngs, pdfname='%s.tsys_trx.pdf'%(self.vis))
        for spw in spws:
          if (spw in trxDifferences['spw']):
            print("               25%ile  median  75%ile")
            print("Trx: spw %02d = %.2f  %.2f  %.2fK" % (spw,scoreatpercentile(trxDifferences['spw'][spw],25),
                                                         np.median(trxDifferences['spw'][spw]),
                                                         scoreatpercentile(trxDifferences['spw'][spw],75)))
            print("Tsys spw %02d = %.2f  %.2f  %.2fK" % (spw,scoreatpercentile(tsysDifferences['spw'][spw],25),
                                                         np.median(tsysDifferences['spw'][spw]),
                                                         scoreatpercentile(tsysDifferences['spw'][spw],75)))
        for antenna in antennas:
          if (antenna in trxDifferences['antenna']):
            print("Trx: antenna %02d (%s) = %.2f  %.2f  %.2fK" % (antenna,self.antennaNames[antenna],
                                                                  scoreatpercentile(trxDifferences['antenna'][antenna],25),
                                                                  np.median(trxDifferences['antenna'][antenna]),
                                                                  scoreatpercentile(trxDifferences['antenna'][antenna],75)))
            print("Tsys antenna %02d (%s) = %.2f  %.2f  %.2fK" % (antenna,self.antennaNames[antenna],
                                                                  scoreatpercentile(tsysDifferences['antenna'][antenna],25),
                                                                  np.median(tsysDifferences['antenna'][antenna]),
                                                                  scoreatpercentile(tsysDifferences['antenna'][antenna],75)))
                                                                  
        return(trxDifferences, tsysDifferences)
    
    def plotTrec2(self, scan, antenna, pol, spw, tdmspw=None, asdm=None, etaF=0.98,
                  lo1=None, dataFraction=[0.0, 1.0], parentms=None, 
                  siteAltitude_m=5059, altscan=None, overlayTelcal=True,
                  fdmCorrection=False, tdmscan=None, tdmdataset=None, plotfile='',
                  showAttenuators=False):
        """
        Computes Trec2 then plots it, and optionally overlays the Telcal result.
        """
#        if (tdmspw is not None or tdmdataset is not None or tdmspw is not None):
#            fdmCorrection = True
        spw = int(spw)
        if (self.unrecognizedAntenna(antenna)): return
        if (self.unrecognizedScan(scan)): return
        if (spw == 'auto'):
            spw = self.spwsforscan[scan][0]
            print("Choosing spw = %d" % (spw))
        if (self.unrecognizedSpw(spw)): return
        antennaId, antennaName = self.getAntenna(antenna)
#        pb.text(0.65, 0.93, 'fractional portion of subscan used', transform=adesc.transAxes)
        if (overlayTelcal):
            trx = self.getTelcalTrx(antenna,spw,scan,pol)  # TelCal's result
            if (trx == None):
                overlayTelcal = False
        result = self.computeTrec2(scan, antenna, pol, spw, tdmspw, asdm=None,
                                   etaF=etaF, lo1=None, dataFraction=dataFraction,
                                   parentms=None, siteAltitude_m=5059, altscan=altscan,
                                   fdmCorrection=fdmCorrection, tdmscan=tdmscan,
                                   tdmdataset=tdmdataset)
        if (result == None):
            return(None)
        trec, gain, tsky, freqHz = result 
        pb.clf()
        adesc = pb.subplot(111)
        freqHz = self.chanfreqs[spw]
        pb.plot(freqHz*1e-9, trec, 'k-')
        pb.text(0.82, 0.85-c*0.07, '[%.2f,%.2f]' % (dataFraction[0],dataFraction[1]),
                color='k', transform=adesc.transAxes)
        pb.xlabel('Frequency (GHz)')
        pb.ylabel('Trec2(K)')
        ylims=pb.ylim()
#        if (ylims[0] < 0):
#            pb.ylim([0,ylims[1]])
        if (overlayTelcal):
#            pb.hold(True) # not needed
            if (fdmCorrection):
                lw = 1
            else:
                lw = 3
            pb.plot(freqHz*1e-9, trx, 'g-', lw=lw)
            print("mean residual = ", np.mean(trx-trec))
            if (fdmCorrection):
                mylabel = ' FDM'
            else:
                mylabel = ''
            pb.text(0.05,0.95, 'TelCal'+mylabel, color='g', transform=adesc.transAxes)
            pb.text(0.42,0.95, 'casa'+mylabel, color='k', transform=adesc.transAxes)
            if (showAttenuators):
                if (self.IFProc[antenna][pol] == None or self.IFSwitch[antenna][pol][1] == None):
                    self.readAttenuatorSettings(antenna,pol)
                if (scan in self.IFProc[antenna][pol]):
                  if (self.IFProc[antenna][pol][scan] != -1):
                    pb.text(0.05,0.85,'IFSw %.1fdB, IFPr %.1fdB'%(self.IFSwitch[antenna][pol][self.sidebandsforspw[spw]][scan],
                                                   self.IFProc[antenna][pol][scan][self.basebands[spw]-1]),
                        color='g',transform=adesc.transAxes)
            if (fdmCorrection):
                if (tdmspw == None): tdmspw = spw
                if (tdmscan == None): tdmscan = scan
                if (tdmdataset == None):
                    tdmdataset = self
                trx = tdmdataset.getTelcalTrx(antenna,tdmspw,tdmscan,pol)  # TelCal's result for the TDM spectrum
#                print "Got TelcalTrx for spw=%d: " % (tdmspw, trx)
                freqHz = tdmdataset.chanfreqs[tdmspw]
                pb.plot(freqHz*1e-9, trx, 'r-')
                y0,y1 = pb.ylim()
                if (y0 < 0 and y0 > -100000): y0 = 0
                pb.ylim([y0,y1+(y1-y0)*0.2])
                pb.text(0.65,0.95, 'TelCal TDM (%s)' % (mjdsecToUTHMS(np.mean(tdmdataset.timerange[tdmscan]))),
                        color='r', transform=adesc.transAxes)
                if (showAttenuators):
                    if (tdmdataset.IFProc[antenna][pol] == None or tdmdataset.IFSwitch[antenna][pol][1] == None):
                        tdmdataset.readAttenuatorSettings(antenna,pol)
                    if (tdmscan in tdmdataset.IFProc[antenna][pol]):
                      if (tdmdataset.IFProc[antenna][pol][tdmscan] != -1):
                        pb.text(0.65,0.85,'IFSw %.1fdB, IFPr %.1fdB'%(tdmdataset.IFSwitch[antenna][pol][tdmdataset.sidebandsforspw[spw]][tdmscan],
                                                       tdmdataset.IFProc[antenna][pol][tdmscan][tdmdataset.basebands[spw]-1]),
                            color='r',transform=adesc.transAxes)
        ut = mjdsecToUTHMS(self.meantime[scan])
        adesc.xaxis.grid(True,which='major')
        adesc.yaxis.grid(True,which='major')
        pb.title('%s  %s  scan=%d  spw=%d  pol=%d  mean_time=%s' % (os.path.basename(self.vis), antennaName, scan, spw, pol, ut), fontsize=11)
        if (plotfile == '' or plotfile==True):
            png = '%s.%s.scan%02d.spw%02d.pol%d.trec2.png' % (os.path.basename(self.vis), antennaName, scan, spw, pol)
        else:
            png = plotfile
        print("Result left in ", png)
        pb.savefig(png)
        pb.draw()
        return(png)
                  
    def computeTrec3(self, scan, antenna, pol, spw, asdm=None, etaF=0.98, lo1=None,
                     dataFraction=[0.0, 1.0], parentms=None, verbose=False,
                     altscan=None, calscandict=None):
        """
        This function is not really used in TelCal (computeTrec2 is used).
        Trec and saturation parameter from a 3-load measurement.
        the formula is
             p = G * (tRec+t) / (1+s*t)
        """
        spw = int(spw)
        if (self.unrecognizedAntenna(antenna)): return
        if (self.unrecognizedScan(scan)): return
        if (self.unrecognizedSpw(spw)): return
        antennaId, antennaName = self.getAntenna(antenna)
        result = self.computeJs(scan, antenna, pol, spw, asdm, etaF, lo1,
                                dataFraction, parentms, verbose,
                                computeJsky=True, altscan=altscan, 
                                calscandict=calscandict)
        t0, t1, t2, freqHz, jatm, jspDSB, jbg, tauA, alpha, gb, atmosphere, tebbsky = result
        p0 = self.getSpectrum(scan,spw,pol,'sky',antennaId,dataFraction)
        p1 = self.getSpectrum(scan,spw,pol,'amb',antennaId,dataFraction)
        p2 = self.getSpectrum(scan,spw,pol,'hot',antennaId,dataFraction)
        a = p0*p1*t2*(t0-t1) + p1*p2*t0*(t1-t2) + p2*p0*t1*(t2-t0)
        c = p0*p1*(t1-t0) + p1*p2*(t2-t1) + p2*p0*(t0-t2)
        b = p0*t0*(t2-t1) + p1*t1*(t0-t2) + p2*t2*(t1-t0)
        d = p0*(t1-t2) + p1*(t2-t0) + p2*(t0-t1)
        if (True):
            trec = a/c
            gain = c/b
            saturation = d/b
        else:
            # DBL_MIN taken from aos-gns:/alma/ACS-12.0/ACSSW/Sources/xercesc/src/xerces-c-src_2_8_0/tests/XSValueTest/XSValueTest.cpp
            DBL_MIN = 2.2250738585072014e-308
            if (np.fabs(c)>DBL_MIN):
                trec = a/c
            else:
                trec=999999
                gain=DBL_MIN
            if (np.fabs(b)>DBL_MIN and np.fabs(c)>DBL_MIN):
                saturation    = d/b
                gain = c/b
            else:
                saturation = 0
                gain = DBL_MAX
        pb.clf()
        adesc = pb.subplot(111)
        trx = self.getTelcalTrx(antenna,spw,scan,pol)
        pb.plot(freqHz*1e-9, trec, 'k-', freqHz*1e-9, trx, 'r-')
        pb.xlabel('Frequency (GHz)')
        pb.ylabel('Trec3(K)')
        print("Median Trx=%f" % (np.median(trec)))
        ut = mjdsecToUTHMS(np.mean(self.mymsmd.timesforscan(scan)))
        adesc.xaxis.grid(True,which='major')
        adesc.yaxis.grid(True,which='major')
        pb.title('Trec3  %s  %s  scan=%d  pol=%d  time=%s' % (os.path.basename(self.vis), antennaName, scan, pol, ut), fontsize=11)
        pb.text(0.5,0.95,'red=TelCal result', transform=adesc.transAxes)
        pb.text(0.5,0.9,'mean=%f'%(np.mean(trec)),transform=adesc.transAxes)
        pb.savefig('%s.scan%d.pol%d.trec3.png' % (os.path.basename(self.vis), scan,pol))
        pb.draw()
        return(trec, saturation, gain)

    def solveTsys(self, pHot, pAmb, pSky, jatm, jbg, gb, alpha, tauA, verbose=False):
        #               dbl   dbl  dbl   vector vec vec  dbl   vector
     #                                  --2 sidebands--
        pRef = 0
        # Equation 17 of Lucas & Corder
        tCal  = jatm[0]-jbg[0]
        if (len(jatm) > 1):
            tCal += (gb[1]/gb[0])*np.exp(-tauA[1]+tauA[0])*(jatm[1]-jbg[1])
        pLoad = alpha*pHot+(1-alpha)*pAmb     # Eq 18b of Lucas & Corder
        tSys  = tCal*(pSky-pRef)/(pLoad-pSky) # Eq 18a of Lucas & Corder
        if verbose: 
            print("solveTsys: shapes: tSys=%s, tCal=%s, pSky=%s, pRef=%s, pLoad=%s, jatm=%s, jbg=%s, gb=%s, tauA=%s" % (np.shape(tSys), np.shape(tCal), np.shape(pSky), np.shape(pRef), np.shape(pLoad), np.shape(jatm), np.shape(jbg), np.shape(gb), np.shape(tauA)))
        return tCal,tSys
        
    def solveAlpha(self, jHot, jAmb, jAtm, jspDSB, etaF):
        # Eq. 16 of Lucas & Corder "Dual Load Amplitude Calibration in ALMA"
        alpha = (etaF*jAtm-jAmb+(1-etaF)*jspDSB) / (jHot-jAmb)
        return(alpha)
        
    def computeJs(self, scan, antenna, pol, spw, asdm=None, etaF=0.98, lo1=None,
                  dataFraction=[0.0, 1.0], parentms=None, verbose=False,
                  siteAltitude_m=5059, computeJsky=False, altscan=None, 
                  calscandict=None, atmosphere=None, newFrequencyAxis=None,
                  maxAltitude=80.0, pwv=None):
        """
        Compute Jamb, Jhot, and (optionally) Jsky
        Reads the number of channels, their widths and freqs for the spw using self.mymsmd
        if computeJsky == True
           returns: jSky, jAmb, jHot, frequency[0], jatmDSB, jspDSB, jbg,   tauA,  alpha, gb, atmosphere, tebbsky
            shapes: (128) (1)   (1)   (128)        (2, 128)  (128) (2,128) (2,128) (128) (2,1)   dict     (2,128)
                    but JspDSB is currently a constant temperature, of order 268K
                    The (2) size is for 2 sidebands.
                    atmosphere is the dictionary returned by CalcAtmosphere
        else:
           returns: jSky=0, jAmb, jHot, frequency[0]
        Inputs:
        scan: integer or string
        antenna: integer ID, string ID or string name
        pol: 0 or 1
        spw: Tsys spw (integer or string integer)
        maxAltitude: of the atmosphere, in km
        atmosphere: dictionary keyed by 'signal', 'image', with values returned by
                    au.CalcAtmosphere()
        pwv: if specified, then use this value to override what was in the ASDM
        newFrequencyAxis: a grid of frequencies to calculate on; direction is unclear
        Note: the ATM model spectrum is Hanning smoothed if self.hanningSmoothed[spw] == True,
            which is set automatically upon initialization of the Atmcal class.  To override
            this feature, one can set that dictionary entry to True or False before calling
            this function.
        """
        spw = int(spw)
        antennaId, antennaName = self.getAntenna(antenna)
        baseband = self.basebands[spw] # mymsmd.baseband(spw)
        result = getCalAtmosphereInfo(self.vis, scan=scan, antenna=antennaName, pol=pol,
                                      baseband=baseband, debug=verbose, altscan=altscan,
                                      calscandict=calscandict, mymsmd=self.mymsmd)
        if (result == None and altscan == 'auto'):
            scanlist = sorted(list(self.scans))
            scanindex = scanlist.index(scan)
            if (scanindex==0):
                altscan = scanlist[1]
            else:
                altscan = scanlist[scanindex-1]
            print("Choosing alternate scan=%d" % (altscan))
            result = getCalAtmosphereInfo(self.vis, scan=scan, antenna=antennaName, pol=pol,
                                          baseband=baseband, debug=verbose, altscan=altscan,
                                          calscandict=calscandict, mymsmd=self.mymsmd)
        if (result == None): return
        myIndex = result[10]
        if (len(myIndex) > 1):
            print("More than 1 row returned.  Be more specific in your selection")
            return
        if (len(myIndex) == 0):
            print("No rows match these values.")
            return
        sbGains = result[0]
        gb = [sbGains, 1-sbGains]  # signal, image
        mjdsec = result[4]
        row = 0
        net_sideband = sidebandToNetSideband(self.mymsmd.sideband(spw))
        field = self.mymsmd.fieldsfortimes([mjdsec[row]])[0]
        if newFrequencyAxis is None:
            freq_signal_sideband = self.mymsmd.chanfreqs(spw)
        else:
            if verbose: print("computeJs(): Using newFrequencyAxis len = ", len(newFrequencyAxis))
            freq_signal_sideband = newFrequencyAxis
        if (len(freq_signal_sideband) == 1):
            chanwidth = 2e9 # channel-averaged spw
        else:
            chanwidth = freq_signal_sideband[1]-freq_signal_sideband[0]
        refFreq = freq_signal_sideband[0] - 0.5*chanwidth
        
        chans_signal = range(len(freq_signal_sideband))
        if field not in self.radec:
            self.radec[field] = getRADecForField(self.vis, field, usemstool=True, mymsmd=self.mymsmd)
        mydirection = self.radec[field]
        myazel = computeAzElFromRADecMJD(mydirection, mjdsec[row]/86400., self.telescopeName, verbose=False)
        airmass = elevationToAirmass(np.degrees(myazel[1]))
        water = result[6]
        if pwv is None:
            pwv = water[row]  # could take the mean instead?
        if (verbose):
            print("Using airmass = %f" % (airmass))
            print("Using 1st pwv of %d = %f" % (len(water), water[row]))
        antenna_id = result[1][row]
        if (antenna_id != antennaName):
            print("Mismatch in antenna ID!  antenna_id=%s, antennaId=%s" % (str(antenna_id), antennaName))
            return
        groundPressure = result[7][row]
        groundTemperature = result[8][row]
        groundRelHumidity = result[9][row]
        Tatm = groundTemperature
        frequency = []
        tebbsky = []
        if (computeJsky):
            if atmosphere is None:
                if verbose: print("computeJs(): Running CalcAtmosphere (chans=%s,signalSB)" % (len(chans_signal)))
                startTime = timeUtilities.time()
                atmosphere = {}
                atmosphere['signal'] = \
                      CalcAtmosphere(chans_signal, freq_signal_sideband*1e-9, pwv, 
                                     refFreq, net_sideband, groundPressure,
                                     groundRelHumidity, Tatm, airmass, 
                                     siteAltitude_m=siteAltitude_m, maxAltitude=maxAltitude, 
                                     hanning=self.hanningSmoothed[spw])
                if (verbose):
                    print("Done CalcAtmosphere after %.1f sec" % (timeUtilities.time()-startTime))
#                print "input freq range: " , freq_signal_sideband[0]*1e-9, freq_signal_sideband[-1]*1e-9
#                print "output freq range: ", atmosphere['signal'][0][0], atmosphere['signal'][0][-1]
            freq, chans, transmission, tebb, opacity = atmosphere['signal']
            tauA = []
            tebbsky.append(tebb)
            tauA.append(opacity)
            frequency.append(freq*1e9)
        else:
            frequency.append(freq_signal_sideband)

        if self.lo1s is not None:
            LO1 = self.lo1s[spw]
        else:
            getLOsStatus = getLOs(self.vis)
            if (verbose):
                print("Done getLOs()")
            if (len(getLOsStatus) > 0):
                startTime = timeUtilities.time()
                intent = 'OBSERVE_TARGET#ON_SOURCE'
                if (intent not in self.intents):
                    # This is necessary in order to correct cal survey datasets
                    # which do not have an OBSERVE_TARGET intent
                    intent = 'CALIBRATE_FLUX#ON_SOURCE'
                    if (intent not in self.intents):
                        intent = 'CALIBRATE_DELAY#ON_SOURCE'
                self.lo1s = interpretLOs(self.vis, parentms, intent=intent, mymsmd=self.mymsmd)
                if (self.lo1s == None): return
                if (spw not in self.lo1s):
                    print("spw map returned by interpretLOs is suspect!")
                    LO1 = self.lo1s[list(self.lo1s.keys())[-1]]
                else:
                    LO1 = self.lo1s[spw]
                if (verbose):
                    print("Found LO1 = %f" % (LO1))
                print("Done getting LOs after %.1f sec" % (timeUtilities.time()-startTime))
            elif (lo1 is not None):
                LO1 = lo1
            else:
                print("Could not find LO1")
                return
            self.LO1 = LO1
        if verbose:
            print("signal SB refFreq = %.0f = %.0f - 0.5*%.0f,  IF=%.0f" % (refFreq, freq_signal_sideband[0], 
                                                                        chanwidth, refFreq-LO1))

        ##### Repeat for image sideband
        refFreq_image = 2*LO1 - refFreq
        freq_image_sideband = 2*LO1 - freq_signal_sideband
        if verbose:
            print(" image SB refFreq = %.0f = %.0f - 0.5*%.0f, IF=%f" % (refFreq_image, freq_image_sideband[0], 
                                                                     chanwidth, refFreq_image-LO1))
        if (computeJsky):
            if 'image' not in atmosphere:
                if verbose: print("computeJs(): Running CalcAtmosphere (chans=%d,imageSB)" % (len(chans)))
                startTime = timeUtilities.time()
                atmosphere['image'] = \
                      CalcAtmosphere(chans, freq_image_sideband*1e-9, pwv,
                                     refFreq_image, net_sideband, groundPressure,
                                     groundRelHumidity, Tatm, airmass, siteAltitude_m=siteAltitude_m,
                                     maxAltitude=maxAltitude, hanning=self.hanningSmoothed[spw])
                if verbose:
                    print("Done CalcAtmosphere after %.1f sec" % (timeUtilities.time()-startTime))
            freq, chans, transmission, tebb, opacity = atmosphere['image']
            frequency.append(freq*1e9)
            tebbsky.append(tebb)
            tauA.append(opacity)  # tauA means tau(Zenith)*Airmass
            jem = frequency*0 # initialize an array to zero
        else:
            frequency.append(freq_image_sideband)
        frequency = np.array(frequency)
        jSky = 0; jAmb = 0; jHot = 0
        jatmDSB = 0
        jspDSB = 0
        jbg = frequency*0 # initialize an array to zero
        jem = frequency*0 # initialize an array to zero
        jsp = frequency*0 # initialize an array to zero
        jatm = frequency*0 # initialize an array to zero
        cosmicBackgroundTemp = Tcmb
        if len(self.loadTemperatures) > 0:
            ambLoad = self.loadTemperatures[antennaId][scan]['amb']
            hotLoad = self.loadTemperatures[antennaId][scan]['hot']
        else:
            ambLoad = 0
            hotLoad = 0
        numSideband = 2
        for iside in range(numSideband):
            hvk = h*np.mean(frequency[iside])/k  # telcal uses the frequency of the middle channel
            if (computeJsky):
                expMinusTau = np.exp(-tauA[iside])
                jebb = hvk/(np.exp(hvk/tebbsky[iside])-1)
                jbg[iside] = hvk/(np.exp(hvk/cosmicBackgroundTemp)-1)
                jem[iside] = jebb - jbg[iside]*expMinusTau
                jsp[iside] = hvk/(np.exp(hvk/groundTemperature)-1)
                jatm[iside] = (jebb - jbg[iside]*expMinusTau) / (1.-expMinusTau)
                jSky += gb[iside]*(etaF*jem[iside] + (1.-etaF)*jsp[iside])
                jatmDSB += gb[iside]*jatm[iside]
                jspDSB += gb[iside]*jsp[iside]
            jAmb += gb[iside]*hvk/(np.exp(hvk/ambLoad)-1)
            jHot += gb[iside]*hvk/(np.exp(hvk/hotLoad)-1)
        if (verbose):
            print("gainratio=%f, ambLoad=%f, hotLoad=%f, means: jSky=%f, jAmb=%f, jHot=%f" % (gb[0], ambLoad, hotLoad, np.mean(jSky),np.mean(jAmb),np.mean(jHot)))
        if (computeJsky):
            if verbose: print("computeJs(): shape jatmDSB = ", np.shape(jatmDSB))
            alpha = self.solveAlpha(jHot, jAmb, jatmDSB, jspDSB, etaF)
            return(jSky, jAmb, jHot, frequency[0], jatm, jspDSB, jbg, tauA, alpha, gb, atmosphere, tebbsky)
        else:
            return(jSky, jAmb, jHot, frequency[0])  # jSky will be 0 here
        # end of computeJs()
# end of class Atmcal

def checkSamplers(vis, state_id=None, ac=None, spws=None, threshold1=1.25, threshold2=2.0,
                  maxChannel=3, maxScan=None, useMedian=False):
    """
    Gets the TDM full-resolution spws for a dataset, finds the sky subscan timerange
    of each ATM cal scan, and computes the ratio of channel zero to the median (or
    minimum) of channels 1-3 (or otherwise specified).  Ratios above the specified
    threshold are indicated with asterisks.

    state_id: the state_id number to use for the sky subscan (default: row with REF=1,SIG=0)
    ac: the Atmcal instance to use (default=None which means create a new one)
    spws: the spws to consider (default=None which means use all spws)
    threshold1: if ratio is above this, then print one pair of asterisks
    threshold2: if ratio is above this, then print two pairs of asterisks
    maxChannel: adjust the upper bound of the block of channels to take the median of
    maxScan: the highest scan number to consider (default=None which means use all scans)
    useMedian: default=False, if True, then take the median of channels 1..maxChannel as
               the denominator of the ratio, otherwise take the mininum
    Return values:
    dictionary of ratios, keyed by antenna name, then baseband number (1..4)
    -Todd Hunter
    """
    if (ac == None):
        ac = Atmcal(vis)
    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)
    if (spws == None):
        if (getCasaSubversionRevision() < casaRevisionWithAlmaspws):
            index1 = mymsmd.tdmspws()
            index2 = mymsmd.chanavgspws()
            index1 = np.setdiff1d(index1,index2)
        else:
            index1 = mymsmd.almaspws(tdm=True)
        index2 = mymsmd.spwsforintent('CALIBRATE_ATMOSPHERE#ON_SOURCE')
        spws = np.intersect1d(index1,index2)
    elif (type(spws) == int):
        spws = [spws]
    print("Processing spws = ", spws)
    scans = []
    band = getBand(mymsmd.chanfreqs(spws[0])[0])
    field = mymsmd.fieldsforspw(spws[0])[0]
    antennaNames = mymsmd.antennanames(range(mymsmd.nantennas()))
    basebands = {}
    uniqueBasebands = []
    scansforspw = {}
    for spw in spws:
        index1 = mymsmd.scansforspw(spw)
        index2 = mymsmd.scansforintent('CALIBRATE_ATMOSPHERE#ON_SOURCE')
        scansforspw[spw] = list(np.intersect1d(index1,index2))
        scans += scansforspw[spw]
        basebands[spw] = mymsmd.baseband(spw)
        if (basebands[spw] not in uniqueBasebands):
            uniqueBasebands.append(basebands[spw])
    scans = np.unique(np.array(scans))
    if (maxScan is not None):
        scans = list(scans[np.where(scans <= maxScan)[0]])
        if (len(scans) < 1):
            print("No scans within the specified value of maxScan=%d" % (maxScan))
            return
    print("Processing scans = ", scans)
    mymsmd.close()
    timerange = ''
    for scan in scans:
        if (scan in list(ac.timestampsString.keys())):
            if (len(timerange) > 0):
                timerange += ','
            timerange += ac.timestampsString[scan][ac.skysubscan]
    scan=','.join([str(a) for a in scans])
    spw = ','.join([str(a) for a in spws])
    if (state_id == None):
        state_id = str(ac.stateID_off_source[0])
    elif (state_id == []):
        state_id = None
    else:
        state_id = str(state_id)
    v = Visibility(vis,antenna1=0,antenna2=0,spwID=spws[0],correctedData=False,
                   scan=scans[0],cross_auto_all='auto',autoSubtableQuery=True,
                   state=state_id)
    print("using state id = ", v.getState())
    ratio = {}
    pols = [0,1] # assume both are present (for now)
    for ant1 in range(len(antennaNames)):
        ratio[ant1] = {}
        for baseband in uniqueBasebands:
            ratio[ant1][baseband] = {}
            for pol in pols:
                ratio[ant1][baseband][pol] = []
            
    for ant1 in range(len(antennaNames)):
        print("Working on antenna %d/%d" % (ant1+1,len(antennaNames)))
        v.autoSubtableQuery = False
        v.setAntennaPair(ant1,ant1)
        for spwID in spws:
            v.autoSubtableQuery = False
            v.setSpwID(spwID)
            myscans = np.intersect1d(scans,scansforspw[spwID])
            for scan in myscans:
              v.autoSubtableQuery = True
              try:
                v.setScan(scan)
                d = v.amp
                if (len(d) < 1):
                    print("ant%d spw%d scan%d has zero length data" % (ant1,spwID,scan))
                else:
                    for pol in pols:
                        if (len(d[pol]) < 1):
                            print("ant%d  spw%d scan%d pol%d has zero length data" % (ant1,spwID,scan,pol))
                        else:
                            if (useMedian):
                                myratio = d[pol][0] / np.median(d[pol][1:maxChannel+1])
                            else:
                                myratio = d[pol][0] / np.min(d[pol][1:maxChannel+1])
                            ratio[ant1][basebands[spwID]][pol].append(myratio)
              except:
                  print("Antenna %d, spw %d: no data for scan " % (ant1,spwID), scan)
                  continue
    print("Band %d   %s   %s" % (band,os.path.basename(vis),getObservationStartDate(vis)))
    print("Ratio of Channel 0 to the median of channels 1~%d in auto-correlation spectra" % (maxChannel))
    outline = "Antenna "
    for baseband in uniqueBasebands:
        for pol in pols:
            outline += " BB%dpol%d  " % (baseband-1,pol)
    print(outline)
    mydict = {}
    for ant1 in range(len(antennaNames)):
        mydict[antennaNames[ant1]] = {}
        outline = "%2d=%4s" % (ant1,antennaNames[ant1])
        for baseband in uniqueBasebands:
            mydict[antennaNames[ant1]][baseband] = []
            for pol in pols:
                myratio = np.median(ratio[ant1][baseband][pol])
                mydict[antennaNames[ant1]][baseband].append(myratio)
                npts = len(ratio[ant1][baseband][pol])
                outline += " "
                if (myratio > threshold2):
                    if (myratio >= 100):
                        outline += "**%4.1f**" % (myratio)
                    else:
                        outline += "**%5.2f**" % (myratio)
                elif (myratio > threshold1):
                    if (myratio >= 100):
                        outline += " *%4.1f* " % (myratio)
                    else:
                        outline += " *%5.2f* " % (myratio)
                else:
                    outline += "  %5.2f  " % (myratio)
        print(outline)
    return(mydict)
    # end of class AtmCal
    
class AtmStates:
    """
    This class is essentially defunct since the table contents were changed
    circa 2011. See the new class Atmcal.
    -Stuartt Corder
    """
    def __init__(self,inputMs):
        self.inputMs = inputMs
        mytb = tbtool()
        mytb.open('%s/STATE' % self.inputMs)
        self.loadTemps    = celsiusToKelvin(mytb.getcol("LOAD"))
        self.stateIntents = mytb.getcol("OBS_MODE")
        mytb.close()
        self.atmSky = []
        self.atmRef = []
        self.atmHot = []
        self.atmAmb = []
        self.atmScans = []
        self.antennaNames = getAntennaNames(self.inputMs)
        self.antennaInfo  = {}
        self.numAtmCals = 0
        for i in self.antennaNames:
            self.antennaInfo[i] = {'AMB' : {'state' : [], 'loadTemp' : []},
                                   'HOT' : {'state' : [], 'loadTemp' : []},
                                   'REF' : {'state' : []},
                                   'SKY' : {'state' : []}
                                   }
        self.getAtmCalTargetStates()
        self.associateStateWithAntenna()

    def getAtmCalTargetStates(self) :
        for i in range(len(self.stateIntents)) :
            if "CALIBRATE_ATMOSPHERE.OFF_SOURCE" in self.stateIntents[i]:
                self.atmSky.append(i)
            if "CALIBRATE_ATMOSPHERE.REFERENCE"  in self.stateIntents[i]:
                self.atmRef.append(i)
            if "CALIBRATE_ATMOSPHERE.ON_SOURCE"  in self.stateIntents[i] and self.loadTemps[i] > 330 : self.atmHot.append(i)
            if "CALIBRATE_ATMOSPHERE.ON_SOURCE"  in self.stateIntents[i] and self.loadTemps[i] < 330 : self.atmAmb.append(i)

    def associateStateWithAntenna(self) :
        self.antennaInfo  = {}
        for i in self.antennaNames:
            self.antennaInfo[i] = {'AMB' : {'state' : [], 'loadTemp' : []},
                                   'HOT' : {'state' : [], 'loadTemp' : []},
                                   'REF' : {'state' : []},
                                   'SKY' : {'state' : []}
                                   }
        visTemp = Visibility(self.inputMs,antenna1=None,antenna2=None,
                             spwID=None,state=None)

        for i in self.atmAmb :
            try:
                visTemp.setState(i)
                ant1 = visTemp.subtable.getcol('ANTENNA1')
                ant2 = visTemp.subtable.getcol('ANTENNA2')
                goodIndex = np.where(ant1 == ant2)[0]
                goodIndex = list(goodIndex)
                antennaIds = np.unique(ant1[goodIndex])
                for j in antennaIds :
                    antName = self.antennaNames[j]
                    antName = getAntennaNames(self.inputMs)[j]
                    self.antennaInfo[antName]['AMB']['state'].append(i)
                    self.antennaInfo[antName]['AMB']['loadTemp'].append(self.loadTemps[i])
            except:
                continue
        for i in self.atmHot :
            try:
                visTemp.setState(i)
                ant1 = visTemp.subtable.getcol('ANTENNA1')
                ant2 = visTemp.subtable.getcol('ANTENNA2')
                goodIndex = np.where(ant1 == ant2)[0]
                goodIndex = list(goodIndex)
                antennaIds = np.unique(ant1[goodIndex])
                for j in antennaIds :
                    antName = self.antennaNames[j]
                    self.antennaInfo[antName]['HOT']['state'].append(i)
                    self.antennaInfo[antName]['HOT']['loadTemp'].append(self.loadTemps[i])
            except:
                continue
        for i in self.atmRef :
            try:
                visTemp.setState(i)
                ant1 = visTemp.subtable.getcol('ANTENNA1')
                ant2 = visTemp.subtable.getcol('ANTENNA2')
                goodIndex = np.where(ant1 == ant2)[0]
                goodIndex = list(goodIndex)
                antennaIds = np.unique(ant1[goodIndex])
                for j in antennaIds :
                    antName = self.antennaNames[j]
                    self.antennaInfo[antName]['REF']['state'].append(i)
            except:
                continue
        for i in self.atmSky :
            try:
               visTemp.setState(i)
               ant1 = visTemp.subtable.getcol('ANTENNA1')
               ant2 = visTemp.subtable.getcol('ANTENNA2')
               goodIndex = np.where(ant1 == ant2)[0]
               goodIndex = list(goodIndex)
               antennaIds = np.unique(ant1[goodIndex])
               for j in antennaIds :
                   antName = self.antennaNames[j]
                   self.antennaInfo[antName]['SKY']['state'].append(i)
            except:
                continue

class processDVTiltMeter:
    def __init__(self,dvTiltmeterFile,outFile=None):
        self.dvTiltmeterFile = dvTiltmeterFile
        self.fulltable = fiop.fileToTable(dvTiltmeterFile,keepType=True)
        self.columns   = fiop.getInvertTable(self.fulltable)
        self.oldtime   = self.columns[0]
        self.time = np.array(convertTimeStamps(self.columns[0]))
        self.time = self.time-self.time[0]
        self.antenna = self.columns[1]
        self.an0    = np.array(self.columns[2])
        self.aw0    = np.array(self.columns[3])
        self.x     = np.array(self.columns[4])
        self.y     = np.array(self.columns[5])
        self.t1    = np.array(self.columns[6])
        self.t2    = np.array(self.columns[7])
        if outFile != None :
            self.outFile = outFile
            self.fitT2Trend()
            self.removeTrend()
            self.writeResiduals()

    def fitT1Trend(self) :
        self.px = polyfit(self.t1,self.x,1)
        self.py = polyfit(self.t1,self.y,1)
        self.newX=polyval(self.px,self.t1)
        self.newY=polyval(self.py,self.t1)

    def fitT2Trend(self) :
        self.px = polyfit(self.t2,self.x,1)
        self.py = polyfit(self.t2,self.y,1)
        self.newX=polyval(self.px,self.t2)
        self.newY=polyval(self.py,self.t2)

    def removeTrend(self) :
        self.x = self.newX-self.x
        self.y = self.newY-self.y

    def restoreTrend(self) :
        self.x = self.x+self.newX
        self.y = self.y+self.newY

    def plotTime(self) :
        return
    def plotT1(self) :
        return
    def plotT2(self) : 
        return

    def writeResiduals(self) :
        f = open(self.outFile,'w')
        for i in range(len(self.antenna)) :
            f.write("%s %s %f %f %f %f %f %f\n" % (self.oldtime[i],self.antenna[i],self.an0[i],self.aw0[i],self.newX[i],self.newY[i],self.t1[i],self.t2[i]) )
        f.close()

def nameforspw(vis, spw):
    """
    Gets the spw name from the name from the
    NAME column of the SPECTRAL_WINDOW table of a measurement set.
    Obsoleted by msmd.namesforspw in CASA >= 4.3.0.
    -Todd Hunter
    """
    if (not os.path.exists(vis)):
        print("Could not find measurement set")
        return
    mytb = createCasaTool(tbtool)
    mytb.open(vis+'/SPECTRAL_WINDOW')
    names = mytb.getcol('NAME')
    mytb.close()
    if (spw >= len(names)):
        print("spw %d is not in this measurement set")
        return
    return(names[spw])
    
def fixMyDelays(asdm,caltableName=None,vis=None,doImport=True,sign=1) :
    """
    This function will extract the TelCal solutions for delay and generate a
    calibration table, which is useful if the solutions were not applied during
    observations.
    This version will handle single or multiple receiver bands in the ASDM.
    There are two main use cases for this command:
      1) If you have already run importasdm with asis='*', then you don't 
         need to specify the asdm, only the ms:
         fixMyDelays('','my.delaycal',vis='uid.ms',False)
     2) If you want to use this function as a wrapper for importasdm, then 
        you can say:  fixMyDelays('uid__blah_blah',None,None,True)
     The optional 'sign' parameter can be used to flip the sign if and when 
     someone changes the sign convention in TelCal.
    - Todd Hunter
    """
    asis = '*'
    if (len(asdm) > 0):
        [asdm,dir] = locate(asdm)
        print("asdm = %s" % (asdm))
    if (vis == None):
        vis = "%s.ms" % asdm.split('/')[-1]
        print("vis = %s" % (vis))
    elif (len(vis) < 1):
        vis = "%s.ms" % asdm.split('/')[-1]
        print("vis = %s" % (vis))
    if (caltableName == None) or (len(caltableName) < 1):
        if (len(asdm) == 0):
            caltableName = "%s.delaycal" % (vis)
        else:
            caltableName = "%s.delaycal" % asdm.split('/')[-1]
    if doImport : importasdm(asdm=asdm,asis=asis,vis=vis,overwrite=True)
    antennaIds = getAntennaNames(vis)
    bbands     = getBasebandAndBandNumbers(vis)
#    print bbands
    mytb = tbtool()
    mytb.open("%s/ASDM_CALDELAY" % vis)
    antennaNames = mytb.getcol("antennaName")
    if (len(antennaNames) < 1):
        print("The ASDM_CALDELAY table has no data.  Delay correction cannot be done.")
        return
    delayOffsets = mytb.getcol("delayOffset")
    basebands    = mytb.getcol("basebandName") # format is 'BB_1'
#    print "basebands = ", basebands
    rxbands    = mytb.getcol("receiverBand")    # format is 'ALMA_RB_%02d'
#    print "rxbands = ", rxbands
    polList = mytb.getcol("polarizationTypes")  # e.g. X or Y or ...
    mytb.close()
    outList = []
    spwNames = []
    rxBands = []
    for j in bbands:
        rxName = "ALMA_RB_%02d" % (j[0])
        rxBands.append(j[0])
        bbName = "BB_%i" % (j[1])
        spwNames.append("%d"%j[2])
#        print "bbName=%s, rxName=%s" % (bbName, rxName)
        for i in antennaIds :
#            print "ith antenna = ", i
            ant = np.where(antennaNames == i)
            bb  = np.where(basebands == bbName)
            rx  = np.where(rxbands == rxName)
#            ind = np.intersect1d_nu(ant,bb)
            newlist = np.intersect1d(ant[0],bb[0])
            newlist = np.intersect1d(newlist,rx[0])
#            print "newlist = ", newlist
            ind = np.unique(newlist)
#            print "ind = ", ind
            if (sign < 0):
                print("Applying the reverse sign of the delays to %s, ant %s" %(bbName,i))
            p  = sign*delayOffsets[:,ind].mean(1)*1e9
            for k in p :
                outList.append(k)
    parameter = outList
    pol = ''
    print("polList[:,0] = ", polList[:,0])
    print("spws = ", spwNames)
    print("rx bands = ", rxBands)
    for k in range(len(polList[:,0])):
        pol += "%s," % (polList[:,0][k])
    pol = pol[:-1]
    
    antenna   = ",".join(np.unique(antennaNames))
    spw       = ",".join(spwNames)
    print("Removing any old caltable = %s." % (caltableName))
    os.system('rm -rf %s'%(caltableName))
    print("Calling gencal('%s','%s','sbd','%s','%s','%s')" % (vis,caltableName,spw,antenna,pol))
    gencal(vis=vis,caltable=caltableName,caltype='sbd',spw=spw,antenna=antenna,pol=pol,parameter=parameter)
    # end of fixMyDelays
            
def getUniqueBasebandNumbers(inputMs) :
    """
    Returns the list of baseband numbers in the specified ms.
    """
    mytb = tbtool()
    mytb.open("%s/SPECTRAL_WINDOW" % inputMs)
    bb = mytb.getcol("BBC_NO")
    mytb.close()
    return np.unique(bb)

def continuumDefault(band):
    """
    Returns the continuum default mean frequency (in GHz) for specified ALMA 
    band.  This value equals LO1 for the 2SB bands.  The value of LO1 is also
    printed for the DSB Bands (9 and 10).
    -Todd Hunter
    """
    meanfreq = {3: 97.5, 4: 145.0, 5: 203.0, 6: 233.0, 7: 343.5, 8: 405.0, 
                9: 679.0, 10: 875.0}
    # LO1 differs from meanfreq only for band 9 and 10
    lo1 = {3: 97.5, 4: 145.0, 5: 203.0, 6: 233.0, 7: 343.5, 8: 405.0, 
           9: 671.0, 10: 867.0}
    bands = list(meanfreq.keys())
    if band not in bands:
        print("Band must be in: %s" % (str(bands)))
        return
    if band in [9,10]:
        print("LO1 = %.1f GHz"% (lo1[band]))
    return meanfreq[band]

def freqToBand(freq):
    """
    Returns the ALMA band integer that can observe the specified frequency.
    It will accept either a single frequency or a list (in Hz).
    It is kind of a kludge until something better is devised.
    Called only by getBasebandAndBandNumbers and plotMosaic.
    See also getBand(), and bandToFreqRange().
    """
    band = []
    if (type(freq) != list and type(freq) != np.ndarray):
        freq = [freq]
    for f in freq:
        if (f > 750e9):
            band.append(10)
        elif (f > 550e9):
            band.append(9)
        elif (f > 379e9):
            band.append(8)
        elif (f > 275e9):
            band.append(7)
        elif (f > 211e9):
            band.append(6)
        elif (f > 163e9): # note: band 5 actually goes down to 158 now (ICT-4688)
            band.append(5)
        elif (f > 120e9):
            band.append(4)
        elif (f > 84e9):
            band.append(3)
        elif (f > 67e9):
            band.append(2)
        elif (f > 35e9 and f <= 52e9):
            band.append(1)
        else:
            band.append(None)
    return(band)
        
def getBasebandAndBandNumbers(inputMs) :
    """
    experimental version to try to deal with band 6/9 phase transfer data
    which uses a bit of a kludge to convert freq to receiverBand
    return orderedlist:  rxBand, baseBand, spw
    Author: unknown
    """
    mytb = tbtool()
    mytb.open("%s/SPECTRAL_WINDOW" % inputMs)
    bb = mytb.getcol("BBC_NO")
    freq = mytb.getcol("REF_FREQUENCY")
    numchan = mytb.getcol("NUM_CHAN")
    tbw = mytb.getcol("TOTAL_BANDWIDTH")
    band = freqToBand(freq)
    pair = []
    for i in range(len(band)):
        # remove the channel-average spws and the WVR spws
        if (numchan[i] > 1 and tbw[i] < 7e9):
            pair.append((band[i],bb[i],i))
    mytb.close()
    return np.unique(pair)

def getBasebandAndBandNumbersTest(inputMs) :
    """
    New experimental version to try to deal with band 6/9 phase transfer data
    which tries to determine receiverBand properly, but seems to be impossible.
    return orderedlist:  rxBand, baseBand, spw
    - Todd Hunter
    """
    mytb = tbtool()
    mytb.open("%s/SPECTRAL_WINDOW" % inputMs)
    bb = mytb.getcol("BBC_NO")
    numchan = mytb.getcol("NUM_CHAN")
    tbw = mytb.getcol("TOTAL_BANDWIDTH")
    mytb.close()
    mytb.open("%s/ASDM_RECEIVER" % inputMs)
    spws = mytb.getcol('spectralWindowId')
    bands = mytb.getcol('frequencyBand')
    mytb.close()
    pair = []
    band = []
    for i in range(len(numchan)):
        window = ('SpectralWindow_%d'%(i))
        print("window = ", window)
        findspw = np.where(spws==window)
        if (len(findspw) > 0):
            print("findspw = ", findspw)
            band.append(bands[findspw[0][0]])
            # remove the channel-average spws and the WVR spws
            if (numchan[i] > 1 and tbw[i] < 7e9):
                pair.append((band[i],bb[i],i))
    return np.unique(pair)

def getSpwsForBasebandFromASDM(asdm, bb):
    """
    Returns a list of spws of the specified baseband in an ASDM.
    The spw numbers are the ones they will have upon loading into a measurement set.
    -Todd Hunter
    """
    mydict = getSpwsFromASDM(asdm)  # this will be ms-numbered spws
    spwmap = asdmspwmap(asdm)
    spws = []
    for key in list(mydict.keys()):
        if mydict[key]['basebandNumber'] == bb:
            spws.append(key)
    return(spws)
    
def getSpwsForBaseband(mymsmd,bb):
    """
    This emulates msmd.spwsforbaseband() for older versions of casa that
    do not contain this method.  
    mymsmd: either an instance of an existing msmd tool, or the name of a
            measurement set
    bb: baseband number
    -Todd Hunter
    """
    needToClose = False
    if (type(mymsmd) == str):
        vis = mymsmd
        if (os.path.exists(vis) == False):
            print("First argument must be either an msmd instance or the name of a measurement set.")
            return
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        needToClose=True
    if (getCasaSubversionRevision() >= '25612'):
        spws = mymsmd.spwsforbaseband(bb)
    else:
        spws = []
        for spw in range(mymsmd.nspw()):
            if (bb == mymsmd.baseband(spw)):
                spws.append(spw)
    if (needToClose):
        mymsmd.close()
    return(spws)
    
def getBasebands(mymsmd, spw=''):
    """
    Takes an msmd tool instance and builds a list of basebands corresponding 
    to the list of spws.
    If no basebands exists (such as simulated data), it returns a blank string.
    Todd Hunter
    """
    basebands = []
    if len(spw) == 0:
        spws = range(mymsmd.nspw())
    elif type(spw) == str:
        spws = spw.split(',')
    else:
        spws = spw
    mytb = tbtool()
    mytb.open(os.path.join(mymsmd.name(),'SPECTRAL_WINDOW'))
    colnames = mytb.colnames()
    mytb.close()
    for spw in spws:
        if 'BBC_NO' in colnames:
            basebands.append(mymsmd.baseband(spw))
        else:
            return('')
    return(basebands)
        
def printLOs(vis, parentms='', showWVR=False,
             showCentralFreq=True, verbose=False, alsoReturnLO2=False,
             showChannelAverageSpws=False, showOnlyScienceSpws=False,
             birdieFreq=None, birdieSpw=None, showEffective=False,
             showWindowFactors=False):
    """
    Print the LO settings for an MS from the ASDM_RECEIVER table.
    The possibility of leakage of YIG and undesired LO2 sideband is printed if present.
    Options:
    showCentralFreq: if True, then show the mean frequency of each spw,
                     otherwise show the frequency of the first channel
    showWVR: include the WVR spw in the list
    parentms:  if the dataset has been split from a parent dataset, then
               you may also need to specify the name of the parent ms.
    alsoReturnLO2: if True, return a second dictionary of the LO2 values
    birdieFreq: if specified, compute the IF of this RF feature
    birdieSpw: only necessary if more than one LO1 in the science spws
    
    Returns: a dictionary of the LO1 values (in Hz) for each spw, keyed by
             integer.  This function will warn if the Band 6,9,10 YIG can leak
             into the spw (i.e. LO2 between 10.5-11.3 GHz). See AIV-2057.
             
    For further help and examples, see:
         https://safe.nrao.edu/wiki/bin/view/ALMA/PrintLOs
    - Todd Hunter
    """
    return(interpretLOs(vis, parentms, showWVR, showCentralFreq, 
                        verbose, show=True, alsoReturnLO2=alsoReturnLO2,
                        showChannelAverageSpws=showChannelAverageSpws,
                        showOnlyScienceSpws=showOnlyScienceSpws,
                        birdieFreq=birdieFreq, birdieSpw=birdieSpw,
                        showEffective=showEffective,
                        showWindowFactors=showWindowFactors))

def sidebandToString(sb):
    """
    Converts -1 to 'LSB' and +1 to 'USB'
    -Todd Hunter
    """
    if sb == -1:
        return 'LSB'
    elif sb == 1:
        return 'USB'
    else:
        return 'none'

def getScienceSpwSidebands(vis, spws=None, mymsmd=None):
    """
    Calls msmd.sideband to determine the sideband of the science spws.  This works
    for both parent and split ms
    spws: if specified, limit the spws to these (python list or comma-delimited string)
    Returns: dictionary keyed by spw ID, with values of 'LSB' or 'USB'
    -Todd Hunter
    """
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    else:
        needToClose = False
    if spws is None:
        spws = getScienceSpws(vis, mymsmd=mymsmd, returnString=False) # list of integers
    else:
        spws = parseSpw(vis, spws, mymsmd=mymsmd)
    mydict = {}
    if True:
        for spw in spws:
            mydict[spw] = sidebandToString(mymsmd.sideband(spw))
    else:
        # old method, only works for non-split ms
        LO1s = interpretLOs(vis, show=False, mymsmd=mymsmd, showOnlyScienceSpws=True)  # dictionary keyed by science spw number
        if LO1s is None: 
            spws = getScienceSpwSidebands(vis)
            [LOs,bands,spws,names,sidebands,receiverIds,spwNames] = getLOs(vis)
            LO1s = {}
            for spw in spws:
                LO1s[spw] = LOs[spw][0]
        for spw in spws:
            print("spw=%s, LOs[spw]=%s" % (str(spw), str(LO1s[spw])))
            if LO1s[spw] - mymsmd.meanfreq(spw) > 0:
                mydict[spw] = 'LSB'
            else:
                mydict[spw] = 'USB'
    if needToClose:
        mymsmd.close()
    return mydict

def getMeanScienceFreqOfSidebands(vis):
    """
    Computes the bandwidth-weighted mean frequency of the spws in each sideband
    Returns: dictionary keyed by 'LSB' and 'USB', with values in Hz
    -Todd Hunter
    """
    if not os.path.exists(vis):
        print("Measurement set not found")
        return
    mymsmd = msmdtool()
    mymsmd.open(vis)
    mydict = getScienceSpwSidebands(vis, mymsmd=mymsmd)
    if mydict is None: return
    lsbFreqs = []
    lsbWeights = []
    usbFreqs = []
    usbWeights = []
    for spw in list(mydict.keys()):
        freq = mymsmd.meanfreq(spw)
        bandwidth = mymsmd.bandwidths(spw)
        if mydict[spw] == 'LSB':
            lsbFreqs.append(freq)
            lsbWeights.append(bandwidth)
        elif mydict[spw] == 'USB':
            usbFreqs.append(freq)
            usbWeights.append(bandwidth)
        else:
            print("Unrecognized sideband (%s) for spw %d" % (mydict[spw], spw)) 
    mymsmd.close()
    mydict = {}
    if len(lsbFreqs) > 0:
        lsbFreq = np.average(lsbFreqs, weights=lsbWeights)
        mydict['LSB'] = lsbFreq
    if len(usbFreqs) > 0:
        usbFreq = np.average(usbFreqs, weights=usbWeights)
        mydict['USB'] = usbFreq
    return mydict

def receiverType(vis, spw=None, showWVR=False, showChannelAverageSpws=False,
                 showOnlyScienceSpws=False, verbose=False, mymsmd=None):
    """
    Uses interpretLOs(alsoReturnReceiverType=True), which ultimately reads the receiverSideband
    field of the ASDM_RECEIVER table.
    spw: if not specified, then use the first science spw
    Returns a string: 'TSB', 'DSB' or 'NOSB'
    """
    if not os.path.exists(vis):
        print("Could not find measurement set.")
        return
    if mymsmd is None:
        mymsmd = msmdtool()
        mymsmd.open(vis)
        needToClose = True
    else:
        needToClose = False
    mydict = interpretLOs(vis, alsoReturnReceiverType=True, showWVR=showWVR, 
                          showChannelAverageSpws=showChannelAverageSpws,
                          showOnlyScienceSpws=showOnlyScienceSpws, verbose=verbose, mymsmd=mymsmd)[1]
    if spw is None:
        spw = getScienceSpws(vis,returnString=False, mymsmd=mymsmd)
        if needToClose:
            mymsmd.close()
        if len(spw) == 0:
            print("No science spws are this dataset, so you must specify the spw parameter.")
            return
        spw = spw[0]
        if verbose: print("Choosing science spw %d" % (spw))
    elif needToClose:
        mymsmd.close()
    if spw not in mydict:
        print("This spw is not in the list.  See the show parameters.")
        print(list(mydict.keys()))
        return
    return mydict[spw]

def interpretLOs(vis, parentms='', showWVR=False,
                 showCentralFreq=False, verbose=False, show=False,
                 alsoReturnLO2=False, showChannelAverageSpws=False,
                 showOnlyScienceSpws=False, birdieFreq=None, birdieSpw=None,
                 intent='OBSERVE_TARGET#ON_SOURCE', spwsForIntent=None,
                 showEffective=False, showWindowFactors=False, mymsmd=None,
                 alsoReturnReceiverType=False):
    """
    Interpret (and optionally print) the LO settings for an MS from the
    ASDM_RECEIVER table.  Note that msmd.spwsforintent is a time hog, and that
    is why passing in spwsForIntent is an option to speed up execution.
    Options:
    showCentralFreq: if True, then show the mean frequency of each spw,
                     otherwise show the frequency of the first channel
    showWVR: include the WVR spw in the list
    parentms:  if the dataset has been split from a parent dataset, then
               you may also need to specify the name of the parent ms.
    alsoReturnLO2: if True, return a second dictionary of the LO2 values
    birdieFreq: if specified, compute the IF of this RF feature
    birdieSpw: only necessary if more than one LO1 in the science spws
    intent: which intent to use in spwsforintent (to find science spws)
    spwsForIntent: if specified, then avoid the call to spwsforintent

    Returns: a dictionary of the LO1 values (in Hz) for each spw, keyed by
             integer.  

    A typical band 7 TDM dataset (prior to splitting) looks like this:
    SPECTRAL_WINDOW table has 39 rows:   row
           WVR                           0
           8 band 3 windows (pointing)   1-8
           8 band 7 windows              9-16
           22 WVR windows                17-38
    The corresponding ASDM_RECEIVER table has only 18 rows:
           WVR                           0
           8 band 3 windows              1-8
           WVR                           9          
           8 band 7 windows              10-17
    After splitting, the ASDM_RECEIVER table remains the same, but the 
    SPECTRAL WINDOW table then has only 4 rows, as the pointing spws and 
    the channel-averaged data are dropped:
           4 band 7 windows               

    Todd Hunter
    """
    if not os.path.exists(vis):
        print("Could not find measurement set.")
        return
    lo1s = {} # initialize dictionary to be returned
    lo2s = {}
    receiverTypes = {}  # 'DSB', 'TSB', 'NOSB'
    try:
        retval =  getLOs(vis, returnReceiverSidebands=True) # does not use msmd
        [LOs,bands,spws,names,sidebands,receiverIds,spwNames,receiverSidebands] = retval
        if verbose: print("%d entries returned by getLOs" % (len(LOs)))
    except:
        print("getLOs failed")
        return(retval)
    if (verbose): print("len(spws) = %d: %s" % (len(spws), str(spws)))
    maxSpw = np.max(spws)
    sawWVR = False
    indices = []  # will exclude the extraneous WVR spws
    for i in range(len(spws)):
        if (names[i].find('WVR') >= 0):
            if (not sawWVR):
                indices.append(i)
                sawWVR = True
        else:
            indices.append(i)
    LOs = np.array(LOs)[indices]
    bands = np.array(bands)[indices]
    spws = list(np.array(spws)[indices])
    names = np.array(names)[indices]
    sidebands = np.array(sidebands)[indices]
    receiverSidebands = np.array(receiverSidebands)[indices]
    receiverIds = np.array(receiverIds)[indices]
    index = range(len(spws))
    mytb = createCasaTool(tbtool)
    mytb.open(vis+'/SPECTRAL_WINDOW')
    # If the data have been split into an ms with fewer spws, then this 
    # table will be smaller (in rows) than the parent MS's table.
    spwNames = mytb.getcol('NAME')
    mytb.close()
    splitted = False
    if (maxSpw != len(spwNames)-1):
        splitted = True
        if (verbose): 
            print("maxSpw=%d != len(spwNames)=%d)" % (maxSpw, len(spwNames)))
        if (parentms == '' or parentms is None):
            print("You appear to have split these data.  Please provide the parentms as an argument.")
            return
        mytb.open(parentms+'/SPECTRAL_WINDOW')
        parentSpwNames = mytb.getcol('NAME')
        mytb.close()
        extractedRows = []
        index = []
        for s in range(len(spwNames)):
            if (len(spwNames[s]) == 0):
                print("This is an old dataset lacking values in the NAME column of the SPECTRAL_WINDOW table.")
                return
            if (verbose): 
                print("Checking for %s in " % (spwNames[s]), parentSpwNames)
            extractedRows.append(np.where(parentSpwNames == spwNames[s])[0][0])
            index.append(spws.index(extractedRows[-1]))
            if (verbose): 
                print("spw %d came from spw %d" % (s, extractedRows[-1]))
# extractedRows = the row of the parent SPECTRAL_WINDOW table that matches 
#                 the split-out spw
#     index = the row of the ASDM_RECEIVER table that matches the split-out spw
        vis = parentms
    if (verbose): 
        print("spwNames = ", spwNames)
        print("spws = ", spws)
        print("bands = ", bands)
        output = "LOs = "
        for LO in LOs:
            output += "%.3f, " % (LO[0]*1e-9)
        print(output)
        print("names = ", names)
        print("index = ", index)

    bbc = getBasebandNumbers(vis) # does not use msmd
    if (show): 
        print('Row refers to the row number in the ASDM_RECEIVER table (starting at 0).')
        if (showCentralFreq):
            myline = 'Row spw BB RxBand CenFreq Nchan LO1(GHz) LO2(GHz) Sampler YIG(GHz) TFBoffset(MHz)'
        else:
            myline = 'Row spw BB RxBand Ch1Freq Nchan LO1(GHz) LO2(GHz) Sampler YIG(GHz) TFBoffset(MHz)'
        if (showEffective):
            myline += ' Eff.BW(MHz) Res(MHz) Width(MHz)'
        if (showWindowFactors):
            myline += ' windowFactors'
        print(myline)

    # Loop over all rows in the ASDM_RECEIVER table, unless we've split, in 
    # which case this will loop over the N spws in the table.
    if (casaVersion >= casaVersionWithMSMD):
        needToClose = False
        if mymsmd is None or mymsmd == '':
            mymsmd = createCasaTool(msmdtool)
            mymsmd.open(vis)
            needToClose = True
        if (spwsForIntent is None):
            if intent in mymsmd.intents():
                scienceSpws = np.setdiff1d(mymsmd.spwsforintent(intent),mymsmd.wvrspws())
            else:
                scienceSpws = []
        else:
            scienceSpws = spwsForIntent
    birdieIF = 0
    if (birdieFreq is not None):
        birdieFreq = parseFrequencyArgumentToHz(birdieFreq)
        birdieFreqGHz = parseFrequencyArgumentToGHz(birdieFreq)
    for i in range(len(index)):
        if (verbose): 
            print("index[%d]=%d" % (i,index[i]))
            print("spws[%d] = %d" % (index[i], spws[index[i]]))
        myspw = i 
        if (myspw not in scienceSpws and showOnlyScienceSpws): continue
        if (birdieFreq is not None and birdieIF == 0):
            if (myspw == birdieSpw):
                if verbose:
                    print("spw=%d, Computing IF = %f - %f" % (myspw, birdieFreq, LOs[index[i]][0]))
                birdieIF = np.fabs(birdieFreq - LOs[index[i]][0])
            elif (myspw in scienceSpws and birdieSpw is None):
                if verbose:
                    print("spw=%d (in %s), Computing IF = %f - %f" % (myspw, str(scienceSpws), birdieFreq, LOs[index[i]][0]))
                birdieIF = np.fabs(birdieFreq - LOs[index[i]][0])
        if (casaVersion >= casaVersionWithMSMD):
            freqs = mymsmd.chanfreqs(myspw)
            meanFreqGHz = mymsmd.meanfreq(myspw) * (1e-9)
        else:
            if (showOnlyScienceSpws):
                print("The option showOnlyScienceSpws is not supported in this old of a CASA.")
                return
            freqs = getFrequencies(vis,myspw) * (1e-9) # does not use msmd
            meanFreqGHz = np.mean(freqs)
        if (len(freqs) < 2 and showChannelAverageSpws==False): 
            continue
        if (bands[index[i]].split('_')[-1].isdigit()):
            rxband = bands[index[i]].split('_')[-1]
        elif (showWVR):
            rxband = 'WVR'
        else:
            continue
        line = "%2d  %2d  %d %3s " % (spws[index[i]], myspw, bbc[myspw], rxband)
        if (showCentralFreq):
            line += "%10.6f %4d " % (meanFreqGHz,len(freqs))
        else:
            line += "%10.6f %4d " % (freqs[0],len(freqs))

        if (LOs[index[i]][0] < 0):
            print(line)
            continue
        if (bbc[myspw] > 0):
            if (splitted):
                lo1s[i] = LOs[index[i]][0]
                lo2s[i] = LOs[index[i]][1]
                receiverTypes[i] = receiverSidebands[index[i]]
            else:
                lo1s[myspw] = LOs[index[i]][0]
                lo2s[myspw] = LOs[index[i]][1]
                receiverTypes[myspw] = receiverSidebands[index[i]]
            for j in range(len(LOs[index[i]])):
                if (j != 2):
                    line = line + '%10.6f' % (LOs[index[i]][j]*1e-9)
                else:
                    line = line + '%5.2f' % (LOs[index[i]][j]*1e-9)
        yig = LOs[index[i]][0] / yigHarmonic(bands[index[i]])
        if (yig > 0):
            line = line + ' %.6f' % (yig*1e-9)
        if (spwType(len(freqs)) == 'FDM'):
            # work out what LO4 must have been
            LO1 = LOs[index[i]][0]
            LO2 = LOs[index[i]][1]
            LO3 = LOs[index[i]][2]
            if (sidebands[index[i]][0] == 'USB'):
                IFlocation = LO3 - (LO2 - (meanFreqGHz*1e9 - LO1))
            else:
                IFlocation = LO3 - (LO2 - (LO1 -  meanFreqGHz*1e9))
            LO4 = 2e9 + IFlocation
            TFBLOoffset = LO4-3e9
            line += '%9.3f %+8.3f ' % (LO4 * 1e-6,  TFBLOoffset * 1e-6)
        else:
            line += 19*' '
        if (showEffective):    
            line += '%9.4f %9.4f %9.4f' % (effectiveBandwidth(vis, myspw)*1e-6,
                                           effectiveResolution(vis, myspw)*1e-6,
                                           getChanWidths(mymsmd, myspw)*1e-6)
        if (showWindowFactors):
            chanwidth = abs(getChanWidths(mymsmd,myspw))
            line += ' %.4f %.4f' % (effectiveBandwidth(vis, myspw)/chanwidth,
                                    effectiveResolution(vis, myspw, mymsmd=mymsmd)/chanwidth)
        if (bands[index[i]].find('ALMA_RB_06')>=0 or bands[index[i]].find('ALMA_RB_09')>=0 or bands[index[i]].find('ALMA_RB_10')>=0):
            if (len(LOs[index[i]]) > 1):
                if (LOs[index[i]][1] < 11.3e9 and LOs[index[i]][1] > 10.5e9):
                    line = line + ' leakage of LO2 undesired sideband may degrade dynamic range'
                    if (bands[index[i]].find('ALMA_RB_06')>=0):
                        line += ' (and YIG may leak in)'
                    yigLeakage = LOs[index[i]][0] + (LOs[index[i]][1] - LOs[index[i]][2]) + (yig - LOs[index[i]][1])
                    if (yigLeakage > 0):
                        line = line + ' at %.6f' % (yigLeakage*1e-9)
        if (show): print(line)
    if (casaVersion >= casaVersionWithMSMD):
        if needToClose:
            mymsmd.done()
    if (birdieIF != 0):
        print("The feature at %f GHz is at IF = %f GHz." % (birdieFreqGHz, birdieIF*1e-9))
    if (alsoReturnLO2):
        if alsoReturnReceiverType:
            return(lo1s, lo2s, receiverTypes)
        else:
            return(lo1s, lo2s)
    else:
        if alsoReturnReceiverType:
            return(lo1s, receiverTypes)
        else:
            return(lo1s)

def yigRange(band):
    """
    band: integer from 3..10
    Returns the required tunable range of the YIG to cover the
    ALMA band. (in Hz)
    """
    harmonic = yigHarmonic(band)
    low = float(bandDefinitions[band][0]+maxIFbandDefinitions[band])
    high = float(bandDefinitions[band][1]-maxIFbandDefinitions[band])
    return(low/harmonic, high/harmonic)

def yigHarmonic(bandString):
    """
    Returns the YIG harmonic for the specified ALMA band, given as a string 
    used in casa tables, or an integer.
    bandString: either an integer, or a string of 10 characters
    For example:  yigHarmonic('ALMA_RB_03')  returns the integer 6.
    Todd Hunter
    """
    # remove any leading spaces
    #bandString = bandString[bandString.find('ALMA_RB'):]
    if (bandString in [3,4,6,7,8,9]):
        bandString = 'ALMA_RB_%02d' % bandString
    harmonics = {'ALMA_RB_03':6, 'ALMA_RB_04':6, 'ALMA_RB_06': 18, 
                 'ALMA_RB_07': 18, 'ALMA_RB_08':18, 'ALMA_RB_09':27}
    try:
        harmonic = harmonics[bandString]
    except:
        harmonic = -1
    return(harmonic)

def printLOsFromASDM(sdmfile, spw='', showCentralFreq=True, showYIG=True,
                     showEffective=False, showChannelAverageSpws=False,
                     showWindowFactors=False):
    """
    Prints the values of LO1, LO2 and the TFB LO offset (if applicable).
    If no spw is specified, then it prints the values for all spws in the ASDM.
    The possibility of leakage of YIG and undesired LO2 sideband is printed if present.

    spw: limit the result to a single spw
    showCentralFreq: if False, then show the first channel freq
    showYIG: if True, compute what the YIG frequency must have been
    showEffective: if True, show effectiveBw and resolution
    showChannelAverageSpws: if True, then also show single-channel spws
    showWindowFactors: if True, then show ratio of effective bandwidth and
         effective resolution to the channel width (ICT-2542).
    - Todd Hunter
    """
    if (os.path.exists(sdmfile)==False):
        print("Could not find this ASDM file = %s." % (sdmfile))
        return
    if (type(spw) == list):
        spw = spw
    elif (type(spw) == str):
        if (spw != ''):
            spw = [int(x) for x in spw.split(',')]
    else:
        spw = [int(spw)]

    scandict = getLOsFromASDM(sdmfile)
    scandictspw = getSpwsFromASDM(sdmfile)
    if (showCentralFreq):
        line = "Window spw BB Chan RxId CenFrq(GHz) LO1(GHz)  LO2(GHz) Sampler"
    else:
        line = "Window spw BB Chan RxId Ch1Frq(GHz) LO1(GHz)  LO2(GHz) Sampler"
    if (showYIG):
        line += ' YIG(GHz) '
    line += ' TFBoffset(MHz)'
    if (showEffective):
        line += ' Eff.BW(MHz) Res(MHz) Width(MHz)'
    if (showWindowFactors):
        line += ' windowFactors'
    print(line)
    sawWVR = False
    truespw = -1
    for j in range(len(scandictspw)):
      # find spw
      myspw = -1
      for i in range(len(scandict)):
          if (scandictspw[j]['spectralWindowId'] == scandict[i]['spectralWindowId']):
              myspw = i
              break
      if (myspw < 0):
            continue
      if (scandictspw[j]['numChan'] == 4):
          if (sawWVR): continue
          sawWVR = True
      truespw += 1
      if (spw == '' or truespw in spw):
        myline =  "%4s  %3d %2d " % (scandictspw[j]['windowFunction'][:4], truespw,
                                         scandictspw[j]['basebandNumber'])
        if (scandict[myspw]['frequencyBand'].split('_')[-1].isdigit()):
            # this will be '06' or '10', etc.
            rxband = scandict[myspw]['frequencyBand'].split('_')[-1]
        else:
            rxband = 'WVR'
        myline += '%4d  %3s ' % (scandictspw[j]['numChan'],rxband)
        if (showCentralFreq):
            myline += '%10.6f  ' % (1e-9*scandictspw[j]['centerFreq'])
        else:
            myline += '%10.6f  ' % (1e-9*scandictspw[j]['chanFreqStart'])
        if (scandictspw[j]['basebandNumber'] == 0):
            # don't show (irrelevant) LO for WVR
            print(myline)
            continue
        for i in range(scandict[myspw]['numLO']):
            if (i<2):
                myline += '%10.6f' % (scandict[myspw]['freqLO'][i]*1e-9)
            else:
                myline += '%5.2f' % (scandict[myspw]['freqLO'][i]*1e-9)
        if (showYIG):
            yigHarm = yigHarmonic(scandict[myspw]['frequencyBand'])
            yig = 1e-9 * scandict[myspw]['freqLO'][0] / (yigHarm*1.0)
            myline += '%10.6f' % yig
        LO2 = scandict[myspw]['freqLO'][1]
        if (spwType(scandictspw[j]['numChan']) == 'FDM'):
            # work out what LO4 must have been
            LO1 = scandict[myspw]['freqLO'][0]
            LO3 = scandict[myspw]['freqLO'][2]
            if (scandictspw[j]['sideband'] > 0):
                IFlocation = LO3 - (LO2 - (scandictspw[j]['centerFreq'] - LO1))
            else:
                IFlocation = LO3 - (LO2 - (LO1 - scandictspw[j]['centerFreq']))
            LO4 = 2e9 + IFlocation
            TFBLOoffset = LO4-3e9
            myline += '%9.3f %+8.3f' % (LO4 * 1e-6,  TFBLOoffset * 1e-6)
        else:
            myline += ' '*18
        if (showEffective):
            myline += '%9.4f  %9.4f  %9.4f' % (scandictspw[j]['effectiveBw']*1e-6,
                                               scandictspw[j]['resolution']*1e-6,
                                               scandictspw[j]['chanWidth']*1e-6)
        if (showWindowFactors):
            myline += ' %.4f %.4f' % (scandictspw[j]['effectiveBw']/scandictspw[j]['chanWidth'],
                                      scandictspw[j]['resolution']/scandictspw[j]['chanWidth'])
        if rxband in ['06','09','10'] and LO2 < 11.3e9 and LO2 > 10.5e9:
            myline += ' leakage of LO2 undesired sideband may degrade dynamic range (and YIG may leak in)'
        if (showChannelAverageSpws or scandictspw[j]['numChan'] > 1):
            print(myline)

def spwType(nchan):
    """
    Implements the logic of msmd.almaspws (CAS-5794) to determine the
    type of ALMA spw based on the number of channels.
    -Todd Hunter
    """
    if (nchan >= 15 and nchan not in [256,128,64,32,16,248,124,62,31]):
        t = 'FDM'
    elif (nchan==1):
        t = 'CA'
    elif (nchan==4):
        t = 'WVR'
    else:
        t = 'TDM'
    return(t)
          
def getLOsFromASDM(sdmfile):
    """
    Returns a dictionary of the LO values for every spw in the specified ASDM.
    Dictionary contents: numLO, freqLO, spectralWindowId.  freqLO is itself a
    list of floating point values.
    Todd Hunter
    """
    if (os.path.exists(sdmfile) == False):
        print("getLOsFromASDM(): Could not find file = ", sdmfile)
        return
    xmlscans = minidom.parse(sdmfile+'/Receiver.xml')
    scandict = {}
    rowlist = xmlscans.getElementsByTagName("row")
    fid = 0
    for rownode in rowlist:
        scandict[fid] = {}
        rownumLO = rownode.getElementsByTagName("numLO")
        numLO = int(rownumLO[0].childNodes[0].nodeValue)
        rowfreqLO = rownode.getElementsByTagName("freqLO")
        rowreceiverId = rownode.getElementsByTagName("receiverId")
        receiverId = int(rowreceiverId[0].childNodes[0].nodeValue)
        rowfrequencyBand = rownode.getElementsByTagName("frequencyBand")
        frequencyBand = str(rowfrequencyBand[0].childNodes[0].nodeValue)
        freqLO = []
        r = filter(None,(rowfreqLO[0].childNodes[0].nodeValue).split(' '))
        for i in range(2,len(r)):
            freqLO.append(float(r[i]))
        
        rowspwid = rownode.getElementsByTagName("spectralWindowId")
        spwid = int(str(rowspwid[0].childNodes[0].nodeValue).split('_')[1])
        scandict[fid]['spectralWindowId'] = spwid
        scandict[fid]['freqLO'] = freqLO
        scandict[fid]['numLO'] = numLO
        scandict[fid]['receiverId'] = receiverId
        scandict[fid]['frequencyBand'] = frequencyBand
        fid +=1
    return(scandict)

def getNumChanFromASDM(sdmfile, scienceOnly=False):
    """
    Gets the number of channels of each spw from an ASDM.
    scienceOnly: if True, limit the result to the science spws
    Returns: a dictionary keyed by spw ID, with value = number of channels.
    -Todd Hunter
    """
    spws = getSpwsFromASDM(sdmfile)
    scienceSpws = getScienceSpwsFromASDM(sdmfile)
    mydict = {}
    for spw in spws:
#        print "%2d: %4d" % (spw, spws[spw]['numChan'])
        if not scienceOnly or spw in scienceSpws:
            mydict[spw] = spws[spw]['numChan']
    return(mydict)

def getMeanFreqFromASDM(sdmfile, minnumchan=64, spws=None, scienceOnly=True):
    """
    Gets the mean frequency of the spectral windows for each sideband, assuming
    that the bandwidth of each spw is the same.  Useful for holography data.
    spws: limit the calculation to the specified spws (as they will be numbered in the ms)
    scienceOnly: ignore spws that do not have OBSERVE_TARGET intent

    Returns:
    dictionary of the form: {'lsb': {'meanfreq': meanfreq, 'spws': nspws},
                             'usb': {'meanfreq': meanfreq, 'spws': nspws},
                             'mean': {'meanfreq': meanfreq, 'spws': nspws}}
    -Todd Hunter
    """
    mydict = getSpwsFromASDM(sdmfile, minnumchan)
    lsb = []
    usb = []
    if (type(spws) == str):
        spws = spws.split(',')
    elif (type(spws) == int):
        spws = [spws]
    if scienceOnly:
        scienceSpws = getScienceSpwsFromASDM(sdmfile)
    for spw in mydict:
        if (spws is not None):
            if (spw not in spws): continue
        if scienceOnly:
            if spw not in scienceSpws: continue
        if (mydict[spw]['sideband'] == -1):
            lsb.append(mydict[spw]['centerFreq'])
        elif (mydict[spw]['sideband'] == 1):
            usb.append(mydict[spw]['centerFreq'])
    if len(lsb) < 1:
        meanfreq = np.mean(usb)
    elif len(usb) < 1:
        meanfreq = np.mean(lsb)
    else:
        meanfreq = np.mean([np.mean(usb),np.mean(lsb)])
    mydict = {'lsb': {'meanfreq': np.mean(lsb), 'spws': len(lsb)},
              'usb': {'meanfreq': np.mean(usb), 'spws':len(usb)},
              'mean': {'meanfreq': meanfreq,
                       'spws':len(usb)+len(lsb)}
              }
    return(mydict)

def listobsasdm(asdm, listfile=''):
    """
    Prints a summary of the scans, fields and spws, similar to listobs for
    a measurement set.
    listfile: True to create asdm'.listobs', or a filename
    -Todd Hunter
    """
    if (not os.path.exists(asdm)):
        print("Could not find ASDM.")
        return
    if listfile != '':
        if (listfile == True):
            listfile = asdm + '.listobs'
        f = open(listfile,'w')
    else:
        f = ''
    a = asdm+' observed from %s' % (getObservationDateRangeFromASDM(asdm))
    if listfile != '':
        f.write(a+'\n')
    print(a)
    listscans(asdm, f)
    mydict = getSpwsFromASDM(asdm)
    a = "SpwID Name %s  #Chans Sideband  Ch0(MHz)  ChanWid(kHz)  TotBW(kHz) CtrFreq(MHz) BB" % (36*' ')
    if listfile != '':
        f.write(a+'\n')
    print(a)
    for i,key in enumerate(mydict.keys()):
        nchan = mydict[key]['numChan']
        sideband = mydict[key]['sideband']
        sb = 'DSB' 
        if sideband==-1:
            sb = 'LSB'
        elif sideband==1:
            sb= 'USB'
        a = "  %2d  %-42s  %4d    %s  %11.3f  %11.3f  %10.1f  %11.3f  %d" % (i,
                            mydict[key]['name'],
                            nchan, sb,
                            mydict[key]['chanFreqStart']*1e-6,
                            mydict[key]['chanWidth']*1e-3,
                            nchan*mydict[key]['chanWidth']*1e-3,
                            mydict[key]['centerFreq']*1e-6,
                            mydict[key]['basebandNumber'],
                            )
        if listfile != '':
            f.write(a+'\n')
        print(a)
    if listfile != '':
        f.close()
        print("listobs = ", listfile)

def getSpwsFromASDM(sdmfile, minnumchan=0, dropExtraWVRSpws=True):
    """
    Returns a dictionary of the spw information for every spw in the specified
    ASDM.  Dictionary contents: chanFreqStart, chanWidth, windowFunction, name
       spectralWindowId, numChan, refFreq, sideband, effectiveBw, basebandNumber,
       resolution, centerFreq.  
    The keys are the spw numbers that the spws will have upon loading into a 
    measurement set.
    minnumchan: only return information on those spws with at least this many 
                channels
    Todd Hunter
    """
    if (os.path.exists(sdmfile) == False):
        print("getSpwsFromASDM(): Could not find file = ", sdmfile)
        return
    xmlscans = minidom.parse(sdmfile+'/SpectralWindow.xml')
    scandict = {}
    rowlist = xmlscans.getElementsByTagName("row")
    fid = 0
    firstWVR = True
    for rowNumber, rownode in enumerate(rowlist):
        rowspwid = rownode.getElementsByTagName("spectralWindowId")
        spwid = int(str(rowspwid[0].childNodes[0].nodeValue).split('_')[1])
        rownumLO = rownode.getElementsByTagName("numChan")
        numChan = int(rownumLO[0].childNodes[0].nodeValue)
        if (numChan < minnumchan): 
            if (numChan == 4):
                if (firstWVR):
                    fid += 1
                    firstWVR = False
            else:
                fid += 1
            continue
        scandict[fid] = {}
        rownumLO = rownode.getElementsByTagName("refFreq")
        refFreq = float(rownumLO[0].childNodes[0].nodeValue)
        rownumNOBB = rownode.getElementsByTagName("basebandName")
        noBB = rownumNOBB[0].childNodes[0].nodeValue
        rownumName = rownode.getElementsByTagName("name")
        name = rownumName[0].childNodes[0].nodeValue
        rownumWF = rownode.getElementsByTagName("windowFunction")
        windowFunction = str(rownumWF[0].childNodes[0].nodeValue)
        try:
            rownumWF = rownode.getElementsByTagName("effectiveBw")
            effectiveBw = float(rownumWF[0].childNodes[0].nodeValue)
        except:
            rownumWF = rownode.getElementsByTagName("effectiveBwArray")
            effectiveBw = float(list(filter(None,(rownumWF[0].childNodes[0].nodeValue).split()))[2])
        try:
            rownumWF = rownode.getElementsByTagName("chanWidth")
            chanWidth = float(rownumWF[0].childNodes[0].nodeValue)
        except:
            rownumWF = rownode.getElementsByTagName("chanWidthArray")
            chanWidth = float(list(filter(None,(rownumWF[0].childNodes[0].nodeValue).split()))[2])
        try:
            rownumWF = rownode.getElementsByTagName("resolution")
            resolution = float(rownumWF[0].childNodes[0].nodeValue)
        except:
            rownumWF = rownode.getElementsByTagName("resolutionArray")
            resolution = float(list(filter(None,(rownumWF[0].childNodes[0].nodeValue).split()))[2])
        scandict[fid]['spectralWindowId'] = spwid
        scandict[fid]['numChan'] = numChan
        scandict[fid]['refFreq'] = refFreq
        scandict[fid]['windowFunction'] = windowFunction
        scandict[fid]['effectiveBw'] = effectiveBw
        scandict[fid]['resolution'] = resolution
        scandict[fid]['chanWidth'] = chanWidth
        scandict[fid]['name'] = name
        if (noBB == 'NOBB'):
            scandict[fid]['basebandNumber'] = 0
        else:
            scandict[fid]['basebandNumber'] = int(noBB.split('_')[-1])
        try:
            rownumLO = rownode.getElementsByTagName("chanFreqStart")
            chanFreqStart = float(rownumLO[0].childNodes[0].nodeValue)
            rownumLO = rownode.getElementsByTagName("chanFreqStep")
            chanFreqStep = float(rownumLO[0].childNodes[0].nodeValue)
            if ((numChan % 2) == 1):
                centerFreq = chanFreqStart + (numChan/2)*chanFreqStep
            else:
                centerFreq = chanFreqStart + (numChan-1)*0.5*chanFreqStep
        except:
            try:
                rownumLO = rownode.getElementsByTagName("chanFreqArray")
                r = list(filter(None,(rownumLO[0].childNodes[0].nodeValue).split(' ')))
                freqLO = []
                for i in range(2,len(r)):
                    freqLO.append(float(r[i]))
                centerFreq = np.mean(freqLO)
                chanFreqStart = freqLO[0]
            except:
                print("Did not find chanFreqStart nor chanFreqArray on row=%d, spw=%d" % (fid,spwid))
                scandict[fid]['centerFreq'] = 0
                continue
        scandict[fid]['centerFreq'] = centerFreq
        scandict[fid]['chanFreqStart'] = chanFreqStart
        if (refFreq > centerFreq):
            scandict[fid]['sideband'] = -1
        else:
            scandict[fid]['sideband'] = +1
        if (scandict[fid]['numChan'] != 4 or firstWVR or dropExtraWVRSpws==False):
            if (firstWVR and scandict[fid]['numChan'] == 4):
                scandict[fid]['sideband'] = 0
                firstWVR = False
            fid +=1
    return(scandict)

def getReceiverId(vis):
    """
    Reads the LO and receiver information from the ASDM_RECEIVER table of 
    a measurement set. -Todd Hunter
    """
    result = getLOs(vis, verbose=False)
    if len(result) == 0: return
    [freqLO,band,spws,names,sidebands,receiverIds,spwNames] = result
    receiverId = np.unique(receiverIds)
    if len(receiverId) == 1: # this is the cyrostat ID number
        # Should only ever be one value (i.e. same for all spws)
        return receiverId[0]
    else:
        return receiverId
    
def getLOs(inputMs, verbose=True, returnLOs_as_list=False, returnReceiverSidebands=False):
    """
    Reads the LO information from an ms's ASDM_RECEIVER table.  It returns
    a list of 7 lists: [freqLO,band,spws,names,sidebands,receiverIDs,spwnames]
    The logic for converting this raw list into sensible association with
    spw numbers is in printLOs().  These lists are longer than the true number 
    of spws by Nantennas-1 due to the extra WVR spws.
    returnReceiverSidebands: if True, then return an 8th list of sideband strings
    -Todd Hunter
    """
    if (os.path.exists(inputMs)):
        mytb = createCasaTool(tbtool)
        if (os.path.exists("%s/ASDM_RECEIVER" % inputMs)):
            try:
                mytb.open("%s/ASDM_RECEIVER" % inputMs)
            except:
                print("Could not open the existing ASDM_RECEIVER table")
                mytb.close()
                return([])
        else:
            if (os.path.exists(inputMs+'/ASDMBinary')):
                print("This is an ASDM, not an ms!  Use printLOsFromASDM.")
            else:
                if (verbose):
                    print("The ASDM_RECEIVER table for this ms does not exist (%s)." % (inputMs))
            mytb.close()
            return([])
    else:
        print("This ms does not exist = %s." % (inputMs))
        return([])
        
    numLO = mytb.getcol('numLO')
    freqLO = []
    band = []
    spws = []
    names = []
    sidebands = []
    receiverIds = []
    if returnReceiverSidebands:
        receiverSidebands = []
    for i in range(len(numLO)):
        spw = int((mytb.getcell('spectralWindowId',i).split('_')[1]))
        if (spw not in spws):
            spws.append(spw)
            if returnLOs_as_list:
                freqLO.append(list(mytb.getcell('freqLO',i)))
            else:
                freqLO.append(mytb.getcell('freqLO',i))
            band.append(mytb.getcell('frequencyBand',i))
            names.append(mytb.getcell('name',i))
            sidebands.append(mytb.getcell('sidebandLO',i))
            receiverIds.append(int(mytb.getcell('receiverId',i)))
            if returnReceiverSidebands:
                receiverSidebands.append(str(mytb.getcell('receiverSideband',i)))
    mytb.close()
    mytb.open("%s/SPECTRAL_WINDOW" % inputMs)
    spwNames = mytb.getcol("NAME")
    mytb.close()
    if returnReceiverSidebands:
        return([freqLO,band,spws,names,sidebands,receiverIds,spwNames,receiverSidebands])
    else:
        return([freqLO,band,spws,names,sidebands,receiverIds,spwNames])
    
def mjdVectorToUTHours(mjd):
    """
    Converts a vector of MJD values to a vector of floating point hours (0-24).
    Todd Hunter
    """
    return(24*(np.modf(mjd)[0]))

def mjdSecondsVectorToMJD(mjdsec):
    return(mjdsec / 86400.0)

def call_qa_time(arg, form='', prec=0, showform=False):
    """
    This is a wrapper for qa.time(), which in casa 4.0.0 returns a list 
    of strings instead of just a scalar string.  
    arg: a time quantity
    - Todd Hunter
    """
    if (type(arg) == dict):
        if (type(arg['value']) == list or 
            type(arg['value']) == np.ndarray):
            if (len(arg['value']) > 1):
                print("WARNING: call_qa_time() received a dictionary containing a list of length=%d rather than a scalar. Using first value." % (len(arg['value'])))
            arg['value'] = arg['value'][0]
    myqa = createCasaTool(qatool)
    result = myqa.time(arg, form=form, prec=prec, showform=showform)
    myqa.done()
    if (type(result) == list or type(result) == np.ndarray):
        return(result[0])
    else:
        return(result)

def call_qa_angle(arg, form='', prec=0, showform=False):
    """
    This is a wrapper for qa.angle(), which in casa 4.0.0 returns a list 
    of strings instead of just a scalar string.  
    - Todd Hunter
    """
    if (type(arg) == dict):
        if (type(arg['value']) == list or 
            type(arg['value']) == np.ndarray):
            if (len(arg['value']) > 1):
                print("WARNING: call_qa_angle() received a dictionary containing a list of length=%d rather than a scalar. Using first value." % (len(arg['value'])))
            arg['value'] = arg['value'][0]
    myqa = createCasaTool(qatool)
    result = myqa.angle(arg, form=form, prec=prec, showform=showform)
    myqa.done()
    if (type(result) == list or type(result) == np.ndarray):
        return(result[0])
    else:
        return(result)

def mjdToYear(mjd):
    """
    Converts 56342.1 to 2013.132876
    -Todd Hunter
    """
    return yearFraction(mjdToDate(mjd))

def epochToDatestring(epoch):
    """
    Inverse of yearFraction().  Runs au.mjdToDatestring(au.epochToMJD()).
    -Todd Hunter
    """
    return mjdToDatestring(epochToMJD(epoch))

def epochToJD(epoch):
    """
    Runs mjdtoJD(epochToMJD())
    Converts 2000.00000000 to 2451545.0 (2000-01-01 12 UT) as per definition
    -Todd Hunter
    """
    return mjdToJD(epochToMJD(epoch))

def epochToMJD(epoch):
    """
    Converts 2013.13315068 to 56342.100  (2013-02-19 02:24:00)
         and 2000.00000000 to 51544.5 (2000-01-01 12 UT) as per definition
    """
    year = int(epoch)
    mjd = strDate2MJD('%d-01-01'%(year))
    mydt = mjdListToDateTime([mjd])[0]
    daysInYear = (datetime.datetime(mydt.year,12,31)-datetime.datetime(mydt.year,1,1)).days+1
    days = daysInYear*(epoch - year) + 0.5
    mjd += days
    return mjd

def yearFraction(date=''):
    """
    Gets the fractional year for a specified date time string.
    Input date format: 2011/10/15 05:00:00  or   2011/10/15-05:00:00
                    or 2011-10-15 05:00:00  or   2011-10-15-05:00:00
                    or 2011-10-15T05:00:00  or   2011-Oct-15 etc.
                    or 20111015 with no time
    The time portion is optional.
    returns:   2011.785502283105
     Note: returns 2000.000 for 2000/01/01 12:00:00 UT as per definition
    -Todd Hunter
    """
    if (date == ''): date = mjdToUT()
    if len(date) == 8:
        date = '-'.join([date[:4],date[4:6],date[6:]])
    doy = doyFromDate(date)
    mjd = dateStringToMJD(date,verbose=False)
    mydt = mjdListToDateTime([mjd])[0]
    daysInYear = (datetime.datetime(mydt.year,12,31)-datetime.datetime(mydt.year,1,1)).days+1
    return mydt.year+(doy-1.5+(mjd%1))*1.0/daysInYear

def doyFromDate(date='', fractional=False):
    """
    Gets the day number for a specified date time string"
    Input date format: 2011/10/15 05:00:00  or   2011/10/15-05:00:00
                    or 2011-10-15 05:00:00  or   2011-10-15-05:00:00
                    or 2011-10-15T05:00:00  or   2011-Oct-15 etc.
                    or 20111015 with no time
    The time portion is optional.
    Returns: integer,  e.g. 288
    -Todd Hunter
    """
    if (date == ''): date = mjdToUT()
    if len(date) == 8:
        date = '-'.join([date[:4],date[4:6],date[6:]])
    mydt = mjdListToDateTime([dateStringToMJD(date,verbose=False)])[0]
    diff = (mydt - datetime.datetime(mydt.year, 1, 1))
    if fractional:
        doy = 1 + diff.days + (diff.seconds % 86400)/86400.
    else:
        doy = 1 + diff.days
    return doy

def dateFromDoy(year, doy, delimiter='/', hms=False):
    """
    Build a date string from the year and day-of-year.
    Inputs:
    year: integer
    doy: integer (1= Jan1)
    delimiter: string to delimit year, month, day
    hms: if True, then also include HH:MM:SS
    Returns: string of format: YYYY/MM/DD
    -Todd Hunter
    """
    mydate = datetime.datetime(year, 1, 1) + datetime.timedelta(days=doy-1)
    if hms:
        myformat = "%%Y%s%%m%s%%d%s%%H:%%M:%%S" % (delimiter, delimiter,delimiter)
    else:
        myformat = "%%Y%s%%m%s%%d" % (delimiter, delimiter)
    mystring = datetime.datetime.strftime(mydate,myformat)
    return mystring

def getMJD():
    """
    Returns the current MJD.  See also getCurrentMJDSec().
    -Todd
    """
    myme = createCasaTool(metool)
    mjd = myme.epoch('utc','today')['m0']['value']
    myme.done()
    return(mjd)
    
def mjdListToDateTime(mjdList):
    """
    Takes a list of mjd values and converts it to a list of datetime 
    structures.
    - Todd Hunter
    """
    return(mjdSecondsListToDateTime(np.array(mjdList)*86400.0))

def dateTimeListToMJDSeconds(dateTimeList, verbose=False):
    """
    Converts a list of datetime structures to a list of MJD seconds (floating point).
    Inverse of mjdSecondsListToDateTime.
    -Todd Hunter.
    """
    mjdsec = []
    for mydate in dateTimeList:
        mystring = datetime.datetime.strftime(mydate,"%Y/%m/%d %H:%M:%S.%f")
        mjdsec.append(dateStringToMJDSec(mystring,verbose=verbose))
    return(mjdsec)
              
def mjdSecondsListToDateTime(mjdsecList, use_metool=True):
    """
    Takes a list of mjd seconds and converts it to a list of datetime 
    structures.
    use_metool: whether or not to use the CASA measures tool if running from 
        CASA. Automatically set False if CASAPATH is not defined.  Note that 
        results are truncated to seconds if use_metool=False.
    - Todd Hunter
    """
    if (casaAvailable and use_metool):
        myme = createCasaTool(metool)
    dt = []
    typelist = type(mjdsecList)
    if not (typelist == list or typelist == np.ndarray):
        mjdsecList = [mjdsecList]
    for mjdsec in mjdsecList:
        if (not casaAvailable or use_metool==False):
            jd = mjdToJD(mjdsec / 86400.)
            trialUnixTime = 1200000000
            diff  = ComputeJulianDayFromUnixTime(trialUnixTime) - jd
            trialUnixTime -= diff*86400
            diff  = ComputeJulianDayFromUnixTime(trialUnixTime) - jd
            trialUnixTime -= diff*86400
            diff  = ComputeJulianDayFromUnixTime(trialUnixTime) - jd
            timeString = timeUtilities.strftime('%d-%m-%Y %H:%M:%S.0', 
                                                timeUtilities.gmtime(trialUnixTime))
        else:
            today = myme.epoch('utc','today')
            mjd = mjdsec / 86400.
            today['m0']['value'] =  mjd
            hhmmss = call_qa_time(today['m0'])
            myqa = createCasaTool(qatool)
            date = myqa.splitdate(today['m0'])  # date is now a dict
            myqa.done()
            timeString = '%d-%d-%d %d:%d:%d.%06d'%(date['monthday'],date['month'],date['year'],date['hour'],date['min'],date['sec'],date['usec'])
        mydate = datetime.datetime.strptime(timeString,'%d-%m-%Y %H:%M:%S.%f')
        # previous implementation truncated to nearest second!
#        mydate = datetime.datetime.strptime('%d-%d-%d %d:%d:%f'%(date['monthday'],date['month'],date['year'],date['hour'],date['min'],date['s']),'%d-%m-%Y %H:%M:%S')
        dt.append(mydate)
    if (casaAvailable and use_metool):
        myme.done()
    return(dt)

def mjdToPredictcomp(MJD=None):
    """
    Converts an MJD into a string suitable for the epoch parameter of
    predictcomp:
    i.e., 2011-12-31-5:34:12
    -- Todd Hunter
    """
    if (MJD is None):
        MJD = getMJD()
    mystring = mjdsecToUT(MJD*86400)
    tokens = mystring.split()
    return(tokens[0]+'-'+tokens[1])
    
def mjdToJD(MJD=None):
    """
    Converts an MJD value to JD.  Default = now.
    """
    if (MJD is None): MJD = getMJD()
    JD = MJD + 2400000.5
    return(JD)

def ymdhmsToMJD(year,month,day,hour=0,minute=0,second=0.0):
    """
    converts 2010,12,31 to MJD on Dec 31, 2010 at UT=0
    converts 2010,12,31,9.50 to MJD on Dec 31, 2010 at UT=09:30
    converts 2010,12,31,9,0,5 to MJD on Dec 31, 2010 at UT=09:05
    required arguments: year, month, day
    optional arguments: hour, minute, second
    """
    if (month < 3):
        month += 12
        year -= 1
    a =  np.floor(year / 100.)
    b = 2 - a + np.floor(a/4.)
    UT = hour+minute/60.+second/3600.
    day += UT/24.
    jd  = np.floor(365.25*(year+4716)) + np.floor(30.6001*(month+1)) + day + b - 1524.5;
    mjd = jdToMJD(jd)
    return(mjd)

def jdToMJD(JD):
    """
    Converts a JD value to MJD by subtracting 2400000.5
    """
    MJD = JD - 2400000.5
    return(MJD)

def mjdToDate(mjd, measuresToolFormat=False):
    """
    Converts an MJD value to the date string YYYY-MM-DD.
    For a string with date and time, use mjdsecToUT().
    -Todd Hunter
    """
    return mjdsecToDate(mjd*86400, measuresToolFormat)

def mjdToDatestring(mjd, delimiter='-'):
    """
    Converts an MJD value to the date string YYYY-MM-DD HH:MM:SS.
    mjd: floating point value or string
    delimiter: string to use to separate YYYY from MM and DD
    -Todd Hunter
    """
    return mjdsecToDatestring(float(mjd)*86400, delimiter)

def mjdsecToDate(mjdsec=None, measuresToolFormat=False):
    """
    Converts an MJD seconds value to the date string YYYY-MM-DD.
    For a string with date and time, use mjdsecToUT().
    -Todd Hunter
    """
    if mjdsec is None:
        mjdsec = getMJDSec()
    if (type(mjdsec) == list or type(mjdsec) == np.ndarray):
        retval = [mjdsecToUT(m).split()[0] for m in mjdsec]
        if measuresToolFormat:
            retval = [mjdsecToUT(m).split()[0].replace(' UT','').replace('-','/').replace(' ','/') for m in mjdsec]
    else:
        retval =  mjdsecToUT(mjdsec).split()[0]
        if measuresToolFormat:
            retval = retval.replace(' UT','').replace('-','/').replace(' ','/')
    return(retval)

def mjdsecToUT(mjdsec=None, prec=6, measuresToolFormat=False):
    """
    Converts an MJD seconds value to a UT date and time string
    such as '2012-03-14 00:00:00 UT'.  For a string with only the
    date, use mjdsecToDate().
    -Todd Hunter
    """
    if mjdsec is None: mjdsec = getCurrentMJDSec()
    if (type(mjdsec) == list or type(mjdsec) == np.ndarray):
        retval = [mjdSecondsToMJDandUT(m, prec=prec)[1] for m in mjdsec]
        if measuresToolFormat:
            retval = [mjdSecondsToMJDandUT(m, prec=prec)[1].replace(' UT','').replace('-','/').replace(' ','/') for m in mjdsec]
    else:
        retval = mjdSecondsToMJDandUT(mjdsec, prec=prec)[1]
        if measuresToolFormat:
            retval = retval.replace(' UT','').replace('-','/').replace(' ','/')
    return(retval)
        
def mjdsecToUTHMS(mjdsec=None, prec=6):
    """
    Converts MJD seconds to HH:MM:SS
    prec: 6 means HH:MM:SS,  7 means HH:MM:SS.S
    -Todd Hunter
    """
    if mjdsec is None: mjdsec = getCurrentMJDSec()
    hms = mjdsecToUT(mjdsec, prec=prec).split()[1]
    return(hms)
    
def mjdsecToUTHM(mjdsec=None):
    """
    Converts MJD seconds to HH:MM
    -Todd Hunter
    """
    if mjdsec is None: mjdsec = getCurrentMJDSec()
    hms = mjdsecToUTHMS(mjdsec)
    hm = hms.split(':')[0] + ':' + hms.split(':')[1]
    return(hm)

def mjdToUT(mjd=None, use_metool=True, prec=6):
    """
    Converts an MJD value to a UT date and time string
    such as '2012-03-14 00:00:00 UT'
    use_metool: whether or not to use the CASA measures tool if running from CASA.
         This parameter is simply for testing the non-casa calculation.
    -Todd Hunter
    """
    if mjd is None:
        mjdsec = getCurrentMJDSec()
    else:
        mjdsec = mjd*86400
    utstring = mjdSecondsToMJDandUT(mjdsec, use_metool, prec=prec)[1]
    return(utstring)
        
def mjdNanosecondsToMJDandUT(mjdnanosec, prec=6, delimiter='-'):
    """
    -Todd Hunter
    """
    return(mjdSecondsToMJDandUT(mjdnanosec*1e-9, prec=prec, delimiter=delimiter))

def acsTimeToUnixTime(acsTime):
    return (acsTime - 122192928e9)/1e7

def dateStringToUnixTime(datestring):
    """
    Converts a UT date string to unix time"
    datestring: format YYYY-MM-DD HH:MM:SS
    -Todd Hunter
    """
    unixtime = dateStringDifference(datestring,'1970-01-01 00:00:00 UT')*60
    return unixtime

def unixTimeToDateString(unixtime=None):
    """
    Converts a time in unix time (seconds since 1970) to a string
    of the format 'YYYY-MM-DD HH:MM:SS UT'
    -Todd Hunter
    """
    if (unixtime == None):
        unixtime = getCurrentUnixTime()
    # could try this someday:
    #      return(datetime.datetime.fromtimestamp(unixtime).astimezone(pytz.utc).strftime('%Y-%m-%d %H:%M:%S UT'))
    jd = ComputeJulianDayFromUnixTime(unixtime)
    mjd = jdToMJD(jd)
    utstring = mjdsecToUT(mjd*86400)
    return(utstring)

def unixTimeToMJD(seconds):
    """
    Uses ComputeJulianDayFromUnixTime and jdToMJD.
    -Todd Hunter
    """
    return(jdToMJD(ComputeJulianDayFromUnixTime(seconds)))

def ComputeJulianDayFromUnixTime(seconds):
    """
    Converts a time expressed in unix time (seconds since Jan 1, 1970)
    into Julian day number as a floating point value.
    - Todd Hunter
    """
    [tm_year, tm_mon, tm_mday, tm_hour, tm_min, tm_sec, tm_wday, tm_yday, tm_isdst] = timeUtilities.gmtime(seconds)
    if (tm_mon < 3):
        tm_mon += 12
        tm_year -= 1
    UT = tm_hour + tm_min/60. + tm_sec/3600.
    a =  np.floor(tm_year / 100.)
    b = 2 - a + np.floor(a/4.)
    day = tm_mday + UT/24.
    jd  = np.floor(365.25*((tm_year)+4716)) + np.floor(30.6001*((tm_mon)+1))  + day + b - 1524.5
    return(jd) 

def casapath():
    """
    Returns the root path of CASA.
    """
    return(os.getenv('CASAPATH').split()[0])

def mjdsecToDatestring(mjdsec='', delimiter='-'):
    """
    Converts a value (or list) in MJD seconds to YYYY-MM-DD HH:MM:SS
    delimiter: string to use to separate YYYY from MM and DD
    """
    if type(mjdsec) == str:
        if mjdsec == '':
            mjdsec = getMJDSec()
    if type(mjdsec) == list or type(mjdsec) == np.ndarray:
        values = []
        for i in mjdsec:
            values.append(mjdSecondsToMJDandUT(i,delimiter=delimiter)[1].split('UT')[0].strip())
        return values
    else:
        return(mjdSecondsToMJDandUT(mjdsec,delimiter=delimiter)[1].split('UT')[0].strip())

def mjdSecondsToMJDandUT(mjdsec, use_metool=True, debug=False, prec=6, delimiter='-'):
    """
    Converts a value of MJD seconds into MJD, and into a UT date/time string.
    prec: 6 means HH:MM:SS,  7 means HH:MM:SS.S
    example: (56000.0, '2012-03-14 00:00:00 UT')
    Caveat: only works for a scalar input value
    Todd Hunter
    """
    if (not casaAvailable or use_metool==False):
        mjd = mjdsec / 86400.
        jd = mjdToJD(mjd)
        trialUnixTime = 1200000000
        diff  = ComputeJulianDayFromUnixTime(trialUnixTime) - jd
        if (debug): print("first difference = %f days" % (diff))
        trialUnixTime -= diff*86400
        diff  = ComputeJulianDayFromUnixTime(trialUnixTime) - jd
        if (debug): print("second difference = %f seconds" % (diff*86400))
        trialUnixTime -= diff*86400
        diff  = ComputeJulianDayFromUnixTime(trialUnixTime) - jd
        if (debug): print("third difference = %f seconds" % (diff*86400))
        # Convert unixtime to date string 
        utstring = timeUtilities.strftime('%Y'+delimiter+'%m'+delimiter+'%d %H:%M:%S UT', 
                                          timeUtilities.gmtime(trialUnixTime))
    else:
        me = createCasaTool(metool)
        today = me.epoch('utc','today')
        mjd = np.array(mjdsec) / 86400.
        today['m0']['value'] =  mjd
        hhmmss = call_qa_time(today['m0'], prec=prec)
        myqa = createCasaTool(qatool)
        date = myqa.splitdate(today['m0'])
        myqa.done()
        utstring = "%s%s%02d%s%02d %s UT" % (date['year'],delimiter,date['month'],delimiter,
                                             date['monthday'],hhmmss)
    return(mjd, utstring)

def mjdsecToTimerange(mjdsec1, mjdsec2=None, decimalDigits=2, includeDate=True, 
                      use_metool=True, debug=False):
    """
    Converts two value of MJD seconds into a UT date/time string suitable for 
    the timerange argument in plotms.  They can be entered as two separated values,
    or a single tuple.
    Example output: '2012/03/14/00:00:00.00~2012/03/14/00:10:00.00'
    input options:
       decimalDigits: how many digits to display after the decimal point
       use_metool: True=use casa tool to convert to UT, False: use formula in aU
    -Todd Hunter
    """
    if (type(mjdsec1) == list or type(mjdsec1)==np.ndarray):
        mjdsec2 = mjdsec1[1]
        mjdsec1 = mjdsec1[0]
    return(mjdsecToTimerangeComponent(mjdsec1, decimalDigits, includeDate, use_metool, debug) + '~' \
           + mjdsecToTimerangeComponent(mjdsec2, decimalDigits, includeDate, use_metool, debug))
        
def mjdsecToTimerangeComponent(mjdsec, decimalDigits=2, includeDate=True, use_metool=True, debug=False):
    """
    Converts a value of MJD seconds into a UT date/time string suitable for one
    member of the timerange argument in plotms.
    example: '2012/03/14/00:00:00.00'
    input options:
       decimalDigits: how many digits to display after the decimal point
       use_metool: True=use casa tool to convert to UT, False: use formula in aU
    Todd Hunter
    """
    if (not casaAvailable or use_metool==False):
        mjd = mjdsec / 86400.
        jd = mjdToJD(mjd)
        trialUnixTime = 1200000000
        diff  = ComputeJulianDayFromUnixTime(trialUnixTime) - jd
        if (debug): print("first difference = %f days" % (diff))
        trialUnixTime -= diff*86400
        diff  = ComputeJulianDayFromUnixTime(trialUnixTime) - jd
        if (debug): print("second difference = %f seconds" % (diff*86400))
        trialUnixTime -= diff*86400
        diff  = ComputeJulianDayFromUnixTime(trialUnixTime) - jd
        if (debug): print("third difference = %f seconds" % (diff*86400))
        # Convert unixtime to date string
        if (includeDate):
            utstring = timeUtilities.strftime('%Y/%m/%d/%H:%M:%S', 
                                              timeUtilities.gmtime(trialUnixTime))
        else:
            utstring = timeUtilities.strftime('%H:%M:%S', 
                                              timeUtilities.gmtime(trialUnixTime))
        utstring += '.%0*d'  % (decimalDigits, np.round(10**decimalDigits*(trialUnixTime % 1)))
    else:
        me = createCasaTool(metool)
        today = me.epoch('utc','today')
        mjd = np.array(mjdsec) / 86400.
        today['m0']['value'] =  mjd
        hhmmss = call_qa_time(today['m0'],prec=6+decimalDigits)
        myqa = createCasaTool(qatool)
        date = myqa.splitdate(today['m0'])
        myqa.done()
        if (includeDate):
            utstring = "%s/%02d/%02d/%s" % (date['year'],date['month'],date['monthday'],hhmmss)
        else:
            utstring = hhmmss
    return(utstring)

def addDays(date1, days=1):
    """
    Takes a string of format 'YYYY-MM-DD' and returns a new string a specified
    number of days later.  Can also use 'YYYYMMDD' or 'YYYY/MM/DD'.
    -Todd Hunter
    """
    return(subtractDays(date1, -days))

def subtractDays(date1, days=1):
    """
    Takes a string of format 'YYYY-MM-DD' and returns a new string a specified
    number of days before.  Can also use 'YYYYMMDD' or 'YYYY/MM/DD'.
    To add days, simply set days to be negative.
    See also computeIntervalBetweenTwoDays.
    -Todd Hunter
    """
    insertDash = False
    insertSlash = False
    if (date1.find('-')>0):
        insertDash = True
    elif (date1.find('/')>0):
        insertSlash = True
    date1 = date1.replace('-','').replace('/','')
    date0 = datetime.date(int(date1[0:4]),int(date1[4:6]),int(date1[6:])) - datetime.timedelta(days=int(days))
    if (insertDash):
        date0 = datetime.datetime.strftime(date0,'%Y-%m-%d')
    elif (insertSlash):
        date0 = datetime.datetime.strftime(date0,'%Y/%m/%d')
    else:
        date0 = datetime.datetime.strftime(date0,'%Y%m%d')
    return(date0)

def computeIntervalBetweenTwoDays(date1, date2):
    """
    Takes 2 strings of format 'YYYY-MM-DD' and returns the number of
    days between them.  Positive if date1 > date2.  See also subtractDays,
    and dateStringDifference (if you need to include HH:MM:SS).
    -Todd Hunter
    """
    date1 = date1.replace('-','').replace('/','')
    date2 = date2.replace('-','').replace('/','')
    delta = datetime.date(int(date1[0:4]),int(date1[4:6]),int(date1[6:])) - \
            datetime.date(int(date2[0:4]),int(date2[4:6]),int(date2[6:]))
    return(delta.days)

def fillZerosInDate(datestring):
    """
    Converts a string like '2014-5-6' into '2014-05-06'.
    """
    if (datestring.find('-') > 0):
        key = '-'
    elif (datestring.find('/') > 0):
        key = '/'
    else:
        return datestring
    datestring = datestring.split(key)
    datestring = ['%02d'%int(i) for i in datestring]
    datestring = key.join(datestring)
    return datestring

def replaceMonth(datestring):
    """
    Replaces a 3-character month string with its 2-character integer string.
    -Todd Hunter
    """
    for i,month in enumerate(['jan','feb','mar','apr','may','jun','jul',
                              'aug','sep','oct','nov','dec']):
        datestring = datestring.lower().replace(month,'%02d'%(i+1))
    return(datestring.upper())
    
def dateStringToMJD(datestring='', datestring2='', verbose=True, use_metool=True):
    """
    Convert a date/time string to floating point MJD
    Input date format: 2011/10/15 05:00:00  or   2011/10/15-05:00:00
                    or 2011-10-15 05:00:00  or   2011-10-15-05:00:00
                    or 2011-10-15T05:00:00  or   2011-Oct-15 etc.
                    or 2011-10-15_05:00:00
    The time portion is optional.
    If a second string is given, both values will be converted and printed,
    but only the first is returned.
    use_metool: this parameter is simply for testing the non-casa calculation
    -- Todd Hunter
    """
    if (datestring == ''):
        datestring = getCurrentDate()
    datestring = replaceMonth(datestring)
    if (datestring2 != ''): datestring2 = replaceMonth(datestring2)
    if (datestring.find('/') < 0):
        if (datestring.count('-') == 3 or datestring.find('T')>0):
            # This is needed to accept 2010-01-01-12:00:00 (accepted by predictcomp)'
            #                     or   2010-01-01T12:00:00 (output by ALMA TMC DB)'
            # by making it look like 2010-01-01 12:00:00'
            datestring = datestring[0:10] + ' ' + datestring[11:]
        if (datestring.count('-') != 2):
            print("Date format: 2011/10/15 05:00:00  or   2011/10/15-05:00:00")
            print("           or 2011-10-15 05:00:00  or   2011-10-15T05:00:00")
            print("The time portion is optional.")
            return(None)
        else:
            d = datestring.split('-')
            datestring = d[0] + '/' + d[1] + '/' + d[2].split('T')[0]
    if (datestring2 != ''):
      if (datestring2.find('/') < 0):
        if (datestring2.count('-') == 3 or datestring2.find('T')>0):
            # This is needed to accept 2010-01-01-12:00:00 (accepted by predictcomp)'
            #                     or   2010-01-01T12:00:00 (output by ALMA TMC DB)'
            # by making it look like 2010-01-01 12:00:00'
            datestring2 = datestring2[0:10] + ' ' + datestring2[11:]
        if (datestring2.count('-') != 2):
            print("Date format: 2011/10/15 05:00:00  or   2011/10/15-05:00:00")
            print("           or 2011-10-15 05:00:00  or   2011-10-15T05:00:00")
            print("The time portion is optional.")
            return(None)
        else:
            d = datestring2.split('-')
            datestring2 = d[0] + '/' + d[1] + '/' + d[2]
    mjd1 = dateStringToMJDSec(datestring, verbose=verbose,
                              use_metool=use_metool) / 86400.
    if (datestring2 != ''):
        mjd2 = dateStringToMJDSec(datestring2, verbose=verbose,
                                  use_metool=use_metool) / 86400.
    return(mjd1)

def replaceFinalAppearanceOfSubstringInString(string, substring, newvalue=' '):
    return string[::-1].replace(substring,newvalue,1)[::-1]

def dateStringListToPlotDate(mylist):
    """
    Converts a list of datetime strings to a list of time axis values that can
    be passed to pylab.plot_date().
    -Todd Hunter
    """
    mjdsecList = dateStringListToMJDSecList(mylist)
    list_of_date_times = mjdSecondsListToDateTime(mjdsecList)
    times = pb.date2num(list_of_date_times)
    return times

def dateStringListToMJDSecList(datestrings):
    """
    Calls dateStringToMJDSec for a list of date strings.
    Returns: a list of MJDseconds
    -Todd Hunter
    """
    mjdsec = []
    for d in datestrings:
        mjdsec.append(dateStringToMJDSec(d, verbose=False))
    return mjdsec
    
def dateStringToMJDSec(datestring='2011/10/15 05:00:00', datestring2='',
                       verbose=True, use_metool=True, plotrangeY=[0,0]):
    """
    Converts one or two date strings into MJD seconds.  This is useful for passing
    time ranges to plotms, because they must be specified in mjd seconds.
    Because me.epoch is used to translate date strings (if use_metool=True), 
    any of these formats is valid: 2011/10/15 05:00:00
                                   2011/10/15/05:00:00
                                   2011/10/15-05:00:00
                                   2011-10-15 05:00:00
                                   2011-10-15_05:00:00
                                   2011-10-15T05:00:00
                                   2011-Oct-15T05:00:00
                                   15-Oct-2011/5:00:00 (listobs format)
    The time portion is optional. The listobs format only works with use_metool=True
    If a second string is given, both values will be converted and a
    string will be created that can be used as a plotrange in plotms.
    To convert from dates like '20110809', see au.strDate2MJD
    use_metool: this parameter is simply for testing the non-casa calculation
    
    For further help and examples, see:
    https://safe.nrao.edu/wiki/bin/view/ALMA/DateStringToMJDSec
    Todd Hunter
    """
    # me.epoch supports '29-Oct-2016 12:12:12', but does not support '29/Oct/2016 12:12:12'
    # so change all slashes to dashes
    datestring = datestring.replace('/','-')
    if (datestring.find('T') > 0):
        datestring = datestring.replace('T',' ')
    if (datestring.find('_') > 0):
        datestring = datestring.replace('_',' ')
    if (datestring.find('"') > 0):
        datestring = datestring.replace('"','')
    if (datestring[-1] in ['-','/']):
        datestring = datestring[:-1]
    mydate = datestring.split()
    if (len(mydate) < 2):
        # count slashes and dashes; if only one, then change it to a space
        n = datestring.count('-')
        if n in [1,3]:
            datestring = replaceFinalAppearanceOfSubstringInString(datestring,'-',' ')
        else:
            n = datestring.count('/')
            if n in [1,3]:
                datestring = replaceFinalAppearanceOfSubstringInString(datestring,'/',' ')
        mydate = datestring.split()
            
    # At this point, the date is in the format 2011-10-15
    hours = 0
    if (len(mydate) > 1):
        mytime = (mydate[1]).split(':')
        for i in range(len(mytime)):
            hours += float(mytime[i])/(60.0**i)
    if (not casaAvailable or not use_metool):
        if (mydate[0][4] in ['-','/']): # '2011/Oct/01'
            delimiter = mydate[0][4]
            year, month, day = mydate[0].split(delimiter)
        elif (mydate[0][2] in ['-','/']): # '01/Oct/2011'
            delimiter = mydate[0][2]
            day, month, year = mydate[0].split(delimiter)
        elif (mydate[0][1] in ['-','/']):  # '1/Oct/2011'
            delimiter = mydate[0][1]
            day, month, year = mydate[0].split(delimiter)
        mjd = ymdhmsToMJD(int(year),int(month),int(day),hours)
    else:
        me = createCasaTool(metool)
        me_epoch = me.epoch('utc',mydate[0])
        if (me_epoch != {}):
            mjd = me_epoch['m0']['value'] + hours/24.
        else:
            print("Bad date = ", mydate[0])
            return(0)
    mjdsec = 86400*mjd
    if (verbose):
        print("MJD= %.5f, MJDseconds= %.3f, JD= %.5f" % (mjd, mjdsec, mjdToJD(mjd)))
    if (len(datestring2) > 0):
        mydate2 = datestring2.split()
        if (len(mydate2) < 1):
            return(mjdsec)
        if (len(mydate2) < 2):
            mydate2 = datestring2.split('-')
        hours = 0
        if (len(mydate2) > 1):
            mytime2 = (mydate2[1]).split(':')
            if (len(mytime2) > 1):
              for i in range(len(mytime2)):
                hours += float(mytime2[i])/(60.0**i)
#            print "hours = %f" % (hours)
        mjd = me.epoch('utc',mydate2[0])['m0']['value'] + hours/24.
        mjdsec2 = mjd*86400
        plotrange = [mjdsec, mjdsec2, plotrangeY[0], plotrangeY[1]]
        if verbose:
            print("plotrange = %s" % (plotrange))
        mystr = '%.0f~%.0f' % (mjdsec, mjdsec2)
        if verbose:
            print("plotrange = '%s'" % (mystr))
        return([mjdsec,mjdsec2], plotrange)
    else:
        return(mjdsec)
        
def plotPointingResults(vis='', figfile=False, source='',buildpdf=False,
                        gs='gs', convert='convert',
                        verbose=False, labels=False, pdftk='pdftk',debug=False,
                        interactive=True, nsigma=2, thresholdArcsec=10.0,
                        fractionOfScansBad=0.60, doplot=True, listAntennas=[], 
                        mymsmd=None):
    """
    This function will plot the pointing results for an ms, assuming that the
    ASDM_CALPOINTING table was filled, e.g. by importasdm(asis='*').  See also
    plotPointingResultsFromASDM(). The default behavior is to plot all sources
    that were pointed on.  In order to plot just one, then give the source
    name.  Setting labels=True will draw tiny antenna names at the points.
    If buildpdf does not work because gs or convert are not in the standard
    location, then you can specify the full paths with gs='' and convert=''.
    vis: the measurement set
    source: limit the results to a specified source
    buildpdf: compile all plots into a multi-page PDF
    gs, convert and pdftk: In case these programs are not found in your path, 
        the full path to the programs can be specified via these parameters.
        Note: The default program for concatenating PDFs is pdftk, but gs will be used if pdftk is not found. 
    interactive: set to False to avoid the need to press return after each scan is plotted.
    labels: set to True to draw the antenna names at each point, otherwise, it only draws the name if the offset is > nsigma times the standard deviation for that scan.
    nsigma: threshold for drawing the antenna name next to its point if labels=False
    fractionOfScansBad: the fraction of scans for which an antenna must exceed thresholdArcsec to be declared "suspect"
    doplot: setting this to False will produce no plots to the screen (or a file) but will return the suspect antenna list 
    listAntennas: list of antenna names for which to print the correction (in addition to plotting them all)
    -Todd Hunter
    """
    if (buildpdf and figfile==False):
        figfile=True
    fname = '%s/ASDM_CALPOINTING' % vis
    if (os.path.exists(fname)):
#        print "Trying to open table = %s" % fname
        mytb = createCasaTool(tbtool)
        mytb.open(fname)
    else:
        fname = './ASDM_CALPOINTING'
        if (os.path.exists(fname)):
            print("Looking for table in current directory")
            mytb.open(fname)
        else:
            print("No ASDM_CALPOINTING table found.")
            return
    colOffsetRelative = ARCSEC_PER_RAD*mytb.getcol('collOffsetRelative')
    calDataId = mytb.getcol('calDataId')
    antennaName = mytb.getcol('antennaName')
    colError = ARCSEC_PER_RAD*mytb.getcol('collError')
    pols = mytb.getcol('polarizationTypes')
    startValidTime = mytb.getcol('startValidTime')
    uniqueAntennaNames = np.unique(antennaName)
    if (len(startValidTime) == 0):
        print("ASDM_CALPOINTING table is empty.")
        return
    (mjd, utstring) = mjdSecondsToMJDandUT(startValidTime[0])
    print('time = %f = %f = %s' % (startValidTime[0], mjd, utstring))
    mytb.close()
    #
    fname = '%s/ASDM_CALDATA'%vis
    matches = None
    needToClose = False
    usingCalDataTable = False
    if (os.path.exists(fname)):
        mytb.open(fname)
        usingCalDataTable = True
    else: # look in current pwd
        fname = './ASDM_CALDATA'
        if (os.path.exists(fname)):
            mytb.open('./ASDM_CALDATA')
            usingCalDataTable = True
        else:
            print("Could not open ASDM_CALDATA table. Will use ms instead.")
            if mymsmd is None:
                needToClose = True
                mymsmd = msmdtool()
                mymsmd.open(vis)
            matches = getPointingScans(vis, mymsmd=mymsmd)
#            return
    if usingCalDataTable:
        calDataList = mytb.getcol('calDataId')
        calType = mytb.getcol('calType')
        startTimeObserved = mytb.getcol('startTimeObserved')
        endTimeObserved = mytb.getcol('endTimeObserved')
        #        matches = np.where(calDataList == calDataId[0])
        matches = np.where(calType == 'CAL_POINTING')[0]
        print("Found %d pointing calibrations, in ASDM_CALDATA rows: " % (len(matches)), matches)
    nscans = len(matches)
    mytb.close()
    if (doplot):
        plotfiles = []
        pointingPlots = []
    filelist = ''
    if (type(figfile)==str):
        if (figfile.find('//') >= 0):
            directories = figfile.split('/')
            directory = ''
            for d in range(len(directories)):
                directory += directories[d] + '/'
            if (os.path.exists(directory)==False):
                print("Making directory = ", directory)
                os.system("mkdir -p %s" % directory)
    if (casaVersion >= casaVersionWithMSMD):
        vm = 0
        if mymsmd is None:
            mymsmd = createCasaTool(msmdtool)
            mymsmd.open(vis)
    else:
        vm = ValueMapping(vis)
    antennaNames = mymsmd.antennanames(range(mymsmd.nantennas()))
    missingAntennas = []
    for antenna in antennaNames:
        if antenna not in uniqueAntennaNames:
            missingAntennas.append(missingAntennas)
    if len(missingAntennas) > 0:
        print("WARNING: There are %d antennas with missing solutions: %s" % (len(missingAntennas), missingAntennas))
    previousSeconds = 0
    antenna = 0
    uniquePointingScans = []
    outlier = {}
    radialErrors = {}
    for i in range(nscans):
        if usingCalDataTable:
            index = matches[i]
        else:
            index = i
        if (debug and usingCalDataTable):
            print("startTimeObserved[%d]=%f" % (index, startTimeObserved[index]))
            print("  endTimeObserved[%d]=%f" % (index, endTimeObserved[index]))
        if usingCalDataTable:
            # determine sourcename
            mytime = 0.5*(startTimeObserved[index]+endTimeObserved[index])
            if (debug):
                print("Time = ", mytime)
            if (casaVersion >= casaVersionWithMSMD):
                scan = mymsmd.scansfortimes(mytime)[0]
            else:
                scan = getScansForTime(vm.scansForTimes, mytime)
        else:
            scan = matches[i]
            scanMeanTime = np.mean(mymsmd.timesforscan(scan))
#            for j,sVT in enumerate(startValidTime):
#                print "%3d) Offset from startValidTime to scan %d meantime = %f sec" % (j,scan,sVT-scanMeanTime)
        uniquePointingScans.append(scan)
        radialErrors[uniquePointingScans[i]] = {}
        outlier[uniquePointingScans[i]] = {}
        if (debug):
            print("Scan = ", scan)
        [avgazim,avgelev] = listazel(vis,scan,antenna,vm,mymsmd=mymsmd)
        if (casaVersion >= casaVersionWithMSMD):
            pointingSource = mymsmd.namesforfields(mymsmd.fieldsforscan(scan)[0])[0]
        else:
            pointingSource = vm.getFieldsForScan(scan)[0].split(';')[0]
        if (source != ''):
            if (source != pointingSource):
                print("Source does not match the request, skipping")
                continue
        if (doplot):
            pb.clf()
            adesc = pb.subplot(111)
            c = ['blue','red']
        seconds = []
        for p in range(len(pols)):
            seconds.append(np.max(np.max(np.abs(colOffsetRelative[p][0])+colError[p][0])))
            seconds.append(np.max(np.max(np.abs(colOffsetRelative[p][1])+colError[p][1])))
        seconds = np.max(seconds)*1.25
        if (seconds <= previousSeconds):
            seconds = previousSeconds
        previousSeconds = seconds
        if (doplot):
            pb.xlim([-seconds,seconds])
            pb.ylim([-seconds,seconds])
#            pb.hold(True) # not needed
        if usingCalDataTable:
            thisscan0 = np.where(startTimeObserved[index] < startValidTime)[0]
            thisscan1 = np.where(endTimeObserved[index] > startValidTime)[0]
            rowsInThisScan = np.intersect1d(thisscan0,thisscan1)
        else:
            rowsInThisScan = np.where(np.abs(scanMeanTime - startValidTime) < 30)[0]
        mystd = []
        for p in range(len(pols)):
            mystd.append([np.std(colOffsetRelative[p][0]), np.std(colOffsetRelative[p][1])])
        for a in rowsInThisScan:
            antenna = np.where(uniqueAntennaNames==antennaName[a])[0][0]  # this is an integer
            for p in range(len(pols)):
#                if (verbose or uniqueAntennaNames[a] in listAntennas):
                if (verbose or antennaName[antenna] in listAntennas):
                    print("%s: %.2f, %.2f = %.2e, %.2e" % (antennaName[antenna],
                       colOffsetRelative[p][0][a], colOffsetRelative[p][1][a], 
                       colOffsetRelative[p][0][a]/ARCSEC_PER_RAD, colOffsetRelative[p][1][a]/ARCSEC_PER_RAD))
                if (doplot):
#                    pb.plot(colOffsetRelative[p][0][a], colOffsetRelative[p][1][a], 'o',
#                        markerfacecolor=overlayColors[antenna], markersize=6, color=overlayColors[antenna],
#                        markeredgecolor=overlayColors[antenna])
                    pb.errorbar(colOffsetRelative[p][0][a], colOffsetRelative[p][1][a], fmt='o',
                            yerr=colError[p][1][a], xerr=colError[p][0][a],
                            color=overlayColors[antenna], markersize=5,
                            markerfacecolor=overlayColors[antenna], markeredgecolor=overlayColors[antenna])
                    if (labels or abs(colOffsetRelative[p][0][a])>nsigma*mystd[p][0] or
                        abs(colOffsetRelative[p][1][a])>nsigma*mystd[p][1]):
                        pb.text(colOffsetRelative[p][0][a], colOffsetRelative[p][1][a],
                                antennaName[a], color='k', size=8)
                totalOffset = (colOffsetRelative[p][0][a]**2 + colOffsetRelative[p][1][a]**2)**0.5
                if (totalOffset > thresholdArcsec):
                    outlier[uniquePointingScans[i]][antennaName[a]] = [True, totalOffset]
                else:
                    outlier[uniquePointingScans[i]][antennaName[a]] = [False]
                radialErrors[uniquePointingScans[i]][antennaName[a]] = totalOffset
                
        if (doplot):
            # Draw spec
            cir = pb.Circle((0, 0), radius=2, facecolor='none', edgecolor='k', linestyle='dotted')
            pb.gca().add_patch(cir)
            pb.title('Relative collimation offsets at %s - %s scan %d' % (utstring,pointingSource,scan),fontsize=12)
            pb.axvline(0,color='k',linestyle='--')
            pb.axhline(0,color='k',linestyle='--')
            yFormatter = ScalarFormatter(useOffset=False)
            adesc.yaxis.set_major_formatter(yFormatter)
            adesc.xaxis.set_major_formatter(yFormatter)
            if (False):
                minorTickSpacing = 1.0
                xminorLocator = MultipleLocator(minorTickSpacing)
                yminorLocator = MultipleLocator(minorTickSpacing)
                adesc.xaxis.set_minor_locator(xminorLocator)
                adesc.yaxis.set_minor_locator(yminorLocator)
                majorTickSpacing = 5.0
                majorLocator = MultipleLocator(majorTickSpacing)
                adesc.xaxis.set_major_locator(majorLocator)
                adesc.yaxis.set_major_locator(majorLocator)
            adesc.xaxis.grid(True,which='major')
            adesc.yaxis.grid(True,which='major')
            pb.xlabel('Cross-elevation offset (arcsec)')
            pb.ylabel('Elevation offset (arcsec)')
            # now draw the legend for plotPointingResults
            myxlim = pb.xlim()
            myylim = pb.ylim()
            myxrange = myxlim[1]-myxlim[0]
            myyrange = myylim[1]-myylim[0]
            x0 = myxlim[0] + 0.05*myxrange
            y0 = myylim[1]-np.abs(myyrange*0.05)
            mystep = 0.025-0.0001*(len(uniqueAntennaNames)) # 0.020 for 50 antennas, 0.0225 for 25 antennas, etc.
            for a in range(len(antennaName[rowsInThisScan])):
                pb.text(myxlim[1]+0.02*myxrange,
                        myylim[1]-(a+1)*myyrange*mystep,
                        antennaName[rowsInThisScan][a],
                        color=overlayColors[list(uniqueAntennaNames).index(antennaName[rowsInThisScan][a])],
                        fontsize=10)
            pb.text(x0,y0,'azim=%+.0f, elev=%.0f'%(avgazim,avgelev),color='k', fontsize=14)
            pb.text(myxlim[0]+0.02*myxrange, myylim[1]+0.05*myyrange,vis,fontsize=12,color='k')
            pb.axis('scaled')
            pb.axis([-seconds,seconds,-seconds,seconds])
            if (figfile==True):
                myfigfile = vis+'.pointing.scan%02d.png' % (scan)
                pointingPlots.append(myfigfile)
                pb.savefig(myfigfile,density=144)
                plotfiles.append(myfigfile)
                print("Figure left in %s" % myfigfile)
            elif (figfile != False):
                myfigfile=figfile
                pb.savefig(figfile,density=144)
                plotfiles.append(myfigfile)
                print("Figure left in %s" % myfigfile)
            else:
                print("To make a hardcopy, re-run with figfile=True or figfile='my.png'")
            pb.draw()
            if (buildpdf == True):
                cmd = '%s -density 144 %s %s.pdf'%(convert,myfigfile,myfigfile)
                print("Running command = ", cmd)
                mystatus = os.system(cmd)
                if (mystatus == 0):
                    print("plotfiles[i] = ", plotfiles[i])
                    filelist += plotfiles[i] + '.pdf '
                else:
                    print("ImageMagick's convert command is missing, no PDF created.")
                    buildpdf = False
        if (i < nscans-1 and interactive):
            mystring = raw_input("Press return for next scan (or 'q' to quit): ")
            if (mystring.lower().find('q') >= 0):
                return
    if (casaVersion >= casaVersionWithMSMD):
        if needToClose:
            mymsmd.close()
    # end 'for' loop over scans
    badAntennas = {}
    for antenna in uniqueAntennaNames:
        goodscans = 0.0
        radialErrorsThisAntenna = {}
        for scan in uniquePointingScans:
            # datasets with only 2 antennas have 1 antenna solution per scan
            if (antenna in radialErrors[scan]):
                radialErrorsThisAntenna[scan] = radialErrors[scan][antenna]
                if (outlier[scan][antenna][0]==False):
                    goodscans += 1.0
                
        badscans = len(uniquePointingScans)-goodscans
        if (badscans/len(uniquePointingScans) >= fractionOfScansBad):
            print("Antenna %s was an outlier (>%.1f arcsec) for %d/%d scans" % (antenna, thresholdArcsec, badscans, len(uniquePointingScans)))
            badAntennas[antenna] = radialErrorsThisAntenna
                
    if (buildpdf and doplot):
        pdfname = vis+'.pointing.pdf'
        if (nscans > 1):
            mystatus = concatenatePDFs(filelist, pdfname, pdftk=pdftk, gs=gs)
        else:
            cmd = 'cp %s %s' % (filelist,pdfname)
            print("Running command = %s" % (cmd))
            mystatus = os.system(cmd)
        if (mystatus == 0):
            print("PDF left in %s" % (pdfname))
            os.system("rm -f %s" % filelist)
        else:
            print("Both pdftk and ghostscript are missing, no PDF created")
    return(pointingPlots, badAntennas)  
# end of plotPointingResults

def getFieldsFromASDMs(asdms):
    """
    Prints a dictionary of the field IDs/names for each ASDM in a list of ASDMs.
    -Todd Hunter
    """
    if type(asdms) == str:
        if asdms.find('*') >= 0:
            asdms = glob.glob(asdms)
        else:
            asdms = asdms.split(',')
    for asdm in asdms:
        if asdm.find('.ms') < 0:
            print('%s: %s' % (asdm,str(getFieldsFromASDM(asdm)[0])))

def getFieldsFromASDM(asdm):
    """
    Returns a dictionary with field IDs as the key, and names as the value,
    and another dictionary with the reverse.
    -Todd Hunter
    """
    if (os.path.exists(asdm) == False):
        print("Could not find file = ", asdm)
        return
    xmlscans = minidom.parse(asdm+'/Field.xml')
    mydict = {}
    mydict2 = {}
    rowlist = xmlscans.getElementsByTagName("row")
    fid = 0
    for rownode in rowlist:
        rowpwv = rownode.getElementsByTagName("fieldId")
        fieldid = int(rowpwv[0].childNodes[0].nodeValue.split('_')[1])
        rowpwv = rownode.getElementsByTagName("fieldName")
        fieldname = str(rowpwv[0].childNodes[0].nodeValue)
        mydict[fieldid] = fieldname
        mydict2[fieldname] = fieldid
    return(mydict, mydict2)

def getScanNumbersFromASDM(asdm, intent='CALIBRATE_ATMOSPHERE'):
    """
    Returns a list of scan numbers in an ASDM that have the
    specified intent, and a list of the number of subscans in each of these scans.
    -Todd Hunter
    """
    if (not os.path.exists(asdm)):
        print("Could not find ASDM")
        return
    mydict = readscans(asdm)[0]
    scans = []
    subscans = []
    for scan in list(mydict.keys()):
        if (intent == '' or mydict[scan]['intent'].find(intent)>=0):
            scans.append(scan)
            subscans.append(mydict[scan]['nsubs'])
    return(scans, subscans)

def getCalibratorFromASDMs(asdm, intent='PHASE'):
    """
    Returns a dictionary of observed calibrator names for the specified intent 
    from a list of ASDMs.
    asdm: a single ASDM or a comma-delimited list, or wildcard string
    intent:  e.g., set to 'CAL' for all calibration intents
    -Todd Hunter
    """
    if type(asdm) == str:
        if asdm.find('*') >= 0:
            asdms = sorted(glob.glob(asdm))
        else:
            asdms = asdm.split(',')
    else:
        asdms = asdm
    cals = {}
    for asdm in asdms:
        cals[asdm] = getCalibratorFromASDM(asdm, intent)
    return cals

def getCalibratorFromASDM(asdm, intent='PHASE'):
    """
    Gets a list of calibrator names for the specified intent from an ASDM.
    asdm: a single ASDM 
    intent:  e.g., set to 'CAL' for all calibration intents
    -Todd Hunter
    """
    dicts = readscans(asdm)
    if (len(dicts[0]) == 0): return
    myscans = dicts[0]
    mysources = dicts[1]
    calibrators = []
    for key in list(myscans.keys()):
        mys = myscans[key]
        if mys['intent'].find(intent) >= 0:
            calibrators.append(mys['source'])
    return list(np.unique(calibrators))

def gaincurve(c,za):
    """
    Evaluates a VLA antenna gain curve
    Todd Hunter
    """
    return(c[0] + c[1]*za + c[2]*za**2 + c[3]*za**3)

def utdatestring(mjdsec):
    """
    Converts MJD seconds to a date string
    Todd Hunter
    """
    (mjd, dateTimeString) = mjdSecondsToMJDandUT(mjdsec)
    tokens = dateTimeString.split()
    return(tokens[0])

def getgain(myAntenna, myBand, myElev=60, tableLocation='', date='', title=None):
    """
    This function will extract the gain curve for a specified JVLA antenna and
    receiver band for the current date and produce a plot vs. elevation.
    It will also list the gain for a specified elevation.
    Elevations are in degrees,  antennas are 1..28,29=PieTown,30=allVLA and 0=mean
    Default date is today, but can be specified as follows:
      Either of these formats is valid: 2011/10/15
                                        2011/10/15 05:00:00
                                        2011/10/15-05:00:00
    Todd Hunter
    """
    defaultTableLocation = os.getenv("CASAPATH").split(' ')[0]+'/data/nrao/VLA/GainCurves'
    if (type(myAntenna) == str):
        loc = myAntenna.find('ea')
        if (loc >= 0):
            myAntenna = int(myAntenna[loc+2:])
        else:
            myAntenna = int(str)
    if (myAntenna < 0 or myAntenna > 30):
        print("Invalid antenna number, must be 1..28,29=PieTown,30=allVLA or 0=mean")
        return
    print("elev=%f" % (myElev))
    print("band=%s" % (myBand))
    if (myAntenna == 30):
        showall = True
        myAntenna = 0
        antennaList = range(1,29)
        antennaList.append(0)
    else:
        antennaList = [myAntenna]
        showall = False
    print("antenna=%d" % (myAntenna))
    if (date == ''):
        currentMJDsec = 86400*unixTimeToMJD(timeUtilities.time())
        print("current time = %f" % (currentMJDsec))
    else:
        currentMJDsec = dateStringToMJDSec(datestring=date)

    if (tableLocation == ''):
        tableLocation = defaultTableLocation
    mytb = createCasaTool(tbtool)
    try:
        mytb.open(tablename=tableLocation)
        print("Using table = ", tableLocation)
    except:
        print("Could not open table = ", tableLocation)
        return
    antennaNumbers = mytb.getcol('ANTENNA')
    bands = mytb.getcol('BANDNAME')
    begintime = mytb.getcol('BTIME')
    endtime = mytb.getcol('ETIME')
    gains = mytb.getcol('GAIN')
    index = np.where(bands == myBand)
    try:
        print("There are %d rows with band %s" % (len(index[0]), myBand))
    except:
        print("Found no entries for band %s" % (myBand))

    pb.clf()
    desc = pb.subplot(111)
    for myAntenna in antennaList:
        yindex = np.where(antennaNumbers[index[0]] == str(myAntenna))
        rows = index[0][yindex[0]]
        list = ''
        for i in rows:
            list += "%d " % (i)
        print("There are %d rows with antenna %d and band %s: %s" % (len(yindex[0]), myAntenna,
                                                                  myBand, list))
        print("Corresponding to dates: ")
        for i in rows:
            print("  %d: %s" % (i, utdatestring(begintime[i])))
        tindex = np.where(begintime[rows] < currentMJDsec)
        tindex = np.where(endtime[rows[tindex[0]]] > currentMJDsec)
        row = rows[tindex[0]]
        if (len(row) < 1):
            closestInterval = 1e10
            for i in rows:
                interval0 = abs(begintime[i]-currentMJDsec)
                interval1 = abs(endtime[i]-currentMJDsec)
                interval = interval1
                if (interval0 < interval1):
                    interval = interval0
                if (interval < closestInterval and i != 895):
                    closestInterval = interval
                    row = i
        else:
            if (np.size(row) > 1):
                row = row[0]
            closestInterval = abs(begintime[row]-currentMJDsec)
        print("Using closest prior measurement in time: row %d (%.0f days away)" %(row,closestInterval/86400.))

        pol = 0
        c = []
        for i in range(4):
            c.append(gains[i][pol][row])
        myZA = 90-myElev
        myGain = gaincurve(c,myZA)
        print("gain=%f at ZA=%.1f or EL=%.1f" % (myGain,myZA,myElev))
        ZA = np.arange(90)
        gain = []
        for i in ZA:
            gain.append(gaincurve(c,i))
        elev = 90-ZA
        if (showall and myAntenna==0):
            pb.plot(elev,gain,'k',lw=4)
        else:
            pb.plot(elev,gain)
    # end 'for' loop over antennaList
     
    pb.xlabel("Elevation (deg)",size=18)
    pb.ylabel("Relative gain", size=18)
    desc.xaxis.grid(True, which='major')
    desc.yaxis.grid(True, which='major')
    datestring = utdatestring(currentMJDsec)
    datestring = utdatestring(begintime[row])
    outname = 'gaincurve.%s.%d.png'%(myBand,myAntenna)
    if (showall):
        outname = 'gaincurves.%s.png'%(myBand)
        titleString = "JVLA gain curves for %s band (%s)"%(myBand,datestring)
    elif (myAntenna == 0):
        titleString = "Mean JVLA gain curve for %s band (%s)"%(myBand,datestring)
    elif (myAntenna == 29):
        titleString = "JVLA gain curve for Pie Town at %s band (%s)"%(myBand,datestring)
    else:
        titleString = "JVLA gain curve for antenna %d at %s band (%s)"%(myAntenna,myBand,datestring)
    if (title == None):
        pb.title(titleString, size=18)
    else:
        pb.title(title, size=18)
    if (os.access('.',os.W_OK)):
        pb.savefig(outname,format='png',dpi=108)
        eps = outname.replace('.png','.eps')
        pb.savefig(eps,format='eps',dpi=1200)
    else:
        print("Could not write plot to current directory.  Will write to /tmp")
        outname = '/tmp/' + outname
        pb.savefig(outname,format='png',dpi=108)      
        eps = outname.replace('.png','.eps')
        pb.savefig(eps,format='eps',dpi=1200)
    pb.draw()
    print("Plot stored in %s and %s" % (outname,eps))
    mytb.close()
    return(myGain)

def readscans(asdm):
    """
    This function was ported from a version originally written by Steve Myers
    for EVLA.  It works for both ALMA and EVLA data.  It returns a dictionary
    containing: startTime, endTime, timerange, sourcename, intent, nsubs, 
    duration.
    Todd Hunter
    """
    # This function lives in a separate .py file.
    return(rs.readscans(asdm))

def getTargetsForIntent(vis, intent='OBSERVE_TARGET', mymsmd=''):
    """
    Gets a list of target names observed with the specified intent.
    intent: wildcards are added on both sides
    """
    if mymsmd == '':
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        needToClose = True
    else:
        needToClose = False
    found = False
    for i in mymsmd.intents():
        if i.find(intent) >= 0:
            found = True
    if found:
        intent = '*' + intent + '*'
        targetNames = list(mymsmd.fieldsforintent(intent, asnames=True))
    else:
        targetNames = []
    if needToClose: mymsmd.close()
    return targetNames

def getTargetsForIntentFromASDM(asdm, intent='OBSERVE_TARGET'):
    """
    Gets a list of target names observed with the specified intent
    from an ASDM (minimum match).  Wildcards (*) are removed.
    intent:  e.g. 'PHASE'
    Returns: e.g. ['3c345']
    -Todd Hunter
    """
    if (not os.path.exists(asdm)):
        print("Could not find ASDM")
        return
    mydict = rs.readscans(asdm)[0]
    targets = []
    intent = intent.replace('*','')
    for scan in list(mydict.keys()):
        if (mydict[scan]['intent'].find(intent) >= 0):
            target = mydict[scan]['source']
            if target not in targets:
                targets.append(target)
    return targets

def getStartTimesForFieldFromASDM(asdm, field):
    """
    Gets a list of start times (in MJD) for the specified fieldname
    from an ASDM.
    -Todd Hunter
    """
    if (not os.path.exists(asdm)):
        print("Could not find ASDM")
        return
    mydict = rs.readscans(asdm)[0]
    mjds = []
    for scan in list(mydict.keys()):
        if (mydict[scan]['source'].find(field) >= 0):
            mjd = mydict[scan]['startmjd']
            mjds.append(mjd)
    return mjds

def getElevationStatsForIntentFromASDM(asdm, intent='OBSERVE_TARGET'):
    """
    Finds all sources observed with a given intent in an ASDM, then
    finds the statistics for the elevation that they were observed at.
    Returns: [minimum, median, max]
    -Todd Hunter
    """
    if (not os.path.exists(asdm)):
        print("Could not find ASDM")
        return
    targets = getTargetsForIntentFromASDM(asdm, intent)
    if (len(targets) == 0):
        print("No targets found with intent = %s in %s" % (intent,asdm))
        return
    observatory = getObservatoryNameFromASDM(asdm)
    elev = []
    for target in targets:
        radec = getRADecForFieldFromASDM(asdm, target, sexagesimal=True)
        mjds = np.array(getStartTimesForFieldFromASDM(asdm, target))
        for mjd in mjds:
            az,el= computeAzElFromRADecMJD(radec, mjd, observatory,degrees=True)
            elev.append(el)
    return [np.min(elev), np.median(elev), np.max(elev)]

def listscans(asdm, listfile='', searchString=''):
    """
    This function was ported from a version originally written by Steve Myers
    for EVLA.  It works for both ALMA and EVLA data.  It prints the summary
    of each scan in the ASDM and the total time on each source.
    asdm: string or comma-delimited list, which can contain a wildcard character
    searchString: only relevant if a list of ASDMs is provided
    Returns:
    * a dictionary or a list of dictionaries if multiple ASDMs are specified
    Useful usage:  a = au.listscans('uid*','FLUX')
    Todd Hunter
    """
    result = []
    listfiles = []
    if (asdm.find('*') >= 0):
        if listfile == '':
            listfile = 'summary'
        asdmlist = glob.glob(asdm)
        for i,asdm in enumerate(asdmlist):
            listfile = listfile+'%02d'%i
            listfiles.append(listfile)
            result.append(rs.listscans(rs.readscans(asdm), listfile, asdm))
        summary = concatenateFiles(listfiles)
        if (searchString != ''):
            print(grep(summary,searchString)[0])
    elif (asdm.find(',')>0):
        if listfile == '':
            listfile = 'summary'
        asdmlist = asdm.split(',')
        for i,asdm in enumerate(asdmlist):
            listfile = listfile+'%02d'%i
            listfiles.append(listfile)
            result.append(rs.listscans(rs.readscans(asdm), listfile, asdm))
        summary = concatenateFiles(listfiles)
        if (searchString != ''):
            print(grep(summary,searchString)[0])
    else:
        result = rs.listscans(rs.readscans(asdm), listfile, asdm)
    return(result)

def plotPointingResultsFromASDMForDatasets(asdmlist, doplot=False,
                                           figfile=False, interactive=False, buildpdf=False,
                                           figfiledir='', checkForMissingEntriesOnly=False):
    """
    Runs plotPointingResultsFromASDM on a list of ASDMs.
    Writes a text file with the list of suspect antennas.
    asdmlist: can be a string containing a wildcard character, which will use all matching files in pwd
         or it can be the name of a file which contains a list of files to use
          or it can be a list of asdms, i.e. ['uid1','uid2']
    doplot: create plots for each scan
    buildpdf: create plots for each scan and assemble them into a PDF
    figfile: specify the file name of the plots
         True:   the filenames will be asdm.pointing.scan%02d.png
         String: the filenames will be String.scan%02d, String.scan%02d, ...
    -Todd Hunter
    """
    if (not figfile and figfiledir != ''):
        figfile = True
    if (buildpdf or figfile):
        doplot = True
    if (type(asdmlist) == str):
        if (asdmlist.find('*') >= 0):
            outfile = 'plotPointingResultsForDatasets.txt'
            asdmlist = glob.glob(asdmlist)
        else:
            outfile = asdmlist+'.plotPointingResultsForDatasets.txt'
            asdmlist = getListOfFilesFromFile(asdmlist, appendms=False)
    else:
        outfile = 'plotPointingResultsForDatasets.txt'
    if (not os.access(".",os.W_OK)):
        outfile = '/tmp/' + outfile
    f = open(outfile,'w')
    pngs = []
    for asdm in asdmlist:
        if (asdm.find('.ms') > 0): continue
        if (not os.path.isdir(asdm)): 
            print("%s does not appear to be an ASDM, skipping." % (asdm))
            continue
        print("Working on %s" % (asdm))
        results = plotPointingResultsFromASDM(asdm,figfile=figfile,doplot=doplot,
                                              interactive=interactive,buildpdf=buildpdf,
                                              figfiledir=figfiledir)
        if (results is not None): 
            pointingPlots, badAntennas = results 
            f.write('%s   %s\nsuspect antennas = %s\n\n' % (asdm,getObservationStartDateFromASDM(asdm),str(badAntennas)))
            pngs += pointingPlots
    f.close()
    if (len(pngs) > 0):
        buildPdfFromPngs(pngs, pdfname=outfile+'.pdf', gs='/usr/bin/gs')
    print("Results left in %s" % (outfile))

def readPointingModelFromASDM(asdm, antenna=None, showBothPolarizations=False,
                              returnDictionary=False, verbose=True, showGlobal=True,
                              showAuxiliary=False):
    """
    Reads and lists the pointing model coefficients for one (or all) 
    antennas in an ASDM.
    asdm: ASDM
    antenna: antenna name string or antenna ID (as integer or string)
    showGlobal: show the local pointing corrections
    showAuxiliary: show the full 19-term model
    returnDictionary: if True, then return dictionary keyed by antenna name,
         then polarization ('X', 'Y', 'id'), then type ('GLOBAL', 'AUXILIARY')
         where 'id' is the antenna integer ID (for convenience)
    -Todd Hunter
    """
    if (os.path.exists(asdm) == False):
        print("readPointingModelFromASDM(): Could not find ASDM = ", asdm)
        return(None)
    if (os.path.exists(asdm+'/PointingModel.xml') == False):
        print("readPointingModelFromASDM(): Could not find PointingModel.xml.", asdm)
        return(None)
    antennaNames = readAntennasFromASDM(asdm, verbose=False)
    if (antenna is not None and antenna != ''):
        if (type(antenna) == str):
            if antenna.isdigit():
                antenna = int(antenna)
        if (type(antenna) == str):
            if (antenna not in antennaNames):
                print("Antenna %s is not in the ASDM." % (antenna))
                return
        else:
            if (antenna < 0 or antenna >= len(antennaNames)):
                print("Antenna %d is not in the ASDM." % (antenna))
                return
            antenna = antennaNames[antenna]
            
    xmlscans = minidom.parse(asdm+'/PointingModel.xml')
    rowlist = xmlscans.getElementsByTagName("row")
    coeff = {}
    for rownode in rowlist:
        rowAntennaId = rownode.getElementsByTagName("antennaId")
        antennaId = str(rowAntennaId[0].childNodes[0].nodeValue)
        rowCoeffName = rownode.getElementsByTagName("coeffName")
        coeffName = str(rowCoeffName[0].childNodes[0].nodeValue)
        coeffNames = ', '.join(coeffName.split()[2:]).replace('"','')
        rowAssocNature = rownode.getElementsByTagName("assocNature")
        assocNature = str(rowAssocNature[0].childNodes[0].nodeValue)
        rowPolarizationType = rownode.getElementsByTagName("polarizationType")
        polarizationType = str(rowPolarizationType[0].childNodes[0].nodeValue)
        rowCoeffVal = rownode.getElementsByTagName("coeffVal")
        coeffVal = str(rowCoeffVal[0].childNodes[0].nodeValue)
        tokens = coeffVal.split()
        antennaId = int(antennaId.split('_')[1])
        antennaName = antennaNames[antennaId]
        if (antennaName not in list(coeff.keys())):
            coeff[antennaName] = {}
        if (polarizationType not in list(coeff[antennaName].keys())):
            coeff[antennaName][polarizationType] = {}
        if (assocNature not in list(coeff[antennaName][polarizationType].keys())):
            coeff[antennaName][polarizationType][assocNature] = []
        coeff[antennaName][polarizationType][assocNature].append([float(token) for token in tokens[2:]])
        coeff[antennaName]['id'] = antennaId
    antennaNames = list(coeff.keys())
    print(' '*21 + str(coeffNames))
    for antennaName in antennaNames:
        if (antenna == None or antenna == '' or antenna == antennaName):
            for polarizationType in list(coeff[antennaName].keys()):
                if polarizationType == 'id': continue
                if (polarizationType == list(coeff[antennaName].keys())[0] or 
                    showBothPolarizations):
                    polString = ' pol'+polarizationType
                    if (not showBothPolarizations): polString = ''
                    for nature in list(coeff[antennaName][polarizationType].keys()):
                        if (verbose and ((nature != 'GLOBAL' or showGlobal) or (nature != 'AUXILIARY' or showAuxiliary))):
                            for i in range(len(coeff[antennaName][polarizationType][nature])):
                                print("Ant%02d: %s%s: %s: %s" % (coeff[antennaName]['id'], antennaName, polString, nature, str(coeff[antennaName][polarizationType][nature][i])))
    if (returnDictionary):                        
        return(coeff)
        
def plotPointingResultsFromASDM(asdm='', figfile=False, buildpdf=False,
                                gs='gs', convert='convert', labels=False,
                                Xrange=[0,0],yrange=[0,0], pdftk='pdftk',
                                interactive=True, thresholdArcsec=10.0,
                                figfiledir='', nsigma=2, 
                                fractionOfScansBad=0.60, doplot=True,
                                antenna=None, listsigma=False,
                                scienceSpws=None, figsize=[9,9], montage=False,
                                tile='', geometry='auto+10', listAntennas=False, checkForMissingEntriesOnly=False):
    """
    This function will plot the pointing results for an ASDM. To use an ms
    instead, see plotPointingResults(). The default behavior is to plot all
    sources that were pointed on.  In order to plot just one, then give the
    source name.
    Setting labels=True will draw tiny antenna names at the points.
    If buildpdf does not work because gs or convert are not in the standard
    location, then you can specify the full paths with gs='' and convert=''.
    figfile: can be False, True, or a string.
         False:  no plot produced
         True:   the filenames will be asdm.pointing.scan%02d.png
         String: the filenames will be String.scan%02d, String.scan%02d, ...
    figfiledir:  useful for figfile=True option, specifies the directory to
                 (create and) write 
    antenna: if name is specified, then plot corrections vs. time for it only
    listsigma: if True, then print ratio of offset/uncertainty
    thresholdArcsec: criterion for bad pointing=min(thresholdArcsec,halfPrimaryBeam)
                 (if a list of scienceSpws is specified)
    scienceSpws: comma-delimited string, or single integer;  if specified, then
         determine halfPrimaryBeam and use that for the threshold
    montage: if True, then place figures into single page
    tile, geometry: passed to montagePngs
    listAntennas: list of antenna names for which to print the correction (in addition to plotting them all)
    Returns: a list of figure file names produced, followed by a list of
        suspect antenna names (those that have errors larger than the
        threshold on all scans).
    For further help and examples, see https://safe.nrao.edu/wiki/bin/view/ALMA/PlotPointingResultsFromASDM
    Todd Hunter
    """
    if (buildpdf and figfile==False):
        figfile = True
    if montage:
        figfile = True
    asdm = asdm.rstrip('/')
    asdmNoPath = os.path.basename(asdm)
    source = ''
    result = readCalPointing(asdm)
    thresholdArcsec = float(thresholdArcsec) 
    if (result is None):
        return
    [colOffsetRelative,antennaName,startValidTime,pols,calDataID,azim,elev,colError,frequency] = result
    antennaName = np.array(antennaName)
    diameter = almaAntennaDiameter(antennaName[0])
    colOffsetRelative = np.array(colOffsetRelative)
    colError = np.array(colError)
    startValidTime = np.array(startValidTime)
    
    # Need to match calDataId to scan numbers
        
    scandict = rs.readscans(asdm)[0]
    caldict = readCalDataFromASDM(asdm)  # this contains translation from calDataId to scan number
    if (caldict == None):
        return
    uniqueScans = 1+np.array(range(len(scandict)))  # in this list, scans appear only once
    scans = []  # in this list, each scan will appear N times, where N=number of antennas
    for i in calDataID:
      for c in range(len(caldict)):
        if (int(caldict[c]['calDataId'].split('_')[1]) == i):
            scans.append(caldict[c]['scan'])

    pointingScans = []
    for i in range(len(caldict)):
        if (int(caldict[i]['calDataId'].split('_')[1]) in calDataID):
            pointingScans.append(caldict[i]['scan'])
    uniquePointingScans = np.unique(pointingScans)
    
    print("pointing scans = ", pointingScans)
    pointingSources = ''
    for i in uniquePointingScans:
        pointingSources += scandict[i]['source'].split(';')[0]
        if (i != uniquePointingScans[-1]):
            pointingSources += ', '
    print("pointing sources = ",pointingSources)
    nPointingScans = len(uniquePointingScans)
    if (len(startValidTime) == 0):
        print("CalPointing.xml table is empty.")
        return
    
    plotfiles = []
    filelist = ''
    if (figfiledir != ''):
        if (os.path.exists(figfiledir) == False):
            print("Making directory: %s" % (figfiledir))
            os.makedirs(figfiledir)
        if (figfiledir[-1] != '/'): figfiledir += '/'
    if (type(figfile)==str):
        if (figfile.find('/')>=0 and figfiledir == ''):
            directories = figfile.split('/')
            directory = ''
            for d in range(len(directories)):
                directory += directories[d] + '/'
            if (os.path.exists(directory)==False):
                print("Making directory = ", directory)
                os.system("mkdir -p %s" % directory)

    previousSeconds = 0
    uniqueAntennaNames = np.unique(antennaName)
    allAntennas = readAntennasFromASDM(asdm)
    missingAntennas = []
    for myantenna in allAntennas:
        if myantenna not in uniqueAntennaNames:
            missingAntennas.append(myantenna)
    if len(missingAntennas) > 0:
        print("WARNING: There are %d antennas with missing solutions: %s" % (len(missingAntennas), missingAntennas))
    if checkForMissingEntriesOnly:
        return missingAntennas
    pointingPlots = []
    outlier = {}
    radialErrors = {}
    meanScienceBeam = False
    halfbeamPointingScan = primaryBeamArcsec(frequency = frequency[scans[0]], showEquation=False, diameter=diameter)*0.5
    meanScienceFreq = getMeanFreqFromASDM(asdm, scienceOnly=True)['mean']['meanfreq'] * 1e-9
    halfbeamScience = primaryBeamArcsec(frequency=meanScienceFreq, showEquation=False, diameter=diameter)*0.5
    if (scienceSpws is None):
        halfbeam = halfbeamPointingScan
        if (halfbeam < thresholdArcsec):
            print("Reducing threshold from %.1f arcsec to half the primary beam of a %.0fm antenna, which is %.1f arcsec for pointing scan %d." % (thresholdArcsec,diameter,halfbeam,scans[0]))
            thresholdArcsec = halfbeam
    else:
        if (type(scienceSpws) == str):
            scienceSpws = scienceSpws.split(',')
        elif (type(scienceSpws) == int):
            scienceSpws = [scienceSpws]
        halfbeam = 0.5*primaryBeamArcsec(frequency = np.mean(getFrequenciesFromASDM(asdm,spws=scienceSpws)),
                                         showEquation=False, diameter=diameter)
        if (halfbeam < thresholdArcsec):
            print("Reducing threshold to half the primary beam of a %.0fm antenna, which is %.1f arcsec for mean science spw." % (diameter,halfbeam))
            thresholdArcsec = halfbeam
            meanScienceBeam = True
    if (antenna is not None and antenna.find('!') < 0):
        print("antenna = ", antenna)
        if (antenna not in uniqueAntennaNames):
            print("%s is not in the list of antennas: %s" % (antenna, uniqueAntennaNames))
            return
        rows = np.where(antennaName==antenna)[0]
        print("Found %d rows for %s = %s" % (len(rows), antenna, str(rows)))
        offsets = []
        times = []
        errors = []
        for row in rows:
            times.append(startValidTime[row])
            offsets.append(colOffsetRelative[row])
            errors.append(colError[row])
        pb.clf()
        pb.gcf().set_size_inches(figsize[0], figsize[1], forward=True)
        adesc = pb.subplot(111)
        # azimuth
        times = (times - times[0])/60.
        pol = 0
        offsets = np.transpose(offsets)
        errors = np.transpose(errors)
        p1 = pb.errorbar(times, offsets[0][pol], yerr=errors[0][pol], fmt='o-',color='b')
#        pb.hold(True) # not needed
        p2 = pb.errorbar(times, offsets[1][pol],fmt='o-',color='r', yerr=errors[1][pol])
        pb.legend([p1,p2],["azimuth","elevation"], loc=1)
        pb.ylabel('Offset (arcsec)')
        pb.title(asdm+'  '+antenna+'   '+getObservationStartDateFromASDM(asdm)[0])
        pb.draw()
        adesc.xaxis.grid(True,which='major')
        adesc.yaxis.grid(True,which='major')
        pb.xlabel('Time (minutes)')
#        pb.ylabel('Elevation offset (arcsec)')
        if (figfile==True):
            myfigfile = figfiledir + asdmNoPath + '.pointing.%s.png' % (antenna)
            pb.savefig(myfigfile,density=144)
            print("Figure left in %s" % myfigfile)
        elif (figfile != False):
            myfigfile = figfiledir + figfile.replace('.png','') + '.%s.png' % (antenna)
            pb.savefig(myfigfile, density=144)
            print("Figure left in %s" % myfigfile)
        else:
            print("To make a hardcopy, re-run with figfile=True or figfile='my.png'")
            myfigfile = None
        return(myfigfile)
    else:
      if (antenna is not None):
          print("The !antenna option is not yet implemented")
      else:
          antenna = 0
      for i in range(nPointingScans):
        outlier[uniquePointingScans[i]] = {}
        radialErrors[uniquePointingScans[i]] = {}
        if (doplot):
            pb.ion()
            pb.clf()
            pb.gcf().set_size_inches(figsize[0], figsize[1], forward=True)
            adesc = pb.subplot(111)
        c = ['blue','red']
        seconds = []
        for p in range(len(pols)):
            seconds.append(np.max(np.max(np.abs(colOffsetRelative[p][0]))) + np.median(colError[0]))
            seconds.append(np.max(np.max(np.abs(colOffsetRelative[p][1]))) + np.median(colError[1]))
        seconds = np.max(seconds)*1.25
        if (seconds <= previousSeconds):
            # don't allow subsequent plots to cover a smaller range of sky
            seconds = previousSeconds
        previousSeconds = seconds
        if (doplot):
            pb.xlim([-seconds,seconds])
            pb.ylim([-seconds,seconds])
#            pb.hold(True) # not needed
        thisscan = np.where(scans == uniquePointingScans[i])[0]
        mystd = []
        mystdOfError = [] 
        colOffsetRelativeTrans = np.transpose(colOffsetRelative[thisscan])
        colErrorTrans = np.transpose(colError[thisscan])
#        print "np.shape(colOffsetRelativeTrans) = ", np.shape(colOffsetRelativeTrans)
        for p in range(len(pols[thisscan[0]])):
            mystd.append([np.std(colOffsetRelativeTrans[0][p]), np.std(colOffsetRelativeTrans[1][p])])
            mystdOfError.append([np.std(colErrorTrans[0][p]), np.std(colErrorTrans[1][p])]) 
        for a in thisscan:
            # 'a' is now a row number that corresponds to the ith pointing scan
            antenna = np.where(uniqueAntennaNames==antennaName[a])[0][0]
            for p in range(len(pols[a])):
                if (doplot):
                    pb.errorbar(colOffsetRelative[a][p][0], colOffsetRelative[a][p][1], fmt='o',
                            yerr=colError[a][p][1], xerr=colError[a][p][0],
                            color=overlayColors[antenna], markersize=5,
                            markerfacecolor=overlayColors[antenna], markeredgecolor=overlayColors[antenna])
                totalOffset = (colOffsetRelative[a][p][0]**2 + colOffsetRelative[a][p][1]**2)**0.5
                totalError = (colError[a][p][0]**2 + colError[a][p][1]**2)**0.5
                if (listsigma and p==0):
                    print("%s: cross-elev: %5.1f  elevation: %5.1f   total: %5.1f" % (antennaName[a],
                                                  np.abs(colOffsetRelative[a][p][0]/colError[a][p][0]),
                                                  np.abs(colOffsetRelative[a][p][1]/colError[a][p][1]),
                                                  np.abs(totalOffset/totalError)))
                if (totalOffset > thresholdArcsec):
                    outlier[uniquePointingScans[i]][antennaName[a]] = [True, totalOffset]
                else:
                    outlier[uniquePointingScans[i]][antennaName[a]] = [False]
                radialErrors[uniquePointingScans[i]][antennaName[a]] = totalOffset
                if (doplot and (labels or abs(colOffsetRelative[a][p][0])>nsigma*mystd[p][0] or
                                abs(colOffsetRelative[a][p][1])>nsigma*mystd[p][1]) or
                                (colError[a][p][0] > nsigma*mystdOfError[p][0]) or
                                (colError[a][p][1] > nsigma*mystdOfError[p][1])
                    ):
                    pb.text(colOffsetRelative[a][p][0], colOffsetRelative[a][p][1], 
                            antennaName[a], color='k', size=8)
        # Draw spec
        if (doplot):
            cir = pb.Circle((0, 0), radius=2, facecolor='none', edgecolor='k', linestyle='dotted')
            pb.gca().add_patch(cir)
#        print "scandict = ", scandict
#        print "len(scandict) = ", len(scandict)
#        print "\n scandict[%d].keys() = " % (uniquePointingScans[i]), scandict[uniquePointingScans[i]].keys()
        pointingSource = scandict[uniquePointingScans[i]]['source'].split(';')[0]
        (mjd, utstring) = mjdSecondsToMJDandUT(startValidTime[thisscan[0]])
        if (doplot):
            pb.title('Relative collimation offsets at %s - %s scan %d' % (utstring,pointingSource,uniquePointingScans[i]),fontsize=12)
            pb.axvline(0,color='k',linestyle='--')
            pb.axhline(0,color='k',linestyle='--')
            pb.xlabel('Cross-elevation offset (arcsec)')
            pb.ylabel('Elevation offset (arcsec)')
            pb.axis('scaled')
            if (Xrange[0] != 0 or Xrange[1] != 0):
                if (yrange[0] != 0 or yrange[1] != 0):
                    pb.axis([Xrange[0],Xrange[1],yrange[0],yrange[1]])
                else:
                    pb.axis([Xrange[0],Xrange[1],-seconds,seconds])
            elif (yrange[0] != 0 or yrange[1] != 0):
                pb.axis([-seconds,seconds, yrange[0], yrange[1]])
            else:
                pb.axis([-seconds,seconds,-seconds,seconds])

            # now draw the legend for plotPointingResultsFromASDM
            myxlim = pb.xlim()
            myylim = pb.ylim()
            myxrange = myxlim[1]-myxlim[0]
            myyrange = myylim[1]-myylim[0]
            x0 = myxlim[0] + 0.05*myxrange
            y0 = myylim[1]-np.abs(myyrange*0.05)
            cornerRadius = (myxlim[0]**2 + myylim[0]**2)**0.5
            if False:
                if (cornerRadius > thresholdArcsec):
                    cir = pb.Circle((0, 0), radius=thresholdArcsec, facecolor='none', edgecolor='r', linestyle='dashed')
                    pb.gca().add_patch(cir)
                    if (meanScienceBeam):
                        x1 = myxlim[1] - 0.05*myxrange
                        pb.text(x1,y0,'mean science beam',color='r',fontsize=14,ha='right')
            x1 = myxlim[1] - 0.05*myxrange
            pb.text(x1,y0,'all-sky specification',color='k',fontsize=14,ha='right')
            y0 -= np.abs(myyrange*0.05)
            if (cornerRadius > halfbeamScience):
                cir = pb.Circle((0, 0), radius=halfbeamScience, facecolor='none', edgecolor='r', linestyle='dashed')
                pb.gca().add_patch(cir)
                pb.text(x1,y0,'mean science beam',color='r',fontsize=14,ha='right')
                y0 -= np.abs(myyrange*0.05)
            if (cornerRadius > halfbeamPointingScan):
                cir = pb.Circle((0, 0), radius=halfbeamPointingScan, facecolor='none', edgecolor='c', linestyle='dashed')
                pb.gca().add_patch(cir)
                pb.text(x1,y0,'pointing beam',color='c',fontsize=14,ha='right')
            pb.text(x0,y0,'azim=%+.0f, elev=%.0f'%(azim[thisscan[0]],elev[thisscan[0]]),color='k', fontsize=14)
            asdmNoPath = asdm.split('/')[-1]
            pb.text(myxlim[0], myylim[1]+0.05*myyrange, asdmNoPath+', pointing=%.1f GHz, science=%.1f GHz, diameter=%.0fm'%(frequency[thisscan[0]],meanScienceFreq,diameter), 
                    fontsize=12,color='k')
            mystep = 0.025-0.0001*(len(uniqueAntennaNames)) # 0.020 for 50 antennas, 0.0225 for 25 antennas, etc.
            for a in range(len(uniqueAntennaNames)):
                pb.text(myxlim[1]+0.02*myxrange, myylim[1]-(a+1)*myyrange*mystep, uniqueAntennaNames[a], color=overlayColors[a],fontsize=10)
            pb.axvline(0,color='k',linestyle='--')
            pb.axhline(0,color='k',linestyle='--')
            yFormatter = ScalarFormatter(useOffset=False)
            adesc.yaxis.set_major_formatter(yFormatter)
            adesc.xaxis.set_major_formatter(yFormatter)
            adesc.xaxis.grid(True,which='major')
            adesc.yaxis.grid(True,which='major')
            if (figfile==True):
                myfigfile = figfiledir + asdmNoPath + '.pointing.scan%02d.png' % (uniquePointingScans[i])
                pointingPlots.append(myfigfile)
                pb.savefig(myfigfile,density=144)
                plotfiles.append(myfigfile)
                print("Figure left in %s" % myfigfile)
            elif (figfile != False):
                myfigfile = figfiledir + figfile.replace('.png','') + '.scan%02d.png' % (uniquePointingScans[i])
                pointingPlots.append(myfigfile)
                pb.savefig(myfigfile, density=144)
                plotfiles.append(myfigfile)
                print("Figure left in %s" % myfigfile)
            else:
                print("To make a hardcopy, re-run with figfile=True or figfile='my.png'")
            if (buildpdf):
                cmd = '%s -density 144 %s %s.pdf'%(convert,myfigfile,myfigfile)
                print("Running command = ", cmd)
                mystatus = os.system(cmd)
                if (mystatus == 0):
                    print("plotfiles[i] = ", plotfiles[i])
                    filelist += plotfiles[i] + '.pdf '
                else:
                    print("ImageMagick's convert command is missing, no PDF created.")
                    buildpdf = False
            pb.draw()
            if (i < nPointingScans-1 and interactive):
                mystring = raw_input("Press return for next scan (or 'q' to quit): ")
                if (mystring.lower().find('q') >= 0):
                    return(pointingPlots)
      # end 'for' loop over scans
      badAntennas = {}
      for antenna in uniqueAntennaNames:
          goodscans = 0.0
          radialErrorsThisAntenna = {}
          for scan in uniquePointingScans:
              # datasets with only 2 antennas have 1 antenna solution per scan
              if (antenna in radialErrors[scan]):
                  radialErrorsThisAntenna[scan] = radialErrors[scan][antenna]
                  if (outlier[scan][antenna][0]==False):
                      goodscans += 1.0
                  
          badscans = len(uniquePointingScans)-goodscans
          if (badscans/len(uniquePointingScans) >= fractionOfScansBad):
              print("Antenna %s was an outlier (>%.1f arcsec) for %d/%d scans" % (antenna, thresholdArcsec, badscans, len(uniquePointingScans)))
              badAntennas[antenna] = radialErrorsThisAntenna
      if (buildpdf and doplot):
          pdfname = ''
          if (os.path.dirname(myfigfile) != ''):
              pdfname += os.path.dirname(myfigfile)+'/'
          pdfname += asdm+'.asdmpointing.pdf'
          if (nPointingScans > 1):
              mystatus = concatenatePDFs(filelist, pdfname, pdftk=pdftk, gs=gs)
          else:
              cmd = 'cp %s %s' % (filelist,pdfname)
              print("Running command = %s" % (cmd))
              mystatus = os.system(cmd)
          if (mystatus == 0):
              print("PDF left in %s" % (pdfname))
              os.system("rm -f %s" % filelist)
          else:
              print("Both pdftk and ghostscript are missing, no PDF created")
      if montage:
          outname = asdm + '.asdmpointing.png'
          montagePngs(pointingPlots, outname=outname, tile=tile, 
                      geometry=geometry, background='white')
          print("Wrote ", outname)
      return(pointingPlots, badAntennas)  
# end of plotPointingResultsFromASDM
            
def offlineTcAtmosphere(asdm, scanlist, mode='AH', origin='specauto', quantcorrection=False) :
    """
    Runs the casapy-telcal command tc_atmosphere to recompute the Trx/Tsys spectra
    in SysCal table.  Need to set showplot=False to avoid crash.
    """
    from tc_atmosphere_cli import tc_atmosphere_cli as tc_atmosphere
    if type(scanlist) == int:
        scanlist = [scanlist]
    for scan in scanlist :
        print(scan)
        tc_atmosphere(asdm=asdm, dataorigin=origin, trecmode=mode, scan=str(scan),
                      antenna='', calresult=asdm, showplot=False, verbose=False,
                      quantcorrection=quantcorrection)

def importasdm2(asdm, intent=''):
    """
    Translates a specified intent into a list of scans and then only imports those
    scans into a measurement set.
    """
    f = open(asdm+'/Scan.xml')
    fc = f.read()
    f.close()

    scanBlockList = re.findall('<row>.+?</row>', fc, re.DOTALL|re.MULTILINE)

    scanNumList = []

    for i in range(len(scanBlockList)):
        scanBlocksIntents = re.findall('<scanIntent>.+?</scanIntent>', scanBlockList[i])[0]
        if re.search(intent, scanBlocksIntents) is not None:
            scanNum1 = re.findall('<scanNumber>[0-9]+</scanNumber>', scanBlockList[i])
            if len(scanNum1) != 1: continue
            scanNum1 = re.findall('[0-9]+', scanNum1[0])
            if len(scanNum1) != 1: continue
            scanNumList.append(scanNum1[0])

    scanNumList = ','.join(scanNumList)

    importasdm(asdm, scans=scanNumList)

def readwvr(sdmfile, verbose=False):
    """
    This function reads the CalWVR.xml table from the ASDM and returns a
    dictionary containing: 'start', 'end', 'startmjd', 'endmjd',
    'startmjdsec', 'endmjdsec',
    'timerange', 'antenna', 'water', 'duration'.
    'water' is the zenith PWV in meters.
    This function is called by readpwv(). -- Todd Hunter
    """
    if (os.path.exists(sdmfile) == False):
        print("readwvr(): Could not find file = ", sdmfile)
        return
    xmlscans = minidom.parse(sdmfile+'/CalWVR.xml')
    scandict = {}
    rowlist = xmlscans.getElementsByTagName("row")
    fid = 0
    myqa = createCasaTool(qatool)
    for rownode in rowlist:
        rowpwv = rownode.getElementsByTagName("water")
        pwv = float(rowpwv[0].childNodes[0].nodeValue)
        water = pwv
        scandict[fid] = {}

        # start and end times in mjd ns
        rowstart = rownode.getElementsByTagName("startValidTime")
        start = int(rowstart[0].childNodes[0].nodeValue)
        startmjd = float(start)*1.0E-9/86400.0
        t = myqa.quantity(startmjd,'d')
        starttime = call_qa_time(t,form="ymd",prec=8)
        rowend = rownode.getElementsByTagName("endValidTime")
        end = int(rowend[0].childNodes[0].nodeValue)
        endmjd = float(end)*1.0E-9/86400.0
        t = myqa.quantity(endmjd,'d')
        endtime = call_qa_time(t,form="ymd",prec=8)
        # antenna
        rowantenna = rownode.getElementsByTagName("antennaName")
        antenna = str(rowantenna[0].childNodes[0].nodeValue)

        scandict[fid]['start'] = starttime
        scandict[fid]['end'] = endtime
        scandict[fid]['startmjd'] = startmjd
        scandict[fid]['endmjd'] = endmjd
        scandict[fid]['startmjdsec'] = startmjd*86400
        scandict[fid]['endmjdsec'] = endmjd*86400
        timestr = starttime+'~'+endtime
        scandict[fid]['timerange'] = timestr
        scandict[fid]['antenna'] = antenna
        scandict[fid]['water'] = water
        scandict[fid]['duration'] = (endmjd-startmjd)*86400
        fid += 1

    if verbose: print('  Found ',rowlist.length,' rows in CalWVR.xml')
    myqa.done()
    # return the dictionary for later use
    return scandict
# Done

def readpwv(asdm):
  """
  This function assembles the dictionary returned by readwvr() into arrays
  containing the PWV measurements written by TelCal into the ASDM.
  Units are in meters.
  -- Todd Hunter
  """
#  print "Entered readpwv"
  dict = readwvr(asdm)
#  print "Finished readwvr"
  bigantlist = []
  for entry in dict:
      bigantlist.append(dict[entry]['antenna'])
  antlist = np.unique(bigantlist)
  watertime = []
  water = []
  antenna = []
  for entry in dict:
      measurements = 1
      for i in range(measurements):
          watertime.append(dict[entry]['startmjdsec']+(i*1.0/measurements)*dict[entry]['duration'])
          water.append(dict[entry]['water'])
          antenna.append(dict[entry]['antenna'])
  return([watertime,water,antenna])   

def readSysCal(asdm):
    """
    This function reads the SysCal.xml table from the ASDM and checks
    how many Tsys entries there are for each combination of Antenna,
    time and spw (which should be 1). Reports the number of duplicates
    and returns a dictionary keyed by [antenna][spw] = list of timestamps
    -- Todd Hunter
    """
    if (os.path.exists(asdm) == False):
        print("readsyscal(): Could not find file = ", asdm)
        return
    xmlscans = minidom.parse(asdm+'/SysCal.xml')
    scandict = {}
    rowlist = xmlscans.getElementsByTagName("row")
    fid = 0
    duplicates = 0
    scandict = {}
    for rownode in rowlist:
        antennaID = rownode.getElementsByTagName("antennaId")
        antenna = int(str(antennaID[0].childNodes[0].nodeValue).split('_')[1])
        if (antenna not in list(scandict.keys())):
            scandict[antenna] = {}
        spwID = rownode.getElementsByTagName("spectralWindowId")
        spw = int(str(spwID[0].childNodes[0].nodeValue).split('_')[1])
        if (spw not in list(scandict[antenna].keys())):
            scandict[antenna][spw] = []
        timeData = rownode.getElementsByTagName("timeInterval")
        timeStamp = int(str(timeData[0].childNodes[0].nodeValue).split()[0])
        timeInterval = int(str(timeData[0].childNodes[0].nodeValue).split()[1])
        if (timeStamp-timeInterval/2 in scandict[antenna][spw]):
            print("Duplicate seen!")
            duplicates += 1
        else:
            scandict[antenna][spw].append(timeStamp-timeInterval/2)

        # start and end times in mjd ns
#        rowstart = rownode.getElementsByTagName("startValidTime")
#        start = int(rowstart[0].childNodes[0].nodeValue)
#        startmjd = float(start)*1.0E-9/86400.0
#        scandict[fid]['start'] = starttime
        fid += 1
    print('  Found ',rowlist.length,' Tsys rows in SysCal.xml')
    print("%d duplicates found" % (duplicates))
    return (scandict)

def replaceTsysFromSQLD(caltable, sqld, asdm, spws, verbose=False):
    """
    This is a utility to replace the Tsys spectrum for one combination
    of antenna+spw+pol+scan with the median value from the square law 
    detector Tsys located in the SysCal.xml table.
    Typically, the SysCal.xml table was produced manually by running 
    casapy-telcal offline.

    caltable: the name of the gencal-produced Tsys cal table to correct
    sqld: the path to the SysCal.xml file to get the SQLD Tsys values from
    asdm: the path to the original ASDM (with SpectralWindow.xml and Scan.xml) 
    spws: a list of spws to replace (in order of baseband number).  The number of
          spws specified must match the number of SQLDs in the SysCal.xml file.
    - Todd Hunter
    """
    tsys = getTsysFromSysCal(asdm, sqld)
    if tsys is None:
        return
    values = 0
    if (type(spws) == str):
        spws = spws.split(',')
    changeFactor = []
    for antenna in list(tsys.keys()):
        for spw in list(tsys[antenna].keys()):
            baseband = tsys[antenna][spw]['baseband']-1 # 1..4  converted here to 0..3
            for scan in list(tsys[antenna][spw]['scans'].keys()):
                for pol in list(tsys[antenna][spw]['scans'][scan].keys()):
                    if (pol == 0): polarization = 'X'
                    if (pol == 1): polarization = 'Y'
                    if (verbose):
                        print("Calling replaceTsysScan('%s', antenna=%d, spw=%d, pol='%s', fromscan=%d, toscan=%d, newvalue=%f)" % (caltable, antenna, int(spws[baseband]), polarization, scan, scan, np.median(tsys[antenna][spw]['scans'][scan][pol])))
                    replaced, change = replaceTsysScan(caltable, antenna, int(spws[baseband]), polarization, scan, scan, tsys[antenna][spw]['scans'][scan][pol],verbose=verbose)
                    values += replaced
                    if (replaced > 0):
                        changeFactor.append(change)
    print("Replaced %d values with a median change of a factor of %f" % (values,np.median(changeFactor)))

def getMedianTsys(asdmlist, verbose=False):
    """
    Gets the median Tsys value for each ASDM in a list.
    asdmlist: either a list of strings or a single string with wildcard character(s).
          Files containing .ms and those not starting with uid___ are ignored.
          If the string does not have a wildcard character, then interpret it as a single file.
          If the file is not a directory (i.e. not an ASDM), then interpret it as a file that
            contains a list of ASDMs to process.
    Returns:
    if a wildcard is given: returns a dictionary of Tsys values keyed by ASDM name
    otherwise: returns the single median Tsys value
    -Todd Hunter
    """
    wildcardPresent = False
    if (type(asdmlist) == str):
        if (asdmlist.find('*') >= 0):
            wildcardPresent = True
            asdmlist = glob.glob(asdmlist)
        elif (os.path.isdir(asdmlist)):
            asdmlist = [asdmlist]
        elif (os.path.exists(asdmlist)):
            asdmlist = getListOfFilesFromFile(asdmlist, appendms=False)
        else:
            print("File not found")
            return
    tsysdict = {}
    tsys = None
    for asdm in asdmlist:
        basename = os.path.basename(asdm)
        if (basename.find('.') < 0 and basename.find('uid___')==0):
            print("Checking ", basename)
            tsys = getTsysFromSysCal(asdm, median=True, verbose=verbose)
            if tsys is None:
                return
            tsys = tsys['median']
            tsysdict[asdm] = tsys
            print("%s:  %f" % (asdm,tsys))
        else:
            print("Skipping %s because it is probably not an ASDM" % (basename))
            
    if (len(tsysdict) < 2 and not wildcardPresent):
        return(tsys)
    else:
        return(tsysdict)
    
def getMedianTsysForChannel(asdm, channel, spw):
    """
    Computes the median over all antennas of a single Tsys channel in one spw in one
    ASDM.
    -Todd Hunter
    """
    mydict = getTsysFromSysCal(asdm,median=True,channel=channel)
    if mydict is None:
        return
    tsys = []
    for antenna in list(mydict.keys()):
        if (antenna != 'median'): 
            tsys.append(mydict[antenna][spw]['median'])
    return(np.median(tsys))

def doubleIntervalInSysCal(vis, rows=range(32,64), factor=2, shift=0.5, 
                           dryrun=False):
    """
    Double the INTERVAL in SYSCAL table, and shift *earlier* by prior
    interval.
    rows: list of rows to change (None means all rows)
    factor: value by which to multiply the interval
    shift: fraction of original interval to shift the TIME column
    -Todd Hunter
    """
    if not os.path.exists(vis):
        print("Could not find measurement set")
        return
    mytb = tbtool()
    mytb.open(os.path.join(vis,'SYSCAL'), nomodify=False)
    if rows is None:
        i = mytb.getcol('INTERVAL')
        new = i*factor
        if dryrun:
            print("New interval would be", new)
        else:
            mytb.putcol('INTERVAL',new)
    else:
        for row in rows:
            i = mytb.getcell('INTERVAL', row)
            new = i*factor
            if dryrun:
                print("New interval would be", new)
            else:
                mytb.putcell('INTERVAL', row, new)
    mytb.close()
    if dryrun:
        print("Shift would be %f" % (-i*shift))
    else:
        adjustStartTimeInSysCal(vis, addSeconds=-i*shift, rows=rows) 
    
def adjustStartTimeInSysCal(vis, addSeconds=-1, rows=None):
    """
    Adds the specified number of seconds to the TIME value for each row of the SYSCAL table
    of a measurement set.
    rows: list of rows to change (None means all rows)
    -Todd Hunter
    """
    if not os.path.exists(vis):
        print("Could not find measurement set")
        return
    mytb = tbtool()
    mytb.open(os.path.join(vis,'SYSCAL'), nomodify=False)
    if rows is None:
        i = mytb.getcol('TIME')
        i += addSeconds
        mytb.putcol('TIME',i)
    else:
        for row in rows:
            i = mytb.getcell('TIME', row)
            i += addSeconds
            mytb.putcell('TIME',row,i)
    mytb.close()

def extendIntervalInSysCal(vis, addSeconds=1):
    """
    Adds the specified number of seconds to the valid interval for each row of the SYSCAL table
    of a measurement set.
    -Todd Hunter
    """
    if not os.path.exists(vis):
        print("Could not find measurement set")
        return
    mytb = tbtool()
    mytb.open(os.path.join(vis,'SYSCAL'), nomodify=False)
    i = mytb.getcol('INTERVAL')
    i += addSeconds
    mytb.putcol('INTERVAL',i)
    mytb.close()
    
def getSpwsFromSysCal(vis):
    """
    Returns the sorted list of unique spw IDs in the SYSCAL table of a measurement set.
    """
    if not os.path.exists(vis):
        print("Could not find measurement set")
        return
    mytb = tbtool()
    mytable = os.path.join(vis,'SYSCAL')
    if not os.path.exists(mytable):
        print("Could not find SYSCAL table")
        return
    mytb.open(mytable)
    spw = sorted(np.unique(mytb.getcol('SPECTRAL_WINDOW_ID')))
    mytb.close()
    return spw

def getTsysFromSysCal(asdm, sqld=None, median=False, verbose=True, channel=None):
    """
    This function reads the Tsys values from a SysCal.xml table 
    associated with an ASDM into a dictionary keyed by antenna ID, 
    spw, scan, and pol (0 or 1).  
    asdm: the path to the original ASDM (containing Scan.xml and SpectralWindow.xml)
    sqld: the path to the SysCal.xml table to use (or its parent directory)
      If sqld is not specified, then assume it is the one inside the ASDM.
      The reason you might need to specify both is because the primary usage is
      to read values from a SysCal.xml file produced by offline casapy-telcal.
    median: if True, then compute the median across frequency on a 
            per-antenna/spw/scan/pol basis and a global basis 
    channel: use the value only from the one specified channel (0..127)
    verbose: if True, print the number of rows and channels found
    -- Todd Hunter
    """
    if (os.path.exists(asdm) == False):
        print("getTsysFromSysCal(): Could not find ASDM = ", asdm)
        return
    if (sqld is None):
        sqld = asdm + '/SysCal.xml'
    if (os.path.exists(sqld) == False):
        print("getTsysFromSysCal(): Could not find file = ", sqld)
        return
    if (sqld.split('/')[-1] != 'SysCal.xml'):
        sqld += '/SysCal.xml'
    scanlist = readscans(asdm)[0]
    wvrSpws = []
    xmlscans = minidom.parse(asdm+'/SpectralWindow.xml')
    rowlist = xmlscans.getElementsByTagName("row")
    baseband = {}
    for rownode in rowlist:
        name = rownode.getElementsByTagName("name")
        name = str(name[0].childNodes[0].nodeValue)
        spwID = rownode.getElementsByTagName("spectralWindowId")
        spw = int(str(spwID[0].childNodes[0].nodeValue).split('_')[1])
        basebandName = rownode.getElementsByTagName("basebandName")
        basebandName = str(basebandName[0].childNodes[0].nodeValue).split('_')  # e.g. "BB_1"
        if (len(basebandName) < 2):
            baseband[spw] = -1  # NOBB
        else:
            baseband[spw] = int(basebandName[1])
        if (name.find('WVR') >= 0):
            wvrSpws.append(spw)
    xmlscans = minidom.parse(sqld)
    rowlist = xmlscans.getElementsByTagName("row")
    scandict = {}
    fid = 0
    duplicates = 0
    scandict = {}
    nchans = []
    if len(rowlist) == 0:
        print("Now rows found in SysCal.xml, probably because all the data are now in the binary file.")
        return
    for rownode in rowlist:
        antennaID = rownode.getElementsByTagName("antennaId")
        antenna = int(str(antennaID[0].childNodes[0].nodeValue).split('_')[1])
        if (antenna not in list(scandict.keys())):
            scandict[antenna] = {}
        spwID = rownode.getElementsByTagName("spectralWindowId")
        asdmspw = int(str(spwID[0].childNodes[0].nodeValue).split('_')[1])
        subtract = len(np.where(asdmspw > np.array(wvrSpws))[0])-1 
        spw = asdmspw-subtract # translate this to actual spw number, as only 1 WVR spw is real
        if (spw not in list(scandict[antenna].keys())):
            scandict[antenna][spw] = {}
            scandict[antenna][spw]['baseband'] = baseband[asdmspw]
            scandict[antenna][spw]['scans'] = {}
        timeData = rownode.getElementsByTagName("timeInterval")
        timeStamp = int(str(timeData[0].childNodes[0].nodeValue).split()[0])
        timeInterval = int(str(timeData[0].childNodes[0].nodeValue).split()[1])
        timeCenter = timeStamp-timeInterval/2
        timeCenterMJD = timeCenter*1e-9/86400.
        scan = -1
        for s in list(scanlist.keys()):
            if (scanlist[s]['endmjd'] >= timeCenterMJD and scanlist[s]['startmjd'] <= timeCenterMJD):
                scan = s
        if (scan not in list(scandict[antenna][spw]['scans'].keys())):
            scandict[antenna][spw]['scans'][scan] = {}
        tsysSpectrum = rownode.getElementsByTagName("tsysSpectrum")
        values = tsysSpectrum[0].childNodes[0].nodeValue.split()
        npol = int(values[1])  # or is it [0] ?
        nchan = int(values[2])
        nchans.append(nchan)
        for pol in range(npol):
            tsys = []
            for i in range(pol*nchan, nchan*(pol+1)):
                tsys.append(float(values[3+i]))
            if (channel is None):
                scandict[antenna][spw]['scans'][scan][pol] = tsys
            else:
                scandict[antenna][spw]['scans'][scan][pol] = [tsys[channel]]
        fid += 1
    if verbose:
        print('  Found ',rowlist.length,' Tsys rows in SysCal.xml (median # chans = %.0f)' % (np.median(nchans)))
    if (median):
        mediandict = {}
        allvalues = []
        for antennaId in list(scandict.keys()):
            if antennaId not in list(mediandict.keys()): mediandict[antennaId] = {}
            spwvalues = []
            for spw in list(scandict[antennaId].keys()):
                if spw not in list(mediandict[antennaId].keys()): mediandict[antennaId][spw] = {}
                scanvalues = []
                for scan in list(scandict[antennaId][spw]['scans'].keys()):
                    if scan not in list(mediandict[antennaId][spw].keys()): mediandict[antennaId][spw][scan] = {}
                    polvalues = []
                    for pol in list(scandict[antennaId][spw]['scans'][scan].keys()):
                        if pol not in list(mediandict[antennaId][spw][scan].keys()): mediandict[antennaId][spw][scan][pol] = {}
                        values = scandict[antennaId][spw]['scans'][scan][pol]
                        polvalues += values
                        # Use 'median' label to allow future keys like 'mean' or 'std'
                        mediandict[antennaId][spw][scan][pol]['median'] = np.nanmedian(values)
                    mediandict[antennaId][spw][scan]['median'] = np.nanmedian(polvalues)
                    scanvalues += polvalues
                mediandict[antennaId][spw]['median'] = np.nanmedian(scanvalues)
                spwvalues += scanvalues
            mediandict[antennaId]['median'] = np.nanmedian(spwvalues)
            allvalues += spwvalues
        mediandict['median'] = np.nanmedian(allvalues)
        return (mediandict)
    else:
        return (scandict)
        

def getNonWvrSpws(mymsmd):
    """
    Uses the msmd tool instance to find a list of the non-WVR spws, in a backward-compatible way.
    -Todd Hunter
    """
    try:
        spws = list(set(range(mymsmd.nspw())).difference(set(mymsmd.almaspws(wvr=True))))
    except:
        spws = list(set(range(mymsmd.nspw())).difference(set(mymsmd.wvrspws())))
    return(spws)

def getNonChanAvgSpws(mymsmd):
    """
    Uses the msmd tool instance to find a list of the non-WVR spws, in a backward-compatible way.
    -Todd Hunter
    """
    try:
        spws = list(set(range(mymsmd.nspw())).difference(set(mymsmd.almaspws(chavg=True))))
    except:
        spws = list(set(range(mymsmd.nspw())).difference(set(mymsmd.chanavgspws())))
    return(spws)

def getFrequenciesFromASDM(asdm, spws=None, minnumchan=128):
    """
    Gets a list of the central frequencies of each spw in the ASDM. Extraneous
    WVR spws are skipped, so the order of the list should match listobs in CASA.
    spws: a list of spws to which to restrict the result
    minnumchan: limit the spws returned to those with at least this many 
                channels
    Returns:
    an array of frequencies (in Hz)
    - Todd Hunter
    """
    mydict = getSpwsFromASDM(asdm, minnumchan, dropExtraWVRSpws=True)
    freqs = []
    if (type(spws) == str):
        spws = [int(i) for i in spws.split(',')]
    elif (type(spws) == int):
        spws = [spws]
    for spw in sorted(mydict.keys()):
        if (spws is None):
            freqs.append(mydict[spw]['centerFreq'])
        elif (spw in spws):
            freqs.append(mydict[spw]['centerFreq'])
    return(np.array(freqs))

def getScienceBasebandRFRanges(vis, mymsmd=None, returnLOs=False):
    """
    Reports the 2GHz-wide ranges of ALMA science basebands.
    Returns: a dictionary keyed by baseband number, with values in Hz
    returnLOs: if True, then also return dictionaries for LO1 and LO2
    """
    needToClose = False
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    freqs = getScienceBasebandFrequencies(vis, mymsmd) # dictionary keyed by baseband
    if returnLOs:
        LO1s, LO2s = interpretLOs(vis, mymsmd=mymsmd, showOnlyScienceSpws=True, alsoReturnLO2=True)
    else:
        LO1s = interpretLOs(vis, mymsmd=mymsmd, showOnlyScienceSpws=True)
    ranges = {}
    sqldSpws = getScienceSpws(vis,sqld=True,fdm=False,tdm=False,returnString=False,mymsmd=mymsmd)
    for spw in LO1s:
        baseband = mymsmd.baseband(spw)
        basebandFreq = freqs[baseband]
        sqldspw = np.intersect1d(mymsmd.spwsforbaseband(baseband), sqldSpws)[0]
        bandwidth = mymsmd.bandwidths(sqldspw)
        ranges[baseband] = [basebandFreq-bandwidth/2, basebandFreq+bandwidth/2]
    if needToClose:
        mymsmd.close()
    if returnLOs:
        return ranges, LO1s, LO2s
    else:
        return ranges

def getScienceSpwBasebands(vis, mymsmd=None):
    """
    Reports the baseband number for each science spw
    Returns: a dictionary keyed by spw number, with values of 1..4
    -Todd Hunter
    """
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    else:
        needToClose = False
    spws = getScienceSpws(vis, returnString=False, mymsmd=mymsmd)
    mydict = {}
    for spw in spws:
        mydict[spw] = mymsmd.baseband(spw)
    if needToClose:
        mymsmd.close()
    return mydict
    
def checkScienceSpwIFRangesRelativeToBasebandIFRanges(vis, thresholdMHz=30, maxBandwidthMHz=1800, verbose=True):
    """
    Computes gap between edges of each spw and edges of its baseband, and
    reports values less than the threshold.
    maxBandwidthMHz: don't report spws with wider bandwidth than this (because pipeline flags 
       ACA 2000 MHz "FDM" spws back to 1875 MHz total bandwidth)
    -Todd Hunter
    """
    if not os.path.exists(vis):
        print("Could not find measurement set.")
        return
    mymsmd = msmdtool()
    mymsmd.open(vis)
    basebandIFs = getScienceBasebandIFRanges(vis, mymsmd)
    spwIFs = getScienceSpwIFRanges(vis, mymsmd)
    basebands = getScienceSpwBasebands(vis, mymsmd)
    sidebands = getScienceSpwSidebands(vis, mymsmd=mymsmd)
    bandwidths = mymsmd.bandwidths()
    mymsmd.close()
    threshold = thresholdMHz * 1e6
    mydict = {}
    for spw in list(spwIFs.keys()):
        diffs = [abs(spwIFs[spw][0] - basebandIFs[basebands[spw]][0]),
                 abs(spwIFs[spw][0] - basebandIFs[basebands[spw]][1]),
                 abs(spwIFs[spw][1] - basebandIFs[basebands[spw]][0]),
                 abs(spwIFs[spw][1] - basebandIFs[basebands[spw]][1])]
        if np.min(diffs) < threshold and bandwidths[spw] < maxBandwidthMHz*1e6:
            mydict[spw] = np.min(diffs)
            if verbose:
                print("spw %d (%s): %s MHz, baseband: %s MHz" % (spw, sidebands[spw], sorted(1e-6*np.array(spwIFs[spw])), sorted(1e-6*np.array(basebandIFs[basebands[spw]]))))
    return mydict

def checkScienceSpwSamplerRanges(vis, thresholdMHz=30, maxBandwidthMHz=1800, verbose=True):
    """
    Computes the science spw sampler rangs, and reports any that are within 
    thresholdMHz of either edge.
    maxBandwidthMHz: don't report spws with wider bandwidth than this (because pipeline flags 
       ACA 2000 MHz "FDM" spws back to 1875 MHz total bandwidth)
    -Todd Hunter
    """
    samplerRanges = getScienceSpwSamplerRanges(vis, verbose=verbose)
    if samplerRanges is None: return
    spws = {}
    threshold = thresholdMHz * 1e6
    for spw in samplerRanges:
        bandwidth = np.abs(np.diff(samplerRanges[spw]))
        minOffset = np.min(np.abs(4e9 - samplerRanges[spw]))
        if (minOffset < threshold and bandwidth < maxBandwidthMHz*1e6):
            spws[spw] = {'4GHz_edge': minOffset, 'bandwidth': bandwidth}
        minOffset = np.min(np.abs(2e9 - samplerRanges[spw]))
        if (minOffset < threshold and bandwidth < maxBandwidthMHz*1e6):
            spws[spw] = {'2GHz_edge': minOffset, 'bandwidth': bandwidth}
    return spws

def getScienceSpwSamplerRanges(vis, mymsmd=None, verbose=False):
    """
    Reports the range within the baseband sampler where the science spws reside 
    (should always be somewhere within 2-4 or 4-2 GHz)
    -Todd Hunter
    """
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        if verbose:
            print("opening msmd")
        mymsmd.open(vis)
    else:
        needToClose = False
    if verbose:
        print("getScienceSpwRFRanges")
    spwFreqs = getScienceSpwRFRanges(vis, mymsmd)
    if verbose:
        print("interpretLOs")
    result = interpretLOs(vis, mymsmd=mymsmd, showOnlyScienceSpws=True, alsoReturnLO2=True)
    if result is None: return
    LO1s, LO2s = result
    if needToClose:
        mymsmd.close()
    samplerFreqs = {}
    for spw in LO1s:
        if spwFreqs[spw][0] > LO1s[spw]:
            samplerFreqs[spw] = LO2s[spw] - (spwFreqs[spw] - LO1s[spw])
        else:
            samplerFreqs[spw] = LO2s[spw] - (LO1s[spw] - spwFreqs[spw])
    return samplerFreqs

def getScienceBasebandSamplerRanges(vis, mymsmd=None, verbose=False):
    """
    Reports the range within the baseband sampler where the baseband resides 
    (should always be 2-4 or 4-2 GHz)
    -Todd Hunter
    """
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        if verbose:
            print("opening msmd")
        mymsmd.open(vis)
    else:
        needToClose = False
    if verbose:
        print("getScienceBasebandRFRanges")
    basebandFreqs, LO1s, LO2s = getScienceBasebandRFRanges(vis, mymsmd, returnLOs=True)
    if verbose:
        print("getScienceSpwBasebands")
    basebands = getScienceSpwBasebands(vis, mymsmd)
    if needToClose:
        mymsmd.close()
    samplerFreqs = {}
    for spw in LO1s:
        if basebandFreqs[basebands[spw]][0] > LO1s[spw]:
            samplerFreqs[spw] = LO2s[spw] - (basebandFreqs[basebands[spw]] - LO1s[spw])
        else:
            samplerFreqs[spw] = LO2s[spw] - (LO1s[spw] - basebandFreqs[basebands[spw]])
    return samplerFreqs

def getScienceBasebandIFRanges(vis, mymsmd=None, verbose=False):
    """
    Reports the 2GHz-wide ranges of ALMA science basebands in the frame after LO1 but 
    before LO2 (i.e. 4-8 or 4-12 GHz)
    Returns: a dictionary keyed by baseband number, with values in Hz
    -Todd Hunter
    """
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    else:
        needToClose = False
    freqs = getScienceBasebandFrequencies(vis, mymsmd) # dictionary keyed by baseband
    LO1s = interpretLOs(vis, mymsmd=mymsmd, showOnlyScienceSpws=True)  # dictionary keyed by science spw number
    ranges = {}
    for spw in LO1s:
        baseband = mymsmd.baseband(spw)
        basebandFreq = freqs[baseband]
        sqldspw = np.intersect1d(mymsmd.spwsforbaseband(baseband), 
                                 getScienceSpws(vis,sqld=True,fdm=False,tdm=False,returnString=False,mymsmd=mymsmd))[0]
        bandwidth = mymsmd.bandwidths(sqldspw)
        if verbose:
            print("spw %d baseband bandwidth = " % (spw), bandwidth)
        ranges[baseband] = [abs(LO1s[spw]-basebandFreq)-bandwidth/2, abs(LO1s[spw]-basebandFreq)+bandwidth/2]
    if needToClose:
        mymsmd.close()
    return ranges

def getScienceSpwIFRanges(vis, mymsmd=None, spws=None, returnGHz=False, units='Hz'):
    """
    Reports the IF ranges of the ALMA science spws of a measurement set.
    spws: if specified, limit the spws to these (python list or comma-delimited string)
    units: 'Hz', 'MHz', or 'GHz'
    Returns: a dictionary keyed by spw number, with values in Hz
    """
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    else:
        needToClose = False
    if spws is None:
        spws = getScienceSpws(vis, mymsmd=mymsmd, returnString=False) # list of integers
    else:
        spws = parseSpw(vis, spws, mymsmd=mymsmd)
    LO1s = interpretLOs(vis, mymsmd=mymsmd, showOnlyScienceSpws=True)  # dictionary keyed by science spw number
    ranges = {}
    for spw in spws:
        bandwidth = mymsmd.bandwidths(spw)
        spwFreq = mymsmd.meanfreq(spw)
        ranges[spw] = np.array([abs(LO1s[spw]-spwFreq)-bandwidth/2, abs(LO1s[spw]-spwFreq)+bandwidth/2])
        if units == 'GHz':
            ranges[spw] *= 1e-9
        elif units == 'MHz':
            ranges[spw] *= 1e-6
        ranges[spw] = list(ranges[spw])
    if needToClose:
        mymsmd.close()
    return ranges

def compareRFandIFRanges(vislist=''):
    """
    Calls getScienceSpwRFRanges and getScienceSpwIFRanges and, if there is more than one vis, it 
    computes max difference in their respective center frequencies.
    vislist: if blank then use all .ms in the current directory (avoiding target.ms if some are target and some are not)
    -Todd Hunter
    """
    if vislist == '':
        vislist = sorted(glob.glob('*.ms'))
        target = 0
        for vis in vislist:
            if vis.find('target') > 0:
                target += 1
        if target != len(vislist):
            newvislist = []
            for vis in vislist:
                if vis.find('target') < 0:
                    newvislist.append(vis)
            vislist = newvislist
    units = 'Hz'
    mymsmd = createCasaTool(msmdtool)
    ifCenter = {}
    rfCenter = {}
    for i,vis in enumerate(vislist):
        print("EB %d of %d" % (i+1, len(vislist)))
        mymsmd.open(vis)
        ifRange = getScienceSpwIFRanges(vis, mymsmd=mymsmd, units=units)
        spwlist = sorted(list(ifRange.keys()))
        spwnames = getScienceSpwNames(vis, mymsmd=mymsmd, uniqueStringOnly=True)
        nTunings = len(np.unique(spwnames))
        rfRange = getScienceSpwRFRanges(vis, mymsmd=mymsmd, units=units)
        tunings = []
        for j,spw in enumerate(spwlist):
            if spwnames[j] not in tunings:
                tunings.append(spwnames[j])
                print("\nTuning %d of %d: %s" % (len(tunings), nTunings, spwnames[j]))
            if spw not in ifCenter.keys():
                ifCenter[spw] = []
                rfCenter[spw] = []
            ifCenter[spw].append(np.mean(ifRange[spw]))
            rfCenter[spw].append(np.mean(rfRange[spw]))
            rf0, rf1 = np.array(rfRange[spw])*1e-9
            if0, if1 = np.array(ifRange[spw])*1e-9
            print("%s: spw%02d: RF: %.6f-%.6f GHz,  IF: %.6f-%.6f GHz" % (os.path.basename(vis), spw, rf0, rf1, if0, if1))
        print("\n spw pair differences:")
        for spw in spwlist[:4]:
            if len(spwlist) >= 8:
                print("spw %02d to %02d: RF difference: %+8.3f MHz   IF difference: %+8.3f MHz" % (spw, spw+20, 1e-6*(rfCenter[spw][i] - rfCenter[spw+20][i]), 1e-6*(ifCenter[spw][i] - ifCenter[spw+20][i])))
            if len(spwlist) >= 12:
                print("spw %02d to %02d: RF difference: %+8.3f MHz   IF difference: %+8.3f MHz" % (spw, spw+40, 1e-6*(rfCenter[spw][i] - rfCenter[spw+40][i]), 1e-6*(ifCenter[spw][i] - ifCenter[spw+40][i])))
        print("\n")
    mymsmd.close()
    if len(vislist) > 1:
        print("\nRF maximum differences between EBs:")
        for spw in spwlist:
            rfDiff = np.max(rfCenter[spw]) - np.min(rfCenter[spw])
            print("spw %02d: %7.3f MHz" % (spw, rfDiff*1e-6))
        print("\nIF maximum differences between EBs:")
        for spw in spwlist:
            ifDiff = np.max(ifCenter[spw]) - np.min(ifCenter[spw])
            print("spw %02d: %7.3f MHz" % (spw, ifDiff*1e-6))
    
def getScienceSpwRFRanges(vis, mymsmd=None, spws=None, returnDistanceFromRXBandEdge=False, units='Hz'):
    """
    Reports the RF ranges of the ALMA science spws of a measurement set.
    spws: if specified, limit the spws to these (python list or comma-delimited string)
    units: Hz, GHz or MHz
    Returns: 1 or 2 dictionaries:
    1) a dictionary keyed by spw number, with values in Hz
    2) if returnDistanceFromRXBandEdge=True, then return smallest gap to a Rx Band edge (in MHz)
    """
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    else:
        needToClose = False
    if spws is None:
        spws = getScienceSpws(vis, mymsmd=mymsmd, returnString=False) # list of integers
    else:
        spws = parseSpw(vis, spws, mymsmd=mymsmd)
    ranges = {}
    distanceFromRxBandEdge = {}
    for spw in spws:
        bandwidth = mymsmd.bandwidths(spw)
        spwFreq = mymsmd.meanfreq(spw)
        ranges[spw] = np.array([spwFreq-bandwidth/2, spwFreq+bandwidth/2])
        if units == 'GHz':
            ranges[spw] *= 1e-9
        elif units == 'MHz':
            ranges[spw] *= 1e-6
        ranges[spw] = list(ranges[spw])
        band = bandforspw(spw, mymsmd=mymsmd)
        distanceFromRxBandLowEdge = np.min(ranges[spw]) - bandDefinitions[band][0]
        distanceFromRxBandHighEdge = bandDefinitions[band][1] - np.max(ranges[spw]) 
        distanceFromRxBandEdge[spw] = np.min([distanceFromRxBandLowEdge, distanceFromRxBandHighEdge]) * 1e-6
    if needToClose:
        mymsmd.close()
    if returnDistanceFromRXBandEdge:
        return ranges, distanceFromRxBandEdge
    else:
        return ranges

def getScienceBasebandFrequencies(vis, mymsmd=None, keyBySQLD=False):
    """
    Uses the ALMA SQLD spws to infer the baseband center frequencies of each science spw on the RF 
    (sky frequency) scale.
    keyBySQLD: if True, then key by SQLD spw instead of baseband number
    Returns: a dictionary keyed by baseband number with values of the baseband center frequency in Hz
    """
    spws = getScienceSpws(vis, sqld=True, tdm=False, fdm=False, returnString=False, mymsmd=mymsmd)
    if len(spws) == 0:
        print("There are no SQLD spws in this dataset, so the baseband frequencies are unknown.")
        return
    freqs = {}
    needToClose = False
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    for spw in spws:
        freq = getMeanFreqOfSpwlist(vis, spw, mymsmd=mymsmd)
        baseband = mymsmd.baseband(spw)
        if keyBySQLD:
            freqs[spw] = freq
        else:
            freqs[baseband] = freq
    if needToClose:
        mymsmd.close()
    return freqs

def getSpwSelForFreqRange(vis, frequency, intent='OBSERVE_TARGET#ON_SOURCE',verbose=False):
    """
    Returns a list of science spws and chans that cover a freq range
    vis: name of measurement set
    frequency: 2 LSRK freqs in Hz, GHz, or a string with units
    -Remy Indebetouw
    """
    if not os.path.exists(vis):
        print("Could not find measurement set.")
        return

    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)

    #spws = getScienceSpws(vis, returnString=False)
    spws = mymsmd.spwsforintent(intent)
    almaspws = mymsmd.almaspws(tdm=True,fdm=True)
    if (len(almaspws) > 0):
        spws = np.intersect1d(spws,almaspws)

    datestring = getObservationStartDate(vis, measuresToolFormat=True)
    observatory = mymsmd.observatorynames()[0]
    field = mymsmd.fieldsforintent(intent)
    if (len(field) == 0):
        print("No fields with that intent")
        return
    else:
        field = field[0]
        # print "Using field %d" % field
    result = getRADecForField(vis, field, returnReferenceFrame=True)
    if result is None: return
    radec, equinox = result
    ra = radec[0][0]
    dec = radec[1][0]
    rastr,decstr=rad2radec(ra,dec,verbose=False).split(",")

    if not isinstance(frequency,list):
        frequency=[frequency]
    if len(frequency)<2:
        frequency=[frequency[0],frequency[0]]
    LSRKfrequency0 = parseFrequencyArgumentToHz(frequency[0])
    if verbose:
        print("using ",datestring, rastr, decstr, equinox, observatory)
    TOPOfreq0 = lsrkToTopo(LSRKfrequency0, datestring, rastr, decstr, equinox, observatory)
    LSRKfrequency1 = parseFrequencyArgumentToHz(frequency[1])
    TOPOfreq1 = lsrkToTopo(LSRKfrequency1, datestring, rastr, decstr, equinox, observatory)
    if TOPOfreq1<TOPOfreq0:
        tmp=TOPOfreq0
        TOPOfreq0=TOPOfreq1
        TOPOfreq1=tmp

    spws2 = []
    spwstr = ""
    for spw in spws:
        freqs = mymsmd.chanfreqs(spw)
        if (np.min(freqs) <= TOPOfreq0 and np.max(freqs) >= TOPOfreq0):
            spws2.append(spw)
            thisdelta=mymsmd.chanwidths(spw).mean()
            if verbose:
                print("spw %i"%spw, "freq ",freqs[0],"-",freqs[-1])

            if thisdelta>0:
                z0=np.where(freqs<=TOPOfreq0)[0].max()
                # increasing freq, freq1>=freq0 so could be beyond maxchan
                z1=np.where(freqs<=TOPOfreq1)[0].max()
            else:
                z1=np.where(freqs<=TOPOfreq0)[0].min()
                # decreasing freq, freq1>=freq0 so could be < 0chan
                z0=np.where(freqs<=TOPOfreq1)[0].min()
            spwstr=spwstr+("%i:%i~%i,"%(spws2[-1],z0,z1))
    mymsmd.close()
    return(spwstr[0:-1])
    
def getScienceSpwsForFrequency(vis, frequency, nearestOnly=False, mymsmd=None):
    """
    Returns a list of science spws that cover a given frequency.
    vis: name of measurement set
    frequency: in Hz, GHz, or a string with units
    nearestOnly: if True, the return only one spw (nearest to center)
    -Todd Hunter
    """
    needToClose = False
    if mymsmd is None:
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        needToClose = True
    spws = getScienceSpws(vis, returnString=False, mymsmd=mymsmd)
    frequency = parseFrequencyArgumentToHz(frequency)
    spws2 = []
    delta = []
    for spw in spws:
        freqs = mymsmd.chanfreqs(spw)
        if (np.min(freqs) <= frequency and np.max(freqs) >= frequency):
            spws2.append(spw)
            delta.append(abs(frequency-mymsmd.meanfreq(spw)))
    if needToClose:
        mymsmd.close()
    if nearestOnly:
        return(spws2[np.argmin(delta)])
    else:
        return(spws2)

def checkScienceSpws(dir='./', intent='OBSERVE_TARGET#ON_SOURCE', tdm=True,
                     fdm=True, ignore='_target.ms'):
    """
    For all measurement sets in the specified directory, get the
    science spws and report any differences in ID number.
    dir: comma-delimited list of directories, wildcard string, or 
         list of directories
    -Todd Hunter
    """
    if type(dir) == str:
        if (dir.find('*')>=0):
            dirs = glob.glob(dir)
        else:
            dirs = dir.split(',')
    else:
        dirs = dir
    for dir in dirs:
        print(dir+":")
        vis = glob.glob(dir+'/*.ms')
        if (len(ignore) > 0):
            tvis = glob.glob(dir+'/*%s*'%(ignore))
            vis = list(set(vis) - set(tvis))
        if (len(vis) < 2):
            print("    Only %d measurement set found" % (len(vis)))
            continue
        spws = getScienceSpws(vis[0], intent, tdm=tdm, fdm=fdm)
        print("%s: %s" % (os.path.basename(vis[0]),spws))
        difference = False
        for v in vis[1:]:
            newspws = getScienceSpws(v, intent, tdm=tdm, fdm=fdm)
            print("%s: %s" % (os.path.basename(v),newspws))
            if spws != newspws:
                difference = True
                print("   Difference!")
    return difference

def getNearestScienceSpw(vis, freqGHz, mymsmd=''):
    """
    Finds the science spw whose mean freq is closest to the specified
    frequency.  - Todd Hunter
    """
    needToClose = False
    if mymsmd == '':
        mymsmd = createCasaTool(msmdtool)
        needToClose = True
    spws = getScienceSpws(vis, mymsmd=mymsmd, returnString=False)
    freqs = []
    for spw in spws:
        freqs.append(mymsmd.meanfreq(spw)*1e-9)
    spw = spws[np.argmin(np.abs(freqs-freqGHz))]
    if needToClose:
        mymsmd.close()
    return spw

def getScienceTimeRanges(vis, field=None, intent='OBSERVE_TARGET*', 
                         mymsmd=None, roundToSmallestRange=False):
    """
    Gets a list of string-formatted time ranges corresponding to the
    scans on the specified field or intent.  Similar to tt.scanTimeRanges.
    roundToSmallestRange: if True, then ceil start and floor end
                            else, then floor start and ceil end
    Returns:  ['2017/10/07/22:18:41~2017/10/07/22:20:07', ...]
    -Todd Hunter
    """
    needToClose = False
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    scans = getScienceScans(vis, field, intent, mymsmd)
    times = []
    for scan in scans:
        t = mymsmd.timesforscan(scan)
        if roundToSmallestRange:
            times.append(mjdsecToTimerange(np.ceil(np.min(t)), np.floor(np.max(t))))
        else:
            times.append(mjdsecToTimerange(np.floor(np.min(t)),np.ceil(np.max(t))))
    return times

def getNonScienceTimeRanges(vis, intent='OBSERVE_TARGET*', obsID=0, tbuffer=0.048):
    """
    Returns a list of timeranges (suitable for flagdata commands) which
    correspond to all times that are not contained by a scan of the
    specified intent. The opposite of this function is getScienceTimeRanges.
    Coped from to tt.nonScanTimeRanges.
    tbuffer: in seconds, flag this much *less* time to avoid flagging good data
           default value is one ALMA timing event
    """
    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)
    scans = mymsmd.scansforintent(intent)
    print("Found %d matching scans: " % (len(scans)), scans)
    startTime = 86400*mymsmd.timerangeforobs(obsID)['begin']['m0']['value']
    endTime = 86400*mymsmd.timerangeforobs(obsID)['end']['m0']['value']
    timeranges = []
    for scan in scans:
        t = mymsmd.timesforscan(scan)
        if scan == scans[0]:  # first scan, start from beginning of obs.
            timeranges.append(mjdsecToTimerange(startTime, np.min(t)-tbuffer))
            endOfPreviousGoodScan = np.max(t) + tbuffer
        elif scan == scans[-1]:  # final scan: finish and then go to end of obs.
            timeranges.append(mjdsecToTimerange(endOfPreviousGoodScan, np.min(t)-tbuffer))
            timeranges.append(mjdsecToTimerange(np.max(t)+tbuffer, endTime))
        else:
            timeranges.append(mjdsecToTimerange(endOfPreviousGoodScan, np.min(t)-tbuffer))
            endOfPreviousGoodScan = np.max(t) + tbuffer
    mymsmd.close()
    return timeranges

def getPointingScans(vis, field=None, mymsmd=None, debug=False):
    """
    Calls getScienceScans using the intent='CALIBRATE_POINTING#ON_SOURCE'
    field: one name or ID
    -Todd Hunter
    """
    return getScienceScans(vis, field, 'CALIBRATE_POINTING#ON_SOURCE', mymsmd, debug)

def getScienceScans(vis, field=None, intent='OBSERVE_TARGET*', 
                    mymsmd=None, debug=False):
    """
    Return a list of scans with the specified intent (and optionally limited 
    to one field).  
    field: one name or integer ID
    intent: either the full name ('CALIBRATE_PHASE#ON_SOURCE', or partial name with wildcard ('*PHAS*'), 
           or a complete abbreviated key like 'PHASE'
    -Todd Hunter
    """
    needToClose = False
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    intents = mymsmd.intents()
    if (intent not in intents):
        for i in intents:
            if i.find(intent) >= 0:
                intent = i
                print("Translated intent to ", i)
                break
    # minimum match OBSERVE_TARGET or OBSERVE_TARGET* to OBSERVE_TARGET#UNSPECIFIED
    value = [i.find(intent.replace('*','')) for i in intents]
    # If any intent gives a match, the mean value of the location list will be > -1
    if np.mean(value) == -1:
        print("%s not found in this dataset. Available intents: " % (intent), intents)
        if needToClose: 
            mymsmd.close()
        return
    sciscans = mymsmd.scansforintent(intent)
    scispws = mymsmd.spwsforintent(intent)
    if debug:
        print("sciscans = ", sciscans)
        print("scispws = ", scispws)
    if field is not None:
        ids, names = parseFieldArgument(vis, field, mymsmd=mymsmd)
        fieldscans = mymsmd.scansforfields()
        spwscans = mymsmd.scansforspws()
        scispwscans = spwscans[str(scispws[0])]
        if debug:
            print("scispwscans = ", scispwscans)
        sciscans2 = []
        for field in [str(i) for i in ids]:
            if debug:
                print("scansforfield%s = " % field, fieldscans[field])
            sciscans2 += list(np.intersect1d(scispwscans,fieldscans[field]))
        sciscans2 = np.unique(sciscans2)
        if debug:
            print("sciscans2 = ", sciscans2)
        sciscans = np.intersect1d(sciscans,sciscans2)
    if needToClose:
        mymsmd.close()
    return sciscans

def getScienceSpwTypes(vis, mymsmd=None):
    """
    Returns a dictionary giving the number of 'TDM' and 'FDM' spws.
    -Todd Hunter
    """
    if (mymsmd is None):
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        needToClose = True
    else:
        needToClose = False
    tdm = len(getScienceSpws(vis, tdm=True, fdm=False, mymsmd=mymsmd, returnString=False))
    fdm = len(getScienceSpws(vis, tdm=False, fdm=True, mymsmd=mymsmd, returnString=False))
    if needToClose: mymsmd.close()
    return {'TDM': tdm, 'FDM': fdm}

def getScienceSpwChanwidths(vis, intent='OBSERVE_TARGET#ON_SOURCE', 
                            tdm=True, fdm=True, mymsmd=None, sqld=False, 
                            verbose=False, returnDict=False, returnMHz=False,
                            absoluteValue=True):
    """
    Returns: an array of channel widths (in Hz) in order sorted by spw ID
    returnDict: if True, then return a dictionary keyed by spw ID
    absoluteValue: if True, take the abs() of the bandwidth
    -Todd Hunter
    """
    if mymsmd is None:
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        needToClose = True
    else:
        needToClose = False
    spws = sorted(getScienceSpws(vis, intent, False, False, tdm, fdm, mymsmd, sqld, verbose))
    bandwidths = []
    for spw in spws:
        bandwidths.append(mymsmd.chanwidths(spw)[0])
    if needToClose:
        mymsmd.close()
    mymsmd.close()
    bandwidths = np.array(bandwidths)
    if returnMHz:
        bandwidths *= 1e-6
    if absoluteValue:
        bandwidths = np.abs(bandwidths)
    if returnDict:
        mydict = {}
        for i, spw in enumerate(spws):
            mydict[spw] = bandwidths[i]
        return mydict
    else:
        return bandwidths

def getScienceSpwBandwidths(vis, intent='OBSERVE_TARGET#ON_SOURCE', 
                             tdm=True, fdm=True, mymsmd=None, sqld=False, 
                             verbose=False, returnDict=False, returnMHz=False):
    """
    Returns: an array of bandwidths (in Hz) in order sorted by spw ID
    returnDict: if True, then return a dictionary keyed by spw ID
    -Todd Hunter
    """
    if mymsmd is None:
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        needToClose = True
    else:
        needToClose = False
    spws = sorted(getScienceSpws(vis, intent, False, False, tdm, fdm, mymsmd, sqld, verbose))
    bandwidths = mymsmd.bandwidths(spws)
    if needToClose:
        mymsmd.close()
    mymsmd.close()
    if returnMHz:
        bandwidths *= 1e-6
    if returnDict:
        mydict = {}
        for i, spw in enumerate(spws):
            mydict[spw] = bandwidths[i]
        return mydict
    else:
        return bandwidths

def getScienceSpwNChannels(vis, intent='OBSERVE_TARGET#ON_SOURCE', 
                           tdm=True, fdm=True, mymsmd=None, sqld=False, 
                           verbose=False, returnDict=False, includeBandwidth=False):
    """
    Returns: an array of nchan in order sorted by spw ID
    returnDict: if True, then return a dictionary keyed by spw ID
    includeBandwidth: if True, then sets returnDict=True and returns 2 keys per spw:
              'nchan' and 'bandwidth'
    -Todd Hunter
    """
    if mymsmd is None:
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        needToClose = True
    else:
        needToClose = False
    spws = sorted(getScienceSpws(vis, intent, False, False, tdm, fdm, mymsmd, sqld, verbose))
    nchan = []
    bandwidth = []
    for spw in spws:
        nchan.append(mymsmd.nchan(spw))
        if includeBandwidth:
            bandwidth.append(mymsmd.bandwidths(spw))
    if needToClose:
        mymsmd.close()
    mymsmd.close()
    if includeBandwidth:
        returnDict = True
    if returnDict:
        mydict = {}
        for i, spw in enumerate(spws):
            if includeBandwidth:
                mydict[spw] = {'nchan': nchan[i], 'bandwidth': bandwidth[i]}
            else:
                mydict[spw] = nchan[i]
        return mydict
    else:
        return nchan

def getTsysSpwsForScan(vis, scan, intent='CALIBRATE_ATMOSPHERE#OFF_SOURCE', 
                       tdm=True, fdm=True, mymsmd=None, sqld=False, 
                       verbose=False):
    """
    A wrapper for getScienceSpwsForScan with intent set to ATMOSPHERE
    -Todd Hunter
    """
    spws = getScienceSpwsForScan(vis, scan, intent, tdm, fdm, mymsmd, sqld, verbose)
    return spws

def getScienceSpwsForScan(vis, scan, intent='OBSERVE_TARGET#ON_SOURCE', 
                          tdm=True, fdm=True, mymsmd=None, sqld=False, 
                          verbose=False):
    """
    Takes the intersection of msmd.spwsforscan with getScienceSpws()
    Returns: list of integer IDs
    -Todd Hunter
    """
    needToClose = False
    if mymsmd is None:
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        needToClose = True
    scanSpws = mymsmd.spwsforscan(scan)
    scienceSpws = getScienceSpws(vis, intent, tdm=tdm, fdm=fdm, mymsmd=mymsmd, sqld=sqld, verbose=verbose, returnString=False)
    if verbose:
        print("scan spws: ", scanSpws)
        print("intent spws: ", scienceSpws)
    if needToClose:
        mymsmd.close()
    return np.intersect1d(scanSpws, scienceSpws)

def getOnlineFlagsForScienceSpws(vis, spwnames=[], verbose=False):
    """
    Searches the pipeline-generated *flagonline.txt file for any flag that
    is applied to a subset of science spws, but not all of them.
    It first finds the full name of the science spws.
    vis:  a measurement set name which has a flagonline.txt file in the same parent directory
    spwnames: if you already know the names, you can pass them as a list
    verbose: if True, then print all flags that include any science spw
             if False, then only print flags that have a subset of science spws
    """
    if not os.path.exists(vis):
        print("Could not find vis: ", vis)
        return
    flagfile = vis.replace('.ms','.flagonline.txt').replace('_target','')
    if len(spwnames) == 0:
        spwnames = getScienceSpwNames(vis)
    f = open(flagfile,'r')
    flags = f.readlines()
    f.close()
    partialFlags = 0  # that flag a subset of the science spws
    spwFlags = 0 # that flag any or all science spws
    if verbose:
        f = open('scienceSpwFlags.txt','w')
    for flag in flags:
        spws = 0
        for name in spwnames:
            if flag.find(name) >= 0:
                spws += 1
        if spws > 0:
            spwFlags += 1
            if verbose:
                print(flag)
                f.write(flag)
            if spws < len(spwnames):
                partialFlags += 1
                print("subset flag: ", flag)
    print("Found %d flags that include science spws" % (spwFlags))
    print("Found %d flags where only a subset of science spws are flagged" % (partialFlags))
    if verbose:
        f.close()
    return partialFlags

def getSpectralSpecNames(vis, intent='OBSERVE_TARGET#ON_SOURCE', tdm=True,
                         fdm=True, mymsmd=None, sqld=False):
    """
    Gets a list of unique prefixes of the spw names, i.e the SpectralSpec names
    by calling getScienceSpwNames(uniqueStringOnly=True)
    intent: either full intent name including #subIntent, or an abbreviated key, like 'PHASE', or blank (for all intents)
    -Todd Hunter
    """
    return np.unique(getScienceSpwNames(vis, intent, tdm=tdm, fdm=fdm, mymsmd=mymsmd, sqld=sqld, uniqueStringOnly=True))

def getScienceSpwNames(vis, intent='OBSERVE_TARGET#ON_SOURCE', 
                       returnString=True, returnListOfStrings=False,
                       tdm=True, fdm=True, mymsmd=None, sqld=False, 
                       verbose=False, uniqueStringOnly=False, returnDict=False):
    """
    intent: either full intent name including #subIntent, or an abbreviated key, like 'PHASE', or blank (to get all intents except WVR spws)
    sqld: setting this true will give BB_1,BB+2, etc. for old datasets (pre Cycle 7)
    Returns: a list of science spw names using getScienceSpws and 
    msmd.namesforspw.
    uniqueStringOnly: if True, then return only the leading string of gibberish 
             that originates in OT
    -Todd Hunter
    """
    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)
    spwlist = getScienceSpws(vis, intent, False, False, tdm, fdm, mymsmd, 
                             sqld, verbose)
    names = mymsmd.namesforspws(spwlist)
    mymsmd.close()
    mydict = {}
    for i,name in enumerate(names):
        mydict[spwlist[i]] = name
    if uniqueStringOnly:
        newnames = []
        mydict = {}
        for i,name in enumerate(names):
            newnames.append(name.split('#')[0])
            mydict[spwlist[i]] = name.split('#')[0]
        names = newnames
    if returnDict:
        return mydict
    else:
        return names

def getScienceSpwFreqRanges(vis, mymsmd=None):
    """
    Calls getScienceSpws with returnFreqRanges=True
    Returns: a dictionary keyed by spw ID, with ranges of TOPO frequency
    """
    mydict = getScienceSpws(vis, returnFreqRanges=True, mymsmd=mymsmd)
    return mydict 

def getScienceSpws(vis, intent='OBSERVE_TARGET#ON_SOURCE', returnString=True, 
                   returnListOfStrings=False, tdm=True, fdm=True, mymsmd=None, 
                   sqld=False, verbose=False, returnFreqRanges=False):
    """
    Return a list of spws with the specified intent.  For ALMA data,
    it ignores channel-averaged and SQLD spws.
    intent: either full intent name including #subIntent, or an abbreviated key, like 'PHASE'
    returnString: if True, return '1,2,3'
                  if False, return [1,2,3]
    returnListOfStrings: if True, return ['1','2','3']  (amenable to tclean spw parameter)
                         if False, return [1,2,3]
    returnFreqRanges: if True, returns a dictionary keyed by spw ID, with values
          equal to the frequency of the middle of the min and max channel (Hz)
    -- Todd Hunter
    """
    if returnString and returnListOfStrings:
        print("You can only specify one of: returnString, returnListOfStrings")
        return
    needToClose = False
    if (mymsmd is None):
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
        needToClose = True
    allIntents = mymsmd.intents()
    if (intent not in allIntents and intent != ''):
        for i in allIntents:
            if i.find(intent) >= 0:
                intent = i
                print("Translated intent to ", i)
                break
    # minimum match OBSERVE_TARGET to OBSERVE_TARGET#UNSPECIFIED
    value = [i.find(intent.replace('*','')) for i in allIntents]
    # If any intent gives a match, the mean value of the location list will be > -1
    if np.mean(value) == -1 and intent != '':
        print("%s not found in this dataset. Available intents: " % (intent), allIntents)
        if needToClose: 
            mymsmd.close()
        return
    if intent == '':
        spws = mymsmd.spwsforintent('*')
    else:
        spws = mymsmd.spwsforintent(intent)
    if (getObservatoryName(vis).find('ALMA') >= 0 or getObservatoryName(vis).find('OSF') >= 0):
        almaspws = mymsmd.almaspws(tdm=tdm,fdm=fdm,sqld=sqld)
        if (len(spws) == 0 or len(almaspws) == 0):
            scienceSpws = []
        else:
            scienceSpws = np.intersect1d(spws,almaspws)
    else:
        scienceSpws = spws
    mydict = {}
    for spw in scienceSpws:
        mydict[spw] = sorted([mymsmd.chanfreqs(spw)[0],mymsmd.chanfreqs(spw)[-1]])
    if needToClose:
        mymsmd.close()
    if returnFreqRanges:
        return mydict
    if returnString:
        return(','.join(str(i) for i in scienceSpws))
    elif returnListOfStrings:
        return list([str(i) for i in scienceSpws])
    else:
        return(list(scienceSpws))

def getScienceTargetFieldIDsFromASDM(asdm, n=-1, returnDict=False):
    """
    Returns an array of the integer field IDs that correspond to the
    nth science target.  If n < 0,  return all field IDs
    """
    names = np.array(getScienceTargetsFromASDM(asdm))
    if n >= len(names):
        print("There are only %d science target names in this ASDM." % (len(names)))
        return
    mydict, ignore = getFieldsFromASDM(asdm)
    # mydict has keys of field IDs and values of field names
    targetdict = {}
    if n < 0:
        scienceNames = np.intersect1d(list(mydict.values()), names)
        fieldIDs = []
        stringValues = np.array([str(i) for i in list(mydict.values())]) # convert np.string to str
        for name in scienceNames:
            idx = np.where(stringValues == name)[0]
            fieldIDs += list(np.array(list(mydict.keys()))[idx])
            targetdict[name] = sorted(list(np.array(list(mydict.keys()))[idx]))
    else:
        idx = np.where(np.array(list(mydict.values())) == names[n])
        fieldIDs = np.array(list(mydict.keys()))[idx]
        targetdict[names[n]] = sorted(list(fieldIDs))
    if returnDict:
        return targetdict
    else:
        return sorted(fieldIDs)

def getScienceTargetRADecsFromASDM(asdm, sexagesimal=False, returndict=False):
    """
    Returns: list of RA, Dec tuples in radians
    sexagesimal: if True, then a list of strings
    returndict: if True, then return dictionary keyed by field ID
    """
    fieldIDs = getScienceTargetFieldIDsFromASDM(asdm)
    print("science field IDs: ", fieldIDs)
    radec = []
    mydict = {}
    for fieldID in fieldIDs:
        r = getRADecForFieldFromASDM(asdm, fieldID)
        if sexagesimal:
            r = rad2radec(r, verbose=False)
        radec.append(r)
        mydict[fieldID] = r
    if returndict:
        radec = mydict
    return radec

def getScienceTargetsFromASDM(asdm):
    """
    Get a list of unique names of the science targets from an ASDM
    -Todd Hunter
    """
    mydict = readscans(asdm)[0]
    targets = []
    for key in list(mydict.keys()):
        if (mydict[key]['intent'].find('OBSERVE_TARGET') >= 0):
            targets.append(mydict[key]['source'])
    return np.unique(targets)

def getScienceTargets(vis, intent='OBSERVE_TARGET*', mymsmd=None, returnNames=True, checkSpw=True, unique=False):
    """
    Get a list of names of the science targets from a measurement set (using msmd).
    checkSpw: if True, confirm that field name was also observed in first science spw
    unique: if True, then return unique list of names, and a corresponding list of first field ID
            of this name that was observed with the specified intent
    -Todd Hunter
    """
    needToClose = False
    if mymsmd is None:
        if not os.path.exists(vis):
            print("Could not open ms: %s" % (vis))
            return
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    ids = mymsmd.fieldsforintent(intent)
    if checkSpw:
        spws = mymsmd.spwsforintent(intent)
        fieldIDs_spw = mymsmd.fieldsforspw(spws[0])
        ids = np.intersect1d(ids,fieldIDs_spw)
    if len(ids) == 0: 
        fields = []
    else:
        fields = mymsmd.namesforfields(ids)
    if unique:
        fields = np.unique(fields)
        firstFieldIDs = []
        for field in fields:
            firstField = sorted(np.intersect1d(mymsmd.fieldsforintent(intent),mymsmd.fieldsforname(field)))[0]
            firstFieldIDs.append(firstField)
    if needToClose:
        mymsmd.close()
    if returnNames:
        if unique:
            return fields, firstFieldIDs
        else:
            return fields
    else:
        return ids

def getCalibrators(vis, intent=['BANDPASS','FLUX','PHASE','AMP','POLARIZATION','CHECK_SOURCE'],
                   mymsmd=None, returnNames=True, returnDict=False, returnListForSingleCalibrator=False,
                   invertDictionary=False):
    """
    Get a list of names of the calibrators from a measurement set (using msmd)
    intent: can be a comma-delimited string or a python list of strings
    invertDictionary: if True, then make the intents be the values and the name or ID as the key
    Returns: 
    a list of names or integers (if returndDict is False), else
    a dictionary keyed by intent with <value> equal to field name or integer ID
         <value> will be a list if it is longer than one name or integer
    """
    needToClose = False
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    ids = []
    intents = mymsmd.intents()
    if type(intent) == str:
        intent = intent.split(',')
    mydict = {}
    for i in intent:
        if 'CALIBRATE_' + i.replace('*','') + '#ON_SOURCE' in intents or 'OBSERVE_' + i.replace('*','') + '#ON_SOURCE' in intents:
            newlist = list(mymsmd.fieldsforintent('*' + i + '*'))
            ids += newlist
            if len(newlist) == 1 and not returnListForSingleCalibrator:
                mydict[i] = newlist[0]
            else:
                mydict[i] = newlist
    if len(ids) == 0: 
        fields = []
    else:
        fields = mymsmd.namesforfields(ids)
    for key in mydict:
        if returnNames:
            mydict[key] = mymsmd.namesforfields(mydict[key])
            if len(mydict[key]) == 1 and not returnListForSingleCalibrator:
                mydict[key] = mydict[key][0]
    if needToClose:
        mymsmd.close()
    if returnDict:
        if invertDictionary:
            newdict = {}
            for key in mydict.keys():
                newdict[mydict[key]] = key
            mydict = newdict
        return mydict
    elif returnNames:
        return fields
    else:
        return ids

def getWeightedMeanScienceFrequency(vis):
    """
    Computes the mean frequency of all science spws, weighted by the mean data-weigths of each spw
    vis: single or multiple measurement sets, as a python list of strings, or comma-delimited string
    -Todd Hunter
    """
    weights = []
    meanfreqs = []
    if type(vis) != list:
        if vis.find('*') >= 0:
            vislist = glob.glob(vis)
        else:
            vislist = vis.split(',')
    else:
        vislist = vis
    for vis in vislist:
        if not os.path.exists(vis):
            print("Could not find measurement set.")
            return
        print("Processing vis: ", os.path.basename(vis))
        mymsmd = msmdtool()
        mymsmd.open(vis)
        freqs = getScienceFrequencies(vis, returnDict=True, mymsmd=mymsmd)
        for spw in list(freqs.keys()):
            meanfreqs.append(freqs[spw])
            weights.append(getMeanWeights(vis, spw=spw, mymsmd=mymsmd))
        mymsmd.close()
    return np.average(meanfreqs, weights=weights)

def getExtremeSpws(vis, intent='OBSERVE_TARGET#ON_SOURCE', verbose=False, mymsmd=None, spw=''):
    """
    Returns the spws IDs containing the minimum and maximum frequencies for the specified intent.
    spw: if specified, then use that spw or list of spws (comma-delimited string or int list)
         if not specified, then use all with the specified intent
         If none have that intent, then use all the spws with more than 4 channels
         i.e. not the WVR data nor the channel-averaged data.  
    -- Todd Hunter
    """
    minfreqs = []
    maxfreqs = []
    if (casaVersion >= casaVersionWithMSMD):
        needToClose = False
        if mymsmd is None or mymsmd == '':
            needToClose = True
            mymsmd = createCasaTool(msmdtool)
            mymsmd.open(vis)
        if (intent not in mymsmd.intents()):
            if intent.split('#')[0] in [i.split('#')[0] for i in mymsmd.intents()]:
                # VLA uses OBSERVE_TARGET#UNSPECIFIED
                intent = intent.split('#')[0]+'*'
            else:
                print("Intent %s not in dataset (nor is %s*)." % (intent,intent.split('#')[0]))

        if (spw == ''):
            argument = [i.find(intent.replace('*','')) for i in mymsmd.intents()]
            if len(argument) == 0:
                spws = []
            elif (np.max(argument) == -1):
                spws = []
            else:
                spws = mymsmd.spwsforintent(intent)
        elif (type(spw) == str):
            spws = [int(i) for i in spw.split(',')]
        elif (type(spw) != list):  
            # i.e. integer
            spws = [spw]
        else:
            spws = spw
        if verbose:
            print("Using spws: ", spws)
        if (getObservatoryName(vis).find('ALMA') >= 0 or getObservatoryName(vis).find('OSF') >= 0):
            almaspws = mymsmd.almaspws(tdm=True,fdm=True)
            if (len(spws) == 0):
                nonwvrspws = getNonWvrSpws(mymsmd)
                for spw in nonwvrspws:
                    minfreqs.append(np.min(mymsmd.chanfreqs(spw)))
                    maxfreqs.append(np.max(mymsmd.chanfreqs(spw)))
            else:
                spws = np.intersect1d(spws,almaspws)
                for spw in spws:
                    minfreqs.append(np.min(mymsmd.chanfreqs(spw)))
                    maxfreqs.append(np.max(mymsmd.chanfreqs(spw)))
        else:
            for spw in spws:
                minfreqs.append(np.min(mymsmd.chanfreqs(spw)))
                maxfreqs.append(np.max(mymsmd.chanfreqs(spw)))
        if needToClose:
            mymsmd.close()
    else:
        mytb = createCasaTool(tbtool)
        try:
            mytb.open("%s/SPECTRAL_WINDOW" % vis)
        except:
            print("Could not open ms table = %s" % (vis))
            return(freqs)
        numChan = mytb.getcol("NUM_CHAN")
        for i in range(len(numChan)):
            if (numChan[i] > 4):
                chanFreq = mytb.getcell("CHAN_FREQ",i)
                minfreqs.append(np.min(chanFreq))
                maxfreqs.append(np.max(chanFreq))
        mytb.close()
    minspw = np.argmin(minfreqs)
    maxspw = np.argmax(maxfreqs)
    return spws[minspw], spws[maxspw]

def getScienceFrequencies(vis, spw='', intent='OBSERVE_TARGET#ON_SOURCE', verbose=False, mymsmd=None, 
                          returnDict=False, returnGHz=False):
    """
    Return a list of the mean frequencies (in Hz) of each spw 
    returnGHz: if True, return values in units of GHz
    spw: if specified, then use that spw or list of spws (comma-delimited string or int list)
         if not specified, then use all with the specified intent
         If none have that intent, then use all the spws with more than 4 channels
         i.e. not the WVR data nor the channel-averaged data.  
    returnDict: if True, return a dictionary keyed by spw ID
    -- Todd Hunter
    """
    freqs = []
    mydict = {}
    if (casaVersion >= casaVersionWithMSMD):
        needToClose = False
        if mymsmd is None or mymsmd == '':
            needToClose = True
            mymsmd = createCasaTool(msmdtool)
            mymsmd.open(vis)
        if (intent not in mymsmd.intents()) and intent != '':
            if intent.split('#')[0] in [i.split('#')[0] for i in mymsmd.intents()]:
                # VLA uses OBSERVE_TARGET#UNSPECIFIED
                intent = intent.split('#')[0]+'*'
            else:
                print("Intent %s not in dataset (nor is %s*)." % (intent,intent.split('#')[0]))

        if (spw == ''):
            argument = [i.find(intent.replace('*','')) for i in mymsmd.intents()]
            if len(argument) == 0:
                spws = []
            elif (np.max(argument) == -1):
                spws = []
            else:
                spws = mymsmd.spwsforintent(intent)
        elif (type(spw) == str):
            spws = [int(i) for i in spw.split(',')]
        elif (type(spw) != list):  
            # i.e. integer
            spws = [spw]
        else:
            spws = spw
        if len(spws) == 0: 
            print("using all spws")
            spws = range(mymsmd.nspw())
        if verbose:
            print("Using spws: ", spws)
        if (getObservatoryName(vis).find('ALMA') >= 0 or getObservatoryName(vis).find('OSF') >= 0):
            almaspws = mymsmd.almaspws(tdm=True,fdm=True)
            if (len(spws) == 0):
                nonwvrspws = getNonWvrSpws(mymsmd)
                for spw in nonwvrspws:
                    freqs.append(mymsmd.meanfreq(spw))
                    mydict[spw] = mymsmd.meanfreq(spw)
            else:
                spws = np.intersect1d(spws,almaspws)
                for spw in spws:
                    freqs.append(mymsmd.meanfreq(spw))
                    mydict[spw] = mymsmd.meanfreq(spw)
        else:
            for spw in spws:
                freqs.append(mymsmd.meanfreq(spw))
                mydict[spw] = mymsmd.meanfreq(spw)
        if returnGHz:
            freqs = list(np.array(freqs)*1e-9)
            for spw in spws:
                mydict[spw] *= 1e-9
        if needToClose:
            mymsmd.close()
        if returnDict:
            return mydict
        else:
            return freqs
    else:
        mytb = createCasaTool(tbtool)
        try:
            mytb.open("%s/SPECTRAL_WINDOW" % vis)
        except:
            print("Could not open ms table = %s" % (vis))
            return(freqs)
        numChan = mytb.getcol("NUM_CHAN")
        for i in range(len(numChan)):
            if (numChan[i] > 4):
                chanFreq = mytb.getcell("CHAN_FREQ",i)
                freqs.append(np.mean(chanFreq))
        mytb.close()
        if returnGHz:
            freqs = list(np.array(freqs)*1e-9)
        return freqs
    
def listConditionsFromASDM(asdm, station=1, verbose=True):
    """
    This function extracts the weather conditions for the specified ASDM,
    and computes and returns a dictionary containing the median values.
    The default weather station to use is 1.
    For further help and examples, see https://safe.nrao.edu/wiki/bin/view/ALMA/ListConditionsFromASDM
    Todd Hunter
    """
    [conditions, medianConditions, stationName] = getWeatherFromASDM(asdm,station=station)
    if (verbose):
        print("Median weather values for %s to %s" % (plotbp3.utstring(conditions[0][0]),plotbp3.utstring(conditions[0][-1])))
        print("  Pressure = %.2f mb" % (medianConditions['pressure']))
        print("  Temperature = %.2f C" % (medianConditions['temperature']))
#        print "  Dew point = %.2f C" % (medianConditions['dewpoint'])
        print("  Relative Humidity = %.2f %%" % (medianConditions['humidity']))
        print("  Wind speed = %.2f m/s" % (medianConditions['windSpeed']))
        print("  Wind max = %.2f m/s" % (np.max(conditions[6])))
        print("  Wind direction = %.2f deg" % (medianConditions['windDirection']))
    return(medianConditions)

def listconditions(vis='', scan='', antenna='0',verbose=True,asdm='',reffreq=0,
                   byscan=False, vm=0, mymsmd='',field='',debug=False, getSolarDirection=True):
    """
    Compiles the mean weather, pwv and opacity values for the given scan
    number or scan list for the specified ms.  If a scan number
    is not provided it returns the average over all science spws in whole ms.
    byscan: setting this True will return a dictionary with conditions per scan.
    field: setting this will run msmd.scansforfield, and use only those scans.
    reffreq: value in GHz (use this to restrict to one spw)
    Note: In casa 4.4 and 4.5, when run on concatenated measurement sets, 
          only the scans from the first obsid are detected due to a change 
          in msmd, a behavior which was fixed in casa 4.6.
    Scan can be a single list: [1,2,3] or '1,2,3' or a single range: '1~4'.
    For further help and examples, see https://safe.nrao.edu/wiki/bin/view/ALMA/Listconditions
    Todd Hunter
    """
    if (os.path.exists(vis)==False):
        print("Could not find the ms = %s." % (vis))
        return
    if (os.path.exists(vis+'/table.dat') == False):
        print("No table.dat.  This does not appear to be an ms.")
        return
    if (type(scan) == str):
        if (scan.find(',')>0):
            scan = [int(k) for k in scan.split(',')]
        elif (scan.find('~')>0):
            scan = range(int(scan.split('~')[0]),int(scan.split('~')[1])+1)
    if (casaVersion >= casaVersionWithMSMD):
        if mymsmd == '':
            mymsmd = createCasaTool(msmdtool)
            mymsmd.open(vis)
            needToClose = True
        else:
            needToClose = False
        if (field != ''):
            scan = mymsmd.scansforfield(int(field))
            print("Will use scans: ", str(scan))
    else:
        if (field != ''):
            print("The field parameter is not supported for this old of a CASA.")
            return
    if (reffreq==0):
        freqs = 1e-9*np.array(getScienceFrequencies(vis, mymsmd=mymsmd))
    else:
        freqs = [reffreq]
    if (byscan and (type(scan) == list or scan=='')):
        conditions = {}
        if (scan == ''):
            if (casaVersion >= casaVersionWithMSMD):
                scan = mymsmd.scannumbers()
            else:
                if (vm == 0):
                    vm = ValueMapping(vis)
                scan = np.unique(vm.scans)
            print("Scans = ", scan)
        pwvstd = 1 # get into the loop the first time, but do not repeat if no WVR file present
        for i in scan:
            if debug:
                print("Calling getWeather('%s',%d,'%s',%s)" % (vis,i,antenna,verbose))
            [cond,myTimes,vm] = getWeather(vis,i,antenna,verbose,vm,mymsmd,getSolarDirection=getSolarDirection)
            if (len(myTimes) > 0 and pwvstd>0):
                [pwv,pwvstd] = getMedianPWV(vis,myTimes,asdm,verbose=False)
                if (pwvstd > 0):
                    if (pwv > 0):
                        tau = []
                        zenithtau = []
                        for myfreq in range(len(freqs)):
                            reffreq = freqs[myfreq]
                            [z,t] = estimateOpacity(pwv, reffreq, cond, verbose)
                            zenithtau.append(z)
                            tau.append(t)
                        d2 = {}
                        d2['tauzenith'] = np.mean(zenithtau)
                        d2['tau'] = np.mean(tau)
                        d2['transmissionzenith'] = np.exp(-np.mean(zenithtau))
                        d2['transmission'] = np.exp(-np.mean(tau))
                        d2['pwv'] = pwv
                        d2['pwvstd'] = pwvstd
                        cond = dict(list(cond.items()) + list(d2.items()))
            conditions[i] = cond
    else:
        if debug:
            print("Calling getWeather('%s','%s','%s',%s), len(freqs)=%d" % (vis,str(scan),antenna,verbose,len(freqs)))
        [conditions,myTimes,vm] = getWeather(vis,scan,antenna,verbose,vm,mymsmd,getSolarDirection=getSolarDirection)
        if debug:
            print("len(myTimes) = %d" % (len(myTimes)))
        if (len(myTimes) < 1):
            return(conditions)
        [pwv,pwvstd] = getMedianPWV(vis,myTimes,asdm,verbose=False)
        if debug:
            print("pwv = %f" % (pwv))
        if (pwvstd < 0):
            return(conditions)
        if (pwv > 0):
            tau = []
            zenithtau = []
            for i in range(len(freqs)):
                reffreq = freqs[i]
                [z,t] = estimateOpacity(pwv, reffreq, conditions,verbose)
                zenithtau.append(z)
                tau.append(t)
            d2 = {}
            d2['tauzenith'] = np.mean(zenithtau)
            d2['tau'] = np.mean(tau)
            d2['transmissionzenith'] = np.exp(-np.mean(zenithtau))
            d2['transmission'] = np.exp(-np.mean(tau))
            d2['pwv'] = pwv
            d2['pwvstd'] = pwvstd
            conditions = dict(list(conditions.items())+list(d2.items()))
    if needToClose:
        mymsmd.close()
    return(conditions)

def buildCalDataIdDictionary(vis, tol=10, debug=False, sbr=True, atm=True, mymsmd=None):
    """
    Provides the mapping from cal data ID to scan number via the times, as an alternative
    to using readCalData which requires the ASDM_CALDATA table.
    vis: string name of ms
    tol: tolerance in seconds (should not have to change this)
    sbr: include SIDEBAND_RATIO scans
    atm: include CALIBRATE_ATMOSPHERE scans
    """
    mytable = vis+'/ASDM_CALATMOSPHERE'
    mytb = createCasaTool(tbtool)
    mytb.open(mytable)
    startValidTime = mytb.getcol('startValidTime')
    calDataId = np.array([int(c.split('_')[-1]) for c in mytb.getcol('calDataId')])
    mytb.close()
    # these range from 2-40 seconds after the first integration of the scan
    # so do a rough correction for this
    startValidTime -= 10  
    needToClose = False
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    scans = mymsmd.scannumbers()
    times = mymsmd.timesforscans(scans)
    if ('CALIBRATE_ATMOSPHERE#ON_SOURCE' in mymsmd.intents()):
        calscans = mymsmd.scansforintent('CALIBRATE_ATMOSPHERE#ON_SOURCE')
    elif ('CALIBRATE_ATMOSPHERE#HOT' in mymsmd.intents()):   
        calscans = mymsmd.scansforintent('CALIBRATE_ATMOSPHERE#HOT')
    else:
        calscans = []
    if ('CALIBRATE_SIDEBAND_RATIO#ON_SOURCE' in mymsmd.intents()):
        sbrscans = mymsmd.scansforintent('CALIBRATE_SIDEBAND_RATIO#ON_SOURCE')
    else:
        sbrscans = []
    if (debug):
        print("%d calscans=%s,  %d sbrscans=%s" % (len(calscans),str(calscans), len(sbrscans), str(sbrscans)))
    if (sbr and atm):
        cal_sbr_scans = np.unique(np.union1d(calscans, sbrscans))
    elif (sbr):
        cal_sbr_scans = sbrscans
    else:
        cal_sbr_scans = calscans
        
    calscantimesdict = {}
    for i in range(len(calDataId)):
        cal = calDataId[i]
        if (cal not in list(calscantimesdict.keys())):
            calscantimesdict[cal] = []
        calscantimesdict[cal].append(startValidTime[i])
    for key in list(calscantimesdict.keys()):
        mylen = len(np.unique(calscantimesdict[key]))
        if (mylen > 1):
            print("Cal data ID %d has multiple startValidTimes (%d). Taking the minimum." % (key, mylen))
        calscantimesdict[key] = np.min(calscantimesdict[key])
        if (debug):
            print("%2d = %s" % (key, mjdsecToUTHMS(calscantimesdict[key])))
        
    calscandict = {}
    scansAssigned = []
    stopTol = 6.01*tol
    for tolerance in np.arange(tol, stopTol, tol):
        if (debug):
            print("tolerance = %.0f sec, calscandict=%s" % (tolerance,str(calscandict)))
        for i in range(len(calDataId)):
            cal = calDataId[i]
#            if (debug):
#                print "Working on row %d, calDataId=%d" % (i,cal)
            if (cal not in list(calscandict.keys())):
                calscandict[cal] = []
            if (tolerance == tol or len(calscandict[cal]) == 0):
                trialscans = mymsmd.scansfortimes(calscantimesdict[cal]+tolerance, tol=tolerance)
                for scan in sorted(list(trialscans)):
                    if (scan not in calscandict[cal] and
                        scan not in scansAssigned):
                        if (scan in cal_sbr_scans):
                            calscandict[cal].append(scan)
                            scansAssigned.append(scan)
                            if (debug):
                                print("tol=%.0f: calscandict[%d].append(%d) of %s (%s-%s)" % (tolerance,cal,scan,str(trialscans), mjdsecToUTHMS(calscantimesdict[cal]), mjdsecToUTHMS(calscantimesdict[cal]+tolerance*2)))
        # loosen the tolerance until all are filled with a value
        if (len(scansAssigned) == len(cal_sbr_scans)):
            allFilled = True
            for key in list(calscandict.keys()):
                if (len(calscandict[key]) == 0):
                    allFilled = False
            if (allFilled): break
        unassigned = np.setdiff1d(cal_sbr_scans, scansAssigned) 
        if (tolerance+tol < stopTol):
            scansAssigned = []
            calscandict = {}
        if ((debug or tolerance+tol > stopTol) and sbr and atm and len(unassigned)>0):
            print("not all scans assigned (%d vs. %d), unassigned = %s" % (len(scansAssigned), len(calscans)+len(sbrscans), str(unassigned)))
    csd = {}
    for key in list(calscandict.keys()):
        if (len(calscandict[key]) > 0):
            csd[key] = np.array(calscandict[key],np.int32)
    if needToClose:
        mymsmd.close()
    if (debug):
        print("returning from buildCalDataIdDictionary, %d/%d scans assigned" % (len(scansAssigned),len(cal_sbr_scans)))
    return(csd)
    
def readCalData(vis, calType='CAL_ATMOSPHERE'):
    """
    Provides the mapping from cal data ID to scan number
    Returns:
       a translation dictionary from calDataId to scan numbers
    Todd Hunter
    """
    mytb = createCasaTool(tbtool)
    if (os.path.exists(vis+'/ASDM_CALDATA') == False):
        print("No ASDM_CALDATA file.  Either importasdm(asis='CalData') or try readCalDataFromASDM instead.")
        return 
    mytb.open(vis+'/ASDM_CALDATA')
    calDataId = np.array([int(c.split('_')[-1]) for c in mytb.getcol('calDataId')])
    calibType = mytb.getcol('calType')
    if (calType not in calibType):
        print("%s is not a calibration type in this dataset.\nAvailable types = %s" % (calType,str(np.unique(calibType))))
        return
    scanSet = []
    for row in range(len(calibType)):
        scanSet.append(mytb.getcell('scanSet',row))
    mytb.close()
    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)
    if (calType == 'CAL_ATMOSPHERE'):
        calatmscans = mymsmd.scansforintent('CALIBRATE_ATMOSPHERE#ON_SOURCE')
    else:
        calatmscans = mymsmd.scansforintent(calType)
    mymsmd.close()
    mydict = {}
    for row in range(len(calibType)):
        cDI = calDataId[row]
        mydict[cDI] = scanSet[row]
    return(mydict)
    
def readCalDataFromASDM(sdmfile):
    """
    Builds a dictionary relating the calDataId to the scan number of an ASDM.
    Todd Hunter
    """
    if (os.path.exists(sdmfile) == False):
        print("readCalDataFromASDM(): Could not find file = ", sdmfile)
        return
    xmlscans = minidom.parse(sdmfile+'/CalData.xml')
    scandict = {}
    rowlist = xmlscans.getElementsByTagName("row")
    fid = 0
    for rownode in rowlist:
        scandict[fid] = {}
        rowscan = rownode.getElementsByTagName("scanSet")
        tokens = rowscan[0].childNodes[0].nodeValue.split()
        scan = int(tokens[2])

        rowcaldataid = rownode.getElementsByTagName("calDataId")
        caldataid = str(rowcaldataid[0].childNodes[0].nodeValue)
        scandict[fid]['calDataId'] = caldataid
        scandict[fid]['scan'] = scan
        fid +=1 
    return(scandict)

def readCalPointingTable(sdmfile):
    """
    Reads the CalPointing.xml table for the specified ASDM
    and returns a dictionary of values.
    Todd Hunter
    """
    if (not os.path.exists(sdmfile)):
        print("readCalPointingTable(): Could not find ASDM = ", sdmfile)
        return(None)
    if (not os.path.exists(sdmfile+'/CalPointing.xml')):
        print("readCalPointingTable(): Could not find %s/CalPointing.xml." % (sdmfile))
        return(None)
    xmlscans = minidom.parse(sdmfile+'/CalPointing.xml')
    scandict = {}
    rowlist = xmlscans.getElementsByTagName("row")
    fid = 0
    myqa = createCasaTool(qatool)
    for rownode in rowlist:
        scandict[fid] = {}
        rowAntennaName = rownode.getElementsByTagName("antennaName")
        antenna = str(rowAntennaName[0].childNodes[0].nodeValue)

        rowDirection = rownode.getElementsByTagName("direction")
#        for r in range(len(rowDirection)):
#            for c in range(len(rowDirection[r].childNodes)):
#                print "%d %d = " % (r,c), rowDirection[r].childNodes[c].nodeValue
        tokens = rowDirection[0].childNodes[0].nodeValue.split()
        azimuth = float(tokens[2])
        elevation = float(tokens[3])

        rowFrequency = rownode.getElementsByTagName("frequencyRange")
        tokens = rowFrequency[0].childNodes[0].nodeValue.split()
        frequency1 = float(tokens[2])
        frequency2 = float(tokens[3])
        frequency = 0.5*(frequency1+frequency2)

        rowRelative = rownode.getElementsByTagName("collOffsetRelative")
        tokens = rowRelative[0].childNodes[0].nodeValue.split()
        azOffset = float(tokens[3])
        elOffset = float(tokens[4])
        azOffset2 = float(tokens[5])
        elOffset2 = float(tokens[6])
        
        rowRelative = rownode.getElementsByTagName("collError")
        tokens = rowRelative[0].childNodes[0].nodeValue.split()
        azError = float(tokens[3])
        elError = float(tokens[4])
        azError2 = float(tokens[5])
        elError2 = float(tokens[6])
        
        rowCalDataId = rownode.getElementsByTagName("calDataId")
        calDataId = str(rowCalDataId[0].childNodes[0].nodeValue)
        scan = int(calDataId.split('_')[1])

        rowpol = rownode.getElementsByTagName("polarizationTypes")
        tokens = rowpol[0].childNodes[0].nodeValue.split()
        poltypes = []
        poltypes.append(str(tokens[2]))
        poltypes.append(str(tokens[3]))

        # start and end times in mjd ns
        rowstart = rownode.getElementsByTagName("startValidTime")
        start = int(rowstart[0].childNodes[0].nodeValue)/1000000000
        startmjd = start/86400.0
        t = myqa.quantity(startmjd,'d')
        starttime = call_qa_time(t,form="ymd",prec=8)
        rowend = rownode.getElementsByTagName("endValidTime")
        end = int(rowend[0].childNodes[0].nodeValue)
        endmjd = float(end)*1.0E-9/86400.0
        t = myqa.quantity(endmjd,'d')
        endtime = call_qa_time(t,form="ymd",prec=8)

        scandict[fid]['startValidTime'] = start
        scandict[fid]['endValidTime'] = end
        scandict[fid]['start'] = starttime
        scandict[fid]['end'] = endtime
        scandict[fid]['startmjd'] = startmjd
        scandict[fid]['endmjd'] = endmjd
        scandict[fid]['startmjdsec'] = startmjd*86400
        scandict[fid]['endmjdsec'] = endmjd*86400
        timestr = starttime+'~'+endtime
        scandict[fid]['azimuth'] = azimuth
        scandict[fid]['elevation'] = elevation
        scandict[fid]['antenna'] = antenna
        scandict[fid]['frequency'] = frequency
        scandict[fid]['azOffset'] = azOffset
        scandict[fid]['elOffset'] = elOffset
        scandict[fid]['azOffset2'] = azOffset2
        scandict[fid]['elOffset2'] = elOffset2
        scandict[fid]['azError'] = azError
        scandict[fid]['elError'] = elError
        scandict[fid]['azError2'] = azError2
        scandict[fid]['elError2'] = elError2
        scandict[fid]['scan'] = scan
        scandict[fid]['duration'] = (endmjd-startmjd)*86400
        scandict[fid]['polarizationTypes'] = poltypes
        fid += 1

    print('  Found ',rowlist.length,' rows in CalPointing.xml')
    myqa.done()
    # return the dictionary for later use
    return scandict
# end of readCalPointingTable(sdmfile):

def readCalPointing(asdm):
    """
    Calls readCalPointingTable() and converts the returned dictionary to a list
    of lists that is subsequently used by plotPointingResultsFromASDM().
    Todd Hunter
    """
    dict = readCalPointingTable(asdm)
    if (dict is None):
        return
    colOffsetRelative = []
    colError = []
    antennaName = []
    pols = []
    scans = []
    startValidTime = []
    azim = []
    elev = []
    frequency = []
    for entry in dict:
        colOffsetRelative.append([[dict[entry]['azOffset'],dict[entry]['elOffset']],[dict[entry]['azOffset2'],dict[entry]['elOffset2']]])
        antennaName.append(dict[entry]['antenna'])
        startValidTime.append(dict[entry]['startValidTime'])
        pols.append(dict[entry]['polarizationTypes'])
        scans.append(dict[entry]['scan'])
        azim.append(dict[entry]['azimuth'])
        elev.append(dict[entry]['elevation'])
        frequency.append(dict[entry]['frequency']*1e-9)
        colError.append([[dict[entry]['azError'], dict[entry]['elError']], [dict[entry]['azError2'], dict[entry]['elError2']]])
    return([ARCSEC_PER_RAD*np.array(colOffsetRelative), antennaName, startValidTime, pols, scans,
            np.array(azim)*180/math.pi,np.array(elev)*180/math.pi, ARCSEC_PER_RAD*np.array(colError), np.array(frequency)])   


def getMedianPWV(vis='.', myTimes=[0,999999999999], asdm='', verbose=False):
    """
    Extracts the PWV measurements from the WVR on all antennas for the
    specified time range.  The time range is input as a two-element list of
    MJD seconds (default = all times).  First, it tries to find the ASDM_CALWVR
    table in the ms.  If that fails, it then tries to find the 
    ASDM_CALATMOSPHERE table in the ms.  If that fails, it then tried to find 
    the CalWVR.xml in the specified ASDM, or failing that, an ASDM of the 
    same name (-.ms).  If neither of these exist, then it tries to find 
    CalWVR.xml in the present working directory. If it still fails, it looks 
    for CalWVR.xml in the .ms directory.  Thus, you only need to copy this 
    xml file from the ASDM into your ms, rather than the entire ASDM. Returns 
    the median and standard deviation in millimeters.
    Returns:
    The median PWV, and the median absolute deviation (scaled to match rms)
    For further help and examples, see https://safe.nrao.edu/wiki/bin/view/ALMA/GetMedianPWV
    -- Todd Hunter
    """
    pwvmean = 0
    success = False
    if (verbose):
        print("in getMedianPWV with myTimes = ", myTimes)
    try:
      if (os.path.exists("%s/ASDM_CALWVR"%vis)):
          mytb = tbtool()
          mytb.open("%s/ASDM_CALWVR" % vis)
          pwvtime = mytb.getcol('startValidTime')  # mjdsec
          antenna = mytb.getcol('antennaName')
          pwv = mytb.getcol('water')
          mytb.close()
          success = True
          if (len(pwv) < 1):
              if (os.path.exists("%s/ASDM_CALATMOSPHERE" % vis)):
                  pwvtime, antenna, pwv = readPWVFromASDM_CALATMOSPHERE(vis)
                  success = True
                  if (len(pwv) < 1):
                      print("Found no data in ASDM_CALWVR nor ASDM_CALATMOSPHERE table")
                      return(0,-1)
              else:
                  if (verbose):
                      print("Did not find ASDM_CALATMOSPHERE in the ms")
                  return(0,-1)
          if (verbose):
              print("Opened ASDM_CALWVR table, len(pwvtime)=", len(pwvtime))
      else:
          if (verbose):
              print("Did not find ASDM_CALWVR table in the ms. Will look for ASDM_CALATMOSPHERE next.")
          if (os.path.exists("%s/ASDM_CALATMOSPHERE" % vis)):
              pwvtime, antenna, pwv = readPWVFromASDM_CALATMOSPHERE(vis)
              success = True
              if (len(pwv) < 1):
                  print("Found no data in ASDM_CALATMOSPHERE table")
                  return(0,-1)
          else:
              if (verbose):
                  print("Did not find ASDM_CALATMOSPHERE in the ms")
    except:
        if (verbose):
            print("Could not open ASDM_CALWVR table in the ms")
    finally:
    # try to find the ASDM table
     if (success == False):
       if (len(asdm) > 0):
           if (os.path.exists(asdm) == False):
               print("Could not open ASDM = ", asdm)
               return(0,-1)
           try:
               [pwvtime,pwv,antenna] = readpwv(asdm)
           except:
               if (verbose):
                   print("Could not open ASDM = %s" % (asdm))
               return(pwvmean,-1)
       else:
           try:
               tryasdm = vis.split('.ms')[0]
               if (verbose):
                   print("No ASDM name provided, so I will try this name = %s" % (tryasdm))
               [pwvtime,pwv,antenna] = readpwv(tryasdm)
           except:
               try:
                   if (verbose):
                       print("Still did not find it.  Will look for CalWVR.xml in current directory.")
                   [pwvtime, pwv, antenna] = readpwv('.')
               except:
                   try:
                       if (verbose):
                           print("Still did not find it.  Will look for CalWVR.xml in the .ms directory.")
                       [pwvtime, pwv, antenna] = readpwv('%s/'%vis)
                   except:
                       if (verbose):
                           print("No CalWVR.xml file found, so no PWV retrieved. Copy it to this directory and try again.")
                       return(pwvmean,-1)
    try:
        matches = np.where(np.array(pwvtime)>myTimes[0])[0]
    except:
        print("Found no times > %d" % (myTimes[0]))
        return(0,-1)
    if (len(pwv) < 1):
        print("Found no PWV data")
        return(0,-1)
    if (verbose):
        print("%d matches = " % (len(matches)), matches)
        print("%d pwv = " % (len(pwv)), pwv)
    ptime = np.array(pwvtime)[matches]
    matchedpwv = np.array(pwv)[matches]
    matches2 = np.where(ptime<=myTimes[-1])[0]
    if (verbose):
        print("matchedpwv = %s" % (matchedpwv))
        print("pwv = %s" % (pwv))
    if (len(matches2) < 1):
        # look for the value with the closest start time
        mindiff = 1e12
        for i in range(len(pwvtime)):
            if (abs(myTimes[0]-pwvtime[i]) < mindiff):
                mindiff = abs(myTimes[0]-pwvtime[i])
#                pwvmean = pwv[i]*1000
        matchedpwv = []
        for i in range(len(pwvtime)):
            if (abs(abs(myTimes[0]-pwvtime[i]) - mindiff) < 1.0):
                matchedpwv.append(pwv[i])
        pwvmean = 1000*np.median(matchedpwv)
        if (verbose):
            print("Taking the median of %d pwv measurements from all antennas = %.3f mm" % (len(matchedpwv),pwvmean))
        pwvstd = 1000*MAD(matchedpwv)
    else:
        pwvmean = 1000*np.median(matchedpwv[matches2])
        pwvstd = 1000*MAD(matchedpwv[matches2])
        if (verbose):
            print("Taking the median of %d pwv measurements from all antennas = %.3f mm" % (len(matches2),pwvmean))
    return(pwvmean,pwvstd)
# end of getMedianPWV

def getAntennaIDsFromWeatherTable(vis):
    """
    Reads the unique list of antenna IDs from the WEATHER table of a measurement set.
    For ALMA, this will be [-1].
    -Todd Hunter
    """
    mytb = createCasaTool(tbtool)
    mytb.open(vis+'/WEATHER')
    ids = mytb.getcol('ANTENNA_ID')
    print("Read %d rows" % (len(ids)))
    mytb.close()
    uniqueIDs = np.unique(ids)
    return uniqueIDs

def ReadWeatherStation(scandict, station):
    """
    Parses a dictionary returned by readWeatherFromASDM and returns a list of
    lists of weather quantities.
    station: station ID (integer)
    """
    timeInterval = []
    pressure = []
    relHumidity = []
    temperature = []
    windDirection = []
    windSpeed = []
    windMax = []
    for entry in range(len(scandict)):
        if (scandict[entry]['stationId'] == station):
            timeInterval.append(scandict[entry]['timeInterval'])
            pressure.append(scandict[entry]['pressure'])
            relHumidity.append(scandict[entry]['relHumidity'])
            temperature.append(scandict[entry]['temperature'])
            windDirection.append(scandict[entry]['windDirection'])
            windSpeed.append(scandict[entry]['windSpeed'])
            windMax.append(scandict[entry]['windMax'])
#        else:
#            print "scandict=%d != %d" % (scandict[entry]['stationId'],station)
    d1 = [timeInterval, pressure, relHumidity, temperature, windDirection,
          windSpeed, windMax, station]
    return(d1)

def convertASDMTimeIntervalToMJDSeconds(s):
    """
    converts a string of format: 'start=2013-11-10T07:44:17.552000000, duration=369.180000'
    to MJD seconds.
    -Todd Hunter
    """
    ymdhms = s.split('=')[1].split(',')[0]
    return(dateStringToMJDSec(ymdhms,verbose=False))

def asdmspwmap(asdm):
    """
    Generate a list that maps the spw number that will be found in the
    measurement set to the corresponding value in the ASDM xml files.
    In general, the order will be [0,n+1,n+2,....] where n=number of antennas
    with WVR data.  
    Example return list:  [0,5,6,7...] if n=4 antennas, meaning
           that spw 1 in the ms = spw 5 in the ASDM xml files.
    -Todd Hunter
    """
    if (asdmLibraryAvailable == False):
        spwmap = au_noASDMLibrary.asdmspwmap(asdm)
        return spwmap
    a = ASDM()
    a.setFromFile(asdm,True)
    spwTable = a.spectralWindowTable().get()
    spwmap = []
    for row in range(len(spwTable)):
        if (spwTable[row].name().find('WVR#Antenna') < 0):
            spwmap.append(row)
    return(spwmap)

def readTcal(asdm, antenna, spw, meantime=None, spectrum='tcal', verbose=False):
    """
    Read the Tcal, Trx, Tsky, or Tsys spectrum for a specific antenna, spw and time. 
    If the time is not specified, it will return a dictionary of spectra keyed
    by the time (in MJD seconds).  The spw should be an spw number
    in the measurement set that is associated with a CALIBRATE_ATMOSPHERE scan.
    A conversion will be made to the spw number
    in the ASDM. If the spw is FDM, it is up to the user to find
    the associated TDM spw (e.g. via tsysspwmap) and pass it in.
    antenna: the antenna ID (string or integer, not the name)
    spw: the spw ID (string or integer)
    spectrum: 'tsys', 'trx' or 'tcal'

    Returns:
    A dictionary keyed by the MJD in seconds, with values equal to a list of
    arrays (one per polarization) each with N channels.
    
    Todd Hunter
    """
    spectrum = spectrum.lower()
    antenna = str(antenna)
    spw = int(spw)
    if (asdmLibraryAvailable == False):
        print("The ASDM bindings library is not available on this machine.")
        return
    a = ASDM()
    a.setFromFile(asdm,True)
    sysCalTable = a.sysCalTable().get()
    # find offset (usually it is the number of antennas with WVR data)
    spwmap = asdmspwmap(asdm)
    if (verbose):
        print("Spwmap = ", spwmap)
    spectra = {}
    timeDelta = 1e20
    for row in sysCalTable:
        antennaId = str(row.antennaId()).split('_')[1]
        if (antennaId == antenna):
            spectralWindowId = str(row.spectralWindowId()).split('_')[1]
            myspw = int(spectralWindowId)
            if (verbose):
                print("spw %d in the MS is spw %d in the ASDM, comparing to ASDM spw %d" % (spw, spwmap[spw], myspw))
            if (spwmap[spw] == myspw):
                mjdsec = convertASDMTimeIntervalToMJDSeconds(str(row.timeInterval()))
                if (spectrum == 'tcal'):
                    values = row.tcalSpectrum()
                elif (spectrum == 'trx'):
                    values = row.trxSpectrum()
                elif (spectrum == 'tsys'):
                    values = row.tsysSpectrum()
                elif (spectrum == 'tsky'):
                    values = row.tskySpectrum()
                elif (spectrum == 'tant'):
                    values = row.tantSpectrum()
                else:
                    print("Unrecognized spectrum type (%s)" % (spectrum))
                    return
                spectra[mjdsec] = []
                for v in values: # there will be one per polarization
                    spectra[mjdsec].append(np.array([v[f].get() for f in range(row.numChan())]))
                if (meantime is not None):
                    if (abs(mjdsec-meantime) < timeDelta):
                        timeDelta = abs(mjdsec-meantime)
                        closestTime = mjdsec
    if (verbose):
        print("Found %d spectra" % (len(spectra)))
    if (meantime is None):
        return(spectra)
    else:
        return(spectra[closestTime], closestTime)

def readStationFromASDMKeyedByAntennaName(sdmfile, station=None):
    """
    Similar to readStationFromASDM and readStationsFromASDM, but returns a dictionary keyed
    by antenna name (instead of antenna ID or pad name).
    Todd Hunter
    """
    mydict = readStationFromASDM(sdmfile, station)
    newdict = {}
    antennaNames = readAntennasFromASDM(sdmfile,verbose=False)
    for key in list(mydict.keys()):
        if (key < len(antennaNames)):
            antennaName = antennaNames[key]
            newdict[antennaName] = mydict[key]
    return(newdict)
    
def getAntennaAmbientTemperaturesFromASDM(asdmFile,  debugPlot=False):
    """
    Access the TMCDB for the antenna amibent temperatures recorded during
    an observation.
    Returns: 4 lists:
        timeList, antennaTemps, meanAntennaTemps, meanAntennaTempsErr
    - Denis Barkats.
    """
    from dateutil import rrule
    
    # get the start time of the ASDM
    dateStart, dateStart_mjdSec = getObservationStartDateFromASDM(asdmFile)
    dateEnd, dateEnd_mjdSec = getObservationEndDateFromASDM(asdmFile)
    antList = readAntennasFromASDM(asdmFile)
    start = dateStart.split(' UT')[0].replace(' ','T')
    end = dateEnd.split(' UT')[0].replace(' ','T')
    
    s = datetime.datetime.strptime(start, '%Y-%m-%dT%H:%M:%S')
    e = datetime.datetime.strptime(end, '%Y-%m-%dT%H:%M:%S')

    # time range to interpolate the other monitor points on.
    timeList = list(rrule.rrule(rrule.SECONDLY,interval=30, dtstart=s,until=e))
    ftimeList = pb.date2num(timeList)

    antennaTemps = {}
    meanAntennaTemps = {}
    meanAntennaTempsErr = {}
    for ant in antList:
        if ant[0:2]=='CM':
            # from ICT-1797
            # CM antennas: quadrapod, 4 total:
            # GET_METR_TEMPS_04[0]...[3]
            mpts = {'METR_TEMPS_04': [0,1,2,3]}
            antennaTemps[ant]=[[],[],[],[]]
                        
        elif  ant[0:2]=='DA':
            # DA antennas: apex, 3 total:
            # GET_METR_TEMPS_14[3] + GET_METR_TEMPS_15[0] + GET_METR_TEMPS_15[1]
            mpts = {'METR_TEMPS_14': [3], 'METR_TEMPS_15':[0,1]}
            antennaTemps[ant]=[[],[],[]]
                                               
        elif  ant[0:2]=='DV':
            # DV antennas: hexapod-subreflector interface cylinder, 8 total:
            # GET_METR_TEMPS_00[0]...[3] + GET_METR_TEMPS_01[0]...[3]
            mpts = {'METR_TEMPS_00': [0,1,2,3], 'METR_TEMPS_01': [0,1,2,3]}
            antennaTemps[ant]=[[],[],[],[],[],[],[],[]]
            time = [[],[],[],[],[],[],[],[]]
            
        elif  ant[0:2]=='PM':
            # PM antennas: 'ambient' temperature, 1 total:
            # GET_METR_TEMPS_1B[0]
            mpts = {'METR_TEMPS_1B': [0]}
            antennaTemps[ant]=[[]]

        meanAntennaTempsErr[ant]=[]
        meanAntennaTempsErr[ant]=[]

        # TODO: Put an offset in DV07 sensors before Nov 25th
        # TODO put offset in CM06 until a certain date.
        
        i = 0
        for m in list(mpts.keys()):
            print(start, end, ant, m)
            t = tmu.get_tmc_data(ant,'Mount',m,start,end, removefile=True, verbose=True)
            q = find((np.array(t['datetime']) > s) & (np.array(t['datetime']) < e))
            time = np.array(t['datetime'])[q]
            # interpolate the temperatures to common timeList range (once every 10s)
            for k in mpts[m]:
                antennaTemps[ant][i]= interp(ftimeList,date2num(time),array(t['value'])[q,k]/100.)
                i +=1
                
        if (debugPlot):
            pb.figure(1); pb.clf()
            for k in range(np.shape(antennaTemps[ant])[0]):
                pb.plot(timeList,antennaTemps[ant][k], '.')
            
        # take the average over monitorpoints.
        std1 = pb.mean(pb.std(antennaTemps[ant],axis=0))  # std from averaging over monitor points
        antennaTemps[ant] = pb.mean(antennaTemps[ant],axis=0)
        if (debugPlot): pb.plot(timeList,antennaTemps[ant], 'ko')
        
        # take the average over time
        if debugPlot: print(ant, pb.mean(antennaTemps[ant]), pb.std(antennaTemps[ant]))
        # the err is the quadratic sum of uncertainty over monitor points and uncertainty over time
        meanAntennaTempsErr[ant]= pb.sqrt(pb.std(antennaTemps[ant])**2+std1**2)
        meanAntennaTemps[ant] = pb.mean(antennaTemps[ant])

        if (debugPlot):
            pb.title('%s-%s, T=%3.2f +- %3.2f'%(asdmFile,ant,antennaTemps[ant],antennaTempsErr[ant]))
            pb.savefig('%s_%s_ambientTemp.png'%(asdmFile,ant))
                
    return timeList, antennaTemps, meanAntennaTemps,meanAntennaTempsErr
    
def plotAntennaAmbientTemperatures(asdm, weatherStation='MeteoTB2'):
    """
    Makes two plots of antenna ambient temperature: vs. time and vs. height
    Inputs:
    asdm: name of the ASDM
    -Denis Barkats
    """
    # TODO: add Nanten temps when discrepancy with ASTE is resolved ?
    stations = [i.strip() for i in list(getWeatherStationNamesFromASDM(asdm).values())]
    if weatherStation not in stations:
        print("%s is not in this dataset" % (weatherStation))
        print("Available: ", str(stations))
        return
    antList = readAntennasFromASDM(asdm, verbose = False)
    print("antList = ", antList)
    baselineLength = getBaselineLengthsFromASDM(asdm,weatherStation)
    heightDiff = getBaselineHeightFromASDM(asdm, weatherStation, verbose=False)
    pads = getAntennaPadsFromASDM(asdm)
    
    h = np.array(sorted(heightDiff.values()))
    ant_by_h = sorted(heightDiff,key=heightDiff.get)
    scaled_h = (h - h.min()) / h.ptp()
    colors_h = matplotlib.cm.coolwarm(scaled_h)

    bl= np.array(sorted(baselineLength.values()))
    ant_by_bl = sorted(baselineLength,key=baselineLength.get)
    scaled_bl = (bl - bl.min()) / bl.ptp()
    colors_bl = matplotlib.cm.jet(scaled_bl)
    
    # get antenna ambient temperatures
    time, temps, meanTemp, meanTempErr = getAntennaAmbientTemperaturesFromASDM(asdm)

    # Get Weather station data. 
    [conditions, medianConditions,stationName] = getWeatherFromASDM(asdm,station=1,verbose=False)
    mjdsec = np.array(conditions[0])
    weatherTime = mjdSecondsListToDateTime(mjdsec)
    temperature = kelvinToCelsius(np.array(conditions[3]))

    # plot temperatures vs time
    figure(1,figsize=(12,9));clf()
    i=0
    for ant in ant_by_h:
        pb.plot(time,temps[ant],'o',color=tableau20[i], markeredgecolor=tableau20[i])
        i+=1
    pb.plot(weatherTime, temperature,'k')
    pb.legend(ant_by_h+[weatherStation], prop={'size':9}, ncol=1,bbox_to_anchor=(1.1,1.02))
    pb.title(asdm+' Weather station and antenna ambient temperatures')
    pb.xlabel('UT time')
    pb.ylabel('Temperature (C)')
    pb.grid()
    fig1 = '%s_ambientTemp_vs_time.png'%(asdm)
    pb.savefig(fig1)

    # plot temperatures vs height diff
    pb.figure(2,figsize=(12,9));clf()
    legendtxt = []
    for ant in ant_by_h:
        marker = 'bo'
        if pads[ant][0] == 'P': marker='ro'
        if pads[ant][0] == 'S': marker='go'
        if pads[ant][0] == 'W': marker='mo'
        if (pads[ant][0] == 'N') or (pads[ant][0] == 'J'): marker='co'
        if pads[ant][0:2] == 'A1': marker='yo'
        pb.errorbar(heightDiff[ant],meanTemp[ant],meanTempErr[ant],fmt= marker)
        legendtxt.append('%s-%s:%2.1f'%(ant,pads[ant],meanTemp[ant]))
    pb.errorbar([0],np.mean(temperature),np.std(temperature),fmt= 'ko')
    x = -500+arange(600)
    pb.plot(x, -0.0065*x+np.mean(temperature),'k-')
    legendtxt.append('%s:%2.1f'%(weatherStation,np.mean(temperature)))
    pb.title(asdm+' Temperature vs height difference wrt Weather station ')
    pb.legend(legendtxt, prop={'size':9}, ncol=1,bbox_to_anchor=(1.1,1.02))
    pb.xlabel('Height Diff [m]')
    pb.ylabel('Temperature (C)')
    pb.grid()
    fig2 = '%s_ambientTemp_vs_height.png'%(asdm)
    pb.savefig(fig2)

    # plot temperatures vs height diff, color coded with distance to Weather station
    fig = pb.figure(3,figsize=(12,9));clf()
    legendtxt = []
    i=0
    for ant in ant_by_bl:
        m = 'o'
        pb.errorbar(heightDiff[ant],meanTemp[ant],meanTempErr[ant],marker=m, mfc=colors_bl[i], mec=colors_bl[i], ecolor=colors_bl[i])
        legendtxt.append('%s-%s:%2.1f'%(ant,pads[ant],meanTemp[ant]))
        i+=1
    pb.errorbar([0],np.mean(temperature),np.std(temperature),fmt= 'ko')
    x = -500+arange(600)
    pb.plot(x, -0.0065*x+np.mean(temperature),'k-')
    legendtxt.append('WSTB1:%2.1f'%(np.mean(temperature)))
    pb.title(asdm+' Temperature vs height difference wrt Weather station')
    pb.legend(legendtxt, prop={'size':9}, ncol=1,bbox_to_anchor=(1.1,1.02))
    pb.xlabel('Height Diff [m]')
    pb.ylabel('Temperature (C)')
    pb.grid()
    ax1 = fig.add_axes([0.17, 0.87, 0.65, 0.02])
    cmap = matplotlib.cm.jet
    norm = matplotlib.colors.Normalize(vmin=min(bl)/1e3, vmax = max(bl)/1e3)
    cb1 = matplotlib.colorbar.ColorbarBase(ax1,cmap=cmap,orientation='horizontal', norm=norm)
    cb1.set_label('Distance to Weather station [km]')
    fig3 = '%s_ambientTemp_vs_height_distance.png'%(asdm)
    pb.savefig(fig3)
    print("Plots left in: ", fig1)
    print("               ", fig2)
    print("               ", fig3)
    
def getWeatherStationNamesFromASDM(asdm, prefix=['WSTB','Meteo','OSF'], 
                                   returnNearestAntennas=False):
    """
    Gets the names of the weather stations in an ASDM.
    Returns: a dictionary keyed by station ID
    -Todd Hunter
    """
    asdm = uidToUnderscores(asdm)
    if (not os.path.exists(asdm)):
        print("Could not find ASDM.")
        return
    if (not os.path.exists(asdm + '/Station.xml')):
        print("Could not find Station.xml, is this really an ASDM?")
        return
    if (type(prefix) != list):
        prefix = [prefix]
    mydict = readStationFromASDM(asdm)
    names = []
    ids = []
    newdict = {}
    for s in list(mydict.keys()):
        for p in prefix:
            if (mydict[s]['name'].lower().find(p.lower()) >= 0):
                names.append(mydict[s]['name'])
                ids.append(s)
                newdict[s] = mydict[s]['name']
    if (returnNearestAntennas):
        stations = getWeatherStationPositionsFromASDM(asdm)
        posdict = getPadPositionsFromASDM(asdm)
        pads = getAntennaPadsFromASDM(asdm,keyByPadName=True)
        for pad in list(posdict.keys()):
            antennaName = pads[pad]
            antennaPosition = readAntennaPositionFromASDM(asdm)[antennaName]['position']
            posdict[pad] += computeITRFCorrection(posdict[pad], antennaPosition)
        newdict = {}
        for station in stations:
            mindistance = 1e12
            for pad in list(posdict.keys()):
                distance = np.linalg.norm([posdict[pad][0] - stations[station][0],
                                           posdict[pad][1] - stations[station][1],
                                           posdict[pad][2] - stations[station][2]])
                if (mindistance > distance):
                    mindistance = distance
                    closestPad = pad
            closestAntenna = pads[closestPad]
            newdict[station] = {'antenna': closestAntenna, 'distance': round(mindistance,1), 'pad': closestPad}
    return(newdict)

def exportAndGetWeatherStationPositionFromASDM(asdm, station='WSOSFLab', outfile=''):
    """
    Returns the position of the specified weather station from an ASDM.
    If asdm is not find, it first exports its metadata from the archive.
    outfile: if specified then append a line with the date, UID and station position vector.
    -Todd Hunter
    """
    if not os.path.exists(asdm):
        asdmExport(asdm, '-m')
    asdm = uidToUnderscores(asdm)
    if not os.path.exists(asdm):
        return
    positions = getWeatherStationPositionsFromASDM(asdm, prefix=station)
    if positions is None:
        print("That weather station was not found.  Present are: ", end=' ') 
        return
    for i in list(positions.keys()):
        if i.find(station) >= 0:
            if outfile != '':
                f = open(outfile,'a')
                date = getObservationStartDateFromASDM(asdm)[0]
                f.write('%s %s %s\n' % (date,asdm,positions[i]))
            return positions[i]
    
def getWeatherStationPositionsFromASDM(asdm, prefix=['WSTB','Meteo','OSF'],
                                       returnDeltaHeights=False,
                                       returnClosestPads=False,
                                       returnNearbyPads=False, radius=100,
                                       referenceStation=None):
    """
    Gets the geocentric XYZ positions of the weather stations in an ASDM.
    Returns: a dictionary keyed by station ID
    returnDeltaHeights: if True, then return height (in m) relative to highest one
           in a dictionary keyed by station name, and the name of the highest station
    referenceStation: the station name from which to base the zero point of deltaHeights
    radius: threshold in meters for associating with a nearby pad
    -Todd Hunter
    """
    asdm = uidToUnderscores(asdm)
    if (not os.path.exists(asdm)):
        print("Could not find ASDM.")
        return
    if (not os.path.exists(asdm + '/Station.xml')):
        print("Could not find Station.xml, is this really an ASDM?")
        return
    if (type(prefix) != list):
        prefix = [prefix]
    mydict = readStationFromASDM(asdm)
    antennaPads = getPadPositionsFromASDM(asdm)
    newdict = {}
    closestPads = {}
    nearbyPads = {}
    stations = []
    for s in list(mydict.keys()):
        for p in prefix:
            if (mydict[s]['name'].lower().find(p.lower()) >= 0):
                newdict[mydict[s]['name']] = np.array(mydict[s]['position'])
                stations.append(mydict[s]['name'])
                minDistance = 1e9
                nearbyPadlist = []
                nearbyAntennas = []
                for pad in antennaPads:
                    distance = np.linalg.norm(np.array(antennaPads[pad])-newdict[mydict[s]['name']])
                    if (distance < minDistance):
                        minDistance = distance
                        closestPad = pad
                        deltaHeight = geocentricXYZToEllipsoidalHeight(np.array(antennaPads[pad]))-\
                                      geocentricXYZToEllipsoidalHeight(newdict[mydict[s]['name']])
                    if (distance < radius):
                        nearbyPadlist.append(pad)
                        nearbyAntennas.append(getAntennaPadsFromASDM(asdm,keyByPadName=True)[pad])
                closestPads[mydict[s]['name']] = {'closestOccupiedPad': closestPad, 
                                                  'distance': minDistance, 
                                                  'deltaHeight': deltaHeight,
                                                  'closestAntenna': getAntennaPadsFromASDM(asdm,keyByPadName=True)[closestPad]}
                nearbyPads[mydict[s]['name']] = {'nearbyPads': nearbyPadlist,
                                                 'nearbyAntennas': nearbyAntennas}
    if (returnDeltaHeights):
        return(computeDeltaEllipsoidalHeights(newdict, referenceStation))
    elif (returnClosestPads):
        return(closestPads)
    elif (returnNearbyPads):
        return(nearbyPads)
    else:
        return(newdict)

def readStationFromASDM(sdmfile, station=None):
    """
    This function uses the ASDM bindings from casapy-telcal and reads the Station 
    position from the Station.xml file ( in the ASDM) and returns a dictionary of 
    station positions if no station is specified.  The dictionary format is:  
    {0: {'name':'J503',position:[x,y,z]}, 1:  etc.}.  If a station is specified, then 
    it returns the name and location as a simple list: ['J503',[x,y,z]].  
    - dbarkats
    """
    if not asdmLibraryAvailable:
#        print "The ASDM bindings library is not available on this machine. Using minidom code instead."
        mydict = au_noASDMLibrary.readStationFromASDM_minidom(sdmfile)
        if (station is None):
            return(mydict)
        else:
            return(mydict[station]['name'],mydict[station]['position'])
    a = ASDM()
    a.setFromFile(sdmfile,True)
    stationTable = a.stationTable().get()  

    staPos = {}
    for row in stationTable:
        stationName = row.name()
        position = [row.position()[0].get(),row.position()[1].get(),row.position()[2].get()]
        id = row.stationId().get()
        if (id not in staPos):  staPos[id] = {}
        staPos[id] = {'position':position, 'name':stationName}
    if (station is not None):
        return(staPos[station]['name'], staPos[station]['position'])
    return staPos

def getPadPositions(vis, includeWeatherStations=False):
    """
    Reads the pad positions in geocentric XYZ coordinates from an ms using tbtool,
    (excluding the weather stations).
    Returns a dictionary keyed by pad name, with value = its XYZ position.
    - Todd Hunter
    """
    if (not os.path.exists(vis)):
        print("Could not find measurement set.")
        return
    antTable = vis+'/ASDM_STATION'
    if (not os.path.exists(antTable)):
        print("Could not find ASDM_STATION table: %s" % (antTable))
        return
    mytb = createCasaTool(tbtool)
    mytb.open(antTable)
    position = np.transpose(mytb.getcol('position'))
    names = list(mytb.getcol('name'))
    types = list(mytb.getcol('type'))
    mytb.close()
    padDict = {}
    for i,name in enumerate(names):
        if (types[i] == 'ANTENNA_PAD' or includeWeatherStations):
            padDict[name] = position[i]
    return padDict

def getPadHeight(vis, pad=None, LOC=False):
    """
    Reads the XYZ geocentric coordinates for the specified pad and computes
    the Pythagorean distance from the center of the Earth.
    vis: name of measurement set
    pad: name of pad, e.g. 'W207'
    -Todd Hunter
    """
    if (not os.path.exists(vis)):
        print("getPadHeight: Could not find measurement set = ", vis)
        return
    padDict = getPadPositions(vis) # uses tbtool
    if (padDict is None): return
    names = list(padDict.keys())
    position = list(padDict.values())
    if (pad is None):
        mydict = {}
        for i,pad in enumerate(names):
            mydict[pad] = np.linalg.norm(position[i])
        return mydict
    elif (pad not in names):
        print("Pad %s not in dataset" % pad)
        return
    else:
        idx = names.index(pad)
        return(np.linalg.norm(position[idx]))
        
def getPadHeightFromASDM(asdm, pad=None, LOC=False):
    """
    Reads the XYZ geocentric coordinates for the specified pad and computes
    the Pythagorean distance from center of the Earth.
    asdm: name of ASDM
    pad: name of pad, e.g. 'W207'
    LOC: if True, use getPadLOC to get local tangent plane
         if False, use raw distance from geocenter
    """
    if LOC:
        return(getPadLOC(pad)[2])
    else:
        result = getPadXYZFromASDM(asdm, pad)
        if result is None: return
        return(np.linalg.norm(result))
    
def getPadXYZFromASDM(asdm, pad=None):
    """
    Reads the XYZ geocentric coordinates for the specified pad
    asdm: name of ASDM
    pad: name of pad, e.g. 'W207'
    """
    if (asdmLibraryAvailable == False):
        mydict = au_noASDMLibrary.readStationsFromASDM_minidom(asdm)
    else:
        mydict = readStationsFromASDM(asdm)
    if (pad is None):
        print("Select one of the available pads: ", list(mydict.keys()))
    if (pad not in list(mydict.keys())):
        print("This pad is not in the ASDM. Available pads: ", list(mydict.keys()))
        return
    position = mydict[pad]
    return(position)

def getDistanceFromMedianPadLOCFromASDM(asdm):
    """
    Computes a dictionary of distance of each antenna pad from the median 
    pad location in ENU coordinates, keyed by antenna name.
    -Todd Hunter
    """
    medianENU = np.array(getMedianPadLOCFromASDM(asdm))
    locs = getPadLOCsFromASDM(asdm)
    antennaNames = list(locs.keys())
    distance = {}
    for antennaName in antennaNames:
        distance[antennaName] = np.linalg.norm(locs[antennaName]-medianENU)
    return distance
    
def getMedianPadLOCFromASDM(asdm):
    """
    Computes the median of pad locations in an ASDM in ENU coordinates
    (in meters).  - Todd Hunter
    """
    mydict = getPadLOCsFromASDM(asdm)
    east,north,up = np.transpose(list(mydict.values()))
    return [np.median(east), np.median(north), np.median(up)]

def getPadLOCsFromASDM(asdm):
    """
    Gets a dictionary of pad locations in an ASDM in ENU coordinates 
    (in meters), keyed by antenna name.
    -Todd Hunter
    """
    mydict = readStationFromASDMKeyedByAntennaName(asdm)
    cx,cy,cz,clong,clat = getCOFAForASDM(asdm)
    antennaNames = list(mydict.keys())
    locs = {}
    for antennaName in antennaNames:
        x,y,z = mydict[antennaName]['position']
        x,y,z = simutil.simutil().itrf2loc(x,y,z,cx,cy,cz)
        locs[antennaName] = np.array([x[0],y[0],z[0]])
    return locs
    
def readAntennaPositionFromASDM(sdmfile, antennaType=''):
    """
    This function uses the ASDM bindings from casapy-telcal and reads the
    Antenna position (relative to its pad) from the Antenna.xml file 
    (in the ASDM) and returns a dictionary of Antenna positions in the 
    ENU coordinate system.
    antennaType: restrict to 'DA' or 'DV' or 'PM' or 'CM'
    - dbarkats
    """
    if (asdmLibraryAvailable == False):
        return(au_noASDMLibrary.readAntennaPositionFromASDM_minidom(sdmfile,antennaType))
#        print "readAntennaPositionFromASDM: The ASDM bindings library is not available on this machine and there is no workaround."
        return
    a = ASDM()
    a.setFromFile(sdmfile,True)
    antennaTable = a.antennaTable().get()  

    antPos = {}
    for row in antennaTable:
        antName = row.name().strip()
        id = row.stationId().get()
        position = [row.position()[0].get(),row.position()[1].get(),row.position()[2].get()]
        if (antName not in antPos):  antPos[antName] = {}
        antPos[antName] = {'position':position, 'id':id}
        
    return antPos

def getAntennaPadsForObsID(vis, obsid=0, keyByAntennaName=False):
    """
    For a concatenated dataset, this function allows you to break the degeneracy when
    a given antenna name occupies more than one pad.  CAS-9358 was created to add this
    ability to msmd.antennaids and msms.antennastations.
    keyByAntennaName: if True, then return a dictionary keyed by antenna name
    Returns: an array of station names
    -Todd Hunter
    """
    if not os.path.exists(vis):
        print("Could not find ms")
        return
    mytb = createCasaTool(tbtool)
    mytb.open(vis)
    myt = mytb.query('OBSERVATION_ID==%d'%obsid)
    ant1 = list(np.unique(myt.getcol('ANTENNA1')))
    ant2 = list(np.unique(myt.getcol('ANTENNA2')))
    myt.close()
    mytb.close()
    mytb.open(vis+'/ANTENNA')
    pads = mytb.getcol('STATION')
    names = mytb.getcol('NAME')
    mytb.close()
    ants = np.unique(ant1 + ant2)
    if keyByAntennaName:
        return(dict(itertools.izip(names[ants],pads[ants])))
    else:
        return(pads[ants])

def convertPadDictionaryToAntenna(mydict, vis):
    """
    Takes a dictionary keyed by pad name and converts it to one keyed
    by antenna name.
    -Todd Hunter
    """
    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)
    names = np.array(mymsmd.antennanames())
    stations = mymsmd.antennastations()
    mymsmd.close()
    newdict = {}
    for i in mydict:
        idx = list(stations).index(i)
        newdict[names[idx]] = mydict[i]
    return newdict

def convertAntennaDictionaryToPad(mydict, vis):
    """
    Takes a dictionary keyed by antenna name and converts it to one keyed
    by pad name.
    -Todd Hunter
    """
    mymsmd = createCasaTool(msmdtool)
    mymsmd.open(vis)
    names = np.array(mymsmd.antennanames())
    stations = mymsmd.antennastations()
    mymsmd.close()
    newdict = {}
    for i in mydict:
        idx = list(names).index(i)
        newdict[stations[idx]] = mydict[i]
    return newdict

def getAntennaPads(vis, exclude='', keyByAntennaName=False, keyByPadName=False, mymsmd=''):
    """
    Gets a list of antenna pads from a measurement set using msmd, excluding
    weather stations.
    exclude: a comma-separated list of pads to exclude
       'LBC' will remove the 21 pads of the ALMA LB campaign
    keyByAntennaName: will return a dictionary keyed by antenna name
    keyByPadName: will return a dictionary keyed by pad
    - Todd Hunter
    """
    stationList = exclude.split(',')
    if stationList[0] == 'LBC':
        stationList = ['P410','P405','P404','P402','P401','W204',
                       'W206','W207','W210','S301','S305','S306',
                       'A113','A118','A121','A122','A124','A127',
                       'A131','A132','A133']
        print("Ignoring LBC stations: ", stationList)
    if (not os.path.exists(vis)):
        print("Could not find measurement set.")
        return
    needToClose = False
    if mymsmd == '':
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    stations = np.array(mymsmd.antennastations())
    antennas = np.array(mymsmd.antennanames())
    if needToClose:
        mymsmd.close()
    if keyByAntennaName:
        if (keyByPadName): 
            print("keyByPadName and keyByAntennaName are mutually exclusive options")
            return
        mydict = {}
        for i,station in enumerate(stations):
            mydict[antennas[i]] = station
        return(mydict)
    elif keyByPadName:
        mydict = {}
        for i,station in enumerate(stations):
            mydict[station] = antennas[i]
        return(mydict)
    else:
        mylist = [x for x in stations if x not in stationList]
        return(mylist)

def padDifferencesFromASDM(asdm1, asdm2, showStableAntennas=False):
    """
    Describes the differences in antennas and pads between two
    datasets by looking at the ASDMs.  It then looks at antennaMoves.txt
    and shows what it thinks was the most recent move for each antenna.
    -Todd Hunter
    """
    obsA = getObservationStartDateFromASDM(asdm1)
    obsB = getObservationStartDateFromASDM(asdm2)
    if (obsA[1] < obsB[1]):
        asdm1dict = getAntennaPadsFromASDM(asdm1, keyByPadName=False)
        asdm2dict = getAntennaPadsFromASDM(asdm2, keyByPadName=False)
        date1 = obsA[0]
        date2 = obsB[0]
    else:
        asdm2dict = getAntennaPadsFromASDM(asdm1, keyByPadName=False)
        asdm1dict = getAntennaPadsFromASDM(asdm2, keyByPadName=False)
        date2 = obsA[0]
        date1 = obsB[0]
    antennas1 = list(asdm1dict.keys())
    antennas2 = list(asdm2dict.keys())
    for antenna in antennas1:
        if (antenna not in antennas2):
            print("Antenna %s (on pad %s) is not in the newer dataset. Last move = %s" % (antenna,asdm1dict[antenna],getMostRecentMove(antenna)))
        else:
            if (asdm1dict[antenna] == asdm2dict[antenna]):
                if (showStableAntennas):
                    print("Antenna %s is on pad %s in both datasets" % (antenna,asdm1dict[antenna]))
            else:
                print("Antenna %s moved from pad %s to %s between datasets (%s to %s). Last move = %s" % (antenna,asdm1dict[antenna], asdm2dict[antenna], date1, date2, getMostRecentMove(antenna)))
    for antenna in antennas2:
        if (antenna not in antennas1):
            print("Antenna %s (on pad %s) is not in the older dataset. Last move = %s" % (antenna,asdm2dict[antenna],getMostRecentMove(antenna)))

def getAntennaPadsFromASDMs(asdmlist, keyByPadName=False):
    """
    Calls getAntennaPadsFromASDM for a list of ASDMs and produces
    a combined dictionary where the values are lists, which will allow it
    to indicate if an antenna appears on multiple stations, or vice-versa.
    asdmlist: a list of ASDMs, a comma-delimeted string,
         or a string with a wildcard ('*.ms')
    keyByPadName: if True, the reverse the dictionary:  {'S301': ['DA41']}
    -Todd Hunter
    """
    if type(asdmlist) == str:
        asdmlist = asdmlist.split(',')
        final_asdmlist = []
        for asdm in asdmlist:
            if (asdm.find('*') >= 0):
                myasdmlist = glob.glob(asdm)
                print("Found %d ASDMs" % (len(myasdmlist)))
                final_asdmlist += myasdmlist
            else:
                final_asdmlist.append(asdm)
        asdmlist = final_asdmlist

    for i,asdm in enumerate(asdmlist):
        if (i == 0):
            mydict = getAntennaPadsFromASDM(asdm, keyByPadName)
            for key in mydict:
                mydict[key] = [mydict[key]]
        else:
            newdict = getAntennaPadsFromASDM(asdm, keyByPadName)
            for key in newdict:
                if key not in mydict:
                    mydict[key] = [newdict[key]]
                elif newdict[key] not in mydict[key]:
                    if keyByPadName:
                        print("Pad %s hosts a second antenna: %s" % (key, newdict[key]))
                    else:
                        print("Antenna %s appears on a second pad: %s" % (key, newdict[key]))
                    mydict[key].append(newdict[key])
    return mydict

def getAntennaPadsFromASDM(asdm, keyByPadName=False):
    """
    Returns a dictionary of antennas-pads association for that ASDM
    e.g.  {'DA41': 'S301'}
    keyByPadName: if True, the reverse the dictionary:  {'S301': 'DA41'}
    -Todd Hunter
    """
    pads = {}
    asdm = asdm.rstrip('/')
    antList = readAntennasFromASDM(asdm, verbose = False)
    antPos = readAntennaPositionFromASDM(asdm)
    if (antPos is None):
        mydict = au_noASDMLibrary.getAntennaPadsFromASDM_minidom(asdm)
        return mydict
    padPos = readStationFromASDM(asdm)
    for ant in antList:
        if (keyByPadName):
            pads[padPos[antPos[ant]['id']]['name']] = ant
        else:
            pads[ant] = padPos[antPos[ant]['id']]['name']
    return pads

def estimateSynthesizedBeamFromASDM(asdm, useL80method=True, verbose=False):
    """
    Estimates the synthesized beam from the L80 percentile length and
    representative frequency in an ASDM.
    useL80method: if True, use .574lambda/L80; if False, then use the max baseline
    -Todd Hunter
    """
    representativeFrequency = representativeFrequencyFromASDM(asdm,verbose=verbose)
    if representativeFrequency is None: return
    if useL80method:
        L80 = getBaselineStatsFromASDM(asdm, percentile=80, verbose=verbose)[-1] # meters
        arcsec = 3600*np.degrees(1)*0.574*c_mks/(representativeFrequency*1e9*L80)
    else:
        maxBaseline = getBaselineStatsFromASDM(asdm, verbose=False)[2] # meters
        arcsec = printBaselineAngularScale(maxBaseline, representativeFrequency)
    return arcsec

def estimateMRS(vis, field='', verbose=False):
    """
    Estimates the maximum recoverable scale (a.k.a. LAS) from the max baseline length and
    representative frequency (or mean of science spw frequencies) in a measurement set.
    field: if not blank, then find the first integration on the specified
           field ID or name, get its az&el and compute the projected baseline
           lengths rather than the unprojected baseline lengths
    Reports the ALMA Cycle-4 approved formula: 0.983*lambda/L5
       and the Cycle-8 tech. handbook formula: 0.6*lambda/Lmin
       and returns the smaller of the two values
    -Todd Hunter
    """
    result = getBaselineStats(vis, field=field, percentile=5, verbose=verbose)
    if (result is None): return
    L5 = result[0] # meters
    Lmin = result[2]
    if os.path.exists(vis+'/ASDM_SBSUMMARY'):
        freqHz = representativeFrequency(vis, verbose=verbose) * 1e9
        print("Using representative frequency")
    else:
        freqHz = np.mean(getScienceFrequencies(vis)) 
        print("Using mean of the science spw mean frequencies.")
    arcsec = 3600*np.degrees(1)*0.983*c_mks/(freqHz*L5)
    print("Cycle 4 handbook:  0.984*lambda/L5 = %g arcsec (L5 = %g m)" % (arcsec, L5))
    arcsec8 = 3600*np.degrees(1)*0.983*c_mks/(freqHz*Lmin)
    print("Cycle 8 handbook:  0.6*lambda/Lmin = %g arcsec (Lmin = %g m)" % (arcsec8, Lmin))
    return np.min([arcsec8,arcsec])

def estimateSynthesizedBeam(vis, field='', useL80method=True, verbose=False, 
                            spw=None, mymsmd=None):
    """
    Estimates the synthesized beam from the max baseline length and
    representative frequency in a measurement set.
    field: if not blank, then find the first integration on the specified
           field ID or name, get its az&el and compute the projected baseline
           lengths rather than the unprojected baseline lengths
    useL80method: if True, use .574lambda/L80; if False, then use the max baseline
    spw: if specified (string or int), then use its meanfreq instead of representative freq
    -Todd Hunter
    """
    needToClose = False
    if mymsmd is None:
        needToClose = True
        mymsmd = createCasaTool(msmdtool)
        mymsmd.open(vis)
    if useL80method:
        result = getBaselineStats(vis, field=field, percentile=80, verbose=verbose, mymsmd=mymsmd)
    else:
        result = getBaselineStats(vis, field=field, verbose=verbose)
    if (result is None): 
        if needToClose:
            mymsmd.close()
        return
    if useL80method:
        L80 = result[0] # meters
        if spw is None:
            repFreqGHz = representativeFrequency(vis,verbose=verbose)
        else:
            repFreqGHz = mymsmd.meanfreq(int(spw)) * 1e-9
        if repFreqGHz is None:
            print("There is no representative frequency. Using mean of science windows...")
            myfreqs = getScienceFrequencies(vis, mymsmd=mymsmd)
            if myfreqs is None:
                print("There are no science windows, using first spw")
                repFreqGHz = mymsmd.meanfreq(0) * 1e-9
            else:
                repFreqGHz = np.mean(myfreqs) * 1e-9
            print("which is %f GHz." % (repFreqGHz))
        arcsec = 3600*np.degrees(1)*0.574*c_mks/(repFreqGHz*1e9*L80)
    else:
        maxBaseline = result[2] # meters
        arcsec = printBaselineAngularScale(maxBaseline, representativeFrequency(vis,verbose=verbose))
    if needToClose:
        mymsmd.close()
    return arcsec

def makeFixephem(vis, scriptname='fixephem.py'):
    """
    Builds a script to fix the target measurement sets for multi-execution ephemeris
    projects of Cycle 3 and 4.  See CAS-8984.
    vis: a single measurement set, or a string with a wildcard ('*.ms')
    -Todd Hunter
    """
    if (vis.find('*') >= 0):
        vis = glob.glob(vis)[0]
        print("Using ", vis)
    fieldnames = getEphemerisFields(vis)
    f = open(scriptname,'w')
    f.write('from recipes.ephemerides import concatephem\n')
    f.write('import glob\n')
    f.write("myvis = glob.glob('*target.ms')\n")
    f.write("concatephem.concatreplaceephem(vis=myvis, field='%s', commontime=True)\n" % fieldnames)
    f.close()

def makeFlagTemplateOutlierAntennas(asdms, templateType='targets', 
       requestedResolution=None, useUvrange=False, verbose=False):
    """
    Builds a flagtemplate file for the pipeline which flags outlier
    antennas.  If the file already exists, it will be appended to.
    asdms: a single ASDM, a list of ASDMs as a python list of strings, or as a
           single comma-delimited string
    templateType: '' or 'targets' 
                  '':  will produce <asdm>_flagtemplate.txt
           'targets':  will produce <asdm>_flagtargetstemplate.txt
    requestedResolution: if specified, then use this value to override what is
           in the ASDM (which in Cycle 3 was the range provided by the 
           configuration that the OT assigned to the SB, and in Cycle 4 is
           the PI requested value).
    useUvrange: if True, then produce a uvrange flag command instead of an
           an antenna name flag command.  This is not recommended as it can
           produce a sharp edge in the uv coverage which will cause ringing
           in the image plane.
    -Todd Hunter
    """
    if (type(asdms) != list):
        asdms = asdms.split(',')
    for asdm in asdms:
        if (not os.path.exists(asdm)):
            print("Could not find ASDM")
            return
        if (templateType not in ['','targets']):
            print("invalid templateType")
            return
        result = findOutlierAntennas(asdm, requestedResolution, returnSensitivityLoss=True)
        if result is None:  # resolution not found in ASDM
            return
        antennas, sensitivityLoss, maxuvdist = result
        if len(antennas) > 0:
            fname = os.path.basename(asdm) + '_flag%stemplate.txt' % (templateType)
            f = open(fname,'a')
            if (useUvrange):
                print("WARNING!: using a uvrange flag can result in a sharp edge in the uv coverage, which will produce ringing in the image domain.")
                if (sensitivityLoss < 20):
                    print("    You should consider flagging by antenna because it will result in only %.1f%% loss in sensitivity." % (sensitivityLoss))
                myline = "mode='manual' uvrange='0~%d' reason='too_many_outlier_antennas'\n" % int(np.round(maxuvdist))
            else:
                myline = "mode='manual' antenna='%s' reason='outlier_antenna'\n" % ';'.join(antennas)
            f.write(myline)
            f.close()
            print("Wrote %s :" % fname)
            print(myline)
        else:
            print("No outliers found in %s." % (asdm))

def findOutlierAntennas(asdm, requestedResolution=None, 
                        returnSensitivityLoss=False, verbose=False):
    """
    Finds outlier antennas in an ASDM by considering the requested
    angular resolution, representative frequency, and minimum elevation of
    the science targets.
    requestedResolution: if specified (in arcsec), then use this value instead
           of what is in the ASDM
    returnSensitivityLoss: if True, then also return the percentage loss in
           sensitivity and the desired max baseline (in meters)
    -Todd Hunter
    """
    if (not os.path.exists(asdm)):
        print("Could not find ASDM")
        return
    baseline = maxBaselineForRequestedResolutionFromASDM(asdm, requestedResolution)
    ignoreAntennas = []
#    if baseline <= 0:
#        if returnSensitivityLoss:
#            return ignoreAntennas, 0.0, baseline
#        else:
#            return ignoreAntennas
    actualMaxBaseline = getBaselineStatsFromASDM(asdm)[2]
    print("Required max baseline: %.1fm.  Actual max baseline: %.1fm" % (baseline,actualMaxBaseline))
    mydict = getMedianBaselinePerAntenna(asdm, ignoreAntennas)
    idx = np.argsort(list(mydict.values()))[::-1]
    antennaNames = np.array(list(mydict.keys()))[idx]
    nAntennas = len(antennaNames)
    medianBaselines = np.array(list(mydict.values()))[idx]
    if baseline <= 0:
        for antennaName in antennaNames:
            if (antennaName.find('CM') >= 0):
                print("Since a 7m antenna is present, setting baseline to 45m")
                baseline = 45
                break
    if baseline > 0:
        while (medianBaselines[0] > baseline):
            ignoreAntennas.append(antennaNames[0])
            mydict = getMedianBaselinePerAntenna(asdm, ignoreAntennas)
            idx = np.argsort(list(mydict.values()))[::-1]
            antennaNames = np.array(list(mydict.keys()))[idx]
            medianBaselines = np.array(list(mydict.values()))[idx]
            if verbose:
                print("Dropping %s, new max median baseline=%.1f" % (antennaNames[0],np.max(medianBaselines)))
    keepAntennas = nAntennas - len(ignoreAntennas)
    baselinePercentage = 100*(1-keepAntennas*(keepAntennas-1)/(nAntennas*(nAntennas-1.0)))
    sensitivityLossPercentage = 100*(1-np.sqrt(keepAntennas*(keepAntennas-1)/(nAntennas*(nAntennas-1.0))))
    print("%d/%d antennas are outliers, representing %.1f%% of the baselines and %.1f%% of the sensitivity."  % (len(ignoreAntennas), nAntennas, baselinePercentage, sensitivityLossPercentage))
    if returnSensitivityLoss:
        return sorted(ignoreAntennas), sensitivityLossPercentage, baseline
    else:
        return sorted(ignoreAntennas)

def plotExtremeBaselinePerAntenna(asdm='', statistic='max', ignoreAntennas=[], 
                                  percentile=0, config='', plotfile='',
                                  labelLengthsLongerThan=100):
    """
    Runs getExtremeBaselinePerAntenna and plots a histogram.
    -Todd Hunter
    """
    mydict = getExtremeBaselinePerAntenna(asdm, statistic, ignoreAntennas, percentile, sort=False, config=config)
    names = list(mydict.keys())
    lengths = np.array(list(mydict.values()))*0.001
    pb.clf()
    desc = pb.subplot(111)
    bins = [.030,.100,.300,1.000,3.000,10.000,30.000,100.000,300.000]
    pb.hist(lengths,bins=bins)
    pb.xscale('log')
    pb.xlabel('Minimum baseline length (km)')
    pb.ylabel('Number of antennas')
#    majorFormatter = matplotlib.ticker.FormatStrFormatter('%g')
#    desc.xaxis.set_major_formatter(majorFormatter)
    pb.xticks(bins,bins)
    pb.xlim([0.9*bins[0],bins[-1]*1.1])
    desc.yaxis.set_minor_locator(MultipleLocator(2))
    pb.title(config,size=18)
    idx = np.where(lengths > labelLengthsLongerThan)[0]
    y1 = pb.ylim()[1]
    for i,j in enumerate(idx):
        pb.plot(2*[lengths[j]], pb.ylim(), 'k:')
        x = lengths[j]
        y = (1-(i+0.22)*0.13)*y1
        pb.text(x,y, names[j], rotation=90, transform=desc.transData, ha='center',size=13.5)
    pb.draw()
    if plotfile != '':
        if plotfile == True:
            plotfile = config + '.png'
        pb.savefig(plotfile)
        print("Wrote ", plotfile)

def getExtremeBaselinePerAntenna(asdm='', statistic='max', ignoreAntennas=[], percentile=0, sort=False, config=''):
    """
    Compute the statistcal extreme baseline length per antenna of the specified ASDM.
    statistic: 'median', 'min', 'max', 'mean', or 'std'
    percentile: only return antennas whose median is > L(percentile,e.g. 80) of the array
    sort: if True, then sort by statistic rather than by antenna name
    Returns: a dictionary keyed by antenna name (if sort=False)
              otherwise, a list of tuples [('name', length)...]
    -Todd Hunter
    """
    if asdm == '' and config == '':
        print("You must specify either asdm or config")
        return
    medians = {}
    if config == '':
        antList = readAntennasFromASDM(asdm, verbose = False)
        antPos = readAntennaPositionFromASDM(asdm)
#        print "antPos: ", antPos # e.g.  {'DV12': {'position': [-0.001052, -1.9e-05, 7.000737], 'id': 32}}
        padPos = readStationFromASDM(asdm)  
#        print "padPos: ", padPos # e.g. {0: {'position': [2225050.61527, -5440086.969379, -2481632.711731], 'type': 'ANTENNA_PAD', 'name': 'A023'}} 
    else:
        returnValue = readPadConfigurationFile(config, verbose=False)
        if (returnValue is None): return
        stations, positions, names, nAntennas, diameters, observatory = returnValue
        padPos = {}
        antPos = {}
        antList = []
        for i,pad in enumerate(stations):
            padPos[i] = {'position': positions[i], 'type': 'ANTENNA_PAD', 'name': stations[i]}
            antPos[stations[i]] = {'position': positions[i], 'id': i}
            antList.append(stations[i])
    if (percentile > 0 and config == ''):
        L80 = getBaselineStatsFromASDM(asdm, percentile=percentile, verbose=False)[-1]
        print("L80 = ", L80)
    else:
        L80 = 0
    for ant in antList:
        if (ant in ignoreAntennas): continue
        lengths = []
        for ant2 in antList:
            if (ant2 in ignoreAntennas): continue
            if (ant2 != ant):
                lengths.append(computeBaselineLength(padPos[antPos[ant]['id']]['position'], 
                                                     padPos[antPos[ant2]['id']]['position']))
        if (statistic == 'median'):
            mymedian = np.median(lengths)
        elif (statistic == 'min'):
            mymedian = np.min(lengths)
        elif (statistic == 'max'):
            mymedian = np.max(lengths)
        elif (statistic == 'mean'):
            mymedian = np.mean(lengths)
        elif (statistic == 'std'):
            mymedian = np.std(lengths)
        else:
            print("Statistic must be one of: 'median', 'min', 'max', 'mean', 'std'")
            return
        if (mymedian > L80):
            medians[ant] = mymedian
    if sort:
        medians = sorted(list(medians.items()), key=operator.itemgetter(1))
    return medians

def getMedianBaselinePerAntenna(asdm, ignoreAntennas=[], percentile=0, sort=False):
    """
    Compute median baseline length per antenna of the specified ASDM.
    percentile: only return antennas whose median is > L80 of the array
    sort: if True, then sort by statistic rather than by antenna name
    Returns: a dictionary keyed by antenna name (if sort=False)
           otherwise, a list of tuples [('name', length)...]
    """
    if (not os.path.exists(asdm+'/Antenna.xml')):
        print("Could not find Antenna.xml.")
        return
    return(getExtremeBaselinePerAntenna(asdm, 'median', ignoreAntennas, percentile, sort))

def getBaselineStatsFromASDM(asdm, percentile=None, verbose=False, units='m'): 
    """
    Compute statistics on the baseline lengths of the specified ASDM.
    unitsL 'm' or 'kl' for kilolambda
    Returns: a list of: number, min, max, median, mean, st.dev, 20%ile, 
             25%ile, 30%ile, 75%ile, 90%ile
    If percentile is specified, then it is appended to the returned list.
    """
    if (not os.path.exists(asdm+'/Antenna.xml')):
        print("Could not find Antenna.xml.")
        return
    if (units not in ['m','kl']):
        print("length units must be either 'm' or 'kl'.")
        return
    antList = readAntennasFromASDM(asdm, verbose = False)
    antPos = readAntennaPositionFromASDM(asdm)
    padPos = readStationFromASDM(asdm)
    lengths = []
    for ant in antList:
        for ant2 in antList:
            if (ant2 > ant):
                lengths.append(computeBaselineLength(padPos[antPos[ant]['id']]['position'], 
                                                     padPos[antPos[ant2]['id']]['position']))
    lengths = np.array(lengths)
    if units == 'kl':
        wavelength = c_mks / getMeanFreqFromASDM(asdm)['mean']['meanfreq']
        lengths /= wavelength
        lengths *= 0.001 # convert lambda to klambda
    twenty = scoreatpercentile(lengths, 20)    
    twentyfive = scoreatpercentile(lengths, 25)    
    thirty = scoreatpercentile(lengths, 30)    
    seventyfive = scoreatpercentile(lengths, 75)
    ninety = scoreatpercentile(lengths, 90)
    number = len(lengths)
    mylist = [number,np.min(lengths),np.max(lengths),np.median(lengths),
           np.mean(lengths), np.std(lengths),twenty,twentyfive,thirty,
           seventyfive,ninety]
    if (percentile is not None):
        value = scoreatpercentile(lengths, percentile)
        if verbose: print("%.0f percentile baseline length is %f %s" % (percentile, value, units))
        mylist = [value] + mylist
    return(mylist)

def getBaselineLengthsFromASDM(asdm, refAnt):
    """
    returns a dictionary of antennas, baseline length from all antennas to
    refAnt.
    refAnt is either antenna name or a weather station name: WSTB1, WSTB2
    -Denis Barkats
    See also: au.getAllBaselineLengthsFromASDM which is analogous to 
              au.getBaselineLengths for measurement sets
    """
    bl = {}
    antList = readAntennasFromASDM(asdm, verbose = False)
    antPos = readAntennaPositionFromASDM(asdm)
    if (antPos == None): return
    padPos = readStationFromASDM(asdm)
    if refAnt.startswith('WSTB1') or refAnt.startswith('Meteo'):
        key = [mykey for mykey,val in list(padPos.items()) if val['name'].find(refAnt)>=0]
        refAntPos = padPos[key[0]]['position']
    else:
        refAntPos = padPos[antPos[refAnt]['id']]['position']
    for ant in antList:
        bl[ant]= computeBaselineLength(padPos[antPos[ant]['id']]['position'], refAntPos)
        #print ant, int(bl[ant]*100)/100.
    return bl

def getAllBaselineLengthsFromASDM(asdm, histogram=False, nbins=None, 
                                  linear=True, plotfile=None,):
    """
    Returns a list of unprojected baseline lengths in meters
    histogram: if True, then produce a histogram
    nbins: number of bins in histogram, default = automatic
    linear: if True, use linear bins along x-axis, otherwise use log
    plotfile: name of plotfile to produce, or True for automatic naming
    -Todd Hunter
    """
    asdm = asdm.rstrip('/')
    antList = readAntennasFromASDM(asdm, verbose=False)
    antPos = readAntennaPositionFromASDM(asdm)
    if (antPos is None): return
    padPos = readStationFromASDM(asdm)
    bl = []
    for ant in antList:
        for ant2 in antList:
            if ant != ant2:
                bl.append(computeBaselineLength(padPos[antPos[ant]['id']]['position'], padPos[antPos[ant2]['id']]['position']))
    if (histogram):
        flatlengths = bl
        if (nbins is None):
            nbins = len(flatlengths)/10
        if linear:
            bins = np.linspace(0,np.max(flatlengths),nbins)
        else:
            bins = np.logspace(0,np.log10(np.max(flatlengths)),nbins)
            print("bins = ", bins)
        pb.clf()
        pb.hist(flatlengths, bins=bins) 
        pb.ylabel('Number of baselines')
        units = 'm'
        pb.xlabel('Baseline length (%s)'%units)
        pb.title(asdm+'  '+getObservationStartDateFromASDM(asdm)[0])
        if (plotfile is not None):
            if plotfile == True:
                png = '%s_baseline_histogram.png' % asdm
            else:
                png = plotfile
            pb.savefig(png)
            print("Left histogram in %s" % png)
    return sorted(bl)

def getBaselineHeightFromASDM(asdm, refAnt, verbose=True):
    """
    -Denis Barkats
    """
    hDiff = {}
    antList = readAntennasFromASDM(asdm, verbose = False)
    antPos = readAntennaPositionFromASDM(asdm)
    padPos = readStationFromASDM(asdm)
    if refAnt.startswith('WSTB1') or refAnt.startswith('Meteo'):
        [key for key,val in list(padPos.items()) if val['name'].find(refAnt) >= 0]
        refAntPos = padPos[key]['position']
    else:
        refAntPos = padPos[antPos[refAnt]['id']]['position']
    for ant in antList:
        hDiff[ant]= computeBaselineHeightDiff(padPos[antPos[ant]['id']]['position'], refAntPos)
        if verbose == True: print('%s, %7.2f'%(ant, hDiff[ant]))
    return hDiff
    
def printSwVersionFromASDM(asdm):
    """
    Reads and prints the contents of the ASDM's Annotation.xml table.
    -Denis Barkats and Todd Hunter
    """
    if (asdmLibraryAvailable == False):
        print("The ASDM bindings library is not available on this machine. Using the minidom code instead.")
        return(au_noASDMLibrary.readSoftwareVersionFromASDM_minidom(asdm))
    a = ASDM()
    a.setFromFile(asdm,True)
    at = a.annotationTable().get()
    print('\n### Annotation table for ASDM: %s contains sw version, atm delayModel... ### \n' %asdm)
    for row in at:
        print("%s: %s" %(row.issue(),row.details()))

    print("\n####")
    return

def compareTimesFromAnnotationTableToBandpassPreAverage(vis, asdm='', atmcal=None, maxvis=-1):
    """
    Computes the change in SNR of the bandpass solution if the "optimized" integration times
    reported by SSR Cycle 6 were to be used instead of the actual integration times.
    vis: single measurement set, comma-delimited list, or pipeline parent directory, 
           e.g. 'E2E6.1.00030.S_yyyy_mm_ddThh_mm_ss.sss'
    asdm: corresponding ASDM(s), not necessary if vis is a directory
    atmcal: for cases of a single vis, you can pass in an existing instance of Atmcal to speed it up
    Returns: bandpassPreAverage dictionary for final vis
    -Todd Hunter
    """
    if os.path.isdir(vis):
        mydir = os.path.join(vis,'S*/G*/M*')
        asdm = sorted(glob.glob(os.path.join(mydir,'rawdata/uid*')))
        vis = [os.path.join(os.path.dirname(myasdm).replace('rawdata','working'),os.path.basename(myasdm)+'.ms') for myasdm in asdm]
    if type(vis) == str:
        vis = vis.split(',')
    if type(asdm) == str:
        asdm = asdm.split(',')
    if len(vis) > 1 and atmcal is not None:
        print("The atmcal option is only supported for single vis datasets.")
        return
    mydict = None
    for i in range(len(vis)):
        if maxvis > 0 and i >= maxvis:
            break
        if not os.path.exists(vis[i]):
            print(vis[i])
            return
        asdmdict = readTimesFromAnnotationTable(asdm[i])
        if asdmdict == {}:
            print("No SSR times found for ", asdm[i])
            continue
        mydict = bandpassPreAverage(vis[i], atmcal=atmcal)
        if mydict is None:
            print(vis[i])
            return
        for spectralSpec in asdmdict:
            ssrOptimumBandpassTime = asdmdict[spectralSpec]['BANDPASS']['optimumIntegration']
            ssrAdoptedBandpassTime = asdmdict[spectralSpec]['BANDPASS']['adoptedIntegration']
        print('\n'+vis[i])
        print('SSR determined times: "optimum"=%.2f sec  adopted=%.2fsec' % (ssrOptimumBandpassTime, ssrAdoptedBandpassTime))
        observedTime = np.min(list(mydict['timeOnSourceMinutes'].values()))
        snr = []
        solint = []
        bandwidth = []
        channels = []
        actualChannelsInSolution = []
        channelsInSolution = []
        actualSnr = []
        bandpassTable = sorted(glob.glob(vis[i]+'*bcal.final.tbl'))
        if len(bandpassTable) > 0:
            actualChannels = getNChanFromCaltable(bandpassTable[0])
        for spw in mydict['spws']:
            snr.append(mydict[spw]['snrPerChannel'])
            solint.append(parseFrequencyArgumentToHz(mydict[spw]['solint']))
            bandwidth.append(mydict[spw]['bandwidth'])
            channels.append(mydict[spw]['totalChannels'])
            channelsInSolution.append(mydict[spw]['channelsInSolution'])
            if len(bandpassTable) > 0:
                result = snrFromCaltable(bandpassTable[0])
                if spw not in list(result.keys()):
                    print("Unexpected error: spw %d not in the solution table" % (spw))
                    return
                result = result[spw]
                fieldIDs = list(result.keys())
                fieldID = fieldIDs[0]
                actualSnr.append(result[fieldID]['median'])
                actualChannelsInSolution.append(actualChannels[spw])
        timetype = ['optimum','adopted']
        idx = np.argmax(solint)
        print("Actual observed time: %.2f minutes = %.1f seconds" % (observedTime, observedTime*60))
        spw = mydict['spws'][idx]
        outputChannels = channelsInSolution[idx]
        print("Pipeline recommended solint in lowest SNR spw%d: %s = %s channels yielding %d output channels" % (spw, mydict[spw]['solint'], mydict[spw]['solintChannelsEvenlyDivisible'], outputChannels))
        print("Pipeline predicted SNR in each spw: ", str([round(s,2) for s in snr]))
        if len(bandpassTable) > 0:
            achievedSnr = [round(s,2) for s in actualSnr]
            print("Pipeline achieved SNR in each spw: ", str(achievedSnr))
            # Fill dictionary entries for use by the analyze=True option of readTimesFromAnnotationTables
            mydict['achievedSnr'] = actualSnr
            mydict['snr'] = snr
            mydict['channelsInSolution'] = channelsInSolution
            mydict['actualChannelsInSolution'] = actualChannelsInSolution
        for i,ssrBandpassTime in enumerate([ssrOptimumBandpassTime, ssrAdoptedBandpassTime]):
            print('Ramifications of using SSR "%s" time:' % (timetype[i]))
            ssrsnrReductionFactor = (ssrBandpassTime/(60*observedTime))**0.5
            snr = np.array(snr)
            ssrsnr = snr*ssrsnrReductionFactor
            spw = mydict['spws'][idx]
            ssrsolint = solint[idx] / (ssrsnrReductionFactor**2)
            nchan = bandwidth[idx] / ssrsolint  # predicted for SSR bandpass solution
            totalChannels = channels[idx]
            print("    SNR reduction factor if SSR-determined time (%.1f sec) were to be used: " % (ssrBandpassTime), ssrsnrReductionFactor)
            print("    Predicted SNR in each spw using SSR-determined time: ", ssrsnr)
            nchanEven = np.min([nextHighestIntegerDivisible(totalChannels, int(np.ceil(nchan))), totalChannels])
            print("    Number of output channels pipeline would choose for spw%d if SSR-determined time were to be used: %.1f --> %d" % (spw,nchan,nchanEven))
            print("----------------------------------------------------------------------------------------------------------------")   
    return mydict
    
def readTimesFromAnnotationTables(mydir, analyze=False, maxvis=-1):
    """
    Given a parent directory, e.g. 'pipeline/root/E2E6*', finds all
    asdms in the containted rawdata directories and runs readTimesFromAnnotationTable on them
    mydir: either the full path ending in '/rawdata', or a wildcard string like 'E2E6*'
    -Todd Hunter
    """
    if mydir.find('*') >= 0:
        asdms = sorted(glob.glob(os.path.join(mydir,'S*/G*/M*/rawdata/uid*')))
    else:
        asdms = sorted(glob.glob(os.path.join(mydir,'uid*')))
    justasdm = []
    mydirs = []
    for asdm in asdms:
        if asdm not in justasdm: # do not repeat the analysis of repeated executions
            mydict = readTimesFromAnnotationTable(asdm)
            project = os.path.dirname(asdm).split('/')[0]
            print(project, os.path.basename(asdm), mydict)
            justasdm.append(os.path.basename(asdm))
            mydirs.append(project)
    if analyze:
        outline = []
        for mydir in mydirs:
            mydict = compareTimesFromAnnotationTableToBandpassPreAverage(mydir, maxvis=maxvis)
            if mydict is None: continue
            if 'achievedSnr' in mydict:
                outline.append(mydict['vis'])
                outline.append('predicted SNRs for %s-channel solutions: %s' % (str(mydict['channelsInSolution']),str([round(s,2) for s in mydict['snr']])))
                outline.append('actual SNRs for %s-channel solutions: %s' % (str(mydict['actualChannelsInSolution']), str([round(s,2) for s in mydict['achievedSnr']])))
                outline.append('')
        filename = 'readTimesFromAnnotationTables.txt'
        f = open(filename,'w')
        for line in outline:
            print(line)
            f.write(line+'\n')
        f.close()

def readTimesFromAnnotationTable(asdm):
    """
    For data from Cycle 6 onward, reads the Parameter Optimization entries from the ASDM's Annotation.xml
    and returns a dictionary keyed by spectralSpec number (starting at zero), then by
    calibration intent: 'BANDPASS', 'PHASE', etc.
    -Todd Hunter
    """
    if (os.path.exists(asdm) == False):
        print("readTimesFromAnnotationTable(): Could not find ASDM = ", asdm)
        return(None)
    if (os.path.exists(asdm+'/Annotation.xml') == False):
        print("readTimesFromAnnotationTable(): Could not find Annotation.xml. This dataset was probably taken prior to R10.6.")
        return(None)
    print("Observation start date: ", getObservationStartDateFromASDM(asdm)[0])
    print("Representative frequency: %.3f GHz" % (representativeFrequencyFromASDM(asdm,verbose=False)))
    xmlscans = minidom.parse(asdm+'/Annotation.xml')
    scandict = {}
    rowlist = xmlscans.getElementsByTagName("row")
    spws = getScienceSpwsFromASDM(asdm)
    mydict = {}
    previousIntent = None
    spectralSpec = 0
    for i,rownode in enumerate(rowlist):
        row = rownode.getElementsByTagName("issue")
        issue = str(row[0].childNodes[0].nodeValue)
        row = rownode.getElementsByTagName("details")
        details = str(row[0].childNodes[0].nodeValue)
        if issue == "Parameter Optimization":
#            print "Parsing line: ", details
            intent, sourcename, originalIntegration, optimumIntegration, adoptedIntegration = details.split(',')
            intent = intent.upper()
            if intent != previousIntent and previousIntent is not None:
                spectralSpec = 0
            previousIntent = intent
            if spectralSpec not in mydict:
                mydict[spectralSpec] = {}
            elif intent in mydict[spectralSpec]:
                spectralSpec += 1
                if spectralSpec not in mydict:
                    mydict[spectralSpec] = {}
            if intent not in list(mydict[spectralSpec].keys()):
                mydict[spectralSpec][intent] = {'sourcename': sourcename, 'originalIntegration': float(originalIntegration), 
                                                'optimumIntegration': float(optimumIntegration), 'adoptedIntegration': float(adoptedIntegration)}
    return mydict

def plotDecorrelation(asdm, scan=[], ylim=[0,1], plotfile='', doplot=True,
                      baseband=0, field=-1, corrected=False):
    """
    Plots decorrelation vs. baseline length from the CalPhase.xml file.
    baseband: 0 means all, otherwise 1,2,3, or 4
    field: which field to plot (-1 means all)
           This will be used to find the list of scans to plot.
    corrected: if True, show AP_CORRECTED data, otherwise AP_UNCORRECTED
    Returns:
    The median decorrelation over all measurements.
    -Todd Hunter
    """
    mydict = readDecorrelationFromASDM(asdm)
    scandict = readscans(asdm)[0]  # each entry contains key='source' with value=fieldname
    fielddict, fieldnamedict = getFieldsFromASDM(asdm) # [1]: key=fieldname, value=fieldid
    for i in list(scandict.keys()):
        scandict[i]['fieldid'] = fieldnamedict[scandict[i]['source']]
    uniqueFieldIDs = list(fielddict.keys())
    scansforfields = {}
    for f in uniqueFieldIDs:
        scansforfields[f] = []
        for i in list(scandict.keys()):
            if scandict[i]['fieldid'] == f:
                scansforfields[f].append(i)
    if (field != -1 and field not in uniqueFieldIDs):
        print("Field %s is not in the xml file." % (str(field)))
        return
    asdm = asdm.rstrip('/')
    if (mydict == None): return
    if (doplot):
        pb.clf()
    if (scan == []):
        scan = range(len(mydict['baselineLengths']))
    medianDecorrelation = []
    elevation = []
    measurements = 0
#    pb.hold(True) # not needed
    for s in scan:
        if ((corrected and mydict['atmPhaseCorrection'][s] == 'AP_CORRECTED') or
            (not corrected and mydict['atmPhaseCorrection'][s] == 'AP_UNCORRECTED')):
            if (baseband == 0 or baseband == int(mydict['basebandName'][s].split('_')[1])):
                if (field==-1 or s in scansforfields[field]):
                    measurements += 1
                    medianDecorrelation.append(mydict['decorrelationFactor'][s])
                    elevation.append(mydict['elevation'][s])
                    if (doplot):
                        length = len(mydict['decorrelationFactor'][s])
                        numReceptor = mydict['numReceptor'][s]
                        for r in range(numReceptor):
                            # plot the individual pols
                            pb.plot(mydict['baselineLengths'][s],
                                    np.array(mydict['decorrelationFactor'][s])[range(r,length,numReceptor)],
                                    'o',color=overlayColors[r])
    print("Processed %d measurements" % (measurements))
    if doplot:
        pb.xlabel('Baseline length (m)')
        pb.ylabel('Decorrelation')
        if (ylim != [0,0]):
            pb.ylim(ylim)
        if baseband==0:
            basebands = 'all basebands'
        else:
            basebands = 'baseband ' + str(baseband)
        elevationString = 'elev %.0f-%.0f' % (np.min(elevation),np.max(elevation))
        pb.title(os.path.basename(asdm) + '  (%s, %s)' % (basebands,elevationString))
        pb.draw()
        if (plotfile != ''):
            if plotfile==True:
                plotfile = asdm.rstrip('/') + '.decorrelation.png'
            pb.savefig(plotfile)
            print("Plot saved in ", plotfile)
    if (measurements > 0):
        medianDecorrelation = np.median(medianDecorrelation)
    else:
        medianDecorrelation = 0
    return(medianDecorrelation)
    
def readDecorrelationFromASDM(asdm):
    """
    This function reads the decorrelation information from the CalPhase.xml file.
    Returns a dictionary with keys:
    'baselineLengths
    'decorrelationFactor'
    'startValidTime'
    'endValidTime'
    'atmPhaseCorrection'
    -Todd Hunter
    """
    if (asdmLibraryAvailable == False):
        print("The ASDM bindings library is not available on this machine. Using minidom instead.")
        mydict = au_noASDMLibrary.readDecorrelationFromASDM_minidom(asdm)
    else:
        print("A version using ASDM bindings library has not yet been written.  Using minidom instead.")
        mydict = au_noASDMLibrary.readDecorrelationFromASDM_minidom(asdm)
    # insert scan number into dictionary
    scandict = readCalDataFromASDM(asdm)
    mydict['scan'] = mydict['calDataId']
    for i in range(len(mydict['calDataId'])):
        mydict['scan'][i] = scandict[mydict['calDataId'][i]]['scan']
    return(mydict)

def plotSeeing(asdm, plotfile=''):
    """
    Reads and plots the seeing from the CalPhase.xml file of an ASDM.
    -Todd Hunter
    """
    mydict = readSeeingFromASDM(asdm)
    pb.clf()
    adesc = pb.subplot(111)
    c = {'AP_UNCORRECTED': 'r', 'AP_CORRECTED': 'g'}
    nMeasurements = len(mydict['atmPhaseCorrection'])
    x = []
    y = []
    for n in range(nMeasurements):
        x.append(mydict['startValidTime'][n]*1e-9)
        y.append(mydict['seeing'][n])
    print("len(x)=%d, min(x)=%f, max(x)=%f, median=%f" % (len(x), np.min(x), np.max(x), np.median(x)))
#    list_of_date_times = mjdSecondsListToDateTime(x)
#    timeplot = pb.date2num(list_of_date_times)
#    pb.plot_date(timeplot, y, 'o', color = c[mydict['atmPhaseCorrection'][n]])
    pb.plot(x-np.min(x), y, 'o', color = c[mydict['atmPhaseCorrection'][n]])
    x0,x1 = pb.xlim()
    print("x0=%f, x1=%f" % (x0,x1))
    pb.xlim([x0-5, x1+5])
    pb.xlabel('Time')
    ymin,ymax = pb.ylim()
    pb.ylim([0,ymax])
    pb.ylabel('Seeing (arcsec)')
    pb.title(asdm)
    adesc.xaxis.set_major_locator(matplotlib.dates.MinuteLocator(byminute=range(0,60,30)))
    adesc.xaxis.set_minor_locator(matplotlib.dates.MinuteLocator(byminute=range(0,60,10)))
    adesc.xaxis.set_major_formatter(matplotlib.dates.DateFormatter('%H:%M'))
    adesc.fmt_xdata = matplotlib.dates.DateFormatter('%H:%M')
    RescaleXAxisTimeTicks(pb.xlim(), adesc)
    pb.draw()
    if (plotfile != ''):
        if (plotfile == True):
            plotfile = asdm + '.seeing.png'
        pb.savefig(plotfile)
        print("Plot left in ", plotfile)
        
def readSeeingFromASDM(asdm):
    """
    This function reads the seeing information from the CalSeeing.xml file.
    Returns a dictionary with the following keys:
    atmPhaseCorrection: AP_UNCORRECTED or AP_CORRECTED
    baselineLengths: typically 3 values (in meters)
    startValidTime: MJD nano seconds
    endValidTime: MJD nano seconds
    phaseRMS:  a value for each baselineLength (radians?) for each timestamp
    seeing: one value per timestamp (arcseconds)
    - thunter
    """
    import au_noASDMLibrary
    if (asdmLibraryAvailable == False):
        print("The ASDM bindings library is not available on this machine. Using minidom instead.")
        mydict = au_noASDMLibrary.readSeeingFromASDM_minidom(asdm)
        return(mydict)
    print("A version using ASDM bindings library has not yet been written.  Using minidom instead.")
    mydict = au_noASDMLibrary.readSeeingFromASDM_minidom(asdm)
    return(mydict)

def getPolarizationsFromASDM(asdm):
    """
    Reads the Polarization.xml file of an ASDM and reports the number of
    polarization products in each polarization ID.
    -Todd Hunter
    """
    if (not os.path.exists(asdm)):
        print("Could not find ASDM.")
        return
    xmlscans = minidom.parse(asdm+'/Polarization.xml')
    rowlist = xmlscans.getElementsByTagName("row")
    mydict = {}
    for rownode in rowlist:
        rowscan = rownode.getElementsByTagName("numCorr")
        tokens = rowscan[0].childNodes[0].nodeValue.split()
        numCorr = int(tokens[0])
        rowscan = rownode.getElementsByTagName("corrType")
        tokens = rowscan[0].childNodes[0].nodeValue.split()
        corrTypes = [str(i) for i in tokens[2:]]
        rowcaldataid = rownode.getElementsByTagName("polarizationId")
        polarizationId = int(str(rowcaldataid[0].childNodes[0].nodeValue).split('_')[1])
        mydict[polarizationId] =  {'numCorr': numCorr, 'corrTypes': corrTypes}
    return(mydict)

def wvrdata(asdm):
    """
    Checks whether a dataset contains WVR corrected data or not, or both.
    Returns: a tuple of two Booleans (uncorrected present and corrected present)
    -Todd Hunter
    """
    if (not os.path.exists(asdm)):
        print("Could not find ASDM.")
        return
    corrected = False
    uncorrected = False
    for line in open(asdm+'/CalPhase.xml'):
        if 'AP_CORRECT' in line:
            corrected = True
        elif 'AP_UNCORRECT' in line:
            uncorrected = True
    if (uncorrected and not corrected):
        print("This ASDM contains only WVR-uncorrected data.")
    elif (corrected and not uncorrected):
        print("This ASDM contains only WVR-corrected data.")
    else:
        print("This ASDM contains both WVR-uncorrected and corrected data.")
    return(uncorrected, corrected)

def printFluxesFromASDM(sdmfile, sourcename='', field=-1, useCalFlux=False, spw=-1, 
                        asdmspw=-1, showCenterFreqs=False, fluxprec=4, freqprec=1,
                        scienceSpws=False):
    """
    Calls readFluxesFromASDM, organizes the output and prints it to
    the screen.  Normally this function will read from the Source.xml table,
    but setting useCalFlux=True will force it to read from CalFlux.xml.
    spw: if specified, (or if neither spw nor asdmspw are specified