package astri.tcs.TCSHardwareDeviceImpl;

import java.util.ArrayList;
import java.util.TreeMap;

import cern.laser.source.alarmsysteminterface.ASIException;
import cern.laser.source.alarmsysteminterface.AlarmSystemInterface;
import cern.laser.source.alarmsysteminterface.FaultState;

public abstract class AlarmHelper {
	

	
	public AlarmHelper() {
			super();
			
		}
   
    /// Check if an alarm code has been configured.
    /// If code is not configured an error log is sent.
    /// @param code. The alarm code to check.
 
	abstract public boolean findAlarm(int code);

    /// Revert the action done by initializeAlarms
    /// @see initializeAlarms()
    abstract public void resetAlarms();

    /// Initialise the alarm list. This method can be called multiple
    /// times. If alarm codes are repeated the latest setting will take
    /// precedence. It is possible that an alarm that is already active
    /// gets reinitialised with a different fault member name. In this
    /// case the control alarm helper will clear the old alarm.
    /// @param family a string which contains the family of the alarms in
    /// alarmInformation.
    /// @param member a string which identifies the meber of the alarms in
    /// alarmInformation.
    /// @param alarmInformation a vector of AlarmInformation
    /// @see AlarmInformation()
    abstract public void initializeAlarms( String family,
         String member, ArrayList<AlarmInformation> alarmInformation);

    /// Activate an alarm.  Calling this method does not guarantee that
    /// the alarm will be triggered. If the activateCommandCounter for the
    /// specified alarm has not reached the alarmActivateCount, it will
    /// only increment the alarCount but will not be published, nor the
    /// action handler called.
    /// @param code as int.
    abstract public void activateAlarm( int code);

    /// Deactivate an alarm.  Calling this method does not guarantee that
    /// the alarm will be turned off.  If the terminateCommandCounter for
    /// the specified alarm has not reached the alarmDeactivateCount, it
    /// will only increment the terminateCommandCounter but will not be
    /// published, nor the action handler called.
    /// @param code as int.
    abstract public void deactivateAlarm( int code) throws ASIException;

    /// Update an alarm which is triggered by a booleanean value.
    /// This is a helper function that will mimic activateAlarm if the
    /// state is TRUE and deactivateAlarm if the state is FALSE.
    /// @param code as int
    /// @param state as booleanean
    abstract public void updateAlarm( int code,  boolean state);

    
    /// \return true or false depending if the alarm identified by code is
    /// set.
    /// @see activateAlarm(int code)
    /// @see deactivateAlarm(int code)
    abstract public boolean isAlarmSet( int code) ;

    /// Inhibit alarms for a period of time.  During this period of not
    /// alarms will be published, unless forceTerminateAllAlarms() is
    /// called.
    /// @param duration as an ACS::Time.
    /// @see forceTerminateAllAlarms()
    abstract public void inhibitAlarms( long duration);

    /// Reset the alarm inhibition time.  This clears the inhibition time,
    /// even if it has not been reach yet.
    abstract public void resetAlarmsInhibitionTime();

    /// Disable activation or clearing of alarms which are not forced.
    abstract public void disableAlarms();

    /// Enable all alarms.
    abstract public void enableAlarms();

    /// Terminates all active alarms.
    abstract public void terminateAllAlarms();

    /// Forcibly terminates all alarms. This will terminate alarms
    /// regardless of the inhibition time and/or if the alarm is active.
    /// Usually this is used on startup to make sure that not alarms are
    /// active in the AlarmSystem and stayed that way due to a container
    /// crash.
    abstract public void forceTerminateAllAlarms();

    /// \return the alam description.  This description is NOT used by the
    /// alarm system.
    abstract public String getAlarmDescription( int code) ;

    /// \return a composite string of all the active alarm descriptions.
    /// This description is NOT used by the alarm system.
    abstract public String createErrorMessage() ;

    /// Member that gets called when an alarm is effectively activated.
    abstract public void handleActivateAlarm(int code);

    /// Member that gets called when an alarm is effectively deactivated.
    abstract public void handleDeactivateAlarm(int code);


    /// Member that actually sends the alarm to the alarm system and calls
    /// the handler.
    abstract public void send( int code,  boolean activate, boolean force) throws ASIException;

    
}


