/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.archive.database.helpers;

import static java.lang.Math.abs;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Random;

/**
 * Cache for SQL strings with one variable parameter (the key).
 *
 */
public class SQLCache {

    private final Map<String, String> cache;
    
    private final int maxSize;
    
    private final String sqlTemplate;
    
    private final Random rnd;
    
    /**
     * The template must contain exactly one place holder (usually the table
     * name) which will be replaced with the key when a call to 
     * {@link #get(String)} happens. 
     */
    public SQLCache(final String sqlTemplate, final int maxSize) {
       cache = new HashMap<String, String>(maxSize);
       this.maxSize = maxSize;
       this.sqlTemplate = sqlTemplate;
       rnd = new Random();
    }
    
    public String get(final String key) {
        String sql = null;
        synchronized (cache) {
            sql = cache.get(key);
            if (sql == null) {
                sql = String.format(sqlTemplate, key);
                if (cache.size() >= maxSize) {
                    int index = abs(rnd.nextInt() % maxSize);
                    final Iterator<String> iterator = 
                        cache.keySet().iterator();
                    while (index > 1) {
                        iterator.next();
                        index--;
                    }
                    cache.remove(iterator.next());
                }
                cache.put(key, sql);
            }
        }
        return sql;
    }
}
