CREATE TABLE `ComponentType` (
	`ComponentTypeId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`IDL` VARCHAR (256) NOT NULL,
	CONSTRAINT `ComponTAltKey` UNIQUE (`IDL`)
) ENGINE=INNODB;
CREATE TABLE `Configuration` (
	`ConfigurationId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`ConfigurationName` VARCHAR (128) NOT NULL,
	`FullName` VARCHAR (256) NOT NULL,
	`Active` BOOLEAN NOT NULL,
	`CreationTime` TIMESTAMP NOT NULL,
	`Description` MEDIUMTEXT NOT NULL,
	CONSTRAINT `ConfigAltKey` UNIQUE (`ConfigurationName`)
) ENGINE=INNODB;
CREATE TABLE `Schemas` (
	`SchemaId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`URN` VARCHAR (512) NOT NULL,
	`ConfigurationId` INTEGER NOT NULL,
	`Schema` MEDIUMTEXT NULL,
	CONSTRAINT `SchemasConfig` FOREIGN KEY (`ConfigurationId`) REFERENCES `Configuration` (`ConfigurationId`),
	CONSTRAINT `SchemasAltKey` UNIQUE (`URN`, `ConfigurationId`)
) ENGINE=INNODB;
CREATE TABLE `NetworkDevice` (
	`NetworkDeviceId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`NetworkName` VARCHAR (256) NOT NULL,
	`ConfigurationId` INTEGER NOT NULL,
	`PhysicalLocation` VARCHAR (256) NULL,
	`Name` VARCHAR (256) NULL,
	CONSTRAINT `NetworkDeviceConfig` FOREIGN KEY (`ConfigurationId`) REFERENCES `Configuration` (`ConfigurationId`),
	CONSTRAINT `NetworDAltKey` UNIQUE (`NetworkName`, `ConfigurationId`)
) ENGINE=INNODB;
CREATE TABLE `Computer` (
	`NetworkDeviceId` INTEGER,
	`ProcessorType` CHAR (3) NOT NULL,
	`RealTime` BOOLEAN NOT NULL,
	`Diskless` BOOLEAN NOT NULL,
	CONSTRAINT `ChildComputerProcessorType` CHECK (`ProcessorType` IN ('uni', 'smp')),
	CONSTRAINT `ComputerKey` PRIMARY KEY (`NetworkDeviceId`),
	CONSTRAINT `ComputerNetworDFKey` FOREIGN KEY (`NetworkDeviceId`) REFERENCES `NetworkDevice` (`NetworkDeviceId`)
) ENGINE=INNODB;
CREATE TABLE `LoggingConfig` (
	`LoggingConfigId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`MinLogLevelDefault` TINYINT DEFAULT 2,
	`MinLogLevelLocalDefault` TINYINT DEFAULT 2,
	`CentralizedLogger` VARCHAR (16) DEFAULT 'Log',
	`DispatchPacketSize` TINYINT DEFAULT 10,
	`ImmediateDispatchLevel` TINYINT DEFAULT 10,
	`FlushPeriodSeconds` TINYINT DEFAULT 10,
	`MaxLogQueueSize` INTEGER DEFAULT 1000,
	`MaxLogsPerSecond` INTEGER DEFAULT -1
) ENGINE=INNODB;
CREATE TABLE `NamedLoggerConfig` (
	`NamedLoggerConfigId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`LoggingConfigId` INTEGER NOT NULL,
	`Name` VARCHAR (64) NOT NULL,
	`MinLogLevel` TINYINT DEFAULT 2,
	`MinLogLevelLocal` TINYINT DEFAULT 2,
	CONSTRAINT `NamedLoggerConfigLoggingConfig` FOREIGN KEY (`LoggingConfigId`) REFERENCES `LoggingConfig` (`LoggingConfigId`),
	CONSTRAINT `NamedLCAltKey` UNIQUE (`LoggingConfigId`, `Name`)
) ENGINE=INNODB;
CREATE TABLE `Manager` (
	`ManagerId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`ConfigurationId` INTEGER NOT NULL,
	`LoggingConfigId` INTEGER NOT NULL,
	`Startup` VARCHAR (256) NULL,
	`ServiceComponents` VARCHAR (512) NULL,
	`ServiceDaemons` VARCHAR (256) NULL,
	`Timeout` INTEGER DEFAULT 50,
	`ClientPingInterval` INTEGER DEFAULT 60,
	`AdministratorPingInterval` INTEGER DEFAULT 45,
	`ContainerPingInterval` INTEGER DEFAULT 30,
	`ServerThreads` TINYINT DEFAULT 10,
	CONSTRAINT `ManagerLoggingConfig` FOREIGN KEY (`LoggingConfigId`) REFERENCES `LoggingConfig` (`LoggingConfigId`),
	CONSTRAINT `ManagerConfig` FOREIGN KEY (`ConfigurationId`) REFERENCES `Configuration` (`ConfigurationId`),
	CONSTRAINT `ManagerAltKey` UNIQUE (`ConfigurationId`, `LoggingConfigId`, `Startup`, `ServiceComponents`, `Timeout`, `ClientPingInterval`, `AdministratorPingInterval`, `ContainerPingInterval`, `ServerThreads`)
) ENGINE=INNODB;
CREATE TABLE `Container` (
	`ContainerId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`ContainerName` VARCHAR (256) NOT NULL,
	`Path` VARCHAR (256) NOT NULL,
	`ConfigurationId` INTEGER NOT NULL,
	`LoggingConfigId` INTEGER NOT NULL,
	`ImplLang` VARCHAR (6) NOT NULL,
	`RealTime` BOOLEAN DEFAULT FALSE,
	`RealTimeType` VARCHAR (4) DEFAULT 'NONE',
	`KernelModuleLocation` MEDIUMTEXT NULL,
	`KernelModule` MEDIUMTEXT NULL,
	`ComputerId` INTEGER NULL,
	`TypeModifiers` VARCHAR (64) NULL,
	`StartOnDemand` BOOLEAN DEFAULT FALSE,
	`KeepAliveTime` INTEGER DEFAULT -1,
	`ServerThreads` INTEGER DEFAULT 5,
	`ManagerRetry` INTEGER DEFAULT 10,
	`CallTimeout` INTEGER DEFAULT 30,
	`PingInterval` INTEGER NULL,
	`Recovery` BOOLEAN DEFAULT TRUE,
	`AutoloadSharedLibs` VARCHAR (1024) NULL,
	CONSTRAINT `ContainerConfig` FOREIGN KEY (`ConfigurationId`) REFERENCES `Configuration` (`ConfigurationId`),
	CONSTRAINT `ContainerLoggingConfig` FOREIGN KEY (`LoggingConfigId`) REFERENCES `LoggingConfig` (`LoggingConfigId`),
	CONSTRAINT `ContainerComputer` FOREIGN KEY (`ComputerId`) REFERENCES `Computer` (`NetworkDeviceId`),
	CONSTRAINT `ContainerRealTimeType` CHECK (`RealTimeType` IN ('NONE', 'ABM', 'CORR')),
	CONSTRAINT `ContainerImplLang` CHECK (`ImplLang` IN ('java', 'cpp', 'py')),
	CONSTRAINT `ContainerAltKey` UNIQUE (`ContainerName`, `Path`, `ConfigurationId`)
) ENGINE=INNODB;
CREATE TABLE `ContainerStartupOption` (
	`ContStartOptId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`ContainerId` INTEGER NOT NULL,
	`OptionType` VARCHAR (16) NOT NULL,
	`OptionName` VARCHAR (256) NOT NULL,
	`OptionValue` VARCHAR (256) NOT NULL,
	CONSTRAINT `ContStartOptContainer` FOREIGN KEY (`ContainerId`) REFERENCES `Container` (`ContainerId`),
	CONSTRAINT `ContStartOptType` CHECK (`OptionType` IN ('ENV_VAR', 'EXEC_ARG', 'EXEC_ARG_LANG', 'CONT_ARG'))
) ENGINE=INNODB;
CREATE TABLE `Component` (
	`ComponentId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`ComponentTypeId` INTEGER NOT NULL,
	`ComponentName` VARCHAR (256) NOT NULL,
	`ConfigurationId` INTEGER NOT NULL,
	`ContainerId` INTEGER NULL,
	`ImplLang` VARCHAR (6) NOT NULL,
	`RealTime` BOOLEAN NOT NULL,
	`Code` VARCHAR (256) NOT NULL,
	`Path` VARCHAR (256) NOT NULL,
	`IsAutostart` BOOLEAN NOT NULL,
	`IsDefault` BOOLEAN NOT NULL,
	`IsStandaloneDefined` BOOLEAN NULL,
	`IsControl` BOOLEAN NOT NULL,
	`KeepAliveTime` INTEGER NOT NULL,
	`MinLogLevel` TINYINT NOT NULL,
	`MinLogLevelLocal` TINYINT NOT NULL,
	`XMLDoc` MEDIUMTEXT NULL,
	`URN` VARCHAR (512) NULL,
	`ActionThreadStackSize` INTEGER DEFAULT 1024,
	`MonitoringThreadStackSize` INTEGER DEFAULT 2048,
	CONSTRAINT `ComponentIDL` FOREIGN KEY (`ComponentTypeId`) REFERENCES `ComponentType` (`ComponentTypeId`),
	CONSTRAINT `ComponentContainer` FOREIGN KEY (`ContainerId`) REFERENCES `Container` (`ContainerId`),
	CONSTRAINT `ComponentConfig` FOREIGN KEY (`ConfigurationId`) REFERENCES `Configuration` (`ConfigurationId`),
	CONSTRAINT `ComponentImplLang` CHECK (`ImplLang` IN ('java', 'cpp', 'py')),
	CONSTRAINT `ComponentAltKey` UNIQUE (`Path`, `ComponentName`, `ConfigurationId`)
) ENGINE=INNODB;
CREATE TABLE `BACIProperty` (
	`BACIPropertyId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`ComponentId` INTEGER NOT NULL,
	`PropertyName` VARCHAR (128) NOT NULL,
	`description` MEDIUMTEXT NOT NULL,
	`format` VARCHAR (16) NOT NULL,
	`units` VARCHAR (24) NOT NULL,
	`resolution` VARCHAR (10) NOT NULL,
	`archive_priority` INTEGER NOT NULL,
	`archive_min_int` DOUBLE NOT NULL,
	`archive_max_int` DOUBLE NOT NULL,
	`archive_mechanism` VARCHAR (24) NOT NULL,
	`archive_suppress` BOOLEAN NOT NULL,
	`default_timer_trig` DOUBLE NOT NULL,
	`min_timer_trig` DOUBLE NOT NULL,
	`initialize_devio` BOOLEAN NOT NULL,
	`min_delta_trig` DOUBLE NULL,
	`default_value` MEDIUMTEXT NOT NULL,
	`graph_min` DOUBLE NULL,
	`graph_max` DOUBLE NULL,
	`min_step` DOUBLE NULL,
	`archive_delta` DOUBLE NOT NULL,
	`archive_delta_percent` DOUBLE NULL,
	`alarm_high_on` DOUBLE NULL,
	`alarm_low_on` DOUBLE NULL,
	`alarm_high_off` DOUBLE NULL,
	`alarm_low_off` DOUBLE NULL,
	`alarm_timer_trig` DOUBLE NULL,
	`min_value` DOUBLE NULL,
	`max_value` DOUBLE NULL,
	`bitDescription` MEDIUMTEXT NULL,
	`whenSet` MEDIUMTEXT NULL,
	`whenCleared` MEDIUMTEXT NULL,
	`statesDescription` MEDIUMTEXT NULL,
	`condition` MEDIUMTEXT NULL,
	`alarm_on` MEDIUMTEXT NULL,
	`alarm_off` MEDIUMTEXT NULL,
	`alarm_fault_family` MEDIUMTEXT NULL,
	`alarm_fault_member` MEDIUMTEXT NULL,
	`alarm_level` INTEGER NULL,
	`Data` MEDIUMTEXT NULL,
	CONSTRAINT `BACIPropertyCompId` FOREIGN KEY (`ComponentId`) REFERENCES `Component` (`ComponentId`),
	CONSTRAINT `BACIPropArchMech` CHECK (`archive_mechanism` IN ('notification_channel', 'monitor_collector')),
	CONSTRAINT `BACIPropertyAltKey` UNIQUE (`PropertyName`, `ComponentId`)
) ENGINE=INNODB;
CREATE TABLE `Location` (
	`LocationId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`Building` VARCHAR (256) NULL,
	`Floor` VARCHAR (128) NULL,
	`Room` VARCHAR (256) NULL,
	`Mnemonic` VARCHAR (256) NULL,
	`LocationPosition` VARCHAR (256) NULL,
	CONSTRAINT `LocationAltKey` UNIQUE (`Building`, `Floor`, `Room`, `Mnemonic`, `LocationPosition`)
) ENGINE=INNODB;
CREATE TABLE `Contact` (
	`ContactId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`ContactName` VARCHAR (256) NOT NULL,
	`Email` VARCHAR (256) NULL,
	`Gsm` VARCHAR (256) NULL,
	CONSTRAINT `ContactAltKey` UNIQUE (`ContactName`)
) ENGINE=INNODB;
CREATE TABLE `AlarmCategory` (
	`AlarmCategoryId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`AlarmCategoryName` VARCHAR (128) NOT NULL,
	`Description` MEDIUMTEXT NOT NULL,
	`Path` VARCHAR (256) NOT NULL,
	`IsDefault` BOOLEAN NOT NULL,
	`ConfigurationId` INTEGER NOT NULL,
	CONSTRAINT `AlarmCategoryConfig` FOREIGN KEY (`ConfigurationId`) REFERENCES `Configuration` (`ConfigurationId`),
	CONSTRAINT `AlarmCAltKey` UNIQUE (`AlarmCategoryName`, `ConfigurationId`)
) ENGINE=INNODB;
CREATE TABLE `FaultFamily` (
	`FaultFamilyId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`FamilyName` VARCHAR (256) NOT NULL,
	`AlarmSource` VARCHAR (256) DEFAULT 'ALARM_SYSTEM_SOURCES',
	`HelpURL` VARCHAR (256) NULL,
	`ContactId` INTEGER NOT NULL,
	`ConfigurationId` INTEGER NOT NULL,
	CONSTRAINT `FaultFamilyContact` FOREIGN KEY (`ContactId`) REFERENCES `Contact` (`ContactId`),
	CONSTRAINT `FaultFamilyConfig` FOREIGN KEY (`ConfigurationId`) REFERENCES `Configuration` (`ConfigurationId`),
	CONSTRAINT `FaultFamilyAltKey` UNIQUE (`FamilyName`, `ConfigurationId`)
) ENGINE=INNODB;
CREATE TABLE `AlarmCategoryFamily` (
	`AlarmCategoryId` INTEGER NOT NULL,
	`FaultFamilyId` INTEGER NOT NULL,
	CONSTRAINT `ACFCategoryId` FOREIGN KEY (`AlarmCategoryId`) REFERENCES `AlarmCategory` (`AlarmCategoryId`),
	CONSTRAINT `ACFFamilyId` FOREIGN KEY (`FaultFamilyId`) REFERENCES `FaultFamily` (`FaultFamilyId`),
	CONSTRAINT `AlarmCFKey` PRIMARY KEY (`AlarmCategoryId`, `FaultFamilyId`)
) ENGINE=INNODB;
CREATE TABLE `FaultMember` (
	`FaultMemberId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`MemberName` VARCHAR (256) NOT NULL,
	`FaultFamilyId` INTEGER NOT NULL,
	`LocationId` INTEGER NULL,
	CONSTRAINT `FaultMemFamilyRef` FOREIGN KEY (`FaultFamilyId`) REFERENCES `FaultFamily` (`FaultFamilyId`),
	CONSTRAINT `FaultMemLocationRef` FOREIGN KEY (`LocationId`) REFERENCES `Location` (`LocationId`),
	CONSTRAINT `FaultMemberAltKey` UNIQUE (`MemberName`, `FaultFamilyId`)
) ENGINE=INNODB;
CREATE TABLE `DefaultMember` (
	`DefaultMemberId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`FaultFamilyId` INTEGER NOT NULL,
	`LocationID` INTEGER NULL,
	CONSTRAINT `DefaultMemberFaultFamilyRef` FOREIGN KEY (`FaultFamilyId`) REFERENCES `FaultFamily` (`FaultFamilyId`),
	CONSTRAINT `DefaultMemberLocationRef` FOREIGN KEY (`LocationID`) REFERENCES `Location` (`LocationId`),
	CONSTRAINT `DefaulMAltKey` UNIQUE (`FaultFamilyId`)
) ENGINE=INNODB;
CREATE TABLE `FaultCode` (
	`FaultCodeId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`FaultFamilyId` INTEGER NOT NULL,
	`CodeValue` INTEGER NOT NULL,
	`Priority` INTEGER NOT NULL,
	`Cause` VARCHAR (256) NULL,
	`Action` MEDIUMTEXT NULL,
	`Consequence` MEDIUMTEXT NULL,
	`ProblemDescription` MEDIUMTEXT NOT NULL,
	`IsInstant` BOOLEAN NOT NULL,
	CONSTRAINT `CodeFaultFamilyRef` FOREIGN KEY (`FaultFamilyId`) REFERENCES `FaultFamily` (`FaultFamilyId`),
	CONSTRAINT `PriorityValue` CHECK (`Priority` IN (0, 1, 2, 3)),
	CONSTRAINT `FaultCodeAltKey` UNIQUE (`FaultFamilyId`, `CodeValue`)
) ENGINE=INNODB;
CREATE TABLE `AlarmDefinition` (
	`AlarmDefinitionId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`ConfigurationId` INTEGER NOT NULL,
	`FaultFamily` VARCHAR (256) NOT NULL,
	`FaultMember` VARCHAR (256) NOT NULL,
	`FaultCode` VARCHAR (256) NOT NULL,
	CONSTRAINT `AlarmDefinitionConfig` FOREIGN KEY (`ConfigurationId`) REFERENCES `Configuration` (`ConfigurationId`),
	CONSTRAINT `AlarmDAltKey` UNIQUE (`ConfigurationId`, `FaultFamily`, `FaultMember`, `FaultCode`)
) ENGINE=INNODB;
CREATE TABLE `ReductionLink` (
	`ReductionLinkId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`ParentAlarmDefId` INTEGER NOT NULL,
	`ChildAlarmDefId` INTEGER NOT NULL,
	`Type` VARCHAR (12) NOT NULL,
	`Action` VARCHAR (6) NOT NULL,
	`ConfigurationId` INTEGER NOT NULL,
	CONSTRAINT `RLParentRef` FOREIGN KEY (`ParentAlarmDefId`) REFERENCES `AlarmDefinition` (`AlarmDefinitionId`),
	CONSTRAINT `RLChildRef` FOREIGN KEY (`ChildAlarmDefId`) REFERENCES `AlarmDefinition` (`AlarmDefinitionId`),
	CONSTRAINT `ReductionLinkConfig` FOREIGN KEY (`ConfigurationId`) REFERENCES `Configuration` (`ConfigurationId`),
	CONSTRAINT `ReductionLinkType` CHECK (`Type` IN ('MULTIPLICITY', 'NODE')),
	CONSTRAINT `ReductionLinkAction` CHECK (`Action` IN ('CREATE', 'REMOVE')),
	CONSTRAINT `ReductLAltKey` UNIQUE (`ParentAlarmDefId`, `ChildAlarmDefId`)
) ENGINE=INNODB;
CREATE TABLE `ReductionThreshold` (
	`AlarmDefinitionId` INTEGER NOT NULL,
	`Value` INTEGER NOT NULL,
	`ConfigurationId` INTEGER NOT NULL,
	CONSTRAINT `RTAlarmRef` FOREIGN KEY (`AlarmDefinitionId`) REFERENCES `AlarmDefinition` (`AlarmDefinitionId`),
	CONSTRAINT `RTConfig` FOREIGN KEY (`ConfigurationId`) REFERENCES `Configuration` (`ConfigurationId`),
	CONSTRAINT `ReductTKey` PRIMARY KEY (`AlarmDefinitionId`)
) ENGINE=INNODB;
CREATE TABLE `EventChannel` (
	`EventChannelId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`ConfigurationId` INTEGER NOT NULL,
	`Name` VARCHAR (256) NOT NULL,
	`Path` VARCHAR (256) NOT NULL,
	`IntegrationLogs` BOOLEAN DEFAULT FALSE,
	`MaxQueueLength` INTEGER DEFAULT 0,
	`MaxConsumers` INTEGER DEFAULT 0,
	`MaxSuppliers` INTEGER DEFAULT 0,
	`RejectNewEvents` BOOLEAN DEFAULT TRUE,
	`DiscardPolicy` VARCHAR (20) DEFAULT 'AnyOrder',
	`EventReliability` VARCHAR (20) DEFAULT 'BestEffort',
	`ConnectionReliability` VARCHAR (20) DEFAULT 'BestEffort',
	`Priority` SMALLINT DEFAULT 0,
	`Timeout` INTEGER DEFAULT 0,
	`OrderPolicy` VARCHAR (20) DEFAULT 'AnyOrder',
	`StartTimeSupported` BOOLEAN DEFAULT FALSE,
	`StopTimeSupported` BOOLEAN DEFAULT FALSE,
	`MaxEventsPerConsumer` INTEGER DEFAULT 0,
	CONSTRAINT `EventChannelConfig` FOREIGN KEY (`ConfigurationId`) REFERENCES `Configuration` (`ConfigurationId`),
	CONSTRAINT `EventChannelDiscardPolicy` CHECK (`DiscardPolicy` IN ('AnyOrder', 'FifoOrder', 'LifoOrder', 'PriorityOrder', 'DeadlineOrder')),
	CONSTRAINT `EventChannelOrderPolicy` CHECK (`OrderPolicy` IN ('AnyOrder', 'FifoOrder', 'LifoOrder', 'PriorityOrder', 'DeadlineOrder')),
	CONSTRAINT `EventChannelEventReliability` CHECK (`EventReliability` IN ('BestEffort', 'Persistent')),
	CONSTRAINT `EventChannelConReliability` CHECK (`ConnectionReliability` IN ('BestEffort', 'Persistent')),
	CONSTRAINT `EventChannelAltKey` UNIQUE (`Name`, `Path`, `ConfigurationId`)
) ENGINE=INNODB;
CREATE TABLE `Event` (
	`EventId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`EventChannelId` INTEGER NOT NULL,
	`Name` VARCHAR (256) NOT NULL,
	`MaxProcessTime` DOUBLE DEFAULT '2.0',
	CONSTRAINT `EventEventChannelRef` FOREIGN KEY (`EventChannelId`) REFERENCES `EventChannel` (`EventChannelId`),
	CONSTRAINT `EventAltKey` UNIQUE (`EventChannelId`, `Name`)
) ENGINE=INNODB;
CREATE TABLE `NotificationServiceMapping` (
	`NotificationServiceMappingId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`ConfigurationId` INTEGER NOT NULL,
	`DefaultNotificationService` VARCHAR (256) NOT NULL,
	CONSTRAINT `NotServMapConfig` FOREIGN KEY (`ConfigurationId`) REFERENCES `Configuration` (`ConfigurationId`),
	CONSTRAINT `NotifiSMAltKey` UNIQUE (`ConfigurationId`)
) ENGINE=INNODB;
CREATE TABLE `DomainsMapping` (
	`DomainsMappingId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`Name` VARCHAR (256) NOT NULL,
	`NotificationService` VARCHAR (256) NOT NULL,
	`NotificationServiceMappingId` INTEGER NOT NULL,
	CONSTRAINT `DomainsNotServMapRef` FOREIGN KEY (`NotificationServiceMappingId`) REFERENCES `NotificationServiceMapping` (`NotificationServiceMappingId`),
	CONSTRAINT `DomainMAltKey` UNIQUE (`NotificationServiceMappingId`, `Name`)
) ENGINE=INNODB;
CREATE TABLE `ChannelMapping` (
	`ChannelMappingId` INTEGER PRIMARY KEY AUTO_INCREMENT,
	`Name` VARCHAR (256) NOT NULL,
	`NotificationService` VARCHAR (256) NOT NULL,
	`NotificationServiceMappingId` INTEGER NOT NULL,
	CONSTRAINT `ChannelNotServMapRef` FOREIGN KEY (`NotificationServiceMappingId`) REFERENCES `NotificationServiceMapping` (`NotificationServiceMappingId`),
	CONSTRAINT `ChanneMAltKey` UNIQUE (`NotificationServiceMappingId`, `Name`)
) ENGINE=INNODB;
 


