#!/bin/bash
#
# Script to rotate table partitions
# The script should be executed before 12:00, this is the (range) time limit for the new partition, that must exists before
# Any time reference is for the beginning of time period. So the partition with time reference 20081001 starts on 1st October 2008
#
#    ALMA - Atacama Large Millimiter Array
#    (c) European Southern Observatory, 2002
#    Copyright by ESO (in the framework of the ALMA collaboration),
#    All rights reserved
#
#    This library is free software; you can redistribute it and/or
#    modify it under the terms of the GNU Lesser General Public
#    License as published by the Free Software Foundation; either
#    version 2.1 of the License, or (at your option) any later version.
#
#    This library is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    Lesser General Public License for more details.
#
#    You should have received a copy of the GNU Lesser General Public
#    License along with this library; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
#    MA 02111-1307  USA
#
#
# Who       When        What
# --------  ----------  --------------------------------------------------------------
# acheccuc  2009/03/12  First release
# acheccuc  2009/08/26  Modified Temporary Table creation to cope with Virtual Columns
#

# Customizable variables
KEEPBACK="6"
KEEPFORWARD="3"
DPDIR="/backup/expdp/ALMA"

# To be used for debugging purposes
# DEBUG="Y"

# Sanity check
if [ "$#" -ne 4 ]; then
	echo $"Usage: $0 USERNAME PASSWORD TABLE_NAME TBS_BASE_NAME"
	exit 1
fi

# Global variables
ORACLE_USER=$(echo "$1" | tr [a-z] [A-Z])
ORACLE_USER_PWD="$2"
TABLE_NAME=$(echo "$3" | tr [a-z] [A-Z])
TBS_PREFIX=$(echo "$4" | tr [a-z] [A-Z])
SCRIPT_FILE="/tmp/parttbs.sql"
# One month interval
INTERVAL="0-1"

# Directory to store extraction related files
BCKPDIR="${DPDIR}/PARTITIONS_$(date +%Y%m%d)"
mkdir -p $BCKPDIR

# Status file
STATUSFILE="/tmp/parttable_status_$ORACLE_USER_$TABLE_NAME-$(date +%Y%m%d).log"

# Status file cleanup
if [ -f $STATUSFILE ]; then
	rm -f $STATUSFILE
fi

# Check connection to Oracle user
sqlplus -L -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF >/dev/null 2>&1
set feedback off
set heading off
quit
EOF
if [ $? -ne 0 ]; then
	echo "Impossible to connect to $ORACLE_USER Oracle account"
	exit 20
fi

TABLE_EXISTS=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$
set feedback off
set heading off
select count(table_name) from user_tables where table_name='${TABLE_NAME}';
quit
EOF)

if [ $TABLE_EXISTS -ne 1 ]; then
	echo "Table $TABLE_NAME does not exists"
	exit 1
fi

# Tablespace extraction function, 2 arguments needed (OLD_TBS -> Tablespace to extract, OLD_PARTITION -> partition to be rotated)
extract_tablespace()
{
# Sanity Check
if [ $# -ne 3 ]; then
	return 1
fi

OLD_TBS="$1"
OLD_PARTITION="$2"
ORACLE_USER="$3"

# Detect if the tablespace is self-contained
TTS_VIOLATIONS=$(sqlplus -S / as sysdba<<EOF | grep -v ^$
set feedback off
set heading off
delete from TRANSPORT_SET_VIOLATIONS;
exec DBMS_TTS.TRANSPORT_SET_CHECK('$OLD_TBS',TRUE);
SELECT count(*) FROM TRANSPORT_SET_VIOLATIONS;
quit
EOF)

if [ $TTS_VIOLATIONS -ne 0 ]; then
        echo
        echo "Tablespace $OLD_TBS is not self-contained"
        echo "The following objects are still defined on the tablespace (and span more tablespaces)"
        sqlplus -S / as sysdba<<-EOF
        set lines 160
        set feedback off
        select table_owner, table_name, partition_name from DBA_TAB_PARTITIONS where tablespace_name='$OLD_TBS';
        quit
	EOF
	echo 
	return 1
fi

# Altering the old tablespace to read-only. Required for transportable tablespace
sqlplus -S / as sysdba<<EOF
set feedback off
set heading off
ALTER TABLESPACE $OLD_TBS READ ONLY;
quit
EOF

# Creating the trasportable tablespace objects
OLD_DATAFILE=$(sqlplus -S / as sysdba<<EOF | grep -v ^$
set feedback off
set heading off
select FILE_NAME from dba_data_files where tablespace_name='$OLD_TBS';
quit
EOF)

echo -e "=> Creating the transportable tablespace metadata set for $OLD_TBS on file ${OLD_TBS}.dmp\n"

if [ -n "$DEBUG" ]; then
	echo "The following datafiles will be extracted: $OLD_DATAFILE"
fi

# Creating the transportable tablespace metadata set
expdp system/system4dba DUMPFILE=${OLD_TBS}.dmp LOGFILE=${OLD_TBS}.log DIRECTORY=ALMA_DATA_PUMP_DIR TRANSPORT_TABLESPACES=$OLD_TBS
if [ $? -ne 0 ]; then
        echo "Error exporting the tablespace metadata"
        exit 4
fi

echo -e "=> Backing up related datafiles: $OLD_DATAFILE\n and compressing in background"
for datafile in $OLD_DATAFILE
do
	EXPORT_FILENAME=${datafile##*/}

	# Export the datafile from ASM and compress it
	rman target /<<-EOF
	copy datafile '$datafile' to '${BCKPDIR}/${EXPORT_FILENAME}';
	exit;
	EOF

	bzip2 ${BCKPDIR}/${EXPORT_FILENAME} &
done

echo -e "\n=> Removing the tablespace $OLD_TBS that hosted the extracted partition\n"
# Remove the exported tablespace
sqlplus -S / as sysdba<<EOF
set feedback off
set heading off
DROP TABLESPACE $OLD_TBS INCLUDING CONTENTS AND DATAFILES;
quit
EOF

#### Should check if bzip2 finished
# mv -f $STATUSFILE $BCKPDIR
mv -f $DPDIR/${OLD_TBS}.{dmp,log} $BCKPDIR

for datafile in $OLD_DATAFILE
do
	EXPORT_FILENAME=${datafile##*/}
	echo " === Exported filename /backup/expdp/ALMA/$EXPORT_FILENAME === "
done
echo
echo " ==> Please store the previous datafiles, ${BCKPDIR}/${OLD_TBS}.dmp, ${BCKPDIR}/${OLD_TBS}.log and the $(command ls ${BCKPDIR}/$(basename ${STATUSFILE})) files in a safe place <== "
return 0
}

##### MAIN #####
echo -e "=> Partition rotation on table $TABLE_NAME\n"

	echo -e "\t### Oracle User: $ORACLE_USER ###\n"

echo "=====  Partition Rotation  ====="
echo
echo " === Table $TABLE_NAME === "

if [ -n "$DEBUG" ]; then
        echo "Partitions status logfile: ${STATUSFILE}"
fi

# STEP 1: If needed (number of partitions in the future less than KEEPFORWARD) a certain number of partitions will be added to fill the gap between the number of available partition in the future and the number of needed partition as defined by KEEPFORWARD
echo 
echo " ---> ADDITION PHASE ---< "

# Detect the last partition timestamp limit
LAST_PART_DATE=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$ | sed -e 's/TIMESTAMP//g'
set feedback off
set heading off
select HIGH_VALUE from user_tab_partitions where PARTITION_NAME=(select max(PARTITION_NAME) from user_tab_partitions where table_name='${TABLE_NAME}') and table_name='${TABLE_NAME}';
quit
EOF)

# Last partition date upper limit
LAST_PART_DATE=$(echo $LAST_PART_DATE | tr -d "'" | sed 's/ *//')
if [ -n "$DEBUG" ]; then
	echo "Last partition end date (LAST_PART_DATE): values less than $LAST_PART_DATE"
fi
	
# Last partition month (upper limit)
LAST_PART_MONTH=$(echo $LAST_PART_DATE |  tr -d "'" | sed 's/ *[0-9]\{4\}-\([0-1][0-9]\)-[0-3][0-9].*/\1/g')
if [ -n "$DEBUG" ]; then
	echo "Last partition ends at the beginning of month: $LAST_PART_MONTH (LAST_PART_MONTH)"
fi

# Gap (in months) between the date the script is executed and the first partition limit
LAST_PART_GAP=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$ | sed 's/[ \t]*//g'
set feedback off
set heading off
select ROUND(months_between(to_date('$LAST_PART_DATE', 'YYYY-MM-DD HH:MI:SS'),sysdate),'0') from dual;
quit
EOF)

if [ -n "$DEBUG" ]; then
	echo "Last partiton is (ends) $LAST_PART_GAP months in the future"
fi

# Count Indexes defined on partitions (at this point only 1 index permitted)
IDX_COUNT=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$ 
set feedback off
set heading off
select count(index_name) from user_indexes where table_name='${TABLE_NAME}' and index_name not like 'SYS%';
quit
EOF)

if [ $IDX_COUNT -gt 1 ]; then
echo "Error: more than one index defined on the table, not permitted!"
	exit 1
fi

# Detect Index and Index Type defined on partitions
IDX_NAME=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$ 
set feedback off
set heading off
select index_name from user_indexes where table_name='${TABLE_NAME}' and index_name not like 'SYS%';
quit
EOF)

IDX_TYPE=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$ 
set feedback off
set heading off
select uniqueness from user_indexes where index_name='${IDX_NAME}';
quit
EOF)

# Detect colums related to index defined on partitions
IDX_FIELD=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$ 
set feedback off
set heading off
select column_name || ',' from user_ind_columns where index_name='$IDX_NAME';
quit
EOF)

IDX_FIELD=$(echo $IDX_FIELD | sed -e 's/\(.*\)\,$/\1/')

if [ -n "$DEBUG" ]; then
	echo "Index defined on the table: $IDX_NAME ($IDX_FIELD)"
fi

# Save a "snapshot" of the current status of the partitions
touch $STATUSFILE
date >> $STATUSFILE
echo >> $STATUSFILE
sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<-EOF >> $STATUSFILE
set echo on
set pagesize 1000
set linesize 160
set feedback off
SELECT table_name, num_rows, partitioned FROM user_tables WHERE table_name='$TABLE_NAME';
SELECT partition_name, num_rows, tablespace_name FROM user_tab_partitions WHERE table_name='$TABLE_NAME' order by partition_name;
SELECT index_name, partition_name, num_rows, tablespace_name, status FROM user_ind_partitions WHERE index_name='$IDX_NAME' order by partition_name;
EOF

# Loops and add the number of partition needed to have KEEPFORWARD (months) partitions in the future
while [ $KEEPFORWARD -gt $LAST_PART_GAP ]
do
        # Name of last partition (chosen names are lexicographically ordered...)
        LAST_PART_NAME=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$ | sed -e 's/TIMESTAMP//g'
        set feedback off
        set heading off
        select PARTITION_NAME from user_tab_partitions where PARTITION_NAME=(select max(PARTITION_NAME) from user_tab_partitions where table_name='${TABLE_NAME}') and table_name='${TABLE_NAME}';
        quit
        EOF)

        echo "A new partition need to be added beyond $LAST_PART_NAME"

        # Generate a date to be appended to various labels
        DATE_POSTFIX=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$ | sed -e 's/TIMESTAMP//g'
        set feedback off
        set heading off
        select to_char(to_date('$LAST_PART_DATE','YYYY-MM-DD HH:MI:SS')+INTERVAL '$INTERVAL' YEAR TO MONTH, 'YYYYMMDD') from dual;
        quit
        EOF
        )
	
	if [ -n "$DEBUG" ]; then
		echo "This label will be used to append a postfix to both new tablespace and new partition names (keeping lexicographical order): $DATE_POSTFIX (DATE_POSTFIX)"
	fi

	# Calculate new partition end limit date
	NEW_PART_DATE=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$ | sed -e 's/TIMESTAMP//g'
	set feedback off
	set heading off
	select to_char(to_date('$LAST_PART_DATE','YYYY-MM-DD HH:MI:SS')+INTERVAL '$INTERVAL' YEAR TO MONTH, 'YYYY-MM-DD HH:MI:SS') from dual;
	quit
	EOF
	)

	# Generate the new tablespace name and a related datafile name to be added to host the new partition
	TBS_NAME="${TBS_PREFIX}${DATE_POSTFIX}"
	DATAFILE_NAME="$(echo $TBS_NAME | tr [A-Z] [a-z]).dbf"
	PART_NAME="${TABLE_NAME}${DATE_POSTFIX}"

	echo -e "=> New partition time limit (values less than): $NEW_PART_DATE\n"

	echo -e "=> Creating a new partition $PART_NAME for user $ORACLE_USER on tablespace $TBS_NAME (datafile $DATAFILE_NAME)\n"

	# New Tablespace creation script generation\
	echo "whenever sqlerror exit 1" > $SCRIPT_FILE
	echo "CREATE TABLESPACE $TBS_NAME LOGGING DATAFILE '+GROUP3/ALMA/$DATAFILE_NAME' size 2048M AUTOEXTEND ON NEXT 256M MAXSIZE 32767M EXTENT MANAGEMENT LOCAL UNIFORM SIZE 64K SEGMENT SPACE MANAGEMENT AUTO;" >> $SCRIPT_FILE
	echo "quit" >> $SCRIPT_FILE
	
	# Let's check if the tablespace already exixts. If so it will be reused, otherwise created.
	TBS_COUNT=$(sqlplus -S / as sysdba<<EOF | grep -v ^$ | tr -d ' '
	set feedback off
	set heading off
	select count(*) from dba_tablespaces where tablespace_name='$TBS_NAME';
	quit
	EOF
	)

	if [ $TBS_COUNT -eq 0 ]; then
		if [ -n "$DEBUG" ]; then
			echo "### Tablespace creation script ###"
			cat $SCRIPT_FILE
			echo "##################################"
		fi
		echo
		# New tablespace creation script execution
		sqlplus -S / as sysdba @$SCRIPT_FILE
		if [ $? -ne 0 ]; then
			echo "Error creating the new partition tablespace"
			exit 1
		fi
	else
		echo "Tablespace $TBS_NAME already exists... and will not be created"
		echo
	fi
	
	# Grant quota on the new tablespace to each user
	echo "ALTER USER $ORACLE_USER QUOTA UNLIMITED ON $TBS_NAME;" > $SCRIPT_FILE
	echo "quit" >> $SCRIPT_FILE
	if [ -n "$DEBUG" ]; then
		echo "### Alter User for proper quota on the tablespace script ###"
		cat $SCRIPT_FILE
		echo "##################################"
	fi
	# User quota script execution
	sqlplus -S / as sysdba @$SCRIPT_FILE
	if [ $? -ne 0 ]; then
		echo "Error assigning quota to user $ORACLE_USER on tablespace $TBS_NAME"
		exit 1
	fi
	echo

	# Add the partition
	echo "whenever sqlerror exit 1" > $SCRIPT_FILE
	echo "ALTER TABLE $TABLE_NAME ADD PARTITION $PART_NAME VALUES LESS THAN (TO_TIMESTAMP('$NEW_PART_DATE', 'YYYY-MM-DD HH:MI:SS')) TABLESPACE $TBS_NAME UPDATE INDEXES;" >> $SCRIPT_FILE
	echo "quit" >> $SCRIPT_FILE
	if [ -n "$DEBUG" ]; then
		echo "### Partition addition script ###"
		cat $SCRIPT_FILE
		echo "##################################"
	fi
	# Partition addition script execution
	sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD @$SCRIPT_FILE
	if [ $? -ne 0 ]; then
		echo "Error adding the new partition"
		exit 2
	fi

	echo " === Added partition $PART_NAME on tablespace $TBS_NAME (datafile $DATAFILE_NAME) for user $ORACLE_USER === "

	# Recalculate LAST_PART_GAP for loop checking
	LAST_PART_DATE=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$ | sed -e 's/TIMESTAMP//g'
        set feedback off
        set heading off
        select HIGH_VALUE from user_tab_partitions where PARTITION_NAME=(select max(PARTITION_NAME) from user_tab_partitions where table_name='${TABLE_NAME}') and table_name='${TABLE_NAME}';
        quit
        EOF)
        LAST_PART_DATE=$(echo $LAST_PART_DATE | tr -d "'" | sed 's/ *//')

        LAST_PART_GAP=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$ | sed 's/[ \t]*//g'
        set feedback off
        set heading off
        select ROUND(months_between(to_date('$LAST_PART_DATE', 'YYYY-MM-DD HH:MI:SS'),sysdate),'0') from dual;
        quit
        EOF)

        if [ -n "$DEBUG" ]; then
                echo "Last partiton is (ends less than) $LAST_PART_GAP months in the future"
        fi
done


# STEP 2: Check if the number of partitions in the past is greater than KEEPBACK (months), if so proceed to extract any partition older than KEEPBACK months with respect to current month

echo 
echo " ---> EXTRACTION PHASE ---< "

# Detect the first partition timestamp limit
FIRST_PART_DATE=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$ | sed -e 's/TIMESTAMP//g'
set feedback off
set heading off
select HIGH_VALUE from user_tab_partitions where PARTITION_NAME=(select min(PARTITION_NAME) from user_tab_partitions where table_name='${TABLE_NAME}') and table_name='${TABLE_NAME}';
quit
EOF)

# First partition date upper limit
FIRST_PART_DATE=$(echo $FIRST_PART_DATE | tr -d "'" | sed 's/ *//')
if [ -n "$DEBUG" ]; then
	echo "First partition end date (FIRST_PART_DATE): values less than $FIRST_PART_DATE"
fi

# First partition month (upper limit)
FIRST_PART_MONTH=$(echo $FIRST_PART_DATE |  tr -d "'" | sed 's/ *[0-9]\{4\}-\([0-1][0-9]\)-[0-3][0-9].*/\1/g')
if [ -n "$DEBUG" ]; then
	echo "First partition ends at the beginning of month: $FIRST_PART_MONTH (FIRST_PART_MONTH)"
fi

# Gap (in months) between the date the script is executed and the first partition limit 
FIRST_PART_GAP=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$ | sed 's/[ \t]*//g'
set feedback off
set heading off
select ROUND(months_between(sysdate,to_date('$FIRST_PART_DATE', 'YYYY-MM-DD HH:MI:SS')),'0') from dual;
quit
EOF)

if [ -n "$DEBUG" ]; then
	echo "First partiton is (ends) $FIRST_PART_GAP months in the past"
fi

# Loops and extracts any partition older than the KEEPBACK time interval (in months)
while [ $FIRST_PART_GAP -gt $KEEPBACK ]
do
	# Name of first partition (chosen names are lexicographically ordered...)
	FIRST_PART_NAME=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$ | sed -e 's/TIMESTAMP//g'
        set feedback off
        set heading off
        select PARTITION_NAME from user_tab_partitions where PARTITION_NAME=(select min(PARTITION_NAME) from user_tab_partitions where table_name='${TABLE_NAME}') and table_name='${TABLE_NAME}';
        quit
        EOF)
	echo "Rotation needed for partition $FIRST_PART_NAME"

	# Extract the oldest partion (name) for the rotation process
	OLD_PARTITION=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$
	set feedback off
	set heading off
	SELECT min(partition_name) FROM user_tab_partitions WHERE table_name='$TABLE_NAME';
	quit
	EOF)

	if [ -n "$DEBUG" ]; then
		echo "Oldest partition for table $TABLE_NAME: $OLD_PARTITION (OLD_PARTITION)"
	fi

	# Extract the tablespace hosting the oldest partition
	OLD_TBS=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$
	set feedback off
	set heading off
	SELECT tablespace_name FROM user_tab_partitions WHERE table_name='$TABLE_NAME' and partition_name='$OLD_PARTITION';
	quit
	EOF)

	# Extract the tablespace hosting the oldest partition
	TABLE_CONSTRAINT=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<EOF | grep -v ^$
	set feedback off
	set heading off
	select CONSTRAINT_TYPE from user_constraints where TABLE_NAME='$TABLE_NAME' and INDEX_NAME='$IDX_NAME';
	quit
	EOF)

	if [ -n "$DEBUG" ]; then
		echo "Tablespace hosting the oldest partition for table: $OLD_TBS (OLD_TBS)"
	fi

	echo -e "=> Ready to extract the oldest partition $OLD_PARTITION on tablespace $OLD_TBS\n"
	echo -e "=> Creating the temporary exchange table ${OLD_PARTITION}_t_${ORACLE_USER} and related index ${OLD_PARTITION}_i_${ORACLE_USER} on tablespace $OLD_TBS\n"

	# Creating a new temporary table aimed to partition exchange (if virtual columns are involved full DDL script has to be extracted)
	if [ -f /tmp/table_temp.sql ]; then
		rm -f /tmp/table_temp.sql
	fi
	ORACLE_COL_TYPES='CHAR\|VARCHAR2\|VARCHAR\|NCHAR\|NVARCHAR2\|BLOB\|CLOB\|NCLOB\|LONG\|NUMBER\|BINARY_FLOAT\|BINARY_DOUBLE\|DATE\|TIMESTAMP\|RAW'

	VIRTUAL_COLUMNS=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<-EOF | grep -v ^$
	set feedback off
	set heading off
	SELECT count(column_name) from user_tab_cols where table_name = '$TABLE_NAME' AND virtual_column='YES';
	quit
	EOF)

	if [ $VIRTUAL_COLUMNS -ne 0 ]; then
		sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<-EOF | grep -v ^$ > $SCRIPT_FILE
		set feedback off
		set heading off
		set lines 500
		column script format a500
		select dbms_metadata.get_ddl('TABLE','$TABLE_NAME') script from dual;
		quit
		EOF
		echo "whenever sqlerror exit 1" > /tmp/table_temp.sql
		cat $SCRIPT_FILE | egrep 'CREATE *TABLE' | tr -d '"' | sed -e 's/'"$TABLE_NAME"'/'"${OLD_PARTITION}_t_${ORACLE_USER}"'/' >> /tmp/table_temp.sql
		cat $SCRIPT_FILE | grep ','$ | grep 'CHAR\|VARCHAR2\|VARCHAR\|NCHAR\|NVARCHAR2\|BLOB\|CLOB\|NCLOB\|LONG\|NUMBER\|BINARY_FLOAT\|BINARY_DOUBLE\|DATE\|TIMESTAMP\|RAW' | sed -e '$s/\(.*\),$/\1\)/' >> /tmp/table_temp.sql
		echo "TABLESPACE $OLD_TBS;" >> /tmp/table_temp.sql
		mv -f /tmp/table_temp.sql $SCRIPT_FILE
	else
		echo "whenever sqlerror exit 1" > $SCRIPT_FILE
		echo "CREATE TABLE ${OLD_PARTITION}_t_${ORACLE_USER} TABLESPACE $OLD_TBS AS SELECT * FROM $TABLE_NAME WHERE 1=0;" >> $SCRIPT_FILE
	fi
	if [ "$TABLE_CONSTRAINT" = 'P' -o "$IDX_TYPE" = 'UNIQUE' ]; then
		echo "CREATE UNIQUE INDEX ${OLD_PARTITION}_i_${ORACLE_USER} ON ${OLD_PARTITION}_t_${ORACLE_USER}(${IDX_FIELD}) tablespace $OLD_TBS;" >> $SCRIPT_FILE
	else
		echo "CREATE INDEX ${OLD_PARTITION}_i_${ORACLE_USER} ON ${OLD_PARTITION}_t_${ORACLE_USER}(${IDX_FIELD}) tablespace $OLD_TBS;" >> $SCRIPT_FILE
	fi
	echo "quit" >> $SCRIPT_FILE

	if [ -n "$DEBUG" ]; then
		echo "### Temporary table creation script ###"
		cat $SCRIPT_FILE
		echo "#######################################"
	fi

	# Executing the temporary table creation script
	sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD @$SCRIPT_FILE
	if [ $? -ne 0 ]; then
        	echo "Error creating the temporary table"
        	exit 3
	fi

	echo -e "=> Exchanging partition $OLD_PARTITION with the temporary table ${OLD_PARTITION}_t_${ORACLE_USER} including indexes\n"

	# Execute partition exchange with the temporary table
	if [ -n "$DEBUG" ]; then
                echo "Executing: ALTER TABLE $TABLE_NAME EXCHANGE PARTITION $OLD_PARTITION WITH TABLE ${OLD_PARTITION}_t_${ORACLE_USER} INCLUDING INDEXES"
        fi
	sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<-EOF 
	whenever sqlerror exit 2
	set feedback off
	set heading off
	ALTER TABLE $TABLE_NAME EXCHANGE PARTITION $OLD_PARTITION WITH TABLE ${OLD_PARTITION}_t_${ORACLE_USER} INCLUDING INDEXES;
	quit
	EOF
	if [ $? -ne 0 ]; then
        	echo "Error exchanging partition and temporary table"
        	exit 3
	fi

	echo -e "=> Dropping the oldest partition $OLD_PARTITION of table $TABLE_NAME\n"

	# Drop the (now) empty partition exchanged with the temporary table
	if [ -n "$DEBUG" ]; then
                echo "Executing: ALTER TABLE $TABLE_NAME DROP PARTITION $OLD_PARTITION"
        fi
	sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<-EOF
	set feedback off
	set heading off
	ALTER TABLE $TABLE_NAME DROP PARTITION $OLD_PARTITION;
	quit
	EOF

	# Unuseful here, only for echoing purposes
	OLD_DATAFILE=$(sqlplus -S / as sysdba<<-EOF | grep -v ^$
	set feedback off
	set heading off
	select FILE_NAME from dba_data_files where tablespace_name='$OLD_TBS';
	quit
	EOF)

	echo " === Extracted the old partition $OLD_PARTITION on tablespace $OLD_TBS datafile(s) $OLD_DATAFILE === "

	# Real Tablespace extraction (the function will check if it's self contained and so eligible for TTS)
	extract_tablespace $OLD_TBS $OLD_PARTITION $ORACLE_USER
	mv $STATUSFILE $BCKPDIR
	if [ $? -ne 0 ]; then
		echo "An error occurred during TTS set creation. Please Check"
	fi

	# Recalculate FIRST_PART_GAP for loop checking
	FIRST_PART_DATE=$(sqlplus -S ${ORACLE_USER}/$ORACLE_USER_PWD<<-EOF | grep -v ^$ | sed -e 's/TIMESTAMP//g'
        set feedback off
        set heading off
        select HIGH_VALUE from user_tab_partitions where PARTITION_NAME=(select min(PARTITION_NAME) from user_tab_partitions where table_name='${TABLE_NAME}') and table_name='${TABLE_NAME}';
        quit
        EOF)
        FIRST_PART_DATE=$(echo $FIRST_PART_DATE | tr -d "'" | sed 's/ *//')

	FIRST_PART_GAP=$(sqlplus -S $ORACLE_USER/$ORACLE_USER_PWD<<-EOF | grep -v ^$ | sed 's/[ \t]*//g'
        set feedback off
        set heading off
        select ROUND(months_between(sysdate,to_date('$FIRST_PART_DATE', 'YYYY-MM-DD HH:MI:SS')),'0') from dual;
        quit
        EOF)

        if [ -n "$DEBUG" ]; then
                echo "First partiton is (ends) $FIRST_PART_GAP months in the past"
        fi
done

