#!/bin/bash
#
# Script to create partitioned tables
#
# This script will create a partitioned table and related index given a "clean" script is available in the SCRIPTDIR directory and a lump of tablespaces have already being created
# This script will create one partitioned table, the base script must already be available (TABLE_NAME.sql) and contain the create table definition only (no indexes...). Any index (if any...) will be create if separated files of type TABLE_NAME_idx*.sql will stay in the same directory
# 
# TABLE_NAME -> Name of the partitioned table to create (will be used to check for a script called $TABLE_NAME.sql from which to extract the table definition
# TBS_BASE   -> Base name of the tablespaces to be used for partitioning (a timestamp label will be added along the script)
# PART_COL   -> Column (timestamp based) to partition upon
#
#    ALMA - Atacama Large Millimiter Array
#    (c) European Southern Observatory, 2002
#    Copyright by ESO (in the framework of the ALMA collaboration),
#    All rights reserved
#
#    This library is free software; you can redistribute it and/or
#    modify it under the terms of the GNU Lesser General Public
#    License as published by the Free Software Foundation; either
#    version 2.1 of the License, or (at your option) any later version.
#
#    This library is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#    Lesser General Public License for more details.
#
#    You should have received a copy of the GNU Lesser General Public
#    License along with this library; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
#    MA 02111-1307  USA
#
#
# Who       When        What
# --------  ----------  -------------------------------------------------------
# acheccuc  2009/03/12  First release
#

if [ $# -ne 5 ]; then
        echo $"Usage: $0 ORACLE_USER ORACLE_PASSWD TABLE_NAME TBS_BASE_NAME PART_COLUMN"
        exit 1
fi

# Global variables
ORACLE_USER="$1"
ORACLE_PASSWD="$2"
TABLE_NAME=$(echo "$3" | tr [a-z] [A-Z])
TBS_BASE=$(echo $4 | tr [a-z] [A-Z])
PART_COL="$5"

# Customizable variables
SCRIPTDIR="./PartScripts"

# A lump of sanity checks
if [ -z "$TABLE_NAME" ]; then
	echo "Table name can't be empty"
        echo $"Usage: $0 ORACLE_USER ORACLE_PASSWD TABLE_NAME TBS_BASE_NAME PART_COLUMN"
	exit 1
fi

if [ -z "$TBS_BASE" ]; then
        echo "Tablespace base name can't' be empty"
        echo $"Usage: $0 ORACLE_USER ORACLE_PASSWD TABLE_NAME TBS_BASE_NAME PART_COLUMN"
        exit 2
fi

# Check if table script exists
TABLE_SCRIPT=$(command ls -1 $SCRIPTDIR | grep -i ^"${TABLE_NAME}.sql"$)
if [ -z "$TABLE_SCRIPT" ]; then
        echo "Table script for table ${TABLE_NAME} does not exist"
        exit 1
fi
cp -f ${SCRIPTDIR}/${TABLE_SCRIPT} ${SCRIPTDIR}/${TABLE_SCRIPT%%.*}.bak

TABLE_EXISTS=$(sqlplus -S $ORACLE_USER/$ORACLE_PASSWD<<EOF  | grep -v ^$
set feedback off
set heading off
select count(*) from user_tables where table_name='$TABLE_NAME';
quit
EOF)

# If the table exists... cleanup before starting
if [ $TABLE_EXISTS -eq 1 ]; then

	# Collect indexed defined on the table 
	TABLE_IDX=$(sqlplus -S $ORACLE_USER/$ORACLE_PASSWD<<-EOF  | grep -v ^$
	set feedback off
	set heading off
	select index_name from user_indexes where table_name='$TABLE_NAME' and index_name not like 'SYS%';
	quit
	EOF)

	# Drop any index
	for idx in $TABLE_IDX
	do
		sqlplus -S $ORACLE_USER/$ORACLE_PASSWD<<-EOF
		drop index $idx;
		quit
		EOF
	done

	# Drop the table
	sqlplus -S $ORACLE_USER/$ORACLE_PASSWD<<-EOF
	drop table $TABLE_NAME cascade constraints purge;
	quit
	EOF

fi

TMPFILE=$(mktemp || exit 1)
cat ${SCRIPTDIR}/$TABLE_SCRIPT | sed -e 's/^[ \t\n]*$//' | sed -e '/^$/d' > $TMPFILE
BAKIFS=$IFS
IFS=$(echo -en "\n\b")
COMMENT=0
TABLE=0
exec 3<&0
exec 0<$TMPFILE
set -f
while read line
do
	line=$(echo "$line" | sed -e 's/^[ \t]*//' -e 's/[ \t]*$//' | grep -v ^'--' | sed -e 's|^/\*.*\*/$||')
        if echo $line | grep -q ^'/\*'; then
                COMMENT=1
                continue
        elif echo $line | grep -q '\*/'$; then
                COMMENT=0
                continue
        elif [ $COMMENT -eq 1 ]; then
                continue
        fi

        if  echo $line | grep -qi ^'create table' ; then
                TABLE=1
                echo "$line" > ${SCRIPTDIR}/$TABLE_SCRIPT
                continue
        elif (echo $line | grep -q ';'$) && [ $TABLE -eq 1 ] ; then
                TABLE=0
                echo "$line" >> ${SCRIPTDIR}/$TABLE_SCRIPT
                continue
        elif [ $TABLE -eq 1 ]; then
                echo "$line" >> ${SCRIPTDIR}/$TABLE_SCRIPT
                continue
        fi

done
exec 0<&3
set +f

rm $TMPFILE

cat ${SCRIPTDIR}/$TABLE_SCRIPT | grep -v ^$ | grep -v ^'--' | tr -d ';' > /tmp/spoolfile.sql

# Insert partitioning specific parts in the script
echo "PARTITION BY RANGE ($PART_COL) (" >> /tmp/spoolfile.sql

# Create X partitions on related tablespaces covering 6 months in advance
for INTERVAL in $(seq 1 6)
do

TBS_NAME=$(sqlplus -S $ORACLE_USER/$ORACLE_PASSWD<<EOF | grep -v ^$
set feedback off
set heading off
SELECT '$TBS_BASE' || TO_CHAR(TRUNC(SYSDATE+INTERVAL '$INTERVAL' MONTH, 'MM'), 'YYYYMMDD')  FROM dual;
quit
EOF
)

PART_NAME=$(sqlplus -S $ORACLE_USER/$ORACLE_PASSWD<<EOF | grep -v ^$
set feedback off
set heading off
SELECT '$TABLE_NAME' || TO_CHAR(TRUNC(SYSDATE+INTERVAL '$INTERVAL' MONTH, 'MM'), 'YYYYMMDD')  FROM dual;
quit
EOF
)

PART_LIMIT=$(sqlplus -S $ORACLE_USER/$ORACLE_PASSWD<<EOF | grep -v ^$
set feedback off
set heading off
SELECT '''' || TO_CHAR(TRUNC(SYSDATE+INTERVAL '$INTERVAL' MONTH, 'MM'), 'YYYY-MM-DD HH:MI:SS') || '''' FROM dual;
quit
EOF
)

echo "PARTITION $PART_NAME VALUES LESS THAN (TO_TIMESTAMP($PART_LIMIT, 'YYYY-MM-DD HH:MI:SS'))
   TABLESPACE $TBS_NAME," >> /tmp/spoolfile.sql
done

cat /tmp/spoolfile.sql | tr '\n' ' ' > /tmp/tmp.sql
echo >> /tmp/tmp.sql
cat /tmp/tmp.sql | sed -e 's/ \+/ /g' | sed -e 's/,\(.\)$/);\1/g' > /tmp/spoolfile.sql
rm -f  /tmp/tmp.sql 

# Properly create table
sqlplus -S $ORACLE_USER/$ORACLE_PASSWD<<EOF 
set echo on
@/tmp/spoolfile.sql
quit
EOF

# Save the table creation script as modified for partitioning 
cp /tmp/spoolfile.sql ${SCRIPTDIR}/${TABLE_SCRIPT%%.*}.part

# Create related indexes
for idx_script in $(command ls -1 $SCRIPTDIR | egrep -i ^"${TABLE_NAME}_idx_?[1-9]?.sql"$)
do
	cat ${SCRIPTDIR}/$idx_script | grep -v ^$ | grep -v ^'--' | tr -d ';' > /tmp/spoolfile.sql
	echo ' local;' >> /tmp/spoolfile.sql
	cat /tmp/spoolfile.sql | tr '\n' ' ' > /tmp/tmp.sql
	echo >> /tmp/tmp.sql
	cat /tmp/tmp.sql | sed -e 's/ \+/ /g' | sed -e 's/,\(.\)$/);\1/g' > /tmp/spoolfile.sql
	rm -f  /tmp/tmp.sql
	sqlplus -S $ORACLE_USER/$ORACLE_PASSWD<<-EOF 
	set echo on
	@/tmp/spoolfile.sql
	quit
	EOF
done

# Print the result...
sqlplus -S $ORACLE_USER/$ORACLE_PASSWD<<EOF
set echo off
set feedback off
break on table_name
set pages 1000
set lines 160
select table_name, partition_name, tablespace_name from user_tab_partitions where table_name='${TABLE_NAME}' order by 2;
quit
EOF

sqlplus -S $ORACLE_USER/$ORACLE_PASSWD<<EOF
set echo off
set feedback off
break on index_name
set pages 1000
set lines 160
select index_name, partition_name, tablespace_name from user_ind_partitions where index_name=(select index_name from user_indexes where table_name='${TABLE_NAME}'and index_name not like 'SYS%') order by 2;
quit
EOF

# Final cleanup
mv -f ${SCRIPTDIR}/${TABLE_SCRIPT%%.*}.bak ${SCRIPTDIR}/${TABLE_SCRIPT} 
rm -f /tmp/spoolfile.sql

