/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.utils;

import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.DefaultCanAddress;
import alma.obops.tmcdbgui.utils.conversation.BaciConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ContainerConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.DefaultCanAddressConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaType;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.Coordinate;
import alma.tmcdb.domain.HwConfiguration;

public class TestCopyBaseElementUseCase extends AbstractSampleTmcdbTestCase 
{
	private static final String TEST_CLONE2 = "Test-clone12";
	private static final String TEST = "Test";
	private static final String CONTROL = TmcdbConstants.CONTROL_PREFIX;
	private static final String SLASH = TmcdbConstants.SLASH;
	private static final String TEST_CLONE1 = "Test-clone11";

	public void testCopyBaseElementUseCase() 
	   throws Exception
	{	
		List<HwConfiguration> configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(TEST);
		HwConfiguration config = configs.get(0);
		assertNotNull(config);
		int preConfigsSize = configs.size();

		addAntenna(config, DV01);
		HwConfigurationConversationUtils.getInstance().hydrateConfiguration( config );	
		HwConfigurationConversationUtils.getInstance().hydrateBaseElements( config );
		
		assertNotNull(config.getBaseElements());
		assertTrue(config.getBaseElements().size() > 0);
		
		HwConfiguration clonedConfig = HwConfigurationConversationUtils.getInstance().cloneConfiguration(config, TEST_CLONE1);
		assertNotNull(clonedConfig);
		
		configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(TEST_CLONE1);
		assertNotNull(configs);
		assertTrue(0 < configs.size());
		
		clonedConfig = configs.get(0);
		assertNotNull(clonedConfig);

		configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(TEST);
		config = configs.get(0);
		config = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForExport( config );	

		int baseElementsBefore = countAntennas();

		for(BaseElement baseElementToClone: config.getBaseElements()) 
		{
			// iterate over the original config's base elements, looking for antenna DV01
			if( baseElementToClone instanceof Antenna && baseElementToClone.getName().equals(DV01) ) 
			{
				// ok, we found antenna DV01; we will copy it into the cloned configuration, but
				// first, we do some bookkeeping that will be used, later, in our verification steps
				int initialProps = 0;
				int finalProps = 0;

				Collection<Component> compsForDV01 = new HashSet<Component>();
				Collection<Container> totalContsForDV01 = new HashSet<Container>();
				Collection<Container> namedContsForDV01 = new HashSet<Container>();
				Collection<DefaultCanAddress> totalDcasForDV01 = new HashSet<DefaultCanAddress>();

				// additional bookkeeping and hydration
				for(Component c: config.getComponents()) 
				{
					if( c.getPath().contains(DV01) || c.getComponentName().equals(DV01) ) 
					{
						BaciConversationUtils.getInstance().hydrateBACIProperties(c);
						initialProps += c.getBACIProperties().size();
						compsForDV01.add(c);
						totalContsForDV01.add(c.getContainer());
						if(c.getContainer().getPath().contains(DV01)) 
						{
							namedContsForDV01.add(c.getContainer());
						}
					}
				}
				List<DefaultCanAddress> dcas = DefaultCanAddressConversationUtils.getInstance().findAll(config.getSwConfiguration());
				for(DefaultCanAddress dca: dcas) {
					if( dca.getComponent().getPath().contains(DV01) )
						totalDcasForDV01.add(dca);
				}

				HwConfigurationConversationUtils.getInstance().hydrateBaseElements( clonedConfig );

				// now copy the antenna from the original config to the cloned config (to be named DV06 in the cloned config)
				BaseElement be = null;
				try {
					be = BaseElementConversationUtils.getInstance().copyAntenna((Antenna)baseElementToClone, DV06, clonedConfig);
				} catch (Exception e) {
					e.printStackTrace();
					throw e;
				}
				assertNotNull(be);
				assertTrue(be instanceof Antenna);
				assertFalse( be == baseElementToClone );
				assertFalse( be.equals(baseElementToClone) );

				clonedConfig = HwConfigurationConversationUtils.getInstance().findConfigurationById(clonedConfig.getId());
				clonedConfig = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForCloning(clonedConfig);
				assertNotNull(clonedConfig);
				
				// And check that the numbers of items in the clone and in the original are equal
				Collection<Component> compsForDV06 = new HashSet<Component>();
				Collection<Container> totalContsForDV06 = new HashSet<Container>();
				Collection<Container> namedContsForDV06 = new HashSet<Container>();
				Collection<DefaultCanAddress> totalDcasForDV06 = new HashSet<DefaultCanAddress>();

				for(Component c: clonedConfig.getComponents()) 
				{
					if( c.getPath().contains(DV06) || c.getComponentName().equals(DV06) ) 
					{
						finalProps += c.getBACIProperties().size();
						compsForDV06.add(c);

						totalContsForDV06.add(c.getContainer());
						if(c.getContainer().getPath().contains(DV06)) 
						{
							namedContsForDV06.add(c.getContainer());
						}
					}
				}
				List<DefaultCanAddress> dcas2 = DefaultCanAddressConversationUtils.getInstance().findAll(clonedConfig.getSwConfiguration());
				for(DefaultCanAddress dca: dcas2) {
					if( dca.getComponent().getPath().contains(DV06))
						totalDcasForDV06.add(dca);
				}

				assertEquals(compsForDV01.size(), compsForDV06.size());
				assertEquals(totalContsForDV01.size(), totalContsForDV06.size());
				assertEquals(namedContsForDV01.size(), namedContsForDV06.size());
				assertEquals(totalDcasForDV01.size(), totalDcasForDV06.size());
				assertEquals(initialProps, finalProps);
				int baseElementsAfter = countAntennas();
				assertEquals(baseElementsBefore + 1, baseElementsAfter);
				
				configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(TEST);
				config = configs.get(0);
				HwConfigurationConversationUtils.getInstance().hydrateBaseElements( config );
				
				assertEquals(config.getBaseElements().size() + 1, clonedConfig.getBaseElements().size());

				List<HwConfiguration> configsAfter = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(TEST);
				assertEquals(preConfigsSize + 1, configsAfter.size());
				
				return;
			}
		}

		fail("No antenna DV01 found in TMCDB, nothing was actually tested");
	}
	
	public void testCopyBaseElementWithPreexistingItemsUseCase() 
	   throws Exception
	{	
		List<HwConfiguration> configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(TEST);
		HwConfiguration config = configs.get(0);
		assertNotNull(config);
		int preConfigsSize = configs.size();
		
		HwConfigurationConversationUtils.getInstance().hydrateConfiguration( config );	
		HwConfigurationConversationUtils.getInstance().hydrateBaseElements( config );
		
		assertNotNull(config.getBaseElements());
		assertTrue(config.getBaseElements().size() > 0);
		
		HwConfiguration clonedConfig = HwConfigurationConversationUtils.getInstance().cloneConfiguration(config, TEST_CLONE2);
		assertNotNull(clonedConfig);
		
		configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(TEST_CLONE2);
		assertNotNull(configs);
		assertTrue(0 < configs.size());
		
		clonedConfig = configs.get(0);
		assertNotNull(clonedConfig);
		
		int baseElementsBefore = countAntennas();
		Antenna existingAntenna = BaseElementConversationUtils.getInstance().findAntennaByName(clonedConfig.getId(), "DV02");
		int addedAntennas = (existingAntenna == null) ? 1 : 0;
		
		for(BaseElement baseElementToClone: config.getBaseElements()) 
		{
			if( baseElementToClone instanceof Antenna && baseElementToClone.getName().equals(DV01) ) {
				
				int initialComps = 0;
				int initialConts = 0;
				int initialProps = 0;
				int finalComps = 0;
				int finalConts = 0;
				int finalProps = 0;

				// Count the initial comps and conts for the antenna
				Map<String, Component> compsMap = new HashMap<String, Component>();
				HwConfigurationConversationUtils.getInstance().hydrateComponents(config);
				BaciConversationUtils.getInstance().hydrateBACIProperties(config.getSwConfiguration());
				for(Component comp: config.getComponents()) {
					if( comp.getPath().equals(CONTROL + SLASH + DV01) ) {
						initialComps++;
						String key = CONTROL + SLASH + DV02 + comp.getComponentName();
						compsMap.put(key, comp);
					}
				}
				
				Map<String, Container> contsMap = new HashMap<String, Container>();
				ContainerConversationUtils.getInstance().hydrateContainers(config.getSwConfiguration());
				for(Container cont: config.getSwConfiguration().getContainers()) {
					if( cont.getPath().equals(CONTROL + SLASH + DV01) ) {
						initialConts++;
						String key = CONTROL + SLASH + DV02 + cont.getContainerName();
						contsMap.put(key, cont);
					}
				}

				// count any existing components / containers in the destination configuration
				// if (and only if) they weren't already counted - i.e. they aren't in the maps that we build previously.
				clonedConfig = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForCloning(clonedConfig);
				for(Component comp : clonedConfig.getComponents()) {
					if( comp.getPath().equals(CONTROL + SLASH + DV02) ) {
						String key = CONTROL + SLASH + DV02 + comp.getComponentName();
						if(compsMap.get(key) == null) {
							initialComps++;
							initialProps += comp.getBACIProperties().size();
						} else {
							initialProps += compsMap.get(key).getBACIProperties().size();
						}
					}
				}
				for(Container cont: config.getSwConfiguration().getContainers()) {
					if( cont.getPath().equals(CONTROL + SLASH + DV02) ) {
						String key = CONTROL + SLASH + DV02 + cont.getContainerName();
						if(contsMap.get(key) == null) {
							initialConts++;
						}
					}
				}
				
				HwConfigurationConversationUtils.getInstance().hydrateBaseElements( clonedConfig );
				BaseElement be = BaseElementConversationUtils.getInstance().copyAntenna((Antenna)baseElementToClone, DV02, clonedConfig);
				assertNotNull(be);
				assertTrue(be instanceof Antenna);
				
				clonedConfig = HwConfigurationConversationUtils.getInstance().findConfigurationById(clonedConfig.getId());
				clonedConfig = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForCloning(clonedConfig);
				assertNotNull(clonedConfig);
				
				// And check that the numbers are the same
				HwConfigurationConversationUtils.getInstance().hydrateComponents(clonedConfig);
				for(Component comp: clonedConfig.getComponents()) {
					if( comp.getPath().equals(CONTROL + SLASH + DV02) ) {
						finalComps++;
						finalProps += comp.getBACIProperties().size();
					}
				}
				ContainerConversationUtils.getInstance().hydrateContainers(clonedConfig.getSwConfiguration());
				for(Container cont: clonedConfig.getSwConfiguration().getContainers()) {
					if( cont.getPath().equals(CONTROL + SLASH + DV02) ) {
						finalConts++;
					}
				}

				assertEquals(initialComps, finalComps);
				assertEquals(initialConts, finalConts);

				int baseElementsAfter = countAntennas();
				assertEquals(baseElementsBefore + addedAntennas, baseElementsAfter);
				
				HwConfigurationConversationUtils.getInstance().hydrateBaseElements( clonedConfig );
				configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(TEST);
				config = configs.get(0);
				HwConfigurationConversationUtils.getInstance().hydrateBaseElements( config );
				
				assertEquals(config.getBaseElements().size() + addedAntennas, clonedConfig.getBaseElements().size());
				
				List<HwConfiguration> configsAfter = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(TEST);
				assertEquals(preConfigsSize + 1, configsAfter.size());
				
				return;
			}
		}

		fail("No antenna DV01 found in TMCDB, nothing was actually tested");
	}

	private Antenna addAntenna(HwConfiguration config, String antennaName)
	throws Exception 
	{
		// Create and save an antenna with the given info
		Antenna newAntenna = new Antenna();
		newAntenna.setName(antennaName);
		newAntenna.setConfiguration(config);
		newAntenna.setType(BaseElementType.Antenna);
		newAntenna.setAntennaType(AntennaType.VA); // required non-null...
		newAntenna.setPosition(new Coordinate(0,1,2)); // required non-null...
		newAntenna.setOffset(new Coordinate(3,4,5)); // required non-null...

		HwConfigurationConversationUtils.getInstance().hydrateComponentsShallow(config);

		newAntenna.setDiameter(12.0); // required non-null...
		newAntenna.setCommissionDate(new Date().getTime()); // required non-null...
		newAntenna.setAvgDelay(12.4);

		HwConfigurationConversationUtils.getInstance().addBaseElement(config, newAntenna);
		return newAntenna;
	}
}
