/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.utils;

import java.util.List;

import alma.obops.tmcdbgui.utils.conversation.AssemblyConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.BaciConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComponentConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ContainerConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.StartupScenarioConversationUtils;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.StartupScenario;

public class TestCloneConfigurationUseCase extends AbstractSampleTmcdbTestCase 
{
	/**
	 * Tests the use case of cloning an existing HW configuration.
	 * @throws Exception 
	 */
	public void testCloneConfigurationUseCase() throws Exception
	{
		HwConfiguration config = HwConfigurationConversationUtils.getInstance().findConfigurationsByName("Test").get(0);
		assertNotNull(config);
		
		int numLruTypesBefore = countLruTypes();
		int numAssemblyTypesBefore = countAssemblyTypes();
		int numAssemblyRolesBefore = countAssemblyRoles();
		int numComponentsBefore = countComponents();
		int numContainersBefore = countContainers();
		int numAntennasBefore = countAntennas();
		int numPadsBefore = countPads();
		int numAssembliesBefore = countAssemblies();
		int numStartupScenariosBefore = countStartupScenarios();
		int numBaseElementStartupsBefore = countBaseElementStartups();
		int numAssemblyStartupsBefore = countAssemblyStartups();
		int numBaciPropertiesBefore = countBaciProperties();
		int numConfigurationsBefore = countConfigurations();
		int numHwConfigurationsBefore = countHwConfigurations();
		int numPointingModelsBefore = countPointingModels();
		int numDefaultCanAddressesBefore = countDefaultCanAddresses();
		
		// clone the configuration
		HwConfiguration clonedConfig = HwConfigurationConversationUtils.getInstance().cloneConfiguration(config, config.getName() + "-clone");
		assertNotNull(clonedConfig);
		
		// set the clone's name back to the original's name, so that when we compare we don't
		// get a 'false negative' (failure) due to mismatching configuration names
		String clonedName = clonedConfig.getName();
		clonedConfig.setName(config.getName());
		
		logger.info("original config id is: " + config.getId() + "; cloned config id is: " + clonedConfig.getId());
		
		// check that the before & after are equivalent
		HwConfigurationConversationUtils.getInstance().compareConfigurations(config, clonedConfig);
			
		// perform some additional 'sanity' checks on the DB, to verify that we did indeed clone things
		List<HwConfiguration> configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(clonedName);
		assertNotNull(configs);
		assertTrue(1 == configs.size());
		
		int numLruTypesAfter= countLruTypes();
		int numAssemblyTypesAfter = countAssemblyTypes();
		int numAssemblyRolesAfter = countAssemblyRoles();
		int numComponentsAfter = countComponents();
		int numContainersAfter = countContainers();
		int numPadsAfter = countPads();
		int numAssembliesAfter = countAssemblies();
		int numStartupScenariosAfter = countStartupScenarios();
		int numBaseElementStartupsAfter = countBaseElementStartups();
		int numAssemblyStartupsAfter = countAssemblyStartups();
		int numAntennasAfter = countAntennas();
		int numBaciPropertiesAfter = countBaciProperties();
		int numConfigurationsAfter = countConfigurations();
		int numHwConfigurationsAfter = countHwConfigurations();
		int numPointingModelsAfter = countPointingModels();
		int numDefaultCanAddressesAfter = countDefaultCanAddresses();
		
		// global items should *NOT* have been cloned; before count & after count should be the same
		assertEquals(numLruTypesBefore, numLruTypesAfter);
		assertEquals(numAssemblyTypesBefore, numAssemblyTypesAfter);
		assertEquals(numAssemblyRolesBefore, numAssemblyRolesAfter);
		
		ComponentConversationUtils.getInstance().hydrateComponents(config.getSwConfiguration());
		
		// did the number of components double?
		assertEquals(numComponentsBefore + config.getComponents().size(), numComponentsAfter);
		assertTrue(numComponentsBefore > 0);
		assertTrue(numComponentsAfter > 0);
		
		ContainerConversationUtils.getInstance().hydrateContainers(config.getSwConfiguration());
		
		// did the number of containers double?
		assertEquals(numContainersBefore + config.getSwConfiguration().getContainers().size(), numContainersAfter);
		
		HwConfigurationConversationUtils.getInstance().hydrateBaseElements(config);
		
		// did the number of antennas double?
		assertEquals(numAntennasBefore + countAntennasInConfig(config), numAntennasAfter);
		assertTrue(numAntennasBefore > 0);
		assertTrue(numAntennasAfter > 0);
		
		// did the number of pads double?
		assertEquals(numPadsBefore + countPadsInConfig(config), numPadsAfter);
		assertTrue(numPadsBefore > 0);
		assertTrue(numPadsAfter > 0);
		
		BaciConversationUtils.getInstance().hydrateBACIProperties(config.getSwConfiguration());
		
		// did the number of baciproperties double?
		assertEquals(numBaciPropertiesBefore + countBaciPropertiesInConfig(config), numBaciPropertiesAfter);
		
		// did the number of startup scenarios double?
		assertEquals(numStartupScenariosBefore + countStartupScenariosInConfig(config), numStartupScenariosAfter);

		for(StartupScenario scenario : config.getStartupScenarios()) {
			StartupScenarioConversationUtils.getInstance().hydrateBaseElementStartups(scenario);
			StartupScenarioConversationUtils.getInstance().hydrateAssemblyStartups(scenario);
		}
		
		// did the number of base element startups double?
		assertEquals(numBaseElementStartupsBefore + countBaseElementStartupsInConfig(config), numBaseElementStartupsAfter);

		// did the number of assembly startups double?
		assertEquals(numAssemblyStartupsBefore + countAssemblyStartupsInConfig(config), numAssemblyStartupsAfter);
		
		AssemblyConversationUtils.getInstance().hydrateAssemblies(config);
		
		// did the number of assemblies startups double?
		assertEquals(numAssembliesBefore + countAssembliesInConfig(config), numAssembliesAfter);
		
		// did the number of sw configs double?
		assertEquals(numConfigurationsBefore + 1, numConfigurationsAfter);
		assertTrue(numConfigurationsBefore > 0);
		assertTrue(numConfigurationsAfter > 0);
		
		// did the number of hw configs double?
		assertEquals(numHwConfigurationsBefore + 1, numHwConfigurationsAfter);
		assertTrue(numHwConfigurationsBefore > 0);
		assertTrue(numHwConfigurationsAfter > 0);
		
		// did the number of pointingModels double?
		assertEquals(numPointingModelsBefore + countPointingModelsInConfig(config), numPointingModelsAfter);

		// did the number of defaultCanAddresses double?
		assertEquals(numDefaultCanAddressesBefore + countDefaultCanAddressesInConfig(config), numDefaultCanAddressesAfter);

		// TODO: further checks similar to the above, for other domain objects?
		
		logger.info("SLH: globals - should not change before & after...");
		logger.info("-----------------------------------------------------");
		logger.info("SLH: lruTypes before is: " + numLruTypesBefore + " and after is: " + numLruTypesAfter);
		logger.info("SLH: assemblyTypes before is: " + numAssemblyTypesBefore + " and after is: " + numAssemblyTypesAfter);
		logger.info("SLH: assemblyRoles before is: " + numAssemblyRolesBefore + " and after is: " + numAssemblyRolesAfter);
		logger.info("SLH: locals (counts should double after cloning)...");
		logger.info("-----------------------------------------------------");
		logger.info("SLH: components before is: " + numComponentsBefore + " and after is: " + numComponentsAfter);
		logger.info("SLH: containers before is: " + numContainersBefore + " and after is: " + numContainersAfter);
		logger.info("SLH: antennas before is: " + numAntennasBefore + " and after is: " + numAntennasAfter);
		logger.info("SLH: pads before is: " + numPadsBefore + " and after is: " + numPadsAfter);
		logger.info("SLH: startups before is: " + numStartupScenariosBefore + " and after is: " + numStartupScenariosAfter);
		logger.info("SLH: baseElementStartups before is: " + numBaseElementStartupsBefore + " and after is: " + numBaseElementStartupsAfter);
		logger.info("SLH: assemblyStartups before is: " + numAssemblyStartupsBefore + " and after is: " + numAssemblyStartupsAfter);
		logger.info("SLH: assemblies before is: " + numAssembliesBefore + " and after is: " + numAssembliesAfter);
		logger.info("SLH: baciproperties before is: " + numBaciPropertiesBefore + " and after is: " + numBaciPropertiesAfter);
		logger.info("SLH: pointingmodels before is: " + numPointingModelsBefore + " and after is: " + numPointingModelsAfter);
		logger.info("SLH: defaultCanAddresses before is: " + numDefaultCanAddressesBefore + " and after is: " + numDefaultCanAddressesAfter);
		logger.info("SLH: hwconfigs before is: " + numHwConfigurationsBefore + " and after is: " + numHwConfigurationsAfter);
		logger.info("SLH: swconfigs before is: " + numConfigurationsBefore + " and after is: " + numConfigurationsAfter);

	}
}
