/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.utils;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import alma.ReceiverBandMod.ReceiverBand;
import alma.acs.logging.AcsLogLevel;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.DefaultCanAddress;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.DefaultCanAddressConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.FEDelay;
import alma.tmcdb.domain.FocusModel;
import alma.tmcdb.domain.FocusModelCoeff;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.IFDelay;
import alma.tmcdb.domain.LODelay;
import alma.tmcdb.domain.PointingModel;
import alma.tmcdb.domain.PointingModelCoeff;


public class TestCloneBaseElementUseCase extends AbstractSampleTmcdbTestCase 
{
	private static final String TEST = "Test";

	/**
	 * Tests the clone base element use case and check delay / fm / pm for zero vals.
	 * @throws Exception 
	 */
	public void testCloneAntennaAndVerifyDelayPointingFocusValuesUseCase() 
	   throws Exception
	{
		List<HwConfiguration> configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(TEST);
		assertNotNull(configs);
		assertTrue(0 < configs.size());

		HwConfiguration config = configs.get(0);
		assertNotNull(config);

		config = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForExport( config );

		assertNotNull(config.getBaseElements());
		assertTrue(config.getBaseElements().size() > 0);
		
		int preSize = config.getBaseElements().size();
		
		for(BaseElement baseElementToClone: config.getBaseElements()) 
		{
			if( baseElementToClone instanceof Antenna && baseElementToClone.getName().equals(DV01) ) 
			{
				// Clone it
				BaseElement clonedBe = BaseElementConversationUtils.getInstance().cloneBaseElement(baseElementToClone, DV22);
				assertTrue(clonedBe instanceof Antenna);

				// Get the config again from the DB...
				configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(TEST);
				assertNotNull(configs);
				assertTrue(0 < configs.size());

				config = configs.get(0);
				assertNotNull(config);

				// And check that the numbers of items in the clone and in the original are equal
				config = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForExport(config);

				// Verify that we have a new (cloned) antenna
				HwConfigurationConversationUtils.getInstance().hydrateBaseElements(config);
				int postSize = config.getBaseElements().size();
				assertTrue((preSize + 1) == postSize);
				
				// Verify that the cloned antenna has zero's for delay model, pointing model, focus model
				for(BaseElement be : config.getBaseElements()) 
				{
					if(be instanceof Antenna && be.getName().equals(DV22))
					{
						Antenna dv22 = (Antenna) be;
						Set<FEDelay> feDelays = dv22.getFrontEndDelays();
						for(FEDelay fedelay : feDelays) {
							assertEquals(0.0, fedelay.getDelay());
						}
						
						Set<LODelay> loDelays = dv22.getLoDelays();
						for(LODelay lodelay : loDelays) {
							assertEquals(0.0, lodelay.getDelay());
						}
						
						Set<IFDelay> ifDelays = dv22.getIfDelays();
						for(IFDelay ifdelay : ifDelays) {
							assertEquals(0.0, ifdelay.getDelay());
						}
						
						Set<PointingModel> pointingModels = dv22.getPointingModels();
						if(pointingModels.size() > 0) 
						{
							PointingModel pm = pointingModels.iterator().next();
							Map<String, PointingModelCoeff> terms = pm.getTerms();
							for(PointingModelCoeff term : terms.values())
							{
								assertEquals(0.0f, term.getValue());
								Map<ReceiverBand, Double> offsets = term.getOffsets();
								for(Entry<ReceiverBand, Double> entry : offsets.entrySet()) {
									assertEquals(0.0, entry.getValue());
								}
							}
						}
						
						Set<FocusModel> focusModels = dv22.getFocusModels();
						if(focusModels.size() > 0) 
						{
							FocusModel pm = focusModels.iterator().next();
							Map<String, FocusModelCoeff> terms = pm.getTerms();
							for(FocusModelCoeff term : terms.values())
							{
								assertEquals(0.0f, term.getValue());
								Map<ReceiverBand, Double> offsets = term.getOffsets();
								for(Entry<ReceiverBand, Double> entry : offsets.entrySet()) {
									assertEquals(0.0, entry.getValue());
								}
							}
						}
					}
				}
				return;
			}
		}

		fail("No antenna DV01 found in TMCDB, nothing was actually tested");	
	}
	
	/**
	 * Tests the clone base element use case.
	 * @throws Exception 
	 */
	public void testCloneBaseElementUseCase() 
	   throws Exception
	{
		List<HwConfiguration> configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(TEST);
		assertNotNull(configs);
		assertTrue(0 < configs.size());

		HwConfiguration config = configs.get(0);
		assertNotNull(config);

		config = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForExport( config );

		assertNotNull(config.getBaseElements());
		assertTrue(config.getBaseElements().size() > 0);
		
		int preSize = config.getBaseElements().size();
		
		for(BaseElement baseElementToClone: config.getBaseElements()) 
		{
			if( baseElementToClone instanceof Antenna && baseElementToClone.getName().equals(DA41) ) {
				
				int initialProps = 0;
				int finalProps = 0;
			
				// Count the initial comps, conts and DCAs for the antenna
				Collection<Component> compsForDA41 = new HashSet<Component>();
				Collection<Container> totalContsForDA41 = new HashSet<Container>();
				Collection<Container> namedContsForDA41 = new HashSet<Container>();
				Collection<DefaultCanAddress> totalDcasForDA41 = new HashSet<DefaultCanAddress>();
				for(Component c: config.getComponents()) 
				{
					if( c.getPath().contains(DA41) || c.getComponentName().equals(DA41) ) 
					{
						initialProps += c.getBACIProperties().size();
						compsForDA41.add(c);
					}
				}
				for(Component comp : compsForDA41) 
				{
					totalContsForDA41.add(comp.getContainer());
					if(comp.getContainer().getPath().contains(DA41)) 
					{
						namedContsForDA41.add(comp.getContainer());
					}
				}
				List<DefaultCanAddress> dcas = DefaultCanAddressConversationUtils.getInstance().findAll(config.getSwConfiguration());
				for(DefaultCanAddress dca: dcas)
					if( dca.getComponent().getPath().contains(DA41) )
						totalDcasForDA41.add(dca);

				// Clone it
				BaseElement clonedBe = BaseElementConversationUtils.getInstance().cloneBaseElement(baseElementToClone, DA45);
				assertTrue(clonedBe instanceof Antenna);

				// Get the config again from the DB...
				configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(TEST);
				assertNotNull(configs);
				assertTrue(0 < configs.size());

				config = configs.get(0);
				assertNotNull(config);

				// And check that the numbers of items in the clone and in the original are equal
				config = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForExport(config);

				Collection<Component> compsForDA45 = new HashSet<Component>();
				Collection<Container> totalContsForDA45 = new HashSet<Container>();
				Collection<Container> namedContsForDA45 = new HashSet<Container>();
				Collection<DefaultCanAddress> totalDcasForDA45 = new HashSet<DefaultCanAddress>();

				// Check components
				for(Component comp: config.getComponents()) 
				{
					// Original components remain in their containers
					if( comp.getPath().contains(DA41) || comp.getComponentName().equals(DA41) )
						assertFalse( "Original component '" + comp.getPath() + "/" + comp.getComponentName() + "' was moved of container", comp.getContainer().getPath().contains(DA45) );

					if( comp.getPath().contains(DA45) || comp.getComponentName().equals(DA45) ) 
					{
						finalProps += comp.getBACIProperties().size();
						compsForDA45.add(comp);

						// Get the original component for "comp"
						String expectedContName = "";
						for(Component c2: compsForDA41) {
							if( (comp.getComponentName().equals(DA45)) &&
							    (c2.getComponentName().equals(DA41))) { // Antenna component
								expectedContName = c2.getContainer().getPath() + "/" + c2.getContainer().getContainerName();
								break;
							}
							if( comp.getPath().contains(DA45) && c2.getPath().contains(DA41) &&
							    comp.getComponentName().equalsIgnoreCase(c2.getComponentName()) ) {
								expectedContName = c2.getContainer().getPath() + "/" + c2.getContainer().getContainerName();
								break;
							}
						}

						// Those containers which were CONTROL/DA41 now should be CONTROL/DA45
						// Others (like CONTROL/DMC or CONTROL/ATRM should remain the same
						expectedContName = expectedContName.replace(DA41, DA45);
						assertEquals(expectedContName,comp.getContainer().getPath() + "/" + comp.getContainer().getContainerName());

						// Check that the container is in the right computer
						Container cont = comp.getContainer();
						totalContsForDA45.add(cont);
						if(cont.getPath().contains(DA45)) { 
							namedContsForDA45.add(cont);
							assertEquals(DA45.toLowerCase() + "-abm", cont.getComputer().getNetworkName());
						}
					
					}
				}
				List<DefaultCanAddress> dcas2 = DefaultCanAddressConversationUtils.getInstance().findAll(config.getSwConfiguration());
				for(DefaultCanAddress dca: dcas2)
					if( dca.getComponent().getPath().contains(DA45) )
						totalDcasForDA45.add(dca);

				assertEquals(compsForDA41.size(), compsForDA45.size());
				assertEquals(totalContsForDA41.size(), totalContsForDA45.size());
				assertEquals(namedContsForDA41.size(), namedContsForDA45.size());
				assertEquals(totalDcasForDA41.size(), totalDcasForDA45.size());
				assertEquals(initialProps, finalProps);

				HwConfigurationConversationUtils.getInstance().hydrateBaseElements(config);
				int postSize = config.getBaseElements().size();
				assertTrue((preSize + 1) == postSize);
				return;
			}
		}

		fail("No antenna DA41 found in TMCDB, nothing was actually tested");
 	}
	
	/**
	 * Tests the clone base element use case.
	 * @throws Exception 
	 */
	public void testCloneBaseElementWithPreexistingItemsUseCase() 
	   throws Exception
	{
		List<HwConfiguration> configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(TEST);
		assertNotNull(configs);
		assertTrue(0 < configs.size());
		
		HwConfiguration config = configs.get(0);
		assertNotNull(config);
		

		config = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForExport( config );
				
		assertNotNull(config.getBaseElements());
		assertTrue(config.getBaseElements().size() > 0);
		
		int preSize = config.getBaseElements().size();
		
		for(BaseElement baseElementToClone: config.getBaseElements()) 
		{
			if( baseElementToClone instanceof Antenna && baseElementToClone.getName().equals(DA41) ) {
				
				int initialProps = 0;
				int finalProps = 0;
			
				// Count the initial comps and conts for the antenna
				Collection<Component> compsForDA41 = new HashSet<Component>();
				Collection<Container> totalContsForDA41 = new HashSet<Container>();
				Collection<Container> namedContsForDA41 = new HashSet<Container>();
				Collection<DefaultCanAddress> totalDcasForDA41 = new HashSet<DefaultCanAddress>();
				for(Component c: config.getComponents()) 
				{
					if( c.getPath().contains(DA41) || c.getComponentName().equals(DA41) ) 
					{
						initialProps += c.getBACIProperties().size();
						compsForDA41.add(c);
					}
				}
				for(Component comp : compsForDA41) 
				{
					totalContsForDA41.add(comp.getContainer());
					if(comp.getContainer().getPath().contains(DA41)) 
					{
						namedContsForDA41.add(comp.getContainer());
					}
				}
				List<DefaultCanAddress> dcas = DefaultCanAddressConversationUtils.getInstance().findAll(config.getSwConfiguration());
				for(DefaultCanAddress dca: dcas)
					if( dca.getComponent().getPath().contains(DA41) )
						totalDcasForDA41.add(dca);

				// Some logging
				for(Container c: totalContsForDA41)
					logger.log(AcsLogLevel.DEBUG, "Container found for DA41: " + c.getPath() + "/" + c.getContainerName());

				// Clone it
				BaseElement clonedBe = BaseElementConversationUtils.getInstance().cloneBaseElement(baseElementToClone, DA46);
				assertTrue(clonedBe instanceof Antenna);

				// Get the config again from the DB...
				configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(TEST);
				assertNotNull(configs);
				assertTrue(0 < configs.size());
				
				config = configs.get(0);
				assertNotNull(config);
				config = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForExport(config);

				// And check that the numbers of items in the clone and in the original are equal
				Collection<Component> compsForDA46 = new HashSet<Component>();
				Collection<Container> totalContsForDA46 = new HashSet<Container>();
				Collection<Container> namedContsForDA46 = new HashSet<Container>();
				Collection<DefaultCanAddress> totalDcasForDA46 = new HashSet<DefaultCanAddress>();

				// Check components
				for(Component comp: config.getComponents()) 
				{
					// Original components remain in their containers
					if( comp.getPath().contains(DA41) || comp.getComponentName().equals(DA41) )
						assertFalse( comp.getContainer().getPath().contains(DA46) );

					if( comp.getPath().contains(DA46) || comp.getComponentName().equals(DA46) ) 
					{
						finalProps += comp.getBACIProperties().size();
						compsForDA46.add(comp);

						// Get the original component for "comp"
						String expectedContName = "";
						for(Component c2: compsForDA41) {
							if( (comp.getComponentName().equals(DA46)) &&
							    (c2.getComponentName().equals(DA41))) { // Antenna component
								expectedContName = c2.getContainer().getPath() + "/" + c2.getContainer().getContainerName();
								break;
							}
							if( comp.getPath().contains(DA46) && c2.getPath().contains(DA41) &&
							    comp.getComponentName().equalsIgnoreCase(c2.getComponentName()) ) {
								expectedContName = c2.getContainer().getPath() + "/" + c2.getContainer().getContainerName();
								break;
							}
						}

						// Those containers which were CONTROL/DA41 now should be CONTROL/DA46
						// Others (like CONTROL/DMC or CONTROL/ATRM should remain the same
						expectedContName = expectedContName.replace(DA41, DA46);
						assertEquals(expectedContName,comp.getContainer().getPath() + "/" + comp.getContainer().getContainerName());

						// Check that the container is in the right computer
						Container cont = comp.getContainer();
						totalContsForDA46.add(cont);
						if(cont.getPath().contains(DA46)) { 
							namedContsForDA46.add(cont);
							assertEquals(DA46.toLowerCase() + "-abm", cont.getComputer().getNetworkName());
						}
					
					}
				}
				List<DefaultCanAddress> dcas2 = DefaultCanAddressConversationUtils.getInstance().findAll(config.getSwConfiguration());
				for(DefaultCanAddress dca: dcas2)
					if( dca.getComponent().getPath().contains(DA46) )
						totalDcasForDA46.add(dca);

				assertEquals(compsForDA41.size(), compsForDA46.size());
				assertEquals(namedContsForDA41.size(), namedContsForDA46.size());
				assertEquals(totalDcasForDA41.size(), totalDcasForDA46.size());
				assertEquals(initialProps, finalProps);

				HwConfigurationConversationUtils.getInstance().hydrateBaseElements(config);
				int postSize = config.getBaseElements().size();
				assertTrue((preSize + 1) == postSize);
				return;
			}
		}

		fail("No antenna DA41 found in TMCDB, nothing was actually tested");		
 	}
}
