/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.utils;

import java.util.Date;
import java.util.List;
import java.util.Set;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.obops.tmcdbgui.utils.conversation.AssemblyConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComponentConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.StartupScenarioConversationUtils;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaType;
import alma.tmcdb.domain.AssemblyRole;
import alma.tmcdb.domain.AssemblyStartup;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.BaseElementStartupType;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.Coordinate;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.LruType;
import alma.tmcdb.domain.Pad;
import alma.tmcdb.domain.StartupScenario;

public class TestBackendHydrationUtils extends AbstractSampleTmcdbTestCase 
{
	/**
	 * Tests the query configurations use case. 
	 * SLH: updated 03-04-2010 to synchronize with GUI.
	 * @throws Exception 
	 */
	public void testQueryConfigurationsUseCase() 
	   throws Exception
	{
		List<HwConfiguration> configs = queryConfigs();
		assertNotNull(configs);
	}

	/**
	 * Tests the expansion of the configurations tree in the hw view use case. 
	 * SLH: updated 03-04-2010 to synchronize with GUI.
	 * @throws Exception 
	 */
	public void testExpandHwConfigurationsTreeUseCase() 
	   throws Exception
	{
		List<HwConfiguration> configs = queryConfigs();
		
		HwConfiguration configToExpand = configs.get(0);
		expandHwConfiguration(configToExpand);
	}
	
	/**
	 * Tests the expansion of the configurations tree in the hw view use case, and selection of an antenna viewing the 'basic' tab.
	 * SLH: updated 03-04-2010 to synchronize with GUI.
	 * @throws Exception 
	 */
	public void testViewAntennaBasicUseCase() 
	   throws Exception
	{
		List<HwConfiguration> configs = queryConfigs();
		
		HwConfiguration configToExpand = configs.get(0);
		expandHwConfiguration(configToExpand);
		
		Antenna antennaToBrowse = null;
		for(BaseElement baseElement : configToExpand.getBaseElements())
		{
			if(baseElement.getType().equals(BaseElementType.Antenna)) {
				antennaToBrowse = (Antenna) baseElement;
				break;
			}
		}
		assertNotNull(antennaToBrowse);
		
		BaseElementConversationUtils.getInstance().hydrateAntenna(antennaToBrowse);
	}
	
	/**
	 * Tests the expansion of the configurations tree in the hw view use case. 
	 * SLH: updated 03-04-2010 to synchronize with GUI.
	 * @throws Exception 
	 */
	// TODO: reenable (and refactor!) this after CONTROL changes pointing model domain classes
	//       temporarily commenting it out until then
//	public void testViewAntennaPointingModelUseCase() 
//	   throws Exception
//	{
//		List<HwConfiguration> configs = queryConfigs();
//		
//		hydrateConfigsToStartups(configs);
//		
//		HwConfiguration configToExpand = configs.get(0);
//		expandHwConfiguration(configToExpand);
//		
//		Antenna antennaToBrowse = null;
//		for(BaseElement baseElement : configToExpand.getBaseElements())
//		{
//			if(baseElement.getType().equals(BaseElementType.Antenna)) {
//				antennaToBrowse = (Antenna) baseElement;
//				break;
//			}
//		}
//		assertNotNull(antennaToBrowse);
//		
//		BackendConversationUtils.getInstance().hydrateAntennaToPadAssignment(antennaToBrowse);
//		BackendConversationUtils.getInstance().hydrateConfiguration(configToExpand);
//		BackendConversationUtils.getInstance().hydrateAntennaToPointingModels(antennaToBrowse);
//	}
//		
	/**
	 * Test case which hydrates a pad to pointing model depth.
	 * @throws Exception 
	 */
	public void testHydrateAntennaToPointingModel() 
	   throws Exception
	{
		// hard-coding confidId of 0 for testing...
		HwConfiguration config = HwConfigurationConversationUtils.getInstance().findConfigurationsByName("Test").get(0);
		Antenna antenna = BaseElementConversationUtils.getInstance().findAntennaByName(config.getId(), DA41);
		assertNotNull(antenna);
		BaseElementConversationUtils.getInstance().hydrateAntenna(antenna);
	}

	/**
	 * Tests the expansion of the configurations tree in the hw view use case, and selection of a pad viewing the 'basic' tab.
	 * SLH: updated 03-04-2010 to synchronize with GUI.
	 * @throws Exception 
	 */
	public void testViewPadBasicUseCase() 
	   throws Exception
	{
		List<HwConfiguration> configs = queryConfigs();
		
		HwConfiguration configToExpand = configs.get(0);
		expandHwConfiguration(configToExpand);
		
		Pad padToBrowse = null;
		for(BaseElement baseElement : configToExpand.getBaseElements())
		{
			if(baseElement.getType().equals(BaseElementType.Pad)) {
				padToBrowse = (Pad) baseElement;
				break;
			}
		}
		assertNotNull(padToBrowse);
	}
	
	/**
	 * Tests the expansion of the configurations tree in the hw view use case, and selection of a 
	 * pad viewing the 'pointing model' tab & selecting the pointing model in order to see its terms.
	 * 
	 * SLH: updated 03-04-2010 to synchronize with GUI.
	 * @throws Exception 
	 */
	public void testHydratePadUseCase() 
	   throws Exception
	{
		List<HwConfiguration> configs = queryConfigs();
		
		HwConfiguration configToExpand = configs.get(0);
		expandHwConfiguration(configToExpand);
		
		Pad padToBrowse = null;
		// this is an approximation of the GUI use case; it's slightly different in that we're looping, etc.
		for(BaseElement baseElement : configToExpand.getBaseElements())
		{
			if(baseElement.getType().equals(BaseElementType.Pad)) 
			{
				padToBrowse = (Pad) baseElement;
				BaseElementConversationUtils.getInstance().hydratePad(padToBrowse);
			}
		}
	}

	/**
	 * Tests adding a base element startup (simulating, e.g. drag-n-drop of an antenna into a startup scenario).
	 * 
	 * SLH: updated 03-05-2010 to agree with GUI.
	 * @throws Exception 
	 */
	public void testAddBaseElementStartupToStartupScenario()
	   throws Exception
	{
		List<HwConfiguration> configs = queryConfigs();
		
		HwConfiguration configToExpand = configs.get(0);
		expandHwConfiguration(configToExpand);
		
		// create a new antenna; this way we for certain won't 
		// already have a baseelementstartup associated with the antenna
		Antenna antennaToAdd = addAntenna(configToExpand, "DV04");
		assertNotNull(antennaToAdd);
		
		StartupScenarioConversationUtils.getInstance().findLruTypesByBaseElementStartupType(BaseElementStartupType.Antenna);
		Set<StartupScenario> startupScenarios = configToExpand.getStartupScenarios();
		assertNotNull(startupScenarios);
		assertTrue(0 < startupScenarios.size());
		
		StartupScenario scenario = startupScenarios.iterator().next();
		assertNotNull(scenario);
		
		StartupScenarioConversationUtils.getInstance().hydrateBaseElementStartups(scenario);
		int beforeCount = scenario.getBaseElementStartups().size();
		
		BaseElementStartup baseElementStartup = 
			StartupScenarioConversationUtils.getInstance().addBaseElementToStartupScenario(antennaToAdd, scenario);
		
		StartupScenarioConversationUtils.getInstance().hydrateBaseElementStartupToChildren(baseElementStartup);
		
		int afterCount = scenario.getBaseElementStartups().size();
		assertEquals(beforeCount + 1, afterCount);
		
		StartupScenario readBackFromDbScenario = 
			StartupScenarioConversationUtils.getInstance().findStartupScenario(configToExpand, scenario.getName());
		
		HwConfigurationConversationUtils.getInstance().hydrateConfiguration(readBackFromDbScenario.getConfiguration());
		StartupScenarioConversationUtils.getInstance().hydrateBaseElementStartups(readBackFromDbScenario);
		
		assertEquals(beforeCount + 1, readBackFromDbScenario.getBaseElementStartups().size());
	}
	
	/**
	 * Tests adding a new antenna to an existing HW configuration.
	 * 
	 * SLH: updated 03-04-2010 to synchronize with GUI.
	 * @throws Exception 
	 */
	public void testAddAntennaUseCase() throws Exception
	{
		List<HwConfiguration> configs = queryConfigs();
		
		HwConfiguration config = configs.get(0);
		expandHwConfiguration(config);
		assertNotNull(config);
		
		int numAntennasBefore = countAntennas();
		
		Antenna newAnt = addAntenna(config, "DV03");	
		assertNotNull(newAnt);
		
		int numAntennasAfter = countAntennas();
		assertEquals(numAntennasBefore + 1, numAntennasAfter);
	}

	private Antenna addAntenna(HwConfiguration config, String antennaName)
			throws Exception 
	{
		// Create and save an antenna with the given info
		Antenna newAntenna = new Antenna();
		newAntenna.setName(antennaName);
		newAntenna.setConfiguration(config);
		newAntenna.setType(BaseElementType.Antenna);
		newAntenna.setAntennaType(AntennaType.VA); // required non-null...
		newAntenna.setPosition(new Coordinate(0,1,2)); // required non-null...
		newAntenna.setOffset(new Coordinate(3,4,5)); // required non-null...
		newAntenna.setLoOffsettingIndex(1); // required non-null...
		newAntenna.setWalshSeq(1); // required non-null...
		
		HwConfigurationConversationUtils.getInstance().hydrateComponentsShallow(config);
		
		newAntenna.setDiameter(12.0); // required non-null...
		newAntenna.setCommissionDate(new Date().getTime()); // required non-null...
		newAntenna.setAvgDelay(12.4);

		HwConfigurationConversationUtils.getInstance().addBaseElement(config, newAntenna);
		return newAntenna;
	}
	
	/**
	 * Tests adding a new antenna to an existing HW configuration.
	 * @throws Exception 
	 */
	public void testAddPadUseCase() throws Exception
	{
		List<HwConfiguration> configs = queryConfigs();
		
		HwConfiguration config = configs.get(0);
		expandHwConfiguration(config);
		assertNotNull(config);
		
		int numPadsBefore = countPads();
		
		// Create and save a pad with the given info
		Pad newPad = new Pad();
		newPad.setName("w201");
		newPad.setConfiguration(config);
		newPad.setType(BaseElementType.Pad);
		newPad.setPosition(new Coordinate(0,1,2)); // required non-null...
		newPad.setCommissionDate(new Date().getTime()); // required non-null...
		newPad.setAvgDelay(2.3);
			
		HwConfigurationConversationUtils.getInstance().hydrateBaseElements(config);
		AssemblyConversationUtils.getInstance().hydrateAssemblies(config);
		HwConfigurationConversationUtils.getInstance().addBaseElement(config, newPad);
		
		int numPadsAfter = countPads();
		assertTrue(numPadsBefore + 1 == numPadsAfter);
	}

	/**
	 * Tests the browsing of a startup scenario use case.
	 * 
	 * SLH: updated 03-04-2010 to synchronize with GUI.
	 * @throws Exception 
	 */
	public void testBrowseStartupScenarioUseCase()
	   throws Exception
	{
		List<HwConfiguration> configs = queryConfigs();
		
		HwConfiguration configuration = configs.get(0);
		HwConfigurationConversationUtils.getInstance().hydrateConfiguration(configuration);
		
		StartupScenarioConversationUtils.getInstance().findLruTypesByBaseElementStartupType(BaseElementStartupType.Antenna);
		Set<StartupScenario> startupScenarios = configuration.getStartupScenarios();
		assertNotNull(startupScenarios);
		assertTrue(0 < startupScenarios.size());
		
		StartupScenario scenario = startupScenarios.iterator().next();
		assertNotNull(scenario);
		
		StartupScenarioConversationUtils.getInstance().hydrateBaseElementStartups(scenario);
		
		Set<BaseElementStartup> baseElementStartups = scenario.getBaseElementStartups();
		assertNotNull(baseElementStartups);
		assertTrue(0 < baseElementStartups.size());
		BaseElementStartup antennaStartup = null;
		for(BaseElementStartup bes : baseElementStartups) {
			if(bes.getType().equals(BaseElementType.Antenna) || 
			   bes.getBaseElement().getType().equals(BaseElementType.Antenna)) {
				antennaStartup = bes;
				break;
			}
		}
		StartupScenarioConversationUtils.getInstance().hydrateBaseElementStartupToChildren(antennaStartup);		
	}
	
	public void testEditStartupScenario() throws Exception
	{
		List<HwConfiguration> configs = queryConfigs();
		HwConfiguration configuration = configs.get(0);
		
		Set<StartupScenario> startupScenarios = configuration.getStartupScenarios();
		assertNotNull(startupScenarios);
		assertTrue(0 < startupScenarios.size());
		
		StartupScenario scenario = startupScenarios.iterator().next();
		assertNotNull(scenario);
		
		// edit the name of the startup scenario
		scenario.setName(scenario.getName() + " edited");
		
		StartupScenarioConversationUtils.getInstance().saveOrUpdateStartupScenario(scenario);
	}

	@SuppressWarnings("null")
	public void testAddAssemblyStartupToStartupScenario()
	   throws Exception
	{
		List<HwConfiguration> configs = queryConfigs();
		
		HwConfiguration configuration = configs.get(0);
		HwConfigurationConversationUtils.getInstance().hydrateConfiguration(configuration);
		
		LruType[] lruTypes = StartupScenarioConversationUtils.getInstance().findLruTypesByBaseElementStartupType(BaseElementStartupType.Antenna);
		assertNotNull(lruTypes);
		assertTrue(0 < lruTypes.length);

		Set<StartupScenario> startupScenarios = configuration.getStartupScenarios();
		assertNotNull(startupScenarios);
		assertTrue(0 < startupScenarios.size());
		
		StartupScenario scenario = startupScenarios.iterator().next();
		assertNotNull(scenario);
		StartupScenarioConversationUtils.getInstance().hydrateBaseElementStartups(scenario);
		Set<BaseElementStartup> baseElementStartups = scenario.getBaseElementStartups();
		assertNotNull(baseElementStartups);
		assertTrue(0 < baseElementStartups.size());
		
		BaseElementStartup antennaStartup = null;
		for(BaseElementStartup bes : baseElementStartups) {
			if(bes.getType().equals(BaseElementType.Antenna) || 
			   bes.getBaseElement().getType().equals(BaseElementType.Antenna)) {
				antennaStartup = bes;
				break;
			}
		}
		StartupScenarioConversationUtils.getInstance().hydrateBaseElementStartupToChildren(antennaStartup);
		
		// pick the IFProc assemblyrole to designate as an assemblystartup: 
		// NOTE that this is a bit
		// of a hard-code / hack, because if the IFProc doesn't have a component related to it then 
		// this unit test will fail. In the existing sample DB, that is not the case, so it's okay,
		// but should this test fail in the future, consider that possibility
		LruType lruType = null;
		for(LruType lru : lruTypes) {
			if(lru.getName().equals("IFProc")) {
				lruType = lru;
			}
		}

		StartupScenarioConversationUtils.getInstance().hydrateLruType(lruType);
		Set<AssemblyType> assemblyTypes = lruType.getAssemblyTypes();
		assertNotNull(assemblyTypes);
		assertTrue(0 < assemblyTypes.size());
		
		AssemblyType assemblyType = assemblyTypes.iterator().next();
		assertNotNull(assemblyType);
		
		Set<AssemblyRole> assemblyRoles = assemblyType.getRoles();
		assertNotNull(assemblyRoles);
		assertTrue(0 < assemblyRoles.size());
		
		AssemblyRole assemblyRole = assemblyRoles.iterator().next();
		
		ComponentType componentType = assemblyRole.getAssemblyType().getComponentType();
        Component component = ComponentConversationUtils.getInstance().findComponentByComponentTypeId(componentType, configuration);
        assertNotNull(component);
        
		// add an assembly startup
		int numAssemblyStartups = antennaStartup.getAssemblyStartups().size();
		AssemblyStartup assemblyStartup = new AssemblyStartup(antennaStartup, assemblyRole);
		assemblyStartup.setSimulated(false);
		antennaStartup.addAssemblyStartup(assemblyStartup);
		assertEquals(numAssemblyStartups + 1, antennaStartup.getAssemblyStartups().size());
		
		// save the configuration
		HwConfigurationConversationUtils.getInstance().updateConfiguration(configuration);
		
		Long antennaStartupId = antennaStartup.getId();
		
		// verify that things were properly saved
		StartupScenario readBackScenario = StartupScenarioConversationUtils.getInstance().findStartupScenario(scenario.getConfiguration(), scenario.getName());
		assertNotNull(readBackScenario);

		HwConfigurationConversationUtils.getInstance().hydrateConfiguration(readBackScenario.getConfiguration());
		StartupScenarioConversationUtils.getInstance().hydrateBaseElementStartups(readBackScenario);
		
		BaseElementStartup antennaStartupRead = null;
		for(BaseElementStartup bes : readBackScenario.getBaseElementStartups()) {
			if(bes.getId().equals(antennaStartupId)) {
				antennaStartupRead = bes;
				break;
			}
		}
		assertNotNull(antennaStartupRead);
		
		StartupScenarioConversationUtils.getInstance().hydrateAssemblyStartups(readBackScenario);
		
		AssemblyStartup assemblyStartupRead = null;
		for(AssemblyStartup as : antennaStartupRead.getAssemblyStartups()) {
			if(as.getAssemblyRole().equals(assemblyRole)) {
				assemblyStartupRead = as;
				break;
			}
		}
		assertEquals(numAssemblyStartups + 1, antennaStartupRead.getAssemblyStartups().size());
		assertNotNull(assemblyStartupRead);
	}
	
	public void testHydrateBaseElements() throws Exception
	{
		List<HwConfiguration> configs = queryConfigs();
		HwConfiguration configuration = configs.get(0);
		HwConfigurationConversationUtils.getInstance().hydrateConfiguration(configuration);
		HwConfigurationConversationUtils.getInstance().hydrateConfiguration(configuration);
		HwConfigurationConversationUtils.getInstance().hydrateBaseElements(configuration);
		HwConfigurationConversationUtils.getInstance().hydrateBaseElements(configuration);
	}
}
