/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.utils;

import java.lang.reflect.Method;
import java.util.List;
import java.util.Set;

import org.springframework.context.ConfigurableApplicationContext;

import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.DefaultCanAddress;
import alma.obops.dam.testutils.TmcdbTestCase;
import alma.obops.dam.tmcdb.service.AntennaService;
import alma.obops.dam.tmcdb.service.AssemblyRoleService;
import alma.obops.dam.tmcdb.service.AssemblyService;
import alma.obops.dam.tmcdb.service.AssemblyStartupService;
import alma.obops.dam.tmcdb.service.AssemblyTypeService;
import alma.obops.dam.tmcdb.service.BaseElementStartupService;
import alma.obops.dam.tmcdb.service.ComponentService;
import alma.obops.dam.tmcdb.service.ConfigurationService;
import alma.obops.dam.tmcdb.service.ContainerService;
import alma.obops.dam.tmcdb.service.DefaultCanAddressService;
import alma.obops.dam.tmcdb.service.LruTypeService;
import alma.obops.dam.tmcdb.service.PadService;
import alma.obops.dam.tmcdb.service.PointingModelService;
import alma.obops.dam.tmcdb.service.StartupScenarioService;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.obops.tmcdbgui.utils.conversation.AssemblyConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.tmcdb.cloning.CloningTestUtils;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.AssemblyRole;
import alma.tmcdb.domain.AssemblyStartup;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.LruType;
import alma.tmcdb.domain.Pad;
import alma.tmcdb.domain.PointingModel;
import alma.tmcdb.domain.StartupScenario;

public abstract class AbstractSampleTmcdbTestCase extends TmcdbTestCase 
{
	protected static final String DV01 = "DV01";
	protected static final String DV22 = "DV22";
	protected static final String DA41 = "DA41";
	protected static final String DA45 = "DA45";
	protected static final String DA46 = "DA46";
	protected static final String DV02 = "DV02";
	protected static final String DV05 = "DV05";
	protected static final String DV06 = "DV06";

	private ConversationInterceptor conversationInterceptor;
	private AntennaService antennaService;
	private ComponentService componentService;
	private ContainerService containerService;
	private DefaultCanAddressService defaultCanAddressService;
	private PadService padService;
	private AssemblyService assemblyService;
	private StartupScenarioService startupScenarioService;
	private BaseElementStartupService baseElementStartupService;
	private ConfigurationService configurationService;
	private AssemblyStartupService assemblyStartupService;
	private AssemblyRoleService assemblyRoleService;
	private AssemblyTypeService assemblyTypeService;
	private LruTypeService lruTypeService;
	private PointingModelService pointingModelService;

	public void setConversationInterceptor(ConversationInterceptor conversationInterceptor) {
		this.conversationInterceptor = conversationInterceptor;
	}

	public void setAntennaService(AntennaService antennaService) {
		this.antennaService = antennaService;
	}

	public void setComponentService(ComponentService componentService) {
		this.componentService = componentService;
	}

	public void setContainerService(ContainerService containerService) {
		this.containerService = containerService;
	}

	public void setDefaultCanAddressService(DefaultCanAddressService defaultCanAddressService) {
		this.defaultCanAddressService = defaultCanAddressService;
	}

	public void setPadService(PadService padService) {
		this.padService = padService;
	}

	public void setAssemblyService(AssemblyService assemblyService) {
		this.assemblyService = assemblyService;
	}

	public void setStartupScenarioService(StartupScenarioService startupScenarioService) {
		this.startupScenarioService = startupScenarioService;
	}

	public void setBaseElementStartupService(BaseElementStartupService baseElementStartupService) {
		this.baseElementStartupService = baseElementStartupService;
	}

	public void setConfigurationService(ConfigurationService configurationService) {
		this.configurationService = configurationService;
	}

	public void setAssemblyStartupService(AssemblyStartupService assemblyStartupService) {
		this.assemblyStartupService = assemblyStartupService;
	}

	public void setAssemblyRoleService(AssemblyRoleService assemblyRoleService) {
		this.assemblyRoleService = assemblyRoleService;
	}

	public void setAssemblyTypeService(AssemblyTypeService assemblyTypeService) {
		this.assemblyTypeService = assemblyTypeService;
	}

	public void setLruTypeService(LruTypeService lruTypeService) {
		this.lruTypeService = lruTypeService;
	}

	public void setPointingModelService(PointingModelService pointingModelService) {
		this.pointingModelService = pointingModelService;
	}

	@Override
	protected void onTearDownInTransaction() throws Exception
	{
		logger.info("tearDown: starting");
		logger.info("tearDown: cleaning up sample tmcdb database");
		CloningTestUtils.removeSampleTmcdbDatabase();
		logger.info("tearDown: ending");
	}

	@Override
	protected ConfigurableApplicationContext createApplicationContext(String[] locations) {
		try {
			CloningTestUtils.unzipSampleTmcdbDatabase();
	    	CloningTestUtils.untarSampleTmcdbDatabase();
			BackendUtils.initializeBackendForTesting(logger);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return super.createApplicationContext(locations);
	}

	@Override
	protected void onSetUpInTransaction() throws Exception {
		// no-op
	}

	protected int countAntennas() 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountAntennas", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		return retVal;
	}
	
	// must be public
	public ConversationTokenProvider privateCountAntennas(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<Antenna> antennas = antennaService.findAll();
		resultHolder.setValue(antennas.size());		
		return retVal;
	}
	
/********************************** utility methods ***************************************/ 
	
	protected int countAntennasInConfig(HwConfiguration config) 
	{
		int retVal = 0;
		
		for(BaseElement be : config.getBaseElements()) {
			if(be.getType().equals(BaseElementType.Antenna)) {
				retVal++;
			}
		}
		
		return retVal;
	}
	
	protected int countPointingModelsInConfig(HwConfiguration config) throws Exception 
	{
		int retVal = 0;
		
		for(BaseElement be: config.getBaseElements()) 
		{
			if(be.getType().equals(BaseElementType.Antenna)) {
				Antenna antenna = (Antenna)be;
				BaseElementConversationUtils.getInstance().hydrateAntenna(antenna);
				retVal += antenna.getPointingModels().size();
			}
		}
		
		return retVal;
	}
	
	protected int countAssembliesInConfig(HwConfiguration config) 
	{
		int retVal = 0;
		
		retVal = config.getAssemblies().size();
		
		return retVal;
	}
	
	protected int countStartupScenariosInConfig(HwConfiguration config) 
	{
		int retVal = 0;
		
		retVal = config.getStartupScenarios().size();
		
		return retVal;
	}
	
	protected int countBaseElementStartupsInConfig(HwConfiguration config) 
	{
		int retVal = 0;
		
		for(StartupScenario scenario : config.getStartupScenarios()) 
		{
			retVal = countBaseElementStartupsInSet(scenario.getBaseElementStartups());
		}
		
		return retVal;
	}
	
	protected int countBaseElementStartupsInSet(Set<BaseElementStartup> toCount)
	{
		int retVal = 0;
		for(BaseElementStartup bestartup : toCount) 
		{
			retVal++;
			if(bestartup.getChildren() != null && bestartup.getChildren().size() > 0) 
			{
				retVal += countBaseElementStartupsInSet(bestartup.getChildren());
			}
		}
		return retVal;
	}
	
	protected int countAssemblyStartupsInConfig(HwConfiguration config) 
	{
		int retVal = 0;
		
		for(StartupScenario scenario : config.getStartupScenarios()) 
		{
			retVal += countAssemblyStartupsInSet(scenario.getBaseElementStartups());
		}
		
		return retVal;
	}
	
	protected int countAssemblyStartupsInSet(Set<BaseElementStartup> toCount)
	{
		int retVal = 0;
		for(BaseElementStartup bestartup : toCount) 
		{
			retVal += bestartup.getAssemblyStartups().size();
			if(bestartup.getChildren() != null && bestartup.getChildren().size() > 0) 
			{
				retVal += countAssemblyStartupsInSet(bestartup.getChildren());
			}
		}
		return retVal;
	}
	
	protected int countPadsInConfig(HwConfiguration config) 
	{
		int retVal = 0;
		
		for(BaseElement be : config.getBaseElements()) {
			if(be.getType().equals(BaseElementType.Pad)) {
				retVal++;
			}
		}
		
		return retVal;
	}

	protected int countBaciPropertiesInConfig(HwConfiguration config) 
	{
		int retVal = 0;
		
		for(Component comp : config.getComponents()) {
			retVal += comp.getBACIProperties().size();
		}
		
		return retVal;
	}
	
	protected int countStartupScenarios() 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountStartupScenarios", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}	
	
	protected int countPointingModels() 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountPointingModels", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}	
	
	protected int countBaseElementStartups() 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountBaseElementStartups", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}	
	
	protected int countComponents() 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountComponents", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}	
	
	protected int countContainers() 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountContainers", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}

	
	protected int countDefaultCanAddresses() 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountDefaultCanAddresses", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}

	protected int countDefaultCanAddressesInConfig(HwConfiguration config) 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountDefaultCanAddressesInConfig", HwConfiguration.class, IntegerResultHolder.class);
		Object[] args = new Object[2];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = config;
		args[1] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}

	protected int countAssemblyStartups() 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountAssemblyStartups", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}	
	
	protected int countAssemblies() 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountAssemblies", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}	
	
	

	protected int countPads()
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountPads", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}
	
	protected int countHwConfigurations() 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountHwConfigurations", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}
	
	protected int countBaciProperties() 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountBACIProperties", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}
	
	protected int countLruTypes() 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountLruTypes", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}
	
	protected int countAssemblyTypes() 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountAssemblyTypes", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}
	
	protected int countAssemblyRoles() 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountAssemblyRoles", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}
	
	protected int countConfigurations() 
	   throws Exception 
	{
		int retVal = 0;
		
		Method methodToInvoke = TestBackendHydrationUtils.class.getMethod("privateCountConfigurations", IntegerResultHolder.class);
		Object[] args = new Object[1];
		IntegerResultHolder resultHolder =  new IntegerResultHolder();
		args[0] = resultHolder;
		
		conversationInterceptor.invoke(methodToInvoke, this, args);
		
		retVal = resultHolder.getValue();
		
		return retVal;
	}

	protected List<HwConfiguration> queryConfigs() 
	   throws Exception 
	{
		List<HwConfiguration> configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName("");
		assertNotNull(configs);
		assertTrue(0 < configs.size());
		return configs;
	}
	
	// must be public
	public ConversationTokenProvider privateCountContainers(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<Container> containers = containerService.findAll();
		resultHolder.setValue(containers.size());		
		return retVal;
	}
	
	// must be public
	public ConversationTokenProvider privateCountDefaultCanAddresses(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<DefaultCanAddress> defaultCanAddresses = defaultCanAddressService.findAll();
		resultHolder.setValue(defaultCanAddresses.size());		
		return retVal;
	}

	// must be public
	public ConversationTokenProvider privateCountDefaultCanAddressesInConfig(HwConfiguration config, IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<DefaultCanAddress> defaultCanAddresses = defaultCanAddressService.findAll(config.getSwConfiguration());
		resultHolder.setValue(defaultCanAddresses.size());		
		return retVal;
	}

	// must be public
	public ConversationTokenProvider privateCountPads(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<Pad> pads = padService.findAll();
		resultHolder.setValue(pads.size());		
		return retVal;
	}
	
	// must be public
	public ConversationTokenProvider privateCountAssemblies(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<Assembly> assemblies = assemblyService.findAll();
		resultHolder.setValue(assemblies.size());		
		return retVal;
	}
	
	// must be public
	public ConversationTokenProvider privateCountStartupScenarios(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<StartupScenario> startups = startupScenarioService.findAll();
		resultHolder.setValue(startups.size());		
		return retVal;
	}
	
	// must be public
	public ConversationTokenProvider privateCountBaseElementStartups(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<BaseElementStartup> startups = baseElementStartupService.findAll();
		resultHolder.setValue(startups.size());		
		return retVal;
	}

	// must be public
	public ConversationTokenProvider privateCountBACIProperties(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<BACIProperty> properties = componentService.findAllBaciProperties();
		resultHolder.setValue(properties.size());		
		return retVal;
	}

	// must be public
	public ConversationTokenProvider privateCountConfigurations(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<Configuration> properties = configurationService.findAllSwConfigurations();
		resultHolder.setValue(properties.size());		
		return retVal;
	}

	// is 'private' but due to interceptor, must be public
	public ConversationTokenProvider privateCountHwConfigurations(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<HwConfiguration> properties = configurationService.findAll();
		resultHolder.setValue(properties.size());		
		return retVal;
	}
	
	// is 'private' but due to interceptor, must be public
	public ConversationTokenProvider privateCountAssemblyStartups(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<AssemblyStartup> assemblystartups = assemblyStartupService.findAll();
		resultHolder.setValue(assemblystartups.size());		
		return retVal;
	}
	
	// is 'private' but due to interceptor, must be public
	public ConversationTokenProvider privateCountAssemblyRoles(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<AssemblyRole> assemblyRoles = assemblyRoleService.findAll();
		resultHolder.setValue(assemblyRoles.size());		
		return retVal;
	}
	
	// is 'private' but due to interceptor, must be public
	public ConversationTokenProvider privateCountAssemblyTypes(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<AssemblyType> assemblyTypes = assemblyTypeService.findAll();
		resultHolder.setValue(assemblyTypes.size());		
		return retVal;
	}
	
	// is 'private' but due to interceptor, must be public
	public ConversationTokenProvider privateCountLruTypes(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<LruType> lruTypes = lruTypeService.findAll();
		resultHolder.setValue(lruTypes.size());		
		return retVal;
	}
	
	// must be public
	public ConversationTokenProvider privateCountComponents(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<Component> components = componentService.findAll();
		resultHolder.setValue(components.size());		
		return retVal;
	}
	
	// must be public
	public ConversationTokenProvider privateCountPointingModels(IntegerResultHolder resultHolder) 
	{
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		List<PointingModel> pms = pointingModelService.findAll();
		resultHolder.setValue(pms.size());		
		return retVal;
	}
	
	protected void expandHwConfiguration(HwConfiguration configToExpand)
	throws Exception 
	{
		HwConfigurationConversationUtils.getInstance().hydrateBaseElements(configToExpand);
		AssemblyConversationUtils.getInstance().hydrateAssemblies(configToExpand);
	}
	
	private class IntegerResultHolder
	{
		private Integer value;
		
		public Integer getValue() { return value; }
		public void setValue(Integer val) { this.value = val; }
		
	}
}
