/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * 
 */
package alma.obops.tmcdbgui.handlers;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.AcsServiceServiceType;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.ComputerProcessorType;
import alma.acs.tmcdb.NetworkDevice;
import alma.obops.dam.tmcdb.domain.TMCDBExport;
import alma.obops.tmcdbgui.utils.AbstractSampleTmcdbTestCase;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.utils.DatetimeUtils;
import alma.obops.utils.DatetimeUtils.Style;
import alma.tmcdb.cloning.CloningTestUtils;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.AntennaType;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.Coordinate;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.Pad;
import alma.tmcdb.domain.StartupScenario;

/**
 * Tests the XML Export feature for a Configuration
 * 
 * @author rtobar
 */
public class ConfigurationExportActionTest extends AbstractSampleTmcdbTestCase {

	private static final String XML_FILE = "exported-conf.xml";
	private static final String XML_FILE2 = "external-configuration.xml";

	private static HwConfiguration _prevConfig;

	public void testConfigurationExport() throws Exception {

		HwConfiguration conf;
		conf = HwConfigurationConversationUtils.getInstance().findConfigurationsByName("Test").get(0);

		logger.info("Starting to fully hydrate the HW Configuration");
		conf = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForExport(conf);

		logger.info("Exporting HW Configuration as XML");
		File xmlFile = new File(XML_FILE);
		FileWriter fw = new FileWriter(xmlFile);
		long start = System.currentTimeMillis();
		fw.write(HwConfigurationConversationUtils.getInstance().exportConfigurationAsXml(conf));
		long end = System.currentTimeMillis();
		fw.close();
		logger.info("Exporting took " + (end-start) + ", done exporting HW Configuration to XML");

		_prevConfig = conf;

	}

	public void testConfigurationImport() throws Exception {

		int prevDCAcount, finalDCAcount;

		logger.info("Reading the configuration back from the XML file");
		File xmlFile = new File(XML_FILE);
		assertNotNull(xmlFile);

		FileInputStream fis = new FileInputStream(xmlFile);
		byte[] b = new byte[fis.available ()];
		fis.read(b);
		fis.close();
		//assertTrue(xmlFile.delete());

		prevDCAcount = countDefaultCanAddresses();

		long start = System.currentTimeMillis();
		TMCDBExport export = HwConfigurationConversationUtils.getInstance().importConfigurationFromXml(new String(b));
		HwConfiguration config = export.get_hwconfig();
		long end = System.currentTimeMillis();
		logger.info("Importing took " + (end-start) + ", done importing HW Configuration from XML");

		logger.info("Comparting imported Config v/s exported Config");
 		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(_prevConfig, config);
		String[] probs = CloningTestUtils.getListOfProblems();
		assertEquals(0, probs.length);
		CloningTestUtils.clearListOfProblems();

		logger.info("Cloning and saving imported configuration");
		HwConfiguration finalConfig = HwConfigurationConversationUtils.getInstance().cloneImportedConfiguration(export,
				   config.getName()+"-imported-"+DatetimeUtils.getIsoDateFormat(Style.SHORTER).format(new Date()));
		logger.info("Comparing imported-cloned Config v/s exported Config");
		finalConfig = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForExport(finalConfig);
 		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(_prevConfig, finalConfig);
		probs = CloningTestUtils.getListOfProblems();
		assertEquals(0, probs.length);
		CloningTestUtils.clearListOfProblems();

		logger.info("Comparing number of DefaultCanAddress rows");
		finalDCAcount = countDefaultCanAddresses();
		assertEquals( prevDCAcount*2, finalDCAcount);
	}

	public void testExportImportWithAcsServices() throws Exception {

		HwConfiguration conf;
		HwConfiguration prevConfig;

		conf = CloningTestUtils.createConfiguration("tmp");

		Set<NetworkDevice> networkDevices = new HashSet<NetworkDevice>();
		Computer comp = new Computer();
		comp.setNetworkName("name");
		comp.setProcessorType(ComputerProcessorType.UNI);
		comp.setRealTime(false);
		comp.setDiskless(false);
		comp.setConfiguration(conf.getSwConfiguration());
		networkDevices.add(comp);

		Set<AcsService> services = new HashSet<AcsService>();
		AcsService acsservice = new AcsService();
		acsservice.setConfiguration(conf.getSwConfiguration());
		acsservice.setServiceType(AcsServiceServiceType.MANAGER);
		acsservice.setComputer(comp);
		services.add(acsservice);

		comp.setAcsServices(services);
		conf.getSwConfiguration().setAcsServices(services);
		conf.getSwConfiguration().setNetworkDevices(networkDevices);

		HwConfigurationConversationUtils.getInstance().updateConfiguration(conf);
		
		logger.info("Exporting HW Configuration as XML");
		File xmlFile = new File(XML_FILE);
		FileWriter fw = new FileWriter(xmlFile);
		long start = System.currentTimeMillis();
		fw.write(HwConfigurationConversationUtils.getInstance().exportConfigurationAsXml(conf));
		long end = System.currentTimeMillis();
		fw.close();
		logger.info("Exporting took " + (end-start) + ", done exporting HW Configuration to XML");

		prevConfig = conf;

		logger.info("Reading the configuration back from the XML file");

		FileInputStream fis = new FileInputStream(xmlFile);
		byte[] b = new byte[fis.available ()];
		fis.read(b);
		fis.close();
		//assertTrue(xmlFile.delete());

		start = System.currentTimeMillis();
		TMCDBExport export = HwConfigurationConversationUtils.getInstance().importConfigurationFromXml(new String(b));
		conf = export.get_hwconfig();
		end = System.currentTimeMillis();
		logger.info("Importing took " + (end-start) + ", done importing HW Configuration from XML");

		logger.info("Comparing imported Config v/s exported Config");
 		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(conf, prevConfig);
		String[] probs = CloningTestUtils.getListOfProblems();
		assertEquals(0, probs.length);
 		CloningTestUtils.clearListOfProblems();

		logger.info("Cloning imported config so it can be saved");
		HwConfiguration finalConfig = HwConfigurationConversationUtils.getInstance().cloneImportedConfiguration(export,
				   conf.getName()+"-imported-"+DatetimeUtils.getIsoDateFormat(Style.SHORTER).format(new Date()));
 		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(prevConfig, finalConfig);
		probs = CloningTestUtils.getListOfProblems();
		assertEquals(0, probs.length);
 		CloningTestUtils.clearListOfProblems();

		logger.info("Comparing imported-cloned-hydrated Config v/s exported Config");
		finalConfig = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForCloning(finalConfig);
 		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(prevConfig, finalConfig);
		probs = CloningTestUtils.getListOfProblems();
		assertEquals(0, probs.length);
 		CloningTestUtils.clearListOfProblems();

	}
	
	public void testExportImportWithGlobalConfigurationReference() throws Exception {

		HwConfiguration conf;
		HwConfiguration prevConfig;

		conf = CloningTestUtils.createConfiguration("tmp2");
		setUpConfigurationForGlobalImportExportTest(conf);
		
		HwConfiguration confGlobal = CloningTestUtils.createConfiguration("tmpGlobal");
		setUpConfigurationForGlobalImportExportTest(confGlobal);

		// create a cross-configuration baseelementstartup
		Antenna antenna = null;
		for(BaseElement be: confGlobal.getBaseElements()) {
			if(be.getType().equals(BaseElementType.Antenna) || be instanceof Antenna)
			{
				antenna = (Antenna)be;
				break;
			}
		}
		assert(null != antenna);
		
		StartupScenario startup = null;
		for(StartupScenario ss: conf.getStartupScenarios()) {
			startup = ss;
			break;
		}
		assert(null != startup);
		
		BaseElementStartup baseElementStartup = new BaseElementStartup(antenna, startup);
		baseElementStartup.setSimulated(false);
		startup.addBaseElementStartup(baseElementStartup);
		
		// create a cross-configuration AntennaToPad mapping
		Pad pad = null;
		for(BaseElement be: confGlobal.getBaseElements()) {
			if(be.getType().equals(BaseElementType.Pad) || be instanceof Pad)
			{
				pad = (Pad)be;
				break;
			}
		}
		assert(null != pad);
		new AntennaToPad(antenna, pad, new Long(0),Long.valueOf(0), true);

		// save the global configuration
		HwConfigurationConversationUtils.getInstance().updateConfiguration(confGlobal);
		
		// set the local configuration's global configuration reference
		conf.setGlobalConfiguration(confGlobal);
		
		// save the 'local' configuration
		HwConfigurationConversationUtils.getInstance().updateConfiguration(conf);
		
		logger.info("Exporting HW Configuration as XML");
		File xmlFile = new File(XML_FILE);
		FileWriter fw = new FileWriter(xmlFile);
		long start = System.currentTimeMillis();
		fw.write(HwConfigurationConversationUtils.getInstance().exportConfigurationAsXml(conf));
		long end = System.currentTimeMillis();
		fw.close();
		logger.info("Exporting took " + (end-start) + ", done exporting HW Configuration to XML");

		prevConfig = conf;

		logger.info("Reading the configuration back from the XML file");

		FileInputStream fis = new FileInputStream(xmlFile);
		byte[] b = new byte[fis.available ()];
		fis.read(b);
		fis.close();
		//assertTrue(xmlFile.delete());

		start = System.currentTimeMillis();
		TMCDBExport export = HwConfigurationConversationUtils.getInstance().importConfigurationFromXml(new String(b));
		conf = export.get_hwconfig();
		end = System.currentTimeMillis();
		logger.info("Importing took " + (end-start) + ", done importing HW Configuration from XML");

		logger.info("Comparing imported Config v/s exported Config");
 		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(conf, prevConfig);
		String[] probs = CloningTestUtils.getListOfProblems();
		assertEquals(0, probs.length);
 		CloningTestUtils.clearListOfProblems();

		logger.info("Cloning imported config so it can be saved");
		HwConfiguration finalConfig = HwConfigurationConversationUtils.getInstance().cloneImportedConfiguration(export,
				   conf.getName()+"-imported-"+DatetimeUtils.getIsoDateFormat(Style.SHORTER).format(new Date()));
 		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(prevConfig, finalConfig);
		probs = CloningTestUtils.getListOfProblems();
		assertEquals(0, probs.length);
 		CloningTestUtils.clearListOfProblems();

		logger.info("Comparing imported-cloned-hydrated Config v/s exported Config");
		finalConfig = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForCloning(finalConfig);
 		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(prevConfig, finalConfig);
		probs = CloningTestUtils.getListOfProblems();
		assertEquals(0, probs.length);
 		CloningTestUtils.clearListOfProblems();

	}

	// TODO: This test is disabled until we have a valid huge XML file to be imported
	public void notestExternalConfigurationImport() throws Exception {

		logger.info("Reading the configuration from the XML file");
		File xmlFile = new File(XML_FILE2);
		assertNotNull(xmlFile);

		FileInputStream fis = new FileInputStream(xmlFile);
		byte[] b = new byte[fis.available ()];
		fis.read(b);
		fis.close();

		long start = System.currentTimeMillis();
		TMCDBExport imported = HwConfigurationConversationUtils.getInstance().importConfigurationFromXml(new String(b));
		HwConfiguration config = imported.get_hwconfig();
		long end = System.currentTimeMillis();
		logger.info("Importing took " + (end-start) + ", done importing HW Configuration from XML");

		logger.info("Cloning imported config so it can be saved");
		HwConfiguration clonedConfig = HwConfigurationConversationUtils.getInstance().cloneImportedConfiguration(imported, config.getName()+"-imported-"+DatetimeUtils.getIsoDateFormat(Style.SHORTER).format(new Date()));
		clonedConfig = HwConfigurationConversationUtils.getInstance().hydrateConfigurationForExport(clonedConfig);
 		CloningTestUtils.clearListOfProblems();
		CloningTestUtils.compareConfigurations(config, clonedConfig);
		String[] probs = CloningTestUtils.getListOfProblems();
		assertEquals(0, probs.length);
 		CloningTestUtils.clearListOfProblems();
	}
	
	private void setUpConfigurationForGlobalImportExportTest(HwConfiguration config) throws Exception
	{
		StartupScenario startup = null;
		Antenna antenna = null;
		Pad pad = null;
		BaseElementStartup baseElementStartup = null;

		antenna = new Antenna("DV01",
				AntennaType.ACA,
				new Coordinate(0.0, 0.0, 0.0),
				new Coordinate(0.0, 0.0, 0.0),
				4.5,
				0L,
				0, 
				0);
		
		config.addBaseElement(antenna);
		pad = new Pad("PAD01", new Coordinate(0.0, 0.0, 0.0), Long.valueOf(0));
		config.addBaseElement(pad);
		HwConfigurationConversationUtils.getInstance().updateConfiguration(config);
		
		startup = new StartupScenario("startup");
		config.addStartupScenario(startup);
		baseElementStartup = new BaseElementStartup(antenna, startup);
		baseElementStartup.setSimulated(false);
		startup.addBaseElementStartup(baseElementStartup);
	}
}
