/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.wizards.support;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.VerifyEvent;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.widgets.Text;

/**
 * Used to verify that input in a text field is of decimal type.
 * @author sharring (borrowing heavily from an internet posting on a similar topic).
 */
public class VerifyDecimalListener implements VerifyListener 
{
	private final static char DECIMAL_SYMBOL = '.';
	private final static int DEFAULT_NUMBER_OF_DIGITS_AFTER_DECIMAL = 8;
	
	private int numDigitsAfterDecimal = DEFAULT_NUMBER_OF_DIGITS_AFTER_DECIMAL;

	public VerifyDecimalListener()
	{
		this(DEFAULT_NUMBER_OF_DIGITS_AFTER_DECIMAL);
	}
	
	public VerifyDecimalListener(int numDigitsAfterDecimal)
	{
		this.numDigitsAfterDecimal = numDigitsAfterDecimal;
	}
	
	@Override
	public void verifyText(VerifyEvent e) 
	{
		Text control = (Text)e.widget;
		String text = control.getText();
		e.doit = false;

		if (e.keyCode == SWT.BS || e.keyCode == SWT.DEL){
			e.doit = true;
			return;
		}

		String newText = text.substring(0, e.start) + e.text +
		text.substring(e.end);

		if (newText.equals("")){
			e.doit = true;
			return;
		}

		Pattern realNumberPattern = Pattern.compile("^[-+]?[0-9]*" + "\\" + DECIMAL_SYMBOL+ "?[0-9]*([eE][-+]?[0-9]+)?$");
		Matcher matcher = realNumberPattern.matcher(newText);
		boolean valid = matcher.matches();

		e.doit = valid;
		if(valid) {
			return;
		}

		if (newText.length() > 1) 
		{
			if(e.character == '-')
			{
				// check the minus notation
				int pos = newText.indexOf('-');

				// allow 2 minus signs in the string ONLY if the second one is for the exponent
				if (pos != -1 && newText.indexOf('-', pos + 1) != -1)
				{
					int pos2 = newText.indexOf('-', pos + 1);
					if(newText.charAt(pos2 - 1) == 'E' || newText.charAt(pos2 - 1) == 'e' && 
							newText.indexOf('-', pos2 + 1) == -1) 
					{
						e.doit = true;
						return;
					}
					e.doit = false;
					return;
				}
				else if(pos != -1) 
				{
					if(newText.charAt(pos - 1) == 'E' || newText.charAt(pos - 1) == 'e') 
					{
						e.doit = true;
						return;
					}
					e.doit = false;
					return;
				}
			}
			
			if(e.character == '+')
			{
				// check the plus notation
				int pos = newText.indexOf('+');

				// allow 2 plus signs in the string ONLY if the second one is for the exponent
				if (pos != -1 && newText.indexOf('+', pos + 1) != -1)
				{
					int pos2 = newText.indexOf('+', pos + 1);
					if(newText.charAt(pos2 - 1) == 'E' || newText.charAt(pos2 - 1) == 'e' && 
							newText.indexOf('-', pos2 + 1) == -1) 
					{
						e.doit = true;
						return;
					}
					e.doit = false;
					return;
				}
				else if(pos != -1) 
				{
					if(newText.charAt(pos - 1) == 'E' || newText.charAt(pos - 1) == 'e') 
					{
						e.doit = true;
						return;
					}
					e.doit = false;
					return;
				}
			}
			
			if(e.character == DECIMAL_SYMBOL)
			{
				// check the decimal notation
				int pos = newText.indexOf(DECIMAL_SYMBOL);
				if (pos != -1) 
				{
					// only allow one decimal symbol
					if (newText.indexOf(DECIMAL_SYMBOL, pos + 1) != -1) 
					{
						e.doit = false;
						return;
					}

					// make sure we don't have too many digits
					if (newText.substring(pos + 1).length() > this.numDigitsAfterDecimal) 
					{
						e.doit = false;
						return;
					}
				}
			}
			
			// check the scientific notation with capital 'E'
			if(e.character == 'E')
			{
				int pos = newText.indexOf("E");
				if (pos != -1) 
				{
					// only allow one E/e symbol
					if (newText.indexOf("E", pos + 1) != -1) 
					{
						e.doit = false;
						return;
					}
					if (newText.indexOf("e") != -1) 
					{
						e.doit = false;
						return;
					}
					e.doit = true;
					return;
				}
			}
			
			// check the scientific notation with capital 'e'
			if(e.character == 'e')
			{
				int pos = newText.indexOf("e");
				if (pos != -1) 
				{
					// only allow one e/E symbol
					if (newText.indexOf("e", pos + 1) != -1) 
					{
						e.doit = false;
						return;
					}
					if (newText.indexOf("E") != -1) 
					{
						e.doit = false;
						return;
					}
					e.doit = true;
					return;
				}
			}
			
		}
		else if (newText.length() == 1 && (newText.charAt(0) == '-' || newText.charAt(0) == '.'))
		{
			e.doit = true;
			return;
		}
	}
}

