/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * NewContainerPage.java
 */
package alma.obops.tmcdbgui.wizards;

import java.util.List;

import org.eclipse.jface.dialogs.DialogPage;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.hibernate.criterion.MatchMode;

import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Container;
import alma.obops.tmcdbgui.dialogs.ComputerSelectionDialog;
import alma.obops.tmcdbgui.dialogs.ComputerSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.dialogs.ConfigurationSelectionDialog;
import alma.obops.tmcdbgui.dialogs.ConfigurationSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Wizard page for creation of a new {@link Container} object.
 * 
 * This page is used the {@link NewContainerWizard}
 * 
 * @author rtobar, Mar 1, 2010
 */
public class NewContainerPage extends WizardPage {

	private Container _container;

	protected NewContainerPage(String pageName, Container container) {
		super(pageName);
		_container = container;
		setTitle("New Container");
		setDescription("Specify the initial basic details of the new container");
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createControl(Composite parent) {

		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout(3, false));

		/* Configuration */
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label configurationLabel = new Label(composite, SWT.NONE);
		configurationLabel.setText("Configuration");
		configurationLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		final Text configurationName = new Text(composite, SWT.BORDER | SWT.SINGLE);
		configurationName.setEditable(false);
		if( _container.getConfiguration() != null) {
			configurationName.setText(_container.getConfiguration().getConfigurationName());
			configurationName.setToolTipText(_container.getConfiguration().getFullName());
		}
		configurationName.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Button browseConfigs = new Button(composite, SWT.PUSH);
		browseConfigs.setText("Browse...");
		browseConfigs.setLayoutData(gd);


		/* Computer */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label computerLabel = new Label(composite, SWT.NONE);
		computerLabel.setText("Computer");
		computerLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		final Text computerName = new Text(composite, SWT.BORDER | SWT.SINGLE);
		computerName.setEditable(false);
		if( _container.getComputer() != null) {
			computerName.setText(_container.getComputer().getName());
			computerName.setToolTipText(_container.getComputer().getNetworkName());
		}
		computerName.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		final Button browseComputers = new Button(composite, SWT.PUSH);
		browseComputers.setText("Browse...");
		browseComputers.setLayoutData(gd);

		/* Name */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label nameLabel = new Label(composite, SWT.NONE);
		nameLabel.setText("Name");
		nameLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		final Text containerName = new Text(composite, SWT.BORDER | SWT.SINGLE);
		containerName.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label dummyLabel = new Label(composite, SWT.NONE);
		dummyLabel.setLayoutData(gd);

		/* Path */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label pathLabel = new Label(composite, SWT.NONE);
		pathLabel.setText("Path");
		pathLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		final Text containerPath = new Text(composite, SWT.BORDER | SWT.SINGLE);
		containerPath.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		dummyLabel = new Label(composite, SWT.NONE);
		dummyLabel.setLayoutData(gd);


		// Setup the browser buttons and logic
		// Configurations should always be enabled, and browse button provokes to appear a dialog
		browseConfigs.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				ElementListSelectionDialog d = new ConfigurationSelectionDialog(getShell(), new ConfigurationSelectionDialogLabelProvider());
				d.open();
				Object configs[] = d.getResult();
				String selectedConfigName = (String)configs[0];
			
				List<HwConfiguration> configsFound = null;
				try {
					configsFound = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(selectedConfigName, MatchMode.EXACT);
				} catch (Exception e1) {
					configsFound = null;
				}
				
				if(null != configsFound && configsFound.size() == 1) {
					_container.setConfiguration( configsFound.get(0).getSwConfiguration() );
					configurationName.setText( _container.getConfiguration().getConfigurationName() );
					computerName.setEnabled(true);
					browseComputers.setEnabled(true);
				}
				toggleIsComplete();
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});

		// Computers are enabled if a configuration exists, and browse button provokes to appear a dialog
		if( _container.getConfiguration() == null ) {
			computerName.setEnabled(false);
			browseComputers.setEnabled(false);
		}
		browseComputers.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				ElementListSelectionDialog d = new ComputerSelectionDialog(getShell(), new ComputerSelectionDialogLabelProvider(), _container.getConfiguration());
				d.open();
				Object computers[] = d.getResult();
				if( computers != null && computers.length == 1 ) {
					_container.setComputer( ((Computer)computers[0]) );
					computerName.setText( _container.getComputer().getName() );
				}
				toggleIsComplete();
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});

		// Listener for container name
		containerName.addListener(SWT.KeyUp, new Listener() {
			public void handleEvent(Event event) {
				_container.setContainerName( containerName.getText() );
				toggleIsComplete();
			}
		});
		// Listener for container path
		containerPath.addListener(SWT.KeyUp, new Listener() {
			public void handleEvent(Event event) {
				_container.setPath( containerPath.getText() );
				toggleIsComplete();
			}
		});

		setControl( composite );
		setPageComplete(false);
	}

	private void toggleIsComplete() {

		// Errors
		if( _container.getConfiguration() == null ) {
			setErrorMessage("Configuration missing");
			setPageComplete(false);
		}
		else if( _container.getContainerName() == null || _container.getContainerName().trim().equals("") ) {
			setErrorMessage("Container name missing");
			setPageComplete(false);
		}

		// Warnings
		else if( _container.getContainerName() != null && !_container.getContainerName().trim().matches("^[\\w\\-_]+$") ) {
			setMessage("Container name should only contain alphanumeric, '-' or '_' characters", DialogPage.WARNING);
			setErrorMessage(null);
			setPageComplete(true);
		}
		else if( _container.getPath() != null && !_container.getPath().trim().equals("") && !_container.getPath().trim().matches("^[\\w\\-_/]+$") ) {
			setMessage("Container path should only contain alphanumeric, '-', '_' or '/' characters", DialogPage.WARNING);
			setErrorMessage(null);
			setPageComplete(true);
		}

		else {
			setErrorMessage(null);
			setMessage(null, DialogPage.WARNING);
			setPageComplete(true);
		}
	}
}