/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * NewComputerPage.java
 */
package alma.obops.tmcdbgui.wizards;

import java.util.List;

import org.eclipse.jface.dialogs.DialogPage;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.hibernate.criterion.MatchMode;

import alma.acs.tmcdb.Computer;
import alma.obops.tmcdbgui.dialogs.ConfigurationSelectionDialog;
import alma.obops.tmcdbgui.dialogs.ConfigurationSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Wizard page for creation of a new {@link Computer} object.
 * 
 * This page is used the {@link NewComputerWizard}
 * 
 * @author rtobar, Mar 2, 2010
 */
public class NewComputerPage extends WizardPage {

	private Computer _computer;

	protected NewComputerPage(String pageName, Computer computer) {
		super(pageName);
		_computer = computer;
		setTitle("New Computer");
		setDescription("Specify the initial basic details of the new computer");
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createControl(Composite parent) {

		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout(3, false));

		/* Configuration */
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label configurationLabel = new Label(composite, SWT.NONE);
		configurationLabel.setText("Configuration");
		configurationLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		final Text configurationName = new Text(composite, SWT.BORDER | SWT.SINGLE);
		configurationName.setEditable(false);
		if( _computer.getConfiguration() != null) {
			configurationName.setText(_computer.getConfiguration().getConfigurationName());
			configurationName.setToolTipText(_computer.getConfiguration().getFullName());
		}
		configurationName.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Button browseConfigs = new Button(composite, SWT.PUSH);
		browseConfigs.setText("Browse...");
		browseConfigs.setLayoutData(gd);

		/* Name */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label nameLabel = new Label(composite, SWT.NONE);
		nameLabel.setText("Name");
		nameLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		final Text computerName = new Text(composite, SWT.BORDER | SWT.SINGLE);
		computerName.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label dummyLabel = new Label(composite, SWT.NONE);
		dummyLabel.setLayoutData(gd);

		/* Path */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label netNameLabel = new Label(composite, SWT.NONE);
		netNameLabel.setText("Network Name");
		netNameLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		final Text computerNetName = new Text(composite, SWT.BORDER | SWT.SINGLE);
		computerNetName.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		dummyLabel = new Label(composite, SWT.NONE);
		dummyLabel.setLayoutData(gd);


		// Setup the browser buttons and logic
		// Configurations should always be enabled, and browse button provokes to appear a dialog
		browseConfigs.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				ElementListSelectionDialog d = new ConfigurationSelectionDialog(getShell(), new ConfigurationSelectionDialogLabelProvider());
				d.open();
				Object configs[] = d.getResult();
				String selectedConfigName = (String)configs[0];
			
				List<HwConfiguration> configsFound = null;
				try {
					configsFound = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(selectedConfigName, MatchMode.EXACT);
				} catch (Exception e1) {
					configsFound = null;
				}
				
				if(null != configsFound && configsFound.size() == 1) {
					_computer.setConfiguration( configsFound.get(0).getSwConfiguration() );
					configurationName.setText( _computer.getConfiguration().getConfigurationName() );
				}
				toggleIsComplete();
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});

		// Listener for computer name
		computerName.addListener(SWT.KeyUp, new Listener() {
			public void handleEvent(Event event) {
				_computer.setName( computerName.getText() );
				toggleIsComplete();
			}
		});
		// Listener for computer network name
		computerNetName.addListener(SWT.KeyUp, new Listener() {
			public void handleEvent(Event event) {
				_computer.setNetworkName( computerNetName.getText().trim() );
				toggleIsComplete();
			}
		});

		setControl( composite );
		setPageComplete(false);
	}

	private void toggleIsComplete() {

		// Errors
		if( _computer.getConfiguration() == null ) {
			setErrorMessage("Configuration missing");
			setPageComplete(false);
		}
		else if ( _computer.getNetworkName() == null || _computer.getNetworkName().equals("") ) {
			setErrorMessage("Network name missing");
			setPageComplete(false);
		}

		// Warnings
		else if( _computer.getName() == null || _computer.getName().trim().equals("") ) {
			setMessage("Empty computer name is discouraged", DialogPage.WARNING);
			setErrorMessage(null);
			setPageComplete(true);
		}

		else {
			setErrorMessage(null);
			setMessage(null, DialogPage.WARNING);
			setPageComplete(true);
		}
	}
}