/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * NewComponentWizard.java
 */
package alma.obops.tmcdbgui.wizards;

import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.WorkbenchException;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentImplLang;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Container;
import alma.obops.tmcdbgui.editors.ComponentEditor;
import alma.obops.tmcdbgui.editors.inputs.ComponentEditorInput;
import alma.obops.tmcdbgui.perspectives.SwConfigurationPerspective;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Wizard for creating a new {@link Component} object.
 * 
 * If during the launching of this wizard a {@link HwConfiguration} was selected,
 * its associated SW configuration is used as the default configuration
 * for the new component.
 * 
 * If a {@link Container} object was selected during the launching of this wizard,
 * its associated SW configuration is used as the default configuration
 * for the new component, and the container itself is used as the default container
 * for the new component.
 * 
 * @author rtobar, Mar 2, 2010
 *
 */
public class NewComponentWizard extends Wizard implements INewWizard {

	private IWorkbench _workbench;
	private Component _comp;
	private Configuration configuration;
	
	public NewComponentWizard(Configuration config)
	{
		this.configuration = config;
	}

	@Override
	public boolean performFinish() {

		// Fill the not-null properties with default values
		_comp.setRealTime(false);
		_comp.setIsAutostart(false);
		_comp.setIsDefault(false);
		_comp.setIsStandaloneDefined(false);
		_comp.setIsControl(false);
		_comp.setKeepAliveTime(-1);
		_comp.setMinLogLevel((byte)-1);
		_comp.setMinLogLevelLocal((byte)-1);

		if( _comp.getPath() == null || _comp.getPath().trim().equals("") )
			_comp.setPath("/");
 
		// Set initial values for implementation language
		if( _comp.getContainer() != null )
			_comp.setImplLang(ComponentImplLang.valueOfForEnum(_comp.getContainer().getImplLang().toString()));

		// Guess a nice name for the code
		if( _comp.getImplLang() != null ) {

			String idl = _comp.getComponentType().getIDL();
			String []packages = idl.replaceAll("^IDL:", "").replaceAll(":1.0", "").split("/");

			if( _comp.getImplLang().equals(ComponentImplLang.CPP) )
				_comp.setCode( packages[packages.length-1] + "Impl" );

			else if( _comp.getImplLang().equals(ComponentImplLang.JAVA) ) {
				StringBuilder sb = new StringBuilder();
				for(int i=0; i!= packages.length; i++)
					sb.append(packages[i] + ( i!=packages.length-1 ? "." : "") );
				sb.append( "Impl." + packages[packages.length-1] + "Helper");
				_comp.setCode( sb.toString() );
			}

			else if( _comp.getImplLang().equals(ComponentImplLang.PY) ) {
				StringBuilder sb = new StringBuilder();
				for(int i=1; i!= packages.length-1; i++)
					sb.append(packages[i] + ( i!=packages.length-2 ? "." : "") );
				sb.append( "Impl." + packages[packages.length-1] );
				_comp.setCode( sb.toString() );
			}
		}

		// Open the Component editor with the new component information
		ComponentEditorInput cei = new ComponentEditorInput(_comp);
		try {
			_workbench.getActiveWorkbenchWindow().getActivePage().openEditor(cei, ComponentEditor.ID);
			_workbench.showPerspective(SwConfigurationPerspective.ID, _workbench.getActiveWorkbenchWindow());
		} catch (WorkbenchException e) {
			e.printStackTrace();
			return false;
		}
		return true;
	}

	@Override
	public void init(IWorkbench workbench, IStructuredSelection selection) {

		_workbench = workbench;
		_comp = new Component();
		_comp.setConfiguration(configuration);

		if( selection != null && selection.size() == 1 ) {
			if( selection.getFirstElement() instanceof Container ) {
				Container cont = (Container)selection.getFirstElement();
				_comp.setContainer(cont);
				_comp.setConfiguration(cont.getConfiguration());
			}
		}

	}

	@Override
	public void addPages() {
		addPage(new NewComponentPage("New Component", _comp));
	}

	
}
