/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * NewComponentPage.java
 */
package alma.obops.tmcdbgui.wizards;

import java.util.List;

import org.eclipse.jface.dialogs.DialogPage;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.hibernate.criterion.MatchMode;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Container;
import alma.obops.tmcdbgui.dialogs.ComponentTypeSelectionDialog;
import alma.obops.tmcdbgui.dialogs.ComponentTypeSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.dialogs.ConfigurationSelectionDialog;
import alma.obops.tmcdbgui.dialogs.ConfigurationSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.dialogs.ContainerSelectionDialog;
import alma.obops.tmcdbgui.dialogs.ContainerSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Wizard page for creation of a new {@link Component} object.
 * 
 * This page is used the {@link NewComponentWizard}
 * 
 * @author rtobar, Mar 2, 2010
 */
public class NewComponentPage extends WizardPage {

	private Component _component;

	protected NewComponentPage(String pageName, Component component) {
		super(pageName);
		_component = component;
		setTitle("New Component");
		setDescription("Specify the initial basic details of the new component");
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createControl(Composite parent) {

		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout(3, false));

		/* Configuration */
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label configurationLabel = new Label(composite, SWT.NONE);
		configurationLabel.setText("Configuration");
		configurationLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		final Text configurationName = new Text(composite, SWT.BORDER | SWT.SINGLE);
		configurationName.setEditable(false);
		if( _component.getConfiguration() != null) {
			configurationName.setText(_component.getConfiguration().getConfigurationName());
			configurationName.setToolTipText(_component.getConfiguration().getFullName());
		}
		configurationName.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Button browseConfigs = new Button(composite, SWT.PUSH);
		browseConfigs.setText("Browse...");
		browseConfigs.setLayoutData(gd);

		/* Container */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label containerLabel = new Label(composite, SWT.NONE);
		containerLabel.setText("Container");
		containerLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		final Text containerName = new Text(composite, SWT.BORDER | SWT.SINGLE);
		containerName.setEditable(false);
		if( _component.getContainer() != null) {
			containerName.setText(_component.getContainer().getContainerName() );
			containerName.setToolTipText( LabelHelper.getFullPath(_component.getContainer(), false) );
		}
		containerName.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		final Button browseContainers = new Button(composite, SWT.PUSH);
		browseContainers.setText("Browse...");
		browseContainers.setLayoutData(gd);

		/* Component Type */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label compTypeLabel = new Label(composite, SWT.NONE);
		compTypeLabel.setText("Component Type");
		compTypeLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		final Text componentTypeIDL = new Text(composite, SWT.BORDER | SWT.SINGLE);
		componentTypeIDL.setEditable(false);
		componentTypeIDL.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Button browseCompTypes = new Button(composite, SWT.PUSH);
		browseCompTypes.setText("Browse...");
		browseCompTypes.setLayoutData(gd);

		/* Name */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label nameLabel = new Label(composite, SWT.NONE);
		nameLabel.setText("Name");
		nameLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		final Text componentName = new Text(composite, SWT.BORDER | SWT.SINGLE);
		componentName.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label dummyLabel = new Label(composite, SWT.NONE);
		dummyLabel.setLayoutData(gd);

		/* Path */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label pathLabel = new Label(composite, SWT.NONE);
		pathLabel.setText("Path");
		pathLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		final Text componentPath = new Text(composite, SWT.BORDER | SWT.SINGLE);
		componentPath.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		dummyLabel = new Label(composite, SWT.NONE);
		dummyLabel.setLayoutData(gd);


		// Setup the browser buttons and logic
		// Configurations should always be enabled, and browse button provokes to appear a dialog
		browseConfigs.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				ElementListSelectionDialog d = new ConfigurationSelectionDialog(getShell(), new ConfigurationSelectionDialogLabelProvider());
				d.open();
				Object configs[] = d.getResult();
				String selectedConfigName = (String)configs[0];
			
				List<HwConfiguration> configsFound = null;
				try {
					configsFound = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(selectedConfigName, MatchMode.EXACT);
				} catch (Exception e1) {
					configsFound = null;
				}
				
				if(null != configsFound && configsFound.size() == 1) {
					_component.setConfiguration( configsFound.get(0).getSwConfiguration() );
					configurationName.setText( _component.getConfiguration().getConfigurationName() );
					containerName.setEnabled(true);
					browseContainers.setEnabled(true);
				}
				toggleIsComplete();
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});

		if( _component.getConfiguration() == null ) {
			containerName.setEnabled(false);
			browseContainers.setEnabled(false);
		}
		browseContainers.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				ElementListSelectionDialog d = new ContainerSelectionDialog(getShell(), new ContainerSelectionDialogLabelProvider(), _component.getConfiguration());
				d.open();
				Object containers[] = d.getResult();
				if( containers != null && containers.length == 1 ) {
					_component.setContainer( (Container)containers[0] );
					containerName.setText( _component.getContainer().getContainerName() );
				}
				toggleIsComplete();
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});

		browseCompTypes.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				ElementListSelectionDialog d = new ComponentTypeSelectionDialog(getShell(), new ComponentTypeSelectionDialogLabelProvider());
				d.open();
				Object compTypes[] = d.getResult();
				if( compTypes != null && compTypes.length == 1 ) {
					_component.setComponentType( (ComponentType)compTypes[0] );
					componentTypeIDL.setText( _component.getComponentType().getIDL() );
				}
				toggleIsComplete();
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});

		// Listener for component name
		componentName.addListener(SWT.KeyUp, new Listener() {
			public void handleEvent(Event event) {
				_component.setComponentName( componentName.getText() );
				toggleIsComplete();
			}
		});
		// Listener for component path
		componentPath.addListener(SWT.KeyUp, new Listener() {
			public void handleEvent(Event event) {
				_component.setPath( componentPath.getText() );
				toggleIsComplete();
			}
		});

		setControl( composite );
		setPageComplete(false);
	}

	private void toggleIsComplete() {

		// Errors
		if( _component.getConfiguration() == null ) {
			setErrorMessage("Configuration missing");
			setPageComplete(false);
		}
		else if( _component.getComponentType() == null ) {
			setErrorMessage("Component Type missing");
			setPageComplete(false);
		}
		else if( _component.getComponentName() == null || _component.getComponentName().trim().equals("") ) {
			setErrorMessage("Component name missing");
			setPageComplete(false);
		}

		// Warnings
		else if( _component.getComponentName() != null && !_component.getComponentName().trim().matches("^[\\w\\-_]+$") ) {
			setMessage("Component name should only contain alphanumeric, '-' or '_' characters", DialogPage.WARNING);
			setErrorMessage(null);
			setPageComplete(true);
		}
		else if( _component.getPath() != null &&
				!_component.getPath().trim().equals("") &&
				!_component.getPath().trim().matches("^[\\w\\-_/]+$") ) {
			setMessage("Component path should only contain alphanumeric, '-', '_' or '/' characters", DialogPage.WARNING);
			setErrorMessage(null);
			setPageComplete(true);
		}

		else {
			setErrorMessage(null);
			setMessage(null, DialogPage.WARNING);
			setPageComplete(true);
		}
	}
}