/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * NewComponentWizard.java
 */
package alma.obops.tmcdbgui.wizards;

import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.WorkbenchException;

import alma.acs.tmcdb.BACIPropArchMech;
import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Configuration;
import alma.obops.tmcdbgui.editors.BACIPropertyEditor;
import alma.obops.tmcdbgui.editors.inputs.BACIPropertyEditorInput;
import alma.obops.tmcdbgui.perspectives.SwConfigurationPerspective;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Wizard for creating a new {@link BACIProperty} object.
 * 
 * If during the launching of this wizard a {@link HwConfiguration} was selected,
 * its associated SW configuration is used as the default configuration
 * to search for candidate Components parents of the new BACIProperty.
 * 
 * If a {@link Component} object was selected during the launching of this wizard,
 * it is used as the parent Component of the new BACIProperty
 * 
 * @author rtobar, Oct 16th, 2010
 *
 */
public class NewBACIPropertyWizard extends Wizard implements INewWizard {

	private IWorkbench    _workbench;
	private BACIProperty  _baciProp;
	private Configuration _configuration;

	public NewBACIPropertyWizard(Configuration config) 
	{
		this._configuration = config;
	}
	
	@Override
	public boolean performFinish() {

		// Fill the not-null properties with default values
		BACIPropertyEditorInput bpei = new BACIPropertyEditorInput(_baciProp);
		try {
			_workbench.getActiveWorkbenchWindow().getActivePage().openEditor(bpei, BACIPropertyEditor.ID);
			_workbench.showPerspective(SwConfigurationPerspective.ID, _workbench.getActiveWorkbenchWindow());
		} catch (WorkbenchException e) {
			e.printStackTrace();
			return false;
		}
		return true;
	}

	@Override
	public void init(IWorkbench workbench, IStructuredSelection selection) {

		_workbench = workbench;
		_baciProp = new BACIProperty();
		_baciProp.setDescription("description");
		_baciProp.setFormat("format");
		_baciProp.setUnits("units");
		_baciProp.setResolution("0");
		_baciProp.setArchive_priority(3);
		_baciProp.setArchive_min_int(0.0);
		_baciProp.setArchive_max_int(0.0);
		_baciProp.setDefault_timer_trig(1.0);
		_baciProp.setMin_timer_trig(.048);
		_baciProp.setInitialize_devio(false);
		_baciProp.setMin_delta_trig(0.0);
		_baciProp.setDefault_value("0");
		_baciProp.setGraph_min(0.0);
		_baciProp.setGraph_max(0.0);
		_baciProp.setMin_step(0.0);
		_baciProp.setArchive_delta(0.0);
		_baciProp.setAlarm_high_on(0.0);
		_baciProp.setAlarm_low_on(0.0);
		_baciProp.setAlarm_high_off(0.0);
		_baciProp.setAlarm_low_off(0.0);
		_baciProp.setAlarm_timer_trig(0.0);
		_baciProp.setMin_value(-2147483648.0);
		_baciProp.setMax_value(2147483647.0);
		_baciProp.setAlarm_fault_family("!");
		_baciProp.setAlarm_fault_member("!");
		_baciProp.setAlarm_level(0);
		_baciProp.setArchive_suppress(true);
		_baciProp.setArchive_mechanism(BACIPropArchMech.MONITOR_COLLECTOR);
		
		if( selection != null && selection.size() == 1 ) {
			if( selection.getFirstElement() instanceof Component ) {
				Component component = (Component)selection.getFirstElement();
				_baciProp.setComponent(component);
			}
		}

	}

	@Override
	public void addPages() {
		addPage(new NewBACIPropertyPage("New BACI Property", _baciProp, _configuration));
	}

	
}
