/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * NewComponentPage.java
 */
package alma.obops.tmcdbgui.wizards;

import java.util.List;

import org.eclipse.jface.dialogs.DialogPage;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.hibernate.criterion.MatchMode;

import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Configuration;
import alma.obops.tmcdbgui.dialogs.ComponentSelectionDialog;
import alma.obops.tmcdbgui.dialogs.ConfigurationSelectionDialog;
import alma.obops.tmcdbgui.dialogs.ConfigurationSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.views.providers.SwDeploymentTreeLabelProvider;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Wizard page for creation of a new {@link BACIProperty} object.
 * 
 * This page is used the {@link NewBACIPropertyWizard}
 * 
 * @author rtobar, Mar 2, 2010
 */
public class NewBACIPropertyPage extends WizardPage {

	private BACIProperty  _baciProp;
	private Configuration _configuration;

	protected NewBACIPropertyPage(String pageName, BACIProperty baciProp, Configuration configuration) {
		super(pageName);
		_baciProp = baciProp;
		_configuration = configuration;
		setTitle("New BACI Property");
		setDescription("Specify the initial basic details of the new BACI Property");
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createControl(Composite parent) {

		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout(3, false));

		/* Configuration */
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label configurationLabel = new Label(composite, SWT.NONE);
		configurationLabel.setText("Configuration");
		configurationLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		final Text configurationName = new Text(composite, SWT.BORDER | SWT.SINGLE);
		configurationName.setEditable(false);
		if( _configuration != null) {
			configurationName.setText(_configuration.getConfigurationName());
			configurationName.setToolTipText(_configuration.getFullName());
		}
		configurationName.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Button browseConfigs = new Button(composite, SWT.PUSH);
		browseConfigs.setText("Browse...");
		browseConfigs.setLayoutData(gd);

		/* Component */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label componentLabel = new Label(composite, SWT.NONE);
		componentLabel.setText("Component");
		componentLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		final Text componentName = new Text(composite, SWT.BORDER | SWT.SINGLE);
		componentName.setEditable(false);
		if( _baciProp.getComponent() != null) {
			componentName.setText(_baciProp.getComponent().getComponentName() );
			componentName.setToolTipText( LabelHelper.getFullPath(_baciProp.getComponent(), false) );
		}
		componentName.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		final Button browseComponents = new Button(composite, SWT.PUSH);
		browseComponents.setText("Browse...");
		browseComponents.setLayoutData(gd);

		/* Name */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label nameLabel = new Label(composite, SWT.NONE);
		nameLabel.setText("Name");
		nameLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		final Text baciPropName = new Text(composite, SWT.BORDER | SWT.SINGLE);
		baciPropName.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label dummyLabel = new Label(composite, SWT.NONE);
		dummyLabel.setLayoutData(gd);

		// Setup the browser buttons and logic
		// Configurations should always be enabled, and browse button provokes to appear a dialog
		browseConfigs.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				ElementListSelectionDialog d = new ConfigurationSelectionDialog(getShell(), new ConfigurationSelectionDialogLabelProvider());
				d.open();
				Object configs[] = d.getResult();
				if( configs != null && configs.length == 1 ) {
					String selectedConfigName = (String)configs[0];
					List<HwConfiguration> configsFound = null;
					try {
						configsFound = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(selectedConfigName, MatchMode.EXACT);
					} catch (Exception e1) {
						configsFound = null;
					}
					
					if(null != configsFound && configsFound.size() == 1) {
						_configuration = configsFound.get(0).getSwConfiguration();
						configurationName.setText( _configuration.getConfigurationName() );
						componentName.setEnabled(true);
						browseComponents.setEnabled(true);
					}
				}
				toggleIsComplete();
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});

		if( _configuration == null ) {
			componentName.setEnabled(false);
			browseComponents.setEnabled(false);
		}
		browseComponents.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				ElementListSelectionDialog d = new ComponentSelectionDialog(getShell(), new SwDeploymentTreeLabelProvider(), _configuration);
				d.open();
				Object components[] = d.getResult();
				if( components != null && components.length == 1 ) {
					_baciProp.setComponent( (Component)components[0] );
					componentName.setText( _baciProp.getComponent().getComponentName() );
				}
				toggleIsComplete();
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});

		// Listener for BACI property name
		baciPropName.addListener(SWT.KeyUp, new Listener() {
			public void handleEvent(Event event) {
				_baciProp.setPropertyName( baciPropName.getText() );
				toggleIsComplete();
			}
		});

		setControl( composite );
		setPageComplete(false);
	}

	private void toggleIsComplete() {

		// Errors
		if( _configuration == null ) {
			setErrorMessage("Configuration missing");
			setPageComplete(false);
		}
		else if( _baciProp.getComponent() == null ) {
			setErrorMessage("Component missing");
			setPageComplete(false);
		}
		else if( _baciProp.getPropertyName() == null || _baciProp.getPropertyName().trim().equals("") ) {
			setErrorMessage("BACI Property name missing");
			setPageComplete(false);
		}

		else {
			setErrorMessage(null);
			setMessage(null, DialogPage.WARNING);
			setPageComplete(true);
		}
	}
}