/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.wizards;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.tmcdb.domain.HwConfiguration;

public class HwConfigurationChooserWizardPage extends WizardPage {
	
	private static final String CONFIGURATION_NAME = "Configuration name";
	private static final String SELECT_THE_CONFIGURATION = "configuration into which antenna is copied.";
	private AntennaNameWizardPage namePage;
	private Combo configCombo;
	private Text name;
	private String searchName;
	private HwConfiguration configuration;
	private HwConfiguration configToExclude;
	private Map<String, HwConfiguration> configsMap = new HashMap<String, HwConfiguration>();
	
	public HwConfigurationChooserWizardPage(String pageName, AntennaNameWizardPage namePage, HwConfiguration configToExclude) 
	{
		super(pageName);
		this.namePage = namePage;
		this.setDescription(SELECT_THE_CONFIGURATION);
		this.setTitle(pageName);
		this.configToExclude = configToExclude;
	}

	/* (non-Javadoc)
	 * @see alma.obops.tmcdbgui.wizards.ConfigurationFilterListener#setSearchName(java.lang.String)
	 */
	public void setSearchName(String searchName)
	{
		this.searchName = searchName; 
		populateCombo();
	}
	
	@Override
	public void createControl(Composite parent) 
	{	
		Composite composite = new Composite(parent, SWT.NONE);

		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		composite.setLayout( layout );

		// Configuration name
		Label lName = new Label( composite, SWT.NULL );
		lName.setText( CONFIGURATION_NAME );
		name = new Text( composite, SWT.BORDER | SWT.SINGLE );
		GridData gd = new GridData( GridData.FILL_HORIZONTAL );
		name.setLayoutData( gd );
		name.addKeyListener(new ConfigurationKeyListener());

		configCombo = new Combo(composite, SWT.NONE);
		configCombo.setEnabled(false);

		if(searchName == null) {
			searchName = "";
		}

		setPageComplete( false );
		setControl( composite );

		addSelectionListener();
	}
	
	private void populateCombo()
	{
		try {
			configCombo.removeAll();
			configsMap.clear();
			List<HwConfiguration> configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(searchName);
			for(HwConfiguration config : configs) {
				if(!config.getName().equals(this.configToExclude.getName())) {
					if(null == configsMap.get(config.getName())) {
						configCombo.add(config.getName());
					}
					configsMap.put(config.getName(), config);
				}
			}
			if(configsMap.size() > 0) {
				configCombo.setEnabled(true);
			} else {
				configCombo.setEnabled(false);
			}
		} catch (Exception e) {
			throw new RuntimeException("Could not query configurations.", e);
		}
	}
	
	private void addSelectionListener() {
		ConfigurationSelectionListener listener = new ConfigurationSelectionListener();
		configCombo.addSelectionListener( listener );
	}
	
	public HwConfiguration getConfiguration()
	{
		return configuration;
	}
	
	private class ConfigurationSelectionListener implements SelectionListener
	{
		@Override
		public void widgetDefaultSelected(SelectionEvent e) {
			widgetSelected(e);
		}

		@Override
		public void widgetSelected(SelectionEvent e) {
			configuration = configsMap.get(configCombo.getItem(configCombo.getSelectionIndex()));
			namePage.setConfiguration(configuration);
			setPageComplete(true);
		}
	}
	
	private class ConfigurationKeyListener implements KeyListener
	{

		@Override
		public void keyPressed(KeyEvent e) {
			// noop
		}

		@Override
		public void keyReleased(KeyEvent e) {
			setSearchName(name.getText());
		}
		
	}
}
