/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * NewComponentPage.java
 */
package alma.obops.tmcdbgui.wizards;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Widget;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Configuration;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.EclipseProgressMonitor;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.ComponentConversationUtils;

/**
 * Wizard page for creation of a new {@link Component} object.
 *
 * This page is used the {@link NewComponentWizard}
 *
 * @author rtobar, Mar 2, 2010
 */
public class ComponentChooserPageWithoutBaciProperties extends WizardPage implements Listener, ISelectionChangedListener {

	private static class ComponentChooserLabelProvider extends ColumnLabelProvider {

		public Image getImage( Object element )  {
			return ImageHelper.getImage((Component)element);
		}

		public String getText( Object element ) {
			return LabelHelper.getFullPath((Component)element, false);
		}
	}

	private class ComponentChooserContentProvider implements
			ITreeContentProvider 
	{

		private Configuration configuration;
		private ComponentType _compType;

		public ComponentChooserContentProvider(Configuration config, ComponentType type) {
			configuration = config;
			_compType = type;
		}

		@Override
		public Object[] getChildren(Object parentElement) {
			return null;
		}

		@Override
		public Object getParent(Object element) {
			return null;
		}

		@Override
		public boolean hasChildren(Object element) {
			return false;
		}

		@Override
		public Object[] getElements(Object inputElement) {

			final List<Component> fComps = new ArrayList<Component>();
			try {
				getContainer().run(false, false, new IRunnableWithProgress() {
					public void run(IProgressMonitor monitor) throws InvocationTargetException,
							InterruptedException {
						try {
							List<Component> comps;
							comps = ComponentConversationUtils.getInstance().findComponentsByComponentTypeId(_compType, configuration, new EclipseProgressMonitor(monitor));
							fComps.addAll(comps);
						} catch(Exception e) {
							RcpUtils.errorMessage(e, getContainer().getShell(), "Cannot get components", "Failed while trying to get components");
						}
					}
				});
			} catch (Exception e) {
				RcpUtils.errorMessage(e, getContainer().getShell(), "Cannot get components", "Failed while trying to get components");
			}
			return fComps.toArray();
		}

		@Override
		public void dispose() {}

		@Override
		public void inputChanged(Viewer theViewer, Object oldInput, Object newInput) {
			if( newInput != null ) {
				_compType = (ComponentType)newInput;
			}
		}

	}

	private Button selectAllButton;
	private Button selectNoneButton;
	private CheckboxTableViewer viewer;
	private Configuration _config;
	private ComponentTypeChooserPage _compTypeChooser;
	private Component[] _components;

	protected ComponentChooserPageWithoutBaciProperties(ComponentTypeChooserPage compTypeChooser, Configuration config) {
		super("Component Chooser");
		setTitle("Component Chooser");
		setDescription("Choose the components of interest");
		_compTypeChooser = compTypeChooser;
		_config = config;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createControl(Composite parent) {

		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout(1, false));

		GridData gd = new GridData(SWT.FILL, SWT.FILL, true, true);
		viewer = CheckboxTableViewer.newCheckList(composite, SWT.BORDER);
		viewer.getControl().setLayoutData(gd);
		viewer.setContentProvider( new ComponentChooserContentProvider(_config, this._compTypeChooser.getComponentType()) );
		viewer.setLabelProvider(new ComponentChooserLabelProvider() );
		viewer.setSorter(new ViewerSorter());
		viewer.addSelectionChangedListener(this);

		gd = new GridData(SWT.FILL, SWT.TOP, true, false);
		Composite downBar = new Composite(composite, SWT.NONE);
		downBar.setLayout(new GridLayout(2, false));
		downBar.setLayoutData( gd );

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		selectAllButton = new Button(downBar, SWT.PUSH);
		selectAllButton.setText("Select All");
		selectAllButton.addListener( SWT.Selection, this);
		selectAllButton.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalIndent = 10;
		selectNoneButton = new Button(downBar, SWT.PUSH);
		selectNoneButton.setText("Select None");
		selectNoneButton.addListener( SWT.Selection, this);
		selectNoneButton.setLayoutData(gd);

		setControl( composite );
		setPageComplete(false);
	}

	public void setVisible(boolean visible){
		if( visible )
			viewer.setInput( _compTypeChooser.getComponentType() );
		super.setVisible(visible);
	}

	@Override
	public void handleEvent(Event event) {

		SelectionChangedEvent selectionEvent = null;
		Widget w = event.widget;

		if( w == selectAllButton ) {
			selectionEvent = new SelectionChangedEvent(viewer, viewer.getSelection());
			viewer.setAllChecked(true);
			selectionChanged(selectionEvent);
		}
		else if( w == selectNoneButton ) {
			selectionEvent = new SelectionChangedEvent(viewer, viewer.getSelection());
			viewer.setAllChecked(false);
			selectionChanged(selectionEvent);
		}

	}

	@Override
	public void selectionChanged(SelectionChangedEvent event) {
		if( event.getSelectionProvider() == viewer ) {
			Object[] objs = viewer.getCheckedElements();	
			int length = objs.length;
			boolean isComplete = length != 0 ? true : false;
			setPageComplete( isComplete );
			_components = new Component[objs.length];
			System.arraycopy(objs, 0, _components, 0, objs.length);
		}
	}

	public Component[] getComponents() {
		return _components;
	}

}
