/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * NewComponentPage.java
 */
package alma.obops.tmcdbgui.wizards;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Widget;

import alma.acs.tmcdb.Component;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.EclipseProgressMonitor;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.LabelHelper;

/**
 * Wizard page for selection of {@link Component} objects.
 *
 * This page is used the {@link NewComponentWizard}
 *
 * @author rtobar, Mar 2, 2010
 */
public class ComponentChooserPage extends WizardPage implements Listener, ISelectionChangedListener, SelectionListener {

	public enum SelectionCriteriaForComponents {
		ALL_PROPERTIES,
		COMMON_PROPERTIES
	}

	private static class ComponentChooserLabelProvider extends ColumnLabelProvider {

		public Image getImage( Object element )  {
			return ImageHelper.getImage((Component)element);
		}

		public String getText( Object element ) {
			return LabelHelper.getFullPath((Component)element, false);
		}
	}

	private class ComponentChooserContentProvider implements
			ITreeContentProvider {

		private List<Component> contentProviderComponents;

		@Override
		public Object[] getChildren(Object parentElement) {
			return null;
		}

		@Override
		public Object getParent(Object element) {
			return null;
		}

		@Override
		public boolean hasChildren(Object element) {
			return false;
		}

		@Override
		public Object[] getElements(Object inputElement) {
			return contentProviderComponents.toArray();
		}

		@Override
		public void dispose() {}

		@Override
		public void inputChanged(Viewer theViewer, Object oldInput, final Object newInput) {

			contentProviderComponents = new ArrayList<Component>();
			if( newInput != null ) {
					try {
						getContainer().run(false, false, new IRunnableWithProgress() {
							public void run(IProgressMonitor monitor) throws InvocationTargetException,
									InterruptedException {
								ComponentProvider cp = (ComponentProvider)newInput;
								try {
									contentProviderComponents = cp.getComponents(new EclipseProgressMonitor(monitor));
								} catch(Exception e) {
									RcpUtils.errorMessage(e, getShell(), "Error while getting components", e.getMessage());
								}
							}
						});
					} catch (InvocationTargetException e) {
						e.printStackTrace();
					} catch (InterruptedException e) {
						e.printStackTrace();
					}
			}
		}

	}

	private Button selectAllButton;
	private Button selectNoneButton;
	private Button allPropertiesButton;
	private Button commonPropertiesButton;
	private CheckboxTableViewer viewer;

	private ComponentProvider _componentProvider;

	private SelectionCriteriaForComponents _selectionCriteria;
	private Component[] _components;

	private Composite topLevelComposite;
	private Composite filterComposite;

	public ComponentChooserPage() {
		super("Component Chooser");
		setTitle("Component Chooser");
		setDescription("Choose the components for which you want to modify BACI properties");
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createControl(Composite parent) {

		topLevelComposite = new Composite(parent, SWT.NONE);
		topLevelComposite.setLayout(new GridLayout(1, false));

		GridData gd;
		gd = new GridData(SWT.FILL, SWT.FILL, true, true);
		viewer = CheckboxTableViewer.newCheckList(topLevelComposite, SWT.BORDER);
		viewer.getControl().setLayoutData(gd);
		viewer.setContentProvider( new ComponentChooserContentProvider() );
		viewer.setLabelProvider(new ComponentChooserLabelProvider() );
		viewer.setSorter(new ViewerSorter());
		viewer.addSelectionChangedListener(this);

		// Buttons to select all/none components at once
		gd = new GridData(SWT.FILL, SWT.TOP, true, false);
		Composite buttonsBar = new Composite(topLevelComposite, SWT.NONE);
		buttonsBar.setLayout(new GridLayout(2, false));
		buttonsBar.setLayoutData( gd );

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		selectAllButton = new Button(buttonsBar, SWT.PUSH);
		selectAllButton.setText("Select All");
		selectAllButton.addListener( SWT.Selection, this);
		selectAllButton.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalIndent = 10;
		selectNoneButton = new Button(buttonsBar, SWT.PUSH);
		selectNoneButton.setText("Select None");
		selectNoneButton.addListener( SWT.Selection, this);
		selectNoneButton.setLayoutData(gd);

		// Radio buttons to select BACI property selection method
		gd = new GridData(SWT.FILL, SWT.CENTER, false, false);
		Group radioButtonsComposite = new Group(topLevelComposite, SWT.BORDER);
		radioButtonsComposite.setText("BACI properties selection");
		radioButtonsComposite.setLayoutData(gd);
		radioButtonsComposite.setLayout(new GridLayout(1, false));

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 10;
		allPropertiesButton = new Button(radioButtonsComposite, SWT.RADIO);
		allPropertiesButton.setText("Get all BACI properties for the selected components.");
		allPropertiesButton.setData(SelectionCriteriaForComponents.ALL_PROPERTIES);
		allPropertiesButton.setLayoutData(gd);
		allPropertiesButton.addSelectionListener(this);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 10;
		commonPropertiesButton = new Button(radioButtonsComposite, SWT.RADIO);
		commonPropertiesButton.setText("Get only the common BACI properties for the selected components.");
		commonPropertiesButton.setData(SelectionCriteriaForComponents.COMMON_PROPERTIES);
		commonPropertiesButton.setLayoutData(gd);
		commonPropertiesButton.addSelectionListener(this);

		if( _selectionCriteria != null && _selectionCriteria.equals(SelectionCriteriaForComponents.ALL_PROPERTIES) )
			allPropertiesButton.setSelection(true);
		else if( _selectionCriteria != null && _selectionCriteria.equals(SelectionCriteriaForComponents.COMMON_PROPERTIES) )
			commonPropertiesButton.setSelection(true);

		setControl( topLevelComposite );
		setPageComplete(false);
	}

	public void setComponentProvier(ComponentProvider componentProvider) {
		_componentProvider = componentProvider;
	}

	public void setVisible(boolean visible){
		if( visible )
			viewer.setInput( _componentProvider );
		super.setVisible(visible);
	}

	public void setUseFilter(boolean useFilter) {

		if( topLevelComposite == null )
			return;

		if( useFilter )
			addFilterWidgets();
		else
			removeFilterWidgets();
	}

	private void addFilterWidgets() {

		// The widgets are already in place
		if( filterComposite != null )
			return;

		GridData gd;

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		filterComposite = new Composite(topLevelComposite, SWT.NONE);
		filterComposite.setLayoutData(gd);
		filterComposite.setLayout(new GridLayout(2, false));

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label filterLabel = new Label(filterComposite, SWT.NONE);
		filterLabel.setText("Filter:");
		filterLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		final Text filterText = new Text(filterComposite, SWT.BORDER | SWT.SINGLE);
		filterText.setLayoutData(gd);

		filterText.addKeyListener(new KeyListener() {
			public void keyReleased(KeyEvent e) {
				viewer.refresh();
			}
			public void keyPressed(KeyEvent e) {
			}
		});

		ViewerFilter[] filters = new ViewerFilter[] {
				new ViewerFilter() {
					public boolean select(Viewer theViewer, Object parentElement, Object element) {

						String filter = filterText.getText().trim().toLowerCase();
						if( filter.length() == 0 )
							return true;

						Component c = (Component)element;
						if( c.getPath().toLowerCase().contains(filter) ||
						    c.getComponentName().toLowerCase().contains(filter) ||
						    LabelHelper.getFullPath(c, false).toLowerCase().contains(filter) )
							return true;

						return false;
					}
				}
		};
		viewer.setFilters(filters);

		filterComposite.moveAbove(null);
		topLevelComposite.layout();
	}

	private void removeFilterWidgets() {

		// The widgets have been already removed
		if( filterComposite == null )
			return;

		viewer.resetFilters();
		filterComposite.dispose();
		filterComposite = null;

		topLevelComposite.layout();
	}

	@Override
	public void handleEvent(Event event) {

		SelectionChangedEvent selectionEvent = null;
		Widget w = event.widget;

		if( w == selectAllButton ) {
			selectionEvent = new SelectionChangedEvent(viewer, viewer.getSelection());
			viewer.setAllChecked(true);
			selectionChanged(selectionEvent);
		}
		else if( w == selectNoneButton ) {
			selectionEvent = new SelectionChangedEvent(viewer, viewer.getSelection());
			viewer.setAllChecked(false);
			selectionChanged(selectionEvent);
		}

	}

	@Override
	public void selectionChanged(SelectionChangedEvent event) {
		if( event.getSelectionProvider() == viewer ) {
			togglePageComplete();
			Object[] objs = viewer.getCheckedElements();
			_components = new Component[objs.length];
			System.arraycopy(objs, 0, _components, 0, objs.length);
		}
	}

	private void togglePageComplete() {
		setPageComplete(viewer.getCheckedElements().length != 0 && _selectionCriteria != null);
	}

	public Component[] getComponents() {
		return _components;
	}

	public SelectionCriteriaForComponents getSelectionCriteria() {
		return _selectionCriteria;
	}

	public void setSelectionCriteria(SelectionCriteriaForComponents criteria) {

		_selectionCriteria = criteria;

		if( allPropertiesButton != null && criteria.equals(SelectionCriteriaForComponents.ALL_PROPERTIES) )
			allPropertiesButton.setSelection(true);
		else if( commonPropertiesButton != null && criteria.equals(SelectionCriteriaForComponents.COMMON_PROPERTIES) )
			commonPropertiesButton.setSelection(true);

	}

	@Override
	public void widgetSelected(SelectionEvent e) {
		Button b = (Button)e.widget;
		if( !b.getSelection() )
			return;

		_selectionCriteria = (SelectionCriteriaForComponents)b.getData();
		togglePageComplete();
	}

	@Override
	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}


	public static void main(String[] args) {
		Display d = new Display();
		Shell window = new Shell(d);
		WizardDialog w = new WizardDialog(window, new Wizard() {
			public void addPages() {
				ComponentChooserPage page = new ComponentChooserPage();
				page.setUseFilter(false);
				page.setComponentProvier(new ComponentProvider() {
					public List<Component> getComponents(EclipseProgressMonitor monitor)
							throws Exception {
						List<Component> components = new ArrayList<Component>();

						Component c = new Component();
						c.setComponentName("C");
						c.setPath("CONTROL");
						components.add(c);

						c = new Component();
						c.setComponentName("A");
						c.setPath("CONTROL");
						components.add(c);
						return components;
					}
				});
				addPage(page);
			}
			@Override
			public boolean performFinish() {
				return false;
			}
		});
		w.create();
		w.open();
	}
}
