/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.wizards;

import java.util.Set;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;

import alma.obops.tmcdbgui.utils.TmcdbConstants;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaType;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;


/**
 * Wizard page for cloning of an antenna.
 * @author sharring
 */
public class CloneAntennaWizardPage extends WizardPage 
{
	private static final String NAME = "Name";
	private static final String NUMBER = "Number";
	private static final String PREFIX = "Prefix";
	private Combo namePrefixCombo, nameNumberCombo;
	private Group namePrefixAndNumber;
    private Set<BaseElement> baseElements;
    private Antenna antennaToClone;
    private boolean complete;
	

	protected CloneAntennaWizardPage(String pageName, Antenna antennaToClone) 
	{
		super(pageName);
		setTitle(pageName);
		setDescription("Configure the cloned antenna");
		this.antennaToClone = antennaToClone;
		this.setPageComplete(false);
		this.complete = false;
	}

	@Override
	public void createControl(Composite parent) 
	{
		namePrefixAndNumber = new Group(parent, SWT.NONE);
		namePrefixAndNumber.setText(NAME);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 4;
		namePrefixAndNumber.setLayout(gridLayout);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 4;
		namePrefixAndNumber.setLayoutData(gridData);

		// Antenna name prefix
		Label lName = new Label( namePrefixAndNumber, SWT.NULL );
		lName.setText( PREFIX );
		namePrefixCombo = new Combo( namePrefixAndNumber, SWT.READ_ONLY );
		
		// Antenna number
		Label lNumber = new Label( namePrefixAndNumber, SWT.NULL );
		lNumber.setText( NUMBER );
		nameNumberCombo = new Combo( namePrefixAndNumber, SWT.READ_ONLY );
		
		if(antennaToClone.getAntennaType().equals(AntennaType.ACA)) 
		{
			String originalPrefix = this.antennaToClone.getName().substring(0, 2);
			if(originalPrefix.equals(TmcdbConstants.PM)) {
				String [] items = { TmcdbConstants.PM };
				namePrefixCombo.setItems(items);
				nameNumberCombo.setItems(TmcdbConstants.PM_NUMBERS);
			} else {
				String [] items = { TmcdbConstants.CM };
				namePrefixCombo.setItems(items);
				nameNumberCombo.setItems(TmcdbConstants.CM_NUMBERS);
			}
		} 
		else if(antennaToClone.getAntennaType().equals(AntennaType.AEC)) {
			String [] items = { TmcdbConstants.DA };
			namePrefixCombo.setItems(items);
			nameNumberCombo.setItems(TmcdbConstants.DA_NUMBERS);
		}
		else if(antennaToClone.getAntennaType().equals(AntennaType.VA) && parsePrefix(antennaToClone.getName()).equals(TmcdbConstants.LA)) {
			String [] items = { TmcdbConstants.LA };
			namePrefixCombo.setItems(items);	
			nameNumberCombo.setItems(TmcdbConstants.DV_NUMBERS);
		}
		else if(antennaToClone.getAntennaType().equals(AntennaType.VA) && parsePrefix(antennaToClone.getName()).equals(TmcdbConstants.DV)) {
			String [] items = { TmcdbConstants.DV };
			namePrefixCombo.setItems(items);	
			nameNumberCombo.setItems(TmcdbConstants.DV_NUMBERS);
		}
		
		namePrefixCombo.select(0);
		namePrefixCombo.setEnabled(false);
		namePrefixCombo.setLayoutData( new GridData() );
		nameNumberCombo.select(-1);
		nameNumberCombo.setLayoutData( new GridData() );
		
		setControl(namePrefixAndNumber);
		
		addSelectionListeners();
	}
	
	private String parsePrefix(String name) {
		String retVal = null;
		retVal = name.substring(0, 2);
		return retVal;
	}
	
	private void updateComplete()
	{
		this.complete = 
			(namePrefixCombo.getSelectionIndex() != -1) &&
			(nameNumberCombo.getSelectionIndex() != -1) &&
			!antennaExistsInConfig();
		this.setPageComplete(complete);
	}
	
	@Override
	public boolean isPageComplete()
	{
		return this.complete;
	}
	
	private boolean antennaExistsInConfig()
	{
		boolean retVal = false;
		
		if(null == this.baseElements) {
			this.baseElements = antennaToClone.getConfiguration().getBaseElements();
		}
		// TODO: replace with a find, for performance(?)
		for(BaseElement be: baseElements) 
		{
			if(be.getType().equals(BaseElementType.Antenna) && be.getName().equals(getName())) 
			{
				retVal = true;
				break;
			}
		}
		
		if(retVal == true) {
			this.setErrorMessage("Antenna already exists: prefix + number must be unique");
		} else {
			this.setErrorMessage(null);
		}
		return retVal;
	}

	public String getName() 
	{
		return namePrefixCombo.getText() + nameNumberCombo.getText();
	}
	
	private void addSelectionListeners() 
	{
		AntennaAttributesSelectionListener listener = new AntennaAttributesSelectionListener();
        namePrefixCombo.addSelectionListener(listener);
        nameNumberCombo.addSelectionListener(listener);
	}
	
	/**
	 * Private class to handle the dependencies (interrelatedness) between the widgets, so that invalid
	 * choices are not possible.
	 * 
	 * @author sharring
	 */
	private class  AntennaAttributesSelectionListener implements SelectionListener
	{
		private String previousNamePrefix = "";
		
		public void widgetDefaultSelected( SelectionEvent e ) 
		{
			if(e.widget == namePrefixCombo && namePrefixCombo.getSelectionIndex() != -1) {
				updateForNamePrefixSelection();
			}
			updateComplete();
		}

		public void widgetSelected( SelectionEvent e ) 
		{
			if(e.widget == namePrefixCombo && namePrefixCombo.getSelectionIndex() != -1) {
				updateForNamePrefixSelection();
			}
			updateComplete();
		}
		
		private void updateForNamePrefixSelection() {
			if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(TmcdbConstants.PM) && !previousNamePrefix.equals(TmcdbConstants.PM)) {
				updateFieldsForPMPrefix();
			} else if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(TmcdbConstants.CM) && !previousNamePrefix.equals(TmcdbConstants.CM)) {
				updateFieldsForCMPrefix();
			}
			previousNamePrefix = namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex());
		}
	}
	
	private void updateFieldsForPMPrefix() {
		nameNumberCombo.setItems(TmcdbConstants.PM_NUMBERS);
	}

	private void updateFieldsForCMPrefix() {
		nameNumberCombo.setItems(TmcdbConstants.CM_NUMBERS);
	}
}
