/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.wizards;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.widgets.support.HwConfigurationListener;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.Pad;

/**
 * Wizard page which allows a user to choose a pad.
 * @author sharring
 */
public class ChoosePadWizardPage extends WizardPage implements HwConfigurationListener
{
	private HwConfiguration config, workingConfig;
	private Pad pad;
	private PadChooserComposite comp;
	
	public ChoosePadWizardPage(String pageName, HwConfiguration workingConfig) 
	{
		super(pageName);
		setPageComplete(false);
		setTitle("Please select a pad");
		this.workingConfig = workingConfig;
	}

	@Override
	public void createControl(Composite parent) 
	{
		comp = new PadChooserComposite(parent, SWT.NONE);
		comp.refreshItems();
		this.setControl(comp);
	}
	
	private class PadChooserComposite extends Composite
	{
		private Map<String, Pad> padsMap;
		private Combo typeCombo;
		
		public PadChooserComposite(Composite parent, int style) 
		{
			super(parent, style);
			padsMap = new HashMap<String, Pad>();
			createControl();
		}

		private void createControl() 
		{
			GridLayout gridLayout = new GridLayout();
			gridLayout.numColumns = 2;
			this.setLayout(gridLayout);
			
			GridData gd = new GridData();
			gd.grabExcessHorizontalSpace = false;
			
			Label typeLabel = new Label(this, SWT.NONE);
			typeLabel.setText("Pad: ");
			typeLabel.setLayoutData(gd);
			
			typeCombo = new Combo(this, SWT.NONE);
			
			typeCombo.setLayoutData(new GridData());
			typeCombo.setText("Select pad");
			
			typeCombo.addSelectionListener(new SelectionListener() 
			{
				@Override
				public void widgetDefaultSelected(SelectionEvent selEvt) {
					widgetSelected(selEvt);
				}

				@Override
				public void widgetSelected(SelectionEvent selEvt) {
					setPageComplete(true);
					String item = typeCombo.getItem(typeCombo.getSelectionIndex());
					pad  = padsMap.get(item);
				}
			});
		}
		
		// this method creates an array of strings representing the available pads
		// for selection; note that it filters out any pads which already exist in the
		// working config (this is used e.g. when working with a 'global' configuration, 
		// wherein we don't allow the user to pick a pad that exists in the 'local' config.
		private String[] getPadStrings() 
		{
			HwConfiguration configToUse = (config != null) ? config : workingConfig;
			String [] retVal = null;
			try 
			{
				List<Pad> pads = new ArrayList<Pad>();
				for(BaseElement be : configToUse.getBaseElements())
				{
					if(be instanceof Pad)
					{
						// if the pad is not already in our working config
						// then we will add it; else it should *not* be added
						if(!padExistsInWorkingConfig((Pad)be)) 
						{
							pads.add((Pad)be);
						}
					}
				}
				
				retVal = new String[pads.size()];
				int count = 0;
				for(Pad iteratedPad: pads)
				{
					retVal[count++] = iteratedPad.getName();
					padsMap.put(iteratedPad.getName(), iteratedPad);
				}
			} 
			catch (Exception e) 
			{
				throw new RuntimeException("Could not get names of pads");
			}
			Arrays.sort(retVal);
			return retVal;
		}
		
		private boolean padExistsInWorkingConfig(Pad padToFind) 
		{
			{
				boolean retVal = false;
				
				for(BaseElement be: workingConfig.getBaseElements()) 
				{
					if(be.getType().equals(BaseElementType.Pad) && be instanceof Pad) 
					{
						Pad padIterated = (Pad) be;
						// if the names are identical, but the configuration id differs,
						// then we have found a pad which already exists in the working config
						if(padIterated.getName().equals(padToFind.getName()) && 
						   !padIterated.getConfiguration().getId().equals(padToFind.getConfiguration().getId())) 
						{
							retVal = true;
							break;
						}
					}
				}
				
				return retVal;
			}
		}

		public void refreshItems() {
			typeCombo.setItems(getPadStrings());
		}
	}

	public Pad getPad() 
	{
		return pad;
	}

	@Override
	public void setHwConfiguration(HwConfiguration config) {
		try {
			HwConfigurationConversationUtils.getInstance().hydrateBaseElements(config);
		} catch (Exception e) {
			e.printStackTrace();
			throw new RuntimeException(e);
		}
		this.config = config;
		if(null != comp) {
			comp.refreshItems();
		}
	}

}
