/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.wizards;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import alma.obops.tmcdbgui.widgets.support.HwConfigurationListener;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.HolographyTower;
import alma.tmcdb.domain.HwConfiguration;

public class ChooseHolographyTowerWizardPage extends WizardPage implements HwConfigurationListener
{
	private HolographyTowerChooserComposite comp;
	private HwConfiguration config, workingConfig;
	private HolographyTower holographyTower;
	
	public ChooseHolographyTowerWizardPage(String pageName, HwConfiguration config) 
	{
		super(pageName);
		setPageComplete(false);
		setTitle("Please select a holography tower");
		this.workingConfig = config;
	}

	@Override
	public void createControl(Composite parent) 
	{
		comp = new HolographyTowerChooserComposite(parent, SWT.NONE);
		comp.refreshItems();
		this.setControl(comp);
	}
	
	private class HolographyTowerChooserComposite extends Composite
	{
		private Combo typeCombo;
		private Map<String, HolographyTower> holographyTowersMap;
		
		public HolographyTowerChooserComposite(Composite parent, int style) 
		{
			super(parent, style);
			holographyTowersMap = new HashMap<String, HolographyTower>();
			String items[] = getHolographyTowerStrings();
			createControl(items);
		}

		public void refreshItems() {
			typeCombo.setItems(getHolographyTowerStrings());
		}

		private void createControl(String items[]) 
		{
			GridLayout gridLayout = new GridLayout();
			gridLayout.numColumns = 2;
			this.setLayout(gridLayout);
			
			GridData gd = new GridData();
			gd.grabExcessHorizontalSpace = false;
			
			Label typeLabel = new Label(this, SWT.NONE);
			typeLabel.setText("HolographyTower: ");
			typeLabel.setLayoutData(gd);
			
			typeCombo = new Combo(this, SWT.NONE);
			
			typeCombo.setItems(items);
			typeCombo.setLayoutData(new GridData());
			typeCombo.setText("Select tower");
			
			typeCombo.addSelectionListener(new SelectionListener() 
			{
				@Override
				public void widgetDefaultSelected(SelectionEvent selEvt) {
					widgetSelected(selEvt);
				}

				@Override
				public void widgetSelected(SelectionEvent selEvt) {
					setPageComplete(true);
					String item = typeCombo.getItem(typeCombo.getSelectionIndex());
					holographyTower  = holographyTowersMap.get(item);
				}
			});
		}
		
		private String[] getHolographyTowerStrings() 
		{
			HwConfiguration configToUse = (config != null) ? config : workingConfig;
			String [] retVal = null;
			try 
			{
				List<HolographyTower> holographyTowers = new ArrayList<HolographyTower>();
				for(BaseElement be : configToUse.getBaseElements())
				{
					if(be instanceof HolographyTower)
					{
						if(!holographyTowerExistsInWorkingConfig((HolographyTower)be)) 
						{
							holographyTowers.add((HolographyTower)be);
						} 
					}
				}
				
				retVal = new String[holographyTowers.size()];
				int count = 0;
				for(HolographyTower ht: holographyTowers)
				{
					retVal[count++] = ht.getName();
					holographyTowersMap.put(ht.getName(), ht);
				}
			} 
			catch (Exception e) 
			{
				throw new RuntimeException("Could not get names of holography towers");
			}
			return retVal;
		}

		private boolean holographyTowerExistsInWorkingConfig(HolographyTower htToFind) 
		{
			boolean retVal = false;
			
			for(BaseElement be: workingConfig.getBaseElements()) {
				if(be.getType().equals(BaseElementType.HolographyTower) && be instanceof HolographyTower) {
					HolographyTower htIterated = (HolographyTower) be;
					if(htIterated.getName().equals(htToFind.getName()) && 
					   !htIterated.getConfiguration().getId().equals(htToFind.getConfiguration().getId())) 
					{
						retVal = true;
						break;
					}
				}
			}
			
			return retVal;
		}
	}

	public HolographyTower getHolographyTower() 
	{
		return holographyTower;
	}

	@Override
	public void setHwConfiguration(HwConfiguration config) 
	{
		this.config = config;
		if(null != comp) {
			comp.refreshItems();
		}
	}

}
