/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.wizards;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.hibernate.criterion.MatchMode;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.dialogs.ConfigurationSelectionDialog;
import alma.obops.tmcdbgui.dialogs.ConfigurationSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.widgets.support.HwConfigurationListener;
import alma.tmcdb.domain.HwConfiguration;

public class ChooseConfigurationWizardPage extends WizardPage 
{
	private HwConfiguration configuration;
	private HwConfigurationListener listener;
	private Text configurationNameText;
	
	public ChooseConfigurationWizardPage(String pageName, HwConfigurationListener listener, HwConfiguration configuration) 
	{
		super(pageName);
		this.configuration = configuration;
		this.listener = listener;
		this.setDescription("Choose a configuration");
		this.setTitle(pageName);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createControl(Composite parent) 
	{
		final Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout(2, false));

		/* Configuration */
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label configurationLabel = new Label(composite, SWT.NONE);
		configurationLabel.setText("Configuration");
		configurationLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		configurationNameText = new Text(composite, SWT.BORDER | SWT.SINGLE);
		configurationNameText.setEditable(false);
		if( configuration != null) {
			listener.setHwConfiguration(configuration);
			configurationNameText.setText(configuration.getName());
			configurationNameText.setToolTipText(configuration.getFullName());
		}
		configurationNameText.setLayoutData(gd);
		
		// Setup the browser buttons and logic
		// Configurations should always be enabled, and browse button provokes to appear a dialog
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Button browseConfigs = new Button(composite, SWT.PUSH);
		browseConfigs.setText("Browse...");
		browseConfigs.setLayoutData(gd);
		
		browseConfigs.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				Cursor cursor = composite.getCursor();
				composite.setCursor(composite.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
				displayDialog();
				composite.setCursor(cursor);
			}
			
			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		
		composite.pack();
		setPageComplete(configuration != null);
		this.setControl(composite);
	}
	
	private void displayDialog() 
	{
		Collection<HwConfiguration> configsToUse = new ArrayList<HwConfiguration>();
		configsToUse.add(configuration);
		if(configuration.getGlobalConfiguration() != null) {
			configsToUse.add(configuration.getGlobalConfiguration());
		}
		List<String> configNamesToUse = new ArrayList<String>();
		for(HwConfiguration config : configsToUse) {
			configNamesToUse.add(config.getSwConfiguration().getConfigurationName());
		}
		ElementListSelectionDialog d = 
			new ConfigurationSelectionDialog(configNamesToUse.toArray(new String[0]), getShell(), 
					new ConfigurationSelectionDialogLabelProvider());
		d.open();
		Object configNames[] = d.getResult();
		
		if( configNames == null || configNames.length != 1) {
			RcpUtils.infoMessage(getShell(), "No Configuration selected", "No configuration was selected. No action will be taken");
			return;
		}
		
		List<HwConfiguration> matchingHwConfigs = null;
		try {
			matchingHwConfigs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName((String) configNames[0], MatchMode.EXACT);
		} catch (Exception e) {
			matchingHwConfigs = null;
		}
		
		if(matchingHwConfigs != null && matchingHwConfigs.size() == 1) {
			HwConfiguration configurationSelected = matchingHwConfigs.get(0);
			listener.setHwConfiguration(configurationSelected);
			configurationNameText.setText(configurationSelected.getName());
			setPageComplete(true);
		}
		else {
			setPageComplete(false);
			RcpUtils.infoMessage(getShell(), "Error", "Problem loading chosen configuration. No action will be taken");
			return;
		}
	}
}
