/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.wizards;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.StartupScenarioConversationUtils;
import alma.obops.tmcdbgui.widgets.support.HwConfigurationListener;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.StartupScenario;

public class ChooseBaseElementsWizardPage extends WizardPage implements HwConfigurationListener
{
	private HwConfiguration config;
	private BaseElement[] baseElements;
	private StartupScenario destinationStartupScenario;
	
	public ChooseBaseElementsWizardPage(String pageName, HwConfiguration config, StartupScenario destinationStartupScenario) 
	{
		super(pageName);
		setPageComplete(false);
		setTitle("Please select base elements");
		this.config = config;
		this.destinationStartupScenario = destinationStartupScenario;
	}

	@Override
	public void createControl(Composite parent) 
	{
		Composite comp = new BaseElementsChooserComposite(parent, SWT.NONE);
		this.setControl(comp);
	}
	
	private class BaseElementsChooserComposite extends Composite
	{	
		public BaseElementsChooserComposite(Composite parent, int style) 
		{
			super(parent, style);
			createControl();
		}

		private void createControl() 
		{
			setLayout(new GridLayout(2, false));
			GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
			Label baseElementsLabel = new Label(this, SWT.NONE);
			baseElementsLabel.setText("Base Element(s)");
			baseElementsLabel.setLayoutData(gd);

			gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
			gd.horizontalIndent = 20;
			final Text baseElementsName = new Text(this, SWT.BORDER | SWT.SINGLE);
			baseElementsName.setEditable(false);
			baseElementsName.setLayoutData(gd);

			gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
			Button browseButton = new Button(this, SWT.PUSH);
			browseButton.setText("Browse...");
			browseButton.setLayoutData(gd);
			
			// Setup the browser buttons and logic
			// Configurations should always be enabled, and browse button provokes to appear a dialog
			browseButton.addSelectionListener(new SelectionListener() {
				public void widgetSelected(SelectionEvent e) {
					setPageComplete(true);
					ElementListSelectionDialog d = new BaseElementSelectionDialog(getShell(), new BaseElementSelectionDialogLabelProvider());
					d.open();
					Object selectedItems[] = d.getResult();
					List<BaseElement> selectedBes = new ArrayList<BaseElement>();
					for(Object be: selectedItems) {
						selectedBes.add((BaseElement)be);
					}
					baseElements = selectedBes.toArray(new BaseElement[selectedBes.size()]);
					if(baseElements.length > 1) {
						int count = 0;
						StringBuffer sb = new StringBuffer();
						for(BaseElement baseEl : baseElements) {
							sb.append( baseEl.getName());
							sb.append((++count >= baseElements.length ? "" : ", "));	
						}
						baseElementsName.setText(sb.toString());
					} else if(baseElements.length == 1) {
						baseElementsName.setText( baseElements[0].getName());
					}
				}
				public void widgetDefaultSelected(SelectionEvent e) {}
			});
		}
	}

	public BaseElement[] getBaseElements() 
	{
		return baseElements;
	}

	@Override
	public void setHwConfiguration(HwConfiguration config) {
		this.config = config;
	}
	
	private class BaseElementSelectionDialog extends ElementListSelectionDialog {

		public BaseElementSelectionDialog(Shell parent, ILabelProvider renderer) {
			super(parent, renderer);

			BaseElement[] topLevelBaseElements = null;
			try {
				parent.setCursor(parent.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
				topLevelBaseElements = HwConfigurationConversationUtils.getInstance().findTopLevelBaseElementsByConfiguration(config);
				for(BaseElement be: topLevelBaseElements) {
					if(!be.getConfiguration().getId().equals(destinationStartupScenario.getConfiguration().getId())) {
						HwConfigurationConversationUtils.getInstance().hydrateConfigurationHashCode(be.getConfiguration());
					}
				}
				topLevelBaseElements = filterOutBaseElementsAlreadyInStartup(topLevelBaseElements);
			} catch(Exception e) {
				e.printStackTrace();
			} finally {
				parent.setCursor(null);
			}
			setElements(topLevelBaseElements);

			setIgnoreCase(true);
			setMessage("Select the base element(s) you wish to add");
			setMultipleSelection(true);
		}

		private BaseElement[] filterOutBaseElementsAlreadyInStartup(BaseElement[] topLevelBaseElements) 
		{	
			try {
				StartupScenarioConversationUtils.getInstance().hydrateBaseElementStartups(destinationStartupScenario);
			} catch (Exception e) {
				e.printStackTrace();
				throw new RuntimeException("Could not hydrate baseelementstartups in startup scenario");
			}
			List<BaseElement> filteredList = new ArrayList<BaseElement>();
			for(BaseElement be: topLevelBaseElements ) {
				if(!baseElementStartupInScenario(be)) {
					filteredList.add(be);
				}
			}
			return filteredList.toArray(new BaseElement[0]);
		}

		private boolean baseElementStartupInScenario(BaseElement be) 
		{
			boolean retVal = false;
			for(BaseElementStartup bes : destinationStartupScenario.getBaseElementStartups()) {
				if(bes.getBaseElement().getType().equals(be.getType()) && bes.getBaseElement().getName().equals(be.getName())) {
					return true;
				}
			}
			return retVal;
		}

		protected Control createDialogArea(Composite parent) {
			Control control = super.createDialogArea(parent);
			getShell().setText("Base Element selection");
			return control;
		}
	}
	
	private static class BaseElementSelectionDialogLabelProvider implements ILabelProvider {

		@Override
		public Image getImage(Object element) 
		{
			Image retVal = null;
			if(element instanceof BaseElement) 
			{
				BaseElement be = (BaseElement) element;
				switch(be.getType()) 
				{
				// only dealing with top level BE's (antenna, aostiming, centralLO, weatherstation)
				case Antenna:
					retVal = RcpUtils.getImage("icons/antenna.png");
					break;
				case AOSTiming:
					retVal = RcpUtils.getImage("icons/masterclock.gif");
					break;
				case CentralLO:
					retVal = RcpUtils.getImage("icons/centralrack.png");
					break;
				case WeatherStationController:
					retVal = RcpUtils.getImage("icons/weatherstation.png");
					break;
				default: 
					retVal = null;
					break;
				}
			}
			return retVal;
		}

		@Override
		public String getText(Object element) {
			if( element instanceof BaseElement )
				return ((BaseElement)element).getName();
			return null;
		}

		public void addListener(ILabelProviderListener listener) { }

		public void dispose() {}

		public boolean isLabelProperty(Object element, String property) { return false; }

		public void removeListener(ILabelProviderListener listener) { }

	}
	
}
