/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.wizards;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.Wizard;

import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Configuration;
import alma.obops.dam.utils.ProgressMonitor;
import alma.obops.tmcdbgui.rcp.TmcdbExplorer;
import alma.obops.tmcdbgui.utils.EclipseProgressMonitor;
import alma.obops.tmcdbgui.utils.conversation.BACIPropertyConversationUtils;
import alma.obops.tmcdbgui.wizards.ComponentChooserPage.SelectionCriteriaForComponents;

public class BulkBACIPropertiesDeletionWizard extends Wizard
{

	private Configuration _config;
	private ComponentType _compType;

	private ComponentTypeChooserPage componentTypeChooserPage;
	private ComponentChooserPage componentChooserPage;
	private BACIPropertiesChooserPage baciPropertiesChooserPage;

	public BulkBACIPropertiesDeletionWizard(Configuration config, ComponentType compType) {
		_config   = config;
		_compType = compType;
		setNeedsProgressMonitor(true);
	}

	@Override
    public void addPages()
    {
    	componentTypeChooserPage = new ComponentTypeChooserPage(_compType);
    	componentChooserPage = new ComponentChooserPage();
    	baciPropertiesChooserPage = new BACIPropertiesChooserPage();

    	addPage( componentTypeChooserPage );
        addPage( componentChooserPage );
        addPage( baciPropertiesChooserPage );
    }

	@Override
	public IWizardPage getNextPage(IWizardPage page) {

		IWizardPage next = super.getNextPage(page);

		// If we come from the ComponentType chooser, we setup the
		// component chooser page correctly
		if( page == componentTypeChooserPage ) {
			componentChooserPage.setComponentProvier(new ComponentTypeComponentProvider(componentTypeChooserPage.getComponentType(), _config));
			componentChooserPage.setUseFilter(false);
			componentChooserPage.setSelectionCriteria(SelectionCriteriaForComponents.COMMON_PROPERTIES);
			next = componentChooserPage;
		}

		// If we come from the Component chooser, we setup the BACI property chooser
		// page according to the selection choice
		if( page == componentChooserPage ) {

			BACIPropertiesProvider provider = null;
			SelectionCriteriaForComponents selectionCriteria = componentChooserPage.getSelectionCriteria();

			if( selectionCriteria.equals(SelectionCriteriaForComponents.ALL_PROPERTIES) ) {
				provider = new AllBACIPropertiesProvider(componentChooserPage.getComponents());
				baciPropertiesChooserPage.setShowComponentName(true);
			}
			else if( selectionCriteria.equals(SelectionCriteriaForComponents.COMMON_PROPERTIES) ) {
				provider = new CommonBACIPropertiesProvider(componentChooserPage.getComponents());
				baciPropertiesChooserPage.setShowComponentName(false);
			}

			baciPropertiesChooserPage.setBACIPropertiesProdiver(provider);
			next = baciPropertiesChooserPage;
		}
		return next;
	}

	@Override
	public boolean performFinish()
	{

		Component[] components = componentChooserPage.getComponents();
		BACIProperty[] selectedProperties = baciPropertiesChooserPage.getBACIProperties();
		List<BACIProperty> propsToDelete = new ArrayList<BACIProperty>();

		// With the selected components, and selected BACI props,
		// compute the actual BACIProperty objects that are meant to be updated
		// in the database
		for(Component comp: components) {
			for(BACIProperty prop: selectedProperties) {
				for(BACIProperty compProp: comp.getBACIProperties()) {
					if( compProp.getPropertyName().equals(prop.getPropertyName()) ) {
						propsToDelete.add(compProp);
						break;
					}
				}
			}
		}

		final BACIProperty[] props = propsToDelete.toArray(new BACIProperty[0]);

		// TODO: Temporary solution doesn't allow to delete properties that contain monitor data
		// This is because the BACIPropertyService.delete() method must be improved to do so.
		// When done, the openConfirm() call should be put inside AND'd in the if
		try {
			if( BACIPropertyConversationUtils.getInstance().baciPropertiesHaveMonitorData(props) ) {
			    MessageDialog.openConfirm(getShell(), "Properties contain Monitoring Data",
			    "One or more of the selected BACI Properties contain monitoring data. Do you whish to continue?");
				return true;
			}
		} catch (Exception e1) {
			e1.printStackTrace();
			MessageDialog.openConfirm(getShell(), "There was an error while determining if the properties contain monitoring data",
		    "One or more of the selected BACI Properties may contain monitoring data. Cannot continue until this error is resolved.");
			return true;
		}

		Job job = new Job("Deleting BACI properties") {
			protected IStatus run(IProgressMonitor monitor) {

				ProgressMonitor eclipseMonitor = new EclipseProgressMonitor(monitor);
				try {
					BACIPropertyConversationUtils.getInstance().bulkDeleteBACIProperties(props, eclipseMonitor);
				} catch(Exception e) {
					return new Status(IStatus.ERROR, TmcdbExplorer.PLUGIN_ID, e.getMessage(), e);
				} finally {
					eclipseMonitor.done();
				}

				return Status.OK_STATUS;
			}
		};

		job.setUser(true);
		job.schedule();
        return true;
	}

}
