/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.wizards;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.Wizard;

import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Configuration;
import alma.obops.dam.utils.ProgressMonitor;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.rcp.TmcdbExplorer;
import alma.obops.tmcdbgui.utils.EclipseProgressMonitor;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.BACIPropertyConversationUtils;
import alma.obops.tmcdbgui.widgets.BACIPropertyEditingComposite;

public class BulkBACIPropertiesCreationWizard extends Wizard
{

	private Configuration _config;
	private ComponentType _compType;

	private ComponentTypeChooserPage componentTypeChooserPage;
	private ComponentChooserPage componentChooserPage;
	private BACIPropertiesEditorPage baciPropertiesEditorPage;

	public BulkBACIPropertiesCreationWizard(Configuration config, ComponentType compType) {
		_config   = config;
		_compType = compType;
		setNeedsProgressMonitor(true);
	}

    public void addPages()
    {
    	componentTypeChooserPage = new ComponentTypeChooserPage(_compType);
    	componentChooserPage = new ComponentChooserPage();
    	baciPropertiesEditorPage = new BACIPropertiesEditorPage("BACI Property specification", "Specify which values are to be taken into acount to search for matching BACI properties");

    	addPage( componentTypeChooserPage );
        addPage( componentChooserPage );
        addPage( baciPropertiesEditorPage );
    }

	@Override
	public IWizardPage getNextPage(IWizardPage page) {

		IWizardPage next = super.getNextPage(page);
		if( page == componentChooserPage ) {
			BACIProperty newProperty = new BACIProperty();
			Component[] comps = componentChooserPage.getComponents();
			if( comps != null && comps.length != 0 ) {
				newProperty.setComponent(comps[0]); // Whatever, just to have a BACI->Comp->Conf link stablished
				baciPropertiesEditorPage.setBACIProperty(newProperty);
			}
		}

		if( page == componentTypeChooserPage ) {
			componentChooserPage.setComponentProvier(new ComponentTypeComponentProvider(componentTypeChooserPage.getComponentType(), _config));
	    	componentChooserPage.setUseFilter(false);
			next = componentChooserPage;
		}

		return next;
	}

	@Override
	public boolean performFinish()
	{

		final String[] objectProperties = baciPropertiesEditorPage.getObjectProperties();
		final Object[] values = baciPropertiesEditorPage.getPropertiesValues();
		final Component[] components = componentChooserPage.getComponents();

		// Check that a name has been given
		String newName = null;
		for (int i=0; i!= objectProperties.length; i++) {
			String objectProperty = objectProperties[i];
			if( objectProperty.equals(BACIPropertyEditingComposite.PROPERTY_NAME) ) {
				newName = (String)values[i];
				break;
			}
		}

		if( newName == null || newName.trim().length() == 0 ) {
			RcpUtils.infoMessage(getShell(), "No BACI Property name give",
					"The new BACI property to be created has no name specified, " +
					"or is empty. A valid name should be given instead.");
			return false;
		}

		// Check that components don't have already a property with the new name
		for (Component component : components) {
			try {
				if( BACIPropertyConversationUtils.getInstance().componentHasProperty(component, newName) ) {
					RcpUtils.infoMessage(getShell(), "BACI Property already exists",
							"A BACI property named '" + newName + "' already exists for component " +
							LabelHelper.getFullPath(component, false));
					return false;
				}
			} catch (Exception e) {
				e.printStackTrace();
				return false;
			}
		}

		Job job = new Job("Creating BACI properties") {
			protected IStatus run(IProgressMonitor monitor) {
				ProgressMonitor eclipseMonitor = new EclipseProgressMonitor(monitor);
				try {
					BACIPropertyConversationUtils.getInstance().bulkCreateBACIProperties(components, objectProperties, values, eclipseMonitor);
				} catch(Exception e) {
					return new Status(IStatus.ERROR, TmcdbExplorer.PLUGIN_ID, e.getMessage(), e);
				}

				return Status.OK_STATUS;
			}
		};

		job.setUser(true);
		job.schedule();
        return true;
	}

}
