/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.wizards;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Configuration;
import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.utils.ProgressMonitor;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.TmcdbGui;
import alma.obops.tmcdbgui.rcp.TmcdbExplorer;
import alma.obops.tmcdbgui.utils.EclipseProgressMonitor;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.BACIPropertyConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.widgets.BACIPropertyEditingComposite;
import alma.obops.tmcdbgui.wizards.BACIPropertiesSearchCriteriaChooserPage.SearchCriteria;
import alma.obops.tmcdbgui.wizards.ComponentChooserPage.SelectionCriteriaForComponents;

public class BulkBACIPropertiesChangesWizard extends Wizard
{

	private Configuration _config;
	private ComponentType _compType;

	private ComponentTypeChooserPage componentTypeChooserPage;
	private ComponentChooserPage componentChooserPage;
	private BACIPropertiesChooserPage baciPropertiesChooserPage;
	private BACIPropertiesEditorPage baciPropertiesEditorPage;
	private BACIPropertiesEditorPage baciPropertyExamplePage;
	private BACIPropertiesSearchCriteriaChooserPage criteriaChooserPage;

	public BulkBACIPropertiesChangesWizard(Configuration config, ComponentType compType) {
		_config   = config;
		_compType = compType;
		setNeedsProgressMonitor(true);
	}

	 /** @see org.eclipse.jface.wizard.Wizard#addPages() */
    public void addPages()
    {

    	baciPropertiesChooserPage = new BACIPropertiesChooserPage();
    	baciPropertiesEditorPage = new BACIPropertiesEditorPage("BACI Properties Chooser", "Select which fields you want to change with the checkboxes. Then, enter the desired value");

    	// If component type already specified, we don't need to
    	// ask which criteria to use, it's already clear
    	if( _compType == null ) {
    		criteriaChooserPage = new BACIPropertiesSearchCriteriaChooserPage();
    		componentTypeChooserPage = new ComponentTypeChooserPage(_compType);
        	componentChooserPage = new ComponentChooserPage();
        	baciPropertyExamplePage = new BACIPropertiesEditorPage("BACI Property specification", "Specify which values are to be taken into acount to search for matching BACI properties");

        	addPage(criteriaChooserPage);
        	addPage(baciPropertyExamplePage);
        	addPage(componentTypeChooserPage);
            addPage(componentChooserPage);
    	} else {
    		ComponentProvider cProvider = new ComponentTypeComponentProvider(_compType, _config);
    		componentChooserPage = new ComponentChooserPage();
    		componentChooserPage.setComponentProvier(cProvider);
			componentChooserPage.setSelectionCriteria(SelectionCriteriaForComponents.COMMON_PROPERTIES);
    		addPage(componentChooserPage);
    	}

        addPage(baciPropertiesChooserPage);
        addPage(baciPropertiesEditorPage);
    }

	@Override
	public IWizardPage getNextPage(IWizardPage page) {

		IWizardPage next = super.getNextPage(page);

		// If we're coming from the criteria chooser, the next page depends on
		// our criteria selection
		if( page == criteriaChooserPage ) {
			SearchCriteria criteria = criteriaChooserPage.getSearchCriteria();
			if( criteria.equals(SearchCriteria.BY_COMPONENT_TYPE) ) {
				next = componentTypeChooserPage;

				baciPropertyExamplePage.setPageComplete(true);
			}
			else if( criteria.equals(SearchCriteria.BY_COMPONENT_NAME) ) {
				componentChooserPage.setComponentProvier(new ConfigurationComponentProvider(_config));
				componentChooserPage.setUseFilter(true);
				componentChooserPage.setSelectionCriteria(SelectionCriteriaForComponents.ALL_PROPERTIES);
				next = componentChooserPage;

				componentTypeChooserPage.setPageComplete(true);
				baciPropertyExamplePage.setPageComplete(true);
			}
			else if( criteria.equals(SearchCriteria.BY_BACIPROPERTY_CONTENTS) ) {
				next = baciPropertyExamplePage;

				componentTypeChooserPage.setPageComplete(true);
				componentChooserPage.setPageComplete(true);
			}
		}

		// If we come from the ComponentType chooser, we setup the
		// component chooser page correctly
		if( page == componentTypeChooserPage ) {
			componentChooserPage.setUseFilter(false);
			componentChooserPage.setSelectionCriteria(SelectionCriteriaForComponents.COMMON_PROPERTIES);
			componentChooserPage.setComponentProvier(new ComponentTypeComponentProvider(componentTypeChooserPage.getComponentType(), _config));
			next = componentChooserPage;
		}

		// If we come from the Component chooser, we setup the BACI property chooser
		// page according to the selection choice
		if( page == componentChooserPage ) {

			BACIPropertiesProvider provider = null;
			SelectionCriteriaForComponents selectionCriteria = componentChooserPage.getSelectionCriteria();

			if( selectionCriteria.equals(SelectionCriteriaForComponents.ALL_PROPERTIES) ) {
				provider = new AllBACIPropertiesProvider(componentChooserPage.getComponents());
				baciPropertiesChooserPage.setShowComponentName(true);
			}
			else if( selectionCriteria.equals(SelectionCriteriaForComponents.COMMON_PROPERTIES) ) {
				provider = new CommonBACIPropertiesProvider(componentChooserPage.getComponents());
				baciPropertiesChooserPage.setShowComponentName(false);
			}

			baciPropertiesChooserPage.setBACIPropertiesProdiver(provider);
			next = baciPropertiesChooserPage;
		}

		// If we're coming from the property chooser page,
		// we set the property to edit in the BACI property editor page
		if( page == baciPropertiesChooserPage )
			baciPropertiesEditorPage.setBACIProperty(baciPropertiesChooserPage.getBACIProperties()[0]);

		if( page == baciPropertyExamplePage ) {
			BACIPropertiesProvider provider = new ByExampleBACIPropertyProvider(baciPropertyExamplePage.getObjectProperties(), baciPropertyExamplePage.getPropertiesValues(), _config);
			baciPropertiesChooserPage.setBACIPropertiesProdiver(provider);
			baciPropertiesChooserPage.setShowComponentName(true);
			next = baciPropertiesChooserPage;
		}

		return next;
	}

	@Override
	public boolean performFinish()
	{

		final String[] objectProperties = baciPropertiesEditorPage.getObjectProperties();
		final Object[] values = baciPropertiesEditorPage.getPropertiesValues();
		final Component[] components = componentChooserPage.getComponents();

		// If a new name has been given for the BACI property,
		// check that components don't have already a property with the new name
		String newName = null;
		for (int i=0; i!= objectProperties.length; i++) {
			String objectProperty = objectProperties[i];

			if( objectProperty.equals(BACIPropertyEditingComposite.PROPERTY_NAME) ) {
				newName = (String)values[i];

				for (Component component : components) {
					try {
						if( BACIPropertyConversationUtils.getInstance().componentHasProperty(component, newName) ) {
							RcpUtils.infoMessage(getShell(), "BACI Property already exists",
									"A BACI property named '" + newName + "' already exists for component " +
									LabelHelper.getFullPath(component, false));
							return false;
						}
					} catch (Exception e) {
						e.printStackTrace();
						return false;
					}
				}
			}
		}

		Job job = new Job("Updating BACI properties") {
			protected IStatus run(IProgressMonitor monitor) {

				Component[] comps = componentChooserPage.getComponents();
				BACIProperty[] selectedProperties = baciPropertiesChooserPage.getBACIProperties();
				List<BACIProperty> propsToUpdate = new ArrayList<BACIProperty>();

				SearchCriteria searchCriteria = criteriaChooserPage.getSearchCriteria();
				if( searchCriteria.equals(SearchCriteria.BY_BACIPROPERTY_CONTENTS) )
					propsToUpdate = Arrays.asList(selectedProperties);
				else {
					// Depending on the BACI Properties selection criteria, we want to
					// update all properties, or only the common ones
					// In both cases, we need to loop through the components' properties,
					// since the properties that we get from the wizard page have been not
					// retrieved from the DB, but instead are only created in memory,
					// and thus we cannot reattach them to hibernate
					SelectionCriteriaForComponents criteria = componentChooserPage.getSelectionCriteria();

					if( criteria.equals(SelectionCriteriaForComponents.COMMON_PROPERTIES) ) {
						for(Component comp: comps) {
							for(BACIProperty prop: selectedProperties) {
								for(BACIProperty compProp: comp.getBACIProperties()) {
									if( compProp.getPropertyName().equals(prop.getPropertyName()) ) {
										propsToUpdate.add(compProp);
										break;
									}
								}
							}
						}
					}
					else if( criteria.equals(SelectionCriteriaForComponents.ALL_PROPERTIES) ) {
						for(BACIProperty prop: selectedProperties) {
							boolean foundProperty = false;
							for(Component comp: comps) {
								for(BACIProperty compProp: comp.getBACIProperties()) {
									if( compProp.getBACIPropertyId().equals(prop.getBACIPropertyId()) ) {
										propsToUpdate.add(compProp);
										foundProperty = true;
										break;
									}
								}
								if( foundProperty )
									break;
							}
							if( !foundProperty )
								return new Status(IStatus.ERROR, TmcdbExplorer.PLUGIN_ID, "Property '" +  prop.getPropertyName() +  "' (id=" + prop.getBACIPropertyId() + ") wasn't found in any of the selected components");
						}
					}
				}

				// Get the object properties and values to set in the to-be-updated objects
				String[] objProps = baciPropertiesEditorPage.getObjectProperties();
				Object[] objVals = baciPropertiesEditorPage.getPropertiesValues();

				ProgressMonitor eclipseMonitor = new EclipseProgressMonitor(monitor);

				try {
					BACIPropertyConversationUtils.getInstance().bulkUpdateBACIProperties(propsToUpdate.toArray(new BACIProperty[propsToUpdate.size()]),
							objProps, objVals, eclipseMonitor);
					System.out.println("Updated " + propsToUpdate.size() + " properties");
				} catch(Exception e) {
					e.printStackTrace();
					return new Status(IStatus.ERROR, TmcdbExplorer.PLUGIN_ID, e.getMessage());
				}

				return Status.OK_STATUS;
			}
		};

		job.setUser(true);
		job.schedule();
        return true;
	}

	public static void main(String[] args) throws Exception {
		Display display = new Display();
		Shell shell = new Shell(display);

		TmcdbContextFactory.INSTANCE.init("config/tmcdbExplorerAppContext.xml", TmcdbGui.getLogger());
		Configuration config = HwConfigurationConversationUtils.getInstance().findConfigurationsByName("ALMA-8_0_0_5.AOS-imported-2011-02-10T20:49").get(0).getSwConfiguration();

		ComponentType ct = new ComponentType();
		ct.setIDL("IDL:alma/Control/Mount:1.0");
//		BulkBACIPropertiesChangesWizard wizard = new BulkBACIPropertiesChangesWizard(new Configuration(), ct);
		BulkBACIPropertiesChangesWizard wizard = new BulkBACIPropertiesChangesWizard(config, null);

		WizardDialog dialog = new WizardDialog(shell, wizard);
    	dialog.create();
    	dialog.open();
	}
}