/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * NewComponentPage.java
 */
package alma.obops.tmcdbgui.wizards;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Widget;

import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.EclipseProgressMonitor;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.LabelHelper;

/**
 * Wizard page for creation of a new {@link Component} object.
 *
 * This page is used the {@link NewComponentWizard}
 *
 * @author rtobar, Mar 2, 2010
 */
public class BACIPropertiesChooserPage extends WizardPage implements Listener, ISelectionChangedListener {

	public class BACIPropertiesChooserLabelProvider extends ColumnLabelProvider {

		public Image getImage( Object element )  {
			return ImageHelper.getImage((BACIProperty)element);
		}

		public String getText( Object element ) {
			if( _showComponentName ) {
				BACIProperty p = (BACIProperty)element;
				if( p.getComponent() != null )
					return LabelHelper.getFullPath(p.getComponent(), false) + ":" +
					       LabelHelper.getFullPath(p, false);
			}

			return LabelHelper.getFullPath((BACIProperty)element, false);
		}
	}

	public class BACIPropertiesChooserContentProvider implements
			ITreeContentProvider {

		private List<BACIProperty> properties;

		@Override
		public Object[] getChildren(Object parentElement) {
			return null;
		}

		@Override
		public Object getParent(Object element) {
			return null;
		}

		@Override
		public boolean hasChildren(Object element) {
			return false;
		}

		@Override
		public Object[] getElements(Object inputElement) {
			return properties.toArray();
		}

		@Override
		public void dispose() {

		}

		@Override
		public void inputChanged(Viewer theViewer, Object oldInput, final Object newInput) {

			properties = new ArrayList<BACIProperty>();
			if( newInput != null ) {
				try {
					getContainer().run(false, false, new IRunnableWithProgress() {
						public void run(IProgressMonitor monitor) throws InvocationTargetException,
								InterruptedException {
							BACIPropertiesProvider provider = (BACIPropertiesProvider)newInput;
							try {
								properties = provider.getBACIProperties(new EclipseProgressMonitor(monitor));
							} catch(Exception e) {
								RcpUtils.errorMessage(e, getShell(), "Error while getting components", e.getMessage());
							}
						}
					});
				} catch (InvocationTargetException e) {
					e.printStackTrace();
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		}

	}

	private Button selectAllButton;
	private Button selectNoneButton;
	private CheckboxTableViewer viewer;
	private BACIProperty[] _properties;

	private boolean _showComponentName = false;
	private BACIPropertiesProvider _propertiesProvider;

	protected BACIPropertiesChooserPage() {
		super("BACI Properties Chooser");
		setTitle("BACI Properties Chooser");
		setDescription("Select which BACI Properties you want to change");
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createControl(Composite parent) {

		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout(1, false));

		GridData gd = new GridData(SWT.FILL, SWT.FILL, true, true);
		viewer = CheckboxTableViewer.newCheckList(composite, SWT.BORDER);
		viewer.getControl().setLayoutData(gd);
		viewer.setContentProvider( new BACIPropertiesChooserContentProvider() );
		viewer.setLabelProvider(new BACIPropertiesChooserLabelProvider() );
		viewer.setSorter(new ViewerSorter());
		viewer.addSelectionChangedListener(this);

		gd = new GridData(SWT.FILL, SWT.TOP, true, false);
		Composite downBar = new Composite(composite, SWT.NONE);
		downBar.setLayout(new GridLayout(2, false));
		downBar.setLayoutData( gd );

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		selectAllButton = new Button(downBar, SWT.PUSH);
		selectAllButton.setText("Select All");
		selectAllButton.addListener( SWT.Selection, this);
		selectAllButton.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalIndent = 10;
		selectNoneButton = new Button(downBar, SWT.PUSH);
		selectNoneButton.setText("Select None");
		selectNoneButton.addListener( SWT.Selection, this);
		selectNoneButton.setLayoutData(gd);

		setControl( composite );
		setPageComplete(false);
	}

	public void setVisible(boolean visible){
		if( visible )
			viewer.setInput( _propertiesProvider );
		super.setVisible(visible);
	}

	public void setBACIPropertiesProdiver(BACIPropertiesProvider provider) {
		_propertiesProvider = provider;
	}

	public void setShowComponentName(boolean show) {
		_showComponentName = show;
	}

	@Override
	public void handleEvent(Event event) {

		SelectionChangedEvent selectionEvent = null;
		Widget w = event.widget;

		if( w == selectAllButton ) {
			selectionEvent = new SelectionChangedEvent(viewer, viewer.getSelection());
			viewer.setAllChecked(true);
			selectionChanged(selectionEvent);
		}
		else if( w == selectNoneButton ) {
			selectionEvent = new SelectionChangedEvent(viewer, viewer.getSelection());
			viewer.setAllChecked(false);
			selectionChanged(selectionEvent);
		}

	}

	@Override
	public void selectionChanged(SelectionChangedEvent event) {
		if( event.getSelectionProvider() == viewer ) {
			Object[] objs = viewer.getCheckedElements();
			_properties = new BACIProperty[objs.length];
			System.arraycopy(objs, 0, _properties, 0, objs.length);
			setPageComplete( objs.length != 0 );
		}
	}

	public BACIProperty[] getBACIProperties() {
		return _properties;
	}
}