/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.wizards;

import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import alma.obops.tmcdbgui.views.providers.helpers.config.ConfigHelperFactory;
import alma.tmcdb.domain.AssemblyType;

public class AssemblyTypeChooserWizardPage extends WizardPage 
{
	private AssemblyAttributesWizardPage nextPage;
	private AssemblyType initialAssemblyType;
	
	public AssemblyTypeChooserWizardPage(String pageName, AssemblyAttributesWizardPage attributesPage, AssemblyType assemblyType) 
	{
		super(pageName);
		setPageComplete(false);
		setTitle("Please select the assembly type for the new assembly");
		this.nextPage = attributesPage;
		this.initialAssemblyType = assemblyType;
	}

	@Override
	public void createControl(Composite parent) 
	{
		Composite comp = new AssemblyTypeChooserComposite(parent, SWT.NONE);
		this.setControl(comp);
	}
	
	private class AssemblyTypeChooserComposite extends Composite
	{
		private Map<String, AssemblyType> typesMap;
		
		public AssemblyTypeChooserComposite(Composite parent, int style) 
		{
			super(parent, style);
			typesMap = new HashMap<String, AssemblyType>();
			String items[] = getAssemblyTypeStrings();
			createControl(items);
		}

		private void createControl(String items[]) 
		{
			GridLayout gridLayout = new GridLayout();
			gridLayout.numColumns = 2;
			this.setLayout(gridLayout);
			
			GridData gd = new GridData();
			gd.grabExcessHorizontalSpace = false;
			
			Label typeLabel = new Label(this, SWT.NONE);
			typeLabel.setText("Assembly type: ");
			typeLabel.setLayoutData(gd);
			
			final Combo typeCombo = new Combo(this, SWT.NONE);
			
			typeCombo.setItems(items);
			typeCombo.setLayoutData(new GridData());
			typeCombo.setText("Select assembly type");
			
			if(initialAssemblyType != null)
			{
				typeCombo.select(typeCombo.indexOf(initialAssemblyType.getName()));
				String item = typeCombo.getItem(typeCombo.getSelectionIndex());
				AssemblyType type = typesMap.get(item);
				nextPage.setAssemblyType(type);
				setPageComplete(true);
			}
			
			typeCombo.addSelectionListener(new SelectionListener() 
			{
				@Override
				public void widgetDefaultSelected(SelectionEvent selEvt) {
					widgetSelected(selEvt);
				}

				@Override
				public void widgetSelected(SelectionEvent selEvt) {
					setPageComplete(true);
					String item = typeCombo.getItem(typeCombo.getSelectionIndex());
					AssemblyType type = typesMap.get(item);
					nextPage.setAssemblyType(type);
				}
			});
		}
		
		private String[] getAssemblyTypeStrings() 
		{
			String [] retVal = null;
			try {
				List<AssemblyType> types = 
					((ConfigHelperFactory)ConfigHelperFactory.getInstance()).getAssemblyTypes();

				Collections.sort(types, new Comparator<AssemblyType>() {
					@Override
					public int compare(AssemblyType o1, AssemblyType o2) {
						return o1.getName().compareTo(o2.getName());
					}
				});

				retVal = new String[types.size()];
				int count = 0;
				for(AssemblyType type: types)
				{
					retVal[count++] = type.getName();
					typesMap.put(type.getName(), type);
				}
			} catch (Exception e) {
				throw new RuntimeException("Could not get assembly types from db");
			}
			return retVal;
		}
	}
}
