/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.wizards;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import alma.obops.tmcdbgui.widgets.StatusPublishingComposite;
import alma.obops.tmcdbgui.widgets.support.StatusListener;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.HwConfiguration;

public class AssemblyAttributesWizardPage extends WizardPage implements StatusListener
{
	private static final String ASSEMBLY_ALREADY_EXISTS = "An assembly with this type & serial number already exists";
	private HwConfiguration configuration;
	private AssemblyAttributesComposite comp;
	private AssemblyType asstype;
	private String serialNumber;
	
	protected AssemblyAttributesWizardPage(String pageName, HwConfiguration config)
	{
		super(pageName);
		this.setPageComplete(false);
		this.setTitle("Please enter the serial number for the new assembly");
		this.configuration = config;
	}
	

	public AssemblyType getAssemblyType() {
		return asstype;
	}
	
	public void setAssemblyType(AssemblyType type)
	{
		this.asstype = type;
	}
	
	@Override
	public void createControl(Composite parent) 
	{
		comp = new AssemblyAttributesComposite(parent, SWT.NONE, this);
		this.setControl(comp);
	}

	@Override
	public void notifyOfCompletion(boolean complete) 
	{
		this.setPageComplete(complete);
	}

	@Override
	public void updateErrorStatus(String newStatusMessage) 
	{
		this.setErrorMessage(newStatusMessage);
	}

	private class AssemblyAttributesComposite extends StatusPublishingComposite
	{
		private Text serialNumberText;
		public AssemblyAttributesComposite(Composite parent, int style, StatusListener listener) 
		{
			super(parent, style);
			createControl();
			addKeyListeners();
			this.addStatusListener(listener);
		}
		
		private void createControl() 
		{
			GridLayout layout = new GridLayout();
			layout.numColumns = 2;  // label, entry
			setLayout( layout );

			createSerialNumberControl();
		}

		private void createSerialNumberControl() 
		{
			Label nameLabel = new Label(this, SWT.NONE);
			nameLabel.setText("Serial number");
			
			serialNumberText = new Text(this, SWT.BORDER);
			GridData gridData = new GridData();
			gridData.grabExcessHorizontalSpace = true;
			gridData.horizontalAlignment = GridData.FILL;
			serialNumberText.setLayoutData(gridData);
		}

		private void addKeyListeners()
		{
			KeyListener kListener = new KeyListener() 
			{
				@Override
				public void keyPressed(KeyEvent evt) {
				}

				@Override
				public void keyReleased(KeyEvent evt) {
					isComplete();
				}	
			};
			serialNumberText.addKeyListener(kListener); 
		}
		
		/**
		 * Getter for the new assembly's serial number;
		 * @return the new assembly's serial number.
		 */
		public String getSerialNumber() 
		{
			String retVal = null;
			
			retVal = serialNumber;
			
			return retVal;
		}
		
		private boolean assemblyExistsInConfig()
		{
			boolean retVal = false;

			try	{
				retVal = foundCorrespondingAssembly();
			}
			catch(Exception ex) {
				throw new RuntimeException("Unable to get the assemblies for the configuration", ex);
			}
			
			if(retVal == true) {
				this.setStatus(ASSEMBLY_ALREADY_EXISTS);
			} else {
				this.setStatus(null);
			}
			return retVal;
		}
		
		private boolean foundCorrespondingAssembly() 
		{
			boolean retVal = false;
			String serialNumber1 = comp.getSerialNumber();
			
			for(Assembly ass: configuration.getAssemblies()) 
			{
				if(ass.getAssemblyType().equals(asstype) && ass.getSerialNumber().equals(serialNumber1)) 
				{
					retVal = true;
					break;
				}
			}
			return retVal;
		}
		
		/** @return <code>true</code> when all required fields are populated */
		public boolean isComplete() 
		{	
			boolean complete = 
				(serialNumberText.getText().length() > 0) &&
				!assemblyExistsInConfig();

			notifyListenersOfCompletion(complete);
			if(complete) {
				serialNumber = serialNumberText.getText();
			}
			return complete;
		}
	}

	public String getSerialNumber() {
		return comp.getSerialNumber();
	}


	public void setConfiguration(HwConfiguration config) 
	{
		this.configuration = config;
	}
}
