/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.wizards;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;

import alma.obops.tmcdbgui.utils.TmcdbConstants;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.widgets.support.HwConfigurationListener;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.HwConfiguration;

public class AntennaNameWizardPage extends WizardPage implements HwConfigurationListener
{
	private static final String SELECT_NAME = "Choose name for copied antenna in destination configuration";
	private Map<String, BaseElement> baseElementMap = new HashMap<String, BaseElement>();
	private String antennaName;
	HwConfiguration configuration;
	
	protected AntennaNameWizardPage(String pageName, String antennaName) 
	{
		super(pageName);
		this.antennaName = antennaName;
		setDescription( SELECT_NAME );
		setTitle(pageName);
	}

	@Override
	public void createControl(Composite parent) 
	{
		Composite composite = new AntennaNameAndPrefixComposite(parent, SWT.None, this);
		setPageComplete( false );
		setControl( composite );
	}
	
	private class AntennaNameAndPrefixComposite extends Composite
	{
		private Combo namePrefixCombo, nameNumberCombo;
		private WizardPage owningPage;
		
		/**
		 * Constructor.
		 * @param parent the parent composite.
		 * @param style the style for this composite.
		 */
		public AntennaNameAndPrefixComposite(Composite parent, int style, WizardPage owningPage) 
		{
			super(parent, style);
			this.owningPage = owningPage;
			createControl();
		}

		private void createControl()
		{
			GridLayout layout = new GridLayout();
			layout.numColumns = 1;  // label, entry
			setLayout( layout );
			
			Group namePrefixAndNumber = new Group(this, SWT.NONE);
			namePrefixAndNumber.setText("Name");
			GridLayout gridLayout = new GridLayout();
			gridLayout.numColumns = 4;
			namePrefixAndNumber.setLayout(gridLayout);
			GridData gridData = new GridData();
			gridData.horizontalSpan = 4;
			namePrefixAndNumber.setLayoutData(gridData);

			GridData gd;
			// Antenna name prefix
			Label lName = new Label( namePrefixAndNumber, SWT.NULL );
			lName.setText( "Prefix" );
			namePrefixCombo = new Combo( namePrefixAndNumber, SWT.READ_ONLY );
			gd = new GridData();
			
			namePrefixCombo.setLayoutData( gd );
			namePrefixCombo.setEnabled(false);
			
			// Antenna number
			Label lNumber = new Label( namePrefixAndNumber, SWT.NULL );
			lNumber.setText( "Number" );
			nameNumberCombo = new Combo( namePrefixAndNumber, SWT.READ_ONLY );
			gd = new GridData();
			
			nameNumberCombo.setLayoutData( gd );
			nameNumberCombo.addSelectionListener(new NumberSelectionListener(this, owningPage));
			
			populateCombos();
		}
		
		
		private void populateCombos()
		{
			namePrefixCombo.setItems(TmcdbConstants.NAME_PREFIX_ARRAY );
			
			String prefixOfAntennaBeingCopied = parsePrefix(antennaName);
			namePrefixCombo.select(namePrefixCombo.indexOf(prefixOfAntennaBeingCopied));
			
			if(prefixOfAntennaBeingCopied.equals(TmcdbConstants.DA)) {
				nameNumberCombo.setItems(TmcdbConstants.DA_NUMBERS);
			}
			else if(prefixOfAntennaBeingCopied.equals(TmcdbConstants.DV) || prefixOfAntennaBeingCopied.equals(TmcdbConstants.LA)) 
			{
				nameNumberCombo.setItems(TmcdbConstants.DV_NUMBERS);
			} else if(prefixOfAntennaBeingCopied.equals(TmcdbConstants.PM)) {
				nameNumberCombo.setItems(TmcdbConstants.PM_NUMBERS);
			} else if(prefixOfAntennaBeingCopied.equals(TmcdbConstants.CM)) {
				nameNumberCombo.setItems(TmcdbConstants.CM_NUMBERS);
			}
		}
	}

	private String parsePrefix(String name) {
		String retVal = null;
		retVal = name.substring(0, 2);
		return retVal;
	}
	
	private class NumberSelectionListener implements SelectionListener
	{
		private AntennaNameAndPrefixComposite nameAndPrefixComposite;
		private WizardPage owningPage;
		
		public NumberSelectionListener(AntennaNameAndPrefixComposite composite, WizardPage owningPage)
		{
			this.nameAndPrefixComposite = composite;
			this.owningPage = owningPage;
		}
		
		@Override
		public void widgetDefaultSelected(SelectionEvent e) {
			widgetSelected(e);
		}

		@Override
		public void widgetSelected(SelectionEvent e) {
		   String selectedAntennaName = nameAndPrefixComposite.namePrefixCombo.getText()  
		      + nameAndPrefixComposite.nameNumberCombo.getText();
		   
		   if(null != baseElementMap.get(selectedAntennaName)) 
		   {
			   owningPage.setErrorMessage("Antenna '" + nameAndPrefixComposite.namePrefixCombo.
					   getItem(nameAndPrefixComposite.namePrefixCombo.getSelectionIndex()) + 
					   nameAndPrefixComposite.nameNumberCombo.
					   getItem(nameAndPrefixComposite.nameNumberCombo.getSelectionIndex()) + 
					   "' already exists in destination configuration");
			   owningPage.setPageComplete(false);
		   } 
		   else {
			   owningPage.setErrorMessage(null);
			   owningPage.setPageComplete(true);
			   antennaName = selectedAntennaName;
		   }
		}
	}
	
	public void setConfiguration(HwConfiguration config)
	{
		this.configuration = config;
		try {
			HwConfigurationConversationUtils.getInstance().hydrateBaseElements(config);
		} 
		catch (Exception e) {
			throw new RuntimeException("Could not hydrate base elements", e);
		}
		Set<BaseElement> baseElements = config.getBaseElements();
		for(BaseElement baseElement: baseElements) {
			this.baseElementMap.put(baseElement.getName(), baseElement);
		}
	}
	
	public HwConfiguration getConfiguration() {
		return this.configuration;
	}

	/**
	 * Getter for the name (prefix + number; e.g. DV01, etc.) for the antenna (in the destination configuration).
	 * @return the name for the copied antenna.
	 */
	public String getAntennaName() {
		return this.antennaName;
	}

	@Override
	public void setHwConfiguration(HwConfiguration config) {
		setConfiguration(config);
	}
}
