/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.widgets;

import java.util.Calendar;
import java.util.Date;
import java.util.Set;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DateTime;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.obops.tmcdbgui.widgets.support.StatusListener;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.WeatherStationController;

/**
 * Used to define/edit the basic attributes of a weather station.
 * @author sharring
 */
public class WeatherStationAttributesComposite extends StatusPublishingComposite 
{
	public static final String WEATHERSTATION_ALREADY_EXISTS = "Weather station already exists: name must be unique";
	private WeatherStationController weatherstation;
	private Text nameText;
	private DateTime commissionDate;
	private Set<BaseElement> baseElements;
    private HwConfiguration configuration;
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 * @param weatherstation the weatherstation that is being "dealt" with.
	 */
	public WeatherStationAttributesComposite(Composite parent, int style, WeatherStationController weatherstation, StatusListener statusListener, DirtyListener dirtyListener)
	{
		super(parent, style);
		this.addStatusListener(statusListener);
		this.addDirtyListener(dirtyListener);
		createControl(weatherstation);
	}
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public WeatherStationAttributesComposite(Composite parent, int style, DirtyListener dirtyListener) 
	{
		this(parent, style, null, null, dirtyListener);
	}

	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public WeatherStationAttributesComposite(Composite parent, int style, StatusListener statusListener, HwConfiguration config) 
	{
		this(parent, style, null, statusListener, null);
		this.configuration = config;
	}

	@Override
	public boolean setFocus() {
		return nameText.setFocus();
	}
	
	private void createControl(WeatherStationController ws) 
	{
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;  // label, entry
		setLayout( layout );
		GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);
		setLayoutData(gridData);

		createNameControl();
		createCommissionDateControl();
        setWeatherStation(ws);
        addKeyListeners();
	}	
	

	private void createNameControl() 
	{
        Label nameLabel = new Label(this, SWT.NONE);
        nameLabel.setText("Name");
        nameText = new Text(this, SWT.BORDER);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        nameText.setLayoutData(gridData);
        nameText.addKeyListener(new SetDirtyKeyListener());
	}

	private void createCommissionDateControl() 
	{
		Composite commissionDateComposite = new Composite(this, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 4;
		commissionDateComposite.setLayout(gridLayout);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 4;
		commissionDateComposite.setLayoutData(gridData);
		
		Label commissionDateLabel = new Label(commissionDateComposite, SWT.NONE);
        commissionDateLabel.setText("Commission date");
        commissionDate = new DateTime(commissionDateComposite, SWT.DATE | SWT.MEDIUM);
        commissionDate.addSelectionListener(new SelectionListener() {

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				setDirty(true);
			}

			@Override
			public void widgetSelected(SelectionEvent e) {
				setDirty(true);
			}
        	
        });
	}
	
	private void addKeyListeners() 
	{
		// At each keystroke computes whether this page is complete
		KeyListener completionKL = new KeyListener() 
		{
			public void keyPressed( KeyEvent e ) {
				// ignore
			}

			public void keyReleased( KeyEvent e ) {
				isComplete();
			}
		};
		nameText.addKeyListener(completionKL);
	}
	
	/**
	 * Getter for the new antenna's commission date.
	 * @return the new antenna's commission date.
	 */
	public Date getCommissionDate() 
	{
		Date retVal = null;
		
		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.YEAR, this.commissionDate.getYear());
		cal.set(Calendar.MONTH, this.commissionDate.getMonth());
		cal.set(Calendar.DAY_OF_MONTH, this.commissionDate.getDay());
		cal.set(Calendar.HOUR_OF_DAY, this.commissionDate.getHours());
		cal.set(Calendar.MINUTE, this.commissionDate.getMinutes());
		cal.set(Calendar.SECOND, this.commissionDate.getSeconds());
		retVal = cal.getTime();
		
		return retVal;
	}
	
	/**
	 * Getter for the new antenna's name.
	 * @return the new antenna's name.
	 */
	public String getWeatherStationName() 
	{
		String retVal = null;
		retVal = nameText.getText();
		return retVal;
	}
	
	private boolean weatherstationExistsInConfig()
	{
		boolean retVal = false;

		if(null == baseElements) 
		{
			this.baseElements = configuration.getBaseElements();
		} 	

		try	{
			retVal = foundCorrespondingBaseElement();
		}
		catch(Exception ex) {
			throw new RuntimeException("Unable to get the base elements for the configuration", ex);
		}
		
		if(retVal == true) {
			this.setStatus(WEATHERSTATION_ALREADY_EXISTS);
		} else {
			this.setStatus(null);
		}
		return retVal;
	}
	
	private boolean foundCorrespondingBaseElement() {
		boolean retVal = false;
		for(BaseElement be: baseElements) 
		{
			if(be.getType().equals(BaseElementType.WeatherStationController) && be.getName().equals(getWeatherStationName())) 
			{
				retVal = true;
				break;
			}
		}
		return retVal;
	}
	
	
	/** @return <code>true</code> when all required fields are populated */
	public boolean isComplete() 
	{	
		boolean complete = 
			!weatherstationExistsInConfig() &&
			(this.getCommissionDate()!= null ) && 
			(nameText.getText().length() > 0);

		notifyListenersOfCompletion(complete);
		return complete;
	}

	private class SetDirtyKeyListener implements KeyListener 
	{
		@Override
		public void keyPressed(KeyEvent e) {
			setDirty(true);
		}

		@Override
		public void keyReleased(KeyEvent e) {
		}
	}

	public void setWeatherStation(WeatherStationController ws) 
	{
		if(null == ws) 
		{
			return;
		}
		this.weatherstation = ws;
		
		// commissionDate
		Calendar cal = Calendar.getInstance();
		cal.setTimeInMillis(weatherstation.getCommissionDate());
		this.commissionDate.setDate(cal.get(Calendar.YEAR), cal.get(Calendar.MONTH), cal.get(Calendar.DAY_OF_MONTH));

		// name
		this.nameText.setText(weatherstation.getName());
		
		this.configuration = weatherstation.getConfiguration();
	}
}
