/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.widgets;

import java.text.DecimalFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Set;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DateTime;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Text;

import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.obops.tmcdbgui.wizards.support.VerifyDecimalListener;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.Coordinate;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.Pad;

public class PadAttributesComposite extends StatusPublishingComposite
{
	public static final String SAMPLE_COORDINATE_STRING = "-5555555.5555555";
	public static final int NUM_CHARS_FOR_DELAY = 12;
	public static final String PAD_DELAY_UNITS =  "(s):";
	public static final String PAD_ALREADY_EXISTS = "Pad already exists: prefix + number must be unique";
	protected static final int COORDINATES_TEXT_WIDTH = 130;
	private static final String PREFIX = "Prefix";
	private static final String COMMISSION_DATE = "Commission date";
	private static final String NUMBER = "Number";
	private static final String NAME = "Name";
	private static final String POSITION = "Position (m)";
	public static final String PAD_DELAY = "Pad delay";
	private static final int INCREMENT = 1;
	private static final int NUMBER_OF_DECIMALS = 0;
	private static final int PAGE_INCREMENT = 10;
	private static final int TOTAL_POWER_MAX_NUMBER = 704;
	private static final int TOTAL_POWER_MIN_NUMBER = 701;
	private static final int NORTH_SOUTH_EXTENSION_MAX_NUMBER = 606;
	private static final int NORTH_SOUTH_EXTENSION_MIN_NUMBER = 601;
	private static final int ACA_MAX_NUMBER = 512;
	private static final int ACA_MIN_NUMBER = 501;
	private static final int PAMPA_LA_BOLA_ARM_MAX_NUMBER = 413;
	private static final int PAMPA_LA_BOLA_ARM_MIN_NUMBER = 401;
	private static final int SOUTHERN_ARM_MAX_NUMBER = 309;
	private static final int SOUTHERN_ARM_MIN_NUMBER = 301;
	private static final int WESTERN_ARM_MAX_NUMBER = 210;
	private static final int WESTERN_ARM_MIN_NUMBER = 201;
	private static final int INNER_ARRAY_MIN_NUMBER = 1;
	private static final int INNER_ARRAY_MAX_NUMBER = 138;
	private static final int TEST_ARRAY_MIN_NUMBER = 1;
	private static final int TEST_ARRAY_MAX_NUMBER = 20;
	
	private static final String TOTAL_POWER_PREFIX = "T";
	private static final String NORTH_SOUTH_ACA_EXTENSION_PREFIX = "N";
	private static final String ACA_PREFIX = "J";
	private static final String PAMPA_LA_BOLA_ARM_PREFIX = "P";
	private static final String SOUTHERN_ARM_PREFIX = "S";
	private static final String WESTERN_ARM_PREFIX = "W";
	private static final String INNER_ARRAY_PREFIX = "A";
	private static final String TEST_ARRAY_PREFIX = "TF";
	private static final String[] NAME_PREFIX_ARRAY = new String[] { INNER_ARRAY_PREFIX, WESTERN_ARM_PREFIX, SOUTHERN_ARM_PREFIX, 
		PAMPA_LA_BOLA_ARM_PREFIX, ACA_PREFIX, NORTH_SOUTH_ACA_EXTENSION_PREFIX, TOTAL_POWER_PREFIX, TEST_ARRAY_PREFIX };
	
	private Text positionX, positionY, positionZ;
	private DateTime commissionDate;
	private Combo namePrefixCombo;
	private Spinner nameNumberSpinner;
	private Set<BaseElement> baseElements;
	private Text cableDelay; 
	private HwConfiguration configuration;
	private String previousNamePrefix = "";
	private KeyListener completionKL;
	private PadAttributesModifyListener updateCombosBasedOnSelectionsML;
		
	public PadAttributesComposite(Composite parent, int style, DirtyListener listener)
	{
		super(parent, style);

		this.addDirtyListener(listener);
		
		createControl();
		
		addKeyListener();
		
		addModifyListeners();
	}
	
	public PadAttributesComposite(Composite parent, int style, HwConfiguration configuration)
	{
		this(parent, style, (DirtyListener)null);
		this.setConfiguration(configuration);
	}

	/**
	 * Getter for the new antenna's position.
	 * @return the position of the new antenna.
	 */
	public Coordinate getPosition() {
		Coordinate retVal = new Coordinate();
		if(positionX.getText() != null && positionX.getText().trim().length() > 0) {
		   retVal.setX(Double.valueOf(positionX.getText()));
		}
		if(positionY.getText() != null && positionY.getText().trim().length() > 0) {
			retVal.setY(Double.valueOf(positionY.getText()));			
		}
		if((positionZ.getText() != null && positionZ.getText().trim().length() > 0)) {
		   retVal.setZ(Double.valueOf(positionZ.getText()));
		}
		return retVal;
	}

	/**
	 * Getter for the new antenna's commission date.
	 * @return the new antenna's commission date.
	 */
	public Date getCommissionDate() 
	{
		Date retVal = null;

		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.YEAR, this.commissionDate.getYear());
		cal.set(Calendar.MONTH, this.commissionDate.getMonth());
		cal.set(Calendar.DAY_OF_MONTH, this.commissionDate.getDay());
		cal.set(Calendar.HOUR_OF_DAY, this.commissionDate.getHours());
		cal.set(Calendar.MINUTE, this.commissionDate.getMinutes());
		cal.set(Calendar.SECOND, this.commissionDate.getSeconds());
		retVal = cal.getTime();

		return retVal;
	}

	/**
	 * Getter for the new antenna's name.
	 * @return the new antenna's name.
	 */
	public String getPadName() 
	{
		String retVal = null;

		String prefix = (namePrefixCombo.getSelectionIndex() == -1) ? "" : namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex());
		String number = (namePrefixCombo.getSelectionIndex() == -1) ? "" : nameNumberSpinner.getText();

		// preserve 2 digit pad names (e.g. instead of pad name 'A1' we prefer 'A01')
		if(number.length() == 1)
		{
			String numberWithLeadingZeros = null;
			
			// special logic for TF pads -> they don't have three digits
			if(prefix.equals(TEST_ARRAY_PREFIX)) {
				numberWithLeadingZeros = "0" + number;
			} else {
				numberWithLeadingZeros = "00" + number;	
			}
			
			number = numberWithLeadingZeros;
		}
		else if(number.length() == 2 && !prefix.equals(TEST_ARRAY_PREFIX))
		{
			String numberWithLeadingZeros = "0" + number;
			number = numberWithLeadingZeros;
		}
		
		retVal = prefix + number;

		return retVal;
	}

	/**
	 * Getter for the cable delay for the new pad.
	 * @return the cable delay for the new pad.
	 */
	public Double getCableDelay()
	{
		Double retVal = null;

		if(cableDelay.getText() != null && cableDelay.getText().trim().length() > 0)
		retVal = Double.valueOf(cableDelay.getText());

		return retVal;
	}
	
	public void setPad(Pad pad)
	{
		if(pad == null) {
			return;
		}
		
		this.configuration = pad.getConfiguration();
		
		DecimalFormat formatter = new DecimalFormat(AntennaAttributesComposite.COORDINATE_FORMAT);
		
		// position
		if(null != pad.getPosition()) {
			String formattedX = formatter.format(pad.getPosition().getX());
			String formattedY = formatter.format(pad.getPosition().getY());
			String formattedZ = formatter.format(pad.getPosition().getZ());
			this.positionX.setText(formattedX);
			this.positionY.setText(formattedY);
			this.positionZ.setText(formattedZ);
		} else {
			this.positionX.setText("");
			this.positionY.setText("");
			this.positionZ.setText("");
		}
		
		// commissionDate
		Calendar cal = Calendar.getInstance();
		cal.setTimeInMillis(pad.getCommissionDate());
		this.commissionDate.setDate(cal.get(Calendar.YEAR), cal.get(Calendar.MONTH), cal.get(Calendar.DAY_OF_MONTH));
		
		// name
		int prefixEndIndex = 1;
		if(pad.getName().startsWith(TEST_ARRAY_PREFIX))
		{
			prefixEndIndex = 2;
		}
		String prefix = pad.getName().substring(0, prefixEndIndex).toUpperCase();	
		String suffix = pad.getName().substring(prefixEndIndex);
		updateForNamePrefixSelection();
		this.namePrefixCombo.select(this.namePrefixCombo.indexOf(prefix));
		this.nameNumberSpinner.setSelection(Integer.valueOf(suffix));
		
		// cable delay
		if(null != pad.getAvgDelay()) {
			formatter = new DecimalFormat(AntennaAttributesComposite.OFFSET_FORMAT);
			String formattedDelay = formatter.format(pad.getAvgDelay());
			this.cableDelay.setText(formattedDelay);	
		} else {
			this.cableDelay.setText("");
		}
	}

	/** @return <code>true</code> when this page is complete */
	public boolean isComplete() 
	{	
		boolean namePrefixComplete = (namePrefixCombo.getSelectionIndex() != -1);
		boolean padExistsComplete = !padExistsInConfig();
		boolean positionXComplete = (positionX.getText().length() > 0);
		boolean positionYComplete = (positionY.getText().length() > 0);
		boolean positionZComplete = (positionZ.getText().length() > 0);
		boolean cableDelayComplete = (cableDelay.getText().length() > 0);
		
		boolean complete = namePrefixComplete 
			&& padExistsComplete
			&& (positionXComplete &&
				 positionYComplete && 
				 positionZComplete) 
			&&  cableDelayComplete;

		notifyListenersOfCompletion(complete);
		return complete;
	}

	public void setConfiguration(HwConfiguration config) {
		this.configuration = config;
	}

	/**
	 * Private class to handle the dependencies (interrelatedness) between the widgets, so that invalid
	 * choices are not possible.
	 * 
	 * @author sharring
	 */
	private class PadAttributesModifyListener implements ModifyListener
	{
		@Override
		public void modifyText(ModifyEvent e) {
			isComplete();
			if(e.widget == namePrefixCombo && namePrefixCombo.getSelectionIndex() != -1) 
			{
				updateForNamePrefixSelection();
			}			
		} 
	}

	private void createControl()
	{
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;  // label, entry
		setLayout( layout );
		GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);
		setLayoutData(gridData);
		
		createPadPrefixAndNumberControl();
		createCableDelayControl();
		createCommissionDateControl();
		createPadPositionControl();
	}
	
	private void addKeyListener() 
	{
		// At each keystroke computes whether this page is complete
		completionKL =  new KeyListener() 
		{
			public void keyPressed( KeyEvent e ) {
				// ignore
			}

			public void keyReleased( KeyEvent e ) {
				isComplete();
			}
		};
		positionX.addKeyListener(completionKL);
		positionY.addKeyListener(completionKL);
		positionZ.addKeyListener(completionKL);
		cableDelay.addKeyListener(completionKL);
	}

	private void addModifyListeners() 
	{
		updateCombosBasedOnSelectionsML = new PadAttributesModifyListener();
		namePrefixCombo.addModifyListener(updateCombosBasedOnSelectionsML);
		nameNumberSpinner.addModifyListener(updateCombosBasedOnSelectionsML);
	}

	private void updateForNamePrefixSelection() 
	{
		isComplete();
		
		if(namePrefixCombo.getSelectionIndex() == -1) {
			return;
		}
		
		if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(INNER_ARRAY_PREFIX) 
				&& !previousNamePrefix.equals(INNER_ARRAY_PREFIX)) 
		{
			nameNumberSpinner.setValues(INNER_ARRAY_MIN_NUMBER, INNER_ARRAY_MIN_NUMBER, 
					INNER_ARRAY_MAX_NUMBER,	NUMBER_OF_DECIMALS, INCREMENT, PAGE_INCREMENT);
		} 
		else if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(WESTERN_ARM_PREFIX) 
				&& !previousNamePrefix.equals(WESTERN_ARM_PREFIX)) 
		{
			nameNumberSpinner.setValues(WESTERN_ARM_MIN_NUMBER, WESTERN_ARM_MIN_NUMBER, 
					WESTERN_ARM_MAX_NUMBER,	NUMBER_OF_DECIMALS, INCREMENT, PAGE_INCREMENT);
		}
		else if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(SOUTHERN_ARM_PREFIX) 
				&& !previousNamePrefix.equals(SOUTHERN_ARM_PREFIX)) 
		{
			nameNumberSpinner.setValues(SOUTHERN_ARM_MIN_NUMBER, SOUTHERN_ARM_MIN_NUMBER, 
					SOUTHERN_ARM_MAX_NUMBER, NUMBER_OF_DECIMALS, INCREMENT, PAGE_INCREMENT);
		}
		else if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(PAMPA_LA_BOLA_ARM_PREFIX) 
				&& !previousNamePrefix.equals(PAMPA_LA_BOLA_ARM_PREFIX)) 
		{
			nameNumberSpinner.setValues(PAMPA_LA_BOLA_ARM_MIN_NUMBER, PAMPA_LA_BOLA_ARM_MIN_NUMBER, 
					PAMPA_LA_BOLA_ARM_MAX_NUMBER, NUMBER_OF_DECIMALS, INCREMENT, PAGE_INCREMENT);
		}
		else if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(ACA_PREFIX) 
				&& !previousNamePrefix.equals(ACA_PREFIX)) 
		{
			nameNumberSpinner.setValues(ACA_MIN_NUMBER, ACA_MIN_NUMBER, 
					ACA_MAX_NUMBER, NUMBER_OF_DECIMALS, INCREMENT, PAGE_INCREMENT);
		}
		else if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(NORTH_SOUTH_ACA_EXTENSION_PREFIX) 
				&& !previousNamePrefix.equals(NORTH_SOUTH_ACA_EXTENSION_PREFIX)) 
		{
			nameNumberSpinner.setValues(NORTH_SOUTH_EXTENSION_MIN_NUMBER, NORTH_SOUTH_EXTENSION_MIN_NUMBER, 
					NORTH_SOUTH_EXTENSION_MAX_NUMBER, NUMBER_OF_DECIMALS, INCREMENT, PAGE_INCREMENT);
		}
		else if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(TOTAL_POWER_PREFIX) 
				&& !previousNamePrefix.equals(TOTAL_POWER_PREFIX)) 
		{
			nameNumberSpinner.setValues(TOTAL_POWER_MIN_NUMBER, TOTAL_POWER_MIN_NUMBER, 
					TOTAL_POWER_MAX_NUMBER, NUMBER_OF_DECIMALS, INCREMENT, PAGE_INCREMENT);
		}
		else if(namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex()).equals(TEST_ARRAY_PREFIX) 
				&& !previousNamePrefix.equals(TEST_ARRAY_PREFIX)) 
		{
			nameNumberSpinner.setValues(TEST_ARRAY_MIN_NUMBER, TEST_ARRAY_MIN_NUMBER, 
					TEST_ARRAY_MAX_NUMBER, NUMBER_OF_DECIMALS, INCREMENT, PAGE_INCREMENT);
		}
		previousNamePrefix = namePrefixCombo.getItem(namePrefixCombo.getSelectionIndex());
	}

	private boolean padExistsInConfig()
	{
		boolean retVal = false;

		if(null == baseElements) {
			this.baseElements = configuration.getBaseElements();
		}
		try {
			retVal = foundCorrespondingBaseElement();			
		} catch(Exception e) {
			// problem encountered; probably not hydrated, so hydrate & try again
			try {
				HwConfigurationConversationUtils.getInstance().hydrateBaseElements(configuration);
				retVal = foundCorrespondingBaseElement();
			} catch(Exception ex) {
				// still a problem!? throw an exception.
				throw new RuntimeException("Unable to hydrate base elements", ex);
			}
		}


		if(retVal == true) {
			this.setStatus(PAD_ALREADY_EXISTS);
		} else {
			this.setStatus(null);
		}
		return retVal;
	}

	private boolean foundCorrespondingBaseElement() 
	{
		boolean retVal = false;
		for(BaseElement be: baseElements) 
		{
			if(be.getType().equals(BaseElementType.Pad) && be.getName().equals(getPadName())) 
			{
				retVal = true;
				break;
			}
		}
		return retVal;
	}

	private void createCableDelayControl()
	{
		new Label(this, SWT.NONE).setText(PAD_DELAY + " " + PAD_DELAY_UNITS);
		cableDelay = new Text(this, SWT.SINGLE | SWT.BORDER);
		GridData gd = GuiUtils.getGridDataForCharWidth(NUM_CHARS_FOR_DELAY, cableDelay);
		cableDelay.setLayoutData(gd);
		cableDelay.addVerifyListener(new VerifyDecimalListener());
		cableDelay.addKeyListener(new SetDirtyKeyListener());
	}
	
	private void createPadPositionControl() 
	{
		Composite positionGroup = new Composite(this, SWT.NONE);
		GridLayout gridLayoutOuter = new GridLayout();
		gridLayoutOuter.numColumns = 4;
		positionGroup.setLayout(gridLayoutOuter);
		GridData gridDataOuter = new GridData();
		gridDataOuter.horizontalSpan = 4;
		positionGroup.setLayoutData(gridDataOuter);

		// Pad position
		Group coordinates = new Group(positionGroup, SWT.NONE);
		coordinates.setText(POSITION);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		coordinates.setLayout(gridLayout);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 4;
		coordinates.setLayoutData(gridData);

		new Label(coordinates, SWT.NONE).setText("x:");
		positionX = new Text(coordinates, SWT.SINGLE | SWT.BORDER);
		GridData gd = new GridData();
		GC gc = new GC(positionX);
		gd.widthHint = gc.stringExtent(SAMPLE_COORDINATE_STRING).x;
		positionX.setLayoutData(gd);
		positionX.addVerifyListener(new VerifyDecimalListener());
		positionX.addKeyListener(new SetDirtyKeyListener());

		new Label(coordinates, SWT.NONE).setText("y:");
		positionY = new Text(coordinates, SWT.SINGLE | SWT.BORDER);
		positionY.setLayoutData(gd);
		positionY.addVerifyListener(new VerifyDecimalListener());
		positionY.addKeyListener(new SetDirtyKeyListener());

		new Label(coordinates, SWT.NONE).setText("z:");
		positionZ = new Text(coordinates, SWT.SINGLE | SWT.BORDER);
		positionZ.setLayoutData(gd);
		positionZ.addVerifyListener(new VerifyDecimalListener());
		positionZ.addKeyListener(new SetDirtyKeyListener());
	}

	private void createPadPrefixAndNumberControl() 
	{
		Group prefixAndNumberComposite = new Group(this, SWT.NONE);
		prefixAndNumberComposite.setText(NAME);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 4;
		prefixAndNumberComposite.setLayout(gridLayout);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 4;
		prefixAndNumberComposite.setLayoutData(gridData);
		GridData gd;

		// Antenna name prefix
		Label lName = new Label( prefixAndNumberComposite, SWT.NULL );
		lName.setText( PREFIX );
		namePrefixCombo = new Combo( prefixAndNumberComposite, SWT.READ_ONLY ); 
		gd = new GridData();
		namePrefixCombo.setItems(NAME_PREFIX_ARRAY );
		//namePrefixCombo.select(0);
		namePrefixCombo.setLayoutData( gd );
		
		namePrefixCombo.addSelectionListener(new SetDirtySelectionListener());

		// Antenna number
		Label lNumber = new Label( prefixAndNumberComposite, SWT.NULL );
		lNumber.setText( NUMBER );
		nameNumberSpinner = new Spinner( prefixAndNumberComposite, SWT.READ_ONLY );
		gd = new GridData();
		nameNumberSpinner.setValues(INNER_ARRAY_MIN_NUMBER, INNER_ARRAY_MIN_NUMBER, 
				INNER_ARRAY_MAX_NUMBER, NUMBER_OF_DECIMALS, INCREMENT, PAGE_INCREMENT);
		nameNumberSpinner.setLayoutData( gd );
		nameNumberSpinner.addSelectionListener(new SetDirtySelectionListener());
	}

	private class SetDirtySelectionListener implements SelectionListener 
	{
		@Override
		public void widgetDefaultSelected(SelectionEvent e) {
			setDirty(true);
		}

		@Override
		public void widgetSelected(SelectionEvent e) {
			setDirty(true);
		}
	}
	
	private class SetDirtyKeyListener implements KeyListener 
	{
		@Override
		public void keyPressed(KeyEvent e) {
			setDirty(true);
		}

		@Override
		public void keyReleased(KeyEvent e) {
		}
	}
	
	private void createCommissionDateControl() 
	{
		Label commissionDateLabel = new Label(this, SWT.NONE);
		commissionDateLabel.setText(COMMISSION_DATE);
		commissionDate = new DateTime(this, SWT.DATE | SWT.MEDIUM);
		
		commissionDate.addSelectionListener(new SetDirtySelectionListener());
	}
}
