/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.widgets;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Text;

import alma.acs.logging.level.AcsLogLevelDefinition;
import alma.acs.tmcdb.LoggingConfig;
import alma.obops.tmcdbgui.editors.TmcdbObjectEditor;

public class LoggingConfigComposite extends DirtyPublishingComposite
{
	/**
	 * 
	 */
	private static final int DEFAULT_LOG_LEVEL_INDEX_VALUE = 2;
	private LoggingConfig loggingConfig;
	private Combo lMinLLDefCombo;
	private Combo lMinLLLDefCombo;
	private Text    lCentralLoggerText;
	private Spinner lDispPackSizeSpin;
	private Spinner lImmDispLvlSpin;
	private Spinner lFlushPerSpin;
	private Spinner lMaxLQSSpin;
	private Spinner lMaxLPSSpin;
	
	public LoggingConfigComposite(Composite parent, int style, LoggingConfig loggingConfig) {
		super(parent, style);
		this.loggingConfig = loggingConfig;
		createLoggingConfigWidgets(this);
	}

	@Override
	public boolean setFocus() {
		return lMinLLDefCombo.setFocus();
	}
	
	private void createLoggingConfigWidgets(Composite composite) {
		composite.setLayout(new GridLayout(2, false));
		GridData gd = new GridData(SWT.FILL, SWT.TOP, true, false);
		gd.horizontalSpan = 2;
		Group g = new Group(composite, SWT.BORDER);
		g.setLayout(new GridLayout(2, false));
		g.setLayoutData(gd);
		g.setText("Logging configuration");

		/* Min Log Level Default */ 
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label lMinLogLevelDefLabel = new Label(g, SWT.NONE);
		lMinLogLevelDefLabel.setText("Default Min Log Level");
		lMinLogLevelDefLabel.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalIndent = 20;
		gd.minimumWidth = 50;
		lMinLLDefCombo = new Combo(g, SWT.DROP_DOWN | SWT.READ_ONLY);
		initCombo(lMinLLDefCombo);
		lMinLLDefCombo.setLayoutData(gd);

		/* Min Log Level Default */ 
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label lMinLogLevelLocalDefLabel = new Label(g, SWT.NONE);
		lMinLogLevelLocalDefLabel.setText("Default Min Log Level (Local)");
		lMinLogLevelLocalDefLabel.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalIndent = 20;
		gd.minimumWidth = 50;
		lMinLLLDefCombo = new Combo(g, SWT.DROP_DOWN | SWT.READ_ONLY);
		initCombo(lMinLLLDefCombo);
		lMinLLLDefCombo.setLayoutData(gd);

		/* Centralized logger */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label lCentralLoggerLabel = new Label(g, SWT.NONE);
		lCentralLoggerLabel.setText("Centralized logger");
		lCentralLoggerLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		gd.minimumWidth = 50;
		lCentralLoggerText = new Text(g, SWT.BORDER);
		lCentralLoggerText.setLayoutData(gd);

		/* Dispatch Packet Size */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label lDispatchPacketSizeLabel = new Label(g, SWT.NONE);
		lDispatchPacketSizeLabel.setText("Dispatch Packet Size");
		lDispatchPacketSizeLabel.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalIndent = 20;
		gd.minimumWidth = 50;
		lDispPackSizeSpin = new Spinner(g, SWT.NONE);
		lDispPackSizeSpin.setLayoutData(gd);

		/* Immediate Dispatch Level */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label lImmediateDispatchLevelLabel = new Label(g, SWT.NONE);
		lImmediateDispatchLevelLabel.setText("Immediate Dispatch Level");
		lImmediateDispatchLevelLabel.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalIndent = 20;
		gd.minimumWidth = 50;
		lImmDispLvlSpin = new Spinner(g, SWT.NONE);
		lImmDispLvlSpin.setLayoutData(gd);

		/* Flush Period (in seconds) */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label lFlushPeriodLabel = new Label(g, SWT.NONE);
		lFlushPeriodLabel.setText("Flush Period (in seconds)");
		lFlushPeriodLabel.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalIndent = 20;
		gd.minimumWidth = 50;
		lFlushPerSpin = new Spinner(g, SWT.NONE);
		lFlushPerSpin.setLayoutData(gd);

		/* Max Log Queue Size */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label lMaxLogQueueLabel = new Label(g, SWT.NONE);
		lMaxLogQueueLabel.setText("Max Log Queue Size");
		lMaxLogQueueLabel.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalIndent = 20;
		gd.minimumWidth = 50;
		lMaxLQSSpin = new Spinner(g, SWT.NONE);
		lMaxLQSSpin.setLayoutData(gd);

		/* Max Log per second */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label lMaxLogPerSecondLabel = new Label(g, SWT.NONE);
		lMaxLogPerSecondLabel.setText("Max Logs per second");
		lMaxLogPerSecondLabel.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalIndent = 20;
		gd.minimumWidth = 50;
		lMaxLPSSpin = new Spinner(g, SWT.NONE);
		lMaxLPSSpin.setMinimum(-1);
		lMaxLPSSpin.setLayoutData(gd);

		initializeWidgets();

		// Subscribe to changes
		subscribeToChanges( lMinLLDefCombo, lMinLLLDefCombo );
		subscribeToChanges( lDispPackSizeSpin, lImmDispLvlSpin, lFlushPerSpin, lMaxLQSSpin, lMaxLPSSpin );
		subscribeToChanges( lCentralLoggerText );
	}

	private void initCombo(Combo combo) {
		AcsLogLevelDefinition[] comboValues = AcsLogLevelDefinition.values();
	    String[] comboStringValues = new String[comboValues.length];
	    for(int i = 0; i < comboValues.length; i++) {
	    	comboStringValues[i] = comboValues[i].toString();
	    }
	    combo.setItems(comboStringValues);
	}

	private void initializeWidgets() 
	{
		// Fill with the initial contents
		if( loggingConfig != null ) {
			lMinLLDefCombo.select( indexOfLogLevel(Integer.valueOf( TmcdbObjectEditor.nullSafeByte(loggingConfig.getMinLogLevelDefault(), (byte)2 )))) ;
			lMinLLLDefCombo.select( indexOfLogLevel(Integer.valueOf(TmcdbObjectEditor.nullSafeByte(loggingConfig.getMinLogLevelLocalDefault(), (byte)2 ))));
			lCentralLoggerText.setText( TmcdbObjectEditor.nullSafeString(loggingConfig.getCentralizedLogger(), "Log"));
			lDispPackSizeSpin.setSelection( TmcdbObjectEditor.nullSafeByte(loggingConfig.getDispatchPacketSize(), (byte)10) );
			lImmDispLvlSpin.setSelection( TmcdbObjectEditor.nullSafeByte(loggingConfig.getImmediateDispatchLevel(), (byte)10) );
			lFlushPerSpin.setSelection( TmcdbObjectEditor.nullSafeByte(loggingConfig.getFlushPeriodSeconds(), (byte)10) );
			lMaxLQSSpin.setSelection( TmcdbObjectEditor.nullSafeInteger(loggingConfig.getMaxLogQueueSize(), 1000) );
			lMaxLPSSpin.setSelection( TmcdbObjectEditor.nullSafeInteger(loggingConfig.getMaxLogsPerSecond(), -1) );
		}
	}
	
	private int indexOfLogLevel(Integer level) {
		if(null == level) {
			return DEFAULT_LOG_LEVEL_INDEX_VALUE;
		}
		int count = 0;
		for(AcsLogLevelDefinition def : AcsLogLevelDefinition.values()) {
			if(level.equals(def.value)) {
				return count;
			}
			count++;
		}
		
		return count;
	}

	protected void subscribeToChanges(Combo ... widgets) {
		SelectionListener listener = new SelectionListener() {
			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				setDirty(true);
			}

			@Override
			public void widgetSelected(SelectionEvent e) {
				setDirty(true);
			}
		};
		for(Combo c: widgets) {
			c.addSelectionListener(listener);
		}
	}

	protected void subscribeToChanges(Text ... widgets) {
		ModifyListener listener = new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				setDirty(true);
			}
		};
		for(Text t: widgets) {
			t.addModifyListener(listener);
		}
	}
	
	protected void subscribeToChanges(Spinner ... widgets) {
		ModifyListener listener = new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				setDirty(true);
			}
		};
		for(Spinner s: widgets) {
			s.addModifyListener(listener);
		}
	}
	
	public void setLoggingConfig(LoggingConfig config)
	{
		this.loggingConfig = config;
		initializeWidgets();
	}
	
	public Integer getMaxLogsPerSecond() {
		return lMaxLPSSpin.getSelection();
	}
	public Integer getMaxLogQueueSize()
	{
		return lMaxLQSSpin.getSelection();
	}
	
	public Byte getFlushPeriodSeconds()  
	{
		return (byte)lFlushPerSpin.getSelection(); 
	}
	
	public Byte getImmediateDispatchLevel() 
	{
		return (byte)lImmDispLvlSpin.getSelection();
	}
	
	public Byte getDispatchPacketSize() {
		return (byte)lDispPackSizeSpin.getSelection();
	}
	
	public String getCentralizedLogger() {
		return lCentralLoggerText.getText();
	}
	
	public Byte getMinLogLevelLocalDefault() {
		return (byte) AcsLogLevelDefinition.valueOf(lMinLLLDefCombo.getItem(lMinLLLDefCombo.getSelectionIndex())).value;
	}
	
	public Byte getMinLogLevelDefault()
	{
		return (byte) AcsLogLevelDefinition.valueOf(lMinLLDefCombo.getItem(lMinLLDefCombo.getSelectionIndex())).value;
	}
	
}
