/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.widgets;

import java.text.DecimalFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Set;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DateTime;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.obops.tmcdbgui.widgets.support.StatusListener;
import alma.obops.tmcdbgui.wizards.support.VerifyDecimalListener;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.Coordinate;
import alma.tmcdb.domain.HolographyTower;
import alma.tmcdb.domain.HwConfiguration;

public class HolographyTowerAttributesComposite extends StatusPublishingComposite 
{
	public static final String HOLOGRAPHYTOWER_ALREADY_EXISTS = "Holography tower already exists: name must be unique";
	private HolographyTower holographytower;
	private Text nameText;
	private DateTime commissionDate;
	private Set<BaseElement> baseElements;
    private HwConfiguration configuration;
	private Text positionX, positionY, positionZ;
	private static final String POSITION = "Position (m)";
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 * @param holographytower the holographytower that is being "dealt" with.
	 */
	public HolographyTowerAttributesComposite(Composite parent, int style, HolographyTower holographytower, StatusListener statusListener, DirtyListener dirtyListener)
	{
		super(parent, style);
		this.addStatusListener(statusListener);
		this.addDirtyListener(dirtyListener);
		createControl(holographytower);
	}
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public HolographyTowerAttributesComposite(Composite parent, int style, DirtyListener dirtyListener) 
	{
		this(parent, style, null, null, dirtyListener);
	}
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public HolographyTowerAttributesComposite(Composite parent, int style, StatusListener statusListener, HwConfiguration config) 
	{
		this(parent, style, null, statusListener, null);
		this.configuration = config;
	}

	@Override
	public boolean setFocus() {
		return nameText.setFocus();
	}
	
	private void createControl(HolographyTower ht) 
	{
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;  // label, entry
		setLayout( layout );
		GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);
		setLayoutData(gridData);

		createNameControl();
		createCommissionDateControl();
		createPositionControl();
		
        setHolographyTower(ht);
        addKeyListeners();
	}	
	
	private void createPositionControl() 
	{
		Composite positionGroup = new Composite(this, SWT.NONE);
		GridLayout gridLayoutOuter = new GridLayout();
		gridLayoutOuter.numColumns = 4;
		positionGroup.setLayout(gridLayoutOuter);
		GridData gridDataOuter = new GridData();
		gridDataOuter.horizontalSpan = 4;
		positionGroup.setLayoutData(gridDataOuter);

		// Antenna position
		Group coordinates = new Group(positionGroup, SWT.NONE);
		coordinates.setText(POSITION);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		coordinates.setLayout(gridLayout);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 4;
		coordinates.setLayoutData(gridData);

		new Label(coordinates, SWT.NONE).setText("x:");
		positionX = new Text(coordinates, SWT.SINGLE | SWT.BORDER);
		GridData gd = new GridData();
		GC gc = new GC(positionX);
		gd.widthHint = gc.stringExtent(PadAttributesComposite.SAMPLE_COORDINATE_STRING).x;
		positionX.setLayoutData(gd);
		positionX.addVerifyListener(new VerifyDecimalListener());
		positionX.addKeyListener(new SetDirtyKeyListener());

		new Label(coordinates, SWT.NONE).setText("y:");
		positionY = new Text(coordinates, SWT.SINGLE | SWT.BORDER);
		positionY.setLayoutData(gd);
		positionY.addVerifyListener(new VerifyDecimalListener());
		positionY.addKeyListener(new SetDirtyKeyListener());

		new Label(coordinates, SWT.NONE).setText("z:");
		positionZ = new Text(coordinates, SWT.SINGLE | SWT.BORDER);
		positionZ.setLayoutData(gd);
		positionZ.addVerifyListener(new VerifyDecimalListener());
		positionZ.addKeyListener(new SetDirtyKeyListener());
	}

	private void createNameControl() 
	{
        Label nameLabel = new Label(this, SWT.NONE);
        nameLabel.setText("Name");
        nameText = new Text(this, SWT.BORDER);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        nameText.setLayoutData(gridData);
        nameText.addKeyListener(new SetDirtyKeyListener());
	}

	private void createCommissionDateControl() 
	{
		Composite commissionDateComposite = new Composite(this, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 4;
		commissionDateComposite.setLayout(gridLayout);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 4;
		commissionDateComposite.setLayoutData(gridData);
		
		Label commissionDateLabel = new Label(commissionDateComposite, SWT.NONE);
        commissionDateLabel.setText("Commission date");
        commissionDate = new DateTime(commissionDateComposite, SWT.DATE | SWT.MEDIUM);
        commissionDate.addSelectionListener(new SelectionListener() {

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				setDirty(true);
			}

			@Override
			public void widgetSelected(SelectionEvent e) {
				setDirty(true);
			}
        	
        });
	}
	
	private void addKeyListeners() 
	{
		// At each keystroke computes whether this page is complete
		KeyListener completionKL = new KeyListener() 
		{
			public void keyPressed( KeyEvent e ) {
				// ignore
			}

			public void keyReleased( KeyEvent e ) {
				isComplete();
			}
		};
		nameText.addKeyListener(completionKL);
		positionX.addKeyListener(completionKL);
		positionY.addKeyListener(completionKL);
		positionZ.addKeyListener(completionKL);
	}
	
	/**
	 * Getter for the new antenna's commission date.
	 * @return the new antenna's commission date.
	 */
	public Date getCommissionDate() 
	{
		Date retVal = null;
		
		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.YEAR, this.commissionDate.getYear());
		cal.set(Calendar.MONTH, this.commissionDate.getMonth());
		cal.set(Calendar.DAY_OF_MONTH, this.commissionDate.getDay());
		cal.set(Calendar.HOUR_OF_DAY, this.commissionDate.getHours());
		cal.set(Calendar.MINUTE, this.commissionDate.getMinutes());
		cal.set(Calendar.SECOND, this.commissionDate.getSeconds());
		retVal = cal.getTime();
		
		return retVal;
	}
	
	/**
	 * Getter for the new antenna's name.
	 * @return the new antenna's name.
	 */
	public String getHolographyTowerName() 
	{
		String retVal = null;
		retVal = nameText.getText();
		return retVal;
	}
	
	/**
	 * Getter for the new antenna's position.
	 * @return the position of the new antenna.
	 */
	public Coordinate getPosition() 
	{
		Coordinate retVal = new Coordinate();
		if(positionX.getText() != null && positionX.getText().trim().length() > 0) {
		   retVal.setX(Double.valueOf(positionX.getText()));
		}
		if(positionY.getText() != null && positionY.getText().trim().length() > 0) {
			retVal.setY(Double.valueOf(positionY.getText()));			
		}
		if((positionZ.getText() != null && positionZ.getText().trim().length() > 0)) {
		   retVal.setZ(Double.valueOf(positionZ.getText()));
		}
		return retVal;
	}

	private boolean holographytowerExistsInConfig()
	{
		boolean retVal = false;

		if(null == baseElements) 
		{
			this.baseElements = configuration.getBaseElements();
		} 	

		try	{
			retVal = foundCorrespondingBaseElement();
		}
		catch(Exception ex) {
			throw new RuntimeException("Unable to get the base elements for the configuration", ex);
		}
		
		if(retVal == true) {
			this.setStatus(HOLOGRAPHYTOWER_ALREADY_EXISTS);
		} else {
			this.setStatus(null);
		}
		return retVal;
	}
	
	private boolean foundCorrespondingBaseElement() {
		boolean retVal = false;
		for(BaseElement be: baseElements) 
		{
			if(be.getType().equals(BaseElementType.HolographyTower) && be.getName().equals(getHolographyTowerName())) 
			{
				retVal = true;
				break;
			}
		}
		return retVal;
	}
	
	
	/** @return <code>true</code> when all required fields are populated */
	public boolean isComplete() 
	{	
		boolean complete = 
			!holographytowerExistsInConfig() &&
			(positionX.getText().length() > 0) &&
			(positionY.getText().length() > 0) &&
			(positionZ.getText().length() > 0) &&
			(this.getCommissionDate()!= null ) && 
			(nameText.getText().length() > 0);

		notifyListenersOfCompletion(complete);
		return complete;
	}

	private class SetDirtyKeyListener implements KeyListener 
	{
		@Override
		public void keyPressed(KeyEvent e) {
			setDirty(true);
		}

		@Override
		public void keyReleased(KeyEvent e) {
		}
	}

	public void setHolographyTower(HolographyTower ht) 
	{
		if(null == ht || this.holographytower == ht) 
		{
			return;
		}
		this.holographytower = ht;
		
		// commissionDate
		Calendar cal = Calendar.getInstance();
		cal.setTimeInMillis(holographytower.getCommissionDate());
		this.commissionDate.setDate(cal.get(Calendar.YEAR), cal.get(Calendar.MONTH), cal.get(Calendar.DAY_OF_MONTH));

		// name
		this.nameText.setText(holographytower.getName());
		
		// position
		if(null != holographytower.getPosition()) {
			DecimalFormat formatter = new DecimalFormat(AntennaAttributesComposite.COORDINATE_FORMAT);
			String formattedX = formatter.format(holographytower.getPosition().getX());
			String formattedY = formatter.format(holographytower.getPosition().getY());
			String formattedZ = formatter.format(holographytower.getPosition().getZ());
			this.positionX.setText(formattedX);
			this.positionY.setText(formattedY);
			this.positionZ.setText(formattedZ);
		} else {
			this.positionX.setText("");
			this.positionY.setText("");
			this.positionZ.setText("");
		}
		
		this.configuration = holographytower.getConfiguration();
	}
}
