/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.widgets;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.swt.widgets.Composite;

import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.obops.tmcdbgui.widgets.support.DirtyPublisher;


/**
 * "Adapter" class for composite classes which wish to be able to notify listeners when they are 'dirty' 
 * (i.e. modified and possibly in need of their changes being saved to persistent storage).
 * 
 * @author sharring
 */
public abstract class DirtyPublishingComposite extends Composite implements DirtyPublisher
{
	protected boolean dirty;
	protected List<DirtyListener> listeners = new ArrayList<DirtyListener>();
	   
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style.
	 */
	public DirtyPublishingComposite(Composite parent, int style)
	{
		super(parent, style);
		this.dirty = false;
	}
	
	/**
	 * Adds a listener, which will be notified when the dirty state has changed.
	 * @param listener the listener to be notified upon dirty state changing.
	 */
	public void addDirtyListener(DirtyListener listener)
	{
		if(listener != null) {
			this.listeners.add(listener);
		}
	}
	
	/**
	 * Removes a dirty listener.
	 * @param listener the listener to remove.
	 */
	public void removeDirtyListener(DirtyListener listener) 
	{
		if(listener != null) {
			this.listeners.remove(listener);
		}
	}
	
	/**
	 * Notifies all listeners that the dirty state has changed.
	 */
	public void notifyDirtyListeners()
	{
		for(DirtyListener listener : listeners) {
			listener.setDirty(dirty);
		}
	}
	
	/**
	 * Sets the dirty state & notifies listeners, if any.
	 * @param dirty the new state of dirtiness.
	 */
	public void setDirty(boolean dirty) {
		if(this.dirty != dirty) {
			this.dirty = dirty;
			notifyDirtyListeners();
		}
	}
	
	/**
	 * Getter for the dirty state.
	 * @return the dirty flag.
	 */
	public boolean getDirty() {
		return this.dirty;
	}
}