/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.widgets;

import java.text.DecimalFormat;
import java.util.Date;
import java.util.List;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.hibernate.criterion.MatchMode;

import alma.acs.tmcdb.Configuration;
import alma.obops.tmcdbgui.dialogs.ConfigurationSelectionDialog;
import alma.obops.tmcdbgui.dialogs.ConfigurationSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.StartupScenarioConversationUtils;
import alma.obops.tmcdbgui.views.providers.helpers.config.AntennaHelper;
import alma.obops.tmcdbgui.views.providers.helpers.config.HolographyTowerHelper;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.obops.tmcdbgui.widgets.support.StatusListener;
import alma.obops.tmcdbgui.wizards.support.VerifyDecimalListener;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.ArrayReference;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.HolographyTower;
import alma.tmcdb.domain.HolographyTowerToPad;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.StartupScenario;

/**
 * Composite used to edit (or create) a configuration.
 * @author sharring
 */
public class ConfigurationAttributesComposite extends StatusPublishingComposite 
{
	private static final String POSITION = "Array Reference Position (m)";
	protected static final int TEXT_WIDTH_SMALL = 130;
	private static final String ACTIVE = "Active";
	private static final String FULL_NAME = "Full name";
	private static final String TELESCOPE_NAME = "Telescope name";
	private static final String GLOBAL_CONFIG = "Global configuration";
	private static final String NAME = "Name";
	private static final String DESCRIPTION = "Description";
	private Text descriptionText, configurationNameText, fullNameText, telescopeNameText, globalConfigText;
	private Text arrayReferenceX, arrayReferenceY, arrayReferenceZ;
	private Button activeButton;
	private Button removeGlobalConfigButton;
	private HwConfiguration config;
	private HwConfiguration globalConfig;
	private ModifyListener modifyListener;
	
	public ConfigurationAttributesComposite(Composite parent, int style, StatusListener listener, 
		HwConfiguration config, DirtyListener dirtyListener) 
	{
		super(parent, style);
		this.addStatusListener(listener);
		this.addDirtyListener(dirtyListener);
		createControl();
		addModifyListener();
		addSelectionListener();
		setConfiguration(config);
	}
	
	private void addSelectionListener()
	{
		SelectionListener listener = new SelectionListener() 
		{

			@Override
			public void widgetDefaultSelected(SelectionEvent arg0) {
			}

			@Override
			public void widgetSelected(SelectionEvent arg0) {
				isComplete();
				setDirty(true);
			}
			
		};
		
		activeButton.addSelectionListener(listener);
	}
	
	private void addModifyListener() 
	{
		// At each keystroke computes whether this page is complete
		modifyListener =  new ModifyListener() 
		{

			@Override
			public void modifyText(ModifyEvent arg0) {
				isComplete();
				setDirty(true);
			}
		};
		arrayReferenceX.addModifyListener(modifyListener);
		arrayReferenceY.addModifyListener(modifyListener);
		arrayReferenceZ.addModifyListener(modifyListener);
		descriptionText.addModifyListener(modifyListener);
		configurationNameText.addModifyListener(modifyListener);
		fullNameText.addModifyListener(modifyListener);
		telescopeNameText.addModifyListener(modifyListener);
	}

	private void createControl() 
	{
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;  // label, entry
		this.setLayout(layout);
		GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);
		setLayoutData(gridData);
		
		new Label(this, SWT.NONE).setText(NAME);
		GridData configurationNameTextLData = new GridData();
		configurationNameTextLData.grabExcessHorizontalSpace = true;
		configurationNameTextLData.verticalAlignment = GridData.FILL;
		configurationNameTextLData.horizontalAlignment = GridData.FILL;
		configurationNameText = new Text(this, SWT.SINGLE | SWT.BORDER);
		configurationNameText.setLayoutData(configurationNameTextLData);
		configurationNameText.setEnabled(GuiUtils.isGodUser());

		new Label(this, SWT.NONE).setText(FULL_NAME);
		GridData fullNameTextLData = new GridData();
		fullNameTextLData.grabExcessHorizontalSpace = true;
		fullNameTextLData.horizontalAlignment = GridData.FILL;
		fullNameText = new Text(this, SWT.SINGLE | SWT.BORDER);
		fullNameText.setLayoutData(fullNameTextLData);
		fullNameText.setEnabled(GuiUtils.isGodUser());
		
		new Label(this, SWT.NONE).setText(DESCRIPTION);
		GridData descriptionTextLData = new GridData();
		descriptionTextLData.grabExcessHorizontalSpace = true;
		descriptionTextLData.horizontalAlignment = GridData.FILL;
		descriptionText = new Text(this, SWT.SINGLE | SWT.BORDER);
		descriptionText.setLayoutData(descriptionTextLData);
		descriptionText.setEnabled(GuiUtils.isGodUser());

		new Label(this, SWT.NONE).setText(TELESCOPE_NAME);
		GridData telescopeNameTextLData = new GridData();
		telescopeNameTextLData.grabExcessHorizontalSpace = true;
		fullNameTextLData.horizontalAlignment = GridData.FILL;
		telescopeNameText = new Text(this, SWT.SINGLE | SWT.BORDER);
		telescopeNameText.setLayoutData(fullNameTextLData);
		telescopeNameText.setEnabled(GuiUtils.isGodUser());
		
		new Label(this, SWT.NONE).setText(GLOBAL_CONFIG);
		GridData globalConfigTextLData = new GridData();
		globalConfigTextLData.grabExcessHorizontalSpace = true;
		globalConfigTextLData.horizontalAlignment = GridData.FILL;
		globalConfigText = new Text(this, SWT.SINGLE | SWT.BORDER);
		globalConfigText.setLayoutData(fullNameTextLData);
		globalConfigText.setEnabled(false);
		
		GridData gd = new GridData(SWT.LEFT, SWT.LEFT, false, false);
		Button browseConfigsButton = new Button(this, SWT.PUSH);
		browseConfigsButton.setText("Add global");
		browseConfigsButton.setEnabled(GuiUtils.isGodUser());
		browseConfigsButton.setLayoutData(gd);
		browseConfigsButton.addSelectionListener(new SelectionListener() 
		{
			public void widgetSelected(SelectionEvent e) 
			{
				Cursor cursor = globalConfigText.getCursor();
				globalConfigText.setCursor(globalConfigText.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
				if(config.getGlobalConfiguration() == null) {
					displayDialog();
				} else if(configHasNoGlobalReferences()) {
					displayDialog();
				} else {
					displayGlobalCrossReferencesWarning();
				}
				globalConfigText.setCursor(cursor);
			}
			
			private void displayDialog() 
			{
				ElementListSelectionDialog d = 
					new ConfigurationSelectionDialog(true, false, getShell(), 
							new ConfigurationSelectionDialogLabelProvider());
				d.open();
				Object configs[] = d.getResult();
				if( configs != null && configs.length == 1 ) {
					String globalConfigName = (String)configs[0];
					List<HwConfiguration> globalConfigs;
					try {
						globalConfigs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(globalConfigName, MatchMode.EXACT);
					} catch (Exception e) {
						globalConfigs = null;
					}
					if(globalConfigs != null && globalConfigs.size() == 1) {
						globalConfig = globalConfigs.get(0);
						globalConfigText.setText(globalConfig.getName());
						setDirty(true);
					}
				}
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		
		gd = new GridData(SWT.LEFT, SWT.LEFT, false, false);
		removeGlobalConfigButton = new Button(this, SWT.PUSH);
		removeGlobalConfigButton.setText("Remove global");
		removeGlobalConfigButton.setEnabled(GuiUtils.isGodUser() && globalConfigText.getText() != null && globalConfigText.getText().trim().length() > 0);
		removeGlobalConfigButton.setLayoutData(gd);
		removeGlobalConfigButton.addSelectionListener(new SelectionListener() 
		{
			public void widgetSelected(SelectionEvent e) 
			{
				if(configHasNoGlobalReferences()) {
					globalConfigText.setText("");
					globalConfig = null;
					removeGlobalConfigButton.setEnabled(false);
					setDirty(true);
				} else {
					displayGlobalCrossReferencesWarning();
				}
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				// TODO Auto-generated method stub
			}
		});
		
		new Label(this, SWT.NONE);
		
		activeButton = new Button(this, SWT.CHECK);
		GridData activeButtonLData = new GridData();
		activeButtonLData.verticalIndent = 12;
		activeButton.setLayoutData(activeButtonLData);
		activeButton.setText(ACTIVE);
		activeButton.setEnabled(GuiUtils.isGodUser());
		
		createArrayReferenceControl();
		pack();
	}
	
	private boolean configHasNoGlobalReferences() 
	{
		boolean retVal = true;
		
		if(retVal)
		{
			for(BaseElement be: config.getBaseElements()) 
			{
				if(be.getType().equals(BaseElementType.Antenna) || be instanceof Antenna) {
					Antenna antenna = (Antenna) be;
					AntennaToPad a2p = AntennaHelper.findCurrentAntennaToPadForAntenna(antenna);
					if(a2p != null && !a2p.getPad().getConfiguration().getId().equals(antenna.getConfiguration().getId())) {
						retVal = false;
						break;
					} 
				}
				else if(be.getType().equals(BaseElementType.HolographyTower) || be instanceof HolographyTower) {
					HolographyTower holoTower = (HolographyTower) be;
					HolographyTowerToPad h2p = HolographyTowerHelper.findCurrentHolographyTowerToPadForHolographyTower(holoTower);
					if(h2p != null && !h2p.getPad().getConfiguration().getId().equals(holoTower.getConfiguration().getId())) {
						retVal = false;
						break;
					} 
				}
			}
		}
		
		for(StartupScenario startup: config.getStartupScenarios()) {
			try {
				StartupScenarioConversationUtils.getInstance().hydrateBaseElementStartups(startup);
			} catch (Exception e) {
				e.printStackTrace();
				retVal = false;
			}
			if(retVal)
			{
				for(BaseElementStartup bes: startup.getBaseElementStartups()) {
					if(bes.getBaseElement() != null && !bes.getBaseElement().getConfiguration().getId().equals(config.getId())) {
						retVal = false;
						break;
					}
				}
			}
		}
		
		return retVal;
	}
	
	private void displayGlobalCrossReferencesWarning() {
		MessageDialog.openError(globalConfigText.getShell(), "Cannot reassign global config", 
				"You must delete all existing cross-configuration references (antennaToPad, holographyTowerToPad, baseElementStartup) items before choosing a new global configuration");
		
	}
	
	@Override 
	public void dispose()
	{
		super.dispose();
	}
	
	/** @return <code>true</code> when this page is complete */
	public boolean isComplete() 
	{	
		boolean positionXComplete = (arrayReferenceX.getText().length() > 0);
		boolean positionYComplete = (arrayReferenceY.getText().length() > 0);
		boolean positionZComplete = (arrayReferenceZ.getText().length() > 0);
		boolean configNameComplete = (configurationNameText.getText().length() > 0);
		boolean configFullNameComplete = (fullNameText.getText().length() > 0);
		boolean configDescriptionComplete = (descriptionText.getText().length() > 0);
		boolean telescopeNameComplete = (telescopeNameText.getText().length() > 0);
		
		boolean complete =  (positionXComplete &&
				 positionYComplete && 
				 positionZComplete) 
			&&  configNameComplete &&
			configFullNameComplete &&
			configDescriptionComplete &&
			telescopeNameComplete;

		notifyListenersOfCompletion(complete);
		return complete;
	}
	
	public void setConfiguration(HwConfiguration configuration) 
	{
		// position
		DecimalFormat formatter = new DecimalFormat(AntennaAttributesComposite.COORDINATE_FORMAT);
		
		this.config = configuration;
		if(null == this.config)
		{
			Configuration swConfig = new Configuration();
			swConfig.setCreationTime(new Date());
			swConfig.setActive(false);
			this.config = new HwConfiguration(swConfig);
		}
		
		if(null != config.getArrayReference()) {
			String formattedX = formatter.format(configuration.getArrayReference().getX());
			String formattedY = formatter.format(configuration.getArrayReference().getY());
			String formattedZ = formatter.format(configuration.getArrayReference().getZ());
			this.arrayReferenceX.setText(formattedX);
			this.arrayReferenceY.setText(formattedY);
			this.arrayReferenceZ.setText(formattedZ);
		} else {
			this.arrayReferenceX.setText("");
			this.arrayReferenceY.setText("");
			this.arrayReferenceZ.setText("");
		}
		
		boolean active = this.config.getActive() == null ? false : this.config.getActive();
		this.activeButton.setSelection(active);
		
		String name = this.config.getName() == null ? "" : this.config.getName();
		this.configurationNameText.setText(name);
		
		String descr = this.config.getDescription() == null ? "" : this.config.getDescription();
		this.descriptionText.setText(descr);
		
		String fullName = this.config.getFullName() == null ? "" : this.config.getFullName();
		this.fullNameText.setText(fullName);
		
		String telescopeName = this.config.getTelescopeName() == null ? "" : this.config.getTelescopeName();
		this.telescopeNameText.setText(telescopeName);
		
		String globalConfigName = this.config.getGlobalConfiguration() == null ? "" : this.config.getGlobalConfiguration().getName();
		this.globalConfigText.setText(globalConfigName);
		
		removeGlobalConfigButton.setEnabled(GuiUtils.isGodUser() && globalConfigName.length() > 0);
		this.setDirty(false);
	}
	
	/**
	 * Getter for the new antenna's position.
	 * @return the position of the new antenna.
	 */
	public ArrayReference getArrayReference() {
		ArrayReference retVal = new ArrayReference();
		if(arrayReferenceX.getText() != null && arrayReferenceX.getText().trim().length() > 0) {
		   retVal.setX(Double.valueOf(arrayReferenceX.getText()));
		}
		if(arrayReferenceY.getText() != null && arrayReferenceY.getText().trim().length() > 0) {
			retVal.setY(Double.valueOf(arrayReferenceY.getText()));			
		}
		if((arrayReferenceZ.getText() != null && arrayReferenceZ.getText().trim().length() > 0)) {
		   retVal.setZ(Double.valueOf(arrayReferenceZ.getText()));
		}
		return retVal;
	}
	
	private void createArrayReferenceControl() 
	{
		Composite positionGroup = new Composite(this, SWT.NONE);
		GridLayout gridLayoutOuter = new GridLayout();
		gridLayoutOuter.numColumns = 4;
		positionGroup.setLayout(gridLayoutOuter);
		GridData gridDataOuter = new GridData();
		gridDataOuter.horizontalSpan = 4;
		positionGroup.setLayoutData(gridDataOuter);

		// Array Reference position
		Group coordinates = new Group(positionGroup, SWT.NONE);
		coordinates.setText(POSITION);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		coordinates.setLayout(gridLayout);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 4;
		coordinates.setLayoutData(gridData);

		new Label(coordinates, SWT.NONE).setText("x:");
		arrayReferenceX = new Text(coordinates, SWT.SINGLE | SWT.BORDER);
		GridData gd = new GridData();
		gd.widthHint = TEXT_WIDTH_SMALL;
		arrayReferenceX.setLayoutData(gd);
		arrayReferenceX.addVerifyListener(new VerifyDecimalListener());

		new Label(coordinates, SWT.NONE).setText("y:");
		arrayReferenceY = new Text(coordinates, SWT.SINGLE | SWT.BORDER);
		arrayReferenceY.setLayoutData(gd);
		arrayReferenceY.addVerifyListener(new VerifyDecimalListener());

		new Label(coordinates, SWT.NONE).setText("z:");
		arrayReferenceZ = new Text(coordinates, SWT.SINGLE | SWT.BORDER);
		arrayReferenceZ.setLayoutData(gd);
		arrayReferenceZ.addVerifyListener(new VerifyDecimalListener());
	}
	
	public HwConfiguration getConfiguration()
	{
		return this.config;
	}

	public Boolean getActive() {
		return activeButton.getSelection();
	}
	
	public String getDescription() {
		return descriptionText.getText();
	}

	public String getFullName() {
		return fullNameText.getText();
	}

	public String getTelescopeName() {
		return telescopeNameText.getText();
	}

	public String getConfigurationName() {
		return configurationNameText.getText();
	}

	public void setDirtyNoUpdate(boolean dirty) {
		this.dirty = dirty;
	}

	public HwConfiguration getGlobalConfiguration() {
		return this.globalConfig;
	}
	
}
