/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.widgets;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.action.Action;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchWindow;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentImplLang;
import alma.obops.tmcdbgui.editors.TmcdbObjectEditor;
import alma.obops.tmcdbgui.handlers.EditXmlAction;
import alma.obops.tmcdbgui.utils.GuiUtils;

/**
 * Composite used in displaying and/or editing component objects.
 * @author sharring
 */
public class ComponentEditingComposite extends Composite implements SelectionListener, PropertyChangeListener
{
	private static final String IMPLEMENTATION_LANGUAGE = "Implementation Language";
	private static final String KEEP_ALIVE_TIME = "Keep Alive Time";
	private static final String KEEP_ALIVE_TIME_PROPERTY = "keepAliveTime";
	private static final String EDIT = "Edit...";
	private static final String XML_DOCUMENTATION = "XML Documentation";
	private static final String MINIMUM_LOG_LEVEL_LOCAL = "Minimum Log Level (local)";
	private static final String MINIMUM_LOG_LEVEL = "Minimum Log Level";
	private static final String IS_STAND_ALONE_DEFINED = "Is stand alone defined?";
	private static final String IS_DEFAULT = "Is default?";
	private static final String IS_AUTOSTART = "Is autostart?";
	private static final String REAL_TIME = "Real-time?";
	private static final String IS_CONTROL_COMPONENT = "Is CONTROL Component?";
	private static final String CODE = "Code";
	private static final String COMPONENT_TYPE_IDL = "Component Type (IDL)";
	private static final String PATH = "Path";
	private static final String NAME = "Name";
	public static final String XML_DOC_PROPERTY = "XMLDoc";
	private static final String MIN_LOG_LEVEL_LOCAL_PROPERTY = "minLogLevelLocal";
	private static final String MIN_LOG_LEVEL_PROPERTY = "minLogLevel";
	private static final String IS_STANDALONE_DEFINED_PROPERTY = "isStandaloneDefined";
	private static final String IS_DEFAULT_PROPERTY = "isDefault";
	private static final String IS_AUTOSTART_PROPERTY = "isAutostart";
	private static final String REAL_TIME_PROPERTY = "realTime";
	private static final String IS_CONTROL_PROPERTY = "isControl";
	private static final String COMPONENT_CODE_PROPERTY = "code";
	private static final String COMPONENT_TYPE_IDL_PROPERTY = "componentType.IDL";
	private static final String COMPONENT_PATH_PROPERTY = "path";
	private static final String COMPONENT_NAME_PROPERTY = "componentName";
	private static final String IMPL_LANG_PROPERTY = "implLang";
	public static final String WIDGET_ENABLER = "widgetEnabler";
	private static final int TEXT_WIDTH = 150;
	private static final String SLASH = "/";
	
	private static class WidgetAssociation {
		public Button checkBox;
		public Control widget;
	}
	
	private boolean includeCheckboxes;
	private Map<String, WidgetAssociation> associationMap = new HashMap<String, WidgetAssociation>();
	private Listener listener;
	
	/* Widgets */
	private Button cXmlDocButton;
	
	private Action editXmlAction;
	private TmcdbObjectEditor editor;
	private Component _component;
	private IWorkbenchWindow _window;
	
	/**
	 * Constructor.
	 * @param editor the tmcdbobjecteditor, if any, used to edit the object
	 * @param parent parent composite.
	 * @param style swt style.
	 * @param comp the component that we're editing. 
	 * @param window the workbenchwindow that contains the editor (or wizardpage)
	 */
	public ComponentEditingComposite(TmcdbObjectEditor editor, Composite parent, int style, Component comp, IWorkbenchWindow window, Listener listener) 
	{
		super(parent, style);
		_window = window;
		this.listener = listener;
		this.editor = editor;
		this.includeCheckboxes = (editor == null);
		this._component = comp;
		initialize(comp);
		if( editor != null ) {
			bindDataEditorToGuiWidgets();
		} 
	}

	@Override 
	public boolean setFocus() {
		return (associationMap.get(COMPONENT_NAME_PROPERTY)).widget.setFocus();
	}
	
	@Override
	public void dispose()
	{
		super.dispose();
		if(null != _component) {
			this._component.removePropertyChangeListener(XML_DOC_PROPERTY, this);
		}
	}
	
	private void bindDataEditorToGuiWidgets()
	{
		// Data Binding
		editor.bind( COMPONENT_NAME_PROPERTY, widgetForProperty(COMPONENT_NAME_PROPERTY));
		editor.bind( COMPONENT_CODE_PROPERTY, widgetForProperty(COMPONENT_CODE_PROPERTY));
		editor.bind( COMPONENT_PATH_PROPERTY, widgetForProperty(COMPONENT_PATH_PROPERTY));
		editor.bindComponentImplLangCombo( IMPL_LANG_PROPERTY, (Combo)(widgetForProperty(IMPL_LANG_PROPERTY)));
		editor.bind( IS_CONTROL_PROPERTY, widgetForProperty(IS_CONTROL_PROPERTY));
		editor.bind( REAL_TIME_PROPERTY, widgetForProperty(REAL_TIME_PROPERTY));
		editor.bind( IS_AUTOSTART_PROPERTY, widgetForProperty(IS_AUTOSTART_PROPERTY));
		editor.bind( IS_DEFAULT_PROPERTY, widgetForProperty(IS_DEFAULT_PROPERTY));
		editor.bind( IS_STANDALONE_DEFINED_PROPERTY, widgetForProperty(IS_STANDALONE_DEFINED_PROPERTY));
		editor.bind( XML_DOC_PROPERTY, widgetForProperty(XML_DOC_PROPERTY));
		this._component.addPropertyChangeListener(XML_DOC_PROPERTY, this);
		editor.bind( COMPONENT_TYPE_IDL_PROPERTY, widgetForProperty(COMPONENT_TYPE_IDL_PROPERTY));
		editor.bindLogLevelCombo( MIN_LOG_LEVEL_PROPERTY, (Combo)widgetForProperty(MIN_LOG_LEVEL_PROPERTY) );
		editor.bindLogLevelCombo( MIN_LOG_LEVEL_LOCAL_PROPERTY, (Combo)widgetForProperty(MIN_LOG_LEVEL_LOCAL_PROPERTY) );
		editor.bind( KEEP_ALIVE_TIME_PROPERTY, widgetForProperty(KEEP_ALIVE_TIME_PROPERTY) );
		subscribeToChanges((Text)widgetForProperty(COMPONENT_TYPE_IDL_PROPERTY)); // we need to subscribe manually since the property name is not standard for this binded widget
	}
	
	private void initialize(Component comp) 
	{
		/* Name */
		createWidget(this, COMPONENT_NAME_PROPERTY, NAME, SWT.BORDER, Text.class, new GridData(SWT.FILL, SWT.CENTER, true, false));

		/* Path */
		createWidget(this, COMPONENT_PATH_PROPERTY, PATH, SWT.BORDER, Text.class, new GridData(SWT.FILL, SWT.CENTER, true, false));

		/* Component Type */
		createWidget(this, COMPONENT_TYPE_IDL_PROPERTY, COMPONENT_TYPE_IDL, SWT.BORDER, Text.class, new GridData(SWT.FILL, SWT.CENTER, true, false));

		/* Code */
		createWidget(this, COMPONENT_CODE_PROPERTY, CODE, SWT.BORDER, Text.class, new GridData(SWT.FILL, SWT.CENTER, true, false));
		
		/* Impl. Lang */
		//createImplLangWidget();
		createWidget(this, IMPL_LANG_PROPERTY, IMPLEMENTATION_LANGUAGE, SWT.DROP_DOWN | SWT.READ_ONLY, Combo.class, new GridData(SWT.LEFT, SWT.CENTER, false, false));

		/* Is Control? */
		createWidget(this, IS_CONTROL_PROPERTY, IS_CONTROL_COMPONENT, SWT.CHECK, Button.class, new GridData(SWT.LEFT, SWT.CENTER, false, false));

		/* Real-time */
		createWidget(this, REAL_TIME_PROPERTY, REAL_TIME, SWT.CHECK, Button.class, new GridData(SWT.LEFT, SWT.CENTER, false, false));

		/* Is auto-start */
		createWidget(this, IS_AUTOSTART_PROPERTY, IS_AUTOSTART, SWT.CHECK, Button.class, new GridData(SWT.LEFT, SWT.CENTER, false, false));
		
		/* Is default */
		createWidget(this, IS_DEFAULT_PROPERTY, IS_DEFAULT, SWT.CHECK, Button.class, new GridData(SWT.LEFT, SWT.CENTER, false, false));

		/* Is stand-alone defined */
		createWidget(this, IS_STANDALONE_DEFINED_PROPERTY, IS_STAND_ALONE_DEFINED, SWT.CHECK, Button.class, new GridData(SWT.LEFT, SWT.CENTER, false, false));

		/* Keep Alive Time */
		createWidget(this, KEEP_ALIVE_TIME_PROPERTY, KEEP_ALIVE_TIME, SWT.NONE, Spinner.class, new GridData(SWT.LEFT, SWT.CENTER, false, false));

		/* Minimum log level */
		createWidget(this, MIN_LOG_LEVEL_PROPERTY, MINIMUM_LOG_LEVEL, SWT.DROP_DOWN | SWT.READ_ONLY, Combo.class, new GridData(SWT.LEFT, SWT.CENTER, false, false));

		/* Minimum log level local */
		createWidget(this, MIN_LOG_LEVEL_LOCAL_PROPERTY, MINIMUM_LOG_LEVEL_LOCAL, SWT.DROP_DOWN | SWT.READ_ONLY, Combo.class, new GridData(SWT.LEFT, SWT.CENTER, false, false));

		/* XML Doc */
		createXmlDocWidget(comp);
		
		GridLayout gridLayout = new GridLayout();
		gridLayout.makeColumnsEqualWidth = false;
		gridLayout.numColumns = ( includeCheckboxes ? 3 : 2);
		gridLayout.horizontalSpacing = 10;
		this.setLayout(gridLayout);
		
		// If including checkboxes, set everything to disabled at the beginning
		if( includeCheckboxes ) {
			for (WidgetAssociation association: associationMap.values()) {
				association.checkBox.setSelection(false);
				association.widget.setEnabled(false);
			}
		}
		
		setComponent(comp);
		this.pack();
	}

	private void createXmlDocWidget(Component comp) 
	{
		int style = SWT.BORDER | SWT.MULTI;
		if(!includeCheckboxes) {
			style |= SWT.READ_ONLY;
		} else {
			style |= SWT.V_SCROLL | SWT.H_SCROLL;
		}
		createWidget(this, XML_DOC_PROPERTY, XML_DOCUMENTATION, style, Text.class, new GridData(SWT.FILL, SWT.FILL, true, true));
		Text widgetForXmlDoc = ((Text)widgetForProperty(XML_DOC_PROPERTY));
		GridData gd = (GridData)widgetForXmlDoc.getLayoutData();
		
		int cols = 80;
		int rows = 10;
		GC gc = new GC(widgetForXmlDoc);
		FontMetrics fm = gc.getFontMetrics ();
		int width = cols * fm.getAverageCharWidth();
		int height = rows * fm.getHeight ();
		gc.dispose ();
		
		gd.heightHint = height;
		gd.widthHint = width;
		widgetForXmlDoc.setLayoutData(gd);

		if(!includeCheckboxes)  {
			/* Dummy label and "Edit" button */
			editXmlAction = new EditXmlAction(_window, comp);
			new Label(this, SWT.NONE);
			GridData gd2 = new GridData(SWT.LEFT, SWT.TOP, false, false);
			gd2.horizontalIndent = 20;
			cXmlDocButton = new Button(this, SWT.PUSH );
			cXmlDocButton.setText(EDIT);
			cXmlDocButton.addListener(SWT.Selection, new Listener() {
				public void handleEvent(Event event) {
					editXmlAction.run();
				}
			});
			cXmlDocButton.setLayoutData(gd2);
		}
	}

	public Control widgetForProperty(String propertyName) {
		return associationMap.get(propertyName).widget;
	}
	
	private void createWidget( Composite parent, String propertyName, String labelText, int style, Class<?> clazz, GridData gdata) 
	{
		// Initial checkbox, if needed
		Button checkBox = null;
		if( includeCheckboxes ) {
			checkBox = new Button(this, SWT.CHECK);
			checkBox.setData(WIDGET_ENABLER, true);
		}
		
		// Create the label
		Label label = new Label(parent, SWT.NONE);
		label.setText(labelText);

		GridData gd;
		// Create the widget
		if(gdata == null) {
			gd = new GridData(); 
   		    gd.widthHint = TEXT_WIDTH;
		} else {
			gd = gdata;
		}
		gd.horizontalIndent = 20;
		gd.verticalIndent = 5;
		gd.horizontalIndent = 5;
		Control widget = null;
		
		if( clazz.equals(Text.class) ) {
			widget = new Text(parent, style);
			if( listener != null )
				widget.addListener(SWT.KeyUp, listener);
		}
		else if( clazz.equals(Combo.class) && !propertyName.equals(IMPL_LANG_PROPERTY)) {
			widget = GuiUtils.createLogLevelCombo(parent);
			if( listener != null )
				widget.addListener(SWT.Selection, listener);
		}
		else if( clazz.equals(Combo.class) && propertyName.equals(IMPL_LANG_PROPERTY)) {
			widget = new Combo(parent, style);
			String[] choices = {ComponentImplLang.CPP.toString(), ComponentImplLang.JAVA.toString(), ComponentImplLang.PY.toString()};
			((Combo)widget).setItems(choices);
			((Combo)widget).select(0);
    			((Combo)widget).setData("type", "implLang");
			if( listener != null )
				widget.addListener(SWT.Selection, listener);
		}
		else if( clazz.equals(Button.class) ) {
			widget = new Button(parent, style);
			if( listener != null )
				widget.addListener(SWT.Selection, listener);
		} 
		else if( clazz.equals(Spinner.class)) {
			Spinner spinner = new Spinner(parent, style);
			spinner.setMinimum(-1);
			spinner.setMaximum(Integer.MAX_VALUE);
			widget = spinner;
			if( listener != null )
				widget.addListener(SWT.Modify, listener);
		}
		else {
			throw new RuntimeException("Unsupported class: " + clazz);
		}
		widget.setLayoutData(gd);

		// Index the widget with its property name (also the checkbox, if needed)
		WidgetAssociation association = new WidgetAssociation();
		association.widget = widget;
		if( includeCheckboxes && checkBox != null) {
			checkBox.addSelectionListener(this);
			if( listener != null )
				checkBox.addListener(SWT.Selection, listener);
			association.checkBox = checkBox;
		}
		associationMap.put(propertyName, association);
	}
	
	protected Boolean nullSafeBoolean(Boolean b, Boolean v) {
		if( b == null )
			return v;
		return b;
	}
	
	protected void subscribeToChanges(Text ... widgets) {
		KeyListener theListener = new KeyListener() {
			public void keyReleased(KeyEvent e) { }
			public void keyPressed(KeyEvent e) {
				switch(e.keyCode) {
					case SWT.CR:
					case SWT.LF:
					case SWT.COMMAND:
					case SWT.KEYPAD_CR:
					case SWT.ESC:
					case SWT.NUM_LOCK:
					case SWT.CAPS_LOCK:
					case SWT.SCROLL_LOCK:
					case SWT.CTRL:
					case SWT.SHIFT:
					case SWT.ALT:
					case SWT.INSERT:
					case SWT.PAGE_DOWN:
					case SWT.PAGE_UP:
					case SWT.HOME:
					case SWT.END:
						break;
					default:
						editor.setDirty(true);
				}
			}
		};
		for(Text t: widgets)
			t.addKeyListener(theListener);
	}

	@Override
	public void propertyChange(PropertyChangeEvent evt) 
	{
		editor.propertyChange(evt);
		Control widgetForProperty = widgetForProperty(XML_DOC_PROPERTY);
		Text textWidget = (Text)widgetForProperty;
		if(null != textWidget && !textWidget.isDisposed() && evt.getPropertyName().equals(XML_DOC_PROPERTY) ) 
		{
			if( _component.getXMLDoc() != null ) {
				textWidget.setText(_component.getXMLDoc());
			}
			else {
				textWidget.setText("");
			}
		}
	}

	@Override
	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}

	@Override
	public void widgetSelected(SelectionEvent e) {

		Control w = null;
		for(WidgetAssociation association: associationMap.values()) {
			if( association.checkBox == e.widget )
				w = association.widget;
			
		}
		if( w == null )
			return;
		w.setEnabled(!w.getEnabled());
		
	}
	
	public String[] getEnabledProperties() {

		List<String> result = new ArrayList<String>();
		for (String propName : associationMap.keySet()) {
			if( associationMap.get(propName).widget.isEnabled() )
				result.add(propName);
		}

		return result.toArray(new String[0]);
	}

	public Object[] getValuesForEnabledProperties() {

		List<Object> result = new ArrayList<Object>();
		for (String propName : associationMap.keySet()) {
			Control widget = associationMap.get(propName).widget;
			if( widget.isEnabled() ) {
				if( widget instanceof Text ) {
					result.add(getObjectSafeValueForProperty(propName, ((Text)widget).getText()));
				}
				else if( widget instanceof Button ) {
					result.add(((Button)widget).getSelection());
				}
				else if( widget instanceof Combo ) {
					Combo c = (Combo)widget;
					int selIndex = c.getSelectionIndex();
					result.add( getObjectSafeValueForProperty(propName, c.getItems()[selIndex]) );
				} else if( widget instanceof Spinner ) {
					Spinner s = (Spinner) widget;
					result.add( getObjectSafeValueForProperty(propName, s.getText()));
				}
			}
		}

		return result.toArray();
	}
	
	private Object getObjectSafeValueForProperty(String propName, String value) {

		Object retVal = null;

		// Booleans
		if( propName.equals(REAL_TIME_PROPERTY) ||
		    propName.equals(IS_STANDALONE_DEFINED_PROPERTY)  ||
		    propName.equals(IS_DEFAULT_PROPERTY) ||
		    propName.equals(IS_DEFAULT_PROPERTY) ||
		    propName.equals(IS_CONTROL_PROPERTY)
		    )
		{
			retVal = Boolean.parseBoolean(value);
		}

		// Integers
		else if( propName.equals(KEEP_ALIVE_TIME_PROPERTY) ) 
		{
			retVal = Integer.parseInt(value);
		}

		// Regular strings (i.e. no special handling)
		else if( propName.equals(COMPONENT_NAME_PROPERTY) ||
			propName.equals(COMPONENT_CODE_PROPERTY) ||	
		    propName.equals(COMPONENT_TYPE_IDL_PROPERTY)) 
		{
			retVal = value;
		}
		
		// Special handling of XML_DOC_PROPERTY String:
		// Oracle doesn't like empty string for XML fields, let's null it if empty
		else if(propName.equals(XML_DOC_PROPERTY))
		{
			retVal = value;
			if(null != retVal && (((String)retVal).trim().equals("") || ((String)retVal).length() == 0)) 
			{
				retVal = null;
			}
		}
		
		// Special case of string for path (which uses a slash '/' in lieu of null)
		else if(propName.equals(COMPONENT_PATH_PROPERTY)) 
		{
			retVal = value;
			if(retVal == null || ((String)retVal).trim().equals("") || (((String)retVal).length() == 0)) 
			{
				retVal = SLASH;
			}
		}

		// Bytes
		else if(propName.equals(MIN_LOG_LEVEL_PROPERTY) ||
			propName.equals(MIN_LOG_LEVEL_LOCAL_PROPERTY) )
		{
			retVal = getByteForLogLevel(value);
		}
		
		// impl lang combo just returns enum (based on string from combo)
		else if(propName.equals(IMPL_LANG_PROPERTY)) {
			retVal = ComponentImplLang.valueOfForEnum(value);
		}
		
		return retVal;
	}
	
	private Byte getByteForLogLevel(String level) {
		Byte retVal = null;
		if(level.equals(alma.AcsLogLevels.TRACE_NAME.value)) {
			retVal = (byte)alma.AcsLogLevels.TRACE_VAL.value;
		} 
		else if(level.equals(alma.AcsLogLevels.DEBUG_NAME.value)) {
			retVal = (byte)alma.AcsLogLevels.DEBUG_VAL.value;
		}
		else if(level.equals(alma.AcsLogLevels.INFO_NAME.value)) {
			retVal = (byte)alma.AcsLogLevels.INFO_VAL.value;
		}
		else if(level.equals(alma.AcsLogLevels.NOTICE_NAME.value)) {
			retVal = (byte)alma.AcsLogLevels.NOTICE_VAL.value;
		} 
		else if(level.equals(alma.AcsLogLevels.WARNING_NAME.value)) {
			retVal = (byte)alma.AcsLogLevels.WARNING_VAL.value;
		} 
		else if(level.equals(alma.AcsLogLevels.ERROR_NAME.value)) {
			retVal = (byte)alma.AcsLogLevels.ERROR_VAL.value;
		}
		else if(level.equals(alma.AcsLogLevels.CRITICAL_NAME.value)) {
			retVal = (byte)alma.AcsLogLevels.CRITICAL_VAL.value;
		}
		else if(level.equals(alma.AcsLogLevels.ALERT_NAME.value)) {
			retVal = (byte)alma.AcsLogLevels.ALERT_VAL.value;
		}
		else if(level.equals(alma.AcsLogLevels.EMERGENCY_NAME.value)) {
			retVal = (byte)alma.AcsLogLevels.EMERGENCY_VAL.value;
		}
		else if(level.equals(alma.AcsLogLevels.OFF_NAME.value)) {
			retVal = (byte)alma.AcsLogLevels.OFF_VAL.value;
		}
		else if(level.equals(GuiUtils.LOG_LEVEL_NOT_SPECIFIED)) {
			retVal = (byte)-1;
		}
		return retVal;
	}

	public void setComponent(Component comp) {
		this._component = comp;
		populateLogLevelCombo();
		populateLogLevelLocalCombo();
	}

	private void populateLogLevelLocalCombo() 
	{
		if( _component == null )
			return;
		Combo combo = (Combo)widgetForProperty(MIN_LOG_LEVEL_PROPERTY);
		Byte val = _component.getMinLogLevelLocal();
		Integer valueInt = (val == null) ? 0 : val.intValue();
		combo.select(valueInt);
	}


	private void populateLogLevelCombo() {
		if( _component == null )
			return;
		Combo combo = (Combo)widgetForProperty(MIN_LOG_LEVEL_LOCAL_PROPERTY);
		Byte val = _component.getMinLogLevel();
		Integer valueInt = (val == null) ? 0 : val.intValue();
		combo.select(valueInt);
		
	}
}
